/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "BaseWSManClient.h"
#include "UNSCommonDefs.h"
#include "UNSEventLogger.h"
#include "UNSStatus.h"
#include "Credentials.h"

#ifdef _WIN32
#include <iphlpapi.h>
#else
#include <netdb.h>
#include <unistd.h>
#endif

#include <string>

#define WSMAN_PATH           "/wsman"
#define WSMAN_HTTPS_SCHEME   "https"
#define WSMAN_HTTP_SCHEME    "http"
#define WSMAN_AUTH_DIGEST    "digest"
using namespace std;

const string BaseWSManClient::IP_LOCALHOST = "127.0.0.1";


//************************************************************************
// Default Constructor.
//************************************************************************
BaseWSManClient::BaseWSManClient(const CredentialsRegInfo &CredRegInfo,
				 const std::string &defaultUser,
				 const std::string &defaultPass):
  m_CredRegInfo(CredRegInfo),
  m_defaultUser(defaultUser),
  m_defaultPass(defaultPass)
{
  Init();
}

//************************************************************************
// Name			: BaseWSManClient
// Description	: Destructor.
//************************************************************************
BaseWSManClient::~BaseWSManClient()
{
  delete m_client;
  delete m_ClientCred;
}

//************************************************************************
// Name:		BaseWSManClient::SetLogger
// Description: Set a pointer to an event logger.
//************************************************************************
void BaseWSManClient::SetLogger (UNSEventLogger *logger)
{
  m_Logger = logger;
}

//************************************************************************
// Name:		BaseWSManClient::Init
// Description: Get AMT FQDN, and set wsman endpoint.
//************************************************************************
void BaseWSManClient::Init()
{
  m_ip = IP_LOCALHOST; //Default ip is local host - 127.0.0.1

  // Retreive credentials if failed set the default.
  if ((m_ClientCred = RetrieveCredentials(m_CredRegInfo)) == NULL) {
    SetCredentials(Credentials(m_defaultUser, m_defaultPass, ""));
  }

  if (m_ClientCred->username.empty())
    m_ClientCred->username = m_defaultUser;

  if (m_ClientCred->password.empty())
    m_ClientCred->password = m_defaultPass;

  // Init some flags.
  m_endpoint   = false; // Endpoint not resolved yet.
  m_client	 = NULL;


  // Set certificate if given
}

//************************************************************************
// Name			: SetEndpoint.
// Description	: Set soap endpoint
//************************************************************************
int BaseWSManClient::SetEndpoint(bool secure)
{
  int status;
  char fqdn[FQDN_MAX_SIZE];

  // Get FQDN of local host
  if (m_fqdn.empty() && secure) {
    // First try to get the FQDN from the hosts file.
    status = FindHostsFileFQDN(fqdn);
    if (status < 0) {
      if (m_Logger) {
        m_Logger->DebugLog("Error while reading hosts file");
      }
    }

    // If FQDN wasn't found, try to get the local host FQDN.
    if (status != 1) {
      status = GetNetworkFQDN(fqdn);
    }

    // IF both failed return ERROR.
    if (status != 1) {
      if (m_Logger) {
        m_Logger->GetFqdnFailure();
      }
      return UNS_STATUS_SUBSCRIPTION_ERROR;
    }
    m_fqdn = fqdn;
  }

  if (secure) {
    if(m_client != NULL) delete m_client;
    m_client = new OpenWsmanClient(m_fqdn.c_str(),
				   AMT_SECURE_PORT,
				   WSMAN_PATH,
				   WSMAN_HTTPS_SCHEME,
				   WSMAN_AUTH_DIGEST,
				   m_ClientCred->username.c_str(),
				   m_ClientCred->password.c_str());

    m_client->SetServerCert(m_ClientCred->cainfo.empty() ? NULL : m_ClientCred->cainfo.c_str(),
			    m_ClientCred->capath.empty() ? NULL : m_ClientCred->capath.c_str());
    m_client->SetClientCert(m_ClientCred->certificate.empty() ? NULL : m_ClientCred->certificate.c_str(),
			    m_ClientCred->certificate.empty() ? NULL : m_ClientCred->sslkey.c_str());

  }
  else if (!secure){
    if(m_client != NULL) delete m_client;
    m_client = new OpenWsmanClient(m_ip.c_str(),
				   AMT_NON_SECURE_PORT,
				   WSMAN_PATH,
				   WSMAN_HTTP_SCHEME,
				   WSMAN_AUTH_DIGEST,
				   m_ClientCred->username.c_str(),
				   m_ClientCred->password.c_str());

  }

  return UNS_STATUS_SUCCESS;
}

//************************************************************************
// Name			: FindHostsFileFQDN
// Description	: Find the LMS inserted FQDN in the hosts file.
// Params		: fqdn - A pointer to a buffer at least FQDN_MAX_SIZE in length
//
// Returns		: 1 if FQDN was found (it is placed in @fqdn)
//		          0 if FQDN wasn't found
//				  < 0 on error.
//************************************************************************
int	BaseWSManClient::FindHostsFileFQDN (char *fqdn)
{
#define LMS_LINE_SIG_FIRST_WORDS "# LMS GENERATED "
#define LMS_LINE_SIG_LAST_WORD "LINE"
#define LMS_LINE_SIG_LAST_WORD_LEN 4
#define LMS_LINE_SIG LMS_LINE_SIG_FIRST_WORDS LMS_LINE_SIG_LAST_WORD
#define lmsstr(s) lmsname(s)
#define lmsname(s) #s
#define LMS_LINE_FORMAT "127.0.0.1       %s %s " LMS_LINE_SIG
#define LMS_LINE_SCAN_FORMAT "127.0.0.1 %" lmsstr(FQDN_MAX_SIZE) "s %" lmsstr(FQDN_MAX_SIZE) "s " LMS_LINE_SIG_FIRST_WORDS "%" lmsstr(LMS_LINE_SIG_LAST_WORD_LEN) "c"

	bool hasFqdn = false;
	char tmpFqdn[FQDN_MAX_SIZE + 1];
	char tmphost[FQDN_MAX_SIZE + 1];
	char tmpsige[LMS_LINE_SIG_LAST_WORD_LEN];
	char line[1024];
#ifdef _WIN32
	char *sysDrive;
	char *dir = "\\system32\\drivers\\etc\\";
	char *hostFile = "hosts";
	char inFileName[1024];
#else
	char *inFileName = "/etc/hosts";
#endif

	// NUL-terminate the FQDN string before doing anything
	*fqdn = '\0';

#ifdef _WIN32
	sysDrive = getenv("SystemRoot");
	if (NULL == sysDrive) {
		// Can't find hosts file
		return -1;
	}

	// sanity check before string copying
	if (1024 < (strlen(sysDrive) + strlen(dir) + strlen(hostFile) + 1)) {
		return -1;
	}

	strcpy(inFileName, sysDrive);
	strcat(inFileName, dir);
	strcat(inFileName, hostFile);
#endif

	FILE *ifp = fopen(inFileName, "r");
	if (NULL == ifp) {
		// Can't open hosts file
		return -1;
	}

	// Go over each line and check for LMS signature
	while (fgets(line, sizeof(line), ifp)) {
		memset(tmpFqdn, 0, sizeof(tmpFqdn));
		memset(tmpsige, 0, sizeof(tmpsige));
		if (0 == (
		    (3 == sscanf(line, LMS_LINE_SCAN_FORMAT, tmpFqdn, tmphost, tmpsige))
		    ? strncmp(tmpsige, LMS_LINE_SIG_LAST_WORD, LMS_LINE_SIG_LAST_WORD_LEN)
		    : (-2))
		) {
			strncpy(fqdn, tmpFqdn, FQDN_MAX_SIZE);
			hasFqdn = true;
			break;
		}
		// Badly formatted line, even though it has LMS sig; ignore it.
	}

	fclose(ifp);

	return (hasFqdn) ? 1 : 0;
}


//************************************************************************
// Name			: GetNetworkFQDN.
// Description	: Use the GetNetworkParams to query the local fqdn.
// Params		: fqdn - A pointer to a buffer at least FQDN_MAX_SIZE in length
// Returns		: 1 if FQDN was found (it is placed in @fqdn)
//		          <= 0 on error
//************************************************************************
int	BaseWSManClient::GetNetworkFQDN (char *fqdn)
{
#ifdef _WIN32
  FIXED_INFO *FixedInfo;
  ULONG		ulOutBufLen;
  DWORD		dwRetVal;

  // NUL-terminate the FQDN string before doing anything
  *fqdn = '\0';

  // Allocate memory.
  FixedInfo	= (FIXED_INFO *) GlobalAlloc( GPTR, sizeof( FIXED_INFO ) );
  ulOutBufLen = sizeof( FIXED_INFO );

  // If GetNetworkParams returns ERROR_BUFFER_OVERFLOW, realloc the required memory.
  if( ERROR_BUFFER_OVERFLOW == GetNetworkParams( FixedInfo, &ulOutBufLen ) ) {
    GlobalFree( FixedInfo );
    FixedInfo = (FIXED_INFO *) GlobalAlloc( GPTR, ulOutBufLen );
  }

  if ( dwRetVal = GetNetworkParams( FixedInfo, &ulOutBufLen ) ) {
    printf("Call to GetNetworkParams failed. Return Value: %08x\n", dwRetVal );

    return -1;
  }

  if ((strlen(FixedInfo -> HostName) + strlen(FixedInfo -> DomainName) + 2) > FQDN_MAX_SIZE){
    printf ("FQDN too long: %s.%s\n",  FixedInfo -> HostName, FixedInfo -> DomainName);
    return -1;
  }
  wsprintf(fqdn, "%s.%s", FixedInfo -> HostName, FixedInfo -> DomainName);
#else // LINUX
  char hostname[HOST_NAME_MAX];
  if (gethostname(hostname, HOST_NAME_MAX) != 0) {
    return -1;
  }
  struct hostent *ent = gethostbyname(hostname);
  if (ent == NULL) {
    return -1;
  }
  strncpy(fqdn, ent->h_name, FQDN_MAX_SIZE);
#endif
  return 1;
}

void BaseWSManClient::SetCredentials(const Credentials &creds)
{
  if (m_ClientCred == NULL)
    m_ClientCred = new Credentials;

  *m_ClientCred = creds;
}
