#ifndef _MISSING_CIPHERSUITES_H
#define _MISSING_CIPHERSUITES_H


/* At run-time, the findMissingCiphers() function will enumerate all the ciphers that OpenSSL knows of, and compares them each to the missing_ciphersuites list.  Afterwards, the 'check_tls_versions' field will have the set of TLS versions that OpenSSL does not have an implementation for (i.e.: if the TLS_RSA_WITH_IDEA_CBC_SHA cipher is set to V1_2, then OpenSSL has an implementation for it for TLS v1.0 and v1.1, but not for v1.2). */


#define V1_0 (1)
#define V1_1 (1 << 1)
#define V1_2 (1 << 2)
#define VALL (V1_0 | V1_1 | V1_2)


struct missing_ciphersuite {
  unsigned short id;      /* TLS protocol ID. */
  char protocol_name[48]; /* IANA name, as defined in the RFCs. */
  int bits;               /* Bit strength of the cipher.  -1 if unknown. */
  unsigned int check_tls_versions; /* OR'ed list of V1_? defines.  Refers to TLS version that OpenSSL does not have a ciphersuite implementation for.  Hence, it should be tested under this version of TLS. */
  unsigned int accepted_tls_versions;  /* OR'ed list of V1_? defines.  Set at run-time if this ciphersuite was accepted by the server by the specified TLS version. */
};


/* Auto-generated by ./iana_tls_ciphersuite_parser.py on December 10, 2019. */
struct missing_ciphersuite missing_ciphersuites[] = {
  {0x0000, "TLS_NULL_WITH_NULL_NULL", -1, VALL, 0},
  {0x0001, "TLS_RSA_WITH_NULL_MD5", -1, VALL, 0},
  {0x0002, "TLS_RSA_WITH_NULL_SHA", -1, VALL, 0},
  {0x0003, "TLS_RSA_EXPORT_WITH_RC4_40_MD5", 40, VALL, 0},
  {0x0004, "TLS_RSA_WITH_RC4_128_MD5", 128, VALL, 0},
  {0x0005, "TLS_RSA_WITH_RC4_128_SHA", 128, VALL, 0},
  {0x0006, "TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5", 40, VALL, 0},
  {0x0007, "TLS_RSA_WITH_IDEA_CBC_SHA", 128, VALL, 0},
  {0x0008, "TLS_RSA_EXPORT_WITH_DES40_CBC_SHA", 40, VALL, 0},
  {0x0009, "TLS_RSA_WITH_DES_CBC_SHA", 56, VALL, 0},
  {0x000A, "TLS_RSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x000B, "TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA", 40, VALL, 0},
  {0x000C, "TLS_DH_DSS_WITH_DES_CBC_SHA", 56, VALL, 0},
  {0x000D, "TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x000E, "TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA", 40, VALL, 0},
  {0x000F, "TLS_DH_RSA_WITH_DES_CBC_SHA", 56, VALL, 0},
  {0x0010, "TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x0011, "TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA", 40, VALL, 0},
  {0x0012, "TLS_DHE_DSS_WITH_DES_CBC_SHA", 56, VALL, 0},
  {0x0013, "TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x0014, "TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA", 40, VALL, 0},
  {0x0015, "TLS_DHE_RSA_WITH_DES_CBC_SHA", 56, VALL, 0},
  {0x0016, "TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x0017, "TLS_DH_anon_EXPORT_WITH_RC4_40_MD5", 40, VALL, 0},
  {0x0018, "TLS_DH_anon_WITH_RC4_128_MD5", 128, VALL, 0},
  {0x0019, "TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA", 40, VALL, 0},
  {0x001A, "TLS_DH_anon_WITH_DES_CBC_SHA", 56, VALL, 0},
  {0x001B, "TLS_DH_anon_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x001E, "TLS_KRB5_WITH_DES_CBC_SHA", 56, VALL, 0},
  {0x001F, "TLS_KRB5_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x0020, "TLS_KRB5_WITH_RC4_128_SHA", 128, VALL, 0},
  {0x0021, "TLS_KRB5_WITH_IDEA_CBC_SHA", 128, VALL, 0},
  {0x0022, "TLS_KRB5_WITH_DES_CBC_MD5", 56, VALL, 0},
  {0x0023, "TLS_KRB5_WITH_3DES_EDE_CBC_MD5", 112, VALL, 0},
  {0x0024, "TLS_KRB5_WITH_RC4_128_MD5", 128, VALL, 0},
  {0x0025, "TLS_KRB5_WITH_IDEA_CBC_MD5", 128, VALL, 0},
  {0x0026, "TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA", 56, VALL, 0},
  {0x0027, "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA", 40, VALL, 0},
  {0x0028, "TLS_KRB5_EXPORT_WITH_RC4_40_SHA", 40, VALL, 0},
  {0x0029, "TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5", 56, VALL, 0},
  {0x002A, "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5", 40, VALL, 0},
  {0x002B, "TLS_KRB5_EXPORT_WITH_RC4_40_MD5", 40, VALL, 0},
  {0x002C, "TLS_PSK_WITH_NULL_SHA", -1, VALL, 0},
  {0x002D, "TLS_DHE_PSK_WITH_NULL_SHA", -1, VALL, 0},
  {0x002E, "TLS_RSA_PSK_WITH_NULL_SHA", -1, VALL, 0},
  {0x002F, "TLS_RSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0030, "TLS_DH_DSS_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0031, "TLS_DH_RSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0032, "TLS_DHE_DSS_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0033, "TLS_DHE_RSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0034, "TLS_DH_anon_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0035, "TLS_RSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x0036, "TLS_DH_DSS_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x0037, "TLS_DH_RSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x0038, "TLS_DHE_DSS_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x0039, "TLS_DHE_RSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x003A, "TLS_DH_anon_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x003B, "TLS_RSA_WITH_NULL_SHA256", -1, VALL, 0},
  {0x003C, "TLS_RSA_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x003D, "TLS_RSA_WITH_AES_256_CBC_SHA256", 256, VALL, 0},
  {0x003E, "TLS_DH_DSS_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x003F, "TLS_DH_RSA_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x0040, "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x0041, "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA", 128, VALL, 0},
  {0x0042, "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA", 128, VALL, 0},
  {0x0043, "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA", 128, VALL, 0},
  {0x0044, "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA", 128, VALL, 0},
  {0x0045, "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA", 128, VALL, 0},
  {0x0046, "TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA", 128, VALL, 0},
  {0x0067, "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x0068, "TLS_DH_DSS_WITH_AES_256_CBC_SHA256", 256, VALL, 0},
  {0x0069, "TLS_DH_RSA_WITH_AES_256_CBC_SHA256", 256, VALL, 0},
  {0x006A, "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256", 256, VALL, 0},
  {0x006B, "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256", 256, VALL, 0},
  {0x006C, "TLS_DH_anon_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x006D, "TLS_DH_anon_WITH_AES_256_CBC_SHA256", 256, VALL, 0},
  {0x0084, "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA", 256, VALL, 0},
  {0x0085, "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA", 256, VALL, 0},
  {0x0086, "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA", 256, VALL, 0},
  {0x0087, "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA", 256, VALL, 0},
  {0x0088, "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA", 256, VALL, 0},
  {0x0089, "TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA", 256, VALL, 0},
  {0x008A, "TLS_PSK_WITH_RC4_128_SHA", 128, VALL, 0},
  {0x008B, "TLS_PSK_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x008C, "TLS_PSK_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x008D, "TLS_PSK_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x008E, "TLS_DHE_PSK_WITH_RC4_128_SHA", 128, VALL, 0},
  {0x008F, "TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x0090, "TLS_DHE_PSK_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0091, "TLS_DHE_PSK_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x0092, "TLS_RSA_PSK_WITH_RC4_128_SHA", 128, VALL, 0},
  {0x0093, "TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0x0094, "TLS_RSA_PSK_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0x0095, "TLS_RSA_PSK_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0x0096, "TLS_RSA_WITH_SEED_CBC_SHA", 128, VALL, 0},
  {0x0097, "TLS_DH_DSS_WITH_SEED_CBC_SHA", 128, VALL, 0},
  {0x0098, "TLS_DH_RSA_WITH_SEED_CBC_SHA", 128, VALL, 0},
  {0x0099, "TLS_DHE_DSS_WITH_SEED_CBC_SHA", 128, VALL, 0},
  {0x009A, "TLS_DHE_RSA_WITH_SEED_CBC_SHA", 128, VALL, 0},
  {0x009B, "TLS_DH_anon_WITH_SEED_CBC_SHA", 128, VALL, 0},
  {0x009C, "TLS_RSA_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x009D, "TLS_RSA_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x009E, "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x009F, "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00A0, "TLS_DH_RSA_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x00A1, "TLS_DH_RSA_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00A2, "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x00A3, "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00A4, "TLS_DH_DSS_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x00A5, "TLS_DH_DSS_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00A6, "TLS_DH_anon_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x00A7, "TLS_DH_anon_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00A8, "TLS_PSK_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x00A9, "TLS_PSK_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00AA, "TLS_DHE_PSK_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x00AB, "TLS_DHE_PSK_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00AC, "TLS_RSA_PSK_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x00AD, "TLS_RSA_PSK_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x00AE, "TLS_PSK_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x00AF, "TLS_PSK_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0x00B0, "TLS_PSK_WITH_NULL_SHA256", -1, VALL, 0},
  {0x00B1, "TLS_PSK_WITH_NULL_SHA384", -1, VALL, 0},
  {0x00B2, "TLS_DHE_PSK_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x00B3, "TLS_DHE_PSK_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0x00B4, "TLS_DHE_PSK_WITH_NULL_SHA256", -1, VALL, 0},
  {0x00B5, "TLS_DHE_PSK_WITH_NULL_SHA384", -1, VALL, 0},
  {0x00B6, "TLS_RSA_PSK_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0x00B7, "TLS_RSA_PSK_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0x00B8, "TLS_RSA_PSK_WITH_NULL_SHA256", -1, VALL, 0},
  {0x00B9, "TLS_RSA_PSK_WITH_NULL_SHA384", -1, VALL, 0},
  {0x00BA, "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0x00BB, "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0x00BC, "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0x00BD, "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0x00BE, "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0x00BF, "TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0x00C0, "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256", 256, VALL, 0},
  {0x00C1, "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256", 256, VALL, 0},
  {0x00C2, "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256", 256, VALL, 0},
  {0x00C3, "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256", 256, VALL, 0},
  {0x00C4, "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256", 256, VALL, 0},
  {0x00C5, "TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256", 256, VALL, 0},
  {0x00C6, "TLS_SM4_GCM_SM3", 128, VALL, 0},
  {0x00C7, "TLS_SM4_CCM_SM3", 128, VALL, 0},
  {0x1301, "TLS_AES_128_GCM_SHA256", 128, VALL, 0},
  {0x1302, "TLS_AES_256_GCM_SHA384", 256, VALL, 0},
  {0x1303, "TLS_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0x1304, "TLS_AES_128_CCM_SHA256", 128, VALL, 0},
  {0x1305, "TLS_AES_128_CCM_8_SHA256", 128, VALL, 0},
  {0xC001, "TLS_ECDH_ECDSA_WITH_NULL_SHA", -1, VALL, 0},
  {0xC002, "TLS_ECDH_ECDSA_WITH_RC4_128_SHA", 128, VALL, 0},
  {0xC003, "TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC004, "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC005, "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC006, "TLS_ECDHE_ECDSA_WITH_NULL_SHA", -1, VALL, 0},
  {0xC007, "TLS_ECDHE_ECDSA_WITH_RC4_128_SHA", 128, VALL, 0},
  {0xC008, "TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC009, "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC00A, "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC00B, "TLS_ECDH_RSA_WITH_NULL_SHA", -1, VALL, 0},
  {0xC00C, "TLS_ECDH_RSA_WITH_RC4_128_SHA", 128, VALL, 0},
  {0xC00D, "TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC00E, "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC00F, "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC010, "TLS_ECDHE_RSA_WITH_NULL_SHA", -1, VALL, 0},
  {0xC011, "TLS_ECDHE_RSA_WITH_RC4_128_SHA", 128, VALL, 0},
  {0xC012, "TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC013, "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC014, "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC015, "TLS_ECDH_anon_WITH_NULL_SHA", -1, VALL, 0},
  {0xC016, "TLS_ECDH_anon_WITH_RC4_128_SHA", 128, VALL, 0},
  {0xC017, "TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC018, "TLS_ECDH_anon_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC019, "TLS_ECDH_anon_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC01A, "TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC01B, "TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC01C, "TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC01D, "TLS_SRP_SHA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC01E, "TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC01F, "TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC020, "TLS_SRP_SHA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC021, "TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC022, "TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC023, "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0xC024, "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0xC025, "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0xC026, "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0xC027, "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0xC028, "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0xC029, "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0xC02A, "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0xC02B, "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0xC02C, "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0xC02D, "TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0xC02E, "TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0xC02F, "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0xC030, "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0xC031, "TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0xC032, "TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0xC033, "TLS_ECDHE_PSK_WITH_RC4_128_SHA", 128, VALL, 0},
  {0xC034, "TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA", 112, VALL, 0},
  {0xC035, "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA", 128, VALL, 0},
  {0xC036, "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA", 256, VALL, 0},
  {0xC037, "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256", 128, VALL, 0},
  {0xC038, "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384", 256, VALL, 0},
  {0xC039, "TLS_ECDHE_PSK_WITH_NULL_SHA", -1, VALL, 0},
  {0xC03A, "TLS_ECDHE_PSK_WITH_NULL_SHA256", -1, VALL, 0},
  {0xC03B, "TLS_ECDHE_PSK_WITH_NULL_SHA384", -1, VALL, 0},
  {0xC03C, "TLS_RSA_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC03D, "TLS_RSA_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC03E, "TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC03F, "TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC040, "TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC041, "TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC042, "TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC043, "TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC044, "TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC045, "TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC046, "TLS_DH_anon_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC047, "TLS_DH_anon_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC048, "TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC049, "TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC04A, "TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC04B, "TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC04C, "TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC04D, "TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC04E, "TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC04F, "TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC050, "TLS_RSA_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC051, "TLS_RSA_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC052, "TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC053, "TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC054, "TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC055, "TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC056, "TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC057, "TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC058, "TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC059, "TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC05A, "TLS_DH_anon_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC05B, "TLS_DH_anon_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC05C, "TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC05D, "TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC05E, "TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC05F, "TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC060, "TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC061, "TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC062, "TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC063, "TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC064, "TLS_PSK_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC065, "TLS_PSK_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC066, "TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC067, "TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC068, "TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC069, "TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC06A, "TLS_PSK_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC06B, "TLS_PSK_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC06C, "TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC06D, "TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC06E, "TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC06F, "TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC070, "TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC071, "TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC072, "TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC073, "TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC074, "TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC075, "TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC076, "TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC077, "TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC078, "TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC079, "TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC07A, "TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC07B, "TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC07C, "TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC07D, "TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC07E, "TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC07F, "TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC080, "TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC081, "TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC082, "TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC083, "TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC084, "TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC085, "TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC086, "TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC087, "TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC088, "TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC089, "TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC08A, "TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC08B, "TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC08C, "TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC08D, "TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC08E, "TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC08F, "TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC090, "TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC091, "TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC092, "TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256", 128, VALL, 0},
  {0xC093, "TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384", 256, VALL, 0},
  {0xC094, "TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC095, "TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC096, "TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC097, "TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC098, "TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC099, "TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC09A, "TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256", 128, VALL, 0},
  {0xC09B, "TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384", 256, VALL, 0},
  {0xC09C, "TLS_RSA_WITH_AES_128_CCM", 128, VALL, 0},
  {0xC09D, "TLS_RSA_WITH_AES_256_CCM", 256, VALL, 0},
  {0xC09E, "TLS_DHE_RSA_WITH_AES_128_CCM", 128, VALL, 0},
  {0xC09F, "TLS_DHE_RSA_WITH_AES_256_CCM", 256, VALL, 0},
  {0xC0A0, "TLS_RSA_WITH_AES_128_CCM_8", 128, VALL, 0},
  {0xC0A1, "TLS_RSA_WITH_AES_256_CCM_8", 256, VALL, 0},
  {0xC0A2, "TLS_DHE_RSA_WITH_AES_128_CCM_8", 128, VALL, 0},
  {0xC0A3, "TLS_DHE_RSA_WITH_AES_256_CCM_8", 256, VALL, 0},
  {0xC0A4, "TLS_PSK_WITH_AES_128_CCM", 128, VALL, 0},
  {0xC0A5, "TLS_PSK_WITH_AES_256_CCM", 256, VALL, 0},
  {0xC0A6, "TLS_DHE_PSK_WITH_AES_128_CCM", 128, VALL, 0},
  {0xC0A7, "TLS_DHE_PSK_WITH_AES_256_CCM", 256, VALL, 0},
  {0xC0A8, "TLS_PSK_WITH_AES_128_CCM_8", 128, VALL, 0},
  {0xC0A9, "TLS_PSK_WITH_AES_256_CCM_8", 256, VALL, 0},
  {0xC0AA, "TLS_PSK_DHE_WITH_AES_128_CCM_8", 128, VALL, 0},
  {0xC0AB, "TLS_PSK_DHE_WITH_AES_256_CCM_8", 256, VALL, 0},
  {0xC0AC, "TLS_ECDHE_ECDSA_WITH_AES_128_CCM", 128, VALL, 0},
  {0xC0AD, "TLS_ECDHE_ECDSA_WITH_AES_256_CCM", 256, VALL, 0},
  {0xC0AE, "TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8", 128, VALL, 0},
  {0xC0AF, "TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8", 256, VALL, 0},
  {0xC0B0, "TLS_ECCPWD_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0xC0B1, "TLS_ECCPWD_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0xC0B2, "TLS_ECCPWD_WITH_AES_128_CCM_SHA256", 128, VALL, 0},
  {0xC0B3, "TLS_ECCPWD_WITH_AES_256_CCM_SHA384", 256, VALL, 0},
  {0xC0B4, "TLS_SHA256_SHA256", -1, VALL, 0},
  {0xC0B5, "TLS_SHA384_SHA384", -1, VALL, 0},
  {0xC100, "TLS_GOSTR341112_256_WITH_KUZNYECHIK_CTR_OMAC", 256, VALL, 0},
  {0xC101, "TLS_GOSTR341112_256_WITH_MAGMA_CTR_OMAC", 256, VALL, 0},
  {0xC102, "TLS_GOSTR341112_256_WITH_28147_CNT_IMIT", 256, VALL, 0},
  {0xCCA8, "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0xCCA9, "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0xCCAA, "TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0xCCAB, "TLS_PSK_WITH_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0xCCAC, "TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0xCCAD, "TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0xCCAE, "TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256", 256, VALL, 0},
  {0xD001, "TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256", 128, VALL, 0},
  {0xD002, "TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384", 256, VALL, 0},
  {0xD003, "TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256", 128, VALL, 0},
  {0xD005, "TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256", 128, VALL, 0},

  /* The ciphers below are reserved for private use (see RFC8446). */
  {0xFF00, "PRIVATE_CIPHER_0", -1, VALL, 0},
  {0xFF01, "PRIVATE_CIPHER_1", -1, VALL, 0},
  {0xFF02, "PRIVATE_CIPHER_2", -1, VALL, 0},
  {0xFF03, "PRIVATE_CIPHER_3", -1, VALL, 0},
  {0xFF04, "PRIVATE_CIPHER_4", -1, VALL, 0},
  {0xFF05, "PRIVATE_CIPHER_5", -1, VALL, 0},
  {0xFF06, "PRIVATE_CIPHER_6", -1, VALL, 0},
  {0xFF07, "PRIVATE_CIPHER_7", -1, VALL, 0},
  {0xFF08, "PRIVATE_CIPHER_8", -1, VALL, 0},
  {0xFF09, "PRIVATE_CIPHER_9", -1, VALL, 0},
  {0xFF0A, "PRIVATE_CIPHER_10", -1, VALL, 0},
  {0xFF0B, "PRIVATE_CIPHER_11", -1, VALL, 0},
  {0xFF0C, "PRIVATE_CIPHER_12", -1, VALL, 0},
  {0xFF0D, "PRIVATE_CIPHER_13", -1, VALL, 0},
  {0xFF0E, "PRIVATE_CIPHER_14", -1, VALL, 0},
  {0xFF0F, "PRIVATE_CIPHER_15", -1, VALL, 0},
  {0xFF10, "PRIVATE_CIPHER_16", -1, VALL, 0},
  {0xFF11, "PRIVATE_CIPHER_17", -1, VALL, 0},
  {0xFF12, "PRIVATE_CIPHER_18", -1, VALL, 0},
  {0xFF13, "PRIVATE_CIPHER_19", -1, VALL, 0},
  {0xFF14, "PRIVATE_CIPHER_20", -1, VALL, 0},
  {0xFF15, "PRIVATE_CIPHER_21", -1, VALL, 0},
  {0xFF16, "PRIVATE_CIPHER_22", -1, VALL, 0},
  {0xFF17, "PRIVATE_CIPHER_23", -1, VALL, 0},
  {0xFF18, "PRIVATE_CIPHER_24", -1, VALL, 0},
  {0xFF19, "PRIVATE_CIPHER_25", -1, VALL, 0},
  {0xFF1A, "PRIVATE_CIPHER_26", -1, VALL, 0},
  {0xFF1B, "PRIVATE_CIPHER_27", -1, VALL, 0},
  {0xFF1C, "PRIVATE_CIPHER_28", -1, VALL, 0},
  {0xFF1D, "PRIVATE_CIPHER_29", -1, VALL, 0},
  {0xFF1E, "PRIVATE_CIPHER_30", -1, VALL, 0},
  {0xFF1F, "PRIVATE_CIPHER_31", -1, VALL, 0},
  {0xFF20, "PRIVATE_CIPHER_32", -1, VALL, 0},
  {0xFF21, "PRIVATE_CIPHER_33", -1, VALL, 0},
  {0xFF22, "PRIVATE_CIPHER_34", -1, VALL, 0},
  {0xFF23, "PRIVATE_CIPHER_35", -1, VALL, 0},
  {0xFF24, "PRIVATE_CIPHER_36", -1, VALL, 0},
  {0xFF25, "PRIVATE_CIPHER_37", -1, VALL, 0},
  {0xFF26, "PRIVATE_CIPHER_38", -1, VALL, 0},
  {0xFF27, "PRIVATE_CIPHER_39", -1, VALL, 0},
  {0xFF28, "PRIVATE_CIPHER_40", -1, VALL, 0},
  {0xFF29, "PRIVATE_CIPHER_41", -1, VALL, 0},
  {0xFF2A, "PRIVATE_CIPHER_42", -1, VALL, 0},
  {0xFF2B, "PRIVATE_CIPHER_43", -1, VALL, 0},
  {0xFF2C, "PRIVATE_CIPHER_44", -1, VALL, 0},
  {0xFF2D, "PRIVATE_CIPHER_45", -1, VALL, 0},
  {0xFF2E, "PRIVATE_CIPHER_46", -1, VALL, 0},
  {0xFF2F, "PRIVATE_CIPHER_47", -1, VALL, 0},
  {0xFF30, "PRIVATE_CIPHER_48", -1, VALL, 0},
  {0xFF31, "PRIVATE_CIPHER_49", -1, VALL, 0},
  {0xFF32, "PRIVATE_CIPHER_50", -1, VALL, 0},
  {0xFF33, "PRIVATE_CIPHER_51", -1, VALL, 0},
  {0xFF34, "PRIVATE_CIPHER_52", -1, VALL, 0},
  {0xFF35, "PRIVATE_CIPHER_53", -1, VALL, 0},
  {0xFF36, "PRIVATE_CIPHER_54", -1, VALL, 0},
  {0xFF37, "PRIVATE_CIPHER_55", -1, VALL, 0},
  {0xFF38, "PRIVATE_CIPHER_56", -1, VALL, 0},
  {0xFF39, "PRIVATE_CIPHER_57", -1, VALL, 0},
  {0xFF3A, "PRIVATE_CIPHER_58", -1, VALL, 0},
  {0xFF3B, "PRIVATE_CIPHER_59", -1, VALL, 0},
  {0xFF3C, "PRIVATE_CIPHER_60", -1, VALL, 0},
  {0xFF3D, "PRIVATE_CIPHER_61", -1, VALL, 0},
  {0xFF3E, "PRIVATE_CIPHER_62", -1, VALL, 0},
  {0xFF3F, "PRIVATE_CIPHER_63", -1, VALL, 0},
  {0xFF40, "PRIVATE_CIPHER_64", -1, VALL, 0},
  {0xFF41, "PRIVATE_CIPHER_65", -1, VALL, 0},
  {0xFF42, "PRIVATE_CIPHER_66", -1, VALL, 0},
  {0xFF43, "PRIVATE_CIPHER_67", -1, VALL, 0},
  {0xFF44, "PRIVATE_CIPHER_68", -1, VALL, 0},
  {0xFF45, "PRIVATE_CIPHER_69", -1, VALL, 0},
  {0xFF46, "PRIVATE_CIPHER_70", -1, VALL, 0},
  {0xFF47, "PRIVATE_CIPHER_71", -1, VALL, 0},
  {0xFF48, "PRIVATE_CIPHER_72", -1, VALL, 0},
  {0xFF49, "PRIVATE_CIPHER_73", -1, VALL, 0},
  {0xFF4A, "PRIVATE_CIPHER_74", -1, VALL, 0},
  {0xFF4B, "PRIVATE_CIPHER_75", -1, VALL, 0},
  {0xFF4C, "PRIVATE_CIPHER_76", -1, VALL, 0},
  {0xFF4D, "PRIVATE_CIPHER_77", -1, VALL, 0},
  {0xFF4E, "PRIVATE_CIPHER_78", -1, VALL, 0},
  {0xFF4F, "PRIVATE_CIPHER_79", -1, VALL, 0},
  {0xFF50, "PRIVATE_CIPHER_80", -1, VALL, 0},
  {0xFF51, "PRIVATE_CIPHER_81", -1, VALL, 0},
  {0xFF52, "PRIVATE_CIPHER_82", -1, VALL, 0},
  {0xFF53, "PRIVATE_CIPHER_83", -1, VALL, 0},
  {0xFF54, "PRIVATE_CIPHER_84", -1, VALL, 0},
  {0xFF55, "PRIVATE_CIPHER_85", -1, VALL, 0},
  {0xFF56, "PRIVATE_CIPHER_86", -1, VALL, 0},
  {0xFF57, "PRIVATE_CIPHER_87", -1, VALL, 0},
  {0xFF58, "PRIVATE_CIPHER_88", -1, VALL, 0},
  {0xFF59, "PRIVATE_CIPHER_89", -1, VALL, 0},
  {0xFF5A, "PRIVATE_CIPHER_90", -1, VALL, 0},
  {0xFF5B, "PRIVATE_CIPHER_91", -1, VALL, 0},
  {0xFF5C, "PRIVATE_CIPHER_92", -1, VALL, 0},
  {0xFF5D, "PRIVATE_CIPHER_93", -1, VALL, 0},
  {0xFF5E, "PRIVATE_CIPHER_94", -1, VALL, 0},
  {0xFF5F, "PRIVATE_CIPHER_95", -1, VALL, 0},
  {0xFF60, "PRIVATE_CIPHER_96", -1, VALL, 0},
  {0xFF61, "PRIVATE_CIPHER_97", -1, VALL, 0},
  {0xFF62, "PRIVATE_CIPHER_98", -1, VALL, 0},
  {0xFF63, "PRIVATE_CIPHER_99", -1, VALL, 0},
  {0xFF64, "PRIVATE_CIPHER_100", -1, VALL, 0},
  {0xFF65, "PRIVATE_CIPHER_101", -1, VALL, 0},
  {0xFF66, "PRIVATE_CIPHER_102", -1, VALL, 0},
  {0xFF67, "PRIVATE_CIPHER_103", -1, VALL, 0},
  {0xFF68, "PRIVATE_CIPHER_104", -1, VALL, 0},
  {0xFF69, "PRIVATE_CIPHER_105", -1, VALL, 0},
  {0xFF6A, "PRIVATE_CIPHER_106", -1, VALL, 0},
  {0xFF6B, "PRIVATE_CIPHER_107", -1, VALL, 0},
  {0xFF6C, "PRIVATE_CIPHER_108", -1, VALL, 0},
  {0xFF6D, "PRIVATE_CIPHER_109", -1, VALL, 0},
  {0xFF6E, "PRIVATE_CIPHER_110", -1, VALL, 0},
  {0xFF6F, "PRIVATE_CIPHER_111", -1, VALL, 0},
  {0xFF70, "PRIVATE_CIPHER_112", -1, VALL, 0},
  {0xFF71, "PRIVATE_CIPHER_113", -1, VALL, 0},
  {0xFF72, "PRIVATE_CIPHER_114", -1, VALL, 0},
  {0xFF73, "PRIVATE_CIPHER_115", -1, VALL, 0},
  {0xFF74, "PRIVATE_CIPHER_116", -1, VALL, 0},
  {0xFF75, "PRIVATE_CIPHER_117", -1, VALL, 0},
  {0xFF76, "PRIVATE_CIPHER_118", -1, VALL, 0},
  {0xFF77, "PRIVATE_CIPHER_119", -1, VALL, 0},
  {0xFF78, "PRIVATE_CIPHER_120", -1, VALL, 0},
  {0xFF79, "PRIVATE_CIPHER_121", -1, VALL, 0},
  {0xFF7A, "PRIVATE_CIPHER_122", -1, VALL, 0},
  {0xFF7B, "PRIVATE_CIPHER_123", -1, VALL, 0},
  {0xFF7C, "PRIVATE_CIPHER_124", -1, VALL, 0},
  {0xFF7D, "PRIVATE_CIPHER_125", -1, VALL, 0},
  {0xFF7E, "PRIVATE_CIPHER_126", -1, VALL, 0},
  {0xFF7F, "PRIVATE_CIPHER_127", -1, VALL, 0},
  {0xFF80, "PRIVATE_CIPHER_128", -1, VALL, 0},
  {0xFF81, "PRIVATE_CIPHER_129", -1, VALL, 0},
  {0xFF82, "PRIVATE_CIPHER_130", -1, VALL, 0},
  {0xFF83, "PRIVATE_CIPHER_131", -1, VALL, 0},
  {0xFF84, "PRIVATE_CIPHER_132", -1, VALL, 0},
  {0xFF85, "PRIVATE_CIPHER_133", -1, VALL, 0},
  {0xFF86, "PRIVATE_CIPHER_134", -1, VALL, 0},
  {0xFF87, "PRIVATE_CIPHER_135", -1, VALL, 0},
  {0xFF88, "PRIVATE_CIPHER_136", -1, VALL, 0},
  {0xFF89, "PRIVATE_CIPHER_137", -1, VALL, 0},
  {0xFF8A, "PRIVATE_CIPHER_138", -1, VALL, 0},
  {0xFF8B, "PRIVATE_CIPHER_139", -1, VALL, 0},
  {0xFF8C, "PRIVATE_CIPHER_140", -1, VALL, 0},
  {0xFF8D, "PRIVATE_CIPHER_141", -1, VALL, 0},
  {0xFF8E, "PRIVATE_CIPHER_142", -1, VALL, 0},
  {0xFF8F, "PRIVATE_CIPHER_143", -1, VALL, 0},
  {0xFF90, "PRIVATE_CIPHER_144", -1, VALL, 0},
  {0xFF91, "PRIVATE_CIPHER_145", -1, VALL, 0},
  {0xFF92, "PRIVATE_CIPHER_146", -1, VALL, 0},
  {0xFF93, "PRIVATE_CIPHER_147", -1, VALL, 0},
  {0xFF94, "PRIVATE_CIPHER_148", -1, VALL, 0},
  {0xFF95, "PRIVATE_CIPHER_149", -1, VALL, 0},
  {0xFF96, "PRIVATE_CIPHER_150", -1, VALL, 0},
  {0xFF97, "PRIVATE_CIPHER_151", -1, VALL, 0},
  {0xFF98, "PRIVATE_CIPHER_152", -1, VALL, 0},
  {0xFF99, "PRIVATE_CIPHER_153", -1, VALL, 0},
  {0xFF9A, "PRIVATE_CIPHER_154", -1, VALL, 0},
  {0xFF9B, "PRIVATE_CIPHER_155", -1, VALL, 0},
  {0xFF9C, "PRIVATE_CIPHER_156", -1, VALL, 0},
  {0xFF9D, "PRIVATE_CIPHER_157", -1, VALL, 0},
  {0xFF9E, "PRIVATE_CIPHER_158", -1, VALL, 0},
  {0xFF9F, "PRIVATE_CIPHER_159", -1, VALL, 0},
  {0xFFA0, "PRIVATE_CIPHER_160", -1, VALL, 0},
  {0xFFA1, "PRIVATE_CIPHER_161", -1, VALL, 0},
  {0xFFA2, "PRIVATE_CIPHER_162", -1, VALL, 0},
  {0xFFA3, "PRIVATE_CIPHER_163", -1, VALL, 0},
  {0xFFA4, "PRIVATE_CIPHER_164", -1, VALL, 0},
  {0xFFA5, "PRIVATE_CIPHER_165", -1, VALL, 0},
  {0xFFA6, "PRIVATE_CIPHER_166", -1, VALL, 0},
  {0xFFA7, "PRIVATE_CIPHER_167", -1, VALL, 0},
  {0xFFA8, "PRIVATE_CIPHER_168", -1, VALL, 0},
  {0xFFA9, "PRIVATE_CIPHER_169", -1, VALL, 0},
  {0xFFAA, "PRIVATE_CIPHER_170", -1, VALL, 0},
  {0xFFAB, "PRIVATE_CIPHER_171", -1, VALL, 0},
  {0xFFAC, "PRIVATE_CIPHER_172", -1, VALL, 0},
  {0xFFAD, "PRIVATE_CIPHER_173", -1, VALL, 0},
  {0xFFAE, "PRIVATE_CIPHER_174", -1, VALL, 0},
  {0xFFAF, "PRIVATE_CIPHER_175", -1, VALL, 0},
  {0xFFB0, "PRIVATE_CIPHER_176", -1, VALL, 0},
  {0xFFB1, "PRIVATE_CIPHER_177", -1, VALL, 0},
  {0xFFB2, "PRIVATE_CIPHER_178", -1, VALL, 0},
  {0xFFB3, "PRIVATE_CIPHER_179", -1, VALL, 0},
  {0xFFB4, "PRIVATE_CIPHER_180", -1, VALL, 0},
  {0xFFB5, "PRIVATE_CIPHER_181", -1, VALL, 0},
  {0xFFB6, "PRIVATE_CIPHER_182", -1, VALL, 0},
  {0xFFB7, "PRIVATE_CIPHER_183", -1, VALL, 0},
  {0xFFB8, "PRIVATE_CIPHER_184", -1, VALL, 0},
  {0xFFB9, "PRIVATE_CIPHER_185", -1, VALL, 0},
  {0xFFBA, "PRIVATE_CIPHER_186", -1, VALL, 0},
  {0xFFBB, "PRIVATE_CIPHER_187", -1, VALL, 0},
  {0xFFBC, "PRIVATE_CIPHER_188", -1, VALL, 0},
  {0xFFBD, "PRIVATE_CIPHER_189", -1, VALL, 0},
  {0xFFBE, "PRIVATE_CIPHER_190", -1, VALL, 0},
  {0xFFBF, "PRIVATE_CIPHER_191", -1, VALL, 0},
  {0xFFC0, "PRIVATE_CIPHER_192", -1, VALL, 0},
  {0xFFC1, "PRIVATE_CIPHER_193", -1, VALL, 0},
  {0xFFC2, "PRIVATE_CIPHER_194", -1, VALL, 0},
  {0xFFC3, "PRIVATE_CIPHER_195", -1, VALL, 0},
  {0xFFC4, "PRIVATE_CIPHER_196", -1, VALL, 0},
  {0xFFC5, "PRIVATE_CIPHER_197", -1, VALL, 0},
  {0xFFC6, "PRIVATE_CIPHER_198", -1, VALL, 0},
  {0xFFC7, "PRIVATE_CIPHER_199", -1, VALL, 0},
  {0xFFC8, "PRIVATE_CIPHER_200", -1, VALL, 0},
  {0xFFC9, "PRIVATE_CIPHER_201", -1, VALL, 0},
  {0xFFCA, "PRIVATE_CIPHER_202", -1, VALL, 0},
  {0xFFCB, "PRIVATE_CIPHER_203", -1, VALL, 0},
  {0xFFCC, "PRIVATE_CIPHER_204", -1, VALL, 0},
  {0xFFCD, "PRIVATE_CIPHER_205", -1, VALL, 0},
  {0xFFCE, "PRIVATE_CIPHER_206", -1, VALL, 0},
  {0xFFCF, "PRIVATE_CIPHER_207", -1, VALL, 0},
  {0xFFD0, "PRIVATE_CIPHER_208", -1, VALL, 0},
  {0xFFD1, "PRIVATE_CIPHER_209", -1, VALL, 0},
  {0xFFD2, "PRIVATE_CIPHER_210", -1, VALL, 0},
  {0xFFD3, "PRIVATE_CIPHER_211", -1, VALL, 0},
  {0xFFD4, "PRIVATE_CIPHER_212", -1, VALL, 0},
  {0xFFD5, "PRIVATE_CIPHER_213", -1, VALL, 0},
  {0xFFD6, "PRIVATE_CIPHER_214", -1, VALL, 0},
  {0xFFD7, "PRIVATE_CIPHER_215", -1, VALL, 0},
  {0xFFD8, "PRIVATE_CIPHER_216", -1, VALL, 0},
  {0xFFD9, "PRIVATE_CIPHER_217", -1, VALL, 0},
  {0xFFDA, "PRIVATE_CIPHER_218", -1, VALL, 0},
  {0xFFDB, "PRIVATE_CIPHER_219", -1, VALL, 0},
  {0xFFDC, "PRIVATE_CIPHER_220", -1, VALL, 0},
  {0xFFDD, "PRIVATE_CIPHER_221", -1, VALL, 0},
  {0xFFDE, "PRIVATE_CIPHER_222", -1, VALL, 0},
  {0xFFDF, "PRIVATE_CIPHER_223", -1, VALL, 0},
  {0xFFE0, "PRIVATE_CIPHER_224", -1, VALL, 0},
  {0xFFE1, "PRIVATE_CIPHER_225", -1, VALL, 0},
  {0xFFE2, "PRIVATE_CIPHER_226", -1, VALL, 0},
  {0xFFE3, "PRIVATE_CIPHER_227", -1, VALL, 0},
  {0xFFE4, "PRIVATE_CIPHER_228", -1, VALL, 0},
  {0xFFE5, "PRIVATE_CIPHER_229", -1, VALL, 0},
  {0xFFE6, "PRIVATE_CIPHER_230", -1, VALL, 0},
  {0xFFE7, "PRIVATE_CIPHER_231", -1, VALL, 0},
  {0xFFE8, "PRIVATE_CIPHER_232", -1, VALL, 0},
  {0xFFE9, "PRIVATE_CIPHER_233", -1, VALL, 0},
  {0xFFEA, "PRIVATE_CIPHER_234", -1, VALL, 0},
  {0xFFEB, "PRIVATE_CIPHER_235", -1, VALL, 0},
  {0xFFEC, "PRIVATE_CIPHER_236", -1, VALL, 0},
  {0xFFED, "PRIVATE_CIPHER_237", -1, VALL, 0},
  {0xFFEE, "PRIVATE_CIPHER_238", -1, VALL, 0},
  {0xFFEF, "PRIVATE_CIPHER_239", -1, VALL, 0},
  {0xFFF0, "PRIVATE_CIPHER_240", -1, VALL, 0},
  {0xFFF1, "PRIVATE_CIPHER_241", -1, VALL, 0},
  {0xFFF2, "PRIVATE_CIPHER_242", -1, VALL, 0},
  {0xFFF3, "PRIVATE_CIPHER_243", -1, VALL, 0},
  {0xFFF4, "PRIVATE_CIPHER_244", -1, VALL, 0},
  {0xFFF5, "PRIVATE_CIPHER_245", -1, VALL, 0},
  {0xFFF6, "PRIVATE_CIPHER_246", -1, VALL, 0},
  {0xFFF7, "PRIVATE_CIPHER_247", -1, VALL, 0},
  {0xFFF8, "PRIVATE_CIPHER_248", -1, VALL, 0},
  {0xFFF9, "PRIVATE_CIPHER_249", -1, VALL, 0},
  {0xFFFA, "PRIVATE_CIPHER_250", -1, VALL, 0},
  {0xFFFB, "PRIVATE_CIPHER_251", -1, VALL, 0},
  {0xFFFC, "PRIVATE_CIPHER_252", -1, VALL, 0},
  {0xFFFD, "PRIVATE_CIPHER_253", -1, VALL, 0},
  {0xFFFE, "PRIVATE_CIPHER_254", -1, VALL, 0},
  {0xFFFF, "PRIVATE_CIPHER_255", -1, VALL, 0},
};

#endif
