/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 */

#define SIMDE_TEST_X86_AVX512_INSN setr

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/setr.h>

static int
test_simde_mm512_setr_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    int32_t e15; int32_t e14; int32_t e13; int32_t e12; int32_t e11; int32_t e10; int32_t  e9; int32_t  e8;
    int32_t  e7; int32_t  e6; int32_t  e5; int32_t  e4; int32_t  e3; int32_t  e2; int32_t  e1; int32_t  e0;
    simde__m512i r;
  } test_vec[8] = {
    { INT32_C( -225639004),
      INT32_C( -150299519),
      INT32_C(-1534790107),
      INT32_C( 1014269333),
      INT32_C(   29394004),
      INT32_C(  599047736),
      INT32_C( 1587483242),
      INT32_C(-1377286712),
      INT32_C(  298403451),
      INT32_C( -826427815),
      INT32_C(  694491833),
      INT32_C( -705464027),
      INT32_C( -970263706),
      INT32_C( -734607127),
      INT32_C(  554049123),
      INT32_C(  -70690832),
      simde_mm512_set_epi32(INT32_C(  -70690832), INT32_C(  554049123), INT32_C( -734607127), INT32_C( -970263706),
                            INT32_C( -705464027), INT32_C(  694491833), INT32_C( -826427815), INT32_C(  298403451),
                            INT32_C(-1377286712), INT32_C( 1587483242), INT32_C(  599047736), INT32_C(   29394004),
                            INT32_C( 1014269333), INT32_C(-1534790107), INT32_C( -150299519), INT32_C( -225639004)) },
    { INT32_C(  -94411288),
      INT32_C( -666296442),
      INT32_C( -469504489),
      INT32_C( 1755610261),
      INT32_C( -444648579),
      INT32_C(  539810698),
      INT32_C(-1045838623),
      INT32_C( 1780875011),
      INT32_C(-1180587774),
      INT32_C( 1720505332),
      INT32_C( -289874239),
      INT32_C(-1063749100),
      INT32_C(-1248309679),
      INT32_C(-1329997639),
      INT32_C(  575866898),
      INT32_C( -286711209),
      simde_mm512_set_epi32(INT32_C( -286711209), INT32_C(  575866898), INT32_C(-1329997639), INT32_C(-1248309679),
                            INT32_C(-1063749100), INT32_C( -289874239), INT32_C( 1720505332), INT32_C(-1180587774),
                            INT32_C( 1780875011), INT32_C(-1045838623), INT32_C(  539810698), INT32_C( -444648579),
                            INT32_C( 1755610261), INT32_C( -469504489), INT32_C( -666296442), INT32_C(  -94411288)) },
    { INT32_C( -592672865),
      INT32_C(-1173697634),
      INT32_C(-2138963688),
      INT32_C( 1137021966),
      INT32_C( -277367531),
      INT32_C( 1535015576),
      INT32_C( -300840189),
      INT32_C(  970793246),
      INT32_C( 2002409399),
      INT32_C(-1909422707),
      INT32_C( 1060757649),
      INT32_C(  187969074),
      INT32_C(-1128231220),
      INT32_C(-1406585232),
      INT32_C(  325181376),
      INT32_C(-1609780510),
      simde_mm512_set_epi32(INT32_C(-1609780510), INT32_C(  325181376), INT32_C(-1406585232), INT32_C(-1128231220),
                            INT32_C(  187969074), INT32_C( 1060757649), INT32_C(-1909422707), INT32_C( 2002409399),
                            INT32_C(  970793246), INT32_C( -300840189), INT32_C( 1535015576), INT32_C( -277367531),
                            INT32_C( 1137021966), INT32_C(-2138963688), INT32_C(-1173697634), INT32_C( -592672865)) },
    { INT32_C( 1559684973),
      INT32_C( -969314800),
      INT32_C(-1209891452),
      INT32_C( 1346728079),
      INT32_C( 1672631336),
      INT32_C(-1432148577),
      INT32_C(-2062863806),
      INT32_C(   47399907),
      INT32_C( 1393917193),
      INT32_C( 2115773956),
      INT32_C( -279257842),
      INT32_C( 1954534792),
      INT32_C( 2098673509),
      INT32_C(-1488841900),
      INT32_C(-1097418668),
      INT32_C(  924984498),
      simde_mm512_set_epi32(INT32_C(  924984498), INT32_C(-1097418668), INT32_C(-1488841900), INT32_C( 2098673509),
                            INT32_C( 1954534792), INT32_C( -279257842), INT32_C( 2115773956), INT32_C( 1393917193),
                            INT32_C(   47399907), INT32_C(-2062863806), INT32_C(-1432148577), INT32_C( 1672631336),
                            INT32_C( 1346728079), INT32_C(-1209891452), INT32_C( -969314800), INT32_C( 1559684973)) },
    { INT32_C(  168008623),
      INT32_C( -453715155),
      INT32_C(-1391419453),
      INT32_C(  -49732507),
      INT32_C(-1285867712),
      INT32_C(-1989456497),
      INT32_C( -747356949),
      INT32_C(  490906758),
      INT32_C( 1610110312),
      INT32_C( 1643820847),
      INT32_C( -516397675),
      INT32_C(  100321539),
      INT32_C( 1161753679),
      INT32_C( 1545726171),
      INT32_C(  793769069),
      INT32_C( 1644133252),
      simde_mm512_set_epi32(INT32_C( 1644133252), INT32_C(  793769069), INT32_C( 1545726171), INT32_C( 1161753679),
                            INT32_C(  100321539), INT32_C( -516397675), INT32_C( 1643820847), INT32_C( 1610110312),
                            INT32_C(  490906758), INT32_C( -747356949), INT32_C(-1989456497), INT32_C(-1285867712),
                            INT32_C(  -49732507), INT32_C(-1391419453), INT32_C( -453715155), INT32_C(  168008623)) },
    { INT32_C(-1778931889),
      INT32_C( -605375026),
      INT32_C( 1673050617),
      INT32_C( -625199109),
      INT32_C(  -96995330),
      INT32_C(-1228008621),
      INT32_C(-2102854258),
      INT32_C(  134984452),
      INT32_C(  310905618),
      INT32_C(-2014521010),
      INT32_C(-1089463947),
      INT32_C(-2013872078),
      INT32_C(  801481447),
      INT32_C(  737497431),
      INT32_C( 1618614949),
      INT32_C(-1017834831),
      simde_mm512_set_epi32(INT32_C(-1017834831), INT32_C( 1618614949), INT32_C(  737497431), INT32_C(  801481447),
                            INT32_C(-2013872078), INT32_C(-1089463947), INT32_C(-2014521010), INT32_C(  310905618),
                            INT32_C(  134984452), INT32_C(-2102854258), INT32_C(-1228008621), INT32_C(  -96995330),
                            INT32_C( -625199109), INT32_C( 1673050617), INT32_C( -605375026), INT32_C(-1778931889)) },
    { INT32_C(  -93411582),
      INT32_C(-2041831255),
      INT32_C(  -37603949),
      INT32_C(-1753490957),
      INT32_C( -415214737),
      INT32_C(  910720124),
      INT32_C(-1202548944),
      INT32_C( -274420795),
      INT32_C( 1822689778),
      INT32_C(  894388184),
      INT32_C(-1563709444),
      INT32_C(   53989264),
      INT32_C(  680272844),
      INT32_C( -566635982),
      INT32_C( 1666979976),
      INT32_C(-1823765080),
      simde_mm512_set_epi32(INT32_C(-1823765080), INT32_C( 1666979976), INT32_C( -566635982), INT32_C(  680272844),
                            INT32_C(   53989264), INT32_C(-1563709444), INT32_C(  894388184), INT32_C( 1822689778),
                            INT32_C( -274420795), INT32_C(-1202548944), INT32_C(  910720124), INT32_C( -415214737),
                            INT32_C(-1753490957), INT32_C(  -37603949), INT32_C(-2041831255), INT32_C(  -93411582)) },
    { INT32_C(-1594363360),
      INT32_C( 1289095896),
      INT32_C( 1158627349),
      INT32_C( -895971493),
      INT32_C( -373133234),
      INT32_C(-1633749559),
      INT32_C( 1579262744),
      INT32_C( -827722334),
      INT32_C( -839338980),
      INT32_C( 1359699192),
      INT32_C( -211246622),
      INT32_C(-1308719740),
      INT32_C( -442822787),
      INT32_C(  253799860),
      INT32_C(-1608354611),
      INT32_C(  -49457301),
      simde_mm512_set_epi32(INT32_C(  -49457301), INT32_C(-1608354611), INT32_C(  253799860), INT32_C( -442822787),
                            INT32_C(-1308719740), INT32_C( -211246622), INT32_C( 1359699192), INT32_C( -839338980),
                            INT32_C( -827722334), INT32_C( 1579262744), INT32_C(-1633749559), INT32_C( -373133234),
                            INT32_C( -895971493), INT32_C( 1158627349), INT32_C( 1289095896), INT32_C(-1594363360)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_setr_epi32( test_vec[i].e15, test_vec[i].e14, test_vec[i].e13, test_vec[i].e12, test_vec[i].e11, test_vec[i].e10, test_vec[i].e9, test_vec[i].e8,
                                             test_vec[i].e7, test_vec[i].e6, test_vec[i].e5, test_vec[i].e4, test_vec[i].e3, test_vec[i].e2, test_vec[i].e1, test_vec[i].e0);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_setr_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
     int64_t e7; int64_t e6; int64_t e5; int64_t e4; int64_t e3; int64_t e2; int64_t e1; int64_t e0;
    simde__m512i r;
  } test_vec[8] = {
     { INT64_C( 1888907496602700549),
      INT64_C(-4376226703742367928),
      INT64_C( 1319808302306308723),
      INT64_C(-8416436825139627772),
      INT64_C(-7030268320569234373),
      INT64_C( 4053306656518847875),
      INT64_C(-7288565613940945788),
      INT64_C( 3412457605701455282),
      simde_mm512_set_epi64(INT64_C( 3412457605701455282), INT64_C(-7288565613940945788),
                            INT64_C( 4053306656518847875), INT64_C(-7030268320569234373),
                            INT64_C(-8416436825139627772), INT64_C( 1319808302306308723),
                            INT64_C(-4376226703742367928), INT64_C( 1888907496602700549)) },
    { INT64_C(-9149987363116605946),
      INT64_C(-8568066162437210648),
      INT64_C( 5724154895417122611),
      INT64_C( 3016754579022165386),
      INT64_C(-6067588729873055686),
      INT64_C( 2536453724300579147),
      INT64_C( 4833323501533385465),
      INT64_C(-7273821104851084568),
      simde_mm512_set_epi64(INT64_C(-7273821104851084568), INT64_C( 4833323501533385465),
                            INT64_C( 2536453724300579147), INT64_C(-6067588729873055686),
                            INT64_C( 3016754579022165386), INT64_C( 5724154895417122611),
                            INT64_C(-8568066162437210648), INT64_C(-9149987363116605946)) },
    { INT64_C( 8826953524808214711),
      INT64_C( 3028688385033096274),
      INT64_C( 7940008473030673245),
      INT64_C(-3386602331089063366),
      INT64_C(-5551104753805792180),
      INT64_C( 8877895432400251040),
      INT64_C( 3968860811702288437),
      INT64_C(-8457518168881001349),
      simde_mm512_set_epi64(INT64_C(-8457518168881001349), INT64_C( 3968860811702288437),
                            INT64_C( 8877895432400251040), INT64_C(-5551104753805792180),
                            INT64_C(-3386602331089063366), INT64_C( 7940008473030673245),
                            INT64_C( 3028688385033096274), INT64_C( 8826953524808214711)) },
    { INT64_C(-1749629734112662134),
      INT64_C(-4615379630913236957),
      INT64_C(-1845654885783324819),
      INT64_C(-4590913868344348334),
      INT64_C(-8403233988575943004),
      INT64_C(-2056952412471198236),
      INT64_C(-2511875144579280083),
      INT64_C(-8623043688818523441),
      simde_mm512_set_epi64(INT64_C(-8623043688818523441), INT64_C(-2511875144579280083),
                            INT64_C(-2056952412471198236), INT64_C(-8403233988575943004),
                            INT64_C(-4590913868344348334), INT64_C(-1845654885783324819),
                            INT64_C(-4615379630913236957), INT64_C(-1749629734112662134)) },
    { INT64_C( 1761036457964785466),
      INT64_C(-7054972876528755116),
      INT64_C(-7738430230159702471),
      INT64_C( 5826683286381985571),
      INT64_C(-4997893423121397361),
      INT64_C(   95646196860293964),
      INT64_C( -268412225668008681),
      INT64_C(-4388731883907908541),
      simde_mm512_set_epi64(INT64_C(-4388731883907908541), INT64_C( -268412225668008681),
                            INT64_C(   95646196860293964), INT64_C(-4997893423121397361),
                            INT64_C( 5826683286381985571), INT64_C(-7738430230159702471),
                            INT64_C(-7054972876528755116), INT64_C( 1761036457964785466)) },
    { INT64_C(-1634122079602729117),
      INT64_C( -197548662778062026),
      INT64_C( 8430179558397234507),
      INT64_C(-6847424589158690122),
      INT64_C(  108510191489750517),
      INT64_C(-5960504929867173559),
      INT64_C( 7692198613837645958),
      INT64_C(-5086195662698067529),
      simde_mm512_set_epi64(INT64_C(-5086195662698067529), INT64_C( 7692198613837645958),
                            INT64_C(-5960504929867173559), INT64_C(  108510191489750517),
                            INT64_C(-6847424589158690122), INT64_C( 8430179558397234507),
                            INT64_C( -197548662778062026), INT64_C(-1634122079602729117)) },
    { INT64_C(-2013722991297495958),
      INT64_C( -331332851212596030),
      INT64_C(-7644478161857365687),
      INT64_C(-5200258926547250644),
      INT64_C(-7000971193485373522),
      INT64_C( 3470026623990179641),
      INT64_C(-5506978056254809529),
      INT64_C( 5213166683745714657),
      simde_mm512_set_epi64(INT64_C( 5213166683745714657), INT64_C(-5506978056254809529),
                            INT64_C( 3470026623990179641), INT64_C(-7000971193485373522),
                            INT64_C(-5200258926547250644), INT64_C(-7644478161857365687),
                            INT64_C( -331332851212596030), INT64_C(-2013722991297495958)) },
    { INT64_C(-7035344274950827789),
      INT64_C( 6421674353147720323),
      INT64_C( 7861915403667476835),
      INT64_C(-7744076081560701528),
      INT64_C(-5000728962608243188),
      INT64_C(-2643855462940399334),
      INT64_C( -358888979465373416),
      INT64_C( 5439855456779249715),
      simde_mm512_set_epi64(INT64_C( 5439855456779249715), INT64_C( -358888979465373416),
                            INT64_C(-2643855462940399334), INT64_C(-5000728962608243188),
                            INT64_C(-7744076081560701528), INT64_C( 7861915403667476835),
                            INT64_C( 6421674353147720323), INT64_C(-7035344274950827789)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_setr_epi64( test_vec[i].e7, test_vec[i].e6, test_vec[i].e5, test_vec[i].e4, test_vec[i].e3, test_vec[i].e2, test_vec[i].e1, test_vec[i].e0);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_setr_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde_float32 e15; simde_float32 e14; simde_float32 e13; simde_float32 e12;
    simde_float32 e11; simde_float32 e10; simde_float32  e9; simde_float32  e8;
    simde_float32  e7; simde_float32  e6; simde_float32  e5; simde_float32  e4;
    simde_float32  e3; simde_float32  e2; simde_float32  e1; simde_float32  e0;
    simde__m512 r;
  } test_vec[8] = {
    { SIMDE_FLOAT32_C(  -693.34),
      SIMDE_FLOAT32_C(  -391.55),
      SIMDE_FLOAT32_C(   340.77),
      SIMDE_FLOAT32_C(   396.59),
      SIMDE_FLOAT32_C(   776.91),
      SIMDE_FLOAT32_C(   317.46),
      SIMDE_FLOAT32_C(  -394.47),
      SIMDE_FLOAT32_C(   664.34),
      SIMDE_FLOAT32_C(   353.81),
      SIMDE_FLOAT32_C(  -365.45),
      SIMDE_FLOAT32_C(   141.75),
      SIMDE_FLOAT32_C(   338.26),
      SIMDE_FLOAT32_C(  -898.63),
      SIMDE_FLOAT32_C(  -717.56),
      SIMDE_FLOAT32_C(  -763.21),
      SIMDE_FLOAT32_C(  -594.53),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -594.53), SIMDE_FLOAT32_C(  -763.21), SIMDE_FLOAT32_C(  -717.56), SIMDE_FLOAT32_C(  -898.63),
                         SIMDE_FLOAT32_C(   338.26), SIMDE_FLOAT32_C(   141.75), SIMDE_FLOAT32_C(  -365.45), SIMDE_FLOAT32_C(   353.81),
                         SIMDE_FLOAT32_C(   664.34), SIMDE_FLOAT32_C(  -394.47), SIMDE_FLOAT32_C(   317.46), SIMDE_FLOAT32_C(   776.91),
                         SIMDE_FLOAT32_C(   396.59), SIMDE_FLOAT32_C(   340.77), SIMDE_FLOAT32_C(  -391.55), SIMDE_FLOAT32_C(  -693.34)) },
    { SIMDE_FLOAT32_C(  -212.87),
      SIMDE_FLOAT32_C(  -107.93),
      SIMDE_FLOAT32_C(   395.61),
      SIMDE_FLOAT32_C(   321.82),
      SIMDE_FLOAT32_C(   802.07),
      SIMDE_FLOAT32_C(  -157.82),
      SIMDE_FLOAT32_C(   306.15),
      SIMDE_FLOAT32_C(   501.93),
      SIMDE_FLOAT32_C(  -150.29),
      SIMDE_FLOAT32_C(  -247.68),
      SIMDE_FLOAT32_C(   472.68),
      SIMDE_FLOAT32_C(   263.49),
      SIMDE_FLOAT32_C(   775.84),
      SIMDE_FLOAT32_C(   573.19),
      SIMDE_FLOAT32_C(  -361.82),
      SIMDE_FLOAT32_C(   173.65),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   173.65), SIMDE_FLOAT32_C(  -361.82), SIMDE_FLOAT32_C(   573.19), SIMDE_FLOAT32_C(   775.84),
                         SIMDE_FLOAT32_C(   263.49), SIMDE_FLOAT32_C(   472.68), SIMDE_FLOAT32_C(  -247.68), SIMDE_FLOAT32_C(  -150.29),
                         SIMDE_FLOAT32_C(   501.93), SIMDE_FLOAT32_C(   306.15), SIMDE_FLOAT32_C(  -157.82), SIMDE_FLOAT32_C(   802.07),
                         SIMDE_FLOAT32_C(   321.82), SIMDE_FLOAT32_C(   395.61), SIMDE_FLOAT32_C(  -107.93), SIMDE_FLOAT32_C(  -212.87)) },
    { SIMDE_FLOAT32_C(    44.50),
      SIMDE_FLOAT32_C(  -431.57),
      SIMDE_FLOAT32_C(   308.32),
      SIMDE_FLOAT32_C(  -958.90),
      SIMDE_FLOAT32_C(  -174.46),
      SIMDE_FLOAT32_C(  -298.65),
      SIMDE_FLOAT32_C(   209.15),
      SIMDE_FLOAT32_C(   439.10),
      SIMDE_FLOAT32_C(  -926.44),
      SIMDE_FLOAT32_C(  -768.87),
      SIMDE_FLOAT32_C(   291.37),
      SIMDE_FLOAT32_C(   727.05),
      SIMDE_FLOAT32_C(  -573.28),
      SIMDE_FLOAT32_C(    36.75),
      SIMDE_FLOAT32_C(  -302.25),
      SIMDE_FLOAT32_C(   571.00),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   571.00), SIMDE_FLOAT32_C(  -302.25), SIMDE_FLOAT32_C(    36.75), SIMDE_FLOAT32_C(  -573.28),
                         SIMDE_FLOAT32_C(   727.05), SIMDE_FLOAT32_C(   291.37), SIMDE_FLOAT32_C(  -768.87), SIMDE_FLOAT32_C(  -926.44),
                         SIMDE_FLOAT32_C(   439.10), SIMDE_FLOAT32_C(   209.15), SIMDE_FLOAT32_C(  -298.65), SIMDE_FLOAT32_C(  -174.46),
                         SIMDE_FLOAT32_C(  -958.90), SIMDE_FLOAT32_C(   308.32), SIMDE_FLOAT32_C(  -431.57), SIMDE_FLOAT32_C(    44.50)) },
    { SIMDE_FLOAT32_C(     7.49),
      SIMDE_FLOAT32_C(  -164.63),
      SIMDE_FLOAT32_C(   120.97),
      SIMDE_FLOAT32_C(  -584.72),
      SIMDE_FLOAT32_C(  -840.08),
      SIMDE_FLOAT32_C(  -642.99),
      SIMDE_FLOAT32_C(  -632.99),
      SIMDE_FLOAT32_C(   911.46),
      SIMDE_FLOAT32_C(   708.81),
      SIMDE_FLOAT32_C(   815.55),
      SIMDE_FLOAT32_C(   163.40),
      SIMDE_FLOAT32_C(    42.16),
      SIMDE_FLOAT32_C(  -604.25),
      SIMDE_FLOAT32_C(  -667.70),
      SIMDE_FLOAT32_C(  -123.35),
      SIMDE_FLOAT32_C(  -145.23),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -145.23), SIMDE_FLOAT32_C(  -123.35), SIMDE_FLOAT32_C(  -667.70), SIMDE_FLOAT32_C(  -604.25),
                         SIMDE_FLOAT32_C(    42.16), SIMDE_FLOAT32_C(   163.40), SIMDE_FLOAT32_C(   815.55), SIMDE_FLOAT32_C(   708.81),
                         SIMDE_FLOAT32_C(   911.46), SIMDE_FLOAT32_C(  -632.99), SIMDE_FLOAT32_C(  -642.99), SIMDE_FLOAT32_C(  -840.08),
                         SIMDE_FLOAT32_C(  -584.72), SIMDE_FLOAT32_C(   120.97), SIMDE_FLOAT32_C(  -164.63), SIMDE_FLOAT32_C(     7.49)) },
    { SIMDE_FLOAT32_C(   363.26),
      SIMDE_FLOAT32_C(   849.07),
      SIMDE_FLOAT32_C(   296.23),
      SIMDE_FLOAT32_C(  -961.65),
      SIMDE_FLOAT32_C(  -607.94),
      SIMDE_FLOAT32_C(  -947.07),
      SIMDE_FLOAT32_C(     8.40),
      SIMDE_FLOAT32_C(  -646.90),
      SIMDE_FLOAT32_C(   -23.65),
      SIMDE_FLOAT32_C(  -317.45),
      SIMDE_FLOAT32_C(  -389.74),
      SIMDE_FLOAT32_C(   -24.49),
      SIMDE_FLOAT32_C(  -609.72),
      SIMDE_FLOAT32_C(   -45.92),
      SIMDE_FLOAT32_C(    85.01),
      SIMDE_FLOAT32_C(   775.08),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   775.08), SIMDE_FLOAT32_C(    85.01), SIMDE_FLOAT32_C(   -45.92), SIMDE_FLOAT32_C(  -609.72),
                         SIMDE_FLOAT32_C(   -24.49), SIMDE_FLOAT32_C(  -389.74), SIMDE_FLOAT32_C(  -317.45), SIMDE_FLOAT32_C(   -23.65),
                         SIMDE_FLOAT32_C(  -646.90), SIMDE_FLOAT32_C(     8.40), SIMDE_FLOAT32_C(  -947.07), SIMDE_FLOAT32_C(  -607.94),
                         SIMDE_FLOAT32_C(  -961.65), SIMDE_FLOAT32_C(   296.23), SIMDE_FLOAT32_C(   849.07), SIMDE_FLOAT32_C(   363.26)) },
    { SIMDE_FLOAT32_C(  -743.67),
      SIMDE_FLOAT32_C(   584.63),
      SIMDE_FLOAT32_C(   124.10),
      SIMDE_FLOAT32_C(    44.15),
      SIMDE_FLOAT32_C(    97.80),
      SIMDE_FLOAT32_C(  -183.84),
      SIMDE_FLOAT32_C(   775.52),
      SIMDE_FLOAT32_C(  -276.30),
      SIMDE_FLOAT32_C(   521.23),
      SIMDE_FLOAT32_C(  -285.93),
      SIMDE_FLOAT32_C(   127.02),
      SIMDE_FLOAT32_C(   104.60),
      SIMDE_FLOAT32_C(   881.59),
      SIMDE_FLOAT32_C(   934.75),
      SIMDE_FLOAT32_C(  -126.94),
      SIMDE_FLOAT32_C(   948.78),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   948.78), SIMDE_FLOAT32_C(  -126.94), SIMDE_FLOAT32_C(   934.75), SIMDE_FLOAT32_C(   881.59),
                         SIMDE_FLOAT32_C(   104.60), SIMDE_FLOAT32_C(   127.02), SIMDE_FLOAT32_C(  -285.93), SIMDE_FLOAT32_C(   521.23),
                         SIMDE_FLOAT32_C(  -276.30), SIMDE_FLOAT32_C(   775.52), SIMDE_FLOAT32_C(  -183.84), SIMDE_FLOAT32_C(    97.80),
                         SIMDE_FLOAT32_C(    44.15), SIMDE_FLOAT32_C(   124.10), SIMDE_FLOAT32_C(   584.63), SIMDE_FLOAT32_C(  -743.67)) },
    { SIMDE_FLOAT32_C(   385.42),
      SIMDE_FLOAT32_C(    69.88),
      SIMDE_FLOAT32_C(   900.97),
      SIMDE_FLOAT32_C(  -107.25),
      SIMDE_FLOAT32_C(  -574.36),
      SIMDE_FLOAT32_C(   748.65),
      SIMDE_FLOAT32_C(  -126.70),
      SIMDE_FLOAT32_C(  -459.43),
      SIMDE_FLOAT32_C(  -231.22),
      SIMDE_FLOAT32_C(   815.77),
      SIMDE_FLOAT32_C(  -476.98),
      SIMDE_FLOAT32_C(   236.36),
      SIMDE_FLOAT32_C(  -834.38),
      SIMDE_FLOAT32_C(  -511.69),
      SIMDE_FLOAT32_C(  -443.09),
      SIMDE_FLOAT32_C(  -508.39),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -508.39), SIMDE_FLOAT32_C(  -443.09), SIMDE_FLOAT32_C(  -511.69), SIMDE_FLOAT32_C(  -834.38),
                         SIMDE_FLOAT32_C(   236.36), SIMDE_FLOAT32_C(  -476.98), SIMDE_FLOAT32_C(   815.77), SIMDE_FLOAT32_C(  -231.22),
                         SIMDE_FLOAT32_C(  -459.43), SIMDE_FLOAT32_C(  -126.70), SIMDE_FLOAT32_C(   748.65), SIMDE_FLOAT32_C(  -574.36),
                         SIMDE_FLOAT32_C(  -107.25), SIMDE_FLOAT32_C(   900.97), SIMDE_FLOAT32_C(    69.88), SIMDE_FLOAT32_C(   385.42)) },
    { SIMDE_FLOAT32_C(  -878.65),
      SIMDE_FLOAT32_C(  -787.20),
      SIMDE_FLOAT32_C(  -378.35),
      SIMDE_FLOAT32_C(  -406.07),
      SIMDE_FLOAT32_C(   822.02),
      SIMDE_FLOAT32_C(  -574.07),
      SIMDE_FLOAT32_C(   229.39),
      SIMDE_FLOAT32_C(    -3.08),
      SIMDE_FLOAT32_C(  -878.41),
      SIMDE_FLOAT32_C(   -80.74),
      SIMDE_FLOAT32_C(   518.86),
      SIMDE_FLOAT32_C(  -831.41),
      SIMDE_FLOAT32_C(  -639.09),
      SIMDE_FLOAT32_C(   136.23),
      SIMDE_FLOAT32_C(  -279.71),
      SIMDE_FLOAT32_C(  -447.29),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -447.29), SIMDE_FLOAT32_C(  -279.71), SIMDE_FLOAT32_C(   136.23), SIMDE_FLOAT32_C(  -639.09),
                         SIMDE_FLOAT32_C(  -831.41), SIMDE_FLOAT32_C(   518.86), SIMDE_FLOAT32_C(   -80.74), SIMDE_FLOAT32_C(  -878.41),
                         SIMDE_FLOAT32_C(    -3.08), SIMDE_FLOAT32_C(   229.39), SIMDE_FLOAT32_C(  -574.07), SIMDE_FLOAT32_C(   822.02),
                         SIMDE_FLOAT32_C(  -406.07), SIMDE_FLOAT32_C(  -378.35), SIMDE_FLOAT32_C(  -787.20), SIMDE_FLOAT32_C(  -878.65)) }
  };


  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_setr_ps( test_vec[i].e15, test_vec[i].e14, test_vec[i].e13, test_vec[i].e12, test_vec[i].e11, test_vec[i].e10, test_vec[i].e9, test_vec[i].e8,
                                         test_vec[i].e7, test_vec[i].e6, test_vec[i].e5, test_vec[i].e4, test_vec[i].e3, test_vec[i].e2, test_vec[i].e1, test_vec[i].e0);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_setr_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde_float64  e7; simde_float64  e6; simde_float64  e5; simde_float64  e4;
    simde_float64  e3; simde_float64  e2; simde_float64  e1; simde_float64  e0;
    simde__m512d r;
  } test_vec[8] = {
    {  SIMDE_FLOAT64_C( -434.21),
      SIMDE_FLOAT64_C( -283.66),
      SIMDE_FLOAT64_C(  252.63),
      SIMDE_FLOAT64_C(   58.11),
      SIMDE_FLOAT64_C(  292.72),
      SIMDE_FLOAT64_C( -266.76),
      SIMDE_FLOAT64_C(  689.73),
      SIMDE_FLOAT64_C(  940.04),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  940.04), SIMDE_FLOAT64_C(  689.73),
                         SIMDE_FLOAT64_C( -266.76), SIMDE_FLOAT64_C(  292.72),
                         SIMDE_FLOAT64_C(   58.11), SIMDE_FLOAT64_C(  252.63),
                         SIMDE_FLOAT64_C( -283.66), SIMDE_FLOAT64_C( -434.21)) },
   {  SIMDE_FLOAT64_C(  736.05),
      SIMDE_FLOAT64_C(  -71.17),
      SIMDE_FLOAT64_C( -331.02),
      SIMDE_FLOAT64_C(  508.70),
      SIMDE_FLOAT64_C(  419.04),
      SIMDE_FLOAT64_C(  700.48),
      SIMDE_FLOAT64_C(  623.94),
      SIMDE_FLOAT64_C(  606.97),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  606.97), SIMDE_FLOAT64_C(  623.94),
                         SIMDE_FLOAT64_C(  700.48), SIMDE_FLOAT64_C(  419.04),
                         SIMDE_FLOAT64_C(  508.70), SIMDE_FLOAT64_C( -331.02),
                         SIMDE_FLOAT64_C(  -71.17), SIMDE_FLOAT64_C(  736.05)) },
   {  SIMDE_FLOAT64_C( -489.77),
      SIMDE_FLOAT64_C(  777.96),
      SIMDE_FLOAT64_C( -178.61),
      SIMDE_FLOAT64_C( -524.51),
      SIMDE_FLOAT64_C( -840.38),
      SIMDE_FLOAT64_C( -796.18),
      SIMDE_FLOAT64_C( -772.79),
      SIMDE_FLOAT64_C(  621.56),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  621.56), SIMDE_FLOAT64_C( -772.79),
                         SIMDE_FLOAT64_C( -796.18), SIMDE_FLOAT64_C( -840.38),
                         SIMDE_FLOAT64_C( -524.51), SIMDE_FLOAT64_C( -178.61),
                         SIMDE_FLOAT64_C(  777.96), SIMDE_FLOAT64_C( -489.77)) },
   {  SIMDE_FLOAT64_C( -273.76),
      SIMDE_FLOAT64_C( -100.14),
      SIMDE_FLOAT64_C( -705.52),
      SIMDE_FLOAT64_C( -912.29),
      SIMDE_FLOAT64_C(  694.94),
      SIMDE_FLOAT64_C(  638.47),
      SIMDE_FLOAT64_C(  -57.86),
      SIMDE_FLOAT64_C(  194.70),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  194.70), SIMDE_FLOAT64_C(  -57.86),
                         SIMDE_FLOAT64_C(  638.47), SIMDE_FLOAT64_C(  694.94),
                         SIMDE_FLOAT64_C( -912.29), SIMDE_FLOAT64_C( -705.52),
                         SIMDE_FLOAT64_C( -100.14), SIMDE_FLOAT64_C( -273.76)) },
   {  SIMDE_FLOAT64_C(   72.92),
      SIMDE_FLOAT64_C( -114.92),
      SIMDE_FLOAT64_C( -543.69),
      SIMDE_FLOAT64_C( -956.64),
      SIMDE_FLOAT64_C( -736.35),
      SIMDE_FLOAT64_C( -239.62),
      SIMDE_FLOAT64_C(  526.64),
      SIMDE_FLOAT64_C( -912.23),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -912.23), SIMDE_FLOAT64_C(  526.64),
                         SIMDE_FLOAT64_C( -239.62), SIMDE_FLOAT64_C( -736.35),
                         SIMDE_FLOAT64_C( -956.64), SIMDE_FLOAT64_C( -543.69),
                         SIMDE_FLOAT64_C( -114.92), SIMDE_FLOAT64_C(   72.92)) },
   {  SIMDE_FLOAT64_C(  460.51),
      SIMDE_FLOAT64_C(  395.84),
      SIMDE_FLOAT64_C(  320.81),
      SIMDE_FLOAT64_C( -538.91),
      SIMDE_FLOAT64_C( -261.04),
      SIMDE_FLOAT64_C( -617.09),
      SIMDE_FLOAT64_C(  102.81),
      SIMDE_FLOAT64_C( -450.10),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -450.10), SIMDE_FLOAT64_C(  102.81),
                         SIMDE_FLOAT64_C( -617.09), SIMDE_FLOAT64_C( -261.04),
                         SIMDE_FLOAT64_C( -538.91), SIMDE_FLOAT64_C(  320.81),
                         SIMDE_FLOAT64_C(  395.84), SIMDE_FLOAT64_C(  460.51)) },
   {  SIMDE_FLOAT64_C( -348.77),
      SIMDE_FLOAT64_C( -294.21),
      SIMDE_FLOAT64_C( -435.22),
      SIMDE_FLOAT64_C(   29.39),
      SIMDE_FLOAT64_C(  461.71),
      SIMDE_FLOAT64_C(  -58.38),
      SIMDE_FLOAT64_C( -276.76),
      SIMDE_FLOAT64_C(    4.34),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    4.34), SIMDE_FLOAT64_C( -276.76),
                         SIMDE_FLOAT64_C(  -58.38), SIMDE_FLOAT64_C(  461.71),
                         SIMDE_FLOAT64_C(   29.39), SIMDE_FLOAT64_C( -435.22),
                         SIMDE_FLOAT64_C( -294.21), SIMDE_FLOAT64_C( -348.77)) },
   {  SIMDE_FLOAT64_C(  -13.27),
      SIMDE_FLOAT64_C(   86.22),
      SIMDE_FLOAT64_C( -609.85),
      SIMDE_FLOAT64_C( -191.12),
      SIMDE_FLOAT64_C( -989.14),
      SIMDE_FLOAT64_C( -547.71),
      SIMDE_FLOAT64_C( -227.52),
      SIMDE_FLOAT64_C(  -15.18),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -15.18), SIMDE_FLOAT64_C( -227.52),
                         SIMDE_FLOAT64_C( -547.71), SIMDE_FLOAT64_C( -989.14),
                         SIMDE_FLOAT64_C( -191.12), SIMDE_FLOAT64_C( -609.85),
                         SIMDE_FLOAT64_C(   86.22), SIMDE_FLOAT64_C(  -13.27)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_setr_pd(test_vec[i].e7, test_vec[i].e6, test_vec[i].e5, test_vec[i].e4, test_vec[i].e3, test_vec[i].e2, test_vec[i].e1, test_vec[i].e0);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_setr_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_setr_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_setr_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_setr_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
