# Code for the GUI interface to RoutePlanner
#
# Copyright (C) 1996-2004 Chris Lawrence
# This file may be freely distributed under the terms of the RoutePlanner
# license.  A copy should appear as 'LICENSE' in the archive that this
# file was included in.
#
# $Id: rpgnome.py,v 1.15 2004/10/02 12:41:34 lordsutch Exp $

import pygtk
pygtk.require('2.0')

import rpdbase

import sys, os, glob, commands, time
from rpcitylist import CityList
from rpprogress import ProgressWin

import gtk
import gtk.gdk
import gnome
import gnome.ui

APP = 'RoutePlanner'
VERSION = rpdbase.VERID
COPYRIGHT = "Copyright (C) 1996-2004 Chris Lawrence"
gnome.init(APP, VERSION)

import gtk.glade
import gobject

from rpunits import *

import gconf
#import gnome.config

# i18n
DIR = 'i18n'

import locale, gettext
locale.setlocale (locale.LC_ALL, '')
gettext.bindtextdomain (APP, DIR)
gettext.textdomain (APP)
gettext.install (APP, DIR, unicode=1)

# Whether or not we use the new file selector
newfilesel = 'FileChooserDialog' in dir(gtk)

PANES = ['shortroute', 'quickroute', 'prefroute', 'leastfuel']
GUIFILE = '/usr/share/routeplanner/routeplanner.glade2'

CONFIG_UNITS   = '/apps/routeplanner/Defaults/units'
CONFIG_DEFMPG  = '/apps/routeplanner/Defaults/default_efficiency'
CONFIG_DEFPREF = '/apps/routeplanner/Defaults/default_preference'

CONFIG_TOLL   = '/apps/routeplanner/Defaults/toll_penalty'
CONFIG_FERRY  = '/apps/routeplanner/Defaults/ferry_penalty'
CONFIG_SCENIC = '/apps/routeplanner/Defaults/scenic_bonus'

class Application:
    def __init__(self, filename=''):
        if os.path.exists('routeplanner.glade2'):
            self.wtree = gtk.glade.XML('routeplanner.glade2')
        else:
            self.wtree = gtk.glade.XML(GUIFILE)
        
        self.appwin = self.wtree.get_widget('app1')
        self.appwin.connect("destroy", gtk.main_quit) 

        dict = {}
        for key in dir(self.__class__):
            dict[key] = getattr(self, key)

        self.wtree.signal_autoconnect(dict)
        self.db = None
        self.cityview = self.wtree.get_widget('citylist')

        self.citystore = gtk.ListStore(gobject.TYPE_PYOBJECT,
                                       gobject.TYPE_STRING)
        renderer = gtk.CellRendererText()
        col = gtk.TreeViewColumn('City', renderer, text=1)
        self.cityview.append_column(col)
        self.cityview.set_reorderable(True)
        self.cityview.set_model(self.citystore)

        sel = self.cityview.get_selection()
        sel.set_mode(gtk.SELECTION_SINGLE)
        sel.connect("changed", self.on_citylist_select_row)
        
        self.citywin = None
        self.make_widgets_sensitive()

        self.gconf_client = gconf.client_get_default()
        self.gconf_client.add_dir('/apps/routeplanner',
                                  gconf.CLIENT_PRELOAD_RECURSIVE)

        self.units = self.gconf_client.get_string(CONFIG_UNITS)
        if not self.units or self.units not in VALID_UNITS:
            self.units = UNITS_US
        
        self.currentunits = self.units
        info = {'metric' : UNITS_METRIC, 'us' : UNITS_US,
                'imperial' : UNITS_IMPERIAL}
        for widget, units in info.items():
            if units == self.units:
                self.wtree.get_widget(widget).set_active(True)

        if self.units == UNITS_METRIC:
            self.wtree.get_widget('effunits').set_text('l/100km')

        self.defmpg = self.gconf_client.get_int(CONFIG_DEFMPG)
        if not self.defmpg or self.defmpg <= 0 or self.defmpg > 100:
            self.defmpg = 20
        effwidget = self.wtree.get_widget('defefficiency')
        effwidget.set_value(self.defmpg)

        self.defpref = self.gconf_client.get_int(CONFIG_DEFPREF)
        if not self.defpref or self.defpref < 0 or self.defpref > 100:
            self.defpref = 50
        prefwidget = self.wtree.get_widget('defpref')
        prefwidget.set_value(self.defpref)

        self.tollpen = self.gconf_client.get_int(CONFIG_TOLL)
        if not self.tollpen or self.tollpen < -100 or self.tollpen > 100:
            self.tollpen = 0
        self.wtree.get_widget('avoidtolls').set_active(self.tollpen)
        self.wtree.get_widget('tollpen').set_sensitive(self.tollpen)
        self.wtree.get_widget('tollpen').set_value(self.tollpen)

        self.ferrypen = self.gconf_client.get_int(CONFIG_FERRY)
        if not self.ferrypen or self.ferrypen < -100 or self.ferrypen > 100:
            self.ferrypen = 0
        self.wtree.get_widget('avoidferries').set_active(self.ferrypen)
        self.wtree.get_widget('ferrypen').set_sensitive(self.ferrypen)
        self.wtree.get_widget('ferrypen').set_value(self.ferrypen)

        self.scenicbonus = self.gconf_client.get_int(CONFIG_SCENIC)
        if not self.scenicbonus or self.scenicbonus < -100 or \
           self.scenicbonus > 100:
            self.scenicbonus = 0
        self.wtree.get_widget('usescenic').set_active(self.scenicbonus)
        self.wtree.get_widget('scenicbonus').set_sensitive(self.scenicbonus)
        self.wtree.get_widget('scenicbonus').set_value(self.scenicbonus)

        if filename:
            self.open_database(filename)

        prefswin = self.wtree.get_widget('prefswin')
        prefswin.hide_on_delete()

        classes = rpdbase.classifications.keys()
        classes.sort()
        classtable = self.wtree.get_widget('classtable')
        classtable.resize(len(classes)-2, 4)

        maxclass = max(rpdbase.classifications.values())+1
        self.speeds = [0]*maxclass
        self.mpgs = [0]*maxclass
        self.prefer = [0]*maxclass

        i = 1
        self.prefwidgets = {}
        for cl in classes:
            classid = rpdbase.classifications[cl]
            if len(rpdbase.defspeed) > classid:
                speed, mpg, pref = (rpdbase.defspeed[classid],
                                    rpdbase.defeff[classid],
                                    rpdbase.defpref[classid])
            else:
                speed, mpg, pref = 50, self.defmpg, self.defpref

            if cl[0:2] == 'XX': continue
            x = gtk.Label(cl)
            x.show()
            classtable.attach(x, 0, 1, i, i+1)
            row = []
            for j in range(1, 4):
                ptype = ['','speed','efficiency', 'preference'][j]
                x = gtk.SpinButton()
                cl2 = cl.replace('/', '_')
                cl2 = cl2.replace(' ', '_')
                val = self.gconf_client.get_int('/apps/routeplanner/%s/%s' %
                                                (ptype, cl2))
                maxval = 100
                if not val:
                    if j == 1:
                        val = Distance(speed, UNITS_US).AsUnit(self.units)
                        maxval = 150
                    elif j == 2:
                        val = ConvertEfficiency(mpg, UNITS_US, self.units)
                        
                    else:
                        val = pref
                        
                x.set_digits(0)
                x.get_adjustment().set_all(val, 1, maxval, 1, 10, 1)
                x.set_value(val)
                x.connect('changed', self.prefswin_changed)
                x.show()
                classtable.attach(x, j, j+1, i, i+1)
                row.append(x)
                if j == 1:
                    self.speeds[classid] = val
                elif j == 2:
                    self.mpgs[classid] = val
                else:
                    self.prefer[classid] = val

            self.prefwidgets[cl] = row
            i += 1

        classtable.show()

    def on_exit1_activate(self,*args):
        gtk.main_quit()

    def on_about1_activate(self,*args):
        about = gnome.ui.About(
            APP, VERSION, COPYRIGHT,
            _("Highway trip planner based on Jim Butterfield's RoadRoute."),
            ['Chris Lawrence <lawrencc@debian.org>',
             'Jim Butterfield'],
            [], '')
        #gtk.gdk.pixbuf_new_from_file(LOGO)
        about.show()

    def on_properties1_activate(self, *args):
        if self.db:
            message = 'Current database: %s\n%s\n'\
                      '%d Cites - %d Routes\nAuthor: %s <%s>' % \
                      (self.db.filename, self.db.comment,
                       len(self.db.cities), len(self.db.routes),
                       self.db.author, self.db.author_email)
        else:
            message = 'No database loaded.'
            
        dialog = gtk.MessageDialog(self.appwin, buttons=gtk.BUTTONS_OK,
                                   message_format=message)
        dialog.run()
        dialog.destroy()

    def on_addcity_clicked(self, *args):
        if not self.citywin:
            self.citywin = CityList(self.wtree)
        self.citywin.startup('', self.db, self.appendcity)

    def on_citylist_button_press_event(self, button, event, *args):
        if event.type == gtk.gdk._2BUTTON_PRESS and event.button == 1 \
           and self.cityview.get_selection().get_selected():
            self.on_changecity_clicked()

    def on_changecity_clicked(self, *args):
        if not self.citywin:
            self.citywin = CityList(self.wtree)
        sel = self.cityview.get_selection()
        city, it = sel.get_selected()
        pos = city.get_path(it)[0]
        cityname = city[pos][0]
        self.citywin.startup(cityname, self.db, self.changecity, pos)

    def appendcity(self, city, ind, *data):
        it = self.citystore.append()
        self.citystore.set(it, 0, city, 1, unicode(city))
        sel = self.cityview.get_selection()
        sel.select_path(len(self.citystore)-1)
        self.make_widgets_sensitive()

    def changecity(self, city, ind, pos):
        sel = self.cityview.get_selection()
        selcity, it = sel.get_selected()
        pos = selcity.get_path(it)[0]
        self.citystore[pos] = (city, unicode(city))
        self.make_widgets_sensitive()

    def on_removecity_clicked(self, *args):
        sel = self.cityview.get_selection()
        selcity, it = sel.get_selected()
        if not it:
            return
        
        pos = selcity.get_path(it)[0]
        del self.citystore[pos]
        self.make_widgets_sensitive()

    def on_citylist_select_row(self, sel, *args):
        wlist = ['changecity', 'removecity']
        if sel.get_selected()[1]:
            state = True
        else:
            state = False
        
        for widget in wlist:
            self.wtree.get_widget(widget).set_sensitive(state)

    def on_filesel_ok_clicked(self, *args):
        filename = self.fw.get_filename()
        self.fw.destroy()
        self.open_database(filename)

    def on_filesel_cancel_clicked(self, *args):
        self.fw.destroy()

    def on_open1_activate(self, *args):
        if newfilesel:
            fs = gtk.FileChooserDialog('Select database', self.appwin,
                                       gtk.FILE_CHOOSER_ACTION_OPEN,
                                       (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                        gtk.STOCK_OPEN, gtk.RESPONSE_ACCEPT))
            fs.set_current_folder('/usr/share/routeplanner')
            response = fs.run()
            fs.hide()
            if response == gtk.RESPONSE_ACCEPT:
                filename = fs.get_filename()
                self.open_database(filename)
            fs.destroy()
        else:
            self.fw = gtk.FileSelection('Select database')
            self.fw.set_transient_for(self.appwin)
            self.fw.selection_entry.connect("activate",
                                             self.on_filesel_ok_clicked)
            self.fw.ok_button.connect("pressed", self.on_filesel_ok_clicked)
            self.fw.cancel_button.connect("pressed",
                                          self.on_filesel_cancel_clicked)
            self.fw.set_modal(True)
            self.fw.hide_on_delete()
            self.fw.set_filename('/usr/share/routeplanner/')
            self.fw.show()

    def on_plan_clicked(self, *args):
        db = self.db
        path = []
        for (info, cname) in self.citystore:
            if path and info == path[-1]:
                dialog = gtk.MessageDialog(self.appwin,
                                           message_format='Same city appears '
                                           'twice in a row:\n'+cname+'\n'
                                           'Please fix this and plan again.',
                                           type=gtk.MESSAGE_ERROR,
                                           buttons=gtk.BUTTONS_CLOSE)
                dialog.run()
                dialog.destroy()
                return
            path.append(info)

        methods = len(rpdbase.methods)
        displist = [None]*methods
        self.routes = [None]*methods
        progwin = ProgressWin("Calculating route", self.appwin)
        progwin.set_label('Planning')
        progwin.set_current(0)
        progwin.set_max(methods)
        progwin.show()

        for method in range(methods):
            progwin.set_label('Planning '+rpdbase.methods[method])
            progwin.set_current(method)
            trail = []
            for i in range(len(path)-1):
                trail.append( db.NewNavigate(
                    path[i], path[i+1], method = method,
                    codetable = self.speeds, prefertable = self.prefer,
                    defaultprefer = self.defpref, mpgtable = self.mpgs,
                    defmpg = self.defmpg, units=self.units,
                    flagbonus = { 'toll' : -self.tollpen,
                                  'scenic' : self.scenicbonus,
                                  'ferry': -self.ferrypen } ))
            progwin.set_current(method+0.5)

            displist[method] = rpdbase.ProduceTrail(trail, db, self.units,
                                                    self.mpgs, self.defmpg,
                                                    self.speeds)

            del trail # Free some memory
            progwin.set_current(method+1)
        
        progwin.destroy()

        planwin = self.wtree.get_widget('planwin')
        startloc = self.wtree.get_widget('startloc')
        endloc = self.wtree.get_widget('endloc')
        vialocs = self.wtree.get_widget('vialocs')

        start, end, via = path[0], path[-1], path[1:-1]
        startloc.set_text(str(start))
        endloc.set_text(str(end))
        if via:
            vialocs.set_text('; '.join(map(str, via)))
        else:
            vialocs.set_text('[none]')

        #font = load_font("fixed")
        #black = GdkColor(0,0,0)

        widgets = PANES
        for j in range(methods):
            route = self.wtree.get_widget(widgets[j])
            text = rpdbase.FormatRoute(displist[j], path, self.units)
            buff = gtk.TextBuffer()
            route.set_buffer(buff)
            tag = gtk.TextTag('routedesc')
            tag.set_property('font', 'Monospace 10')
            buff.get_tag_table().add(tag)
            sob, eob = buff.get_bounds()
            buff.insert_with_tags_by_name(eob, text, "routedesc")
            self.routes[j] = text
        
        planwin.connect("delete_event", self.close_route_win)
        planwin.show()

    def on_close1_activate(self, *args):
        self.close_route_win()

    def close_route_win(self, *args):
        self.wtree.get_widget('planwin').hide()
        return True

    def on_print1_activate(self, *args):
        printwin = self.wtree.get_widget('printwin')
        printwin.connect("delete_event", self.close_print_win)
        printwin.set_transient_for(self.wtree.get_widget('planwin'))
        printwin.hide_on_delete()
        printwin.show()

    def close_print_win(self, *args):
        self.wtree.get_widget('printwin').hide()
        return True

    def on_print_ok_clicked(self, *args):
        self.wtree.get_widget('printwin').hide()
        command = self.wtree.get_widget('printcomm').get_text()
        items = []
        items.append( self.wtree.get_widget('shortest').get_active() )
        items.append( self.wtree.get_widget('quickest').get_active() )
        items.append( self.wtree.get_widget('preferred').get_active() )
        items.append( self.wtree.get_widget('lowestfuel').get_active() )

        separator = '\n\n'
        if self.wtree.get_widget('use_ff').get_active():
            separator = '\f'

        fh = os.popen(command, 'w')
        already_printed = 0

        for j in range(len(rpdbase.methods)):
            if items[j]:
                if already_printed: fh.write(separator)
                fh.write(rpdbase.methods[j]+" Route:\n\n")
                fh.write(self.routes[j]+'\n')
                already_printed = 1

        fh.close()

        self.wtree.get_widget('printwin').hide()
        return True

    def on_save_as1_activate(self, *args):
        if newfilesel:
            fs = gtk.FileChooserDialog('Save routes as...',
                                       self.wtree.get_widget('planwin'),
                                       gtk.FILE_CHOOSER_ACTION_SAVE,
                                       (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                        gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))
            response = fs.run()
            fs.hide()
            if response == gtk.RESPONSE_ACCEPT:
                filename = fs.get_filename()
                do_save(filename)
            fs.destroy()
        else:
            self.fw = gtk.FileSelection('Save routes as...')
            self.fw.set_transient_for(self.wtree.get_widget('planwin'))
            self.fw.selection_entry.connect("activate", self.save_as_done)
            self.fw.ok_button.connect("pressed", self.save_as_done)
            self.fw.cancel_button.connect("pressed", lambda x: self.fw.destroy())
            self.fw.set_modal(True)
            self.fw.hide_on_delete()
            self.fw.show()

    def save_as_done(self, *args):
        fn = self.fw.get_filename()
        self.do_save(fn)

    def do_save(self, fn):
        if os.path.exists(fn):
            dialog = GnomeMessageBox("Overwrite existing file\n"+fn+"?",
                                     MESSAGE_BOX_QUESTION, STOCK_BUTTON_YES,
                                     STOCK_BUTTON_NO)
            dialog.set_parent(self.wtree.get_widget('planwin'))
            if dialog.run_and_close():
                return

        self.fw.destroy()
        del self.fw
        try:
            fh = open(fn, 'w')
        except IOError, x:
            dialog = GnomeErrorDialog(str(x), self.wtree.get_widget('planwin'))
            dialog.run_and_close()
            return
        
        for j in range(len(rpdbase.methods)):
            if j: fh.write('\n\n')
            fh.write(rpdbase.methods[j]+" Route:\n\n")
            fh.write(self.routes[j]+'\n')
        fh.close()

    def make_widgets_sensitive(self):
        if not self.db:
            wlist = {'addcity' : False,
                     'citylist' : False,
                     'plan' : False}
        elif len(self.citystore) > 1:
            wlist = {'addcity' : True,
                     'citylist' : True,
                     'plan' : True}
        else:
            wlist = {'addcity' : True,
                     'citylist' : True,
                     'plan' : False}

        for widget, state in wlist.items():
            self.wtree.get_widget(widget).set_sensitive(state)

    def open_database(self, filename):
        progwin = ProgressWin(os.path.basename(filename)+'...', self.appwin)
        progwin.show()
        try:
            self.db = rpdbase.RPDatabase(filename, quiet=1,
                                         progressbar=progwin)
        except (rpdbase.invalidformat, IOError), x:
            progwin.destroy()
            error = GnomeErrorDialog('Unable to open '+filename+':\n\n'+
                                     str(x), self.appwin)
            error.run_and_close()
            return

        self.make_widgets_sensitive()
        self.citystore.clear()
        self.appwin.set_title('RoutePlanner: '+filename)
        progwin.destroy()

    def on_preferences1_activate(self, *args):
        prefswin = self.wtree.get_widget('prefswin')
        prefswin.close_hides(True)
        prefswin.show()

    def units_changed(self, *args):
        self.prefswin_changed()
        oldunits = newunits = self.currentunits
        info = {'metric' : UNITS_METRIC, 'us' : UNITS_US,
                'imperial' : UNITS_IMPERIAL}
        for widget, units in info.items():
            if self.wtree.get_widget(widget).get_active():
                newunits = units

        if oldunits == newunits:
            return

        if newunits == UNITS_METRIC:
            self.wtree.get_widget('effunits').set_text('l/100km')
        else:
            self.wtree.get_widget('effunits').set_text('mpg')

        effwidget = self.wtree.get_widget('defefficiency')
        current = effwidget.get_value()
        current = ConvertEfficiency(current, oldunits, newunits)
        effwidget.set_value(current)

        for cl, row in self.prefwidgets.items():
            for j in range(1, 3):
                val = row[j-1].get_value()
                if j == 1:
                    row[j-1].set_value(Distance(val, oldunits).
                                       AsUnit(newunits))
                else:
                    row[j-1].set_value(ConvertEfficiency(val, oldunits,
                                                       newunits))
        
        self.currentunits = newunits

    def prefswin_changed(self, *args):
        self.wtree.get_widget('prefswin').set_modified(True)

    def on_prefswin_apply(self, *args):
        # Update prefs here
        info = {'metric' : UNITS_METRIC, 'us' : UNITS_US,
                'imperial' : UNITS_IMPERIAL}
        for widget, units in info.items():
            if self.wtree.get_widget(widget).get_active():
                self.units = units

        self.defmpg = self.wtree.get_widget('defefficiency').get_value_as_int()
        self.defpref = self.wtree.get_widget('defpref').get_value_as_int()

        self.gconf_client.set_string(CONFIG_UNITS, self.units)
        self.gconf_client.set_int(CONFIG_DEFMPG, self.defmpg)
        self.gconf_client.set_int(CONFIG_DEFPREF, self.defpref)

        self.tollpen = (self.wtree.get_widget('avoidtolls').get_active() and
                        self.wtree.get_widget('tollpen').get_value_as_int())
        self.ferrypen = (self.wtree.get_widget('avoidferries').get_active()
                         and self.wtree.get_widget('ferrypen').
                         get_value_as_int())
        self.scenicbonus = (self.wtree.get_widget('usescenic').get_active()
                            and self.wtree.get_widget('scenicbonus').
                            get_value_as_int())
        self.gconf_client.set_int(CONFIG_TOLL, self.tollpen)
        self.gconf_client.set_int(CONFIG_FERRY, self.ferrypen)
        self.gconf_client.set_int(CONFIG_SCENIC, self.scenicbonus)
        
        items = self.prefwidgets.items()
        items.sort()
        for cl, row in items:
            classid = rpdbase.classifications[cl]
            for j in range(1, 4):
                type = ['','speed','efficiency', 'preference'][j]
                val = row[j-1].get_value_as_int()

                cl = cl.replace('/', '_')
                cl = cl.replace(' ', '_')
                self.gconf_client.set_int('/apps/routeplanner/'+type+'/'+cl,
                                          val)

                if j == 1:
                    self.speeds[classid] = val
                elif j == 2:
                    self.mpgs[classid] = val
                else:
                    self.prefer[classid] = val

    def on_avoidtolls_toggled(self, widget, *args):
        self.wtree.get_widget('tollpen').set_sensitive(widget.get_active())
        self.prefswin_changed()

    def on_avoidferries_toggled(self, widget, *args):
        self.wtree.get_widget('ferrypen').set_sensitive(widget.get_active())
        self.prefswin_changed()

    def on_usescenic_toggled(self, widget, *args):
        self.wtree.get_widget('scenicbonus').set_sensitive(widget.get_active())
        self.prefswin_changed()

    def on_prefswin_close(self, *args):
        self.wtree.get_widget('prefswin').hide()
        return True

def main(filename=''):
    global app
    app = Application(filename)
    gtk.main()
    app.appwin.hide()
