# ifndef _RHEO_UZAWA_ABTB_H
# define _RHEO_UZAWA_ABTB_H
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:uzawa_abtb
NAME: @code{uzawa_abtb}, @code{uz_abtb} -- mixed linear solver
@findex uzawa\_abtb
@cindex mixed linear problem
@cindex Stokes problem
@cindex Uzawa algorithm
@cindex finite element method
DESCRIPTION: 
  @noindent
  Uzawa algorithm applied to the mixed problem
  @example
       [ a  b^T ] [ u ]    [ f ]
       [        ] [   ]  = [   ]
       [ b  0   ] [ p ]    [ g ]
  @end example
  with the bock diagonal preconditionner : (inv(a),I).
  Such mixed linear problems appears for instance with the discretization
  of Stokes problems.
  The algorithm requires the resolution of linear
  systems such as: @code{a*x = b}.
  These systems may be solved either by direct or iterative 
  algorithms. Thus, a general matrix solver class is submitted
  to the algorithm.

  @noindent
  @code{uzawa_abtb} is a specialization of the @code{uz_abtb} that
  uses a direct solver.

AUTHOR: Pierre.Saramito@imag.fr
DATE:   19 nov 1997
METHODS: @uzawa_abtb
End:
*/

#include "rheolef/matrix_solver.h"
namespace rheolef { 

//<uzawa_abtb:
template<
    class MatrixSolver, 
    class MatrixPreconditionner, 
    class Matrix, 
    class DiagMatrix,
    class Vector, 
    class Real,
    class Size>
int
uz_abtb(
    const MatrixSolver&          m_solver,
    const Matrix&                a,
    const MatrixPreconditionner& ap,
    const Matrix&                b,
    const DiagMatrix&,
    Vector&                      u,
    Vector&                      p,
    const Vector&                f,
    const Vector&                g,
    const Real&                  rho,
    Size&                        max_iter,
    Real&                        tol,
    std::ostream*                p_cres = 0)
{
    Real residu = 1;
    for (Size k=1; k <= max_iter; k++) {
        m_solver (a, ap, u, f - b.trans_mult(p));  
        Vector R = b*u - g;
        p += rho*R ;
        residu = norm(R);
        if (p_cres) *p_cres << "[uzawa_abtb] " << k << " " << residu << "\n" ;
        if (residu <= tol) {
	    m_solver (a, ap, u, f - b.trans_mult(p));  
	    tol = residu;
	    max_iter = k;
	    return 0;
        }
    }
    tol = residu;
    return 1;
}
template<
    class MatrixPreconditionner, 
    class Matrix, 
    class Vector, 
    class Real,
    class Size>
int
uzawa_abtb(
    const Matrix&                a,
    const MatrixPreconditionner& ap,
    const Matrix&                b,
    Vector&                      u,
    Vector&                      p,
    const Vector&                f,
    const Vector&                g,
    const Real&                  rho,
    Size&                        max_iter,
    Real&                        tol,
    std::ostream*                p_cres = &std::cerr)
{
    return uz_abtb (ldlt_solver<MatrixPreconditionner, Matrix, Vector, Vector>(),
        a, ap, b, EYE, u, p, f, g, rho, max_iter, tol, p_cres);
}
//>uzawa_abtb:
}// namespace rheolef
# endif // _RHEO_UZAWA_ABTB_H
