# ifndef _SKIT_BICG_H
# define _SKIT_BICG_H
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:bicg
NAME: bicg - bi-conjugate gradient method 
SYNOPSIS:
    
    "int bicg" ("const Matrix &A, Vector &x, const Vector &b",
    |   const Preconditioner &M, int &max_iter, Real &tol);

EXAMPLE:

  The simplest call to 'bicg' has the folling form:

  |    int status = bicg(a, x, b, EYE, 100, 1e-7);

DESCRIPTION:       

 'bicg' solves the unsymmetric linear system Ax = b 
  using the preconditioned bi-conjugate gradient method.

  The return value indicates convergence within max_iter (input)
  iterations (0), or no convergence within max_iter iterations (1).
  Upon successful return, output arguments have the following values:

    "x"  approximate solution to Ax = b

    "max_iter" the number of iterations performed before the tolerance was reached

    "tol" the residual after the final iteration

SEE ALSO:

  "cg", class "csr", class "vec"

NOTE: 

  'bicg' is an iterative template routine.

  'bicg' follows the algorithm described on p. 22 in

	Templates for the Solution of Linear Systems: Building Blocks for Iterative Methods, 
	2nd Edition, 
        R. Barrett, M. Berry, T. F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout,
	R. Pozo, C. Romine, H. Van der Vorst,
        SIAM, 1994, "ftp.netlib.org/templates/templates.ps".

  The present implementation is inspired from "IML++ 1.2" iterative method library,
  "http://math.nist.gov/iml++".

AUTHOR: 

    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France

DATE: 
    
    12 march 1997

METHODS: @bicg
End:
*/
// ========== [ method implementation ] ====================================

namespace rheolef { 
//<bicg:
template < class Matrix, class Vector, class Preconditioner, class Real >
int 
bicg(const Matrix &A, Vector &x, const Vector &b,
     const Preconditioner &M, int &max_iter, Real &tol)
{
  Real resid;
  Real rho_1, rho_2=0, alpha, beta;
  Vector z, ztilde, p, ptilde, q, qtilde;

  Real normb = norm(b);
  Vector r = b - A * x;
  Vector rtilde = r;

  if (normb == Real(0))
    normb = 1;
  
  if ((resid = norm(r) / normb) <= tol) {
    tol = resid;
    max_iter = 0;
    return 0;
  }

  for (int i = 1; i <= max_iter; i++) {
    z = M.solve(r);
    ztilde = M.trans_solve(rtilde);
    rho_1 = dot(z, rtilde);
    if (rho_1 == Real(0)) { 
      tol = norm(r) / normb;
      max_iter = i;
      return 2;
    }
    if (i == 1) {
      p = z;
      ptilde = ztilde;
    } else {
      beta = rho_1 / rho_2;
      p = z + beta * p;
      ptilde = ztilde + beta * ptilde;
    }
    q = A * p;
    qtilde = A.trans_mult(ptilde);
    alpha = rho_1 / dot(ptilde, q);
    x += alpha * p;
    r -= alpha * q;
    rtilde -= alpha * qtilde;

    rho_2 = rho_1;
    if ((resid = norm(r) / normb) < tol) {
      tol = resid;
      max_iter = i;
      return 0;
    }
  }
  tol = resid;
  return 1;
}
//>bicg:
}// namespace rheolef
#endif // _SKIT_BICG_H
