
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:cad
NAME: @code{cad} - plot a boundary file 
@pindex cad
@clindex cad
@cindex CAD, Computer Aid Design
DESCRIPTION:
  @noindent
  Manipulates geometry boundary files (CAD, Computer Aid Design):
  visualizations and file conversions.
  This command is under development.
SYNOPSIS:
  @example
	col < @var{file}[.qmgcad] | cad -input-qmg - @var{options...}
  @end example
EXAMPLES:
  @noindent
  The @code{qmg} logo:
  @example
	col < ../data/logo2.qmgcad  | cad -input-qmg - -noclean
	gnuplot output.plot
  @end example
  @noindent
  A torus:
  @example
	col < ../data/test2-out.qmgcad | cad -input-qmg -
  @end example
  In the @code{geomview} window, enter '2as' key sequence to switch to a smooth
  rendering.
UNRESOLVED ISSUES:
  @noindent
  The @code{qmg} demonstration files comes from PC-DOS and contains line-feed
  characters that are not filtered by @code{flex} when reading
  file. Thus, the @code{col} filter is required.

  @noindent
  Included surfaces, such as cracks, are hiden in 3D with @code{geomview}.
  Thus, @code{vtk} rendering with some transparency factor would be better.

  @noindent
  @code{bamg} and @code{grummp} boundary files are not yet supported.
  They will be in the future.
INPUT FILE SPECIFICATION:
  @table @code
  @itemx @var{filename}
        specifies the name of the file containing
        the input mesh. The ".cad" extension is assumed.

@cindex RHEOPATH environment variable

  @itemx -I@var{caddir}
        add @var{caddir} to the mesh search path.
        This mechanism initializes a search path given by the environment variable
        @samp{RHEOPATH}. If the environment variable
        @samp{RHEOPATH} is not set, the
        default value is the current directory
        (@pxref{cad class}).

  @itemx -
        read mesh on standard input instead on a file.
  @end table
INPUT FORMAT OPTIONS:
@cindex cad file format conversion
@fiindex @file{.qmgcad} bamg mesh
@toindex @code{qmg}

  @table @code
  @itemx -input-cad
        read the boundary in the @file{.cad} format.
        This is the default 
	(TODO: not yet, uses qmg instead).

  @itemx -input-qmg
        read the boundary in the @file{.qmgcad} format.
	In @code{qmg}, this file format is known as @emph{brep}, that
	stands for boundary representation.
  @end table

OUTPUT FORMAT OPTIONS:
@toindex @code{geomview}
  @table @code
  @itemx -cad
        output boundary on standard output stream in cad text file format,
        instead of plotting it.

  @itemx -geomview
        use geomview rendering tool.
	This is the default for tridimensional data.

@toindex @code{gnuplot}
  @itemx -gnuplot
        use gnuplot rendering tool.
	This is the default for bidimensional data.
  @end table

RENDER OPTIONS:
  @table @code
  @item -subdivide @var{nsub}
	Subdivide each Bezier patch in @var{degree}*@var{nsub} linear
	sub-element. 
        For rendering tools that does not support Bezier patches, such
	as @code{gnuplot} or @code{geomview}.
	Default is @var{nsub}=3 for tridimensional data and 	
	@var{nsub}=50 for bidimensional data.

  @item -no-bezier-adapt
	Subdivide each Bezier patch in @var{nsub} instead 
	of @var{degree}*@var{nsub}.
  @item -bezier-adapt
	Subdivide each Bezier patch in of @var{degree}*@var{nsub}.
	This is the default.
  @end table

OTHERS OPTIONS:
  @table @code
    @itemx -verbose
    	print messages related to graphic files created and
       	command system calls (this is the default).

    @itemx -noverbose
    	does not print previous messages.

    @itemx -clean
    	clear temporary graphic files (this is the default).
    
    @itemx -noclean
    	does not clear temporary graphic files.

    @itemx -execute
    	execute graphic command (this is the default).
    
    @itemx -noexecute
   	does not execute graphic command. Generates only 
	graphic files. This is usefull in conjuction with the
	"-noclean" command.

  @end table

AUTHOR:
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
SEE ALSO:
    "geo"(3)
DATE:
    2 february 2002
End:
*/

#include "rheolef/cad.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
using namespace rheolef;
using namespace std;
	
bool
atoint (const char* s, int& x)
{
    int tmp;
    if (1 != sscanf(s, "%d", & tmp)) return false;
    x = tmp;
    return true;
}
void usage()
{
      cerr << "cad: usage:" << endl
           << "cad "
           << "{-Icaddir}* "
           << "[-input-cad|-input-qmg] "
           << "[-cad|-gnuplot|-geomview] "
           << "[-[no-]bezier-adapt] "
           << "[-subdivide int] "
           << "[-[no]clean] [-[no]execute] [-[no]verbose] "
           << "{-|file[.cad[.gz]]}"
           << endl;
      exit (1);
}
typedef enum {
        output_unspecified,
        output_cad,
        output_gnuplot,
        output_geomview
} output_type;

int
main(int argc, char** argv)
{
    if (argc <= 1) usage();
 
    cad c;
    bool have_cad = false;
    bool have_subdiv = false;
    int digits10 = numeric_limits<Float>::digits10;
    clog << verbose;
    bool bverbose = true;
    output_type output_format = output_unspecified;

    for (int i = 1; i < argc; i++) {

      // input format

	     if (strcmp (argv[i], "-input-cad") == 0) cin >> rheo;
	else if (strcmp (argv[i], "-input-qmg") == 0) cin >> qmg;

      // output format

        else if (strcmp (argv[i], "-cad") == 0)       { cout << rheo;     output_format = output_cad;}
        else if (strcmp (argv[i], "-gnuplot") == 0)   { cout << gnuplot;  output_format = output_gnuplot;}
        else if (strcmp (argv[i], "-geomview") == 0)  { cout << geomview; output_format = output_geomview;}
	
      // rendering options

        else if (strcmp (argv[i], "-bezier-adapt") == 0)     cout << bezieradapt;
        else if (strcmp (argv[i], "-no-bezier-adapt") == 0)  cout << nobezieradapt;
        else if (strcmp (argv[i], "-subdivide") == 0) {

	    int nsub = 1;
            if (i == argc-1 || ! atoint(argv[++i], nsub)) {
                cerr << "cad: -subdivide option: numeric argument missing" << endl;
                usage();
            }
            cout << setsubdivide(nsub);
	    have_subdiv = true;
	}
      // other options

        else if (strcmp (argv[i], "-clean") == 0)     clog << clean;
        else if (strcmp (argv[i], "-noclean") == 0)   clog << noclean;
        else if (strcmp (argv[i], "-execute") == 0)   clog << execute;
        else if (strcmp (argv[i], "-noexecute") == 0) clog << noexecute;
        else if (strcmp (argv[i], "-verbose") == 0)   { bverbose = true; clog << verbose; }
        else if (strcmp (argv[i], "-noverbose") == 0) { bverbose = false; clog << noverbose; }
        else if (argv [i][0] == '-' && argv [i][1] == 'I') {

            append_dir_to_rheo_path (argv[i]+2);
        }
        else if (strcmp (argv[i], "-") == 0) {

            // input on standard input
            cin  >> qmg >> c;
            have_cad = true;
        }
        else if (argv[i][0] != '-') {

            // input geo on file
            c = cad (argv[i]);
            have_cad = true;
        }
        else {
            usage();
        }
    }
    if (!have_cad) {
        cerr << "cad: no input specified" << endl;
        usage();
    }
    // defaul rendering tool depends upon dimension
    if (output_format == output_unspecified) {
        if (c.dimension() == 2) {
	    cout << gnuplot;
        } else if (c.dimension() == 3) {
	    cout << geomview;
        }
    }
    if (!have_subdiv) {
        if (c.dimension() == 2) {
	    cout << setsubdivide(50);
        } else if (c.dimension() == 3) {
	    cout << setsubdivide(3);
        }
    }
    cout << setprecision(digits10)
         << c;
    return 0;
}
