/*P:dirichlet-nh-error
NAME: Error analysis (@PAKAGE@-@VERSION@)
\cindex{error analysis}
\foindex{mass}
PRINCIPE:
  Since the solution $u$ is regular, the following error estimates
  holds:
  $$
      \Vert u - u_h \Vert_{0,2,\Omega} \approx {\cal O}(h^{k+1})  
  $$
  $$
      \Vert u - u_h \Vert_{0,\infty,\Omega} \approx O(h^{k+1})  
  $$
  providing the approximate solution $u_h$ uses
  $P_k$ continuous finite element method, $k \geq 1$.
  Here, $\Vert.\Vert_{0,2,\Omega}$ and  $\Vert.\Vert_{0,\infty,\Omega}$
  denotes as usual the $L^2(\Omega)$ and $L^\infty(\Omega)$ norms.

  By denoting $\pi_h$ the Lagrange interpolation operator,
  the triangular inequality leads to:
  $$
      \Vert u - u_h \Vert_{0,2,\Omega}
      \leq
      \Vert u - \pi_h u \Vert_{0,2,\Omega}
      +
      \Vert u_h - \pi_h u \Vert_{0,2,\Omega}
  $$
  Since $\Vert u - \pi_h u \Vert_{0,2,\Omega} \approx O(h^{k+1})$,
  we have just to check the $\Vert u_h - \pi_h u \Vert_{0,2,\Omega}$
  error term.

END:
*/

//<dirichlet-nh-error:
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

Float u (const point& x) { return sin(x[0]+x[1]+x[2]); }

int main(int argc, char**argv)
{
    field u_h;
    cin >> u_h;
    space Vh = u_h.get_space();
    field pi_h_u = interpolate(Vh, u);
    field eh = pi_h_u - u_h;
    form m(Vh, Vh, "mass");
    cout << "error_inf "  << eh.max_abs()       << endl;
    cout << "error_l2  "  << sqrt(m(eh,eh)) << endl;
    return 0;
}
//>dirichlet-nh-error:
/*P:dirichlet-nh-error
RUNNING THE PROGRAM:
  Remarks on step~(b) the use of the \code{get\_space} member
  function.
  Thus, the previous implementation does not depend upon
  the degree of the polynomial approximation.

  After compilation, run the code by using the command:
  \begin{verbatim}
	dirichlet-nh square-h=0.1.geo P1 | dirichlet-nh-error
  \end{verbatim}

  The two errors in $L^\infty$ and $L^2$ are printed 
  for a $h=0.1$ quasi-uniform mesh.

  Let $nelt$ denotes the number of elements in the mesh.
  Since the mesh is quasi-uniform, we have
  $h \approx nelt^{\frac{1}{N}}$. Here $N=2$ for our
  bidimensionnal mesh.
  The figure~\ref{fig-dirichlet-nh-err} plots in logarithmic scale the
  error versus $nelt^{\frac{1}{2}}$
  for both $P_1$ (on the left) and $P_2$ (on the right) 
  approximations.

  % latex2html does not support tabular in figures yet...
  \begin{figure}[H]
    \begin{center}
      \begin{tabular}{cc}
  	\includegraphics[scale=0.9]{demo2-P1-uniform.pdf}
        &
  	\includegraphics[scale=0.9]{demo2-P2-uniform.pdf}
      \end{tabular}
    \end{center}
    \caption{Error analysis in $L^2$ and $L^\infty$ norms.}
    \label{fig-dirichlet-nh-err}
  \end{figure}

END:
*/
