#ifndef _SKIT_NUMERIC_LIMITS_H
#define _SKIT_NUMERIC_LIMITS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#ifdef _RHEOLEF_HAVE_LIMITS
#include <limits>
#else
// avoid conflicts with previous def.
#define numeric_limits rheo_numeric_limits

#undef FLT_MANT_DIG
#undef FLT_DIG
#undef FLT_ROUNDS
#undef FLT_EPSILON
#undef FLT_MIN_EXP
#undef FLT_MIN
#undef FLT_MIN_10_EXP
#undef FLT_MAX_EXP
#undef FLT_MAX
#undef FLT_MAX_10_EXP
#undef DBL_MANT_DIG
#undef DBL_DIG
#undef DBL_EPSILON
#undef DBL_MIN_EXP
#undef DBL_MIN
#undef DBL_MIN_10_EXP
#undef DBL_MAX_EXP
#undef DBL_MAX
#undef DBL_MAX_10_EXP
#undef LDBL_MANT_DIG
#undef LDBL_DIG
#undef LDBL_EPSILON
#undef LDBL_MIN_EXP
#undef LDBL_MIN
#undef LDBL_MIN_10_EXP
#undef LDBL_MAX_EXP
#undef LDBL_MAX
#undef LDBL_MAX_10_EXP
#include <float.h>
#include <limits.h>

enum float_round_style {
     round_indeterminate       = -1,
     round_toward_zero         =  0,
     round_to_nearest          =  1,
     round_toward_infinity     =  2,
     round_toward_neg_infinity =  3
};

template<class T> 
class numeric_limits {
public:
    static const bool is_specialized = false;
    static T min() throw() {return T();}
    static T max() throw() {return T();}
    static const int  digits = 0;
    static const int  digits10 = 0;
    static const bool is_signed = false;
    static const bool is_integer = false;
    static const bool is_exact = false;
    static const int  radix = 0;
    static T epsilon() throw() {return T();}
    static T round_error() throw() {return T();}

    static const int  min_exponent = 0;
    static const int  min_exponent10 = 0;
    static const int  max_exponent = 0;
    static const int  max_exponent10 = 0;

    static const bool has_infinity = false;
    static const bool has_quiet_NaN = false;
    static const bool has_signaling_NaN = false;
    static const bool has_denorm = false;
    static const bool has_denorm_loss = false;

    // Sept. 1996 draft is vague on how these can be guaranteed to return
    // value without throwing an exception.  
    static T infinity() throw() {return T();};
    static T quiet_NaN() throw() {return T();}
    static T signaling_NaN() throw() {return T();}
    static T denorm_min() throw() {return T();}

    static const bool is_iec559 = false;
    static const bool is_bounded = false;
    static const bool is_modulo = false;

    static const bool traps = false;
    static const bool tinyness_before = false;
    static const float_round_style round_style = round_toward_zero;
};
// obsolete
#ifdef _RHEOLEF_HAVE_STATIC_DATA_MEMBER
template<class T> const bool numeric_limits<T>::is_specialized;
template<class T> const int  numeric_limits<T>::digits;
template<class T> const int  numeric_limits<T>::digits10;
template<class T> const bool numeric_limits<T>::is_signed;
template<class T> const bool numeric_limits<T>::is_integer;
template<class T> const bool numeric_limits<T>::is_exact;
template<class T> const int  numeric_limits<T>::radix;
template<class T> const int  numeric_limits<T>::min_exponent;
template<class T> const int  numeric_limits<T>::min_exponent10;
template<class T> const int  numeric_limits<T>::max_exponent;
template<class T> const int  numeric_limits<T>::max_exponent10;
template<class T> const bool numeric_limits<T>::has_infinity;
template<class T> const bool numeric_limits<T>::has_quiet_NaN;
template<class T> const bool numeric_limits<T>::has_signaling_NaN;
template<class T> const bool numeric_limits<T>::has_denorm;
template<class T> const bool numeric_limits<T>::has_denorm_loss;
template<class T> const bool numeric_limits<T>::is_iec559;
template<class T> const bool numeric_limits<T>::is_bounded;
template<class T> const bool numeric_limits<T>::is_modulo;
template<class T> const bool numeric_limits<T>::traps;
template<class T> const bool numeric_limits<T>::tinyness_before;
template<class T> const float_round_style numeric_limits<T>::round_style;
#endif // _RHEOLEF_HAVE_STATIC_DATA_MEMBER

// The specializations for floating-point types use the following macro 
// to factor out commonality.  They presume IEEE arithmetic.
#define __NUMERIC_LIMITS_FLOAT(T)		\
    static const bool is_specialized = true;	\
    static const int  radix = 2;		\
						\
    static const bool is_signed = true;		\
    static const bool is_integer = false;	\
    static const bool is_exact = false;		\
						\
    static const bool has_infinity = true;	\
    static const bool has_quiet_NaN = true;	\
    static const bool has_signaling_NaN = true;	\
    static const bool has_denorm = false;	\
    static const bool has_denorm_loss = false;	\
						\
    static const bool is_iec559 = sizeof(T)<=8;	\
    static const bool is_bounded = true;	\
    static const bool is_modulo = false;	\
    static const bool traps = true;		\
    static const bool tinyness_before = true;	\
 						\
    static T round_error ()   { return (T)0.5F; }    	\
    static const float_round_style round_style = round_to_nearest;	\
    static T infinity ()      {return *(T*)(void*)data.value[0];}\
    static T quiet_NaN ()     {return *(T*)(void*)data.value[1];}\
    static T signaling_NaN () {return *(T*)(void*)data.value[2];}\
private:					\
    static const struct data_t {		\
	T align;				\
	int value[3][sizeof(T)/sizeof(int) == 0 ? 1 : sizeof(T)/sizeof(int)];	\
    } data;					\
public:						\

template<>
class numeric_limits <float> {
public:
    static const int  digits = FLT_MANT_DIG;
    static const int  digits10 = FLT_DIG;
    static const int  min_exponent = FLT_MIN_EXP;
    static const int  max_exponent = FLT_MAX_EXP;
    static const int  min_exponent10 = FLT_MIN_10_EXP;
    static const int  max_exponent10 = FLT_MAX_10_EXP;

    __NUMERIC_LIMITS_FLOAT(float)

    static float min ()          { return FLT_MIN; }
    static float max ()          { return FLT_MAX; }
    static float epsilon ()      { return FLT_EPSILON; }
    static float denorm_min ()   { return FLT_MIN; }
};

template<>
class numeric_limits <double> {
public:
    static const int  digits = DBL_MANT_DIG;
    static const int  digits10 = DBL_DIG;
    static const int  min_exponent = DBL_MIN_EXP;
    static const int  max_exponent = DBL_MAX_EXP;
    static const int  min_exponent10 = DBL_MIN_10_EXP;
    static const int  max_exponent10 = DBL_MAX_10_EXP;

    __NUMERIC_LIMITS_FLOAT(double)

    static double min ()          { return DBL_MIN; }
    static double max ()          { return DBL_MAX; }
    static double epsilon ()      { return DBL_EPSILON; }
    static double denorm_min ()   { return min (); }
};

template<>
class numeric_limits <long double> {
public:
    static const int  digits = LDBL_MANT_DIG;
    static const int  digits10 = LDBL_DIG;
    static const int  min_exponent = LDBL_MIN_EXP;
    static const int  max_exponent = LDBL_MAX_EXP;
    static const int  min_exponent10 = LDBL_MIN_10_EXP;
    static const int  max_exponent10 = LDBL_MAX_10_EXP;

    __NUMERIC_LIMITS_FLOAT(long double)

    static long double min ()          { return LDBL_MIN; }
    static long double max ()          { return LDBL_MAX; }
    static long double epsilon ()      { return LDBL_EPSILON; }
    static long double denorm_min ()   { return min (); }
};
#endif // _RHEOLEF_HAVE_LIMITS

// The specializations for integral types use three macros to factor out
// commonality.
//
// 	__NUMERIC_LIMITS_INTEGRAL declares members of numeric_limits<T>
// 	whose value does not depend on the signdness of T. 
//
//	__NUMERIC_LIMITS_SIGNED(T) declares members dependent on
//	knowing that T is signed.
//
//	__NUMERIC_LIMITS_UNSIGNED(T) declares members dependent on
//	knowing that T is unsigned.
//
// We could have been real cutesy and come up with definitions that would
// work for both signed and unsigned types, but doing so does not seem
// to be worth the additional obfuscation and overhead for constant folding.
//
// The definitions are not intended to be universally portable.

#define __NUMERIC_LIMITS_INTEGRAL(T)			\
    static const bool is_specialized = true;			\
								\
    static const int radix = 2;					\
    static const int min_exponent = 0;				\
    static const int max_exponent = 0;				\
    static const int min_exponent10 = 0;			\
    static const int max_exponent10 = 0;			\
								\
    static const bool is_integer = true;			\
    static const bool is_exact = true;				\
								\
    static const bool has_infinity = false;			\
    static const bool has_quiet_NaN = false;			\
    static const bool has_signaling_NaN = false;		\
    static const bool has_denorm = false;			\
    static const bool has_denorm_loss = false;			\
								\
    static const bool is_iec559 = false;			\
    static const bool is_bounded = true;			\
    static const bool is_modulo = true;				\
    static const bool traps = false;				\
    static const bool tinyness_before = false;			\
								\
    static T infinity ()       { return 0; }	\
    static T quiet_NaN ()      { return 0; }	\
    static T signaling_NaN () { return 0; }		\
    static T epsilon ()     { return 1; }		\
    static T denorm_min ()  { return min (); }	\
    static T round_error () { return 0; }		\
								\
    static const float_round_style round_style = round_toward_zero;

#define __NUMERIC_LIMITS_SIGNED(T) 				\
    static const int digits = 8*sizeof(T)-1;			\
    /* Following presumes 8, 16, 32, or 64-bit T. */		\
    static const int digits10 = 7*sizeof(T)/3; 			\
    static const bool is_signed = true;				

#define __NUMERIC_LIMITS_UNSIGNED(T) 			\
    static const int digits = 8*sizeof(T);			\
    /* Following presumes 8, 16, 32, or 64-bit T. */		\
    static const int digits10 = 12*sizeof(T)/5;  		\
    static const bool is_signed = false;				

#ifndef _RHEOLEF_HAVE_LIMITS
template<>
class numeric_limits <int> {
public:
    __NUMERIC_LIMITS_INTEGRAL(int)
    __NUMERIC_LIMITS_SIGNED(int)
    static int min() { return INT_MIN; }
    static int max() { return INT_MAX; }
};
template<>
class numeric_limits <unsigned int> {
public:
    __NUMERIC_LIMITS_INTEGRAL(unsigned int)
    __NUMERIC_LIMITS_UNSIGNED(unsigned int)
    static unsigned int min() { return 0; }
    static unsigned int max() { return UINT_MAX; }
};

template<>
class numeric_limits <long> {
public:
    __NUMERIC_LIMITS_INTEGRAL(long)
    __NUMERIC_LIMITS_SIGNED(long)
    static long min() { return LONG_MIN; }
    static long max() { return LONG_MAX; }
};

template<>
class numeric_limits <unsigned long> {
public:
    __NUMERIC_LIMITS_INTEGRAL(unsigned long)
    __NUMERIC_LIMITS_UNSIGNED(unsigned long)
    static unsigned long min() { return 0; }
    static unsigned long max() { return ULONG_MAX; }
};

template<>
class numeric_limits <short> {
public:
    __NUMERIC_LIMITS_INTEGRAL(short)
    __NUMERIC_LIMITS_SIGNED(short)
    static short min () { return SHRT_MIN; }
    static short max () { return SHRT_MAX; }
};

template<>
class numeric_limits <unsigned short> {
public:
    __NUMERIC_LIMITS_INTEGRAL(unsigned short)
    __NUMERIC_LIMITS_UNSIGNED(unsigned short)
    static unsigned short min () { return 0; }
    static unsigned short max () { return USHRT_MAX; }
};

template<>
class numeric_limits <char> {
public:
    __NUMERIC_LIMITS_INTEGRAL(char)
    static const int digits = CHAR_MIN<0 ? 7 : 8;
    static const int digits10 = 2;
    static const bool is_signed = CHAR_MIN<0;				
    static char min () { return CHAR_MIN; }
    static char max () { return CHAR_MAX; }
};

template<>
class numeric_limits <signed char> {
public:
    __NUMERIC_LIMITS_INTEGRAL(signed char)
    __NUMERIC_LIMITS_SIGNED(signed char)
    static signed char min ()         { return SCHAR_MIN; }
    static signed char max ()         { return SCHAR_MAX; }
};

template<>
class numeric_limits <unsigned char> {
public:
    __NUMERIC_LIMITS_INTEGRAL(unsigned char)
    __NUMERIC_LIMITS_UNSIGNED(unsigned char)
    static unsigned char min ()         { return 0; }
    static unsigned char max ()         { return UCHAR_MAX; }
};
// obsolete:
#ifdef _RHEOLEF_HAVE_WCHART
template<>
class numeric_limits <wchar_t> {
public:
    __NUMERIC_LIMITS_INTEGRAL(wchar_t)
    static const bool is_signed = (wchar_t)-1<0;
    static const int digits = 8*sizeof(wchar_t) - is_signed;
    // Following assumes that wchar_t is 8, 16, or 32-bit, 
    // either signed or unsigned.
    static const int digits10 = 7*sizeof(T)/3;
    static char min () { return CHAR_MIN; }
    static char max () { return CHAR_MAX; }
};
#endif /* _RHEOLEF_HAVE_WCHART */

#ifdef _BOOL
template<>
class numeric_limits <bool> {
public:
    static const bool is_specialized = true;

    static const int radix = 2;
    static const int min_exponent = 0;
    static const int max_exponent = 0;
    static const int min_exponent10 = 0;
    static const int max_exponent10 = 0;

    static const bool is_integer = false;
    static const bool is_exact = true;

    static const bool has_infinity = false;
    static const bool has_quiet_NaN = false;
    static const bool has_signaling_NaN = false;
    static const bool has_denorm = false;
    static const bool has_denorm_loss = false;

    static const bool is_iec559 = false;
    static const bool is_bounded = true;
    static const bool is_modulo = false;
    static const bool traps = false;
    static const bool tinyness_before = false;

    static bool infinity ()       { return false; }
    static bool quiet_NaN ()      { return false; }
    static bool signaling_NaN () { return false; }
    static bool epsilon ()     { return false; }
    static bool denorm_min ()  { return min (); }
    static bool round_error () { return false; }
								
    static const float_round_style round_style = round_toward_zero;

    static const int digits = 1;
    static const int digits10 = 0;
    static const bool is_signed = false;

    static bool min ()         { return false; }
    static bool max ()         { return true; }
};
#endif	/* _BOOL */
#endif // ! _RHEOLEF_HAVE_LIMITS

#endif /* _SKIT_NUMERIC_LIMITS_H */

