/* primax_scan -- linux driver for Primax scanners

   Authors:
   Marco Foglia <Marco.Foglia@switzerland.org>
   Thomas Schano <schano@t-online.de>
   Christian Ordig <chr.ordig@gmx.net>

   Copyright (C) 1999 Marco Foglia, Thomas Schano, Christian Ordig

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
*/        




#include <asm/io.h>
#include <stdio.h>
#include <unistd.h>            
#include <sched.h>
#include <sys/time.h>
                        
#include "primax_scan.h"
#include "lp.h"

static int last_set_mode = -1;  /* Mode from last epp_set_mode() */
static int port_feature;
static unsigned char port_mode;
unsigned short port = 0x378;

/*  struct file_operations */
/*  { */
/*aus linux/fs.h*/ 
/*          loff_t (*llseek) (struct file *, loff_t, int); */
/*  	ssize_t (*read) (struct file *, char *, size_t, loff_t *); */
/*  	ssize_t (*write) (struct file *, const char *, size_t, loff_t *); */
/*  	int (*readdir) (struct file *, void *, filldir_t); */
/*  	unsigned int (*poll) (struct file *, struct poll_table_struct *); */
/*  	int (*ioctl) (struct inode *, struct file *, unsigned int, unsigned long); */
/*  	int (*mmap) (struct file *, struct vm_area_struct *); */
/*  	int (*open) (struct inode *, struct file *); */
/*  	int (*flush) (struct file *); */
/*  	int (*release) (struct inode *, struct file *); */
/*  	int (*fsync) (struct file *, struct dentry *); */
/*  	int (*fasync) (int, struct file *, int); */
/*  	int (*check_media_change) (kdev_t dev); */
/*  	int (*revalidate) (kdev_t dev); */
/*  	int (*lock) (struct file *, int, struct file_lock *); */
/*  }; */


inline int epp_set_mode(unsigned char mode);
inline int clear_timeout(void);
static int port_probe(unsigned short port); 

/*-------------------------------------------------------------*/
void epp_off(void) 
{
    DBG(DBG_LOW,"epp_off:\n");
    /*Restore Port-Mode*/
    if (port_feature & 0x010 ){ 
	outb(port_mode, port + 0x402);
        fprintf(stderr, "Port %X is switched to initial mode\n",port);
    }
}

/*-------------------------------------------------------------*/
void epp_on(void)
{
    DBG(DBG_LOW,"epp_on:\n");

    if (iopl(3)) {
	fprintf(stderr, "Could not unlock IO ports. Are you superuser?\n");
	exit(1);
    }

    port_mode = inb(port + 0x402);
    port_feature = port_probe(port);
    
    if (!(port_feature & 0x300 )){ 
	fprintf(stderr, "\nThis Port does not support the  EPP-Mode\n");
	fprintf(stderr, "Please activate EPP-Mode or\nECP-Mode with included"
                " EPP-mode!\n\nThe activated and detected Modis are:\n");
        if (port_feature & PPA_PROBE_SPP)
            fprintf(stderr, "SPP-Port present\n");
        if (port_feature & PPA_PROBE_PS2)
            fprintf(stderr, "PS2-Port present\n");
        if (port_feature & PPA_PROBE_ECR)
            fprintf(stderr, "ECP-Port present\n");
        if (port_feature & PPA_PROBE_EPP17)
            fprintf(stderr, "EPP 1.7-Port present\n");
        if (port_feature & PPA_PROBE_EPP19)
            fprintf(stderr, "EPP 1.9-Port present\n");
	exit(1);
    };
    
    if (port_feature & 0x010 ){
	/* Port is ECP with EPP present*/
	/* switching Port to EPP */
        fprintf(stderr, "Port %X is switched to EPP-Mode\n",port);
	outb((port_mode & 0x1f) | 0x80, port + 0x402); 
    };
}


/*-------------------------------------------------------------*/
/*
 * resets a timer
 */
inline void init_timer(long long *timer, unsigned long us)
{
    struct timeval start_time;
    
    gettimeofday(&start_time, NULL);	
    *timer =  start_time.tv_sec * 1e6 + start_time.tv_usec + us;

    DBG(DBG_TIMEOUT, "init_timer:  %i.%i s   wait until:  %Li \n", 
	(int)start_time.tv_sec, (int)start_time.tv_usec, *timer);
}
  
/*-------------------------------------------------------------*/
/*
 * checks for timeout 
 */
inline int check_timer(long long *timer) 
{
    struct timeval current_time;
    
    gettimeofday(&current_time, NULL);

    /*DBG(DBG_TIMEOUT,"check_timer:  %i.%i s\n", 
      (int)current_time.tv_sec, (int)current_time.tv_usec);   */
    if (current_time.tv_sec * 1e6 + current_time.tv_usec > *timer)
	return TIMEOUT;
    else
	return NO_TIMEOUT;
}

/*-------------------------------------------------------------*/
inline int clear_timeout(void)
{
    unsigned char data = 0;

    data = inb(port + 1);
    if (data & 0x01) {
	outb(0x01 , port + 1);
	DBG(DBG_TIMEOUT, "timeout: %02x -- we are in trouble\n", data);
	return TIMEOUT;
    }
    return NO_TIMEOUT;
} 

/*-------------------------------------------------------------*/
/* 
 * Sets mode of next read/write operation from/to BASE+4.  
 * Does nothing if already in right mode
 */
inline int epp_set_mode(unsigned char mode) 
{
    int status;
    
    if (mode > 0xF){
	printf("epp_set_mode:  error, mode %02x > 0x0F  \n",mode);
	exit(1);
    }
    if (last_set_mode != mode) {
	if ((status = clear_timeout()))
	    return status;
	outb(mode, port + 3);
	if ((status = clear_timeout()))
	    return status;
	last_set_mode = mode;
    }
    return NO_TIMEOUT;
}
/*-------------------------------------------------------------*/
/* 
 * let's wait! 
 */
inline void scan_delay(unsigned long us_delay)
{
    
    /* this solution makes to much load */
    /* long long scan_delay_timer;
       init_timer(&scan_delay_timer, us_delay);
       while (!check_timer(&scan_delay_timer)); */
    
    /*This is the old hack */
    int sched_type;
    struct sched_param priority, save_priority;

    sched_getparam(0, &save_priority);    
    sched_type = sched_getscheduler(0);  

    priority.sched_priority = 1;
    sched_setscheduler(0, SCHED_FIFO, &priority);
    usleep(us_delay);    
    sched_setscheduler(0, sched_type, &save_priority);  
    
    DBG(DBG_LOW, "scan_delay: %li\n", us_delay);    
}

/*-------------------------------------------------------------*/
/* 
 * reads num bytes from scanner 
 */

int epp_read_one(unsigned char *data, unsigned char mode)
{
    int status;

    epp_set_mode(mode);

    if ((status = clear_timeout()))
	return status;
    *data = inb(port + 4);

    if ((status = clear_timeout()))
	return status;

    DBG(DBG_LOW, "epp_read:  %4x           %02x  %02x\n",num,mode, *data);

    return status;
}


int epp_read(unsigned char *data, int num, unsigned char mode) 
{
    int status;

    epp_set_mode(mode);

    if ((status = clear_timeout()))
	return status;
    insb(port + 4, data, num);

    if ((status = clear_timeout()))
	return status;

    DBG(DBG_LOW, "epp_read:  %4x           %02x   --  \n",num, mode);

    return status;
}

/*-------------------------------------------------------------*/
int epp_write_one(unsigned char data, unsigned char mode)
{
    int status;

    epp_set_mode(mode);

    if ((status = clear_timeout()))
	return status;

    outb(data ,port + 4);
    
    if ((status = clear_timeout()))
	return status;

    DBG(DBG_LOW, "epp_write: %4x  %02x  %02x\n",
        num, mode, *data);
    return status;
}

int epp_write(unsigned char *data, int num, unsigned char mode) 
{
    int status;

    epp_set_mode(mode);

    if ((status = clear_timeout()))
	return status;

    outsb(port+4,data,num);

    if ((status = clear_timeout()))
	return status;

    DBG(DBG_LOW, "epp_write: %4x  %02x   --\n",num, mode);

    return status;
}

/*
 *  stolen from the ppa package
 */

int port_probe(unsigned short port)
{
    int retv = 0;
    unsigned char a, b, c;
    unsigned int i, j;

//    printf(PPA_ID "Probing port %04x\n", port);

/*                 #####  ######  ######
 *                #     # #     # #     #
 *                #       #     # #     #
 *                 #####  ######  ######
 *                      # #       #
 *                #     # #       #
 *                 #####  #       #
 */

    outb(0x0c, port + 0x402);
    outb(0x0c, port + 0x002);
    outb(0x55, port);
    a = inb(port);
    if (a != 0x55)
	return retv;
//    printf(PPA_ID "    SPP port present\n");

    retv += PPA_PROBE_SPP;

/*                #######  #####  ######
 *                #       #     # #     #
 *                #       #       #     #
 *                #####   #       ######
 *                #       #       #
 *                #       #     # #
 *                #######  #####  #
 */

    for (i = 1024; i > 0; i--) {	/* clear at most 1k of data from FIFO */
	a = inb(port + 0x402);
	if ((a & 0x03) == 0x03)
	    goto no_ecp;
	if (a & 0x01)
	    break;
	inb(port + 0x400);	/* Remove byte from FIFO */
    }

    if (i <= 0)
	goto no_ecp;

    b = a ^ 3;
    outb(b, port + 0x402);
    c = inb(port + 0x402);

    if (a == c) {
	outb(0xc0, port + 0x402);	/* FIFO test */
	j = 0;
	while (!(inb(port + 0x402) & 0x01) && (j < 1024)) {
	    inb(port + 0x400);
	    j++;
	}
	if (j >= 1024)
	    goto no_ecp;
	i = 0;
	j = 0;
	while (!(inb(port + 0x402) & 0x02) && (j < 1024)) {
	    outb(0x00, port + 0x400);
	    i++;
	    j++;
	}
	if (j >= 1024)
	    goto no_ecp;
	j = 0;
	while (!(inb(port + 0x402) & 0x01) && (j < 1024)) {
	    inb(port + 0x400);
	    j++;
	}
	if (j >= 1024)
	    goto no_ecp;
//	printf(PPA_ID "    ECP with a %i byte FIFO present\n", i);

	retv += PPA_PROBE_ECR;
    }
/*                ######   #####   #####
 *                #     # #     # #     #
 *                #     # #             #
 *                ######   #####   #####
 *                #             # #
 *                #       #     # #
 *                #        #####  #######
 */

  no_ecp:
    if (retv & PPA_PROBE_ECR)
	outb(0x20, port + 0x402);

    outb(0x55, port);
    outb(0x0c, port + 2);
    a = inb(port);
    outb(0x55, port);
    outb(0x2c, port + 2);
    b = inb(port);
    if (a != b) {
//	printf(PPA_ID "    PS/2 bidirectional port present\n");
	retv += PPA_PROBE_PS2;
    }
/*                ####### ######  ######
 *                #       #     # #     #
 *                #       #     # #     #
 *                #####   ######  ######
 *                #       #       #
 *                #       #       #
 *                ####### #       #
 */

    if (port & 0x007) {
//	printf(PPA_ID "    EPP not supported at this address\n");
	return retv;
    }
    if (retv & PPA_PROBE_ECR) {
	for (i = 0x00; i < 0x80; i += 0x20) {
	    outb(i, port + 0x402);

	    a = inb(port + 1);
	    outb(a, port + 1);
	    outb(a & 0xfe, port + 1);
	    a = inb(port + 1);
	    if (!(a & 0x01)) {
		printf(PPA_ID "    Failed Intel bug check. (Phony EPP in ECP)\n");
		return retv;
	    }
	}
//	printf(PPA_ID "    Passed Intel bug check.\n");
	outb(0x80, port + 0x402);
    }
    a = inb(port + 1);
    outb(a, port + 1);
    outb(a & 0xfe, port + 1);
    a = inb(port + 1);

    if (a & 0x01) {
	outb(0x0c, port + 0x402);
	outb(0x0c, port + 0x002);
	return retv;
    }

    outb(0x04, port + 2);
    inb(port + 4);
    a = inb(port + 1);
    outb(a, port + 1);
    outb(a & 0xfe, port + 1);

    if (a & 0x01) {
//	printf(PPA_ID "    EPP 1.9 with hardware direction protocol\n");
	retv += PPA_PROBE_EPP19;
    } else {
	/* The EPP timeout bit was not set, this could either be:
	 * EPP 1.7
	 * EPP 1.9 with software direction
	 */
	outb(0x24, port + 2);
	inb(port + 4);
	a = inb(port + 1);
	outb(a, port + 1);
	outb(a & 0xfe, port + 1);
	if (a & 0x01) {
//	    printf(PPA_ID "    EPP 1.9 with software direction protocol\n");
	    retv += PPA_PROBE_EPP19;
	    } else {
//	      printf(PPA_ID "    EPP 1.7\n");
	      retv += PPA_PROBE_EPP17;
	    }
    }

    outb(0x0c, port + 0x402);
    outb(0x0c, port + 0x002);
    return retv;
}






















