/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment rotations example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>

#define X 0
#define Y 1
#define Z 2

static PgmViewport *viewport = NULL;
static PgmDrawable *txt[3] = { NULL };
static gboolean rotate[3] = { FALSE };
static gfloat angle[3] = { 0.0f };
static gfloat scale = 1.0f;

/* Update the rotation of the image */
static gboolean
update_pass_cb (PgmViewport *viewport,
                gpointer data)
{
  PgmDrawable *drawable = PGM_DRAWABLE (data);

  pgm_drawable_set_scale (drawable, scale);

  if (rotate[X])
    {
      pgm_drawable_set_rotation_x (drawable, angle[X]);
      angle[X] += 0.05f;
    }
  if (rotate[Y])
    {
      pgm_drawable_set_rotation_y (drawable, angle[Y]);
      angle[Y] += 0.05f;
    }
  if (rotate[Z])
    {
      pgm_drawable_set_rotation_z (drawable, angle[Z]);
      angle[Z] += 0.05f;
    }

  return TRUE;
}

/* scroll-event handler */
static void
scroll_event_cb (PgmViewport *viewport,
                 PgmEventScroll *event,
                 gpointer data)
{
  if (event->direction == PGM_SCROLL_UP)
    scale *= 1.1f;
  else
    scale *= 0.9f;
}

/* key-press handler */
static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  static gboolean fullscreen = FALSE;

  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Handle quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

          /* Fullscreen */
        case PGM_f:
          fullscreen = !fullscreen;
          pgm_viewport_set_fullscreen (viewport, fullscreen);
          break;

        case PGM_x:
        case PGM_X:
          rotate[X] = !rotate[X];
          if (rotate[X])
            pgm_text_set_label (PGM_TEXT (txt[X]), "Press 'x': ON");
          else
            pgm_text_set_label (PGM_TEXT (txt[X]), "Press 'x': OFF");
          break;

        case PGM_y:
        case PGM_Y:
          rotate[Y] = !rotate[Y];
          if (rotate[Y])
            pgm_text_set_label (PGM_TEXT (txt[Y]), "Press 'y': ON");
          else
            pgm_text_set_label (PGM_TEXT (txt[Y]), "Press 'y': OFF");
          break;

        case PGM_z:
        case PGM_Z:
          rotate[Z] = !rotate[Z];
          if (rotate[Z])
            pgm_text_set_label (PGM_TEXT (txt[Z]), "Press 'z': ON");
          else
            pgm_text_set_label (PGM_TEXT (txt[Z]), "Press 'z': OFF");
          break;

        default:
          break;
        }
    }
}

/* Entry point */
int
main (int argc,
      char *argv[])
{
  const gchar *img_path = "examples/pictures/meiko.jpg";
  PgmDrawable *img;
  PgmCanvas *canvas;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_size (viewport, 600, 600);
  pgm_viewport_set_title (viewport, "Rotations");

  /* Image definition */
  img = pgm_image_new_from_file (img_path, 0);
  pgm_drawable_set_size (img, 400, 400);
  pgm_drawable_set_position (img, 100, 100, 0);
  pgm_drawable_set_bg_color (img, 0, 0, 255, 0);
  pgm_drawable_show (img);

  /* Text definitions */
  txt[X] = pgm_text_new ("Press 'x': OFF");
  pgm_text_set_font_height (PGM_TEXT (txt[X]), 20.0f);
  pgm_text_set_weight (PGM_TEXT (txt[X]), PGM_TEXT_WEIGHT_BOLD);
  pgm_drawable_set_size (txt[X], 300, 20);
  pgm_drawable_set_position (txt[X], 10, 0, 0);
  pgm_drawable_set_bg_color (txt[X], 0, 0, 0, 0);
  pgm_drawable_show (txt[X]);
  txt[Y] = pgm_text_new ("Press 'y': OFF");
  pgm_text_set_font_height (PGM_TEXT (txt[Y]), 20.0f);
  pgm_text_set_weight (PGM_TEXT (txt[Y]), PGM_TEXT_WEIGHT_BOLD);
  pgm_drawable_set_size (txt[Y], 300, 20);
  pgm_drawable_set_position (txt[Y], 10, 20, 0);
  pgm_drawable_set_bg_color (txt[Y], 0, 0, 0, 0);
  pgm_drawable_show (txt[Y]);
  txt[Z] = pgm_text_new ("Press 'z': OFF");
  pgm_text_set_font_height (PGM_TEXT (txt[Z]), 20.0f);
  pgm_text_set_weight (PGM_TEXT (txt[Z]), PGM_TEXT_WEIGHT_BOLD);
  pgm_drawable_set_size (txt[Z], 300, 20);
  pgm_drawable_set_position (txt[Z], 10, 40, 0);
  pgm_drawable_set_bg_color (txt[Z], 0, 0, 0, 0);
  pgm_drawable_show (txt[Z]);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 600, 600);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img);
  pgm_canvas_add_many (canvas, PGM_DRAWABLE_NEAR,
                       txt[X], txt[Y], txt[Z], NULL);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "scroll-event",
                    G_CALLBACK (scroll_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  g_signal_connect (G_OBJECT (viewport), "update_pass",
                    G_CALLBACK (update_pass_cb), img);
  pgm_viewport_show (viewport);

  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
