/************************************************************************
*                                                                       *
*                Copyright (C)  2000                                    *
*        University Corporation for Atmospheric Research                *
*                All Rights Reserved                                    *
*                                                                       *
*    The use of this Software is governed by a License Agreement.       *
*                                                                       *
************************************************************************/

/*
 *  $Id: graphcap.h,v 1.5 2008-07-27 03:22:39 haley Exp $
 *
 *	graphcap.h:
 *
 *	Author		John clyne	(clyne@bierstadt.ucar.edu)
 *
 *	Date		Thu Mar  9 16:09:47 MST 1989
 *
 *
 *		This file contains a struct that represets the graphcap.
 *	The order in which the struct is defined is, and MUST be, identical
 *	to that of the order of elements in the binary graphcap. This 
 *	allows the binalry graphcap to be read in a single instruction.
 *	Unfortunately, character strings in the graphcap are represented as 
 *	arrays of ints. So, they must be read in as ints and then converted 
 *	to C strings.
 *		A set of macros is provided to access the graphcap. Clients
 *	who access the graphcap should make use of these macros and NOT the
 *	graphcap itself. 
 *		For further information; see the "NCAR Graphics Installers
 *	Guide" and documentation on "graphc", the graphcap preprocessor.
 *	Don't ask me why the order of layout in the graphcap does not 
 *	agree with the documentation in the "Installers Guide"
 *
 */


/*
 *	Device Class info - define array sizes (CAPLIN)
 */

#define DGIMAX     300
#define DGEMAX     150
#define DTIMAX     100
#define DHCMAX     50
#define DCFTMX     30
#define CORFIN     8 
#define CORFMT     DCFTMX * 4

/*
 *	Polyline Information - define array sizes (CAPLIN)
 */

#define LDSMAX     30
#define LDTMAX     15
#define LMSMAX     30
#define LMTMAX     15
#define LCSMAX     30
#define LCTMAX     15
#define LVCFMX     8
#define LWSMAX     30
#define LWTMAX     15
#define LWTFMX     8
#define LINFIN     8
#define LINFMT     LVCFMX * 4
#define LWTFIN     8
#define LWTFMT     LWTFMX * 4
#define LWTRNG     2
#define LBSMAX     30
#define LBTMAX     15
#define LPSMAX     20
#define LPTMAX     20

/* 
 *	User Prompt String - array sizes (CAPUSR)
 */

#define	UPRMAX     80

/*
 *	Contains device color information - array sizes (CAPCOL)
 */
#define	MAPMAX		256
#define	COLINT		MAPMAX * 3   
#define	COLMAX		15
#define	MSTMAX		50
#define	MTRMAX		20
#define	DMPMAX		50
#define	COLFIN		8
#define	DMPFIN		8
#define	COLFMT		COLMAX * 4
#define	DMPFMT		DMPMAX * 4

/*
 *	TEXT class information  - array sizes (CAPTXT)
 */
#define TCSMAX		10
#define TCTMAX		10
#define TVCFMX		5
#define TXSMAX		20
#define TXTMAX		20
#define TXTFIN		5
#define TXTFMT		TVCFMX * 4

/*
 *	POLYMARKER information - array sizes (CAPMAR)
 */
#define	MCSMAX		30
#define	MCTMAX		15
#define	MVCFMX		5
#define	MRSMAX		20
#define	MRTMAX		10
#define	MARFIN		5
#define	MARFMT		MVCFMX * 4

/*
 *	Polygon class information - array sizes (CAPPLG)
 */
#define PCSMAX		20
#define PCTMAX		15
#define PLSMAX		40
#define PLTMAX		20
#define	PBSMAX		30
#define	PBTMAX		15
#define	PPSMAX		20
#define	PPTMAX		20

/*
 *	Common for the bundle tables - array sizes (CAPBND)
 */
#define  TBLSIZ   6

/*
 *	RASTER class information - array sizes (CAPSCN)
 */
#define SCSMAX     50
#define SCTMAX     50
#define SFMMAX     10
#define SFNMAX     8 
#define SCVFMX     10
#define SCVFIX     8 
#define SCNFMT     SFMMAX * 4
#define SCVFMT     SCVFMX * 4

/*
 *	Dummy graphcap space for future expansion array size (CAPSPC)
 */
#define  DUMSIZ    326

typedef	int	Fchar;
typedef	int	Fbool;

/*
 *	the format of graphcap table is identical to the binary file
 *	generated by the graphcap preprocessor. The order of items
 *	in the table *MUST* agree with the order in which the 
 *	graphcap is written. 
 *	All arrays of floats, chars, and ints are followed by a single
 *	int giving the number of elements used in the array
 *	For convience, comments in () corespond to the name given the 
 *	element by the fortran graphcap preprocessor.
 *	The graphcap is a struct of structs. Each little struct coresponds
 *	to a section described in the graphcap preprocessor documentation
 */
typedef	struct	{

	/*
	 *	Device class info (CAPDEV)
	 */

	Fchar	graphic_init[DGIMAX];	/* graphic init string, (DGISTR)*/
	int	graphic_init_size;	/* size of ^, (DGISIZ)		*/
	Fchar	erase[DGEMAX];		/* device erase string (DGESTR)	*/
	int	erase_size;		/* size of ^, (DGESIZ).		*/
	Fchar	text_init[DTIMAX];	/* text init string, (DTISTR)	*/	
	int	text_init_size;		/* size of ^, (DTISIZ)		*/	

	int	lower_left_x;		/* lower left x coord, (DCDLLX)	*/
	int	lower_left_y;		/* lower left y coord, (DCDLLY)	*/
	int	upper_right_x;		/* upper right x coord, (DCDURX)*/
	int	upper_right_y;		/* upper right y coord, (DCDURY)*/

	Fbool	colour_avail;		/* colour available, (DCOAVL)	*/

	int	coord_format[4][DCFTMX];/* device coordinate format (CORFMT) */	

	int	coord_encoding[CORFIN/2];/* array  describing the device 
					 * coordinates as follows: (CORFIN)
					 *
					 * CORFIN[0] -- DEVICE_COORD_ENCODING
					 * CORFIN[1] -- Number of columns 
					 * 	defined in CORFMT.
					 * CORFIN[2] -- Maximum number of 
					 * 	columns  allowed in CORFMT.
					 * CORFIN[3] -- Num of rows in CORFMT.
					 */

	float	coord_inmin[CORFIN/2];	/* if encoding is ASCII real, then
					 * CORRIN[0] -- Minimum data value 
					 *	input to encoder
					 * CORRIN[1] -- Maximum data value 
					 * input to encoder 
					 * CORRIN[2] -- Minimum data value 
					 *	output from encoder.
					 * CORRIN[3] -- Maximum data value 
					 *	output from encoder.as obtained
					 *	from DEVICE_COORD_FLOATING_INFO
					 */

	Fbool	batch;			/* device batch (BATCH)	*/

	int	cursor_home_size;	/* size of DHCSTR, (DHCSIZ)	*/
	Fchar	cursor_home[DHCMAX];	/* cursor home string, (DHCSTR)	*/
	int	xoffset;		/* x coord offset, (CORXOF)	*/
	int	yoffset;		/* y coord offset, (CORYOF)	*/

	int	dash_bit_length;	/* dash bit length, (DASBIT)	*/
	float	xscale;            	/* device coord xscale, (CORXSC)*/
	float	yscale;            	/* device coord yscale, (CORYSC)*/
	int	dev_win_ll_x,		/* device window lower left x (VDWLLX)*/
		dev_win_ll_y,		/* device window lower left y (VDWLLY)*/
		dev_win_ur_x,		/* device window upper righ x (VDWURX)*/
		dev_win_ur_y;		/* device window upper righ y (VDWURY)*/
} Dev_class;

typedef	struct {

	/*
	 *	line drawing (CAPLIN)
	 */

	Fbool	poly_flag;		/* poly flag, (PLAVBL)		*/
	Fchar	line_draw_start[LDSMAX];/* instruction_start, (LDSSTR)	*/
	int	line_draw_start_size;	/* size of ^, (LDSSIZ)		*/
	Fchar	line_draw_term[LDTMAX];	/* instruction_terminator, (LTDSTR) */
	int	line_draw_term_size;	/* size of ^, (LTDSIZ)		*/
	Fchar	line_move_start[LMSMAX];/* move_instruction_start, (LMSSTR) */
	int	line_move_start_size;	/* size of ^, (LMSSIZ)		*/
	Fchar	line_move_term[LMTMAX];	/* instruction_term, (LMTSTR)	*/
	int	line_move_term_size;	/* size of ^, (LMTSIZ)		*/
	Fchar	line_colour_start[LCSMAX];/* instruction start, (LCSSTR)*/ 
	int	line_colour_start_size;	/* size of ^, (LCSSIZ)		*/ 
	Fchar	line_colour_term[LCTMAX];/* instruction term, (LCTSTR)	*/ 
	int	line_colour_term_size;	/* size of ^, (LCTSIZ)		*/ 

	int	vector_count_encoding[LINFIN/2];     
				/* Array dimensioned for 8 describing the
                              	 * device vector counts as follows: (LINFIN)
				 * LINFIN[0] -- DEVICE_VECTOR_COUNT_ENCODING
				 * LINFIN[1] -- Number of columns defined
				 * 	in LINFMT.
				 * LINFIN[2] -- Maximum number of columns 
				 *	allowed  in LINFMT.
				 * LINFIN[3] -- Number of rows in LINFMT.
				 */

	float vector_count_inmin[LINFIN/2];
				/* if encoding is ASCII real, then
				 * LINRIN[0] -- Minimum data value input to 
				 *	encoder
				 * LINRIN[1] -- Maximum data value input to 
				 *	encoder
				 * LINRIN[2] -- Minimum data value output from
				 * 	encoder.
				 * LINRIN[3] -- Maximum data value output from
				 * 	encoder. obtained from 
				 * DEVICE_VECTOR_COUNT_FLOATING_INFO
				 * 	if encoding is not ASCII real, then
				 * 	LINFIN[0] equals 5 .
				 */

	int	vector_count_format[4][LVCFMX];/* vector count format,(LINFMT)*/

	Fchar	line_width_start[LWSMAX];	/* instruction start, (LWSSTR)*/
	int	line_width_start_size;		/* size of ^, (LWSSIZ)	*/
	Fchar	line_width_term[LWTMAX];	/* instruction start, (LWTSTR)*/
	int	line_width_term_size;		/* size of ^, (LWTSIZ)	*/
	int	line_width_encoding[LWTFIN/2];     
				/*  Array dimensioned for 8 describing the
				 * line widths as follows: (LWTFIN)
				 *
				 * LWTFIN[0] -- LINE_WIDTH_ENCODING
				 * LWTFIN[1] -- Number of columns defined
				 * 	in LWTFMT.
				 * LWTFIN[2] -- Maximum number of columns 
				 *	allowed in LWTFMT.
				 * LWTFIN[3] -- Number of rows in LWTFMT.
				 */

	float line_width_inmin[LWTFIN/2]; 
				/*  if encoding is ASCII real, then
				 * LWTRIN[0] -- Minimum data value input to 
				 * encoder
				 * LWTRIN[1] -- Maximum data value input to 
				 *	encoder
				 * LWTRIN[2] -- Minimum data value output from
				 * 	encoder.
				 * LWTRIN[3] -- Maximum data value output from
				 * 	encoder. obtained from 
				 *	LINE_WIDTH_FLOATING_INFO
				 *	if encoding is not ASCII real, then
				 * 	LWTFIN[4] equals 5 . 
				 */

	int	line_width_format[4][LWTFMX];	/* (LWTFMT)		*/
	int	line_width_min[LWTRNG];	/* Two values specifying the 
					 * line_width_range, (LWTRNG)
					 */
	float	line_width_scale;	/* line width scale, (LWTSCF)	*/
	Fchar	line_back_col_start[LBSMAX];	/* line background color 
						 * start (LBSSTR)
						 */
	int	line_back_col_start_size;	/* size of ^, (LBSSIZ)	*/
	Fchar	line_back_col_term[LBTMAX];	/* line background color 
						 * terminator (LBTSTR)
						 */
	int	line_back_col_term_size;	/* size of ^, (LBTSIZ)	*/

	Fchar	line_point_start[LPSMAX];	/* line point start (LPSSTR) */ 
	int	line_point_start_size;		/* size of ^, (LBSSIZ)	*/
	Fchar	line_point_term[LPTMAX];	/* line point term (LPTSTR)  */
	int	line_point_term_size;		/* size of ^, (LPTSIZ)	*/

} Dev_line;

typedef	struct {
	/* 
	 *	User Prompt String (CAPUSR)
	 */

	Fchar	user_prompt[UPRMAX];	/* user_prompt, (UPRSTR)	*/
	int	user_prompt_size;	/* size of ^, (UPRSIZ)		*/

} Dev_user_prompt;

typedef	struct {
	/*
	 *	Contains device color information (CAPCOL)
	 */

	int	map_init[COLINT]; 	/* Contains the DEVICE_MAP_INIT table 
					 * of color intensities -- COLINT is 
					 * dimensioned for 3 * maximum number 
					 * of colors (COLINT)
					 */
	int	map_init_indexs[MAPMAX];/*  Array of indices associated with 
					 * the table of color intensities 
					 * defined in COLINT above. (COLIDX)
					 */
	int	map_index_defined;	/* map_index_range_defined, (IDXCUR)*/
	int	vdmint;			/* Unused, (VDMINT)		*/
	Fbool	map_avail;		/* device_map_available, (DMPAVL)*/

	int	colour_index_format[4][COLMAX];	/* (COLFMT)		*/  
	int	colour_index_encoding[COLFIN/2];	
				/* Array dimensioned for 8  describing  the 
				 * color indices as follows: (COLFIN)
				 * COLFIN[0] -- DEVICE_COLOR_INDEX_ENCODING
				 * COLFIN[1] -- Number of 4-entry columns 
				 *	defined in COLFMT.
				 * COLFIN[2] -- Maximum number of columns 
				 *	allowed in COLFMT.
				 * COLFIN[3] -- Number of rows in COLFMT.
				 * 	If COLFIN[0] is not equal to 5, then
				 * COLFIN[4] is IFIX[ALOG10[FLOAT[IDXMAX+1]]+1]
				 */
	float	colour_index_inmin[COLFIN/2]; 
				/* if COLFIN[0] equals 5, then
				 * COLRIN[4] -- Minimum data value input to 
				 * 	encoder
				 * COLRIN[5] -- Maximum data value input to 
				 *	encoder
				 * COLRIN[6] -- Minimum data value output from
				 * 	encoder.
				 * [7] -- Maximum data value output from
				 * 	encoder. obtained from 
				 * 	DEVICE_COLOR_INDEX_FLOATING_INFO
				 */

	int	map_index_max;		/* map_index_range_max, (IDXMAX) */
	Fchar	map_start[MSTMAX];  	/* instruction_start, (MSTSTR)	*/
	int	map_start_size;		/* size of ^, (MSTSIZ)		*/
	Fchar	map_term[MTRMAX];	/* instr_terminator, (MTRSTR)	*/
	int	map_model;		/* DEVICE_MAP_MODEL, ((DMPMDL)	*/
	int	map_term_size;		/* size of MTRSTR, (MTRSIZ)	*/  
	Fbool	map_individual;         /* map_individual, (DMPIDV)	*/

	int	map_intensity_encoding[DMPFIN/2];    
				/* Array dimensioned for 8 describing the
				 * device map intensities as follows: (DMPFIN)
				 * DMPFIN[0] -- DEVICE_MAP_INTENSITY_ENCODING
				 * DMPFIN[1] -- Number of 4-entry columns 
				 *	defined in DMPFMT.
				 * DMPFIN[2] -- Maximum number of columns 
				 *	allowed in DMPFMT.
				 * DMPFIN[3] -- Number of rows in DMPFMT.
				 * 	If DMPFIN[0] equals 5, then
				 * DMPFIN[4] is IFIX[ALOG10[FLOAT[INTMAX+1]]+1],
				 * 	where INTMAX is the maximum value in 
				 *	the COLINT array.
				 */

	float map_intensity_inmin[DMPFIN/2]; 
				/*  if DMPFIN[0] equals 5, then
				 * DMPRIN[4] -- Minimum data value input to 
				 *	encoder
				 * DMPRIN[5] -- Maximum data value input to 
				 *	encoder
				 * DMPRIN[6] -- Minimum data value output from
				 * 	encoder.
				 * DMPRIN[7] -- Maximum data value output from
				 * 	encoder. as obtained from
				 * 	DEVICE_MAP_INTENSITY_FLOATING_INFO
				 */
	int   map_intensity_format[4][DMPMAX];
				/* A two-dimensional array [dimensioned for
				 * DMPMAX x 4] containing the 
				 * device_map_intensity_format, (DMPFMT)
				 */
} Dev_colour;

#define	USE_DEV_TEXT
#ifdef	USE_DEV_TEXT
typedef	struct {
	/*
	 *	TEXT class information  (CAPTXT)
	 */

	Fchar	text_colour_start[TCSMAX];/* instr start, (TCSSTR) unused */
	int	text_colour_start_size;		/* size of ^ (TCSSIZ)	*/
	Fchar	text_colour_term[TCTMAX];	/* instr term, (TCTSTR)	*/
	int	text_colour_term_size;		/* size of ^, (TCTSIZ)	*/
	int	text_colour_fmt [4][TVCFMX];
				/* A two-dimensional array [dimensioned for
				 * TVCFMX x 4] containing the    
				 * TEXT_VECTOR_COUNT_FORMAT
				 */
	int	text_colour_encoding[TXTFIN];
				/* Array dimensioned for 5 describing the
				 * text vector count format as follows:
				 * TXTFIN[0] -- TEXT_VECTOR_COUNT_ENCODING
				 * TXTFIN[1] -- Number of columns defined
				 * 	in TXTFMT.
				 * TXTFIN[2] -- Currently not used.
				 * TXTFIN[3] -- Currently not used.
				 */
	Fchar	text_start[TXSMAX];	/* instruction_start, (TXSSTR)	*/
	int	text_start_size;	/* size of ^, (TXSSIZ)		*/
	Fchar	text_term[TXTMAX];	/* instruction_start, (TXTSTR)	*/
	int	text_term_size;		/* size of ^, (TXTSIZ)		*/

} Dev_text;
#endif	/*	USE_DEV_TEXT	*/


#define	USE_DEV_MARKER
#ifdef	USE_DEV_MARKER
typedef	struct {
	/*
	 *	POLYMARKER information (CAPMAR)
	 *	Unused (polymarkers are software simulated)
	 */

	Fchar	marker_colour_start[MCSMAX];	/* instr_start, (MCSSTR) */
	int	marker_colour_start_size;	/* size of ^, (MCSSIZ)	*/
	Fchar	marker_colour_term[MCTMAX];	/* instr_start, (MCTSTR) */
	int	marker_colour_term_size;	/* size of ^, (MCTSIZ)	*/
	int	marker_vector_count_format [MARFIN];       
				/* Describing the marker vector counts as 
				 * follows: (MARFIN)
				 * MARFIN[0] -- MARKER_VECTOR_COUNT_ENCODING
				 * MARFIN[1] -- Number of columns defined
				 * in MARFMT.
				 * MARFIN[2] -- Currently undefined.
				 * MARFIN[3] -- Currently undefined.
				 * MARFIN[4] -- Currently undefined.
				 */

	int	marker_vector_count_encoding[4][MVCFMX];
				/* A two-dimensional array [dimensioned for
				 * MVCFMX x 4] containing the 
				 * MARKER_VECTOR_COUNT_ENCODING
				 */
	Fchar	marker_start[MRSMAX];	/* instruction_start, (MRSSTR)	*/
	int	marker_start_siz;	/* size of ^, (MRSSIZ)		*/
	Fchar	marker_term[MRTMAX];	/* instruction_term, (MRTSTR)	*/
	int	marker_term_size;	/* size of ^, (MRTSIZ)		*/
	int	marker_dot_size;	/* polymarker dot size (MDOTSZ)	*/

} Dev_marker;
#endif	/* USE_DEV_MARKER	*/

typedef	struct {

/*
 *	Polygon class information (CAPPLG)
 */

	Fchar	polygon_colour_start[PCSMAX];	/* instr_start, (PCSSTR)*/
	int	polygon_colour_start_size;	/* size of ^, (PCSSIZ)	*/
	Fchar	polygon_colour_term[PCTMAX];	/* instr_start, (PCTSTR)*/
	int	polygon_colour_term_size;	/* size of ^, (PCTSIZ)	*/
  
	Fchar	polygon_start[PLSMAX];	/* instr_start, (PLSSTR)*/
	int	polygon_start_size;	/* size of ^, (PLSSIZ)	*/
	Fchar	polygon_term[PLTMAX];	/* instr_start, (PLTSTR)*/
	int	polygon_term_size;	/* size of ^, (PLTSIZ)	*/


	Fchar	poly_back_col_start[PBSMAX];	/* poly background color 
						 * start (PBSSTR)
						 */
	int	poly_back_col_start_size;	/* size of ^, (PBSSIZ)	*/
	Fchar	poly_back_col_term[PBTMAX];	/* poly background color 
						 * terminator (PBTSTR)
						 */
	int	poly_back_col_term_size;	/* size of ^, (PBTSIZ)	*/


	int	poly_hatch_space;	/* poly hatch spacing (PHATSP)	*/
	int	poly_max_points;	/* poly maximum points (PMAXPT)	*/

	Fchar	poly_point_start[PPSMAX];/* poly point start (PPSSTR)	*/ 
	int	poly_point_start_size;	/* size of ^, (PPSSIZ)		*/
	Fchar	poly_point_term[PPTMAX];/* poly point term (PPTSTR)	*/
	int	poly_point_term_size;	/* size of ^, (PPTSIZ)		*/

	Fbool	polygon_simulate;	/* polygon simulate (PLGSIM)	*/
	int	poly_sim_spacing;	/* poly simulate spacng (PSIMSP)*/
	int	poly_sim_trunc;		/* poly sim truncation (PSIMTR)	*/

} Dev_polygon;

#define	USE_DEV_BUNDLE
#ifdef	USE_DEV_BUNDLE
typedef	struct {

/*
 *	Common for the bundle tables (CAPBND)
 */


	int	num_polyline_bundles;	/* number of tables, (PLBTEC)	*/
	int	polyline_bundles[TBLSIZ];/* indices for POLYLINE bdl. tables.*/

	int	polyline_type_bndl[TBLSIZ]; 
					/* Array of POLYLINE LINETYPE
					 * bundle table entries. (PLTBTB)
					 */
	float	polyline_linewidth_bndl[TBLSIZ];
					/* Array of POLYLINE LINEWIDTH
					 * bundle table entries. (PLWBTB)
					 */
	int	polyline_colour_bndl[TBLSIZ];
					/* Array of POLYLINE COLOR INDEX
					 * bundle table entries. (PLCBTB)
					 */
	int	num_polymarker_bundles;
					/* Number of POLYMARKER bundle tables.
					 * (PMBTEC)
					 */
	int	polymarker_bundles[TBLSIZ];	
					/* Array of indices for POLYMARKER
					 * bundle tables. (PMIBTB)
					 */
	int	polymarker_type_bndl[TBLSIZ];
					/* Array of POLYMARKER MARKERTYPE
					 * bundle table entries. (PMTBTB)
					 */
	float	polymarker_size_bndl[TBLSIZ];
					/* Array of POLYMARKER MARKERSIZE
					 * bundle table entries. (PMSBTB)
					 */
	int	polymarker_colour_bndl[TBLSIZ];
					/* Array of POLYMARKER COLOR INDEX
					 * bundle tbl. entries. (PMCBTB)
					 */
	int	num_text_bundles;	/* Number of TEXT bundle tables.
					 * (TXBTEC)
					 */
	int	text_bundles[TBLSIZ];	/* Array of indices for TEXT bundle 
					 * tables, (TXIBTB)
					 */ 
	int	text_font_bndl[TBLSIZ];	/* Array of TEXT FONT bundle table 
					 * entries. (TXFBTB)
					 */
	int	text_prec_bndl[TBLSIZ];	/* Array of TEXT PRECISION
					 * bundle table entries (TXPBTB)
					 */
	float	char_expan_bndl[TBLSIZ];/* Array of CHARACTER EXPANSION FACTOR
					 * table entries. (TCXBTB)
					 */
	float	char_spacing_bndl[TBLSIZ];
					/* Array of CHARACTER SPACING
					 * bundle table entries. (TCSBTB)
					 */
	int	text_colour_index_bndl[TBLSIZ];
					/* Array of TEXT COLOR INDEX
					 * bundle table entries (TXCBTB)
					 */
	int	num_fill_area_bundles;	/* Number of FILL AREA bundle tables.
					 * (FABTEC)
					 */
	int	fill_area_bundles[TBLSIZ];/* Array of indices for FILL AREA
					 * bundle tables (FAIBTB)
					 */
	int	fill_interior_style_bndl[TBLSIZ];
					/* Array of FILL AREA INTERIOR STYLE
					 * bundle table entries. (FISBTB)
					 */
	int	fill_style_index_bndl[TBLSIZ];
					/* Array of FILL AREA STYLE INDEX
					 * bundle table entries. (FASBTB)
					 */
	int	fill_colour_bndl[TBLSIZ];/* Array of FILL AREA COLOR INDEX
					 * bundle table entries. (FACBTB)
					 */ 
} Dev_bundle;
#endif	/* USE_DEV_BUNDLE	*/
 
typedef	struct {
	/*
	 *	RASTER class information (CAPSCN)
	 */
	Fchar	raster_hor_start[SCSMAX];/* horizontal start,(SCSSTR)	*/
	int	raster_hor_start_size;	/* size of ^ (SCSSIZ)		*/
	Fchar	raster_hor_term[SCTMAX];/* horizontal  term, (SCTSTR)	*/
	int	raster_hor_term_size;	/* size of ^ (SCTSIZ)		*/

	int	raster_lower_left_x;	/* (SCNLLX)			*/
	int	raster_lower_left_y;	/* (SCNLLY)			*/
	int	raster_upper_right_x;	/* (SCNURX)			*/
	int	raster_upper_right_y;	/* (SCNURY)			*/
	int	raster_xoffset;		/* x offset (SCNXOF)		*/
	int	raster_yoffset;		/* y offset (SCNYOF)		*/
	float	raster_xscale;		/* x scale (SCNXSC)		*/
	float	raster_yscale;		/* y scale (SCNYSC)		*/

	int	raster_data_format[4][SFMMAX];
				/* A two-dimensional array  [dimensioned for 
				 * SFMMAX x 4]  containing the 
				 * RASTER_DATA_FORMAT (SCNFMT)
				 */
	int	raster_data_encoding[SFNMAX/2];
				/* Array dimensioned for 8 describing the   
				 * raster data format as follows: (SCNFIN)
				 * SCNFIN[0] -- RASTER_DATA_ENCODING         
				 * SCNFIN[1] -- Number of columns defined     
				 * 	in SCNFMT.                              
				 * SCNFIN[2] -- Maximum number of columns 
				 * 	in SCNFMT.                              
				 * SCNFIN[3] -- Number of rows in SCNFMT.   */                       
	float	raster_data_inmin[SFNMAX/2];
				/* if encoding is ASCII real, then          
				 * SCNRIN[4] -- Minimum data value input to 
				 * 	encoder     
				 * SCNRIN[5] -- Maximum data value input to 
				 *	encoder     
				 * SCNRIN[6] -- Minimum data value output 
				 *	from encoder.
				 * SCNRIN[7] -- Maximum data value output 
				 *	from encoder obtained from 
				 *	RASTER_DATA_FLOATING_INFO
				 */

	int	raster_vector_count_format[4][SCVFMX];
				/* A two-dimensional array [dimensioned for 
				 * SCVFMX x 4] containing the 
				 * RASTER_VECTOR_COUNT_FORMAT
				 */
	int	raster_vector_count_encoding[SCVFIX/2];
				/* Array dimensioned for 8 describing the
				 * raster vector count format as follows:
				 * SCVFIN[0] -- RASTER_VECTOR_COUNT_ENCODING
				 * SCVFIN[1] -- Number of columns defined
				 * 	in SCVFMT.
				 * SCVFIN[2] -- Maximum number of columns 
				 *	allowed in SCVFMT.
				 * SCVFIN[3] -- Number of rows in SCVFMT
				 */

	float	raster_vector_count_inmin[SCVFIX/2];
				/* if encoding is ASCII real, then
				 * SCVRIN[4] -- Minimum data value input to 
				 *	encoder
				 * SCVRIN[5] -- Maximum data value input to 
				 *	encoder
				 * SCVRIN[6] -- Minimum data value output from
				 * 	encoder.
				 * SCVRIN[7] -- Maximum data value output from
				 * 	encoder.  
				 * obtained from 
				 * RASTER_VECTOR_COUNT_FLOATING_INFO
				 */
	Fbool	raster_simulate;	/* (SCNSIM)		*/

} Dev_raster;


#ifdef	USE_FREE_SPACE	
typedef	struct {

	/*
	 *	Dummy graphcap space for future expansion array size (CAPSPC)
	 */
	int	free_space[DUMSIZ];	/* Dummy for future use */
	int	end_space;		/* end of graphcap ?	*/
} Free_space;
#endif	/* USE_FREE_SPACE	*/

/********************************************************************
 *				The Graphcap
 ********************************************************************/
typedef struct	{
	Dev_class	dev_class;
	Dev_line	dev_line;
	Dev_user_prompt	dev_user_prompt;
	Dev_colour	dev_colour;

#ifdef	USE_DEV_TEXT
	Dev_text	dev_text;
#endif

#ifdef	USE_DEV_MARKER
	Dev_marker	dev_marker;
#endif

	Dev_polygon	dev_poly;

#ifdef	USE_DEV_BUNDLE
	Dev_bundle	dev_bundle;
#endif

	Dev_raster	dev_raster;

#ifdef	USE_FREE_SPACE	
	Free_space	free_space;
#endif

} Graphcap;





/*
 *	define Graphcap graphcap in gcread.c only. Else define graphcap
 *	as external
 */
#define	GCAP_SIZE	sizeof (Graphcap)

#ifdef	GCREAD
	Graphcap	graphcap;
#else
	extern	Graphcap	graphcap;
#endif


/*
 *
 *	Graphcap Macros.
 *
 *		Use these macros to acces the graphcap. Do NOT access
 *	the graphcap struct itself! 
 */
#define	GRAPHIC_INIT		((char *) graphcap.dev_class.graphic_init)
#define	GRAPHIC_INIT_SIZE	graphcap.dev_class.graphic_init_size		
#define	ERASE			((char *) graphcap.dev_class.erase)
#define	ERASE_SIZE		graphcap.dev_class.erase_size
#define	TEXT_INIT		((char *) graphcap.dev_class.text_init)
#define	TEXT_INIT_SIZE		graphcap.dev_class.text_init_size			
#define	LOWER_LEFT_X		graphcap.dev_class.lower_left_x			
#define	LOWER_LEFT_Y		graphcap.dev_class.lower_left_y			
#define	UPPER_RIGHT_X		graphcap.dev_class.upper_right_x
#define	UPPER_RIGHT_Y		graphcap.dev_class.upper_right_y
#define COLOUR_AVAIL		graphcap.dev_class.colour_avail


#define	COORD_FORMAT		graphcap.dev_class.coord_format
#define COORD_ENCODING		graphcap.dev_class.coord_encoding[0]
#define COORD_FORMAT_SIZE	graphcap.dev_class.coord_encoding[1]
#define	COORD_INMIN		graphcap.dev_class.coord_inmin[0]
#define	COORD_INMAX		graphcap.dev_class.coord_inmin[1]
#define	COORD_OUTMIN		graphcap.dev_class.coord_inmin[2]
#define	COORD_OUTMAX		graphcap.dev_class.coord_inmin[3]
#define	BATCH			graphcap.dev_class.batch

#define	CURSOR_HOME_SIZE	graphcap.dev_class.cursor_home_size
#define	CURSOR_HOME		((char *) graphcap.dev_class.cursor_home)
#define	XOFFSET			graphcap.dev_class.xoffset
#define	YOFFSET			graphcap.dev_class.yoffset
#define	DASH_BIT_LENGTH		graphcap.dev_class.dash_bit_length
#define XSCALE			graphcap.dev_class.xscale
#define YSCALE			graphcap.dev_class.yscale

#define	DEV_WIN_LL_X		graphcap.dev_class.dev_win_ll_x
#define	DEV_WIN_LL_Y		graphcap.dev_class.dev_win_ll_y
#define	DEV_WIN_UR_X		graphcap.dev_class.dev_win_ur_x
#define	DEV_WIN_UR_Y		graphcap.dev_class.dev_win_ur_y

/*
 *      line drawing (CAPLIN)
 */

#define	POLY_FLAG		graphcap.dev_line.poly_flag
#define	LINE_DRAW_START		((char *) graphcap.dev_line.line_draw_start)
#define	LINE_DRAW_START_SIZE	graphcap.dev_line.line_draw_start_size
#define	LINE_DRAW_TERM		((char *) graphcap.dev_line.line_draw_term)
#define	LINE_DRAW_TERM_SIZE	graphcap.dev_line.line_draw_term_size
#define	LINE_MOVE_START		((char *) graphcap.dev_line.line_move_start)
#define	LINE_MOVE_START_SIZE	graphcap.dev_line.line_move_start_size
#define	LINE_MOVE_TERM		((char *) graphcap.dev_line.line_move_term)
#define	LINE_MOVE_TERM_SIZE	graphcap.dev_line.line_move_term_size

#define	LINE_BACK_COL_START	((char *) graphcap.dev_line.line_back_col_start)
#define	LINE_BACK_COL_START_SIZE graphcap.dev_line.line_back_col_start_size
#define	LINE_BACK_COL_TERM	((char *) graphcap.dev_line.line_back_col_term)
#define	LINE_BACK_COL_TERM_SIZE graphcap.dev_line.line_back_col_term_size

#define	LINE_POINT_START	((char *) graphcap.dev_line.line_point_start)
#define	LINE_POINT_START_SIZE	graphcap.dev_line.line_point_start_size
#define	LINE_POINT_TERM		((char *) graphcap.dev_line.line_point_term)
#define	LINE_POINT_TERM_SIZE	graphcap.dev_line.line_point_term_size

#define LINE_COLOUR_START	((char *) graphcap.dev_line.line_colour_start)
#define LINE_COLOUR_START_SIZE	graphcap.dev_line.line_colour_start_size
#define LINE_COLOUR_TERM	((char *) graphcap.dev_line.line_colour_term)
#define LINE_COLOUR_TERM_SIZE	graphcap.dev_line.line_colour_term_size	

#define VECTOR_COUNT_ENCODING	graphcap.dev_line.vector_count_encoding[0]
#define VECTOR_COUNT_FORMAT_SIZE	graphcap.dev_line.vector_count_encoding[1]

#define	VECTOR_COUNT_INMIN	graphcap.dev_line.vector_count_inmin[0]
#define	VECTOR_COUNT_INMAX	graphcap.dev_line.vector_count_inmin[1]
#define	VECTOR_COUNT_OUTMIN	graphcap.dev_line.vector_count_inmin[2]
#define	VECTOR_COUNT_OUTMAX	graphcap.dev_line.vector_count_inmin[3]
#define	VECTOR_COUNT_FORMAT	graphcap.dev_line.vector_count_format

	/*
	 *	Line Width stuff
	 */
#define LINE_WIDTH_START	((char *) graphcap.dev_line.line_width_start)
#define LINE_WIDTH_START_SIZE	graphcap.dev_line.line_width_start_size
#define LINE_WIDTH_TERM		((char *) graphcap.dev_line.line_width_term)
#define LINE_WIDTH_TERM_SIZE	graphcap.dev_line.line_width_term_size

#define LINE_WIDTH_ENCODING	graphcap.dev_line.line_width_encoding[0]
#define LINE_WIDTH_FORMAT_SIZE	graphcap.dev_line.line_width_encoding[1]

#define	LINE_WIDTH_INMIN	graphcap.dev_line.line_width_inmin[0]
#define	LINE_WIDTH_INMAX	graphcap.dev_line.line_width_inmin[1]
#define	LINE_WIDTH_OUTMIN	graphcap.dev_line.line_width_inmin[2]
#define	LINE_WIDTH_OUTMAX	graphcap.dev_line.line_width_inmin[3]
#define LINE_WIDTH_FORMAT	graphcap.dev_line.line_width_format

#define LINE_WIDTH_MIN	   	graphcap.dev_line.line_width_min[0]
#define LINE_WIDTH_MAX	   	graphcap.dev_line.line_width_min[1]
#define LINE_WIDTH_SCALE	graphcap.dev_line.line_width_scale


/*
 *	User Prompt String
 */

#define USER_PROMPT		((char *) graphcap.dev_user_prompt.user_prompt)
#define USER_PROMPT_SIZE	graphcap.dev_user_prompt.user_prompt_size

/*
 *	Colour information
 */

#define MAP_INIT		graphcap.dev_colour.map_init
#define MAP_INIT_INDEXS		graphcap.dev_colour.map_init_indexs
#define MAP_INDEX_DEFINED	graphcap.dev_colour.map_index_defined
#define MAP_AVAIL		graphcap.dev_colour.map_avail
#define COLOUR_INDEX_FORMAT	graphcap.dev_colour.colour_index_format
#define COLOUR_INDEX_ENCODING	graphcap.dev_colour.colour_index_encoding[0]
#define COLOUR_INDEX_FORMAT_SIZE graphcap.dev_colour.colour_index_encoding[1]

#define	COLOUR_INDEX_INMIN	graphcap.dev_colour.colour_index_inmin[0]
#define	COLOUR_INDEX_INMAX	graphcap.dev_colour.colour_index_inmin[1]
#define	COLOUR_INDEX_OUTMIN	graphcap.dev_colour.colour_index_inmin[2]
#define	COLOUR_INDEX_OUTMAX	graphcap.dev_colour.colour_index_inmin[3]

#define MAP_INDEX_MAX		graphcap.dev_colour.map_index_max
#define MAP_START		((char *) graphcap.dev_colour.map_start)
#define MAP_START_SIZE		graphcap.dev_colour.map_start_size
#define MAP_TERM		((char *) graphcap.dev_colour.map_term)
#define MAP_TERM_SIZE		graphcap.dev_colour.map_term_size
#define MAP_MODEL		graphcap.dev_colour.map_model

#define MAP_INDIVIDUAL		graphcap.dev_colour.map_individual

#define MAP_INTENSITY_ENCODING	graphcap.dev_colour.map_intensity_encoding[0]
#define MAP_INTENSITY_FORMAT_SIZE	graphcap.dev_colour.map_intensity_encoding[1]

#define	MAP_INTENSITY_INMIN	graphcap.dev_colour.map_intensity_inmin[0]
#define	MAP_INTENSITY_INMAX	graphcap.dev_colour.map_intensity_inmin[1]
#define	MAP_INTENSITY_OUTMIN	graphcap.dev_colour.map_intensity_inmin[2]
#define	MAP_INTENSITY_OUTMAX	graphcap.dev_colour.map_intensity_inmin[3]

#define MAP_INTENSITY_FORMAT	graphcap.dev_colour.map_intensity_format

/*
 *	Text stuff - software simulated
 */

/*
 *	Polymarker stuff - software simulated
 */
#define	MARKER_DOT_SIZE		graphcap.dev_marker.marker_dot_size

/*
 *	Polygon stuff
 */
#define POLYGON_COLOUR_START	((char *) graphcap.dev_poly.polygon_colour_start)
#define POLYGON_COLOUR_START_SIZE	graphcap.dev_poly.polygon_colour_start_size
#define POLYGON_COLOUR_TERM	((char *) graphcap.dev_poly.polygon_colour_term)
#define POLYGON_COLOUR_TERM_SIZE graphcap.dev_poly.polygon_colour_term_size
#define POLYGON_START		((char *) graphcap.dev_poly.polygon_start)
#define POLYGON_START_SIZE	graphcap.dev_poly.polygon_start_size
#define POLYGON_TERM		((char *) graphcap.dev_poly.polygon_term)
#define POLYGON_TERM_SIZE	graphcap.dev_poly.polygon_term_size

#define	POLY_BACK_COL_START	((char *) graphcap.dev_poly.poly_back_col_start)
#define	POLY_BACK_COL_START_SIZE graphcap.dev_poly.poly_back_col_start_size
#define	POLY_BACK_COL_TERM	((char *) graphcap.dev_poly.poly_back_col_term)
#define	POLY_BACK_COL_TERM_SIZE graphcap.dev_poly.poly_back_col_term_size

#define	POLY_HATCH_SPACE	graphcap.dev_poly.poly_hatch_space
#define	POLY_MAX_POINTS		graphcap.dev_poly.poly_max_points

#define	POLY_POINT_START	((char *) graphcap.dev_poly.poly_point_start)
#define	POLY_POINT_START_SIZE graphcap.dev_poly.poly_point_start_size
#define	POLY_POINT_TERM	((char *) graphcap.dev_poly.poly_point_term)
#define	POLY_POINT_TERM_SIZE graphcap.dev_poly.poly_point_term_size

#define	POLYGON_SIMULATE	graphcap.dev_poly.polygon_simulate
#define	POLY_SIM_SPACING	graphcap.dev_poly.poly_sim_spacing
#define	POLY_SIM_TRUNC		graphcap.dev_poly.poly_sim_trunc


/*
 *	Bundle table stuff - not supported
 */

 
/*
 *	Raster stuff
 */
#define	RASTER_HOR_START	((char *) graphcap.dev_raster.raster_hor_start)
#define	RASTER_HOR_START_SIZE	graphcap.dev_raster.raster_hor_start_size
#define	RASTER_HOR_TERM		((char *) graphcap.dev_raster.raster_hor_term)
#define	RASTER_HOR_TERM_SIZE	graphcap.dev_raster.raster_hor_term_size

#define	RASTER_LOWER_LEFT_X	graphcap.dev_raster.raster_lower_left_x
#define	RASTER_LOWER_LEFT_Y	graphcap.dev_raster.raster_lower_left_y
#define	RASTER_UPPER_RIGHT_X	graphcap.dev_raster.raster_upper_right_x
#define	RASTER_UPPER_RIGHT_Y	graphcap.dev_raster.raster_upper_right_y
#define	RASTER_XOFFSET		graphcap.dev_raster.raster_xoffset
#define	RASTER_YOFFSET		graphcap.dev_raster.raster_yoffset
#define RASTER_XSCALE		graphcap.dev_raster.raster_xscale
#define RASTER_YSCALE		graphcap.dev_raster.raster_yscale

#define	RASTER_DATA_FORMAT	graphcap.dev_raster.raster_data_format
#define RASTER_DATA_ENCODING	graphcap.dev_raster.raster_data_encoding[0]
#define RASTER_DATA_FORMAT_SIZE	graphcap.dev_raster.raster_data_encoding[1]

#define RASTER_DATA_INMIN	graphcap.dev_raster.raster_data_inmin[0]
#define RASTER_DATA_INMAX 	graphcap.dev_raster.raster_data_inmin[1]
#define RASTER_DATA_OUTMAX	graphcap.dev_raster.raster_data_inmin[2]
#define RASTER_DATA_OUTMIN	graphcap.dev_raster.raster_data_inmin[3]

#define	RASTER_VECTOR_COUNT_FORMAT	graphcap.dev_raster.raster_vector_count_format
#define RASTER_VECTOR_COUNT_ENCODING	graphcap.dev_raster.raster_vector_count_encoding[0]
#define RASTER_VECTOR_COUNT_FORMAT_SIZE graphcap.dev_raster.raster_vector_count_encoding[1]

#define RASTER_VECTOR_COUNT_INMIN	graphcap.dev_raster.raster_vector_count_inmin[0] 
#define RASTER_VECTOR_COUNT_INMAX	graphcap.dev_raster.raster_vector_count_inmin[1]
#define RASTER_VECTOR_COUNT_OUTMAX	graphcap.dev_raster.raster_vector_count_inmin[2]
#define RASTER_VECTOR_COUNT_OUTMIN	graphcap.dev_raster.raster_vector_count_inmin[3]

#define	RASTER_SIMULATE		graphcap.dev_raster.raster_simulate	
