﻿/*
 * Copyright (c) 2021 Belledonne Communications SARL.
 *
 * This file is part of linphone-desktop
 * (see https://www.linphone.org).
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include "ConferenceInfoModel.hpp"

#include <algorithm>

#include <QDateTime>
#include <QDesktopServices>
#include <QElapsedTimer>
#include <QFileInfo>
#include <QMimeDatabase>
#include <QTimer>
#include <QUuid>
#include <QMessageBox>
#include <QUrlQuery>
#include <QImageReader>
#include <qqmlapplicationengine.h>

#include "app/App.hpp"
#include "app/paths/Paths.hpp"
#include "app/providers/ThumbnailProvider.hpp"
#include "components/calls/CallsListModel.hpp"
#include "components/chat-events/ChatCallModel.hpp"
#include "components/chat-events/ChatEvent.hpp"
#include "components/chat-events/ChatMessageModel.hpp"
#include "components/chat-events/ChatNoticeModel.hpp"
#include "components/contact/ContactModel.hpp"
#include "components/contact/VcardModel.hpp"
#include "components/contacts/ContactsListModel.hpp"
#include "components/conferenceScheduler/ConferenceScheduler.hpp"
#include "components/core/CoreHandlers.hpp"
#include "components/core/CoreManager.hpp"
#include "components/notifier/Notifier.hpp"
#include "components/other/timeZone/TimeZoneModel.hpp"
#include "components/settings/AccountSettingsModel.hpp"
#include "components/settings/SettingsModel.hpp"
#include "components/participant/ParticipantModel.hpp"
#include "components/participant/ParticipantListModel.hpp"
#include "components/presence/Presence.hpp"
#include "components/recorder/RecorderManager.hpp"
#include "components/recorder/RecorderModel.hpp"
#include "components/timeline/TimelineModel.hpp"
#include "components/timeline/TimelineListModel.hpp"
#include "components/core/event-count-notifier/AbstractEventCountNotifier.hpp"
#include "utils/QExifImageHeader.hpp"
#include "utils/Utils.hpp"
#include "utils/Constants.hpp"
#include "utils/LinphoneEnums.hpp"



// =============================================================================

using namespace std;

// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
QSharedPointer<ConferenceInfoModel> ConferenceInfoModel::create(std::shared_ptr<linphone::ConferenceInfo> conferenceInfo){
	return QSharedPointer<ConferenceInfoModel>::create(conferenceInfo);
}

// Callable from QML
ConferenceInfoModel::ConferenceInfoModel (QObject * parent) : QObject(parent){
	mTimeZone = QTimeZone::systemTimeZone();
	mConferenceInfo = linphone::Factory::get()->createConferenceInfo();
	QDateTime currentDateTime = QDateTime::currentDateTime();
	QDateTime utc = currentDateTime.addSecs( -mTimeZone.offsetFromUtc(currentDateTime));
	mConferenceInfo->setDateTime(0);
	mConferenceInfo->setDuration(0);
	mIsScheduled = false;
	auto accountAddress = CoreManager::getInstance()->getCore()->getDefaultAccount()->getContactAddress();
	if(accountAddress){
		auto cleanedClonedAddress = accountAddress->clone();
		cleanedClonedAddress->clean();
		mConferenceInfo->setOrganizer(cleanedClonedAddress);
	}
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::timeZoneModelChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::dateTimeChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::durationChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::organizerChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::subjectChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::descriptionChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::participantsChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::uriChanged);// Useless but just in case.
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::isScheduledChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::inviteModeChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::conferenceInfoStateChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::conferenceSchedulerStateChanged);
}

// Callable from C++
ConferenceInfoModel::ConferenceInfoModel (std::shared_ptr<linphone::ConferenceInfo> conferenceInfo, QObject * parent) : QObject(parent){
	App::getInstance()->getEngine()->setObjectOwnership(this, QQmlEngine::CppOwnership);// Avoid QML to destroy it when passing by Q_INVOKABLE
	mTimeZone = QTimeZone::systemTimeZone();
	mConferenceInfo = conferenceInfo;
	mIsScheduled = (mConferenceInfo->getDateTime() != 0 || mConferenceInfo->getDuration() != 0);
	
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::timeZoneModelChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::dateTimeChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::durationChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::organizerChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::subjectChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::descriptionChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::participantsChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::uriChanged);// Useless but just in case.
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::isScheduledChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::inviteModeChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::conferenceInfoStateChanged);
	connect(this, &ConferenceInfoModel::conferenceInfoChanged, this, &ConferenceInfoModel::conferenceSchedulerStateChanged);
}

ConferenceInfoModel::~ConferenceInfoModel () {
}

std::shared_ptr<linphone::ConferenceInfo> ConferenceInfoModel::getConferenceInfo(){
	return mConferenceInfo;
}

std::shared_ptr<linphone::ConferenceInfo> ConferenceInfoModel::findConferenceInfo(const std::shared_ptr<const linphone::ConferenceInfo> & conferenceInfo){
	return CoreManager::getInstance()->getCore()->findConferenceInformationFromUri(conferenceInfo->getUri()->clone());
}

//------------------------------------------------------------------------------------------------

//Note conferenceInfo->getDateTime uses system timezone and fromMSecsSinceEpoch need a UTC
QDateTime ConferenceInfoModel::getDateTimeSystem() const{
	QDateTime reference(QDateTime::fromMSecsSinceEpoch(mConferenceInfo->getDateTime() * 1000));// Get a reference for timezone offset computing
	qint64 utcMs = (mConferenceInfo->getDateTime() - QTimeZone::systemTimeZone().offsetFromUtc(reference)) * 1000;// Remove system timezone offset to get UTC
	return QDateTime::fromMSecsSinceEpoch(utcMs, QTimeZone::systemTimeZone());	// Return a System Timezone datetime based
}

QDateTime ConferenceInfoModel::getDateTimeUtc() const{
	return getDateTimeSystem().toUTC();
}

int ConferenceInfoModel::getDuration() const{
	return mConferenceInfo->getDuration();
}

QDateTime ConferenceInfoModel::getEndDateTime() const{
	return getDateTimeUtc().addSecs(getDuration()*60);
}

QString ConferenceInfoModel::getOrganizer() const{
	return Utils::coreStringToAppString(mConferenceInfo->getOrganizer()->asString());
}

QString ConferenceInfoModel::getSubject() const{
	return Utils::coreStringToAppString(mConferenceInfo->getSubject());
}

QString ConferenceInfoModel::getDescription() const{
	return Utils::coreStringToAppString(mConferenceInfo->getDescription());
}

QString ConferenceInfoModel::displayNamesToString()const{
	QStringList txt;
	for(auto participant : mConferenceInfo->getParticipants()){
		if(participant){
			QString displayName = Utils::getDisplayName(participant);
			if(displayName != "")
				txt << displayName;
		}
	}
	return txt.join(", ");
}

QString ConferenceInfoModel::getUri() const{
	auto address = mConferenceInfo->getUri();
	return address->isValid() && !address->getDomain().empty() ? QString::fromStdString(address->asStringUriOnly()) : "";
}

bool ConferenceInfoModel::isScheduled() const{
	return mIsScheduled;
}

int ConferenceInfoModel::getInviteMode() const{
	return mInviteMode;
}

QVariantList ConferenceInfoModel::getParticipants() const{
	QVariantList addresses;
	for(auto item : mConferenceInfo->getParticipants()){
		QVariantMap participant;
		participant["displayName"] = Utils::getDisplayName(item);
		participant["address"] = QString::fromStdString(item->asStringUriOnly());
		addresses << participant;
	}
	return addresses;
}
QVariantList ConferenceInfoModel::getAllParticipants() const{
	QVariantList addresses = getParticipants();
	QString organizerAddress = QString::fromStdString(mConferenceInfo->getOrganizer()->asStringUriOnly());
	for(auto item : addresses){
		if( item.toMap()["address"] == organizerAddress)
			return addresses;
	}
	QVariantMap participant;
	participant["displayName"] = Utils::getDisplayName(mConferenceInfo->getOrganizer());
	participant["address"] = organizerAddress;
	addresses << participant;
	return addresses;
}


int ConferenceInfoModel::getParticipantCount()const{
	return mConferenceInfo->getParticipants().size();
}

int ConferenceInfoModel::getAllParticipantCount()const{
	return getAllParticipants().size();
}

TimeZoneModel* ConferenceInfoModel::getTimeZoneModel() const{
	TimeZoneModel * model = new TimeZoneModel(mTimeZone);
	App::getInstance()->getEngine()->setObjectOwnership(model, QQmlEngine::JavaScriptOwnership);
	return model;
}

QString ConferenceInfoModel::getIcalendarString() const{
	return Utils::coreStringToAppString(mConferenceInfo->getIcalendarString());
}

LinphoneEnums::ConferenceInfoState ConferenceInfoModel::getConferenceInfoState() const{
	return LinphoneEnums::fromLinphone(mConferenceInfo->getState());
}

LinphoneEnums::ConferenceSchedulerState ConferenceInfoModel::getConferenceSchedulerState() const{
	return LinphoneEnums::fromLinphone(mLastConferenceSchedulerState);
}

//------------------------------------------------------------------------------------------------
// Datetime is in Custom (Locale/UTC/System). Convert into system timezone for conference info
void ConferenceInfoModel::setDateTime(const QDateTime& dateTime){
	QDateTime system = dateTime.toTimeZone(QTimeZone::systemTimeZone());//System
	int offset = QTimeZone::systemTimeZone().offsetFromUtc(system);//Get UTC offset in system coordinate
	system = system.addSecs( offset - mTimeZone.offsetFromUtc(dateTime));// Delta on offsets
	mConferenceInfo->setDateTime(system.toMSecsSinceEpoch() / 1000 + offset);// toMSecsSinceEpoch() is UTC, add system reference.
	
	emit dateTimeChanged();
}

void ConferenceInfoModel::setDuration(const int& duration){
	mConferenceInfo->setDuration(duration);
	emit durationChanged();
}

void ConferenceInfoModel::setSubject(const QString& subject){
	mConferenceInfo->setSubject(Utils::appStringToCoreString(subject));
	emit subjectChanged();
}

void ConferenceInfoModel::setOrganizer(const QString& organizerAddress){
	mConferenceInfo->setOrganizer(Utils::interpretUrl(organizerAddress));
	emit organizerChanged();
}

void ConferenceInfoModel::setDescription(const QString& description){
	mConferenceInfo->setDescription(Utils::appStringToCoreString(description));
	emit descriptionChanged();
}

void ConferenceInfoModel::setParticipants(ParticipantListModel * participants){
	mConferenceInfo->setParticipants(participants->getParticipants());
	emit participantsChanged();
}

void ConferenceInfoModel::setTimeZoneModel(TimeZoneModel * model){
	if( mTimeZone != model->getTimeZone()){
		mTimeZone = model->getTimeZone();
		emit timeZoneModelChanged();
	}
}

void ConferenceInfoModel::setIsScheduled(const bool& on){
	if( mIsScheduled != on){
		mIsScheduled = on;
		if(!mIsScheduled){
			mConferenceInfo->setDateTime(0);
			mConferenceInfo->setDuration(0);
		}else{
			mTimeZone = QTimeZone::systemTimeZone();
			QDateTime currentDateTime = QDateTime::currentDateTime();
			QDateTime utc = currentDateTime.addSecs( -mTimeZone.offsetFromUtc(currentDateTime));
			mConferenceInfo->setDateTime(utc.toMSecsSinceEpoch() / 1000);
			mConferenceInfo->setDuration(1200);
		}
		emit dateTimeChanged();
		emit durationChanged();
		emit isScheduledChanged();
	}
}

void ConferenceInfoModel::setInviteMode(const int& mode){
	if( mode != mInviteMode){
		mInviteMode = mode;
		emit inviteModeChanged();
	}
}

void ConferenceInfoModel::setConferenceInfo(std::shared_ptr<linphone::ConferenceInfo> conferenceInfo){
	mConferenceInfo = conferenceInfo;
	mIsScheduled = (mConferenceInfo->getDateTime() != 0 || mConferenceInfo->getDuration() != 0);
	emit conferenceInfoChanged();
}

//-------------------------------------------------------------------------------------------------

void ConferenceInfoModel::resetConferenceInfo() {
	mConferenceInfo = linphone::Factory::get()->createConferenceInfo();
	mConferenceInfo->setDateTime(0);
	mConferenceInfo->setDuration(0);
	mIsScheduled = false;
	auto accountAddress = CoreManager::getInstance()->getCore()->getDefaultAccount()->getContactAddress();
	if(accountAddress){
		auto cleanedClonedAddress = accountAddress->clone();
		cleanedClonedAddress->clean();
		mConferenceInfo->setOrganizer(cleanedClonedAddress);
	}
}

void ConferenceInfoModel::createConference(const int& securityLevel) {
	CoreManager::getInstance()->getTimelineListModel()->mAutoSelectAfterCreation = false;
	shared_ptr<linphone::Core> core = CoreManager::getInstance()->getCore();
	static std::shared_ptr<linphone::Conference> conference;
	qInfo() << "Conference creation of " << getSubject() << " at " << securityLevel << " security, organized by " << getOrganizer() << " for " << getDateTimeSystem().toString();
	qInfo() << "Participants:";
	for(auto p : mConferenceInfo->getParticipants())
		qInfo() << "\t" << p->asString().c_str();
	
	mConferenceScheduler = ConferenceScheduler::create();
	mConferenceScheduler->mSendInvite = mInviteMode;
	connect(mConferenceScheduler.get(), &ConferenceScheduler::invitationsSent, this, &ConferenceInfoModel::onInvitationsSent);
	connect(mConferenceScheduler.get(), &ConferenceScheduler::stateChanged, this, &ConferenceInfoModel::onConferenceSchedulerStateChanged);
	mConferenceScheduler->getConferenceScheduler()->setInfo(mConferenceInfo);
}

void ConferenceInfoModel::cancelConference(){
	mConferenceScheduler = ConferenceScheduler::create();
	connect(mConferenceScheduler.get(), &ConferenceScheduler::invitationsSent, this, &ConferenceInfoModel::onInvitationsSent);
	connect(mConferenceScheduler.get(), &ConferenceScheduler::stateChanged, this, &ConferenceInfoModel::onConferenceSchedulerStateChanged);
	mConferenceScheduler->getConferenceScheduler()->cancelConference(mConferenceInfo);
}

void ConferenceInfoModel::deleteConferenceInfo(){
	if(mConferenceInfo) {
		CoreManager::getInstance()->getCore()->deleteConferenceInformation(mConferenceInfo);
		emit removed(true);
	}
}

//-------------------------------------------------------------------------------------------------

void ConferenceInfoModel::onConferenceSchedulerStateChanged(linphone::ConferenceScheduler::State state){
	qDebug() << "ConferenceInfoModel::onConferenceSchedulerStateChanged: " << (int) state;
	mLastConferenceSchedulerState = state;
	if( state == linphone::ConferenceScheduler::State::Ready)
		emit conferenceCreated();
	else if( state == linphone::ConferenceScheduler::State::Error)
		emit conferenceCreationFailed();
	emit conferenceInfoChanged();
}
void ConferenceInfoModel::onInvitationsSent(const std::list<std::shared_ptr<linphone::Address>> & failedInvitations) {
	qDebug() << "ConferenceInfoModel::onInvitationsSent";
	emit invitationsSent();
}
