/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "entry.hpp"
#include "diary.hpp"
#include "lifeograph.hpp"


using namespace LIFEO;


// STATIC MEMBERS
ElementShower< Entry >  *Entry::shower( NULL );

Entry::Entry( DiaryBase * const d, const Date::date_t date,const Glib::ustring &text,
              bool flag_favored )
:   DiaryElement( d ), m_date( date ),
    m_date_created( time( NULL ) ), m_date_changed( m_date_created ),
    m_text( text ), m_ptr2theme( NULL ), m_option_favored( flag_favored ),
    m_option_lang( LANG_INHERIT_DIARY ),
    m_option_trashed( false ), m_flag_filteredout( false )
{
    calculate_title( text );
}

Entry::Entry( DiaryBase * const d, const Date::date_t date, bool flag_favored )
:   DiaryElement( d, _( STRING::EMPTY_ENTRY_TITLE ) ), m_date( date ),
    m_date_created( time( NULL ) ), m_date_changed( m_date_created ),
    m_text( "" ), m_ptr2theme( NULL ), m_option_favored( flag_favored ),
    m_option_lang( LANG_INHERIT_DIARY ),
    m_option_trashed( false ), m_flag_filteredout( false )
{
}

const Glib::RefPtr< Gdk::Pixbuf >&
Entry::get_icon( void ) const
{
    return( m_option_favored ?
            Lifeograph::icons->entry_favorite_16 : Lifeograph::icons->entry_16 );
}
const Glib::RefPtr< Gdk::Pixbuf >&
Entry::get_icon32( void ) const
{
    return( m_option_favored ?
            Lifeograph::icons->entry_favorite_32 : Lifeograph::icons->entry_32 );
}

void
Entry::calculate_title( const Glib::ustring &text )
{
    if( text.size() < 1 )
    {
        m_name = _( STRING::EMPTY_ENTRY_TITLE );
        return;
    }
    unsigned int l_pos = text.find( '\n', 0 );
    if( l_pos == std::string::npos )
        m_name = text;
    else
        m_name = text.substr( 0, l_pos );
}

Glib::ustring
Entry::get_date_created_str( void ) const
{
    return Date::format_string( m_date_created );
}

Glib::ustring
Entry::get_date_changed_str( void ) const
{
    return Date::format_string( m_date_changed );
}

void
Entry::set_tags( const Tagset &tagset )
{
    m_tags = tagset;
}

const Tagset&
Entry::get_tags( void ) const
{
    return m_tags;
}

Tagset&
Entry::get_tags( void )
{
    return m_tags;
}

bool
Entry::add_tag( Tag *tag )
{
    if( m_tags.add( tag ) )
    {
        tag->get_items()->insert( this );
        return true;
    }
    else
        return false;
}

bool
Entry::remove_tag( Tag *tag )
{
    if( tag == NULL )
        return false;

    m_tags.erase( tag );
    tag->get_items()->erase( this );

    return true;
}

bool
Entry::get_filtered_out( void )
{
     Diary::FilteringStatus filtering_status = Diary::d->get_filtering_status();

    if( filtering_status & Diary::FS_NEW )  // recalculate
    {
        //m_flag_filteredout = false;

        m_flag_filteredout =
                ( bool( filtering_status & Diary::FS_FILTER_TRASH ) ^ m_option_trashed );

        // FS_NEW alone means not applied clear
        if( m_flag_filteredout || filtering_status == Diary::FS_NEW )
            return m_flag_filteredout;

        // cheapest first: favorites filter
        if( filtering_status & Diary::FS_FILTER_FAVORITES )
            if( ! m_option_favored )
                return( m_flag_filteredout = true ); // no need to continue, already filtered out

        if( filtering_status & Diary::FS_FILTER_DATE )
            if( ( m_date.m_date < Diary::d->get_filter_date_begin() ) ||
                ( m_date.m_date > Diary::d->get_filter_date_end() ) )
                return( m_flag_filteredout = true );

        if( filtering_status & Diary::FS_FILTER_TAG )
            if( ! m_tags.checkfor_member( Diary::d->get_filter_tag() ) )
                return( m_flag_filteredout = true ); // no need to continue, already filtered out

        if( filtering_status & Diary::FS_FILTER_TEXT )
            m_flag_filteredout =
                    ( m_text.lowercase().find( Diary::d->get_filter_text() ) == std::string::npos );
    }

    return m_flag_filteredout;
}

void
Entry::show( void )
{
    if( shower != NULL )
        shower->show( *this );
    else
        PRINT_DEBUG( "Entry has no graphical data!" );
}

void
Entry::prepare_for_hiding( void )
{
    if( shower != NULL )
        shower->prepare_for_hiding( *this );
    else
        PRINT_DEBUG( "Entry has no graphical data!" );
}

// ENTRY SET ===================================================================
PoolEntries::~PoolEntries()
{
    for( iterator iter = begin(); iter != end(); ++iter )
        delete iter->second;
}

void
PoolEntries::clear( void )
{
    for( iterator iter = begin(); iter != end(); ++iter )
        delete iter->second;

    std::map< Date::date_t, Entry*, FuncCompareDates >::clear();
}
