/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <QtGui/QLabel>

#include "container.h"
#include "field.h"
#include "fieldfactory.h"

using namespace Form;

struct FieldEntry
{
  QLabel *nameLabel;
  Field *field;
  QLabel *commentLabel;
};

class Container::Private
{
  public:
    QString name;
    QGridLayout* layout;
    QList<FieldEntry> fields;
    QWidget* baseWidget;
};

Container::Container( const QString &name, QWidget *parent )
  : QScrollArea( parent ), d( new Private )
{
  d->name = name;
  d->baseWidget = new QWidget;
  d->layout = new QGridLayout( d->baseWidget );
  d->layout->setSizeConstraint( QLayout::SetFixedSize );
  d->layout->setMargin( 11 );
  d->layout->setSpacing( 6 );

  setWidget( d->baseWidget );
  setWidgetResizable( true );
//  d->baseWidget->resize( d->baseWidget->sizeHint() );
}

Container::~Container()
{
  delete d;
  d = 0;
}

void Container::build( const Description *description, const PagePolicy *policy, PagePolicy::State state )
{
  Description::Entry::List entries = description->entries();

  int rowCounter = 0;
  for ( int i = 0; i < entries.count(); ++i ) {
    if ( entries[ i ].isGroup() ) {
      QLabel *title = new QLabel( entries[ i ].label(), d->baseWidget );

      QFont font = title->font();
      font.setBold( true );
      font.setPointSize( font.pointSize() + 5 );
      title->setFont( font );

      d->layout->addItem( new QSpacerItem( 0, 20 ), rowCounter, 0 );
      rowCounter++;

      d->layout->addWidget( title, rowCounter, 0, 1, 3 );
      rowCounter++;
    } else {
      FieldEntry entry;

      if ( !policy->fieldVisible( state, entries[ i ].name() ) )
        continue;

      entry.field = FieldFactory::self()->createField( entries[ i ].type(), entries[ i ].name(), d->baseWidget );
      if ( !entry.field )
        continue;

      entry.field->setDefaultValue( entries[ i ].defaultValue() );

      entry.nameLabel = new QLabel( entries[ i ].label(), d->baseWidget );
      entry.commentLabel = new QLabel( entries[ i ].comment(), d->baseWidget );
      entry.commentLabel->setWordWrap( true );

      if ( !entries[ i ].header().isEmpty() ) {
        QLabel *headerLabel = new QLabel( entries[ i ].header(), d->baseWidget );
        //headerLabel->setMargin( 10 );
        headerLabel->setWordWrap( true );
        d->layout->addWidget( headerLabel, rowCounter, 0, 1, 3 );
        rowCounter++;
      }

      d->layout->addWidget( entry.nameLabel, rowCounter, 0, 1, 1, Qt::AlignTop | Qt::AlignLeft );
      d->layout->addWidget( entry.field, rowCounter, 1, 1, 1, Qt::AlignTop | Qt::AlignLeft );
      d->layout->addWidget( entry.commentLabel, rowCounter, 2, 1, 1, Qt::AlignTop | Qt::AlignLeft );

      rowCounter++;

      if ( !policy->fieldEditable( state, entries[ i ].name() ) )
        entry.field->setEditable( false );

      d->fields.append( entry );
    }
  }

  d->layout->setRowStretch( rowCounter, 1 );
}

QString Container::name() const
{
  return d->name;
}

QGridLayout* Container::containerLayout() const
{
  return d->layout;
}

void Container::loadEntry( const Entry &entry )
{
  for ( int i = 0; i < d->fields.count(); ++i )
    d->fields[ i ].field->loadEntry( entry );
}

void Container::saveEntry( Entry &entry ) const
{
  for ( int i = 0; i < d->fields.count(); ++i )
    d->fields[ i ].field->saveEntry( entry );
}

bool Container::isValid( QString &message, PagePolicy::State state ) const
{
  bool valid = true;

  QString fieldMessage;
  for ( int i = 0; i < d->fields.count(); ++i ) {
    QString msg;
    if ( !d->fields[ i ].field->isValid( msg, state ) ) {
      fieldMessage.append( QString( "\n%1" ).arg( msg ) );
      valid = false;
    }
  }

  if ( !valid )
    message = fieldMessage;

  return valid;
}

void Container::setEditable( bool editable )
{
  for ( int i = 0; i < d->fields.count(); ++i ) {
    d->fields[ i ].field->setEditable( editable );
    d->fields[ i ].commentLabel->setVisible( editable );
  }
}
