/*
 *  $Id: laplace.c 24796 2022-04-28 15:20:59Z yeti-dn $
 *  Copyright (C) 2004 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/stats.h>
#include <libprocess/correct.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>

#define LAPLACE_RUN_MODES GWY_RUN_IMMEDIATE

static gboolean module_register(void);
static void     laplace        (GwyContainer *data,
                                GwyRunType run);
static void     zeromasked     (GwyContainer *data,
                                GwyRunType run);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Removes data under mask, "
       "interpolating them with Laplace equation solution."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "2.1",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, laplace)

static gboolean
module_register(void)
{
    gwy_process_func_register("laplace",
                              (GwyProcessFunc)&laplace,
                              N_("/_Correct Data/_Interpolate Data Under Mask"),
                              GWY_STOCK_REMOVE_UNDER_MASK,
                              LAPLACE_RUN_MODES,
                              GWY_MENU_FLAG_DATA_MASK | GWY_MENU_FLAG_DATA,
                              N_("Interpolate data under mask by solution of "
                                 "Laplace equation"));
    gwy_process_func_register("zeromasked",
                              (GwyProcessFunc)&zeromasked,
                              N_("/_Correct Data/_Zero Data Under Mask"),
                              GWY_STOCK_ZERO_UNDER_MASK,
                              LAPLACE_RUN_MODES,
                              GWY_MENU_FLAG_DATA_MASK | GWY_MENU_FLAG_DATA,
                              N_("Fill data under mask with zeros"));

    return TRUE;
}


static void
laplace(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *mfield;
    GQuark dquark;
    gint id;

    g_return_if_fail(run & LAPLACE_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_KEY, &dquark,
                                     GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield && dquark && mfield);
    gwy_app_undo_qcheckpointv(data, 1, &dquark);
    gwy_data_field_laplace_solve(dfield, mfield, -1, 1.0);
    gwy_data_field_data_changed(dfield);
    gwy_app_channel_log_add_proc(data, id, id);
}

static void
zeromasked(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *mfield;
    gint xres, yres;
    GQuark dquark;
    gint id;

    g_return_if_fail(run & LAPLACE_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_KEY, &dquark,
                                     GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield && dquark && mfield);
    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    gwy_app_undo_qcheckpointv(data, 1, &dquark);
    gwy_data_field_area_fill_mask(dfield, mfield, GWY_MASK_INCLUDE,
                                  0, 0, xres, yres, 0.0);
    gwy_data_field_data_changed(dfield);
    gwy_app_channel_log_add_proc(data, id, id);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
