/*
 *  $Id: vtkfile.c 22626 2019-10-30 09:14:30Z yeti-dn $
 *  Copyright (C) 2012-2018 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-USERGUIDE]
 * VTK structured grid file
 * .vtk
 * Export
 **/

/**
 * [FILE-MAGIC-MISSING]
 * Export only.  Avoding clash with a standard file format.
 **/

#include "config.h"
#include <math.h>
#include <locale.h>
#include <string.h>
#include <glib/gstdio.h>
#include <libgwyddion/gwymacros.h>
#include <libgwymodule/gwymodule-file.h>
#include <libprocess/stats.h>
#include <app/gwyapp.h>

#include "err.h"

#define EXTENSION ".vtk"

typedef struct {
    gdouble zscale;
    gboolean autoscale;
} VTKExportArgs;

typedef struct {
    VTKExportArgs *args;
    gdouble autoscale_value;
    GtkWidget *dialogue;
    GtkWidget *autoscale;
    GtkWidget *zscale;
    GtkWidget *zscale_label;
} VTKExportControls;

static gboolean module_register    (void);
static gint     vtk_detect         (const GwyFileDetectInfo *fileinfo,
                                    gboolean only_name);
static gboolean vtk_export         (GwyContainer *data,
                                    const gchar *filename,
                                    GwyRunType mode,
                                    GError **error);
static gboolean vtk_export_dialogue(VTKExportArgs *args,
                                    gdouble autoscale);
static void     zscale_changed     (GtkEntry *entry,
                                    VTKExportControls *controls);
static void     autoscale_changed  (GtkToggleButton *check,
                                    VTKExportControls *controls);
static void     update_zscale_entry(VTKExportControls *controls);
static void     update_sensitivity (VTKExportControls *controls);
static void     load_args          (GwyContainer *settings,
                                    VTKExportArgs *args);
static void     save_args          (GwyContainer *settings,
                                    const VTKExportArgs *args);

static const VTKExportArgs vtkfile_defaults = {
    1.0, TRUE,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Exports data as VTK structured grids."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2012",
};

GWY_MODULE_QUERY2(module_info, vtkfile)

static gboolean
module_register(void)
{
    gwy_file_func_register("vtk",
                           N_("VTK structured grid (.vtk)"),
                           (GwyFileDetectFunc)&vtk_detect,
                           NULL,
                           NULL,
                           (GwyFileSaveFunc)&vtk_export);

    return TRUE;
}

static gint
vtk_detect(const GwyFileDetectInfo *fileinfo,
           G_GNUC_UNUSED gboolean only_name)
{
    return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 30 : 0;
}

static inline gint
print_with_decimal_dot(FILE *fh,
                       gchar *formatted_number,
                       const gchar *decimal_dot,
                       guint decimal_dot_len)
{
    gchar *pos = strstr(formatted_number, decimal_dot);

    if (!pos)
        return fputs(formatted_number, fh);
    else {
        pos[0] = '.';
        if (decimal_dot_len == 1)
            return fputs(formatted_number, fh);
        else {
            gint l1, l2;

            pos[1] = '\0';
            if ((l1 = fputs(formatted_number, fh)) == EOF)
                return EOF;
            if ((l2 = fputs(pos + decimal_dot_len, fh)) == EOF)
                return EOF;
            return l1 + l2;
        }
    }
}

static gdouble
make_autoscale(GwyDataField *dfield)
{
    gdouble min, max;
    gint xres, yres;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    gwy_data_field_get_min_max(dfield, &min, &max);

    return (max == min) ? 0.0 : 0.2*sqrt(xres*yres)/(max - min);
}

static gboolean
vtk_export(G_GNUC_UNUSED GwyContainer *data,
           const gchar *filename,
           GwyRunType mode,
           GError **error)
{
    VTKExportArgs args;
    const guchar *title = NULL;
    GwyDataField *dfield;
    guint xres, yres, i, j, decimal_dot_len;
    gint id;
    struct lconv *locale_data;
    const gchar *decimal_dot;
    gdouble q, min, autoscale;
    gboolean ok;
    gdouble *d;
    gchar buf[40];
    FILE *fh;

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);

    if (!dfield) {
        err_NO_CHANNEL_EXPORT(error);
        return FALSE;
    }

    load_args(gwy_app_settings_get(), &args);
    autoscale = make_autoscale(dfield);
    if (args.autoscale)
        args.zscale = autoscale;

    gwy_container_gis_string(data, gwy_app_get_data_title_key_for_id(id),
                             &title);
    if (!title)
        title = _("Untitled");

    if (mode == GWY_RUN_INTERACTIVE) {
        ok = vtk_export_dialogue(&args, autoscale);
        save_args(gwy_app_settings_get(), &args);
        if (!ok) {
            err_CANCELLED(error);
            return FALSE;
        }
    }

    locale_data = localeconv();
    decimal_dot = locale_data->decimal_point;
    g_return_val_if_fail(decimal_dot, FALSE);
    decimal_dot_len = strlen(decimal_dot);
    g_return_val_if_fail(decimal_dot_len, FALSE);

    if (!(fh = gwy_fopen(filename, "w"))) {
        err_OPEN_WRITE(error);
        return FALSE;
    }

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);

    /* Do not bother checking errors here.  If some write fails we will get
     * more errors below. */
    gwy_fprintf(fh, "# vtk DataFile Version 2.0\n");
    gwy_fprintf(fh, "%s\n", title);
    gwy_fprintf(fh, "ASCII\n");
    gwy_fprintf(fh, "DATASET STRUCTURED_GRID\n");
    gwy_fprintf(fh, "DIMENSIONS %u %u 1\n", xres, yres);
    gwy_fprintf(fh, "POINTS %u float\n", xres*yres);

    d = gwy_data_field_get_data(dfield);
    min = gwy_data_field_get_min(dfield);
    q = args.zscale;

    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++, d++) {
            g_snprintf(buf, sizeof(buf), "%u %u %.6g\n", i, j, q*(*d - min));
            if (print_with_decimal_dot(fh, buf,
                                       decimal_dot, decimal_dot_len) == EOF)
                goto fail;
        }
    }
    fclose(fh);

    return TRUE;

fail:
    err_WRITE(error);
    fclose(fh);
    g_unlink(filename);

    return FALSE;
}

static gboolean
vtk_export_dialogue(VTKExportArgs *args,
                    gdouble autoscale)
{
    VTKExportControls controls;
    GtkWidget *dialogue, *table;
    gint response, row = 0;

    controls.args = args;
    controls.autoscale_value = autoscale;

    dialogue = gtk_dialog_new_with_buttons(_("Export VTK"), NULL, 0,
                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                           GTK_STOCK_OK, GTK_RESPONSE_OK,
                                           NULL);
    controls.dialogue = dialogue;
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_file_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);

    table = gtk_table_new(2, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 12);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), table,
                       TRUE, TRUE, 0);
    row = 0;

    controls.autoscale
        = gtk_check_button_new_with_mnemonic(_("_Automatic Z-scale"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.autoscale),
                                 args->autoscale);
    gtk_table_attach(GTK_TABLE(table), controls.autoscale,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls.autoscale, "toggled",
                     G_CALLBACK(autoscale_changed), &controls);
    row++;

    controls.zscale_label = gtk_label_new_with_mnemonic(_("Ph_ysical scale:"));
    gtk_misc_set_alignment(GTK_MISC(controls.zscale_label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), controls.zscale_label,
                     0, 1, row, row+1, GTK_FILL, 0, 0, 0);


    controls.zscale = gtk_entry_new();
    gtk_entry_set_width_chars(GTK_ENTRY(controls.zscale), 9);
    gtk_table_attach(GTK_TABLE(table), controls.zscale,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(controls.zscale_label),
                                  controls.zscale);
    update_zscale_entry(&controls);
    update_sensitivity(&controls);
    g_signal_connect(controls.zscale, "changed",
                     G_CALLBACK(zscale_changed), &controls);
    row++;

    gtk_widget_show_all(dialogue);
    response = gtk_dialog_run(GTK_DIALOG(dialogue));
    if (response != GTK_RESPONSE_NONE)
        gtk_widget_destroy(dialogue);

    return response == GTK_RESPONSE_OK;
}

static void
zscale_changed(GtkEntry *entry, VTKExportControls *controls)
{
    controls->args->zscale = g_strtod(gtk_entry_get_text(entry), NULL);
}

static void
autoscale_changed(GtkToggleButton *check, VTKExportControls *controls)
{
    VTKExportArgs *args = controls->args;

    args->autoscale = gtk_toggle_button_get_active(check);
    if (args->autoscale) {
        args->zscale = controls->autoscale_value;
        update_zscale_entry(controls);
    }
    update_sensitivity(controls);
}

static void
update_zscale_entry(VTKExportControls *controls)
{
    gchar *s = g_strdup_printf("%.8g", controls->args->zscale);

    gtk_entry_set_text(GTK_ENTRY(controls->zscale), s);
    g_free(s);
}

static void
update_sensitivity(VTKExportControls *controls)
{
    gboolean sens = !controls->args->autoscale;

    gtk_widget_set_sensitive(controls->zscale_label, sens);
    gtk_widget_set_sensitive(controls->zscale, sens);
}

static const gchar autoscale_key[] = "/module/vtkfile/user_zscale";
static const gchar zscale_key[]    = "/module/vtkfile/zscale";

static void
load_args(GwyContainer *settings,
          VTKExportArgs *args)
{
    *args = vtkfile_defaults;

    gwy_container_gis_boolean_by_name(settings, autoscale_key,
                                      &args->autoscale);
    gwy_container_gis_double_by_name(settings, zscale_key, &args->zscale);
}

static void
save_args(GwyContainer *settings,
          const VTKExportArgs *args)
{
    gwy_container_set_boolean_by_name(settings, autoscale_key,
                                      args->autoscale);
    gwy_container_set_double_by_name(settings, zscale_key, args->zscale);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
