/*
 * Copyright (c) 2011 Paulo Zanoni
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <cstdlib>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <iostream>

extern "C" {
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
}

#include "XDGUtils.h"

XDGUtils::XDGLogLevel XDGUtils::logLevel_ = Quiet;

bool XDGUtils::validSemicolonList(const char *list)
{
    size_t len;

    // Last char must be unescaped ';'
    len = strlen(list);
    if (list[len-1] != ';' || list[len-2] == '\\')
        return false;

    return true;
}

char *XDGUtils::escapedStrTok_r(char *s, const char *delim,
				char **ptrptr)
{
    char *checkedString;
    int i;

    if (s)
	checkedString = s;
    else
	checkedString = *ptrptr;

    for(i = 0; checkedString[i] != '\0'; i++) {
	if (checkedString[i] == '\\')
	    i++;
	if (strchr(delim, checkedString[i])) {
	    checkedString[i] = '\0';
	    *ptrptr = &checkedString[i+1];
	    return checkedString;
	}
    }
    return NULL;
}

bool XDGUtils::findFileInPath(const char *fileName,
			      std::string& fullPath)
{
    char *path = strdup(getenv("PATH"));
    char *tokStr, *tokPtr;

    ITERATE_STRTOK_R(path, ":", &tokPtr, tokStr) {
	std::string tmp(std::string(tokStr) + '/' + fileName);
	if (fileExists(tmp.c_str())) {
	    fullPath = tmp;
	    free(path);
	    return true;
	}
    }
    free(path);
    return false;
}

bool XDGUtils::fileExists(const char *filePath)
{
    struct stat statBuffer;
    return (stat(filePath, &statBuffer) == 0);
}

void XDGUtils::setLogLevel(XDGLogLevel logLevel)
{
    logLevel_ = logLevel;
}

void XDGUtils::log(XDGLogLevel logLevel, const char *fmt, ...)
{
    va_list argp;
    va_start(argp, fmt);

    if (logLevel_ >= logLevel)
        vprintf(fmt, argp);
    va_end(argp);
}
