#ifndef  __PREFIX_H
#define  __PREFIX_H

#define CC_PROTOTYPE_ANSI

#include <stdio.h>
#include <stdlib.h>

#endif  /* __PREFIX_H */
/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*                      PROTOTYPES FOR FILES IN UTIL                       */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/


#ifndef __UTIL_H
#define __UTIL_H



/***************************************************************************/
/*                                                                         */
/*                             allocrus.c                                  */
/*                                                                         */
/***************************************************************************/

/***************************************************************************/
/*                                                                         */
/*                   MEMORY ALLOCATION MACROS                              */
/*                                                                         */
/*                           TSP CODE                                      */
/*                                                                         */
/*                                                                         */
/*  Written by:  Applegate, Bixby, Chvatal, and Cook                       */
/*  Date: February 24, 1995 (cofeb24)                                      */
/*                                                                         */
/*                                                                         */
/*  EXPORTED MACROS:                                                       */
/*    CC_SAFE_MALLOC (nnum,type)                                           */
/*         int nnum (the number of objects to be malloced)                 */
/*         data type (the sort of objects to be malloced)                  */
/*         RETURNS a pointer to the allocated space. If out of memory,     */
/*                 it prints an error message and returns NULL.            */
/*                                                                         */
/*    CC_FREE (object,type)                                                */
/*         type *object (pointer to previously allocated space)            */
/*         data type (the sort of object)                                  */
/*         ACTION: frees the memory and sets the object to NULL.           */
/*                                                                         */
/*    CC_IFFREE (object,type)                                              */
/*         type *object (pointer to previously allocated space)            */
/*         data type (the sort of object)                                  */
/*         ACTION: if *object is not NULL, frees the memory and sets       */
/*                 the object to NULL.                                     */
/*                                                                         */
/*    CC_PTR_ALLOC_ROUTINE (type, functionname, chunklist, freelist)       */
/*         data type (the sort of objects)                                 */
/*         string functionname (the generated function)                    */
/*         CCbigchunkptr *chunklist (used to accumulate bigchunks)         */
/*         type *freelist (used for the linked list of objects)            */
/*         ACTION: Generates a function ("functionname") that returns      */
/*                 (type *) objects, keeping the free ones on freelist     */
/*                 and getting its space from calls to bigchunkalloc.      */
/*                                                                         */
/*    CC_PTR_FREE_ROUTINE (type, functionname, freelist)                   */
/*         Parameters as above.                                            */
/*         ACTION: Generates a function that adds an object to the         */
/*                 freelist.                                               */
/*                                                                         */
/*    CC_PTR_FREE_LIST_ROUTINE (type, functionname, freefunction)          */
/*         Parameters defined as above, with freefunction the function     */
/*         generated by PTR_FREE_ROUTINE.                                  */
/*         ACTION: Generates a function to free a linked list of           */
/*                 objects using calls to freefunction.                    */
/*                                                                         */
/*    CC_PTR_FREE_WORLD_ROUTINE( type, functionname, chunklist, freelist)  */
/*         Parameters defined as above.                                    */
/*         ACTION: Generates a function that returns all of the            */
/*                 memory used in the PTR_ALLOC_ROUTINE allocations        */
/*                 back to the global supply of CCbigchunkptrs.            */
/*                                                                         */
/*    CC_PTR_LEAKS_ROUTINE (type, name, chunklist, freelist, field,        */
/*            fieldtype)                                                   */
/*          As above, with "field" the name of a "fieldtype" field in the  */
/*          object type that can be set to 0 or to 1.                      */
/*          ACTION: Generates a function that checks to see that we have   */
/*                  not leaked any of the objects.                         */
/*                                                                         */
/*    CC_PTR_STATUS_ROUTINE (type, name, chunklist, freelist)              */
/*          ACTION: Like LEAKS, but does not check for duplicates (and so  */
/*                  does not corrupt the objects).                         */
/*                                                                         */
/*  NOTES:                                                                 */
/*     These routines use the functions in allocrus.c.  The PTR macros     */
/*  The PTR macros generate the functions for allocating objects for       */
/*  linked lists. They get their raw memory from the bigchunk supply, so   */
/*  so foo_free_world (geneated by PTR_FREE_WORLD_ROUTINE) should be       */
/*  called for each type of linked object "foo" when closing down the      */
/*  local memory.                                                          */
/*     To use these functions, put the macros near the top of the file     */
/*  before any calls to the functions (since the macros also write the     */
/*  function prototypes). If you use PTR_FREE_LIST_ROUTINE for foo, you    */
/*  must also use PTR_FREE_ROUTINE, and PTR_FREE_LIST_ROUTINE must be      */
/*  listed after CC_PTR_FREE_ROUTINE (to get the prototype).               */
/*                                                                         */
/***************************************************************************/


#define CC_SAFE_MALLOC(nnum,type)                                          \
    (type *) CCutil_allocrus (((unsigned int) (nnum)) * sizeof (type))

#define CC_FREE(object,type) {                                             \
    CCutil_freerus ((void *) (object));                                    \
    object = (type *) NULL;                                                \
}

#define CC_IFFREE(object,type) {                                           \
    if ((object)) CC_FREE ((object),type);                                 \
}

#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_ALLOC_ROUTINE(type, functionname)                    \
static type * functionname (void);                                         \
static type * functionname (void)
#else
#define CC_HEADER_PTR_ALLOC_ROUTINE(type, functionname)                    \
static type * functionname ();                                             \
static type * functionname ()
#endif

#define CC_PTR_ALLOC_ROUTINE(type, functionname, chunklist, freelist)      \
static  type * freelist = ( type * ) NULL;                                 \
static  CCbigchunkptr * chunklist = ( CCbigchunkptr * ) NULL;              \
 CC_HEADER_PTR_ALLOC_ROUTINE (type, functionname)                          \
{                                                                          \
    type *p;                                                               \
                                                                           \
    if (! freelist ) {                                                     \
        int count = CC_BIGCHUNK / sizeof ( type );                         \
        CCbigchunkptr *bp;                                                 \
                                                                           \
        bp = CCutil_bigchunkalloc ();                                      \
        if (!bp) {                                                         \
            fprintf (stderr, "ptr alloc failed\n");                        \
            return ( type * ) NULL;                                        \
        }                                                                  \
        freelist = ( type * ) bp->this;                                    \
        bp->next = chunklist ;                                             \
        chunklist = bp;                                                    \
                                                                           \
        for (p = freelist + count - 2; p >= freelist ; p--)                \
            p->next = p + 1;                                               \
        freelist [count - 1].next = ( type * ) NULL;                       \
    }                                                                      \
    p = freelist ;                                                         \
    freelist = p->next;                                                    \
                                                                           \
    return p;                                                              \
}


#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_FREE_ROUTINE(type, functionname)                     \
static void functionname ( type *p );                                      \
static void functionname ( type *p )
#else
#define CC_HEADER_PTR_FREE_ROUTINE(type, functionname)                     \
static void functionname ();                                               \
static void functionname ( p )                                             \
type *p;
#endif

#define CC_PTR_FREE_ROUTINE(type, functionname, freelist)                  \
 CC_HEADER_PTR_FREE_ROUTINE(type, functionname)                            \
{                                                                          \
    p->next = freelist ;                                                   \
    freelist = p;                                                          \
}


#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_FREE_LIST_ROUTINE(type, functionname)                \
static void functionname ( type *p );                                      \
static void functionname ( type *p )
#else
#define CC_HEADER_PTR_FREE_LIST_ROUTINE(type, functionname)                \
static void functionname ();                                               \
static void functionname ( p )                                             \
type *p;
#endif

#define CC_PTR_FREE_LIST_ROUTINE(type, functionname, freefunction)         \
 CC_HEADER_PTR_FREE_LIST_ROUTINE (type, functionname)                      \
{                                                                          \
    type *next;                                                            \
                                                                           \
    while (p) {                                                            \
        next = p->next;                                                    \
        freefunction (p);                                                  \
        p = next;                                                          \
    }                                                                      \
}

#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_FREE_WORLD_ROUTINE(functionname)                     \
static void functionname (void);                                           \
static void functionname (void)
#else
#define CC_HEADER_PTR_FREE_WORLD_ROUTINE(functionname)                     \
static void functionname ();                                               \
static void functionname ()
#endif

#define CC_PTR_FREE_WORLD_ROUTINE(type, functionname, chunklist, freelist) \
 CC_HEADER_PTR_FREE_WORLD_ROUTINE(functionname)                            \
{                                                                          \
    CCbigchunkptr *bp, *bpnext;                                            \
                                                                           \
    for (bp = chunklist ; bp; bp = bpnext) {                               \
        bpnext = bp->next;                                                 \
        CCutil_bigchunkfree (bp);                                          \
    }                                                                      \
    chunklist = (CCbigchunkptr *) NULL;                                    \
    freelist = (type *) NULL;                                              \
}


#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_LEAKS_ROUTINE(functionname)                          \
static int functionname (int *total, int *onlist);                         \
static int functionname (int *total, int *onlist)
#else
#define CC_HEADER_PTR_LEAKS_ROUTINE(functionname)                          \
static int functionname ();                                                \
static int functionname (total, onlist)                                    \
int *total, *onlist;
#endif

#define CC_PTR_LEAKS_ROUTINE(type,name,chunklist,freelist,field,fieldtype) \
 CC_HEADER_PTR_LEAKS_ROUTINE(name)                                         \
{                                                                          \
    int count = CC_BIGCHUNK / sizeof ( type );                             \
    int duplicates = 0;                                                    \
    type * p;                                                              \
    CCbigchunkptr *bp;                                                     \
                                                                           \
    *total = 0;                                                            \
    *onlist = 0;                                                           \
                                                                           \
    for (bp = chunklist ; bp; bp = bp->next)                               \
        (*total) += count;                                                 \
                                                                           \
    for (p = freelist ; p; p = p->next) {                                  \
        (*onlist)++;                                                       \
        p-> field = ( fieldtype ) 0;                                       \
    }                                                                      \
    for (p = freelist ; p; p = p->next) {                                  \
        if (p-> field == ( fieldtype ) 1)                                  \
            duplicates++;                                                  \
        else                                                               \
            p-> field = ( fieldtype ) 1;                                   \
    }                                                                      \
    if (duplicates) {                                                      \
        fprintf (stderr, "WARNING: %d duplicates on ptr free list \n",     \
                 duplicates);                                              \
    }                                                                      \
    return *total - *onlist;                                               \
}

#ifdef CC_PROTOTYPE_ANSI
#define CC_HEADER_PTR_STATUS_ROUTINE(functionname)                         \
static int functionname (int *total, int *onlist);                         \
static int functionname (int *total, int *onlist)
#else
#define CC_HEADER_PTR_STATUS_ROUTINE(functionname)                         \
static int functionname ();                                                \
static int functionname (total, onlist)                                    \
int *total, *onlist;
#endif

#define CC_PTR_STATUS_ROUTINE(type, name, chunklist, freelist)             \
 CC_HEADER_PTR_STATUS_ROUTINE(name)                                        \
{                                                                          \
    int count = CC_BIGCHUNK / sizeof ( type );                             \
    type * p;                                                              \
    CCbigchunkptr *bp;                                                     \
                                                                           \
    *total = 0;                                                            \
    *onlist = 0;                                                           \
                                                                           \
    for (bp = chunklist ; bp; bp = bp->next)                               \
        (*total) += count;                                                 \
                                                                           \
    for (p = freelist ; p; p = p->next)                                    \
        (*onlist)++;                                                       \
    return *total - *onlist;                                               \
}


#define CC_BIGCHUNK ((int) ((1<<16)-16))

typedef struct CCbigchunkptr {
    void                 *this;
    struct CCbigchunkptr *next;
} CCbigchunkptr;


#ifdef CC_PROTOTYPE_ANSI

void
   *CCutil_allocrus (unsigned int size),
   *CCutil_reallocrus (void *ptr, unsigned int size),
    CCutil_freerus (void *p),
    CCutil_bigchunkquery (int *total, int *reserve),
    CCutil_bigchunkfree (CCbigchunkptr *bp);

int
    CCutil_reallocrus_scale (void **pptr, int *pnnum, int count, double scale,
                      unsigned int size),
    CCutil_reallocrus_count (void **pptr, int count, unsigned int size),
    CCutil_bigchunk_free_world (void);

CCbigchunkptr
   *CCutil_bigchunkalloc (void);

#else

void
   *CCutil_allocrus (),
   *CCutil_reallocrus (),
    CCutil_freerus (),
    CCutil_bigchunkquery (),
    CCutil_bigchunkfree ();

int
    CCutil_reallocrus_scale (),
    CCutil_reallocrus_count (),
    CCutil_bigchunk_free_world ();

CCbigchunkptr
    *CCutil_bigchunkalloc ();

#endif


/***************************************************************************/
/*                                                                         */
/*                             bgetopt.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_bix_getopt (int, char **, char *);

#else

int
    CCutil_bix_getopt ();

#endif

#define CC_BIX_GETOPT_UNKNOWN -3038

extern int CCutil_bix_optind;
extern char *CCutil_bix_optarg;



/***************************************************************************/
/*                                                                         */
/*                             dheaps_i.c                                  */
/*                                                                         */
/***************************************************************************/

typedef struct CCdheap {
    double  *key;
    int     *entry;
    int     *loc;
    int     total_space;
    int     size;
} CCdheap;

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_dheap_free (CCdheap *h),
    CCutil_dheap_insert (CCdheap *h, int i),
    CCutil_dheap_delete (CCdheap *h, int i),
    CCutil_dheap_changekey (CCdheap *h, int i, double newkey);
int
    CCutil_dheap_init (CCdheap *h, int k),
    CCutil_dheap_resize (CCdheap *h, int newsize),
    CCutil_dheap_findmin (CCdheap *h),
    CCutil_dheap_deletemin (CCdheap *h);

#else

void
    CCutil_dheap_free (),
    CCutil_dheap_insert (),
    CCutil_dheap_delete (),
    CCutil_dheap_changekey ();
int
    CCutil_dheap_init (),
    CCutil_dheap_resize (),
    CCutil_dheap_findmin (),
    CCutil_dheap_deletemin ();

#endif


/***************************************************************************/
/*                                                                         */
/*                             edg2cyc.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_edge_to_cycle (int ncount, int *elist, int *cyc);

#else

int
    CCutil_edge_to_cycle ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             edgelen.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct CCdatagroup {
    double  *x;
    double  *y;
    double  *z;
    int    **adj;
    int      norm;
} CCdatagroup;


#ifdef CC_PROTOTYPE_ANSI

extern int
  (*CCutil_dat_edgelen) (int i, int j, CCdatagroup *dat);
int
    CCutil_init_dat_edgelen (CCdatagroup *dat),
    CCutil_max_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_euclid_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_ibm_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_euclid_ceiling_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_euclid3d_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_geographic_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_att_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_dsjrand_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_crystal_edgelen (int i, int j, CCdatagroup *dat),
    CCutil_matrix_edgelen (int i, int j, CCdatagroup *dat);
void
    CCutil_dsjrand_init (int maxdist, int seed),
    CCutil_freedatagroup (int ncount, CCdatagroup *dat);

#else

extern int
  (*CCutil_dat_edgelen) ();
int
    CCutil_init_dat_edgelen (),
    CCutil_max_edgelen (),
    CCutil_euclid_edgelen (),
    CCutil_ibm_edgelen (),
    CCutil_euclid_ceiling_edgelen (),
    CCutil_euclid3d_edgelen (),
    CCutil_geographic_edgelen (),
    CCutil_att_edgelen (),
    CCutil_dsjrand_edgelen (),
    CCutil_crystal_edgelen (),
    CCutil_matrix_edgelen ();
void
    CCutil_dsjrand_init (),
    CCutil_freedatagroup ();

#endif


#define CC_KD_NORM_TYPE    128            /* Kdtrees work      */
#define CC_X_NORM_TYPE     256            /* Old nearest works */
#define CC_JUNK_NORM_TYPE  512            /* Nothing works     */

#define CC_D2_NORM_SIZE      1024         /* x,y coordinates   */
#define CC_D3_NORM_SIZE      2048         /* x,y,z coordinates */
#define CC_MATRIX_NORM_SIZE  4096         /* adj matrix        */

#define CC_NORM_BITS      (CC_KD_NORM_TYPE | CC_X_NORM_TYPE | CC_JUNK_NORM_TYPE)
#define CC_NORM_SIZE_BITS (CC_D2_NORM_SIZE | CC_D3_NORM_SIZE | CC_MATRIX_NORM_SIZE)

#define CC_MAXNORM        (0 |   CC_KD_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_EUCLIDEAN_CEIL (1 |   CC_KD_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_EUCLIDEAN      (2 |   CC_KD_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_EUCLIDEAN_3D   (3 |    CC_X_NORM_TYPE |     CC_D3_NORM_SIZE)
#define CC_IBM            (4 | CC_JUNK_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_ATT            (5 |    CC_X_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_GEOGRAPHIC     (6 |    CC_X_NORM_TYPE |     CC_D2_NORM_SIZE)
#define CC_MATRIXNORM     (7 | CC_JUNK_NORM_TYPE | CC_MATRIX_NORM_SIZE)
#define CC_DSJRANDNORM    (8 | CC_JUNK_NORM_TYPE)
#define CC_CRYSTAL        (9 |    CC_X_NORM_TYPE |     CC_D3_NORM_SIZE)

#define CC_GEOGRAPHIC_SCALE (6378.388 * 3.14 / 180.0) /*    see edgelen.c   */
#define CC_ATT_SCALE (.31622)                         /*    sqrt(1/10)      */

/* For X-NORMS, scales are such that |x[i] - x[j]| * scale <= edgelen(i,j). */
/* Ggeographic is slightly off, since the fractional part of x[i] is really */
/* really minutes, not fractional degrees.                                  */



/***************************************************************************/
/*                                                                         */
/*                             fastread.c                                  */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_readint (FILE *);

#else

int
    CCutil_readint ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             genhash.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct CCgenhash {
    int                     nelem;
    int                     maxelem;
    int                     size;
#ifdef CC_PROTOTYPE_ANSI
    int                   (*hcmp) (void *key1, void *key2, void *u_data);
    unsigned int          (*hfunc) (void *key, void *u_data);
#else
    int                   (*hcmp) ();
    unsigned int          (*hfunc) ();
#endif
    void                   *u_data;
    double                  maxdensity;
    double                  lowdensity;
    struct CCgenhash_elem **table;
} CCgenhash;

typedef struct CCgenhash_iter {
    int                    i;
    struct CCgenhash_elem *next;
} CCgenhash_iter;

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_genhash_init (CCgenhash *h, int size,
            int (*hcmp) (void *key1, void *key2, void *u_data),
            unsigned int (*hfunc) (void *key, void *u_data),
            void *u_data, double maxdensity, double lowdensity),
    CCutil_genhash_insert (CCgenhash *h, void *key, void *data),
    CCutil_genhash_insert_h (CCgenhash *h, unsigned int hashval, void *key,
            void *data),
    CCutil_genhash_replace (CCgenhash *h, void *key, void *data),
    CCutil_genhash_replace_h (CCgenhash *h, unsigned int hashval, void *key,
                       void *data),
    CCutil_genhash_delete (CCgenhash *h, void *key),
    CCutil_genhash_delete_h (CCgenhash *h, unsigned int hashval, void *key);

unsigned int
    CCutil_genhash_hash (CCgenhash *h, void *key);

void
   *CCutil_genhash_lookup (CCgenhash *h, void *key),
   *CCutil_genhash_lookup_h (CCgenhash *h, unsigned int hashval, void *key),
   *CCutil_genhash_next (CCgenhash *h, CCgenhash_iter *iter, void **key,
            int *keysize);

void
    CCutil_genhash_u_data (CCgenhash *h, void *u_data),
    CCutil_genhash_free (CCgenhash *h, void (*freefunc)(void *key, void *data,
            void *u_data)),
    CCutil_genhash_start (CCgenhash *h, CCgenhash_iter *iter);

#else

int
    CCutil_genhash_init (),
    CCutil_genhash_insert (),
    CCutil_genhash_insert_h (),
    CCutil_genhash_replace (),
    CCutil_genhash_replace_h (),
    CCutil_genhash_delete (),
    CCutil_genhash_delete_h ();

unsigned int
    CCutil_genhash_hash ();

void
   *CCutil_genhash_lookup (),
   *CCutil_genhash_lookup_h (),
   *CCutil_genhash_next ();

void
    CCutil_genhash_u_data (),
    CCutil_genhash_free (),
    CCutil_genhash_start ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             getdata.c                                   */
/*                                                                         */
/***************************************************************************/

#define CC_MASTER_NO_DAT  100
#define CC_MASTER_DAT     101

#ifdef CC_PROTOTYPE_ANSI

int
    CCutil_getdata (char *datname, int binary_in, int innorm, int *ncount,
            CCdatagroup *dat),
    CCutil_writemaster (char *mastername, int ncount, CCdatagroup *dat,
            int *perm),
    CCutil_getmaster (char *mastername, int *ncount, CCdatagroup *dat,
            int **perm),
    CCutil_getnodeweights (char *weightname, int ncount, int weight_limit,
            double **wcoord),
    CCutil_gettsplib (char *datname, int *ncount, CCdatagroup *dat),
    CCutil_datagroup_perm (int ncount, CCdatagroup *dat, int *perm),
    CCutil_getedgelist (int ncount, char *fname, int *ecount, int **elist,
            int **elen),
    CCutil_getedgelist_n (int *ncount, char *fname, int *ecount, int **elist,
            int **elen),
    CCutil_getcycle_edgelist (int ncount, char *cyclename, int *outcycle),
    CCutil_getcycle (int ncount, char *cyclename, int *outcycle),
    CCutil_getedges_double (int *ncount, char *fname, int *ecount, int **elist,
            double **elen, int binary_in),
    CCutil_writeedges (int ncount, char *outedgename, int ecount, int *elist,
            CCdatagroup *dat),
    CCutil_writecycle_edgelist (int ncount, char *outedgename, int *cycle,
            CCdatagroup *dat),
    CCutil_writecycle (int ncount, char *outcyclename, int *cycle),
    CCutil_writeedges_double (int ncount, char *outedgename, int ecount,
            int *elist, double *elen, int binary_out);

#else

int
    CCutil_getdata (),
    CCutil_writemaster (),
    CCutil_getmaster (),
    CCutil_getnodeweights (),
    CCutil_gettsplib (),
    CCutil_datagroup_perm (),
    CCutil_getedgelist (),
    CCutil_getedgelist_n (),
    CCutil_getcycle_edgelist (),
    CCutil_getcycle (),
    CCutil_getedges_double (),
    CCutil_writeedges (),
    CCutil_writecycle_edgelist (),
    CCutil_writecycle (),
    CCutil_writeedges_double ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             priority.c                                  */
/*                                                                         */
/***************************************************************************/

typedef struct CCpriority {
    CCdheap   heap;
    union pri_data {
        void *data;
        int  next;
    }        *pri_info;
    int       space;
    int       freelist;
} CCpriority;

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_priority_free (CCpriority *pri),
    CCutil_priority_delete (CCpriority *pri, int handle),
    CCutil_priority_changekey (CCpriority *pri, int handle, double newkey),
   *CCutil_priority_findmin (CCpriority *pri, double *keyval),
   *CCutil_priority_deletemin (CCpriority *pri, double *keyval);

int
    CCutil_priority_init (CCpriority *pri, int k),
    CCutil_priority_insert (CCpriority *pri, void *data, double keyval);

#else

void
    CCutil_priority_free (),
    CCutil_priority_delete (),
    CCutil_priority_changekey (),
   *CCutil_priority_findmin (),
   *CCutil_priority_deletemin ();

int
    CCutil_priority_init (),
    CCutil_priority_insert ();

#endif


/***************************************************************************/
/*                                                                         */
/*                             safe_io.c                                   */
/*                                                                         */
/***************************************************************************/

#define CC_SBUFFER_SIZE (4000)
#define CC_SFNAME_SIZE (32)

typedef struct CC_SFILE {
    int           status;
    int           desc;
    int           chars_in_buffer;
    int           current_buffer_char;     /* only used for reading */
    int           bits_in_last_char;       /* writing: number of empty bits in
                                            * buffer[chars_in_buffer];
                                            * reading: number of full bits in
                                            * buffer[?] */
    int           pos;
    char          fname[CC_SFNAME_SIZE];
    unsigned char buffer[CC_SBUFFER_SIZE];
} CC_SFILE;

#ifdef CC_PROTOTYPE_ANSI

CC_SFILE
   *CCutil_sopen (char *f, char *s),
   *CCutil_sdopen (int d, char *s);

int
    CCutil_swrite (CC_SFILE *f, unsigned char *buf, int size),
    CCutil_swrite_bits (CC_SFILE *f, unsigned int x, int xbits),
    CCutil_swrite_char (CC_SFILE *f, unsigned char x),
    CCutil_swrite_string (CC_SFILE *f, unsigned char *x),
    CCutil_swrite_short (CC_SFILE *f, unsigned short x),
    CCutil_swrite_int (CC_SFILE *f, unsigned int x),
    CCutil_swrite_double (CC_SFILE *f, double x),
    CCutil_sread (CC_SFILE *f, unsigned char *buf, int size),
    CCutil_sread_bits (CC_SFILE *f, unsigned int *x, int xbits),
    CCutil_sread_char (CC_SFILE *f, unsigned char *x),
    CCutil_sread_string (CC_SFILE *f, unsigned char *x, int maxlen),
    CCutil_sread_short (CC_SFILE *f, unsigned short *x),
    CCutil_sread_short_r (CC_SFILE *f, unsigned short *x),
    CCutil_sread_int (CC_SFILE *f, unsigned int *x),
    CCutil_sread_int_r (CC_SFILE *f, unsigned int *x),
    CCutil_sread_double (CC_SFILE *f, double *x),
    CCutil_sread_double_r (CC_SFILE *f, double *x),
    CCutil_sflush (CC_SFILE *f),
    CCutil_stell (CC_SFILE *f),
    CCutil_sseek (CC_SFILE *f, int offset),
    CCutil_srewind (CC_SFILE *f),
    CCutil_sclose (CC_SFILE *f),
    CCutil_sbits (unsigned int x),
    CCutil_sdelete_file (char *fname),
    CCutil_sdelete_file_backup (char *fname);

#else

CC_SFILE
   *CCutil_sopen (),
   *CCutil_sdopen ();

int
    CCutil_swrite (),
    CCutil_swrite_bits (),
    CCutil_swrite_char (),
    CCutil_swrite_string (),
    CCutil_swrite_short (),
    CCutil_swrite_int (),
    CCutil_swrite_double (),
    CCutil_sread (),
    CCutil_sread_bits (),
    CCutil_sread_char (),
    CCutil_sread_string (),
    CCutil_sread_short (),
    CCutil_sread_short_r (),
    CCutil_sread_int (),
    CCutil_sread_int_r (),
    CCutil_sread_double (),
    CCutil_sread_double_r (),
    CCutil_sflush (),
    CCutil_stell (),
    CCutil_sseek (),
    CCutil_srewind (),
    CCutil_sclose (),
    CCutil_sbits (),
    CCutil_sdelete_file (),
    CCutil_sdelete_file_backup ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             sortrus.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_int_array_quicksort (int *len, int n),
    CCutil_int_perm_quicksort (int *perm, int *len, int n),
    CCutil_double_perm_quicksort (int *perm, double *len, int n),
    CCutil_rselect (int *arr, int l, int r, int m, double *coord);

char
   *CCutil_linked_radixsort (char *data, char *datanext, char *dataval,
            int valsize);

#else

void
    CCutil_int_array_quicksort (),
    CCutil_int_perm_quicksort (),
    CCutil_double_perm_quicksort (),
    CCutil_rselect ();

char
   *CCutil_linked_radixsort ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             urandom.c                                   */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

void
    CCutil_sprand (int);
int
    CCutil_lprand (void);

#else

void
    CCutil_sprand ();
int
    CCutil_lprand ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             util.c                                      */
/*                                                                         */
/***************************************************************************/


#ifdef CC_PROTOTYPE_ANSI

char
   *CCutil_strrchr (char *s, int c);

unsigned int
    CCutil_nextprime (unsigned int x);

int
    CCutil_our_gcd (int a, int b);

#else

char
   *CCutil_strrchr ();

unsigned int
    CCutil_nextprime ();

int
    CCutil_our_gcd ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             zeit.c                                      */
/*                                                                         */
/***************************************************************************/


#ifdef CC_PROTOTYPE_ANSI

double
    CCutil_zeit (void),
    CCutil_real_zeit (void);

#else

double
    CCutil_zeit (),
    CCutil_real_zeit ();

#endif


#endif /* __UTIL_H */
#ifndef  __BIGGUY_H
#define  __BIGGUY_H


#ifdef  CC_BIGGUY_LONGLONG

typedef long long CCbigguy;

#define CCbigguy_FRACBITS 32
#define CCbigguy_DUALSCALE (((CCbigguy) 1) << CCbigguy_FRACBITS)
#define CCbigguy_FRACPART(x) ((x) & (CCbigguy_DUALSCALE-1))
#define CCbigguy_MAXBIGGUY (((((CCbigguy) 1) << 62) - 1) + \
                            (((CCbigguy) 1) << 62))
#define CCbigguy_MINBIGGUY (-CCbigguy_MAXBIGGUY)
#define CCbigguy_bigguytod(x) (((double) (x)) / ((double) CCbigguy_DUALSCALE))
#define CCbigguy_itobigguy(d) ((CCbigguy) ((d) * (double) CCbigguy_DUALSCALE))
#define CCbigguy_ceil(x) (CCbigguy_FRACPART(x) ? \
        ((x) + (CCbigguy_DUALSCALE - CCbigguy_FRACPART(x))) : (x))
#define CCbigguy_cmp(x,y) (((x) < (y)) ? -1 : ((x) > (y)) ? 1 : 0)
#define CCbigguy_ZERO ((CCbigguy) 0)
#define CCbigguy_ONE ((CCbigguy) CCbigguy_DUALSCALE)
#define CCbigguy_addmult(x,y,m) ((*x) += (y)*(m))
#define CCbigguy_dtobigguy(d) ((CCbigguy) ((d) * (double) CCbigguy_DUALSCALE))

#else /* CC_BIGGUY_LONGLONG */

typedef struct CCbigguy {
    unsigned short ihi;
    unsigned short ilo;
    unsigned short fhi;
    unsigned short flo;
} CCbigguy;

extern const CCbigguy CCbigguy_MINBIGGUY;
extern const CCbigguy CCbigguy_MAXBIGGUY;
extern const CCbigguy CCbigguy_ZERO;
extern const CCbigguy CCbigguy_ONE;

#ifdef CC_PROTOTYPE_ANSI

    void
        CCbigguy_addmult (CCbigguy *x, CCbigguy y, short m);

    int
        CCbigguy_cmp (CCbigguy x, CCbigguy y);

    double
        CCbigguy_bigguytod (CCbigguy x);

    CCbigguy
        CCbigguy_itobigguy (int d),
        CCbigguy_dtobigguy (double d),
        CCbigguy_ceil (CCbigguy x);

#else

    void
        CCbigguy_addmult ();

    int
        CCbigguy_cmp ();

    double
        CCbigguy_bigguytod ();

    CCbigguy
        CCbigguy_itobigguy (),
        CCbigguy_dtobigguy (),
        CCbigguy_ceil ();

#endif

#endif /* CC_BIGGUY_LONGLONG */

#define CCbigguy_add(x,y) (CCbigguy_addmult(x,y,1))
#define CCbigguy_sub(x,y) (CCbigguy_addmult(x,y,-1))

#ifdef CC_PROTOTYPE_ANSI

int
    CCbigguy_swrite (CC_SFILE *f, CCbigguy x),
    CCbigguy_sread (CC_SFILE *f, CCbigguy *x);

#else

int
    CCbigguy_swrite (),
    CCbigguy_sread ();

#endif

#endif /* __BIGGUY_H */
#ifndef __LP_H
#define __LP_H

#define  CClp_METHOD_DUAL    1
#define  CClp_METHOD_BARRIER 2

#define  CClp_SUCCESS        0
#define  CClp_FAILURE        1
#define  CClp_UNBOUNDED      2
#define  CClp_INFEASIBLE     3
#define  CClp_UNKNOWN        4

typedef struct CClp {
    struct cpxenv *cplex_env;
    struct cpxlp  *cplex_lp;
    int            lp_allocated;
} CClp;

typedef struct CClpbasis {
    int       *rstat;
    int       *cstat;
    double    *dnorm;
} CClpbasis;

#ifdef CC_PROTOTYPE_ANSI

int
    CClp_init (CClp *lp),
    CClp_loadlp (CClp *lp, char *name, int ncols, int nrows, int objsense,
            double *obj, double *rhs, char *sense, int *matbeg, int *matcnt,
            int *matind, double *matval, double *lb, double *ub),
    CClp_opt (CClp *lp, int method),
    CClp_dualopt (CClp *lp),
    CClp_limited_dualopt (CClp *lp, int lim, int *status, double *upperbound),
    CClp_primalopt (CClp *lp),
    CClp_addrows (CClp *lp, int newrows, int newnz, double *rhs, char *sense,
            int *rmatbeg, int *rmatind, double *rmatval),
    CClp_addcols (CClp *lp, int newcols, int newnz, double *obj,
            int *cmatbeg, int *cmatind, double *cmatval, double *lb,
            double *ub),
    CClp_delete_row (CClp *lp, int i),
    CClp_delete_set_of_rows (CClp *lp, int *delstat),
    CClp_delete_column (CClp *lp, int i),
    CClp_delete_set_of_columns (CClp *lp, int *delstat),
    CClp_setbnd (CClp *lp, int col, char lower_or_upper, double bnd),
    CClp_get_basis_and_norms (CClp *lp, CClpbasis *b),
    CClp_load_basis_and_norms (CClp *lp, CClpbasis *b),
    CClp_basis (CClp *lp, int *cstat, int *rstat),
    CClp_loadbasis (CClp *lp, int *cstat, int *rstat),
    CClp_getbasis_and_norms (CClp *lp, int *cstat, int *rstat,
            double *dnorm),
    CClp_loadbasis_and_norms (CClp *lp, int *cstat, int *rstat,
                              double *dnorm),
    CClp_x (CClp *lp, double *x),
    CClp_rc (CClp *lp, double *rc),
    CClp_pi_range (CClp *lp, double *pi, int from, int to),
    CClp_objval (CClp *lp, double *obj),
    CClp_nonzeros (CClp *lp),
    CClp_status (CClp *lp, int *status),
    CClp_getweight (CClp *lp, int nrows, int *rmatbeg, int *rmatind,
            double *rmatval, double *weight),
    CClp_dump_lp (CClp *lp, char *fname),
    CClp_getgoodlist (CClp *lp, int *goodlist, int *goodlen_p,
            double *downpen, double *uppen),
    CClp_strongbranch (CClp *lp, int *candidatelist, int ncand,
            double *downpen, double *uppen, int iterations,
            double *upperbound),
    CClp_getfarkasmultipliers (CClp *lp, double *y);

void
    CClp_init_struct (CClp *lp),
    CClp_free (CClp *lp),
    CClp_init_basis (CClpbasis *b),
    CClp_free_basis (CClpbasis *b),
    CClp_pivotin (CClp *lp, int i);

#else

int
    CClp_init (),
    CClp_loadlp (),
    CClp_opt (),
    CClp_dualopt (),
    CClp_limited_dualopt (),
    CClp_primalopt (),
    CClp_addrows (),
    CClp_addcols (),
    CClp_delete_row (),
    CClp_delete_set_of_rows (),
    CClp_delete_column (),
    CClp_delete_set_of_columns (),
    CClp_setbnd (),
    CClp_get_basis_and_norms (),
    CClp_load_basis_and_norms (),
    CClp_basis (),
    CClp_loadbasis (),
    CClp_getbasis_and_norms (),
    CClp_loadbasis_and_norms (),
    CClp_x (),
    CClp_rc (),
    CClp_pi_range (),
    CClp_objval (),
    CClp_nonzeros (),
    CClp_status (),
    CClp_getweight (),
    CClp_dump_lp (),
    CClp_getgoodlist (),
    CClp_strongbranch (),
    CClp_getfarkasmultipliers ();

void
    CClp_init_struct (),
    CClp_free (),
    CClp_init_basis (),
    CClp_free_basis (),
    CClp_pivotin ();

#endif


#endif  /* __LP_H */
#ifndef __KDTREE_H
#define __KDTREE_H


typedef struct CCkdnode {
    double           cutval;
    struct CCkdnode *loson;
    struct CCkdnode *hison;
    struct CCkdnode *father;
    struct CCkdnode *next;
    struct CCkdbnds *bnds;
    int              lopt;
    int              hipt;
    char             bucket;
    char             empty;
    char             cutdim;
} CCkdnode;

typedef struct CCkdtree {
    CCkdnode  *root;
    CCkdnode **bucketptr;
    int       *perm;
} CCkdtree;

typedef struct CCkdbnds {
    double           x[2];
    double           y[2];
    struct CCkdbnds *next;
} CCkdbnds;

#ifdef CC_PROTOTYPE_ANSI

void
    CCkdtree_free (CCkdtree *kt),
    CCkdtree_delete (CCkdtree *kt, int k),
    CCkdtree_delete_all (CCkdtree *kt, int ncount),
    CCkdtree_undelete (CCkdtree *kt, int k),
    CCkdtree_undelete_all (CCkdtree *kt, int ncount);
int
    CCkdtree_build (CCkdtree *kt, int ncount, CCdatagroup *dat, double *wcoord),
    CCkdtree_k_nearest (CCkdtree *kt, int ncount, int k, CCdatagroup *dat,
            double *wcoord, int wantlist, int *ocount, int **olist),
    CCkdtree_quadrant_k_nearest (CCkdtree *kt, int ncount, int k,
            CCdatagroup *dat, double *wcoord, int wantlist, int *ocount,
            int **olist),
    CCkdtree_node_k_nearest (CCkdtree *kt, int ncount, int n, int k,
            CCdatagroup *dat, double *wcoord, int *list),
    CCkdtree_node_quadrant_k_nearest (CCkdtree *kt, int ncount, int n, int k,
            CCdatagroup *dat, double *wcoord, int *list),
    CCkdtree_node_nearest (CCkdtree *kt, int n, CCdatagroup *dat,
            double *wcoord),
    CCkdtree_fixed_radius_nearest (CCkdtree *kt, CCdatagroup *dat,
            double *wcoord, int n, double rad,
            int (*doit_fn) (int, int, void *), void *pass_param),
    CCkdtree_nearest_neighbor_tour (CCkdtree *kt, int ncount, int start,
            CCdatagroup *dat, int *outcycle, double *val),
    CCkdtree_nearest_neighbor_2match (CCkdtree *kt, int ncount, int start,
            CCdatagroup *dat, int *outmatch, double *val),
    CCkdtree_prim_spanningtree (CCkdtree *kt, int ncount, CCdatagroup *dat,
            double *wcoord, int *outtree, double *val),
    CCkdtree_greedy_tour (CCkdtree *kt, int ncount, CCdatagroup *dat,
            int *outcycle, double *val),
    CCkdtree_far_add_tour (CCkdtree *kt, int ncount, int start,
            CCdatagroup *dat, int *outcycle, double *val),
    CCkdtree_qboruvka_tour (CCkdtree *kt, int ncount, CCdatagroup *dat,
            int *outcycle, double *val),
    CCkdtree_boruvka_tour (CCkdtree *kt, int ncount, CCdatagroup *dat,
            int *outcycle, double *val),
    CCkdtree_twoopt_tour (CCkdtree *kt, int ncount, CCdatagroup *dat,
            int *incycle, int *outcycle, double *val,
            int in_run_two_and_a_half_opt, int run_silently),
    CCkdtree_3opt_tour (CCkdtree *kt, int ncount, CCdatagroup *dat,
            int *incycle, int *outcycle, double *val, int run_silently);

#else

void
    CCkdtree_free (),
    CCkdtree_delete (),
    CCkdtree_delete_all (),
    CCkdtree_undelete (),
    CCkdtree_undelete_all ();
int
    CCkdtree_build (),
    CCkdtree_k_nearest (),
    CCkdtree_quadrant_k_nearest (),
    CCkdtree_node_k_nearest (),
    CCkdtree_node_quadrant_k_nearest (),
    CCkdtree_node_nearest (),
    CCkdtree_fixed_radius_nearest (),
    CCkdtree_nearest_neighbor_tour (),
    CCkdtree_nearest_neighbor_2match (),
    CCkdtree_prim_spanningtree (),
    CCkdtree_greedy_tour (),
    CCkdtree_far_add_tour (),
    CCkdtree_qboruvka_tour (),
    CCkdtree_boruvka_tour (),
    CCkdtree_twoopt_tour (),
    CCkdtree_3opt_tour ();
#endif

#endif  /* __KDTREE_H */

/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*                      PROTOTYPES FOR FILES IN CUT                        */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/


#ifndef  __CUT_H
#define  __CUT_H


#define CC_MINCUT_BIGDOUBLE   (100000000000.0)
#define CC_MINCUT_ONE_EPSILON (0.000001)


#ifdef CC_PROTOTYPE_ANSI

int
    CCcut_mincut (int ncount, int ecount, int *elist, double *dlen,
            double *valval, int **cut, int *cutcount),
    CCcut_violated_cuts (int ncount, int ecount, int *elist, double *dlen,
            double cutoff, int (*doit_fn) (double, int, int *, void *),
            void *pass_param),
    CCcut_mincut_st (int ncount, int ecount, int *elist, double *ecap,
            int s, int t, double *value, int **cut, int *cutcount),
    CCcut_linsub (int ncount, int ecount, int *elist, double *x, double cutoff,
        int (*doit_fn) (double, int, int, void *), void *pass_param),
    CCcut_connect_components (int ncount, int ecount, int *elist, double *x,
        int *ncomp, int **compscount, int **comps);

#else

int
    CCcut_mincut (),
    CCcut_violated_cuts (),
    CCcut_mincut_st (),
    CCcut_linsub (),
    CCcut_connect_components ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             shrink.c                                    */
/*                                                                         */
/***************************************************************************/

typedef struct CC_SRKnode {
    struct CC_SRKedge *adj;
    struct CC_SRKnode *next;
    struct CC_SRKnode *prev;
    struct CC_SRKnode *members;
    struct CC_SRKnode *parent;
    struct CC_SRKnode *qnext;
    double             prweight;
    double             weight;
    int                num;
    int                newnum;
    int                onecnt;
    int                onqueue;
} CC_SRKnode;

typedef struct CC_SRKedge {
    struct CC_SRKnode *end;
    struct CC_SRKedge *other;
    struct CC_SRKedge *next;
    struct CC_SRKedge *prev;
    double             weight;
} CC_SRKedge;

typedef struct CC_SRKgraph {
    struct CC_SRKnode  *nodespace;
    struct CC_SRKedge  *edgespace;
    struct CC_SRKnode  *head;
    struct CC_SRKedge **hit;
    int                 original_ncount;
    int                 original_ecount;
} CC_SRKgraph;

typedef struct CC_SRKexpinfo {
    int *members;
    int *memindex;
} CC_SRKexpinfo;

typedef struct CC_SRKcallback {
    double  cutoff;
    void   *pass_param;
#ifdef CC_PROTOTYPE_ANSI
    int   (*doit_fn) (double, int, int *, void *);
#else
    int   (*doit_fn) ();
#endif
} CC_SRKcallback;

#ifdef CC_PROTOTYPE_ANSI

void
    CCcut_SRK_identify_paths (CC_SRKgraph *G, int *newcount, int onecnt_okay),
    CCcut_SRK_identify_paths_to_edges (CC_SRKgraph *G, int *newcount,
        int onecnt_okay),
    CCcut_SRK_identify_ones (CC_SRKgraph *G, int *count, double epsilon),
    CCcut_SRK_identify_one_triangles (CC_SRKgraph *G, int *count,
        CC_SRKnode *qstart, double epsilon),
    CCcut_SRK_identify_nodes (CC_SRKgraph *G, CC_SRKnode *n, CC_SRKnode *m),
    CCcut_SRK_init_graph (CC_SRKgraph *G),
    CCcut_SRK_free_graph (CC_SRKgraph *G),
    CCcut_SRK_init_expinfo (CC_SRKexpinfo *expand),
    CCcut_SRK_free_expinfo (CC_SRKexpinfo *expand),
    CCcut_SRK_init_callback (CC_SRKcallback *cb);

int
    CCcut_SRK_buildgraph (CC_SRKgraph *G, int ncount, int ecount, int *elist,
        double *dlen),
    CCcut_SRK_subtour_shrink (CC_SRKgraph *G, double *minval, double epsilon,
        CC_SRKcallback *cb, int **cut, int *cutcount),
    CCcut_SRK_identify_pr_edges (CC_SRKgraph *G, double *minval, int *count,
        CC_SRKnode *qstart, double epsilon, CC_SRKcallback *cb, int **cut,
        int *cutcount),
    CCcut_SRK_defluff (CC_SRKgraph *G),
    CCcut_SRK_grab_edges (CC_SRKgraph *G, int *oncount, int *oecount,
        int **olist, double **olen, CC_SRKexpinfo *expand),
    CCcut_SRK_grab_nodes (CC_SRKgraph *G, CC_SRKexpinfo *expand),
    CCcut_SRK_trivial (int ncount, CC_SRKexpinfo *expand),
    CCcut_SRK_expand (CC_SRKexpinfo *expand, int *arr, int size, int **pnewarr,
        int *pnewsize);

#else

void
    CCcut_SRK_identify_paths (),
    CCcut_SRK_identify_paths_to_edges (),
    CCcut_SRK_identify_ones (),
    CCcut_SRK_identify_one_triangles (),
    CCcut_SRK_identify_nodes (),
    CCcut_SRK_init_graph (),
    CCcut_SRK_free_graph (),
    CCcut_SRK_init_expinfo (),
    CCcut_SRK_free_expinfo (),
    CCcut_SRK_init_callback ();

int
    CCcut_SRK_buildgraph (),
    CCcut_SRK_subtour_shrink (),
    CCcut_SRK_identify_pr_edges (),
    CCcut_SRK_defluff (),
    CCcut_SRK_grab_edges (),
    CCcut_SRK_grab_nodes (),
    CCcut_SRK_trivial (),
    CCcut_SRK_expand ();

#endif

#endif  /* __CUT_H */
/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*                      PROTOTYPES FOR FILES IN EDGEGEN                    */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#ifndef __EDGEGEN_H
#define __EDGEGEN_H



/***************************************************************************/
/*                                                                         */
/*                             edgegen.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct CCedgegengroup {
    struct {
        int count;
        int quadnearest;
        int nearest;
        int nearest_start;
        int greedy_start;
        int random_start;
        int nkicks;
    } linkern;

    struct {
        int twoopt_count;
        int twoopt5_count;
        int threeopt_count;
        int greedy;
        int nearest_count;
        int random_count;
    } tour;

    struct {
        int wantit;
        int basic;
        int priced;
    } f2match;

    struct {
        int number;
        int basic;
        int priced;
    } f2match_nearest;

    int    nearest;
    int    quadnearest;
    int    want_tree;
    int    nearest_twomatch_count;
    int    delaunay;
    int    mlinkern;
} CCedgegengroup;


#ifdef CC_PROTOTYPE_ANSI

int
    CCedgegen_read (char *egname, CCedgegengroup *plan),
    CCedgegen_edges (CCedgegengroup *plan, int ncount, CCdatagroup *dat,
        double *wcoord, int *ecount, int **elist);
void
    CCedgegen_init_edgegengroup (CCedgegengroup *plan);

#else

int
    CCedgegen_read (),
    CCedgegen_edges ();
void
    CCedgegen_init_edgegengroup ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             xnear.c                                     */
/*                                                                         */
/***************************************************************************/

typedef struct CCxnear {
    struct CCdatagroup  dat;
    double             *w;
    int                *nodenames;
    int                *invnames;
} CCxnear;


#ifdef CC_PROTOTYPE_ANSI

int
    CCedgegen_x_k_nearest (int ncount, int num, CCdatagroup *dat,
        double *wcoord, int wantlist, int *ecount, int **elist),
    CCedgegen_x_quadrant_k_nearest (int ncount, int num, CCdatagroup *dat,
        double *wcoord, int wantlist, int *ecount, int **elist),
    CCedgegen_x_node_k_nearest (CCxnear *xn, int n, int nearnum, int ncount,
        int *list),
    CCedgegen_x_node_quadrant_k_nearest (CCxnear *xn, int n, int nearnum,
        int ncount, int *list),
    CCedgegen_x_node_nearest (CCxnear *xn, int ncount, int ni, char *marks),
    CCedgegen_x_nearest_neighbor_tour (int ncount, int start, CCdatagroup *dat,
        int *outcycle, double *val),
    CCedgegen_junk_k_nearest (int ncount, int num, CCdatagroup *dat,
        double *wcoord, int wantlist, int *ecount, int **elist),
    CCedgegen_junk_node_k_nearest (CCdatagroup *dat, double *wcoord, int n,
        int nearnum, int ncount, int *list),
    CCedgegen_junk_node_nearest (CCdatagroup *dat, double *wcoord, int ncount,
        int n, char *marks),
    CCedgegen_junk_nearest_neighbor_tour (int ncount, int start,
        CCdatagroup *dat, int *outcycle, double *val),
    CCedgegen_xnear_build (int ncount, CCdatagroup *dat, double *wcoord,
        CCxnear *xn);

void
    CCedgegen_xnear_free (int ncount, CCxnear *xn);

#else

int
    CCedgegen_x_k_nearest (),
    CCedgegen_x_quadrant_k_nearest (),
    CCedgegen_x_node_k_nearest (),
    CCedgegen_x_node_quadrant_k_nearest (),
    CCedgegen_x_node_nearest (),
    CCedgegen_x_nearest_neighbor_tour (),
    CCedgegen_junk_k_nearest (),
    CCedgegen_junk_node_k_nearest (),
    CCedgegen_junk_node_nearest (),
    CCedgegen_junk_nearest_neighbor_tour (),
    CCedgegen_xnear_build ();
void
    CCedgegen_xnear_free ();

#endif

#endif  /* __EDGEGEN_H */
/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*                      PROTOTYPES FOR FILES IN CUT                        */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/


#ifndef __TSP_H
#define __TSP_H


/*************** Tolerances for the LP and Cutting routines ***************/

#define CCtsp_MIN_VIOL (0.00001)   /* min violation for cut to be added to lp */
#define CCtsp_CUTS_NEXT_TOL (0.0001)       /* to try next level  */
#define CCtsp_CUTS_NEXT_ROUND (0.00000001) /* if improve is less, stop round */
#define CCtsp_PRICE_RCTHRESH  (-0.00001)   /* to add a bad edge */
#define CCtsp_PRICE_MAXPENALTY (0.49)      /* penalty permitted in addbad */
#define CCtsp_PHASE1_RCTHRESH (-0.000000001)
#define CCtsp_PHASE1_MAXPENALTY (0.00000001)
#define CCtsp_EDGE_LIFE (1000000) /* 200 */  /* Large for subtour runs */
#define CCtsp_CUT_LIFE  (50)
#define CCtsp_CUT_BATCH (250)     /* number of new cuts before lp optimize */
#define CCtsp_STORE_BATCH (50)    /* number of new cuts before lp addrows  */
#define CCtsp_INTTOL (0.0001)     /* used to check if lp soln is integral  */

/************************** Branching Strategies  ************************/

#define CCtsp_BRANCH_MIDDLE 1
#define CCtsp_BRANCH_STRONG 2

/*************************************************************************/

#define CCtsp_LP_MAXDOUBLE  1e30

#define CCtsp_CUTRHS(c) (3*(c)->cliquecount - (c)->handlecount - 1)

typedef struct CCtsp_lpnode {
    int                 deg;
    int                 mark;
    struct CCtsp_lpadj *adj;
} CCtsp_lpnode;

typedef struct CCtsp_lpedge {
    int ends[2];   /* ends[0] should always be < ends[1] */
    int fixed;
    int branch;    /* < 0 means set to 0 and > 0 means set to 1 */
    int len;
    int age;
    int coef;      /* should be maintained at zero */
    int coefnext;  /* should be maintained at -2 */
} CCtsp_lpedge;

typedef struct CCtsp_lpadj {
    int to;
    int edge;
} CCtsp_lpadj;

typedef struct CCtsp_lpgraph {
    int           ncount;
    int           espace;
    int           ecount;
    int           nodemarker;
    CCtsp_lpnode *nodes;
    CCtsp_lpedge *edges;
    CCtsp_lpadj  *adjspace;
    int           adjstart;
    int           adjend;
} CCtsp_lpgraph;

typedef struct CCtsp_predge {
    int    ends[2];
    int    len;
    double rc;
} CCtsp_predge;

typedef struct CCtsp_pricegroup {
    int                    ncount;
    int                    espace;
    int                    ecount;
    CCtsp_lpnode          *nodes;
    CCtsp_predge          *edges;
    int                    cliquecount;
    struct CCtsp_lpclique *cliques; /* just a copy of the pointer */
    CCtsp_lpgraph         *graph;   /* pointer to the copy in a CCtsp_lp */
    CCtsp_lpadj           *adjspace;
    double                *node_pi;
    double                *clique_pi;
    double                 penalty;
} CCtsp_pricegroup;

typedef struct CCtsp_extraedge {
    int ends[2];
} CCtsp_extraedge;

typedef struct CCtsp_sparser {
    unsigned int node : 24;
    unsigned int mult : 8;
} CCtsp_sparser;

typedef struct CCtsp_segment {
    int lo;
    int hi;
} CCtsp_segment;

typedef struct CCtsp_lpclique {
    int                   segcount;
    struct CCtsp_segment *nodes;
    int                   hashnext;
    int                   refcount;
} CCtsp_lpclique;

#define CC_FOREACH_NODE_IN_CLIQUE(i,c,tmp) \
    for(tmp=0;tmp<(c).segcount;tmp++) \
        for(i=(c).nodes[tmp].lo;i<=(c).nodes[tmp].hi;i++)

#define CCtsp_NEWCUT_AGE (-1)

typedef struct CCtsp_lpcut {
    int                   handlecount;
    int                   cliquecount;
    int                   modcount;
    int                   age;
    int                   rhs;
    char                  sense;
    char                  branch;
    int                  *cliques;
    struct CCtsp_sparser *mods;
} CCtsp_lpcut;

typedef struct CCtsp_lpcut_in {
    int                    handlecount;
    int                    cliquecount;
    int                    rhs;
    char                   sense;
    char                   branch;
    CCtsp_lpclique        *cliques;
    struct CCtsp_lpcut_in *next;
    struct CCtsp_lpcut_in *prev;
} CCtsp_lpcut_in;

typedef struct CCtsp_lp_result {
    double         ub;
    double         lb;
    int            ecount;
    int           *elist;
    double        *x;
    double        *rc;
} CCtsp_lp_result;

typedef struct CCtsp_lpcuts {
    int            cutcount;
    int            cliqueend;
    int            cutspace;
    int            cliquespace;
    int            cliquehashsize;
    int            cliquefree;
    int            *cliquehash;
    CCtsp_lpcut    *cuts;
    CCtsp_lpclique *cliques;
    CCgenhash      *cuthash;
    char           *tempcuthash;
    int            tempcuthashsize;
} CCtsp_lpcuts;

typedef struct CCtsp_bigdual {
    int            cutcount;
    CCbigguy      *node_pi;
    CCbigguy      *cut_pi;
} CCtsp_bigdual;

typedef struct CCtsp_tighten_info {
    int    ncall;
    int    nfail;
    int    nadd;
    int    nadd_tied;
    int    ndel;
    int    ndel_tied;
    double add_delta;
    double del_delta;
    double time;
} CCtsp_tighten_info;

typedef struct CCtsp_branchobj {
    int             depth;
    int             rhs;
    int             ends[2];
    char            sense;
    CCtsp_lpclique *clique;
} CCtsp_branchobj;

typedef struct CCtsp_cutselect {
    int    cutpool;
    int    connect;
    int    segments;
    int    exactsubtour;
    int    tighten_lp;
    int    decker_lp;
    int    teething_lp;
    int    tighten_pool;
    int    decker_pool;
    int    teething_pool;
    int    maxchunksize;
    int    Xfastcuts;
    int    Xexactsubtours;
    int    Xslowcuts;
    int    consecutiveones;
    int    necklace;
    int    usetighten;     /* set to 1 to tighten before cuts are added */
    int    extra_connect;  /* set to 1 to force a connected solution */
    double nexttol;
    double roundtol;
} CCtsp_cutselect;

/* nodes are reordered to match compression tour */

typedef struct CCtsp_genadj {
    int                     deg;
    struct CCtsp_genadjobj *list;
} CCtsp_genadj;

typedef struct CCtsp_genadjobj {
    int end;
    int len;
} CCtsp_genadjobj;

typedef struct CCtsp_edgegenerator {
    double                    *node_piest;
    struct CCdatagroup        *dg;
    int                       *supply;
    CCkdtree                  *kdtree;
    CCxnear                   *xnear;
    struct CCtsp_xnorm_pricer *xprice;
    CCtsp_genadjobj           *adjobjspace;
    CCtsp_genadj              *adj;
    int                        ncount;
    int                        nneighbors;
    int                        start;
    int                        current;
    int                        supplyhead;
    int                        supplycount;
} CCtsp_edgegenerator;

typedef struct CCtsp_xnorm_pricer_val {
    double                         val;
    struct CCtsp_xnorm_pricer_val *next;
    struct CCtsp_xnorm_pricer_val *prev;
    int                            index;
} CCtsp_xnorm_pricer_val;

typedef struct CCtsp_xnorm_pricer {
    CCdatagroup            *dat;
    double                 *pi;
    int                    *order;
    CCtsp_xnorm_pricer_val *xminuspi_space;
    CCtsp_xnorm_pricer_val *xminuspi;
    int                    *invxminuspi;
    int                     ncount;
} CCtsp_xnorm_pricer;

typedef struct CCtsp_lp {
    CCtsp_lpgraph              graph;
    CCtsp_lpcuts               cuts;
    CCtsp_lpcuts              *pool;
    CClp                       lp;
    int                       *perm;
    CCdatagroup               *dat;
    int                        fullcount;
    struct CCtsp_genadj       *fulladj;
    struct CCtsp_genadjobj    *fulladjspace;
    int                        nfixededges;
    int                       *fixededges;
    struct CCtsp_qsparsegroup *sparsifier;
    int                        edge_life;
    int                        cut_life;
    char                      *name;
    int                        id;
    int                        parent_id;
    int                        root;
    double                     upperbound;
    double                     lowerbound;
    CCbigguy                   exact_lowerbound;
    CCtsp_bigdual             *exact_dual;
    int                        infeasible;
    int                        full_edges_valid;
    CClpbasis                 *basis;
    CCtsp_lpcut_in             cutqueue;    /* dummy entry for doubly-linked
                                               list */
    CCtsp_lp_result            result;
    CCtsp_tighten_info         tighten_stats;
    int                        branchdepth;
    CCtsp_branchobj           *branchhistory;
} CCtsp_lp;

typedef struct CCtsp_lprow {
    int           rowcnt;
    int           nzcnt;
    char         *sense;
    double       *rhs;
    int          *begin;      /* offset into the array for start of row */
    int           indexspace;
    int          *indices;    /* the column indices of the row entries  */
    int           entryspace;
    double       *entries;    /* the matrix entries                     */
} CCtsp_lprow;



/***************************************************************************/
/*                                                                         */
/*                             tsp_lp.c                                    */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCtsp_cutting_loop (CCtsp_lp *lp, CCtsp_cutselect *sel, int savelp),
    CCtsp_subtour_loop (CCtsp_lp *lp),
    CCtsp_pricing_loop (CCtsp_lp *lp, double *bnd),
    CCtsp_init_cutselect (CCtsp_lp *lp, CCtsp_cutselect *s),
    CCtsp_call_x_heuristic (CCtsp_lp *lp, double *val, int *outcyc),
    CCtsp_bb_cutting (char *probname, int probnum, int ncount,
            CCdatagroup *dat, int *ptour, double *upbound, CCtsp_lpcuts *pool,
            CCtsp_cutselect *sel, double *val, int *prune, int *foundtour,
            int *besttour),
    CCtsp_init_lp (CCtsp_lp **lp, char *probname, int probnum,
            char *probfilename, int ncount, CCdatagroup *dat, int ecount,
            int *elist, int *elen, int excount, int *exlist, int *exlen,
            int exvalid, int *ptour, double initial_ub, CCtsp_lpcuts *pool),
    CCtsp_bb_init_lp (CCtsp_lp **lp, char *probname, int probnum,
            int ncount, CCdatagroup *dat, int *ptour, double initial_ub,
            CCtsp_lpcuts *pool),
    CCtsp_get_lp_result (CCtsp_lp *lp, double *lb, double *ub, int *ecount,
            int **elist, double **x, double **rc, double **node_pi,
            double **cut_pi),
    CCtsp_process_cuts (CCtsp_lp *lp, int *pnadded, int tighten),
    CCtsp_add_cut_to_cutlist (CCtsp_lpcuts *cuts, CCtsp_lpcut *c),
    CCtsp_add_cut (CCtsp_lp *lp, CCtsp_lpcut_in *d, CCtsp_lprow *cr),
    CCtsp_lpcut_in_nzlist (CCtsp_lpgraph *g, CCtsp_lpcut_in *c),
    CCtsp_add_nzlist_to_lp (CCtsp_lp *lp, int nzlist, int rhs, char sense,
            CCtsp_lprow *cr),
    CCtsp_add_vars_to_lp (CCtsp_lp *lp, CCtsp_predge *prlist, int n),
    CCtsp_update_result (CCtsp_lp *lp),
    CCtsp_infeas_recover (CCtsp_lp *lp),
    CCtsp_test_cut_branch (CCtsp_lp *lp, CCtsp_lpclique *c, double *down,
            double *up),
    CCtsp_register_cliques (CCtsp_lpcuts *cuts, CCtsp_lpcut_in *c,
            CCtsp_lpcut *new),
    CCtsp_addbad_variables (CCtsp_lp *lp, struct CCtsp_edgegenerator *eg,
            double *ppenalty, int *pnadded, double rcthresh,
            double maxpenalty, int phase1, int *feasible),
    CCtsp_eliminate_variables (CCtsp_lp *lp),
    CCtsp_build_lpgraph (CCtsp_lpgraph *g, int ncount, int ecount,
            int *elist, int *elen),
    CCtsp_build_lpadj (CCtsp_lpgraph *g, int estart, int eend),
    CCtsp_add_multiple_rows (CCtsp_lp *lp, CCtsp_lprow *cr),
    CCtsp_delete_cut (CCtsp_lp *lp, int i),
    CCtsp_find_edge (CCtsp_lpgraph *g, int from, int to),
    CCtsp_find_branch (CCtsp_lp *lp, int nwant, int *ngot,
            CCtsp_branchobj **bobj, double *val, int **cyc, int usecliques),
    CCtsp_bb_find_branch (char *probname, int probnum, int ncount,
            CCdatagroup *dat, int *ptour, double *upperbound,
            CCtsp_lpcuts *pool, CCtsp_branchobj **b, int usecliques,
            int *foundtour, int *besttour),
    CCtsp_check_integral (CCtsp_lp *lp, double *val, int **cyc, int *yesno),
    CCtsp_find_branch_edge (CCtsp_lp *lp, int *n0, int *n1, double *val,
            int **cyc, int branchtype),
    CCtsp_find_branch_cliques (CCtsp_lp *lp, int nwant, int *ngot,
            CCtsp_lpclique **bcliques, double **bval),
    CCtsp_execute_branch (CCtsp_lp *lp, CCtsp_branchobj *b),
    CCtsp_execute_unbranch (CCtsp_lp *lp, CClpbasis *basis),
    CCtsp_splitprob (CCtsp_lp *lp, CCtsp_branchobj *b, int child0, int child1),
    CCtsp_bb_splitprob (char *probname, int probnum, int ncount,
            CCdatagroup *dat, int *ptour, double initial_ub, CCtsp_lpcuts *pool,
            CCtsp_branchobj *b, int child0, int child1, double *val0,
            double *val1, int *prune0, int *prune1),
    CCtsp_dumptour (int ncount, CCdatagroup *dat, int *perm, char *probname,
            int *tour),
    CCtsp_add_branchhistory_to_lp (CCtsp_lp *lp),
    CCtsp_easy_dfs_brancher (CCtsp_lp *lp, CCtsp_cutselect *sel, int depth,
            double *upbound, int *bbcount, int usecliques, int *besttour),
    CCtsp_bfs_brancher (char *probname, int id, double lowerbound,
            CCtsp_cutselect *sel, double *upbound, int *bbcount, int usecliques,
            CCdatagroup *mydat, int *ptour, CCtsp_lpcuts *pool, int ncount,
            int *besttour),
    CCtsp_do_interactive_branch (CCtsp_lp *lp),
    CCtsp_inspect_full_edges (CCtsp_lp *lp),
    CCtsp_read_probfile (CCtsp_lp *lp, char *fname, int ncount),
    CCtsp_read_probfile_id (CCtsp_lp *lp, char *fname, int id, int ncount),
    CCtsp_write_probfile_sav (CCtsp_lp *lp),
    CCtsp_write_probfile_id (CCtsp_lp *lp),
    CCtsp_dump_x (CCtsp_lp *lp, char *fname),
    CCtsp_exact_price (CCtsp_lp *lp, CCbigguy *bound, int phase1),
    CCtsp_edge_elimination (CCtsp_lp *lp),
    CCtsp_exact_dual (CCtsp_lp *lp),
    CCtsp_verify_infeasible_lp (CCtsp_lp *lp, int *yesno),
    CCtsp_verify_lp_prune (CCtsp_lp *lp, int *yesno),
    CCtsp_tighten_lpcut_in (CCtsp_lpgraph *g, CCtsp_lpcut_in *c,
            double *x, CCtsp_lpcut_in *d, CCtsp_tighten_info *stats,
            double *pimprove),
    CCtsp_tighten_lpcut (CCtsp_lpgraph *g, CCtsp_lpclique *cliques,
            CCtsp_lpcut *c, double *x, CCtsp_lpcut_in *d,
            CCtsp_tighten_info *stats, double *pimprove),
    CCtsp_test_pure_comb (int ncount, CCtsp_lpcut_in *c, int *yes_no,
            int *handle),
    CCtsp_test_pseudocomb (int ncount, CCtsp_lpcut_in *c, int handle,
            int *yes_no),
    CCtsp_test_teeth_disjoint (int ncount, CCtsp_lpcut_in *c, int handle,
            int *yes_no),
    CCtsp_find_pure_handle (int ncount, CCtsp_lpcut_in *c, int *handle),
    CCtsp_comb_to_double_decker (CCtsp_lpgraph *g, double *x,
            CCtsp_lpcut_in *c, CCtsp_lpcut_in **d),
    CCtsp_teething (CCtsp_lpgraph *g, double *x, CCtsp_lpcut_in *cut,
            CCtsp_lpcut_in **newcut),
    CCtsp_init_cutpool (int ncount, char *poolfilename, CCtsp_lpcuts **pool),
    CCtsp_write_cutpool (int ncount, char *poolfilename, CCtsp_lpcuts  *pool),
    CCtsp_search_cutpool (CCtsp_lpcuts *pool, CCtsp_lpcut_in **cuts,
            int *cutcount, int ncount, int ecount, int *elist, double *x),
    CCtsp_search_cutpool_cliques (CCtsp_lpcuts *pool, CCtsp_lpclique **cliques,
            int *cliquecount, int ncount, int ecount, int *elist, double *x,
            double maxdelta, int maxcliques, double **cliquevals),
    CCtsp_branch_cutpool_cliques (CCtsp_lpcuts *pool, CCtsp_lpclique **cliques,
            int *cliquecount, int ncount, int ecount, int *elist, double *x,
            int nwant, double **cliquevals),
    CCtsp_add_to_cutpool (CCtsp_lpcuts *pool, CCtsp_lpcuts *cuts,
            CCtsp_lpcut *c),
    CCtsp_add_to_cutpool_lpcut_in (CCtsp_lpcuts *pool, CCtsp_lpcut_in *cut),
    CCtsp_display_cutpool (CCtsp_lpcuts *pool),
    CCtsp_price_cuts (CCtsp_lpcuts *pool, int ncount, int ecount, int *elist,
            double *x, double *cutval),
    CCtsp_clique_to_array (CCtsp_lpclique *c, int **ar, int *count),
    CCtsp_clique_delta (CCtsp_lpgraph *g, double *x, CCtsp_lpclique *c,
            double *delta),
    CCtsp_x_greedy_tour (CCdatagroup *dat, int ncount, int ecount, int *elist,
            double *x, int *cyc, double *val),
    CCtsp_x_greedy_tour_lk (CCdatagroup *dat, int ncount, int ecount,
            int *elist, double *x, int *cyc, double *val);

void
    CCtsp_init_tsp_lp_struct (CCtsp_lp *lp),
    CCtsp_free_tsp_lp_struct (CCtsp_lp **lp),
    CCtsp_add_cuts_to_queue (CCtsp_lp *lp, CCtsp_lpcut_in **c),
    CCtsp_delete_cut_from_cutlist (CCtsp_lpcuts *cuts, int ind),
    CCtsp_init_lprow (CCtsp_lprow *cr),
    CCtsp_free_lprow (CCtsp_lprow *cr),
    CCtsp_unregister_cliques (CCtsp_lpcuts *cuts, CCtsp_lpcut *c),
    CCtsp_free_cutpool (CCtsp_lpcuts **pool),
    CCtsp_init_lpgraph_struct (CCtsp_lpgraph *g),
    CCtsp_free_lpgraph (CCtsp_lpgraph *g),
    CCtsp_free_lpcut_in (CCtsp_lpcut_in *c),
    CCtsp_free_lpclique (CCtsp_lpclique *c),
    CCtsp_free_bigdual (CCtsp_bigdual **d),
    CCtsp_init_branchobj (CCtsp_branchobj *b),
    CCtsp_free_branchobj (CCtsp_branchobj *b),
    CCtsp_print_branchhistory (CCtsp_lp *lp),
    CCtsp_init_tighten_info (CCtsp_tighten_info *stats),
    CCtsp_print_tighten_info (CCtsp_tighten_info *stats),
    CCtsp_mark_clique (CCtsp_lpclique *c, int *marks, int marker),
    CCtsp_mark_clique_and_neighbors (CCtsp_lpgraph *g, CCtsp_lpclique *c,
           int *marks, int marker),
    CCtsp_mark_cut (CCtsp_lpcut_in *c, int *marks, int marker),
    CCtsp_mark_cut_and_neighbors (CCtsp_lpgraph *g, CCtsp_lpcut_in *c,
           int *marks, int marker),
    CCtsp_mark_clique_and_neighbors_double (CCtsp_lpgraph *g, CCtsp_lpclique *c,
           double *marks, double marker),
    CCtsp_is_clique_marked (CCtsp_lpclique *c, int *marks, int marker,
           int *yes_no),
    CCtsp_clique_count (CCtsp_lpclique *c, int *count);


double
    CCtsp_cutprice (CCtsp_lpgraph *g, CCtsp_lpcut_in *c, double *x);

#else

int
    CCtsp_cutting_loop (),
    CCtsp_subtour_loop (),
    CCtsp_pricing_loop (),
    CCtsp_init_cutselect (),
    CCtsp_call_x_heuristic (),
    CCtsp_bb_cutting (),
    CCtsp_init_lp (),
    CCtsp_bb_init_lp (),
    CCtsp_get_lp_result (),
    CCtsp_process_cuts (),
    CCtsp_add_cut_to_cutlist (),
    CCtsp_add_cut (),
    CCtsp_lpcut_in_nzlist (),
    CCtsp_add_nzlist_to_lp (),
    CCtsp_add_vars_to_lp (),
    CCtsp_update_result (),
    CCtsp_infeas_recover (),
    CCtsp_test_cut_branch (),
    CCtsp_register_cliques (),
    CCtsp_addbad_variables (),
    CCtsp_eliminate_variables (),
    CCtsp_build_lpgraph (),
    CCtsp_build_lpadj (),
    CCtsp_add_multiple_rows (),
    CCtsp_delete_cut (),
    CCtsp_find_edge (),
    CCtsp_find_branch (),
    CCtsp_bb_find_branch (),
    CCtsp_check_integral (),
    CCtsp_find_branch_edge (),
    CCtsp_find_branch_cliques (),
    CCtsp_execute_branch (),
    CCtsp_execute_unbranch (),
    CCtsp_splitprob (),
    CCtsp_bb_splitprob (),
    CCtsp_dumptour (),
    CCtsp_add_branchhistory_to_lp (),
    CCtsp_easy_dfs_brancher (),
    CCtsp_bfs_brancher (),
    CCtsp_do_interactive_branch (),
    CCtsp_inspect_full_edges (),
    CCtsp_read_probfile (),
    CCtsp_read_probfile_id (),
    CCtsp_write_probfile_sav (),
    CCtsp_write_probfile_id (),
    CCtsp_dump_x (),
    CCtsp_exact_price (),
    CCtsp_edge_elimination (),
    CCtsp_exact_dual (),
    CCtsp_verify_infeasible_lp (),
    CCtsp_verify_lp_prune (),
    CCtsp_tighten_lpcut_in (),
    CCtsp_tighten_lpcut (),
    CCtsp_test_pure_comb (),
    CCtsp_test_pseudocomb (),
    CCtsp_test_teeth_disjoint (),
    CCtsp_find_pure_handle (),
    CCtsp_comb_to_double_decker (),
    CCtsp_teething (),
    CCtsp_init_cutpool (),
    CCtsp_write_cutpool (),
    CCtsp_search_cutpool (),
    CCtsp_search_cutpool_cliques (),
    CCtsp_branch_cutpool_cliques (),
    CCtsp_add_to_cutpool (),
    CCtsp_add_to_cutpool_lpcut_in (),
    CCtsp_display_cutpool (),
    CCtsp_price_cuts (),
    CCtsp_clique_to_array (),
    CCtsp_clique_delta (),
    CCtsp_x_greedy_tour (),
    CCtsp_x_greedy_tour_lk ();

void
    CCtsp_init_tsp_lp_struct (),
    CCtsp_free_tsp_lp_struct (),
    CCtsp_add_cuts_to_queue (),
    CCtsp_delete_cut_from_cutlist (),
    CCtsp_init_lprow (),
    CCtsp_free_lprow (),
    CCtsp_unregister_cliques (),
    CCtsp_free_cutpool (),
    CCtsp_init_lpgraph_struct (),
    CCtsp_free_lpgraph (),
    CCtsp_free_lpcut_in (),
    CCtsp_free_lpclique (),
    CCtsp_free_bigdual (),
    CCtsp_init_branchobj (),
    CCtsp_free_branchobj (),
    CCtsp_print_branchhistory (),
    CCtsp_init_tighten_info (),
    CCtsp_print_tighten_info (),
    CCtsp_mark_clique (),
    CCtsp_mark_clique_and_neighbors (),
    CCtsp_mark_cut (),
    CCtsp_mark_cut_and_neighbors (),
    CCtsp_mark_clique_and_neighbors_double (),
    CCtsp_is_clique_marked (),
    CCtsp_clique_count ();


double
    CCtsp_cutprice ();

#endif

/***************************************************************************/
/*                                                                         */
/*                             cliqhash.c                                  */
/*                                                                         */
/***************************************************************************/

#ifdef CC_PROTOTYPE_ANSI

int
    CCtsp_init_cliquehash (CCtsp_lpcuts *cuts, int size),
    CCtsp_register_clique (CCtsp_lpcuts *cuts, CCtsp_lpclique *c);

void
    CCtsp_free_cliquehash (CCtsp_lpcuts *cuts),
    CCtsp_unregister_clique (CCtsp_lpcuts *cuts, int c);

#else

int
    CCtsp_init_cliquehash (),
    CCtsp_register_clique ();

void
    CCtsp_free_cliquehash (),
    CCtsp_unregister_clique ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             cutcall.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct cutinfo {
    CC_SRKexpinfo expand;
    CCtsp_lpcut_in **clist;
    CCtsp_lpcut_in *current;
    int *cutcount;
} cutinfo;

#ifdef CC_PROTOTYPE_ANSI

int
    CCtsp_connect_cuts (CCtsp_lpcut_in **cuts, int *cutcount, int ncount,
            int ecount, int *elist, double *x),
    CCtsp_segment_cuts (CCtsp_lpcut_in **cuts, int *cutcount, int ncount,
            int ecount, int *elist, double *x),
    CCtsp_exact_subtours (CCtsp_lpcut_in **cuts, int *cutcount, int ncount,
            int ecount, int *elist, double *x),
    CCtsp_tighten_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
            CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
            int *elist, double *x, double testtol, int maxcuts),
    CCtsp_double_decker_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
            CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
            int *elist, double *x, double testtol, int maxcuts),
    CCtsp_teething_lp (CCtsp_lpcuts *cuts, CCtsp_tighten_info *stats,
            CCtsp_lpcut_in **cutsout, int *cutcount, int ncount, int ecount,
            int *elist, double *x, double testtol, int maxcuts),
    CCtsp_copy_lpcut_in (CCtsp_lpcut_in *c, CCtsp_lpcut_in *new),
    CCtsp_segment_to_subtour (CCtsp_lpcut_in **cut, int a, int b),
    CCtsp_array_to_subtour (CCtsp_lpcut_in **cut, int *ar, int acount),
    CCtsp_array_to_lpclique (int *ar, int acount, CCtsp_lpclique *cliq),
    CCtsp_seglist_to_lpclique (int nseg, int *list, CCtsp_lpclique *cliq),
    CCtsp_add_node_to_lpclique (CCtsp_lpclique *cin, CCtsp_lpclique *cout,
            int n),
    CCtsp_delete_node_from_lpclique (CCtsp_lpclique *cin,
            CCtsp_lpclique *cout, int n),
    CCtsp_lpcut_to_lpcut_in (CCtsp_lpcuts *cuts, CCtsp_lpcut *c,
            CCtsp_lpcut_in *new),
    CCtsp_copy_lpclique (CCtsp_lpclique *c, CCtsp_lpclique *new),
    CCtsp_file_cuts (char *cutfile, CCtsp_lpcut_in **cuts, int *cutcount,
            int ncount, int *tour),
    CCtsp_file_cuts_write (char *cutfile, CCtsp_lpcuts *cuts, int *tour),
    CCtsp_buildcut_begin (cutinfo *cuts, int init_cliquecount),
    CCtsp_buildcut_addclique (cutinfo *cuts, int *arr, int size, int handle);

void
    CCtsp_init_lpcut_in (CCtsp_lpcut_in *c),
    CCtsp_init_lpclique (CCtsp_lpclique *c),
    CCtsp_print_lpcut_in (CCtsp_lpcut_in *c),
    CCtsp_print_lpclique (CCtsp_lpclique *c),
    CCtsp_lpclique_compare (CCtsp_lpclique *a, CCtsp_lpclique *b, int *diff),
    CCtsp_buildcut_abort (cutinfo *cuts),
    CCtsp_buildcut_finish (cutinfo *cuts, int rhs);

#else

int
    CCtsp_connect_cuts (),
    CCtsp_segment_cuts (),
    CCtsp_exact_subtours (),
    CCtsp_tighten_lp (),
    CCtsp_double_decker_lp (),
    CCtsp_teething_lp (),
    CCtsp_copy_lpcut_in (),
    CCtsp_segment_to_subtour (),
    CCtsp_array_to_subtour (),
    CCtsp_array_to_lpclique (),
    CCtsp_seglist_to_lpclique (),
    CCtsp_add_node_to_lpclique (),
    CCtsp_delete_node_from_lpclique (),
    CCtsp_lpcut_to_lpcut_in (),
    CCtsp_copy_lpclique (),
    CCtsp_file_cuts (),
    CCtsp_file_cuts_write (),
    CCtsp_buildcut_begin (),
    CCtsp_buildcut_addclique ();

void
    CCtsp_init_lpcut_in (),
    CCtsp_init_lpclique (),
    CCtsp_print_lpcut_in (),
    CCtsp_print_lpclique (),
    CCtsp_lpclique_compare (),
    CCtsp_buildcut_abort (),
    CCtsp_buildcut_finish ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             edgemap.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct CCtsp_edgeinf {
    int ends[2];
    int val;
    struct CCtsp_edgeinf *next;
} CCtsp_edgeinf;

typedef struct CCtsp_edgehash {
    struct CCtsp_edgeinf **table;
    unsigned int size;
    unsigned int mult;
} CCtsp_edgehash;


#ifdef CC_PROTOTYPE_ANSI

int
    CCtsp_edgehash_init (CCtsp_edgehash *h, int size),
    CCtsp_edgehash_add (CCtsp_edgehash *h, int end1, int end2, int val),
    CCtsp_edgehash_del (CCtsp_edgehash *h, int end1, int end2),
    CCtsp_edgehash_find (CCtsp_edgehash *h, int end1, int end2);

void
    CCtsp_edgehash_delall (CCtsp_edgehash *h),
    CCtsp_edgehash_free (CCtsp_edgehash *h);

#else

int
    CCtsp_edgehash_init (),
    CCtsp_edgehash_add (),
    CCtsp_edgehash_del (),
    CCtsp_edgehash_find ();

void
    CCtsp_edgehash_delall (),
    CCtsp_edgehash_free ();

#endif


/***************************************************************************/
/*                                                                         */
/*                             generate.c                                  */
/*                                                                         */
/***************************************************************************/

#define CCtsp_PRICE_COMPLETE_GRAPH -1
#define CCtsp_GEN_PRICE_EPSILON 0.0001 /* 0.0000001 */
#define CCtsp_GEN_USE_ADJ 50           /* Cutoff for using explicit adj list */

#ifdef CC_PROTOTYPE_ANSI

void
    CCtsp_free_edgegenerator (CCtsp_edgegenerator *eg);

int
    CCtsp_init_edgegenerator (CCtsp_edgegenerator *eg, int ncount,
            CCdatagroup *dg, CCtsp_genadj *adj, int nneighbors),
    CCtsp_reset_edgegenerator (CCtsp_edgegenerator *eg, double *node_piest),
    CCtsp_generate_edges (CCtsp_edgegenerator *eg, int nwant, int *pngot,
            int *elist, int *elen, int *finished),
    CCtsp_edgelist_to_genadj (int ncount, int ecount, int *elist, int *elen,
            CCtsp_genadj **adj, CCtsp_genadjobj **adjobjspace);

#else

void
    CCtsp_free_edgegenerator ();

int
    CCtsp_init_edgegenerator (),
    CCtsp_reset_edgegenerator (),
    CCtsp_generate_edges (),
    CCtsp_edgelist_to_genadj ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             prob_io.c                                   */
/*                                                                         */
/***************************************************************************/

#define CCtsp_PROB_IO_VERSION  1
#define CCtsp_PROB_FILE_NAME_LEN 128

#define CCtsp_PROB_IO_CUTS_VERSION_BASE  -1000
#define CCtsp_PROB_IO_CUTS_VERSION       -1001   /* Should be <= BASE (-1000) */

typedef struct CCtsp_PROB_FILE {
    CC_SFILE *f;
    char name[CCtsp_PROB_FILE_NAME_LEN];
    int id;
    int parent;
    double ub;
    double lb;
    CCbigguy exactlb;
    int nnodes;
    int child0;
    int child1;
    int real;       /* Set to 1 when we know this is a real child */
    int processed;
    int infeasible;
    struct {
        int dat;
        int edge;
        int fulladj;
        int cut;
        int tour;
        int basis;
        int norms;
        int fix;
        int exactdual;
        int history;
    } offsets;
} CCtsp_PROB_FILE;


#ifdef CC_PROTOTYPE_ANSI

CCtsp_PROB_FILE
    *CCtsp_prob_read (char *f, int n),
    *CCtsp_prob_read_name (char *f),
    *CCtsp_prob_write (char *f, int n),
    *CCtsp_prob_write_name (char *fname, char *pname);

int
    CCtsp_prob_file_delete (char *f, int n),
    CCtsp_prob_getname (CCtsp_PROB_FILE *p, char *name),
    CCtsp_prob_getid (CCtsp_PROB_FILE *p, int *id),
    CCtsp_prob_getparent (CCtsp_PROB_FILE *p, int *parent),
    CCtsp_prob_getub (CCtsp_PROB_FILE *p, double *ub),
    CCtsp_prob_getlb (CCtsp_PROB_FILE *p, double *lb),
    CCtsp_prob_getexactlb (CCtsp_PROB_FILE *p, CCbigguy *lb),
    CCtsp_prob_getnnodes (CCtsp_PROB_FILE *p, int *nnodes),
    CCtsp_prob_getchildren (CCtsp_PROB_FILE *p, int *child0, int *child1),
    CCtsp_prob_getreal (CCtsp_PROB_FILE *p, int *real),
    CCtsp_prob_getprocessed (CCtsp_PROB_FILE *p, int *processed),
    CCtsp_prob_getinfeasible (CCtsp_PROB_FILE *p, int *infeasible),
    CCtsp_prob_gettour (CCtsp_PROB_FILE *p, int **tour),
    CCtsp_prob_getedges (CCtsp_PROB_FILE *p, int *nedges, int **elist,
        int **elen),
    CCtsp_prob_getcuts (CCtsp_PROB_FILE *p, CC_SFILE *s, CCtsp_lpcuts *cuts),
    CCtsp_prob_getbasis (CCtsp_PROB_FILE *p, int *ccount, int *rcount,
        int **cstat, int **rstat),
    CCtsp_prob_getnorms (CCtsp_PROB_FILE *p, int *rcount, double **dnorm),
    CCtsp_prob_getfulladj (CCtsp_PROB_FILE *p, int ncount, int *fullcount,
        CCtsp_genadj **adj, CCtsp_genadjobj **adjspace),
    CCtsp_prob_getfixed (CCtsp_PROB_FILE *p, int *ecount, int **elist),
    CCtsp_prob_getexactdual (CCtsp_PROB_FILE *p, int ncount,
        CCtsp_bigdual **d),
    CCtsp_prob_gethistory (CCtsp_PROB_FILE *p, int *depth,
        CCtsp_branchobj **history),
    CCtsp_prob_rclose (CCtsp_PROB_FILE *p),

    CCtsp_prob_putname (CCtsp_PROB_FILE *p, char *name),
    CCtsp_prob_putid (CCtsp_PROB_FILE *p, int id),
    CCtsp_prob_putparent (CCtsp_PROB_FILE *p, int parent),
    CCtsp_prob_putub (CCtsp_PROB_FILE *p, double ub),
    CCtsp_prob_putlb (CCtsp_PROB_FILE *p, double lb),
    CCtsp_prob_putexactlb (CCtsp_PROB_FILE *p, CCbigguy lb),
    CCtsp_prob_putnnodes (CCtsp_PROB_FILE *p, int nnodes),
    CCtsp_prob_putchildren (CCtsp_PROB_FILE *p, int child0, int child1),
    CCtsp_prob_putreal (CCtsp_PROB_FILE *p, int real),
    CCtsp_prob_putprocessed (CCtsp_PROB_FILE *p, int processed),
    CCtsp_prob_putinfeasible (CCtsp_PROB_FILE *p, int infeasible),
    CCtsp_prob_puttour (CCtsp_PROB_FILE *p, int *tour),
    CCtsp_prob_putedges (CCtsp_PROB_FILE *p, int nedges, int *elist, int *elen),
    CCtsp_prob_putcuts (CCtsp_PROB_FILE *p, CC_SFILE *s, CCtsp_lpcuts *cuts),
    CCtsp_prob_putbasis (CCtsp_PROB_FILE *p, int ccount, int rcount, int *cstat,
        int *rstat),
    CCtsp_prob_putnorms (CCtsp_PROB_FILE *p, int rcount, double *dnorm),
    CCtsp_prob_putfulladj (CCtsp_PROB_FILE *p, int ncount, int fullcount,
        CCtsp_genadj *adj),
    CCtsp_prob_putfixed (CCtsp_PROB_FILE *p, int ecount, int *elist),
    CCtsp_prob_putexactdual (CCtsp_PROB_FILE *p, CCtsp_bigdual *d, int ncount),
    CCtsp_prob_puthistory (CCtsp_PROB_FILE *p, int depth,
        CCtsp_branchobj *history),
    CCtsp_prob_wclose (CCtsp_PROB_FILE *p);

#else

CCtsp_PROB_FILE
    *CCtsp_prob_read (),
    *CCtsp_prob_read_name (),
    *CCtsp_prob_write (),
    *CCtsp_prob_write_name ();

int
    CCtsp_prob_file_delete (),
    CCtsp_prob_getname (),
    CCtsp_prob_getid (),
    CCtsp_prob_getparent (),
    CCtsp_prob_getub (),
    CCtsp_prob_getlb (),
    CCtsp_prob_getexactlb (),
    CCtsp_prob_getnnodes (),
    CCtsp_prob_getchildren (),
    CCtsp_prob_getreal (),
    CCtsp_prob_getprocessed (),
    CCtsp_prob_getinfeasible (),
    CCtsp_prob_gettour (),
    CCtsp_prob_getedges (),
    CCtsp_prob_getcuts (),
    CCtsp_prob_getbasis (),
    CCtsp_prob_getnorms (),
    CCtsp_prob_getfulladj (),
    CCtsp_prob_getfixed (),
    CCtsp_prob_getexactdual (),
    CCtsp_prob_gethistory (),
    CCtsp_prob_rclose (),

    CCtsp_prob_putname (),
    CCtsp_prob_putid (),
    CCtsp_prob_putparent (),
    CCtsp_prob_putub (),
    CCtsp_prob_putlb (),
    CCtsp_prob_putexactlb (),
    CCtsp_prob_putnnodes (),
    CCtsp_prob_putchildren (),
    CCtsp_prob_putreal (),
    CCtsp_prob_putprocessed (),
    CCtsp_prob_putinfeasible (),
    CCtsp_prob_puttour (),
    CCtsp_prob_putedges (),
    CCtsp_prob_putcuts (),
    CCtsp_prob_putbasis (),
    CCtsp_prob_putnorms (),
    CCtsp_prob_putfulladj (),
    CCtsp_prob_putfixed (),
    CCtsp_prob_putexactdual (),
    CCtsp_prob_puthistory (),
    CCtsp_prob_wclose ();

#endif



/***************************************************************************/
/*                                                                         */
/*                             qsparse.c                                   */
/*                                                                         */
/***************************************************************************/

typedef struct CCtsp_qsparsegroup {
    CCdheap *add_queue;   /* An empty heap will be maintained */
    CCdheap *sub_queue;   /* An empty heap will be maintained */
    int *count_m1;        /* The array will be maintained at 0 */
    int *count_non0;      /* The array will be maintained at 0 */
    int *count_1;         /* The array will be maintained at 0 */
    int *on_add_queue;    /* The array will be maintained at 0 */
    int *on_sub_queue;    /* The array will be maintained at 0 */
    int *mults;           /* The array will be maintained at 0 */
} CCtsp_qsparsegroup;

#ifdef CC_PROTOTYPE_ANSI

void
    CCtsp_free_qsparsify (CCtsp_qsparsegroup **pqs);
int
    CCtsp_qsparsify (CCtsp_qsparsegroup **pqs, struct CCtsp_lpgraph *g,
            int *pnzlist, int *scount, struct CCtsp_sparser **slist,
            int *savedcount);
#else

void
    CCtsp_free_qsparsify ();
int
    CCtsp_qsparsify ();

#endif

#endif  /* __TSP_H */
#ifndef __XSTUFF_H
#define __XSTUFF_H

#ifdef CC_PROTOTYPE_ANSI

int
    Xfastcuts (CCtsp_lpcut_in **cuts, int *cutcount, int ncount, int ecount,
               int *elist, double *x),
    Xslowcuts (CCtsp_lpcut_in **cuts, int *cutcount, int ncount, int ecount,
               int *elist, double *x),
    Xfastsubtours (CCtsp_lpcut_in **cuts, int *cutcount, int ncount, int ecount,
               int *elist, double *x),
    Xexactsubtours (CCtsp_lpcut_in **cuts, int *cutcount, int ncount, int ecount,
               int *elist, double *x),
    Xcliquetrees (CCtsp_lpcut_in **cuts, int *cutcount, int ncount, int ecount,
               int *elist, double *x),
    Xconsecutiveones (CCtsp_lpcut_in **cuts, int *cutcount, int ncount, int ecount,
                      int *elist, double *x, CCtsp_lpcuts *pool),
    Xnecklacecuts (CCtsp_lpcut_in **cuts, int *cutcount, int ncount, int ecount,
                      int *elist, double *x, CCtsp_lpcuts *pool);

#else

int
    Xfastcuts (),
    Xslowcuts (),
    Xfastsubtours (),
    Xexactsubtours (),
    Xcliquetrees (),
    Xconsecutiveones (),
    Xnecklacecuts ();

#endif

#endif  /* __XSTUFF_H */
#ifndef __FMATCH_H
#define __FMATCH_H


#ifdef CC_PROTOTYPE_ANSI

int
    CCfmatch_fractional_2match (int ncount, int ecount, int *elist, int *elen,
        CCdatagroup *dat, double *val, int *thematching, int *thedual,
        int *thebasis, int wantbasic);

#else

int
    CCfmatch_fractional_2match ();

#endif

#endif  /* __FMATCH_H */
#ifndef  __LINKERN_H
#define  __LINKERN_H


#ifdef CC_PROTOTYPE_ANSI

int
    CClinkern_tour (int ncount, CCdatagroup *dat, int ecount,
            int *elist, int stallcount, int repeatcount, int *incycle,
            int *outcycle, double *val, int run_silently, double time_bound,
            double length_bound, char *saveit_name);

#else

int
    CClinkern_tour ();

#endif



/***************************************************************************/
/*                                                                         */
/* Must define exactly one of:                                             */
/*                                                                         */
/*          CC_LINKED_LIST_FLIPPER       (flip_llX)                        */
/*          CC_ARRAY_FLIPPER             (flip_ary)                        */
/*          CC_TWO_LEVEL_FLIPPER         (flip_two)                        */
/*          CC_SEGMENTS_FLIPPER          (flip_sg1)                        */
/*          CC_NO_UNDO_SEGMENTS_FLIPPER  (flip_sg2)                        */
/*          CC_FULL_SEGMENTS_FLIPPER     (flip_sg3)                        */
/*          CC_SPLAY_FLIPPER             (flip_sp2)                        */
/*          CC_BTREE_FLIPPER             (flip_btr)                        */
/*                                                                         */
/* NOTE: If MARK_NEIGHBORS is not defined in linkern.c, then               */
/*       NO_UNDO_SEGMENTS may follow a different improving sequence then   */
/*       the other flippers, since the next and prevs in turn () will be   */
/*       with respect to an out-of-date-tour.                              */
/*                                                                         */
/***************************************************************************/


#define CC_TWO_LEVEL_FLIPPER
/* #define BTREE_FLIPPER */

#ifdef CC_LINKED_LIST_FLIPPER
#define CC_EXTRA_INFO_FLIP
#endif

#ifdef CC_ARRAY_FLIPPER
#define CC_USE_FLIP_CLEANING
#endif

#ifdef CC_TWO_LEVEL_FLIPPER
#define CC_USE_FLIP_CLEANING
#endif

#ifdef CC_NO_UNDO_SEGMENTS_FLIPPER
#define CC_USE_FLIP_CLEANING
#define CC_USE_QUICK_FLIPS
#endif

#ifdef CC_FULL_SEGMENTS_FLIPPER
#define CC_USE_FLIP_CLEANING
#endif

#ifdef CC_SPLAY_FLIPPER
#define CC_USE_FLIP_CLEANING
#define CC_EXTRA_INFO_FLIP
#endif

#ifdef CC_BTREE_FLIPPER
#define CC_USE_FLIP_CLEANING
#define CC_EXTRA_INFO_FLIP
#endif

#ifdef CC_PROTOTYPE_ANSI

int
    CClinkern_flipper_init (int ncount, int *cyc),
    CClinkern_flipper_reset_perm (int ncount),
    CClinkern_flipper_reset_temp (int ncount),
    CClinkern_flipper_next (int x),
    CClinkern_flipper_prev (int x),
    CClinkern_flipper_cycle (int *x),
    CClinkern_flipper_sequence_burst (int x, int y, int z),
    CClinkern_flipper_sequence (int x, int y, int z);

void
#ifdef CC_EXTRA_INFO_FLIP
    CClinkern_flipper_flip (int xprev, int x, int y, int ynext),
#else
    CClinkern_flipper_flip (int x, int y),
#endif
    CClinkern_flipper_flip_quick (int x, int y),
    CClinkern_flipper_flip_perm (int x, int y),
    CClinkern_flipper_sequence_burst_init (int x, int z),
    CClinkern_flipper_finish (void),
    CClinkern_flipper_free_world (void);

#else

int
    CClinkern_flipper_init (),
    CClinkern_flipper_reset_perm (),
    CClinkern_flipper_reset_temp (),
    CClinkern_flipper_next (),
    CClinkern_flipper_prev (),
    CClinkern_flipper_cycle (),
    CClinkern_flipper_sequence_burst (),
    CClinkern_flipper_sequence ();

void
    CClinkern_flipper_flip (),
    CClinkern_flipper_flip_quick (),
    CClinkern_flipper_flip_perm (),
    CClinkern_flipper_sequence_burst_init (),
    CClinkern_flipper_finish (),
    CClinkern_flipper_free_world ();

#endif

#endif  /* __LINKERN_H */
#ifndef  __MACRORUS_H
#define  __MACRORUS_H

#define CC_SWAP(a,b,t) (((t)=(a)),((a)=(b)),((b)=(t)))

#define CC_OURABS(a) (((a) >= 0) ? (a) : -(a))

#endif  /* __MACRORUS_H */
