#include <gnome.h>
#include <libgnomeui/gnome-app.h>
#include "glotski.h"

#include "icon.xpm"

#define MYGRAY 164
#define LIGHTEN(a) (255-(255-a)/2)
#define DARKEN(a) (a/2)

#define GNOMEUIINFO_ITEM_STOCK_DATA_(label, tooltip, cb, data, stock_id) { GNOME_APP_UI_ITEM, label, tooltip, (gpointer)cb, data, NULL, GNOME_APP_PIXMAP_STOCK, stock_id, 0, (GdkModifierType) 0, NULL }
#define GNOMEUIINFO_TOGGLEITEM_NONE_DATA_(label, cb, data) { GNOME_APP_UI_TOGGLEITEM, label, NULL, (gpointer)cb, data, NULL, GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType) 0, NULL }

static void select_file(GtkWidget *widget, pair *mypair);


/* The canvas drag logic was originally based on Havoc Pennington's code
   from his GNOME book; however, it has been modified extensively.
*/
/* This shouldn't be necessary, but the rectangle item has slightly 
   different dimensions than the equivalent polygon item, so it can look 
   ugly.
*/
#define UGLY_RECTANGLE

#ifdef UGLY_RECTANGLE
void drawrect(GnomeCanvasGroup *group, double x1, double y1, 
	      double x2, double y2, int r, int g, int b) {
  GnomeCanvasPoints *ptmp = gnome_canvas_points_new(4);
  ptmp->coords[0] = x1;
  ptmp->coords[1] = y1;

  ptmp->coords[2] = x1;
  ptmp->coords[3] = y2;

  ptmp->coords[4] = x2;
  ptmp->coords[5] = y2;

  ptmp->coords[6] = x2;
  ptmp->coords[7] = y1;
  
  gnome_canvas_item_new(group,
			gnome_canvas_polygon_get_type(),
			"points", ptmp,
			"fill_color_rgba",
			GNOME_CANVAS_COLOR(r, g, b), 
			"width_pixels", 0,
			NULL);
}
#endif

void textdisp(char *text) {
  gnome_ok_dialog(N_(text));
}

void errordisp(char *text) {
  g_error(_(text));
}

void moveelement(glot *myglot, int xloc, int yloc) {
  double xblock = myglot->parent->usize[X]*myglot->parent->zoom[X];
  double yblock = myglot->parent->usize[Y]*myglot->parent->zoom[Y];
  double visx = xblock*(double)myglot->loc[X];
  double visy = yblock*(double)myglot->loc[Y];
  double newvisx = xblock*(double)xloc;
  double newvisy = xblock*(double)yloc;
  gnome_canvas_item_move((GnomeCanvasItem *)myglot->element, 
			 newvisx - visx, newvisy - visy);
  glotplace(myglot, xloc, yloc, TRUE);
}

static gint glot_event(GnomeCanvasItem *glotitem, GdkEvent *event, 
		       glot *myglot) {
  static double visx, visy; /* The visible X and Y coords */
  static int ipropx, ipropy; /* The proper (where item is located) X and Y */
  /* Directions tested from propx and propy */

  int inewx, inewy;
  GdkCursor *arrow;
  static int dragging;
  double mousex, mousey; /* Where the mouse is */
  double newvisx, newvisy; /* Where the piece will be */
  double dx, dy;

  static double anchorx, anchory;
  static int origx, origy;

  double xblocksize = myglot->parent->usize[X]*myglot->parent->zoom[X];
  double yblocksize = myglot->parent->usize[Y]*myglot->parent->zoom[Y];

  int valid = TRUE; /* Actually draggable? */

  mousex = event->button.x;
  mousey = event->button.y;
  gnome_canvas_item_w2i(glotitem->parent, &mousex, &mousey);

  switch (event->type) 
    {
    case GDK_BUTTON_PRESS:
      switch (myglot->type) {
      case IMMOBILE:
	arrow = gdk_cursor_new(GDK_X_CURSOR);
	valid = FALSE;
	break;
      case HORIZONTAL:
	arrow = gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);
	break;
      case VERTICAL:
	arrow = gdk_cursor_new(GDK_SB_V_DOUBLE_ARROW);
	break;
      case BIMOBILE:
	arrow = gdk_cursor_new(GDK_FLEUR);
	break;
      default:
	arrow = NULL;
	break;
      }      
      gnome_canvas_item_grab(glotitem,
			     GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
			     arrow, event->button.time);
      ipropx = origx = myglot->loc[X]; ipropy = origy = myglot->loc[Y];
      visx = anchorx = mousex; visy = anchory = mousey;
      gdk_cursor_destroy(arrow);
      if (valid) {
	dragging = TRUE;
	if (!(myglot->parent->move && !myglot->parent->move->next && 
	      myglot == myglot->parent->move->myglot && 
	      !myglot->parent->move->goalsat)) {
	  if (myglot->parent->move && myglot->parent->move->next) {
	    deletemove(myglot->parent->move->next);
	    myglot->parent->move->next = NULL;
	  }
	  makemove(myglot->parent);
	  myglot->parent->move->startloc[X] = myglot->loc[X];
	  myglot->parent->move->startloc[Y] = myglot->loc[Y];
	  myglot->parent->move->myglot = myglot;
	  myglot->parent->nummoves++;
	  updatestatus(myglot->parent);
	}
      }
      break;  
    case GDK_MOTION_NOTIFY:
      if (dragging && (event->motion.state & GDK_BUTTON1_MASK)) 
	{
	  /* First, resolve the new proper location of the piece */
	  inewx = (COMPAT(myglot, LEFT)) 
	    ? (int)((mousex-anchorx) / xblocksize) + origx : origx;
	  inewy = (COMPAT(myglot, UP)) 
	    ? (int)((mousey-anchory) / xblocksize) + origy : origy;

	  if (inewx < 0) inewx = 0;
	  if (inewx + myglot->size[X] >= myglot->parent->size[X]) 
	    inewx = myglot->parent->size[X]-myglot->size[X];
	  if (inewy < 0) inewy = 0;
	  if (inewy + myglot->size[Y] >= myglot->parent->size[Y]) 
	    inewy = myglot->parent->size[Y]-myglot->size[Y];

	  /*	  
	  if (ipropx != inewx || ipropy != inewy) 
	    printf("Attempting move from (%d, %d) [(%d, %d)] to (%d, %d)\n", 
		   ipropx, ipropy, myglot->loc[X], myglot->loc[Y], 
		   inewx, inewy);
	  */
	  while (ipropx != inewx &&  /* First, attempts proper X location */
		 glotmove(myglot, XDIR(inewx, ipropx), TRUE)) {
	    (inewx < ipropx) ? ipropx-- : ipropx++;
	  }

	  while (ipropy != inewy && /* Then, attempts proper Y location */
		 glotmove(myglot, YDIR(inewy, ipropy), TRUE)) {
	    (inewy < ipropy) ? ipropy-- : ipropy++;
	  }

	
	  if (ipropx != myglot->loc[X] || ipropy != myglot->loc[Y])
	    printf("Drag error: I think (%d, %d), should be (%d, %d)\n",
		   ipropx, ipropy, myglot->loc[X], myglot->loc[Y]);
	 
	  /* If the above occurs, that is bad.  
	     Next line should be redundant. */
	  ipropx = myglot->loc[X]; ipropy = myglot->loc[Y];


	  newvisx = (ipropx-origx)*xblocksize + anchorx;
	  newvisy = (ipropy-origy)*yblocksize + anchory;


	  /* Next, determine whether it can move into the new partial domain */
	  
	  if (ipropx == inewx && newvisx != mousex && 
	      glotmove(myglot, XDIR(mousex,newvisx), FALSE))  {	     
	    if (ipropy == inewy && newvisy != mousey &&
		glotmove(myglot, YDIR(mousey, newvisy), FALSE) && 
		glotplace(myglot, 
			  ipropx + ((mousex < newvisx) ? -1 : 1),
			  ipropy + ((mousey < newvisy) ? -1 : 1), FALSE)) {
	      newvisx = mousex; newvisy = mousey;
	    } 
	    else newvisx = mousex;
	  }
	  else 
	    if (ipropy == inewy && newvisy != mousey && 
		glotmove(myglot, YDIR(mousey, newvisy),FALSE)) 
	      newvisy = mousey;
	  
	  /* Finally, perform the move */

	  if ((newvisx != visx) || (newvisy != visy)) {
	    gnome_canvas_item_move(glotitem, newvisx - visx, newvisy - visy);
	    visx = newvisx;
	    visy = newvisy;
	  }
	}
      break;
      
    case GDK_BUTTON_RELEASE:
      gnome_canvas_item_ungrab(glotitem, event->button.time);
      dragging = FALSE;
      /* Must resolve final location */
      dx = (visx - (double)((ipropx-origx)*xblocksize + anchorx))/
	(double)(xblocksize);
      if (dx * dx > 0.25)
	glotmove(myglot, XDIR((int)visx, (ipropx-origx)*xblocksize + anchorx),
		 TRUE);

      dy = (visy - (double)((ipropy-origy)*yblocksize + anchory))/
	(double)(yblocksize);
      if (dy * dy > 0.25)
 	glotmove(myglot, YDIR((int)visy, (ipropy-origy)*yblocksize + anchory), 
		 TRUE);

      inewx = (COMPAT(myglot, LEFT)) 
	? (int)((mousex-anchorx) / xblocksize) + origx : origx;
      inewy = (COMPAT(myglot, UP)) 
	? (int)((mousey-anchory) / xblocksize) + origy : origy;

      newvisx = (myglot->loc[X]-origx)*xblocksize + anchorx;
      newvisy = (myglot->loc[Y]-origy)*yblocksize + anchory;
      gnome_canvas_item_move(glotitem, newvisx - visx, newvisy - visy);
      visx = newvisx;
      visy = newvisy;

      if (myglot->parent->move->startloc[X] == myglot->loc[X] &&
	  myglot->parent->move->startloc[Y] == myglot->loc[Y]) {
	popmove(&(myglot->parent->move));
	if (myglot->parent->move && myglot->parent->move->next) {
	  fprintf(stderr, "glotski: Corrupt movelist\n");
	  exit(1);
	}
	myglot->parent->nummoves--;
	updatestatus(myglot->parent);
      } else {
	myglot->parent->move->endloc[X] = myglot->loc[X];
	myglot->parent->move->endloc[Y] = myglot->loc[Y];
      }
      if (goalequiv(myglot->parent)) { 
	myglot->parent->move->goalsat = TRUE;
	myglot->parent->state++;
	updatestatus(myglot->parent);
	if (myglot->parent->mygoal->next) {
	  myglot->parent->mygoal = myglot->parent->mygoal->next;
	  redrawgoal(myglot->parent);
	}
	else if (!myglot->parent->won) {
	  textdisp("You won!");	
	  myglot->parent->won = TRUE;
	}
      }
      break;
    default:
      break;
    }
  
  return FALSE;
}

static void goalgen(GnomeCanvasGroup *group, level *mylevel) { /* Draw goalwin */
  int x, y;
  double xblock = mylevel->usize[X]*mylevel->gzoom[X];
  double yblock = mylevel->usize[Y]*mylevel->gzoom[Y];
  glot *myglot;
  group = GNOME_CANVAS_GROUP 
    (gnome_canvas_item_new(group,
			   gnome_canvas_group_get_type(),
			   "x", 0.0,
			   "y", 0.0,
			   NULL));

  /* OK to use rectangles because only using rectangles */
   for (x = 0; x < mylevel->mygoal->size[X]; x++)
    for (y = 0; y < mylevel->mygoal->size[Y]; y++) {
      if (!(myglot = LOOKUP(mylevel->mygoal, x, y))) {
	gnome_canvas_item_new(group,
			      gnome_canvas_rect_get_type(),
			      "x1", (double)(x)*xblock,
			      "y1", (double)(y)*yblock,
			      "x2", (double)(x+1)*xblock,
			      "y2", (double)(y+1)*yblock,
			      "fill_color_rgba",
			      GNOME_CANVAS_COLOR(MYGRAY, MYGRAY, MYGRAY),
			      "width_pixels", 0,
			      NULL);
      } else {
	gnome_canvas_item_new(group,
			      gnome_canvas_rect_get_type(),
			      "x1", (double)(x)*xblock,
			      "y1", (double)(y)*yblock,
			      "x2", (double)(x+1)*xblock,
			      "y2", (double)(y+1)*yblock,
			      "fill_color_rgba",
			      GNOME_CANVAS_COLOR(myglot->red, 
						 myglot->green,
						 myglot->blue),
			      "outline_color_rgba",
			      GNOME_CANVAS_COLOR(0,0,0),
			      "width_pixels", 2,
			      NULL);
      }
    }
}

static void glotgen(GnomeCanvasGroup *group, glot *myglot) {
  int x, y;
  int neighbor[4]; /* is neighbor present in direction */
                   /* dirtype order: left, up, right, down */

  double xblock = myglot->parent->usize[X]*myglot->parent->zoom[X];
  double yblock = myglot->parent->usize[Y]*myglot->parent->zoom[Y];

  double xdelta = (xblock/6.0)*myglot->parent->scale[X]*
    myglot->parent->zoom[X];
  double ydelta = (yblock/6.0)*myglot->parent->scale[Y]*
    myglot->parent->zoom[Y];

  double xloc = (double)myglot->loc[X];
  double yloc = (double)myglot->loc[Y];

  GnomeCanvasPoints *side[4]; /* Side highights */
                              /* Each highlight has 4 points */
                              /* The first and fourth points are anchored;
				 depending on the presence of neighbors
				 clockwise (2) or counterclockwise (3),
				 the other two points may change (by delta)
			      */
  GnomeCanvasPoints *ptmp;    /* For generating diagonal in-bevels */

  group = GNOME_CANVAS_GROUP 
    (gnome_canvas_item_new(group,
			   gnome_canvas_group_get_type(),
			   "x", 0.0,
			   "y", 0.0,
			   NULL));


   for (x = 0; x < myglot->size[X]; x++)
    for (y = 0; y < myglot->size[Y]; y++) {
      neighbor[UP] = neighbor[DOWN] = neighbor[LEFT] = neighbor[RIGHT] = -1;
      side[UP] = side[DOWN] = side[LEFT] = side[RIGHT] = NULL;
      /* Building phase */
      /* Creation phase: [L]ower/[U]pper [L]eft/[R]ight */
      if (LOOKUP(myglot, x, y)) {
	if (x == 0 || !LOOKUP(myglot, x-1, y)) {
	  neighbor[LEFT] = 0;
	  side[LEFT] = gnome_canvas_points_new(4);
	  side[LEFT]->coords[0] = (x+xloc)*xblock;              /* UL */
	  side[LEFT]->coords[1] = (y+yloc)*yblock;

	  side[LEFT]->coords[2] = side[LEFT]->coords[0]+xdelta; /* UR */
	  side[LEFT]->coords[3] = side[LEFT]->coords[1];

	  side[LEFT]->coords[4] = side[LEFT]->coords[2];        /* LR */
	  side[LEFT]->coords[5] = side[LEFT]->coords[3]+yblock;

	  side[LEFT]->coords[6] = side[LEFT]->coords[0];        /* LL */
	  side[LEFT]->coords[7] = side[LEFT]->coords[5];
	}
	else neighbor[LEFT] = 1;
	
	if (y == 0 || !LOOKUP(myglot, x, y-1)) {
	  neighbor[UP] = 0;
	  side[UP] = gnome_canvas_points_new(4);
	  side[UP]->coords[0] = (x+xloc+1.0)*xblock;            /* UR */
	  side[UP]->coords[1] = (y+yloc)*yblock;

	  side[UP]->coords[2] = side[UP]->coords[0];            /* LR */
	  side[UP]->coords[3] = side[UP]->coords[1]+ydelta;

	  side[UP]->coords[4] = side[UP]->coords[2]-xblock;     /* LL */
	  side[UP]->coords[5] = side[UP]->coords[3];

	  side[UP]->coords[6] = side[UP]->coords[4];            /* UL */
	  side[UP]->coords[7] = side[UP]->coords[1];
	}
	else neighbor[UP] = 1;
	if (x == myglot->size[X]-1 || !LOOKUP(myglot, x+1, y)) {
	  neighbor[RIGHT] = 0;
	  side[RIGHT] = gnome_canvas_points_new(4);
	  side[RIGHT]->coords[0] = (x+xloc+1.0)*xblock;           /* LR */
	  side[RIGHT]->coords[1] = (y+yloc+1.0)*yblock;

	  side[RIGHT]->coords[2] = side[RIGHT]->coords[0]-xdelta; /* LL */
	  side[RIGHT]->coords[3] = side[RIGHT]->coords[1];

	  side[RIGHT]->coords[4] = side[RIGHT]->coords[2];        /* UL */
	  side[RIGHT]->coords[5] = side[RIGHT]->coords[3]-yblock;

	  side[RIGHT]->coords[6] = side[RIGHT]->coords[0];        /* UR */
	  side[RIGHT]->coords[7] = side[RIGHT]->coords[5];
	}
	else neighbor[RIGHT] = 1;
	if (y == myglot->size[Y]-1 || !LOOKUP(myglot, x, y+1)) { 
	  neighbor[DOWN] = 0;
	  side[DOWN] = gnome_canvas_points_new(4);
	  side[DOWN]->coords[0] = (x+xloc)*xblock;                /* LL */
	  side[DOWN]->coords[1] = (y+yloc+1.0)*yblock;

	  side[DOWN]->coords[2] = side[DOWN]->coords[0];          /* UL */
	  side[DOWN]->coords[3] = side[DOWN]->coords[1]-ydelta;

	  side[DOWN]->coords[4] = side[DOWN]->coords[2]+xblock;   /* UR */
	  side[DOWN]->coords[5] = side[DOWN]->coords[3];

	  side[DOWN]->coords[6] = side[DOWN]->coords[4];          /* LR */
	  side[DOWN]->coords[7] = side[DOWN]->coords[1];
	}
	else neighbor[DOWN] = 1;
    

	/* Adjustment phase */
	if (!neighbor[LEFT] && !neighbor[UP]) {
	  side[LEFT]->coords[3] += ydelta;
	  side[UP]->coords[4] += xdelta;
	}
	if (!neighbor[UP] && !neighbor[RIGHT]) {
	  side[UP]->coords[2] -= xdelta;
	  side[RIGHT]->coords[5] += ydelta;
	}
	if (!neighbor[RIGHT] && !neighbor[DOWN]) {
	  side[RIGHT]->coords[3] -= ydelta;
	  side[DOWN]->coords[4] -= xdelta;
	}
	if (!neighbor[DOWN] && !neighbor[LEFT]) {
	  side[DOWN]->coords[2] += xdelta;
	  side[LEFT]->coords[5] -= ydelta;
	}

	/* Drawing phase */
	/* Simple drawing phase */
#ifdef UGLY_RECTANGLE
	drawrect(group, (x+xloc)*xblock, (y+yloc)*yblock, (x+xloc+1.0)*xblock,
		 (y+yloc+1.0)*yblock, myglot->red, myglot->green, 
		 myglot->blue);
#else
	gnome_canvas_item_new(group,
			      gnome_canvas_rect_get_type(),
			      "x1", (x+xloc)*xblock,
			      "y1", (y+yloc)*yblock,
			      "x2", (x+xloc+1.0)*xblock,
			      "y2", (y+yloc+1.0)*yblock,
			      "fill_color_rgba",
			      GNOME_CANVAS_COLOR(myglot->red, 
						 myglot->green, 
						 myglot->blue),
			      "width_pixels", 0,
			      NULL);
#endif

	/* Outer bevel drawing phase */
	if (!neighbor[LEFT])
	  gnome_canvas_item_new(group,
				gnome_canvas_polygon_get_type(),
				"points", side[LEFT],
				"fill_color_rgba",
				GNOME_CANVAS_COLOR(LIGHTEN(myglot->red), 
						   LIGHTEN(myglot->green), 
						   LIGHTEN(myglot->blue)),
				"width_pixels", 0,
				NULL);
	if (!neighbor[UP])
	  gnome_canvas_item_new(group,
				gnome_canvas_polygon_get_type(),
				"points", side[UP],
				"fill_color_rgba",
				GNOME_CANVAS_COLOR(LIGHTEN(myglot->red), 
						   LIGHTEN(myglot->green), 
						   LIGHTEN(myglot->blue)),
				"width_pixels", 0,
				NULL);

	if (!neighbor[RIGHT])
	  gnome_canvas_item_new(group,
				gnome_canvas_polygon_get_type(),
				"points", side[RIGHT],
				"fill_color_rgba",
				GNOME_CANVAS_COLOR(DARKEN(myglot->red), 
						   DARKEN(myglot->green), 
						   DARKEN(myglot->blue)),
				"width_pixels", 0,
				NULL);
	if (!neighbor[DOWN])
	  gnome_canvas_item_new(group,
				gnome_canvas_polygon_get_type(),
				"points", side[DOWN],
				"fill_color_rgba",
				GNOME_CANVAS_COLOR(DARKEN(myglot->red), 
						   DARKEN(myglot->green), 
						   DARKEN(myglot->blue)),
				"width_pixels", 0,
				NULL);

	/* Center bevel drawing phase */
	
	if (neighbor[UP] && neighbor[LEFT] && !LOOKUP(myglot, x-1, y-1)) {
#ifdef UGLY_RECTANGLE
	  drawrect(group, (x+xloc)*xblock, (y+yloc)*yblock, 
		   (x+xloc)*xblock+xdelta, (y+yloc)*yblock+ydelta,
		   LIGHTEN(myglot->red), LIGHTEN(myglot->green), 
		   LIGHTEN(myglot->blue));
#else
	  gnome_canvas_item_new(group,
				gnome_canvas_rect_get_type(),
				"x1", (x+xloc)*xblock,
				"y1", (y+yloc)*yblock,
				"x2", (x+xloc)*xblock+xdelta,
				"y2", (y+yloc)*yblock+ydelta,
				"fill_color_rgba",
				GNOME_CANVAS_COLOR(LIGHTEN(myglot->red), 
						   LIGHTEN(myglot->green), 
						   LIGHTEN(myglot->blue)),
				"width_pixels", 0,
				NULL);

#endif
	}

	if (neighbor[UP] && neighbor[RIGHT] && !LOOKUP(myglot, x+1, y-1)) {
	  ptmp = gnome_canvas_points_new(3);
	  ptmp->coords[0] = (x+xloc+1.0)*xblock;
	  ptmp->coords[1] = (y+yloc)*yblock;

	  ptmp->coords[2] = ptmp->coords[0]-xdelta;
	  ptmp->coords[3] = ptmp->coords[1];

	  ptmp->coords[4] = ptmp->coords[2];
	  ptmp->coords[5] = ptmp->coords[1]+ydelta;

	  gnome_canvas_item_new(group,
			gnome_canvas_polygon_get_type(),
			"points", ptmp,
			"fill_color_rgba",
				GNOME_CANVAS_COLOR(DARKEN(myglot->red), 
						   DARKEN(myglot->green), 
						   DARKEN(myglot->blue)),
			"width_pixels", 0,
			NULL);

	  ptmp = gnome_canvas_points_new(3);
	  ptmp->coords[0] = (x+xloc+1.0)*xblock;
	  ptmp->coords[1] = (y+yloc)*yblock;

	  ptmp->coords[2] = ptmp->coords[0];
	  ptmp->coords[3] = ptmp->coords[1]+ydelta;

	  ptmp->coords[4] = ptmp->coords[2]-xdelta;
	  ptmp->coords[5] = ptmp->coords[3];

	  gnome_canvas_item_new(group,
			gnome_canvas_polygon_get_type(),
			"points", ptmp,
			"fill_color_rgba",
				GNOME_CANVAS_COLOR(LIGHTEN(myglot->red), 
						   LIGHTEN(myglot->green), 
						   LIGHTEN(myglot->blue)),
			"width_pixels", 0,
			NULL);
	}

	if (neighbor[DOWN] && neighbor[RIGHT] && !LOOKUP(myglot, x+1, y+1)) {
#ifdef UGLY_RECTANGLE
	  drawrect(group, (x+xloc+1.0)*xblock-xdelta, 
		   (y+yloc+1.0)*yblock-ydelta, (x+xloc+1.0)*xblock, 
		   (y+yloc+1.0)*yblock, DARKEN(myglot->red), 
		   DARKEN(myglot->green), DARKEN(myglot->blue));
#else
	  gnome_canvas_item_new(group,
				gnome_canvas_rect_get_type(),
				"x1", (x+xloc+1.0)*xblock-xdelta,
				"y1", (y+yloc+1.0)*yblock-ydelta,
				"x2", (x+xloc+1.0)*xblock,
				"y2", (y+yloc+1.0)*yblock,
				"fill_color_rgba",
				GNOME_CANVAS_COLOR(DARKEN(myglot->red), 
						   DARKEN(myglot->green), 
						   DARKEN(myglot->blue)),
				"width_pixels", 0,
				NULL);
#endif
	}

	if (neighbor[DOWN] && neighbor[LEFT] && !LOOKUP(myglot, x-1, y+1)) {
	  ptmp = gnome_canvas_points_new(3);
	  ptmp->coords[0] = (x+xloc)*xblock;
	  ptmp->coords[1] = (y+yloc+1.0)*yblock;

	  ptmp->coords[2] = ptmp->coords[0];
	  ptmp->coords[3] = ptmp->coords[1]-ydelta;

	  ptmp->coords[4] = ptmp->coords[2]+xdelta;
	  ptmp->coords[5] = ptmp->coords[3];

	  gnome_canvas_item_new(group,
			gnome_canvas_polygon_get_type(),
			"points", ptmp,
			"fill_color_rgba",
				GNOME_CANVAS_COLOR(DARKEN(myglot->red), 
						   DARKEN(myglot->green), 
						   DARKEN(myglot->blue)),
			"width_pixels", 0,
			NULL);

	  ptmp = gnome_canvas_points_new(3);
	  ptmp->coords[0] = (x+xloc)*xblock;
	  ptmp->coords[1] = (y+yloc+1.0)*yblock;

	  ptmp->coords[2] = ptmp->coords[0]+xdelta;
	  ptmp->coords[3] = ptmp->coords[1];

	  ptmp->coords[4] = ptmp->coords[2];
	  ptmp->coords[5] = ptmp->coords[1]-ydelta;

	  gnome_canvas_item_new(group,
			gnome_canvas_polygon_get_type(),
			"points", ptmp,
			"fill_color_rgba",
				GNOME_CANVAS_COLOR(LIGHTEN(myglot->red), 
						   LIGHTEN(myglot->green), 
						   LIGHTEN(myglot->blue)),
			"width_pixels", 0,
			NULL);	
	
	}
      }
    }
   myglot->element = (void *)GNOME_CANVAS_ITEM(group);
   gtk_signal_connect(GTK_OBJECT(GNOME_CANVAS_ITEM(group)), "event", 
		      (GtkSignalFunc)glot_event, myglot);
}

static void makebk(GnomeCanvasGroup *group, level *mylevel) {
  int x, y;
  double x1, y1, x2, y2;

  double xblock = mylevel->usize[X]*mylevel->zoom[X];
  double yblock = mylevel->usize[Y]*mylevel->zoom[Y];

  double xdelta = (xblock/12.0)*mylevel->scale[X]*mylevel->zoom[X];
  double ydelta = (yblock/12.0)*mylevel->scale[Y]*mylevel->zoom[Y];

  GnomeCanvasPoints *ulpoints, *lrpoints;

  group = GNOME_CANVAS_GROUP 
    (gnome_canvas_item_new(group,
			   gnome_canvas_group_get_type(),
			   "x", 0.0,
			   "y", 0.0,
			   NULL));

  for (y = 0; y < mylevel->size[Y]; y++)
    for (x = 0; x < mylevel->size[X]; x++) {
      x1 = (double)(x)*xblock; y1 = (double)(y)*yblock;
      x2 = (double)(x+1)*xblock; y2 = (double)(y+1)*yblock;

      ulpoints = gnome_canvas_points_new(5); /* Upper left */
      ulpoints->coords[0] = x1; ulpoints->coords[1] = y1;
      ulpoints->coords[2] = x1; ulpoints->coords[3] = y2;
      ulpoints->coords[4] = x1+xdelta;
      ulpoints->coords[5] = y2-ydelta;
      ulpoints->coords[6] = x2-xdelta;
      ulpoints->coords[7] = y1+ydelta;
      ulpoints->coords[8] = x2; ulpoints->coords[9] = y1;

      gnome_canvas_item_new(group, /* Upper left triangle */
			gnome_canvas_polygon_get_type(),
			"points", ulpoints,
			"fill_color_rgba", GNOME_CANVAS_COLOR(LIGHTEN(MYGRAY), 
							      LIGHTEN(MYGRAY), 
							      LIGHTEN(MYGRAY)),
			"width_pixels", 0,
			NULL);

      lrpoints = gnome_canvas_points_new(5); /* Lower right */
      lrpoints->coords[0] = x2; lrpoints->coords[1] = y2;
      lrpoints->coords[2] = x1; lrpoints->coords[3] = y2;
      lrpoints->coords[4] = ulpoints->coords[4];
      lrpoints->coords[5] = ulpoints->coords[5];
      lrpoints->coords[6] = ulpoints->coords[6];
      lrpoints->coords[7] = ulpoints->coords[7];
      lrpoints->coords[8] = x2; lrpoints->coords[9] = y1;
      gnome_canvas_item_new(group,
			    gnome_canvas_polygon_get_type(),
			    "points", lrpoints,
			    "fill_color_rgba", 
			    GNOME_CANVAS_COLOR(DARKEN(MYGRAY), DARKEN(MYGRAY),
					       DARKEN(MYGRAY)),
			    "width_pixels", 0,
			    NULL);
      
#ifdef UGLY_RECTANGLE
      drawrect(group, x1+xdelta, y1+ydelta, x2-xdelta, y2-ydelta, 
	       MYGRAY, MYGRAY, MYGRAY);
#else
      gnome_canvas_item_new(group,
			    gnome_canvas_rect_get_type(),
			    "x1", x1+xdelta, "y1", y1+ydelta,
			    "x2", x2-xdelta, "y2", y2-ydelta,
			    "fill_color_rgba", GNOME_CANVAS_COLOR(MYGRAY, 
								  MYGRAY,
								  MYGRAY),
			    "width_pixels", 0,
			    NULL);
#endif
    }
}

static void about_cb(GtkWidget* widget, GtkWidget *app) {
  static GtkWidget *dialog = NULL;  
  if (dialog != NULL) {
      g_assert(GTK_WIDGET_REALIZED(dialog));
      gdk_window_show(dialog->window);
      gdk_window_raise(dialog->window);
  } else {        
      const gchar *authors[] = {
	"Martin Hock <oxymoron@cmu.edu>",
	NULL
      };
      dialog = gnome_about_new (_("Glotski"), GLOTVER,
				"(C) 1999-2000 Martin Hock",
				authors,
				_("Drag the blocks around to reach a goal."),
				NULL);
      gtk_signal_connect(GTK_OBJECT(dialog),
			 "destroy",
			 GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			 &dialog);
      gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(app));
      gtk_widget_show(dialog);
  }
}

static void cancel_file(GtkWidget *widget, level **mylevel) {
  if (!(*mylevel)) gtk_main_quit();
}

static void open_cb(GtkWidget *widget, level **mylevel) {
  static pair mypair;
  GtkWidget *fselect = gtk_file_selection_new("Select level file");
  mypair.a = (void *)fselect; mypair.b = (void *)mylevel;
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fselect), LEVELPATH);
  gtk_file_selection_complete(GTK_FILE_SELECTION(fselect), "*.lev");
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fselect)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(select_file), &mypair);
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fselect)->cancel_button),
		     "clicked", GTK_SIGNAL_FUNC(cancel_file), mylevel);
                           
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fselect)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fselect);

  gtk_signal_connect_object(GTK_OBJECT
			    (GTK_FILE_SELECTION(fselect)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fselect);
  gtk_widget_show(fselect);
}
  
static gint exit_cb(GtkWidget *widget, gpointer data) {
  gtk_main_quit();
  return FALSE;
}

static void undo_cb(GtkWidget *widget, level **mylevel) {
  undo(*mylevel);
}

static void redo_cb(GtkWidget *widget, level **mylevel) {
  redo(*mylevel);
}

static void restart_cb(GtkWidget *widget, level **mylevel) {
  movelist *mymove;
  if ((*mylevel)->move) {
    do {
      mymove = (*mylevel)->move;
      undo(*mylevel);
    } while (mymove != (*mylevel)->move);
    if ((*mylevel)->move->next) {
      deletemove((*mylevel)->move->next);
      (*mylevel)->move->next = NULL;
    }
  }
}

void toggle_off(GtkWidget *toggle, level *mylevel, GtkWidget *window) {
  gtk_widget_hide(window);
  if (GTK_CHECK_MENU_ITEM(toggle)->active) 
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(toggle), FALSE);
  mylevel->showgoal = FALSE;
}

void toggle_on(GtkWidget *toggle, level *mylevel, GtkWidget *window) {
  gtk_widget_show(window);
  if (!GTK_CHECK_MENU_ITEM(toggle)->active)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(toggle), TRUE);
  mylevel->showgoal = TRUE;
}

void toggle_cb(GtkWidget *widget, level **mylevel) {
  if (GTK_CHECK_MENU_ITEM(widget)->active)
    toggle_on(widget, *mylevel, ((GtkWidget *)((*mylevel)->gelement))->parent);
  else toggle_off(widget, *mylevel, 
		  ((GtkWidget *)((*mylevel)->gelement))->parent);

}

static gint appdelete_event(GtkWidget *window, GdkEventAny *e, gpointer data) {
  exit_cb(window, data);
  return FALSE;
}	

static gint gwindelete_event(GtkWidget *window, GdkEventAny *e, 
			     level **mylevel) {
  toggle_off((GtkWidget *)((*mylevel)->telement), *mylevel, window);
  return TRUE;
}

static void drawlevel(level *mylevel, GtkWidget *window) {
  GtkWidget *canvas;
  GnomeCanvasGroup *group;
  setuplist *mysetup;

  if (mylevel->lelement) gtk_widget_destroy((GtkWidget *)(mylevel->lelement));
  gtk_widget_push_visual(gdk_rgb_get_visual());
  gtk_widget_push_colormap(gdk_rgb_get_cmap());
  canvas = gnome_canvas_new();
  gtk_widget_pop_colormap();
  gtk_widget_pop_visual();
  gnome_canvas_set_scroll_region(GNOME_CANVAS(canvas), 0, 0, 
				 mylevel->usize[X]*mylevel->size[X]*
				 mylevel->zoom[X], 
				 mylevel->usize[Y]*mylevel->size[Y]*
				 mylevel->zoom[Y]);
  group = gnome_canvas_root(GNOME_CANVAS(canvas));
  makebk(group, mylevel);
  mysetup = mylevel->setup;
  while (mysetup) {
    glotgen(group, mysetup->this);
    mysetup = mysetup->next;
  }
  gtk_widget_set_usize(canvas,
		       mylevel->usize[X]*mylevel->size[X]*mylevel->zoom[X],
		       mylevel->usize[Y]*mylevel->size[Y]*mylevel->zoom[Y]);
  gnome_app_set_contents(GNOME_APP(window), canvas);
  gtk_widget_show(canvas);
  updatestatus(mylevel);
  mylevel->lelement = (void *)canvas;
}

static GtkWidget *setuplevel(level **mylevel) {
  GtkWidget *app = gnome_app_new(GLOTNAME, GLOTNAME);
  GtkWidget *appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
  GnomeUIInfo gamemenu[] = {
    GNOMEUIINFO_MENU_RESTART_GAME_ITEM(restart_cb, mylevel),
    GNOMEUIINFO_MENU_UNDO_MOVE_ITEM(undo_cb, mylevel),
    GNOMEUIINFO_MENU_REDO_MOVE_ITEM(redo_cb, mylevel),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_TOGGLEITEM_NONE_DATA_(N_("_Toggle goal window"), 
					 toggle_cb, mylevel),
    GNOMEUIINFO_END
  };

  GnomeUIInfo filemenu[] = {
    GNOMEUIINFO_MENU_OPEN_ITEM(open_cb, mylevel),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_EXIT_ITEM(exit_cb, NULL),
    GNOMEUIINFO_END
  };

  GnomeUIInfo helpmenu[] = {
    GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, app),
    GNOMEUIINFO_END
  };

  GnomeUIInfo rootmenu[] = {
    GNOMEUIINFO_MENU_GAME_TREE(gamemenu),
    GNOMEUIINFO_MENU_FILE_TREE(filemenu),
    GNOMEUIINFO_MENU_HELP_TREE(helpmenu),
    GNOMEUIINFO_END
  };

  GnomeUIInfo toolbar[] = {
    GNOMEUIINFO_ITEM_STOCK_DATA_(N_("Restart"), N_("Reset to initial state"),
				 restart_cb, mylevel, 
				 GNOME_STOCK_PIXMAP_REFRESH),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_ITEM_STOCK_DATA_(N_("Undo"), N_("Previous move"), undo_cb, 
				 mylevel, GNOME_STOCK_PIXMAP_UNDO),
    GNOMEUIINFO_ITEM_STOCK_DATA_(N_("Redo"), N_("Next move"), redo_cb,
				 mylevel, GNOME_STOCK_PIXMAP_REDO),
    GNOMEUIINFO_END
  };
  GdkPixmap *icon;
  GdkBitmap *mask;

  gtk_window_set_policy(GTK_WINDOW(app), FALSE, FALSE, TRUE);
  gtk_signal_connect(GTK_OBJECT(app), "delete_event", 
		     GTK_SIGNAL_FUNC(appdelete_event), NULL);
  gnome_app_set_statusbar(GNOME_APP(app), appbar);
  gnome_app_create_menus(GNOME_APP(app), rootmenu);
  if ((*mylevel)->showgoal)
    GTK_CHECK_MENU_ITEM(gamemenu[4].widget)->active = TRUE;
  (*mylevel)->telement = (void *)(gamemenu[4].widget);
  (*mylevel)->selement = (void *)appbar;
  gnome_app_create_toolbar(GNOME_APP(app), toolbar);
  drawlevel(*mylevel, app);
  gtk_widget_show_all(app);
  icon = gdk_pixmap_create_from_xpm_d(app->window, &mask, NULL, icon_xpm);
  gdk_window_set_icon(app->window, NULL, icon, mask);
  return app;
}

static void drawgoal(level *mylevel, GtkWidget *window) {
  GtkWidget *canvas;
  GnomeCanvasGroup *group;
  if (mylevel->gelement) gtk_widget_destroy((GtkWidget *)(mylevel->gelement));
  gtk_widget_push_visual(gdk_rgb_get_visual());
  gtk_widget_push_colormap(gdk_rgb_get_cmap());
  canvas = gnome_canvas_new();
  gtk_widget_pop_colormap();
  gtk_widget_pop_visual();
  gnome_canvas_set_scroll_region(GNOME_CANVAS(canvas), 0, 0, 
				 mylevel->usize[X]*mylevel->size[X]*
				 mylevel->gzoom[X], 
				 mylevel->usize[Y]*mylevel->size[Y]*
				 mylevel->gzoom[Y]);
  group = gnome_canvas_root(GNOME_CANVAS(canvas));
  gtk_widget_set_usize(canvas,
		       mylevel->usize[X]*mylevel->size[X]*mylevel->gzoom[X],
		       mylevel->usize[Y]*mylevel->size[Y]*mylevel->gzoom[Y]);
  goalgen(group, mylevel);
  gtk_container_add(GTK_CONTAINER(window), canvas);
  gtk_widget_show(canvas);
  mylevel->gelement = (void *)canvas;
}

void redrawgoal(level *mylevel) { /* Assumes you've drawn before */
  if (mylevel->gelement && mylevel->mygoal) 
    drawgoal(mylevel, ((GtkWidget *)(mylevel->gelement))->parent);
}

static GtkWidget *setupgoal(level **mylevel) {
  GdkPixmap *icon;
  GdkBitmap *mask;
  GtkWidget *goalwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(goalwin), "Goal");
  gtk_window_set_policy(GTK_WINDOW(goalwin), FALSE, FALSE, TRUE);
  gtk_signal_connect(GTK_OBJECT(goalwin), "delete_event",
		     GTK_SIGNAL_FUNC(gwindelete_event), mylevel);
  drawgoal(*mylevel, goalwin);
  if ((*mylevel)->showgoal) gtk_widget_show_all(goalwin);
  icon = gdk_pixmap_create_from_xpm_d(goalwin->window, &mask, NULL, icon_xpm);
  gdk_window_set_icon(goalwin->window, NULL, icon, mask);
  return goalwin;
}

static void select_file(GtkWidget *widget, pair *mypair) {
  GtkWidget *filesel = (GtkWidget *)mypair->a;
  level **mylevel = (level **)mypair->b;
  char *text = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  level *newlevel = loadlev(text, FALSE);
  if (!newlevel) {
    textdisp("Invalid filename");
    open_cb(NULL, mylevel);
    return;
  }
  if (*mylevel) {
    newlevel->showgoal = (*mylevel)->showgoal;
    newlevel->lelement = (*mylevel)->lelement;
    newlevel->gelement = (*mylevel)->gelement;
    newlevel->telement = (*mylevel)->telement;
    newlevel->selement = (*mylevel)->selement;
    drawlevel(newlevel, 
	      ((GtkWidget *)((*mylevel)->lelement))->parent->parent->parent);
    drawgoal(newlevel, ((GtkWidget *)((*mylevel)->gelement))->parent);
    deletelevel(*mylevel);
    *mylevel = newlevel;
  } else {
    *mylevel = newlevel;
    setuplevel(mylevel);
    setupgoal(mylevel);
  }
}

void updatestatus(level *mylevel) {
  gchar stattxt[150];
  gchar tmptxt[50];
  if (mylevel->nummoves == 1) sprintf(stattxt, "1 move");
  else sprintf(stattxt, "%d moves", mylevel->nummoves);
  if (mylevel->step) {
    sprintf(tmptxt, " (%d minimum)", mylevel->step);
    strcat(stattxt, tmptxt);
  }
  if (mylevel->numgoals-mylevel->state == 1) 
    sprintf(tmptxt, ", 1 goal remaining");
  else 
    sprintf(tmptxt, ", %d goals remaining", mylevel->numgoals-mylevel->state);
  strcat(stattxt, tmptxt);
  gnome_appbar_set_status(GNOME_APPBAR((GtkWidget *)(mylevel->selement)), 
			  stattxt); 
}

int main(int argc, char *argv[]) {
  static level *mylevel = NULL;

  static int filemode = FALSE;
  poptContext pctx;
  char **args;

  struct poptOption options[] = {
    {
      "file",
      'f',
      POPT_ARG_NONE,
      &filemode,
      0,
      N_("Load a file listed on the command line"),
      NULL
    },
    {
      NULL,
      '\0',
      0,
      NULL,
      0,
      NULL,
      NULL
    }
  };

  gnome_init_with_popt_table(GLOTNAME, GLOTVER, argc, argv, options, 0, &pctx);
  gdk_rgb_init();

  args = poptGetArgs(pctx);
 
  if (filemode) {
    if (args) {
      if (!(mylevel = loadlev(args[0], TRUE))) {
	textdisp("Invalid filename; try these.");
	open_cb(NULL, &mylevel);
      } else {
	setuplevel(&mylevel);
	setupgoal(&mylevel);
      }
    }
  }
  else {
    if (args) {
      printf("Invalid argument %s\n", args[0]);
      return 1;
    }
    open_cb(NULL, &mylevel);
  }
  poptFreeContext(pctx);
  gtk_main();
  return 0;
}			 







