/**************************************************************************\
 gfxdump (Program for dumping graphics chip registers)

  Copyright (C) 1999 yvind Aabling.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#include "gfx.h"
#include "pci.c"

#ifdef __unix__
#define MODE	"a"
#else
#define MODE	((binopt)?"ab":"a")
#endif

#define MAXREGLEN	32
#define MAXACCLEN	7
#define MAXPAGELEN	7
#define MAXGROUPLEN	5

#define MAXGROUPS	32
#define MAXREGS		2048

/* ------------------------------------------------------------------------ */
/* --- Structs --- -------------------------------------------------------- */
/* ------------------------------------------------------------------------ */

/* Register Entry */
typedef struct {
  char	name[MAXREGLEN+1] ;	/* Register Name */
  u32	offset ;	/* For Access Method */
  int	acc ;		/* Access Method Index */
  int	grp ;		/* Group Index */
  char	page[MAXPAGELEN+1] ;	/* Page # in Chip Docs */
  int	size ;		/* Register Bytesize */
  int	incl ;		/* Include Register in output */
  int	valid ;		/* Register Value Valid */
  u32	value ;		/* Register Value */
  u32	mask ;		/* Register Value Mask */
} REG ;

/* Register Access Method Entry */
typedef struct {
  char	*name ;		/* Register Access Method Name */
  char	*prtname ;	/* Register Access Method Name (Output) */
  u32	(*acc)(REG*) ;	/* Register Access Function */
  int	(*prt)(REG*) ;	/* Register Address Printing Function */
} accessmethod ;

/* Register Group Entry */
typedef struct {
  char	name1[MAXGROUPLEN+1] ;	/* Register Group Name */
  char	name2[MAXGROUPLEN+1] ;	/* Register Group Name */
  char	opti1[MAXGROUPLEN+2] ;	/* Register Group -Option String */
  char	optx1[MAXGROUPLEN+2] ;	/* Register Group +Option String */
  char	opti2[MAXGROUPLEN+2] ;	/* Register Group -Option String */
  char	optx2[MAXGROUPLEN+2] ;	/* Register Group +Option String */
  int	incl ;			/* Include Register Group in output */
} GRP ;

/* ------------------------------------------------------------------------ */
/* --- Function Predeclarations --- --------------------------------------- */
/* ------------------------------------------------------------------------ */

static void dommap(void) ;
static int prt_acc_idx(REG*) ;
static int prt_acc_4(REG*) ;
static int prt_acc_2(REG*) ;
static int prt_m64_mem(REG*) ;

/* ------------------------------------------------------------------------ */
/* --- Global variables --- ----------------------------------------------- */
/* ------------------------------------------------------------------------ */

/* Chip ID and Memory Mapping */
static int mem=-1, verbose=0, binopt=0, dangeropt=0, repopt=1 ;
static int allopt=0, everyopt=0, groupopt=0, nameopt=0, pageopt=0, xopt=0 ;
static u16 vendor=0, device=0 ;
static u32 base0=0, base1=0, base2=0, baserom=0 ;
static u32 size0=0, size1=0, size2=0 ;
static volatile u8 *ptr0=NULL, *ptr1=NULL, *ptr2=NULL ;

/* Registers and Groups */
static REG *reg ; int nregs=0 ;
static GRP *grp ; int ngrps=0 ;

/* ------------------------------------------------------------------------ */
/* --- Register Access Methods --- ---------------------------------------- */
/* ------------------------------------------------------------------------ */

/* --- I/O Mapped Registers --- */

static u32 acc_io(REG *r) { switch (r->size) {
  case 1: return inb(r->offset) ;
  case 2: return inw(r->offset) ;
  case 4: return inl(r->offset) ; }
  return 0xDEADBEEF ; }

static u32 acc_io1(REG *r) { switch (r->size) {
  case 1: return inb(base1+r->offset) ;
  case 2: return inw(base1+r->offset) ;
  case 4: return inl(base1+r->offset) ; }
  return 0xDEADBEEF ; }

/* --- base[012] Memory Mapped Registers --- */

static u32 acc_mem0(REG *r) { switch (r->size) {
  case 1: return GETREG( u8,ptr0,r->offset) ;
  case 2: return GETREG(u16,ptr0,r->offset) ;
  case 4: return GETREG(u32,ptr0,r->offset) ; }
  return 0xDEADBEEF ; }

static u32 acc_mem1(REG *r) { switch (r->size) {
  case 1: return GETREG( u8,ptr1,r->offset) ;
  case 2: return GETREG(u16,ptr1,r->offset) ;
  case 4: return GETREG(u32,ptr1,r->offset) ; }
  return 0xDEADBEEF ; }

static u32 acc_mem2(REG *r) { switch (r->size) {
  case 1: return GETREG( u8,ptr2,r->offset) ;
  case 2: return GETREG(u16,ptr2,r->offset) ;
  case 4: return GETREG(u32,ptr2,r->offset) ; }
  return 0xDEADBEEF ; }

/* --- Indexed VGA Registers --- */

static u32 acc_vga_atc(REG *r) {
  u8 idx, val=0xFF ; inb(0x3DA) ; idx = inb(0x3C0) ; outb(0x3C0,r->offset) ;
  switch (r->size) { case 1: val = inb(0x3C1) ; break ; }
  inb(0x3DA) ; outb(0x3C0,idx) ; return val ; }

static u32 acc_vga_crt(REG *r) {
  u8 idx, val=0xFF ; idx = inb(0x3D4) ; outb(0x3D4,r->offset) ;
  switch (r->size) { case 1: val = inb(0x3D5) ; break ; }
  outb(0x3D4,idx) ; return val ; }

static u32 acc_vga_gra(REG *r) {
  u8 idx, val=0xFF ; idx = inb(0x3CE) ; outb(0x3CE,r->offset) ;
  switch (r->size) { case 1: val = inb(0x3CF) ; break ; }
  outb(0x3CE,idx) ; return val ; }

static u32 acc_vga_seq(REG *r) {
  u8 idx, val=0xFF ; idx = inb(0x3C4) ; outb(0x3C4,r->offset) ;
  switch (r->size) { case 1: val = inb(0x3C5) ; break ; }
  outb(0x3C4,idx) ; return val ; }

static u32 acc_vga_dac(REG *r) {
  u8 idx, tmp ; u32 val ;
  idx = inb(0x3C7) ; outb(0x3C7,r->offset) ;
  tmp = inb(0x3C9) ; val = tmp<<16 ;
  tmp = inb(0x3C9) ; val += tmp<<8 ;
  tmp = inb(0x3C9) ; val += tmp ;
  outb(0x3C7,idx) ; return val ; }

/* --- Vendor Specific --- ------------------------------------------------ */

/* --- Vendor 0x1002: ATI Technologies --- */

static u32 acc_r128_pll(REG *r) {
  /* MMR_0008, byte 0 and MMR_000C */
  #define R128_CLOCK_CNTL_INDEX0	GETREG( u8,ptr2,0x0008)
  #define R128_CLOCK_CNTL_INDEX0_SET	SETREG( u8,ptr2,0x0008)
  #define R128_CLOCK_CNTL_DATA		GETREG(u32,ptr2,0x000C)
  u8 idx=R128_CLOCK_CNTL_INDEX0 ; u32 val ;
  R128_CLOCK_CNTL_INDEX0_SET = r->offset ;
  val = R128_CLOCK_CNTL_DATA ; R128_CLOCK_CNTL_INDEX0_SET = idx ; return val ; }

static u32 acc_m64_pll(REG *r) {
  /* MEM_0_24, byte 1 and 2 */
  #define M64_CLOCK_CNTL1	GETREG(u8,ptr2,0x491)
  #define M64_CLOCK_CNTL1_SET	SETREG(u8,ptr2,0x491)
  #define M64_CLOCK_CNTL2	GETREG(u8,ptr2,0x492)
  u8 idx=M64_CLOCK_CNTL1, val ; M64_CLOCK_CNTL1_SET = r->offset << 2 ;
  val = M64_CLOCK_CNTL2 ; M64_CLOCK_CNTL1_SET = idx ; return val ; }

static u32 acc_m64_mem(REG *r) { switch (r->size) {
  case 1: return GETREG( u8,ptr2,((r->offset^0x100)<<2)) ;
  case 2: return GETREG(u16,ptr2,((r->offset^0x100)<<2)) ;
  case 4: return GETREG(u32,ptr2,((r->offset^0x100)<<2)) ; }
  return 0xDEADBEEF ; }

/* --- Vendor 0x100C: Tseng Labs --- */

static u32 acc_et6k_pll(REG *r) {
  u8 idx ; u16 val=0xFFFF ;
  idx = inb(base1+0x67) ; outb(base1+0x67,r->offset) ;
  switch (r->size) {
    case 1: val = inb(base1+0x69) ; break ;
    case 2: val = inb(base1+0x69) ;
            val += inb(base1+0x69)<<8 ; break ; }
  outb(base1+0x67,idx) ; return val ; }

/* ------------------------------------------------------------------------ */
/* --- Access Method List --- --------------------------------------------- */
/* ------------------------------------------------------------------------ */

static accessmethod acc[] = {
  /* --- I/O Mapped Registers --- */
  { "I/O",	"IO",	&acc_io,	&prt_acc_4 },
  { "I/O1",	"IO",	&acc_io1,	&prt_acc_2 },
  /* --- base[012] Memory Mapped Registers --- */
  { "MEM0",	"MEM",	&acc_mem0,	&prt_acc_4 },
  { "MEM1",	"MEM",	&acc_mem1,	&prt_acc_4 },
  { "MEM2",	"MEM",	&acc_mem2,	&prt_acc_4 },
  /* --- Indexed VGA Registers --- */
  { "ATC",	"ATC",	&acc_vga_atc,	&prt_acc_idx },
  { "CRT",	"CRT",	&acc_vga_crt,	&prt_acc_idx },
  { "GRA",	"GRA",	&acc_vga_gra,	&prt_acc_idx },
  { "SEQ",	"SEQ",	&acc_vga_seq,	&prt_acc_idx },
  { "DAC",	"DAC",	&acc_vga_dac,	&prt_acc_idx },
  /* --- Vendor 0x1002: ATI Technologies --- */
  { "R128MMR",	"MMR",	&acc_mem2,	&prt_acc_4 },
  { "R128PLL",	"PLL",	&acc_r128_pll,	&prt_acc_idx },
  { "M64MEM",	"MEM",	&acc_m64_mem,	&prt_m64_mem },
  { "M64PLL",	"PLL",	&acc_m64_pll,	&prt_acc_idx },
  /* --- Vendor 0x100C: Tseng Labs --- */
  { "ET6KPLL",	"PLL",	&acc_et6k_pll,	&prt_acc_idx },
  /* --- End Marker --- */
  { NULL, NULL, NULL, NULL } } ;

/* --- Shared Print Routines --- */

static int prt_acc_idx(REG *r) {
  return printf("%s[%02X]",acc[r->acc].prtname,r->offset) ; }

static int prt_acc_4(REG *r) {
  return printf("%s_%04X",acc[r->acc].prtname,r->offset) ; }

static int prt_acc_2(REG *r) {
  return printf("%s_%02X",acc[r->acc].prtname,r->offset) ; }

/* --- ATI Technologies Inc. --- */

static int prt_m64_mem(REG *r) {
  return printf("MEM_%d_%02X",(int)r->offset>>8,(int)r->offset&0xFF) ; }

/* ------------------------------------------------------------------------ */
/* --- Vendor Specific Chip Type Detection --- ---------------------------- */
/* ------------------------------------------------------------------------ */
/* Set size[012]>0 to request memory mapping of
 * region, and modify base[012] if necessary.
 * Return name of <chip>.reg file listing Chip Registers,
 * or NULL if Chip ID not recognized. */

/* Vendor 0x1002: ATI Technologies */			/* ATI Technologies */
static char *ati(void) {
  switch (device) {
    /* Mach 64 */
    case 0x4358: case 0x4354: case 0x4554: case 0x4C54:	/* CX, CT, ET, LT */
    case 0x5654: case 0x5655: case 0x4754: case 0x4755:	/* VT/GT (Rage 1) */
    case 0x4756: case 0x4757: case 0x4758: case 0x4759:	/* Rage II */
    case 0x475A: case 0x5656: case 0x5657:		/* Rage II */
    case 0x4C47:					/* Rage LT */
    case 0x4C44: case 0x4C49: case 0x4C50:		/* Rage LT Pro */
    case 0x4C51: case 0x4C42:				/* Rage LT Pro */
    case 0x4C4D: case 0x4C52:				/* Rage Mobility */
    case 0x474D: case 0x474E: case 0x474F:		/* Rage XL/XC */
    case 0x4752: case 0x4753:				/* Rage XL/XC */
      size2 = 4096 ; dommap() ; return "Mach64.reg" ;
    /* Rage PRO */
    case 0x4742: case 0x4744: case 0x4749:		/* Rage PRO */
    case 0x4750: case 0x4751:				/* Rage PRO */
      size2 = 4096 ; dommap() ; return "RagePRO.reg" ;
    /* Rage 128 */
    case 0x5245: case 0x5246: case 0x524B: case 0x524C:	/* Rage 128 */
      size2 = 8192 ; dommap() ; return "Rage128.reg" ;
    default: return NULL ; } }

/* Vendor 0x100C: Tseng Labs */				/* Tseng Labs */
static char *tseng(void) {
  switch (device) {
    case 0x3202: case 0x3205: case 0x3206: case 0x3207:	/* ET4000w32p [ABDC] */
      dommap() ; return "ET4000w32p.reg" ;
    case 0x3208:					/* ET6000/6100 */
      base0 += 0x3FFF00L ; size0 = 4096 ; dommap() ; return "ET6000.reg" ;
    case 0x4702:					/* ET6300 */
      dommap() ; return "ET6300.reg" ;
    default: return NULL ; } }

/* Vendor 0x100E: Weitek */				/* Weitek */
static char *weitek(void) {
  switch (device) {
    case 0x9001:					/* P9000 */
      dommap() ; return "P9000.reg" ;
    case 0x9100:					/* P9100 */
      dommap() ; return "P9100.reg" ;
    default: return NULL ; } }

/* Vendor 0x1013: Cirrus Logic */			/* Cirrus Logic */
static char *cirrus(void) {
  switch (device) {
    case 0x0038:					/* GD 7548 */
      dommap() ; return "GD7548.reg" ;
    case 0x00A0:					/* GD 5430 */
      dommap() ; return "GD5430.reg" ;
    case 0x00A4:					/* GD 5434-4 */
      dommap() ; return "GD5434-4.reg" ;
    case 0x00A8:					/* GD 5434-8 */
      dommap() ; return "GD5434-8.reg" ;
    case 0x00AC:					/* GD 5436 */
      dommap() ; return "GD5436.reg" ;
    case 0x00B8:					/* GD 5446 */
      dommap() ; return "GD5446.reg" ;
    case 0x00BC:					/* GD 5480 */
      dommap() ; return "GD5480.reg" ;
    case 0x00D0:					/* GD 5462 */
      dommap() ; return "GD5462.reg" ;
    case 0x00D4:					/* GD 5464 */
      dommap() ; return "GD5464.reg" ;
    case 0x00D6:					/* GD 5465 */
      dommap() ; return "GD5465.reg" ;
    case 0x1100:					/* CL 6729 */
      dommap() ; return "CL6729.reg" ;
    case 0x1110:					/* CL 6832 */
      dommap() ; return "CL6832.reg" ;
    case 0x1200:					/* GD 7542 */
      dommap() ; return "GD7542.reg" ;
    case 0x1202:					/* GD 7543 */
      dommap() ; return "GD7543.reg" ;
    case 0x1204:					/* GD 7541 */
      dommap() ; return "GD7541.reg" ;
    default: return NULL ; } }

/* Vendor 0x1023: Trident */				/* Trident */
static char *trident(void) {
  switch (device) {
    case 0x9320:					/* TGUI 9320 */
      dommap() ; return "TGUI9320.reg" ;
    case 0x9420:					/* TGUI 9420 */
      dommap() ; return "TGUI9420.reg" ;
    case 0x9440:					/* TGUI 9440 */
      dommap() ; return "TGUI9440.reg" ;
    case 0x9660:					/* TGUI 9660/9680/9682 */
      dommap() ; return "TGUI9660.reg" ;
    case 0x9388:					/* TGUI 9388 */
      dommap() ; return "TGUI9388.reg" ;
    case 0x9397:					/* TGUI 9397 */
      dommap() ; return "TGUI9397.reg" ;
    case 0x9750:					/* TGUI 9750 */
      dommap() ; return "TGUI9750.reg" ;
    case 0x9850:					/* TGUI 9850 */
      dommap() ; return "TGUI9850.reg" ;
    default: return NULL ; } }

/* Vendor 0x102B: Matrox */				/* Matrox */
static char *matrox(void) {
  switch (device) {
    case 0x0518:					/* MGA-2 Atlas PX2085 */
      dommap() ; return "Atlas.reg" ;
    case 0x0519:					/* MGA Millennium */
      dommap() ; return "Mill.reg" ;
    case 0x051A:					/* MGA Mystique */
      dommap() ; return "Mystique.reg" ;
    case 0x051B:					/* MGA Millennium II */
    case 0x051F:					/* MGA Millennium II AGP */
      dommap() ; return "MillII.reg" ;
    case 0x0520:					/* MGA Millennium G200 PCI */
    case 0x0521:					/* MGA Millennium G200 AGP */
      dommap() ; return "MillG200.reg" ;
    case 0x0D10:					/* MGA Impression */
      dommap() ; return "Impression.reg" ;
    case 0x1000:					/* MGA Productiva G100 PCI */
    case 0x1001:					/* MGA Productiva G100 AGP */
      dommap() ; return "ProdG100.reg" ;
    default: return NULL ; } }

/* Vendor 0x1039: SIS */				/* SIS */
static char *sis(void) {
  switch (device) {
    case 0x0001:					/* 86C201 */
      dommap() ; return "SIS86C201.reg" ;
    case 0x0002:					/* 86C202 */
      dommap() ; return "SIS86C202.reg" ;
    case 0x0008:					/* 85C503 */
      dommap() ; return "SIS85C503.reg" ;
    case 0x0205:					/* 86C205 */
      dommap() ; return "SIS86C205.reg" ;
    case 0x0406:					/* 85C501 */
      dommap() ; return "SIS85C501.reg" ;
    case 0x0496:					/* 85C496 */
      dommap() ; return "SIS85C496.reg" ;
    case 0x0601:					/* 85C601 */
      dommap() ; return "SIS85C601.reg" ;
    case 0x5107:					/* 5107 */
      dommap() ; return "SIS5107.reg" ;
    case 0x5511:					/* 85C5511 */
      dommap() ; return "SIS85C5511.reg" ;
    case 0x5513:					/* 85C5513 */
      dommap() ; return "SIS85C5513.reg" ;
    case 0x5571:					/* 5571 */
      dommap() ; return "SIS5571.reg" ;
    case 0x5597:					/* 5597 */
      dommap() ; return "SIS5597.reg" ;
    case 0x7001:					/* 7001 */
      dommap() ; return "SIS7001.reg" ;
    default: return NULL ; } }

/* Vendor 0x105D: Number Nine */			/* Number Nine */
static char *number9(void) {
  switch (device) {
    case 0x2309:					/* Imagine-128 */
      dommap() ; return "Imagine128.reg" ;
    case 0x2339:					/* Imagine-128-II */
      dommap() ; return "Imagine128II.reg" ;
    case 0x493D:					/* Imagine-128-T2R */
      dommap() ; return "Imagine128T2R.reg" ;
    default: return NULL ; } }

/* Vendor 0x10C8: Neomagic */				/* Neomagic */
static char *neomagic(void) {
  switch (device) {
    case 0x0001:					/* Magicgraph NM2070 */
      dommap() ; return "MagicNM2070.reg" ;
    case 0x0002:					/* Magicgraph 128V */
      dommap() ; return "Magic128V.reg" ;
    case 0x0003:					/* Magicgraph 128ZV */
      dommap() ; return "Magic128ZV.reg" ;
    case 0x0004:					/* Magicgraph NM2160 */
      dommap() ; return "MagicNM2160.reg" ;
    default: return NULL ; } }

/* Vendor 0x1163: Rendition */				/* Rendition */
static char *rendition(void) {
  switch (device) {
    case 0x0001:					/* V1000 */
      dommap() ; return "V1000.reg" ;
    case 0x2000:					/* V2100 */
      dommap() ; return "V2100.reg" ;
    default: return NULL ; } }

/* 0x3D3D: 3Dlabs */					/* 3Dlabs */
static char *labs3d(void) {
  switch (device) {
    case 0x0001:					/* GLINT 300SX */
      dommap() ; return "GLINT300SX.reg" ;
    case 0x0002:					/* GLINT 500TX */
      dommap() ; return "GLINT500TX.reg" ;
    case 0x0003:					/* GLINT Delta */
      dommap() ; return "GLINTDelta.reg" ;
    case 0x0004:					/* GLINT Permedia */
      dommap() ; return "GLINTPermedia.reg" ;
    case 0x0006:					/* GLINT MX */
      dommap() ; return "GLINTMX.reg" ;
    case 0x0007:					/* GLINT Permedia 2 */
      dommap() ; return "GLINTPermedia 2.reg" ;
    default: return NULL ; } }

/* 0x5333: S3 */					/* S3 */
static char *s3(void) {
  switch (device) {
    case 0x0551:					/* Plato/PX */
      dommap() ; return "Plato.reg" ;
    case 0x5631:					/* ViRGE */
      dommap() ; return "ViRGE.reg" ;
    case 0x8811:					/* Trio32/64 */
      dommap() ; return "Trio32+64.reg" ;
    case 0x8812:					/* Aurora64V+ */
      dommap() ; return "Aurora64V.reg" ;
    case 0x8814:					/* Trio64UV+ */
      dommap() ; return "Trio64UV.reg" ;
    case 0x883D:					/* ViRGE/VX */
      dommap() ; return "ViRGEVX.reg" ;
    case 0x8880:					/* 868 */
      dommap() ; return "868.reg" ;
    case 0x88B0:					/* 928 */
      dommap() ; return "928.reg" ;
    case 0x88C0:					/* 864-0 */
      dommap() ; return "864-0.reg" ;
    case 0x88C1:					/* 864-1 */
      dommap() ; return "864-1.reg" ;
    case 0x88D0:					/* 964-0 */
      dommap() ; return "964-0.reg" ;
    case 0x88D1:					/* 964-1 */
      dommap() ; return "964-1.reg" ;
    case 0x88F0:					/* 968 */
      dommap() ; return "968.reg" ;
    case 0x8901:					/* Trio64V2/DX or /GX */
      dommap() ; return "Trio64V2.reg" ;
    case 0x8902:					/* PLATO/PX */
      dommap() ; return "PLATOPX.reg" ;
    case 0x8A01:					/* ViRGE/DX or /GX */
      dommap() ; return "ViRGE.reg" ;
    case 0x8A10:					/* ViRGE/GX2 */
      dommap() ; return "ViRGEGX2.reg" ;
    case 0x8C01:					/* ViRGE/MX */
    case 0x8C02:					/* ViRGE/MX? */
      dommap() ; return "ViRGEMX.reg" ;
    case 0x8C03:					/* ViRGE/MX+ */
      dommap() ; return "ViRGEMX+.reg" ;
    default: return NULL ; } }

/* ------------------------------------------------------------------------ */
/* --- Helper routines --- ------------------------------------------------ */
/* ------------------------------------------------------------------------ */

/* Memory map regions requested by Vendor Chip Detection function */
static void dommap(void) {
  if (size0 && ptr0==NULL) ptr0 = mymmap(mem,base0,size0) ;
  if (size1 && ptr1==NULL) ptr1 = mymmap(mem,base1,size1) ;
  if (size2 && ptr2==NULL) ptr2 = mymmap(mem,base2,size2) ; }

/* Add a Register Group (chip.reg) */
static void addgroup(char *grp1, char *grp2) {
  int i, n ;
  if (ngrps >= MAXGROUPS) return ;
  if (*grp2 == '-') *grp2 = 0 ;
  strncpy(grp[ngrps].name1,grp1,MAXGROUPLEN) ;
  strncpy(grp[ngrps].name2,grp2,MAXGROUPLEN) ;
  n = strlen(grp1) ; for ( i=0; i<n; i++ )
    if (isupper(grp1[i])) grp1[i] = tolower(grp1[i]) ;
  n = strlen(grp2) ; for ( i=0; i<n; i++ )
    if (isupper(grp2[i])) grp2[i] = tolower(grp2[i]) ;
  strncpy(grp[ngrps].opti1+1,grp1,MAXGROUPLEN) ;
  strncpy(grp[ngrps].opti2+1,grp2,MAXGROUPLEN) ;
  strncpy(grp[ngrps].optx1+1,grp1,MAXGROUPLEN) ;
  strncpy(grp[ngrps].optx2+1,grp2,MAXGROUPLEN) ;
  grp[ngrps].opti1[0] = '-' ; grp[ngrps].optx1[0] = '+' ;
  grp[ngrps].opti2[0] = '-' ; grp[ngrps].optx2[0] = '+' ;
  ngrps++ ; }

/* Add a Register (chip.reg) */
static void addreg(char *method, char *offset, char *page, char *size,
                   char *group, char *name, char *mask) {
  int i ; accessmethod *a=acc ;
  if (nregs >= MAXREGS) return ; if (!mask) mask = "0" ;
  for ( i=0; a->name; i++, a++ )
    if (!strcmp(a->name,method)) reg[nregs].acc = i ;
  for ( i=0; i<ngrps; i++ )
    if (!strcmp(grp[i].name1,group)) reg[nregs].grp = i ;
  strncpy(reg[nregs].name,name,MAXREGLEN) ;
  strncpy(reg[nregs].page,page,MAXPAGELEN) ;
  reg[nregs].offset = strtoul(offset,NULL,16) ;
  reg[nregs].mask = strtoul(mask,NULL,16) ;
  reg[nregs].size = strtol(size,NULL,10) ;
  reg[nregs].value = 0xDEADBEEF ; nregs++ ; }

/* For nextline() and reg_*() */
static char buf[256] ;
static char *token[16] ;

/* Read next line from chip.reg file and tokenize (split on tabs/newline) */
static int nextline(FILE *file) {
  int i=0 ; char *str, *ptr ;
  memset(token,0,sizeof(token)) ;
  while (!feof(file)) {
    if ( fscanf(file,"%255[^\n]\n",buf) != 1 ) return 0 ;
    for ( i=0, str=ptr=buf; i<=16 && ptr; i++, str=NULL ) {
      ptr = strtok(str," \t\n") ; if (ptr) {
	while (*ptr==' ' || *ptr=='\t' || *ptr=='\n') ptr++ ;
        if (*ptr == '#') ptr = NULL ; }
      token[i] = ptr ; if ((i && !ptr) || i>=16) return i ; } }
  return 0 ; }

static int reg_groups(FILE *file) {
  int n ;
  while (n=nextline(file)) {
    if (*token[0] == '*') return 1 ;
    if (n>=2) addgroup(token[0],token[1]) ;
    else fprintf(stderr,"gfxdump: Ignored %s\n",buf) ; }
  return 0 ; }

static int reg_regs(FILE *file) {
  int n ;
  while (n=nextline(file)) {
    if (*token[0] == '*') return 1 ;
    if (n>=6) addreg(token[0],token[1],token[2],token[3],
                     token[4],token[5],token[6]) ;
    else fprintf(stderr,"gfxdump: Ignored %s\n",buf) ; }
  return 0 ; }

static int reg_file(char *name) {
  int i, n, full=0 ; FILE *file ;
  if (!(file=fopen(name,"r"))) {
    perror("gfxdump: fopen(<chip>.reg) failed") ; exit(1) ; }
  while (full || (n=nextline(file))) {
    if (n && !strcmp(token[0],"*END*")) { while (nextline(file)) ; full = 0 ; }
    else if (!strcmp(token[0],"*INCLUDE*"))   full = reg_file(token[1]) ;
    else if (!strcmp(token[0],"*GROUPS*"))    full = reg_groups(file) ;
    else if (!strcmp(token[0],"*REGISTERS*")) full = reg_regs(file) ;
    else { full = 0 ; printf("gfxdump: ignored: %s\n",buf) ; } }
  fclose(file) ; return 0 ; }

/* Print out the value of a register */
static void prtreg(REG *r) {
  int i, n ;
  n = acc[r->acc].prt(r) ; for ( i=n; i<=8; i++ ) printf(" ") ;
  printf("= ") ; for ( i=2*(4-r->size); i; i-- ) printf(" ") ;
  if (xopt) printf("0x") ;
  if (r->valid) switch (r->size) {
    case 1: printf("%02X", (u8)r->value) ; break ;
    case 2: printf("%04X",(u16)r->value) ; break ;
    case 3: printf("%06lX",(u32)r->value) ; break ;
    case 4: printf("%08lX",(u32)r->value) ; break ; }
  else for ( i=2*r->size; i; i-- ) printf("?") ;
  if (pageopt||groupopt||nameopt) printf("  ") ;
  if (pageopt) {
    n = printf("%s",r->page) ;
    if (groupopt||nameopt) for ( i=n; i<=6; i++ ) printf(" ") ; }
  if (groupopt) {
    n = printf("%s",grp[r->grp].name1) ;
    if (nameopt) for ( i=n; i<=MAXGROUPLEN; i++ ) printf(" ") ; }
  if (nameopt) {
    n = printf("%s",r->name) ;
    /*for ( i=n; i<=MAXREGLEN; i++ ) printf(" ") ;*/ }
  printf("\n") ; }

/* Parse command line options */
static int parseopts(int argc, char **argv) {
  int i, j, n ;
  for ( i=1; i<argc && (*argv[i]=='-' || *argv[i]=='+'); i++) {
    n = strlen(argv[i]) ; for ( j=0; j<n; j++ )
      if (isupper(argv[i][j])) argv[i][j] = tolower(argv[i][j]) ;
    if      (!strcmp(argv[i],"-a")) allopt = 1 ;
    else if (!strcmp(argv[i],"-b")) binopt = 1 ;
    else if (!strcmp(argv[i],"-d")) dangeropt = 1 ;
    else if (!strcmp(argv[i],"-e")) everyopt = 1 ;
    else if (!strcmp(argv[i],"-g")) groupopt = 1 ;
    else if (!strcmp(argv[i],"-n")) nameopt = 1 ;
    else if (!strcmp(argv[i],"-p")) pageopt = 1 ;
    else if (!strcmp(argv[i],"-v")) verbose = 1 ;
    else if (!strcmp(argv[i],"-x")) xopt = 1 ;
    else if (!strcmp(argv[i],"-r") && i<=argc) repopt = atoi(argv[++i]) ;
    else for ( j=0; j<ngrps; j++ ) {
      if (!strcmp(argv[i],grp[j].opti1)) grp[j].incl = 1 ;
      if (!strcmp(argv[i],grp[j].opti2)) grp[j].incl = 1 ;
      if (!strcmp(argv[i],grp[j].optx1)) grp[j].incl = -1 ;
      if (!strcmp(argv[i],grp[j].optx2)) grp[j].incl = -1 ; } }
  if (everyopt) groupopt = nameopt = pageopt = 1 ;
  return i ; }

/* ------------------------------------------------------------------------ */
/* --- Global variables --- ----------------------------------------------- */
/* ------------------------------------------------------------------------ */

/* Graphics chip vendors */
/* Note: Placed after Vendor Specific Chip Type Detection
 * routines, to avoid predeclaration of same */
static struct { u16 vendor ; char *name ; char *(*func)() ; }
vendors[] = {
  { 0x1002, "ATI Technologies",	&ati },
  { 0x100C, "Tseng Labs",	&tseng },
  { 0x100E, "Weitek",		&weitek },
  { 0x1013, "Cirrus Logic",	&cirrus },
  { 0x1023, "Trident",		&trident },
  { 0x102B, "Matrox",		&matrox },
  { 0x1039, "SIS",		&sis },
  { 0x105D, "Number Nine",	&number9 },
  { 0x10C8, "Neomagic",		&neomagic },
  { 0x1163, "Rendition",	&rendition },
  { 0x3D3D, "3Dlabs",		&labs3d },
  { 0x5333, "S3",		&s3 },
  { 0x0000, NULL,		NULL } } ;

/* ------------------------------------------------------------------------ */
/* --- Main program --- --------------------------------------------------- */
/* ------------------------------------------------------------------------ */

int main(int argc, char **argv) {

  char buf[9], *name=NULL, *file=NULL, *ident=NULL, *(*func)()=NULL ;
  int i, j, k, idx=-1 ; u8 vgakey ; char *sp="" ;

  /* Zero main structs */
  reg = (REG*) malloc(MAXREGS*sizeof(REG)) ;
  grp = (GRP*) malloc(MAXGROUPS*sizeof(GRP)) ;
  memset(reg,0,sizeof(reg)) ;
  memset(grp,0,sizeof(grp)) ;

  /* Parse command line options (recognizes
   * single-char switch options only) */
  i = parseopts(argc,argv) ;

  /* bus:card:func specified ? */
  if (argc-i>=1) ident = argv[i++] ;

  /* Reopen stdout if optional 2nd arg (output filename) given */
  if (i<argc) freopen(argv[i++],MODE,stdout) ;
  if (!stdout) { perror("freopen(stdout) failed") ; exit(1) ; }

  /* Scan PCI/AGP busses */
  pci_scan() ;	/* In pci.c */
  for ( i=0; i<pcicards; i++ ) {
    #ifdef __unix__
    snprintf(buf,sizeof(buf),"%d:%d:%d",
    #else
    sprintf(buf,"%d:%d:%d",
    #endif
      pciinfo[i].bus, pciinfo[i].card, pciinfo[i].func) ;
    if (ident && !strcmp(buf,ident)) idx = i ; }

  /* Desired card present and card vendor supported ? */
  if (ident)
    if (idx>=0) {
      vendor  = pciinfo[idx].vendor ;
      device  = pciinfo[idx].device ;
      base0   = pciinfo[idx].base0 ;
      base1   = pciinfo[idx].base1 ;
      base2   = pciinfo[idx].base2 ;
      baserom = pciinfo[idx].baserom ;
      for ( i=0; vendors[i].func; i++ )
        if (vendors[i].vendor == vendor) {
          name = vendors[i].name ;
          func = vendors[i].func ; }
      if (!func) {
        fprintf(stderr,"Card @ %s (vendor %04X) is not a graphics card or is unsupported\n",
          ident,vendor) ;
        exit(1) ; } }
    else {
      fprintf(stderr,"Invalid PCI/AGP address %s or no card present\n",ident) ;
      exit(1) ; }
  /* No card specified; print out list of cards found */
  else {
    if (!name || verbose) for ( i=0; i<pcicards; i++ ) if (
      pciinfo[i].bus>9 || pciinfo[i].card>9 || pciinfo[i].func>9) sp = " " ;
    for ( i=0; i<pcicards; i++ ) {
      name = NULL ;
      for ( j=0; vendors[j].func; j++ )
        if (vendors[j].vendor == pciinfo[i].vendor) name = vendors[j].name ;
      if (name || verbose) {
        fprintf(stderr,"%d:%d:%d%s : %04X:%04X",
          pciinfo[i].bus, pciinfo[i].card, pciinfo[i].func,
          (pciinfo[i].bus>9 || pciinfo[i].card>9 || pciinfo[i].func>9)
            ? "" : sp,   pciinfo[i].vendor, pciinfo[i].device) ;
        if (name) fprintf(stderr," (%s)\n",name) ;
        else fprintf(stderr,"\n") ; } }
    exit(0) ; }

  /* Print out values */
  if (verbose) {
    fprintf(stderr,"Card @ %s : %04X:%04X (%s):\n", ident,vendor,device,name) ;
    fprintf(stderr,"base[0,1,2] = 0x%08lX, 0x%08lX, 0x%08lX\n",base0,base1,base2) ; }

  /* Enable access to all I/O ports */
  #ifdef __unix__
  if (iopl(3)) { perror("gfxdump: iopl() failed") ; exit(1) ; }
  #endif

  /* Open /dev/mem for access to real memory */
  #ifdef __unix__
  if ( (mem = open("/dev/mem",O_RDWR)) == -1) {
    perror("gfxdump: open(/dev/mem) failed") ; exit(1) ; }
  #endif

  /* Save, then set the VGA KEY */
  vgakey = inb(0x3D8) ; outb(0x3D8,0xA0|vgakey) ;

  /* Determine access methods and register
   * list from Chip Vendor and Device ID */
  /* Note: The vendor function may call dommap() */
  file = func() ; dommap() ;

  /* Read and parse chip.reg file */
  if (!file) { fprintf(stderr,
    "gfxdump: Unknown Chip Device %04X:%04X\n",vendor,device) ; exit(1) ; }
  reg_file("VGA.reg") ; reg_file(file) ;

  /* Reparse command line options to recognize register group options */
  parseopts(argc,argv) ;

  /* All registers ? */
  if (allopt) for ( i=0; i<ngrps; i++ )
    if (grp[i].incl==0) grp[i].incl = 1 ;

  /* Mark registers for inclusion by group */
  for ( i=0; i<nregs; i++ ) if (grp[reg[i].grp].incl>0) reg[i].incl = 1 ;

  for ( k=0; k<repopt; k++ ) {

    /* Short delay between dumps */
    #ifdef __unix__
    if (k) sleep(1) ;
    #endif

    /* Collect register values */
    for ( i=0; i<nregs; i++ )
      if (reg[i].incl && (dangeropt || reg[i].name[0] != '*')) {
	reg[i].value = acc[reg[i].acc].acc(&reg[i]) ; reg[i].valid = 1 ; }

    /* Print out register values */
    for ( i=0; i<nregs; i++ ) if (reg[i].incl>0) prtreg(&reg[i]) ; }

  /* Restore the VGA KEY */
  outb(0x3D8,vgakey) ;

  return 0 ; }
