/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "mainwindow.h"


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    this->setWindowTitle("Dianara");
    this->setWindowIcon(QIcon(":/icon/64x64/dianara.png"));
    this->setMinimumSize(400, 400);

    QSettings settings;

    firstRun = true;
    prepareDataDirectory(); // This sets this->dataDirectory

    reallyQuitProgram = false;
    trayIconAvailable = false;


    QString currentIconset = QIcon::themeName();
    qDebug() << "System iconset:" << currentIconset;
    qDebug() << "Icon theme search paths:" << QIcon::themeSearchPaths();
    if (currentIconset.isEmpty() || currentIconset == "hicolor")
    {
        qDebug() << ">> No system iconset (or hicolor) configured; trying to use Oxygen";
        QIcon::setThemeName("oxygen"); // VERY TMPFIX
    }

    // Network control
    pumpController = new PumpController();


    ////// GUI


    // User's profile editor, in its own window
    profileEditor = new ProfileEditor(pumpController, this);


    // Splitter to divide the window horizontally
    mainSplitter = new QSplitter(Qt::Horizontal, this);
    mainSplitter->setChildrenCollapsible(false);
    mainSplitter->setContentsMargins(0, 0, 0, 0);
    mainSplitter->setHandleWidth(4);


    // Left side
    leftSideWidget = new QWidget();
    leftLayout = new QVBoxLayout();
    leftLayout->setContentsMargins(1, 1, 0, 1);

    avatarIconButton = new QPushButton();
    avatarIconButton->setIcon(QIcon(QPixmap(":/images/no-avatar.png")
                                    .scaled(64, 64,
                                            Qt::KeepAspectRatio,
                                            Qt::SmoothTransformation)));
    avatarIconButton->setSizePolicy(QSizePolicy::Maximum,
                                    QSizePolicy::Maximum);
    avatarIconButton->setFlat(true);
    avatarIconButton->setIconSize(QSize(64, 64));
    connect(avatarIconButton, SIGNAL(clicked()),
            profileEditor, SLOT(show()));
    leftLayout->addWidget(avatarIconButton, 0, Qt::AlignLeft);


    userInfoLayout = new QVBoxLayout();
    userInfoLayout->setContentsMargins(10, 0, 10, 0);

    fullNameLabel = new QLabel("[---------------]");
    fullNameLabel->setWordWrap(true);
    userInfoLayout->addWidget(fullNameLabel);


    QFont userDetailsFont;
    userDetailsFont.setBold(true);
    userDetailsFont.setItalic(true);
    userDetailsFont.setPointSize(userDetailsFont.pointSize() - 2);

    userIdLabel = new QLabel();
    userIdLabel->setWordWrap(true);
    userIdLabel->setFont(userDetailsFont);
    userInfoLayout->addWidget(userIdLabel);

    userDetailsFont.setBold(false);
    userDetailsFont.setItalic(false);

    userHometownLabel = new QLabel();
    userHometownLabel->setWordWrap(true);
    userHometownLabel->setFont(userDetailsFont);
    userInfoLayout->addWidget(userHometownLabel);

    leftLayout->addLayout(userInfoLayout);



    leftPanel = new QToolBox(); // For now, only one "section"...

    meanwhileFeed = new MinorFeed(pumpController);
    connect(meanwhileFeed, SIGNAL(newItemsCountChanged(int)),
            this, SLOT(setMinorFeedTitle(int)));
    leftPanel->addItem(meanwhileFeed,
                       QIcon::fromTheme("clock"),
                       "*meanwhile*");
    this->setMinorFeedTitle(0); // Set initial title
    leftLayout->addWidget(leftPanel);

    this->leftSideWidget->setLayout(leftLayout);



    // Right side
    rightSideWidget = new QWidget();
    rightLayout = new QVBoxLayout();
    rightLayout->setContentsMargins(0, 1, 1, 1);

    publisher = new Publisher(pumpController);


    /// START SETTING UP TIMELINES


    // Main timeline //

    mainTimeline = new TimeLine(TimeLine::TimelineTypeMain,
                                pumpController,
                                this);
    mainTimelineScrollArea = new QScrollArea();
    mainTimelineScrollArea->setWidget(mainTimeline);  // Make timeline scrollable
    mainTimelineScrollArea->setWidgetResizable(true);
    mainTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(mainTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollMainTimelineToTop()));
    connect(mainTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));
    connect(mainTimeline, SIGNAL(unreadPostsCountChanged(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));

    connect(mainTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(notifyTimelineUpdate(int,int)));

    // for post editing
    connect(mainTimeline, SIGNAL(postEditRequested(QString,QString,QString)),
            publisher, SLOT(setEditingMode(QString,QString,QString)));

    // To ensure comment composer is visible
    connect(mainTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollMainTimelineToWidget(QWidget*)));


    // Direct timeline //

    directTimeline = new TimeLine(TimeLine::TimelineTypeDirect,
                                  pumpController,
                                  this);
    directTimelineScrollArea = new QScrollArea();
    directTimelineScrollArea->setWidget(directTimeline);
    directTimelineScrollArea->setWidgetResizable(true);
    directTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(directTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollDirectTimelineToTop()));
    connect(directTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));
    connect(directTimeline, SIGNAL(unreadPostsCountChanged(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));

    connect(directTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(notifyTimelineUpdate(int,int)));

    // direct timeline doesn't need a connection for post editing,
    // since you're not gonna send a message to yourself =)

    // To ensure comment composer is visible
    connect(directTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollDirectTimelineToWidget(QWidget*)));


    // Activity timeline //

    activityTimeline = new TimeLine(TimeLine::TimelineTypeActivity,
                                    pumpController,
                                    this);
    activityTimelineScrollArea = new QScrollArea();
    activityTimelineScrollArea->setWidget(activityTimeline);  // Make it scrollable
    activityTimelineScrollArea->setWidgetResizable(true);
    activityTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(activityTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollActivityTimelineToTop()));
    connect(activityTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));
    connect(activityTimeline, SIGNAL(unreadPostsCountChanged(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));

    connect(activityTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(notifyTimelineUpdate(int,int)));

    // for post editing
    connect(activityTimeline, SIGNAL(postEditRequested(QString,QString,QString)),
            publisher, SLOT(setEditingMode(QString,QString,QString)));

    // To ensure comment composer is visible
    connect(activityTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollActivityTimelineToWidget(QWidget*)));


    // Favorites timeline //

    favoritesTimeline = new TimeLine(TimeLine::TimelineTypeFavorites,
                                     pumpController,
                                     this);
    favoritesTimelineScrollArea = new QScrollArea();
    favoritesTimelineScrollArea->setWidget(favoritesTimeline);
    favoritesTimelineScrollArea->setWidgetResizable(true);
    favoritesTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(favoritesTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollFavoritesTimelineToTop()));
    connect(favoritesTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));
    connect(favoritesTimeline, SIGNAL(unreadPostsCountChanged(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));

    connect(favoritesTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(notifyTimelineUpdate(int,int)));

    // for post editing
    connect(favoritesTimeline, SIGNAL(postEditRequested(QString,QString,QString)),
            publisher, SLOT(setEditingMode(QString,QString,QString)));

    // To ensure comment composer is visible
    connect(favoritesTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollFavoritesTimelineToWidget(QWidget*)));


    /// END SETTING UP TIMELINES


    // The contact list has its own tabs with its own scroll areas
    contactList = new ContactList(pumpController, this);


    tabWidget = new QTabWidget();
    tabWidget->addTab(mainTimelineScrollArea,
                      QIcon::fromTheme("view-list-details"),
                      "MAIN TIMELINE TAB");
    tabWidget->setTabToolTip(0, tr("The main timeline"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeMain, 0);
    tabWidget->addTab(directTimelineScrollArea,
                      QIcon::fromTheme("mail-message"),
                      "MESSAGES TAB");
    tabWidget->setTabToolTip(1, tr("Messages sent explicitly to you"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeDirect, 0);
    tabWidget->addTab(activityTimelineScrollArea,
                      QIcon::fromTheme("user-home"),
                      "ACTIVITY TAB");
    tabWidget->setTabToolTip(2, tr("Your own posts"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeActivity, 0);
    tabWidget->addTab(favoritesTimelineScrollArea,
                      QIcon::fromTheme("folder-favorites"),
                      "FAVORITES TAB");
    tabWidget->setTabToolTip(3, tr("Your favorited posts"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeFavorites, 0);
    tabWidget->addTab(contactList,
                      QIcon::fromTheme("system-users"),
                      tr("&Contacts"));
    tabWidget->setTabToolTip(4, tr("The people you follow, and the ones who follow you"));
    connect(tabWidget, SIGNAL(currentChanged(int)),
            this, SLOT(setTitleAndTrayInfo(int)));



    rightLayout->addWidget(publisher, 1); // stretch 1/10
    rightLayout->addWidget(tabWidget, 9); // stretch 9/10

    this->rightSideWidget->setLayout(rightLayout);


    mainSplitter->addWidget(leftSideWidget);
    mainSplitter->addWidget(rightSideWidget);
    this->setCentralWidget(mainSplitter);



    ////////////////// Load configuration from disk
    loadSettings();


    // FreeDesktop.org notifications handler
    fdNotifier = new FDNotifications();
    fdNotifier->setNotificationType(showNotifications); // valid since loadSettings()
                                                        // was just called
    connect(fdNotifier, SIGNAL(showFallbackNotification(QString)),
            this, SLOT(showTrayFallbackMessage(QString)));


    updateTimer = new QTimer(this);
    updateTimer->setInterval(this->updateInterval * 1000 * 60); // min > msec
    connect(updateTimer, SIGNAL(timeout()),
            this, SLOT(updateMainDirectMinorTimelines()));
    updateTimer->start();


    //// External widgets which live in their own windows

    // Account wizard
    accountDialog = new AccountDialog(pumpController, this);
    connect(accountDialog, SIGNAL(userIDChanged(QString)),
            this, SLOT(updateUserID(QString)));

    // If this is the first run, show the account Dialog
    if (firstRun)
    {
        accountDialog->show();
    }

    // Configuration dialog
    configDialog = new ConfigDialog(this->dataDirectory,
                                    this->updateInterval,
                                    this->postsPerPageMain,
                                    this->postsPerPageOther,
                                    this->tabsPosition,
                                    this->tabsMovable,
                                    this->publicPosts,
                                    this->showNotifications,
                                    this);
    connect(configDialog, SIGNAL(configurationChanged(int,int,int,int,bool,bool,int)),
            this, SLOT(updateSettings(int,int,int,int,bool,bool,int)));


    // Filter editor
    filterEditor = new FilterEditor(pumpController, this);



    ///
    //////////////////////////////// Connections for PumpController //////////
    ///

    connect(pumpController, SIGNAL(profileReceived(QString,QString,QString,QString,QString)),
            this, SLOT(updateProfileData(QString,QString,QString,QString,QString)));

    connect(pumpController, SIGNAL(avatarPictureReceived(QByteArray,QUrl)),
            this, SLOT(storeAvatar(QByteArray,QUrl)));
    connect(pumpController, SIGNAL(imageReceived(QByteArray,QUrl)),
            this, SLOT(storeImage(QByteArray,QUrl)));


    // After receiving timeline contents, update corresponding timeline
    connect(pumpController, SIGNAL(mainTimelineReceived(QVariantList,int,QString,QString)),
            mainTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));
    connect(pumpController, SIGNAL(directTimelineReceived(QVariantList,int,QString,QString)),
            directTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));
    connect(pumpController, SIGNAL(activityTimelineReceived(QVariantList,int,QString,QString)),
            activityTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));
    connect(pumpController, SIGNAL(favoritesTimelineReceived(QVariantList,int,QString,QString)),
            favoritesTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));


    // After sucessful posting, request updated timeline, with your post included
    connect(pumpController, SIGNAL(postPublished()),
            this, SLOT(updateMainActivityMinorTimelines()));


    // After successful liking, update likes count
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            mainTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            directTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            activityTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    // We don't update likes count in favorites timeline,
    // since it still doesn't know about this post

    // Instead, reload favorites timeline
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            favoritesTimeline, SLOT(goToFirstPage()));


    // After commenting successfully, refresh list of comments in that post
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            mainTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            directTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            activityTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            favoritesTimeline, SLOT(setCommentsInPost(QVariantList,QString)));


    // After successful sharing....

    // TODO***


    // After receiving the minor feed ("Meanwhile"), update it
    connect(pumpController, SIGNAL(minorFeedReceived(QVariantList)),
            meanwhileFeed, SLOT(setFeedContents(QVariantList)));


    // After receiving a contact list, update it
    connect(pumpController, SIGNAL(contactListReceived(QString,QVariantList,int)),
            contactList, SLOT(setContactListContents(QString,QVariantList,int)));

    // After receiving the list of lists, update it
    connect(pumpController, SIGNAL(listsListReceived(QVariantList)),
            contactList, SLOT(setListsListContents(QVariantList)));


    // Show notifications for events sent from pumpController
    connect(pumpController, SIGNAL(showNotification(QString)),
            fdNotifier, SLOT(showMessage(QString)));

    // Update statusBar message from pumpController's infos
    connect(pumpController, SIGNAL(currentJobChanged(QString)),
            this, SLOT(setStatusBarMessage(QString)));



    // Add menus
    createMenus();

    // Add the system tray icon
    createTrayIcon();


    // tmp statusBar stuff
    this->statusBar()->showMessage(tr("Initializing..."));


    settings.beginGroup("MainWindow");

    // Now set the "view side panel" checkable menu to its saved state
    // That will also trigger the action to hide/show it
    viewSidePanel->setChecked(settings.value("viewSidePanel", true).toBool());

    // Set the "view status bar" checkable menu to its saved state, which
    // will also trigger the action to hide/show it
    viewStatusBar->setChecked(settings.value("viewStatusBar", true).toBool());

    settings.endGroup();


    // If User ID is defined, set PumpController in motion
    if (!userID.isEmpty())
    {
        pumpController->setUserCredentials(this->userID);
        // getUserProfile() will be called from setUserCredentials()
    }
    else // Otherwise, just say so in the statusbar
    {
        this->setStatusBarMessage(tr("Your account is not configured yet."));
    }

    this->adjustTimelineSizes();
    qApp->processEvents();

    qDebug() << "MainWindow created";
}



MainWindow::~MainWindow()
{
    qDebug() << "MainWindow destroyed";
}




void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "MainWindow::closeEvent()";


    if (reallyQuitProgram)
    {
        event->accept(); // really close, if called from Quit menu
        qDebug() << "Quit called from menu, shutting down program";
    }
    else
    {
        this->hide();     // Hide window, app accessible via tray icon
        qDebug() << "Tried to close main window, so hiding to tray";
        event->ignore();  // ignore the closeEvent
    }
}


void MainWindow::resizeEvent(QResizeEvent *event)
{
    qDebug() << "MainWindow::resizeEvent()" << event->size();
    this->adjustTimelineSizes();

    event->accept();
}




/*
 * Prepare the data directory. Create if necessary
 *
 */
void MainWindow::prepareDataDirectory()
{
    #if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    dataDirectory = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
    #else
    dataDirectory = QStandardPaths::standardLocations(QStandardPaths::DataLocation).first();
    #endif
    qDebug() << "Data directory:" << this->dataDirectory;

    QDir dataDir;
    if (!dataDir.exists(dataDirectory))
    {
        qDebug() << "Creating data directory";
        if (dataDir.mkpath(dataDirectory))
        {
            qDebug() << "Data directory created";
        }
        else
        {
            qDebug() << "Error creating data directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/images"))
    {
        qDebug() << "Creating images directory";
        if (dataDir.mkpath(dataDirectory + "/images"))
        {
            qDebug() << "Images directory created";
        }
        else
        {
            qDebug() << "Error creating images directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/avatars"))
    {
        qDebug() << "Creating avatars directory";
        if (dataDir.mkpath(dataDirectory + "/avatars"))
        {
            qDebug() << "Avatars directory created";
        }
        else
        {
            qDebug() << "Error creating avatars directory!";
        }
    }

}



/*
 * Populate the menus
 *
 */
void MainWindow::createMenus()
{
    sessionMenu = new QMenu(tr("&Session"));

    sessionUpdateMainTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                            tr("&Update Main Timeline"),
                                            this);
    sessionUpdateMainTimeline->setShortcut(QKeySequence(Qt::Key_F5));
    connect(sessionUpdateMainTimeline, SIGNAL(triggered()),
            mainTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateMainTimeline);

    sessionUpdateDirectTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                              tr("Update &Messages Timeline"),
                                              this);
    sessionUpdateDirectTimeline->setShortcut(QKeySequence(Qt::Key_F6));
    connect(sessionUpdateDirectTimeline, SIGNAL(triggered()),
            directTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateDirectTimeline);

    sessionUpdateActivityTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                                tr("Update &Activity Timeline"),
                                                this);
    sessionUpdateActivityTimeline->setShortcut(QKeySequence(Qt::Key_F7));
    connect(sessionUpdateActivityTimeline, SIGNAL(triggered()),
            activityTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateActivityTimeline);


    sessionUpdateFavoritesTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                                 tr("Update Favorites &Timeline"),
                                                 this);
    sessionUpdateFavoritesTimeline->setShortcut(QKeySequence(Qt::Key_F8));
    connect(sessionUpdateFavoritesTimeline, SIGNAL(triggered()),
            favoritesTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateFavoritesTimeline);


    sessionUpdateMinorFeed = new QAction(QIcon::fromTheme("view-refresh"),
                                         tr("Update Minor &Feed"),
                                         this);
    sessionUpdateMinorFeed->setShortcut(QKeySequence(Qt::Key_F10));
    connect(sessionUpdateMinorFeed, SIGNAL(triggered()),
            meanwhileFeed, SLOT(updateFeed()));
    sessionMenu->addAction(sessionUpdateMinorFeed);


    sessionUpdateAllTimelines = new QAction(QIcon::fromTheme("view-refresh"),
                                         tr("Update All Timelines"),
                                         this);
    sessionUpdateAllTimelines->setShortcut(QKeySequence("Ctrl+F5"));
    connect(sessionUpdateAllTimelines, SIGNAL(triggered()),
            this, SLOT(updateAllTimelines()));
    sessionMenu->addAction(sessionUpdateAllTimelines);

    sessionMenu->addSeparator();

    sessionMarkAllAsRead = new QAction(QIcon::fromTheme("mail-mark-read"),
                                       tr("Mark All as Read"),
                                       this);
    sessionMarkAllAsRead->setShortcut(QKeySequence("Ctrl+R"));
    connect(sessionMarkAllAsRead, SIGNAL(triggered()),
            this, SLOT(markAllAsRead()));
    sessionMenu->addAction(sessionMarkAllAsRead);    

    sessionMenu->addSeparator();

    sessionPostNote = new QAction(QIcon::fromTheme("document-edit"),
                                          tr("&Post a Note"),
                                          this);
    sessionPostNote->setShortcut(QKeySequence("Ctrl+N"));
    connect(sessionPostNote, SIGNAL(triggered()),
            publisher, SLOT(setFullMode()));
    connect(sessionPostNote, SIGNAL(triggered()),
            this, SLOT(show()));  // show window, in case it's hidden and
                                  // Post a Note is used from tray menu
    sessionMenu->addAction(sessionPostNote);

    sessionMenu->addSeparator();

    sessionQuit = new QAction(QIcon::fromTheme("application-exit"),
                              tr("&Quit"), this);
    sessionQuit->setShortcut(QKeySequence("Ctrl+Q"));
    connect(sessionQuit, SIGNAL(triggered()),
            this, SLOT(quitProgram()));
    sessionMenu->addAction(sessionQuit);

    this->menuBar()->addMenu(sessionMenu);


    viewMenu = new QMenu(tr("&View"));

    viewSidePanel = new QAction(QIcon::fromTheme("view-sidetree"),
                                tr("Side &Panel"), this);
    connect(viewSidePanel, SIGNAL(toggled(bool)),
            this, SLOT(toggleSidePanel(bool)));
    viewSidePanel->setCheckable(true);
    viewSidePanel->setChecked(true);
    viewSidePanel->setShortcut(Qt::Key_F9);

    viewMenu->addAction(viewSidePanel);

    viewStatusBar = new QAction(QIcon::fromTheme("configure-toolbars"),
                                tr("Status &Bar"), this);
    connect(viewStatusBar, SIGNAL(toggled(bool)),
            this, SLOT(toggleStatusBar(bool)));
    viewStatusBar->setCheckable(true);
    viewStatusBar->setChecked(true);

    viewMenu->addAction(viewStatusBar);


    viewFullscreenAction = new QAction(QIcon::fromTheme("view-fullscreen"),
                                       tr("Full &Screen"), this);
    connect(viewFullscreenAction, SIGNAL(toggled(bool)),
            this, SLOT(toggleFullscreen(bool)));
    viewFullscreenAction->setCheckable(true);
    viewFullscreenAction->setChecked(false);
    viewFullscreenAction->setShortcut(Qt::Key_F11);

    viewMenu->addAction(viewFullscreenAction);


    this->menuBar()->addMenu(viewMenu);



    settingsMenu = new QMenu(tr("S&ettings"));

    settingsEditProfile = new QAction(QIcon::fromTheme("user-properties"),
                                      tr("Edit &Profile"), this);
    connect(settingsEditProfile, SIGNAL(triggered()),
            profileEditor, SLOT(show()));
    settingsMenu->addAction(settingsEditProfile);

    settingsAccount = new QAction(QIcon::fromTheme("dialog-password"),
                                  tr("&Account"), this);
    connect(settingsAccount, SIGNAL(triggered()),
            accountDialog, SLOT(show()));
    settingsMenu->addAction(settingsAccount);

    settingsMenu->addSeparator();

    settingsFilters = new QAction(QIcon::fromTheme("view-filter"),
                                  tr("&Filters"), this);
    connect(settingsFilters, SIGNAL(triggered()),
            filterEditor, SLOT(show()));
    settingsMenu->addAction(settingsFilters);

    settingsConfigure = new QAction(QIcon::fromTheme("configure"),
                                    tr("&Configure Dianara"), this);
    connect(settingsConfigure, SIGNAL(triggered()),
            configDialog, SLOT(show()));
    settingsMenu->addAction(settingsConfigure);
    this->menuBar()->addMenu(settingsMenu);
    this->menuBar()->addSeparator();



    helpMenu = new QMenu(tr("&Help"));

    helpVisitWebsite = new QAction(QIcon::fromTheme("internet-web-browser"),
                            tr("Visit &Website"), this);
    connect(helpVisitWebsite, SIGNAL(triggered()),
            this, SLOT(visitWebSite()));
    helpMenu->addAction(helpVisitWebsite);


    helpVisitPumpFAQ = new QAction(QIcon::fromTheme("internet-web-browser"),
                                  tr("&Frequently Asked Questions about Pump.io"), this);
    connect(helpVisitPumpFAQ, SIGNAL(triggered()),
            this, SLOT(visitFAQ()));
    helpMenu->addAction(helpVisitPumpFAQ);


    helpMenu->addSeparator();

    helpAbout = new QAction(QIcon::fromTheme("system-help"),
                            tr("About &Dianara"), this);
    connect(helpAbout, SIGNAL(triggered()),
            this, SLOT(aboutDianara()));
    helpMenu->addAction(helpAbout);
    this->menuBar()->addMenu(helpMenu);



    // Context menu for the tray icon
    trayContextMenu = new QMenu("Tray Context Menu");
    trayContextMenu->setSeparatorsCollapsible(false);
    trayContextMenu->addSeparator()->setText("Dianara");
    trayContextMenu->addAction(QIcon(":/icon/64x64/dianara.png"),
                               tr("&Show Window"),
                               this, SLOT(show()));
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionUpdateMainTimeline);
    trayContextMenu->addAction(sessionMarkAllAsRead);
    trayContextMenu->addAction(sessionPostNote);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(settingsConfigure);
    trayContextMenu->addAction(helpAbout);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionQuit);

    // FIXME: if mainwindow is hidden, program quits
    // after closing Configure or About window (now partially fixed)


    qDebug() << "Menus created";
}



/*
 * Create an icon in the system tray, define its contextual menu, etc.
 *
 */
void MainWindow::createTrayIcon()
{
    trayIcon = new QSystemTrayIcon(this);

    if (trayIcon->isSystemTrayAvailable())
    {
        trayIconAvailable = true;

        this->setTrayIconPixmap(); // Set icon for "no unread messages" initially
        this->setTitleAndTrayInfo(this->tabWidget->currentIndex());


        // Catch clicks on icon
        connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
                this, SLOT(trayControl(QSystemTrayIcon::ActivationReason)));

        // clicking in a popup notification (balloon-type) will show the window
        connect(trayIcon, SIGNAL(messageClicked()),
                this, SLOT(show()));

        // Set contextual menu for the icon
        trayIcon->setContextMenu(this->trayContextMenu);


        trayIcon->show();

        qDebug() << "Tray icon created";
    }
    else
    {
        trayIconAvailable = false;

        qDebug() << "System tray not available";
    }
}


/*
 * Set the tray icon's pixmap, with number of unread messages, or nothing
 *
 */
void MainWindow::setTrayIconPixmap(int count)
{
    QPixmap iconPixmap = QIcon::fromTheme("dianara",
                                          QIcon(":/icon/32x32/dianara.png")).pixmap(32, 32);

    // Paint the number of unread messages on top of the pixmap, if != 0
    if (count > 0)
    {
        QFont font;
        font.setPointSize(font.pointSize() + 1);
        font.setBold(true);

        QPainter painter(&iconPixmap);
        painter.setPen(QColor(Qt::white));
        painter.setFont(font);
        painter.drawText(0, 0,
                         30, 30, // Not 32, 32, to have some margin
                         Qt::AlignRight | Qt::AlignBottom,
                         QString("%1").arg(count));
    }

    this->trayIcon->setIcon(iconPixmap);
}


/*
 * Adjust timelines maximum widths according to their scrollareas sizes,
 * which in turn depend on the window size
 *
 * Called from the resizeEvent(), among other places
 *
 */
void MainWindow::adjustTimelineSizes()
{
    int scrollbarWidth = mainTimelineScrollArea->verticalScrollBar()->width() + 4;
    // FIXME: adding +4 pixels should not be hardcoded

    int timelineWidth;
    int timelineHeight;

    // Get the right timelinewidth based on currently active tab's timeline width
    switch (this->tabWidget->currentIndex())
    {
    case 0: // main
        timelineWidth = mainTimelineScrollArea->width() - scrollbarWidth;
        timelineHeight = mainTimelineScrollArea->height();
        mainTimeline->resizePosts();
        break;
    case 1: // direct
        timelineWidth = directTimelineScrollArea->width() - scrollbarWidth;
        timelineHeight = directTimelineScrollArea->height();
        directTimeline->resizePosts();
        break;
    case 2: // activity
        timelineWidth = activityTimelineScrollArea->width() - scrollbarWidth;
        timelineHeight = activityTimelineScrollArea->height();
        activityTimeline->resizePosts();
        break;
    case 3: // favorites
        timelineWidth = favoritesTimelineScrollArea->width() - scrollbarWidth;
        timelineHeight = favoritesTimelineScrollArea->height();
        favoritesTimeline->resizePosts();
        break;

    default: // Contacts tab, ATM
        timelineWidth = contactList->width() - scrollbarWidth;
        timelineHeight = contactList->height();
    }

    // Then set the maximum width for all of them based on that
    mainTimeline->setMaximumWidth(timelineWidth);
    directTimeline->setMaximumWidth(timelineWidth);
    activityTimeline->setMaximumWidth(timelineWidth);
    favoritesTimeline->setMaximumWidth(timelineWidth);


    // Some basic 1:2 proportions TMP FIXME
    if (timelineHeight > (timelineWidth * 2))
    {
        timelineHeight = timelineWidth * 2;
    }
    mainTimeline->setMinMaxHeightForPosts(timelineHeight);
    directTimeline->setMinMaxHeightForPosts(timelineHeight);
    activityTimeline->setMinMaxHeightForPosts(timelineHeight);
    favoritesTimeline->setMinMaxHeightForPosts(timelineHeight);
}



/*
 * Load general program settings and state: size, position...
 *
 */
void MainWindow::loadSettings()
{
    QSettings settings;

    firstRun = settings.value("firstRun", true).toBool();
    if (firstRun)
    {
        qDebug() << "This is the first run";
    }


    userID = settings.value("userID", "").toString();
    this->setTitleAndTrayInfo(tabWidget->currentIndex());



    // Main window state
    settings.beginGroup("MainWindow");

    this->resize(settings.value("windowSize", QSize(680, 560)).toSize());
    if (!firstRun)
    {
        this->move(settings.value("windowPosition").toPoint());
    }
    this->mainSplitter->restoreState(settings.value("mainSplitterState",
                                                    mainSplitter->saveState()).toByteArray());
    // Set childrenCollapsible to false AFTER loading state, to make sure state does not mess with it
    mainSplitter->setChildrenCollapsible(false);

    settings.endGroup();


    // General program configuration
    settings.beginGroup("Configuration");

    this->updateInterval = settings.value("updateInterval", 5).toInt();

    this->postsPerPageMain = settings.value("postsPerPageMain", 20).toInt();
    this->pumpController->setPostsPerPageMain(this->postsPerPageMain);

    this->postsPerPageOther = settings.value("postsPerPageOther", 5).toInt();
    this->pumpController->setPostsPerPageOther(this->postsPerPageOther);


    this->tabsPosition = settings.value("tabsPosition", QTabWidget::North).toInt();
    tabWidget->setTabPosition((QTabWidget::TabPosition)tabsPosition);

    this->tabsMovable = settings.value("tabsMovable", true).toBool();
    tabWidget->setMovable(tabsMovable);


    this->publicPosts = settings.value("publicPosts", false).toBool();
    this->publisher->setDefaultPublicPosting(this->publicPosts);


    this->showNotifications = settings.value("showNotifications", 0).toInt();

    settings.endGroup();


    qDebug() << "Settings loaded";
}

/*
 * Save general program settings and state: size, position...
 *
 */
void MainWindow::saveSettings()
{
    QSettings settings;

    settings.setValue("firstRun", false);

    // General main window status
    settings.beginGroup("MainWindow");

    settings.setValue("windowSize", this->size());
    settings.setValue("windowPosition", this->pos());
    settings.setValue("mainSplitterState", this->mainSplitter->saveState());

    settings.setValue("viewSidePanel", this->viewSidePanel->isChecked());
    settings.setValue("viewStatusBar", this->viewStatusBar->isChecked());

    settings.endGroup();


    // From config dialog
    settings.beginGroup("Configuration");

    settings.setValue("updateInterval", this->updateInterval);

    settings.setValue("postsPerPageMain",  this->postsPerPageMain);
    settings.setValue("postsPerPageOther", this->postsPerPageOther);

    settings.setValue("tabsPosition", this->tabsPosition);
    settings.setValue("tabsMovable",  this->tabsMovable);

    settings.setValue("publicPosts", this->publicPosts);

    settings.setValue("showNotifications", this->showNotifications);

    settings.endGroup();


    qDebug() << "Settings saved";
}



//////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// SLOTS ////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



/*
 * Update UserID string from signal emitted in AccountDialog
 *
 */
void MainWindow::updateUserID(QString newUserID)
{
    this->userID = newUserID;

    // update window title and tray icon tooltip
    this->setTitleAndTrayInfo(tabWidget->currentIndex());

    this->pumpController->setUserCredentials(userID);

    // Remove current user's name, id and avatar
    this->fullNameLabel->setText("--");
    this->userIdLabel->setText("_@_");
    this->userHometownLabel->setText("--");

    avatarIconButton->setIcon(QIcon(QPixmap(":/images/no-avatar.png")
                                    .scaled(64, 64,
                                            Qt::KeepAspectRatio,
                                            Qt::SmoothTransformation)));


    qDebug() << "UserID updated from AccountDialog:" << userID;
}



/*
 * Update settings changed from ConfigDialog()
 *
 */
void MainWindow::updateSettings(int newUpdateInterval,
                                int newPostsPerPageMain, int newPostsPerPageOther,
                                int newTabsPosition,bool newTabsMovable,
                                bool newPublicPosts, int newShowNotifications)
{
    this->updateInterval = newUpdateInterval;
    this->updateTimer->setInterval(updateInterval * 1000 * 60);

    this->postsPerPageMain = newPostsPerPageMain;
    this->pumpController->setPostsPerPageMain(this->postsPerPageMain);

    this->postsPerPageOther = newPostsPerPageOther;
    this->pumpController->setPostsPerPageOther(this->postsPerPageOther);

    this->tabsPosition = newTabsPosition;
    this->tabWidget->setTabPosition((QTabWidget::TabPosition)tabsPosition);

    this->tabsMovable = newTabsMovable;
    this->tabWidget->setMovable(tabsMovable);

    this->publicPosts = newPublicPosts;
    this->publisher->setDefaultPublicPosting(this->publicPosts);

    this->showNotifications = newShowNotifications;
    fdNotifier->setNotificationType(showNotifications); // FIXME: Safe?


    qDebug() << "updateInterval updated:" << updateInterval << updateInterval*60000;
    qDebug() << "postsPerPage Main/Other:" << postsPerPageMain << postsPerPageOther;
    qDebug() << "tabsPosition updated:" << tabsPosition << tabWidget->tabPosition();
    qDebug() << "tabsMovable updated:" << tabsMovable;
    qDebug() << "public posts updated:" << publicPosts;
    qDebug() << "Notifications updated:" << showNotifications;
}




/*
 * Control interaction with the system tray icon
 *
 */
void MainWindow::trayControl(QSystemTrayIcon::ActivationReason reason)
{
    qDebug() << "Tray icon activation reason:" << reason;

    if (reason != QSystemTrayIcon::Context) // Simple "main button" click in icon
    {
        /*
        qDebug() << "trayControl()";
        qDebug() << "isHidden?"    << this->isHidden();
        qDebug() << "isVisible?"   << this->isVisible();
        qDebug() << "isMinimized?" << this->isMinimized();
        qDebug() << "hasFocus?"    << this->hasFocus();
        */

        // Hide or show the main window
        if (this->isMinimized())
        {
            // hide and show, because raise() wouldn't work
            this->hide();
            this->showNormal();
            qDebug() << "RAISING!";
        }
        else if (this->isHidden())
        {
            this->show();
            qDebug() << "SHOWING";
        }
        else
        {
            this->hide();
            qDebug() << "HIDING";
        }
    }
}



/*
 * If FreeDesktop.org notifications are not available,
 * fall back to Qt's balloon ones
 *
 */
void MainWindow::showTrayFallbackMessage(QString message)
{
    this->trayIcon->showMessage(tr("Dianara Notification"),
                                message,
                                QSystemTrayIcon::Information,
                                4000); // 4 secs
}



void MainWindow::updateProfileData(QString avatarUrl, QString fullName,
                                   QString hometown, QString bio,
                                   QString eMail)
{
    if (!bio.isEmpty())
    {
        bio.prepend("<b></b>"); // make it rich text, so it gets wordwrap
        bio.replace("\n", "<br>"); // HTML newlines
    }
    this->fullNameLabel->setText(fullName);
    this->fullNameLabel->setToolTip(bio);
    this->userIdLabel->setText(this->userID);
    this->userIdLabel->setToolTip(bio);
    this->userHometownLabel->setText(hometown);
    this->userHometownLabel->setToolTip(bio);
    qDebug() << "Updated profile data from server:" << fullName << " @" << hometown;

    this->avatarURL = avatarUrl;
    qDebug() << "Own avatar URL:" << avatarURL;


    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(avatarURL);

    if (QFile::exists(avatarFilename))
    {
        // Load avatar if already cached
        this->avatarIconButton->setIcon(QIcon(QPixmap(avatarFilename)
                                              .scaled(64, 64,
                                                      Qt::KeepAspectRatio,
                                                      Qt::SmoothTransformation)));

        qDebug() << "Using cached avatar for user";
    }
    else
    {
        pumpController->getAvatar(avatarURL);
    }
    this->avatarIconButton->setToolTip(bio);


    // Fill/update this info in the profile editor too
    this->profileEditor->setProfileData(avatarURL, fullName,
                                        hometown, bio,
                                        eMail);
}


/*
 * Update all timelines
 *
 */
void MainWindow::updateAllTimelines()
{
    mainTimeline->goToFirstPage(); // received timeline will come in a SIGNAL()
    directTimeline->goToFirstPage();
    activityTimeline->goToFirstPage();
    favoritesTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated all timelines by menu";
}

/*
 * Update some of the timelines:
 *
 * Main, Direct messages, and Minor feed
 *
 */
void MainWindow::updateMainDirectMinorTimelines()
{
    mainTimeline->goToFirstPage();
    directTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated some timelines by menu or after:" << this->updateInterval << "min";
}


void MainWindow::updateMainActivityMinorTimelines()
{
    mainTimeline->goToFirstPage();
    activityTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated some timelines by menu or after posting";
}



void MainWindow::scrollMainTimelineToTop()
{
    this->mainTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->mainTimelineScrollArea->horizontalScrollBar()->setValue(0);

    this->adjustTimelineSizes();
}

void MainWindow::scrollDirectTimelineToTop()
{
    this->directTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->directTimelineScrollArea->horizontalScrollBar()->setValue(0);

    this->adjustTimelineSizes();
}

void MainWindow::scrollActivityTimelineToTop()
{
    this->activityTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->activityTimelineScrollArea->horizontalScrollBar()->setValue(0);

    this->adjustTimelineSizes();
}

void MainWindow::scrollFavoritesTimelineToTop()
{
    this->favoritesTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->favoritesTimelineScrollArea->horizontalScrollBar()->setValue(0);

    this->adjustTimelineSizes();
}

/*
 * Scroll timelines to make sure the commenter block of the post
 * currently being commented is shown
 *
 */
void MainWindow::scrollMainTimelineToWidget(QWidget *widget)
{
    this->mainTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}

void MainWindow::scrollDirectTimelineToWidget(QWidget *widget)
{
    this->directTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}

void MainWindow::scrollActivityTimelineToWidget(QWidget *widget)
{
    this->activityTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}

void MainWindow::scrollFavoritesTimelineToWidget(QWidget *widget)
{
    this->favoritesTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}



void MainWindow::notifyTimelineUpdate(int timelineType, int newPostCount)
{
    if (newPostCount > 0)
    {
        QString newPostsString;
        if (newPostCount == 1)
        {
            newPostsString = tr("There is 1 new post.");
        }
        else
        {
            newPostsString = tr("There are %1 new posts.").arg(newPostCount);
        }
        this->setStatusBarMessage(tr("Timeline updated.") + " " + newPostsString);


        // If some type of notifications are enabled, and only for main timeline
        if (this->showNotifications != FDNotifications::NoNotifications
           && timelineType == TimeLine::TimelineTypeMain)
        {
            this->fdNotifier->showMessage(tr("Timeline updated at %1.").arg(QTime::currentTime().toString())
                                        + "\n" + newPostsString);
        }
    }
    else
    {
        this->setStatusBarMessage(tr("Timeline updated. No new posts."));
    }


    // In either case, restart update timer, so every manual update of timeline
    this->updateTimer->stop();  // will postpone the auto-update
    this->updateTimer->start();
}


/*
 * Update timelines titles with number of new posts
 *
 */
void MainWindow::setTimelineTabTitle(int timelineType, int newPostCount)
{
    int updatedTab = 0;

    QString messageCountString;
    if (newPostCount > 0)
    {
        messageCountString = QString(" (%1)").arg(newPostCount);
    }


    switch (timelineType)
    {
    case TimeLine::TimelineTypeMain:
        this->tabWidget->setTabText(0,
                                    tr("&Timeline") + messageCountString);
        updatedTab = 0;
        break;

    case TimeLine::TimelineTypeDirect:
        this->tabWidget->setTabText(1,
                                    tr("&Messages") + messageCountString);
        updatedTab = 1;
        break;

    case TimeLine::TimelineTypeActivity:
        this->tabWidget->setTabText(2,
                                    tr("&Activity") + messageCountString);
        updatedTab = 2;
        break;

    case TimeLine::TimelineTypeFavorites:
        this->tabWidget->setTabText(3,
                                    tr("Fav&orites") + messageCountString);
        updatedTab = 3;
        break;
    }

    // If the updated tab is the current one, set also window title, etc.
    if (updatedTab == tabWidget->currentIndex())
    {
        this->setTitleAndTrayInfo(updatedTab);
    }


    // If it's the main timeline, set the tray icon pixmap with the newmsg count
    if (updatedTab == 0)
    {
        if (trayIconAvailable)
        {
            this->setTrayIconPixmap(newPostCount);
        }
    }

}


/*
 * Set mainWindow's title based on current tab, and user ID
 *
 * Use that same title as tray icon's tooltip
 *
 */
void MainWindow::setTitleAndTrayInfo(int currentTab)
{
    QString currentTabTitle;
    currentTabTitle = this->tabWidget->tabText(currentTab);
    currentTabTitle.remove("&"); // Remove accelators

    QString title = currentTabTitle
                  + " - Dianara";

    /*  Not sure if I like it

    if (currentTabTitle.endsWith(")")) // kinda TMP
    {
        title.prepend("* ");
    }
    */

    this->setWindowTitle(title);
    if (trayIconAvailable)
    {
        title = "<b>Dianara</b>: "
              + currentTabTitle
              + "<br><br>["
              + (userID.isEmpty() ?
                        tr("Your Pump.io account is not configured") : userID)
              + "]";


        this->trayIcon->setToolTip(title);
    }
}


/*
 * Set the title for the minor feed (Meanwhile), with new item count
 *
 */
void MainWindow::setMinorFeedTitle(int newItemsCount)
{
    QString title = tr("Meanwhile...");
    if (newItemsCount > 0)
    {
        title.append(QString(" (%1)").arg(newItemsCount));
    }


    this->leftPanel->setItemText(0, title);
}




/*
 * Store avatars on disk
 *
 */
void MainWindow::storeAvatar(QByteArray avatarData, QUrl avatarUrl)
{
    QString fileName = MiscHelpers::getCachedAvatarFilename(avatarUrl.toString());
    qDebug() << "Saving avatar to disk: " << fileName;

    QFile avatarFile(fileName);
    avatarFile.open(QFile::WriteOnly);
    avatarFile.write(avatarData);
    avatarFile.close();

    this->pumpController->notifyAvatarStored(avatarUrl.toString(),
                                             avatarFile.fileName());

    if (avatarUrl == this->avatarURL)
    {
        this->avatarIconButton->setIcon(QIcon(QPixmap(avatarFile.fileName())
                                              .scaled(64, 64,
                                                      Qt::KeepAspectRatio,
                                                      Qt::SmoothTransformation)));
    }

    qDebug() << "avatarData size:" << avatarData.size();
}





/*
 * Store images on disk
 *
 */
void MainWindow::storeImage(QByteArray imageData, QUrl imageUrl)
{
    QString fileName = MiscHelpers::getCachedImageFilename(imageUrl.toString());
    qDebug() << "Saving image to disk: " << fileName;

    QFile imageFile(fileName);
    imageFile.open(QFile::WriteOnly);
    imageFile.write(imageData);
    imageFile.close();

    this->pumpController->notifyImageStored(imageUrl.toString());

    qDebug() << "imageData size:" << imageData.size();
}



/*
 * Update status bar message, from pumpController's (and others) signals
 *
 */
void MainWindow::setStatusBarMessage(QString message)
{
    this->statusBar()->showMessage("[" + QTime::currentTime().toString() + "] " + message, 0);
}


/*
 * Mark all posts in all timelines as read
 * and clear their counters
 *
 */
void MainWindow::markAllAsRead()
{
    mainTimeline->markPostsAsRead();
    directTimeline->markPostsAsRead();
    activityTimeline->markPostsAsRead();
    favoritesTimeline->markPostsAsRead();

    meanwhileFeed->markAllAsRead();
    this->setMinorFeedTitle(0);
}



/*
 * Hide or show the side panel
 *
 */
void MainWindow::toggleSidePanel(bool shown)
{
    // Unless the publisher (really, the composer) has focus already,
    // give focus to timeline before, to avoid giving focus to the publisher
    if (!this->publisher->hasFocus())
    {
        // FIXME: check if publisher's composer has focus!
        this->mainTimeline->setFocus();
    }

    qDebug() << "Showing side panel:" << shown;
    this->leftSideWidget->setVisible(shown);


    if (!shown)
    {
        qApp->processEvents();
        this->adjustTimelineSizes();
    }
}


/*
 * Hide or show the status bar
 *
 */
void MainWindow::toggleStatusBar(bool shown)
{
    qDebug() << "Showing side panel:" << shown;
    this->statusBar()->setVisible(shown);
}


/*
 * Toggle between fullscreen mode and normal window mode
 *
 */
void MainWindow::toggleFullscreen(bool enabled)
{
    if (enabled)
    {
        this->showFullScreen();
    }
    else
    {
        this->showNormal();
    }
}





/*
 * Open website in browser
 *
 */
void MainWindow::visitWebSite()
{
    qDebug() << "Opening website in browser";
    QDesktopServices::openUrl(QUrl("http://jancoding.wordpress.com/dianara"));
}

/*
 * Open pump.io's FAQ in web browser
 * (currently at GitHub)
 *
 */
void MainWindow::visitFAQ()
{
    qDebug() << "Opening Pump.io FAQ in browser";
    QDesktopServices::openUrl(QUrl("https://github.com/e14n/pump.io/wiki/FAQ"));
}




/*
 * About... message
 *
 */
void MainWindow::aboutDianara()
{
    QMessageBox::about(this, tr("About Dianara"),
                       "<b>Dianara v1.1</b><br>"
                       "Copyright 2012-2014  JanKusanagi<br><br>"
                       "<a href=\"http://jancoding.wordpress.com/dianara\">"
                       "http://jancoding.wordpress.com/dianara</a><br><br>"

                       + tr("Dianara is a pump.io social networking client.")
                       + "<br><br>"

                       + tr("With Dianara you can see your timelines, create new posts, "
                            "upload pictures, interact with posts, manage "
                            "your contacts and follow new people.")
                       + "<br><br>"

                       + tr("Thanks to all the testers, translators and packagers, "
                            "who help make Dianara better!")
                       + "<br><br>"

                       + tr("English translation by JanKusanagi.",
                            "TRANSLATORS: Change this with your language and name ;)")
                       + "<br><br>"

                       + tr("Dianara is licensed under the GNU GPL license, and "
                            "uses some Oxygen icons: http://www.oxygen-icons.org/ "
                            "(LGPL licensed)"));


    if (this->isHidden()) // FIXME: ugly workaround to avoid closing the program
    {                     // after closing About dialog, if mainWindow hidden
        this->show();
        this->hide();     // This hack might be causing problems under LXDE
        qDebug() << "MainWindow was hidden, showing and hiding again";
    }
}



/*
 * Close the program. Needed to quit correctly from context menu
 *
 */
void MainWindow::quitProgram()
{
    // Add more needed shutdown stuff here

    if (this->isHidden())
    {
        this->show();
    }

    saveSettings();

    reallyQuitProgram = true;

    qApp->closeAllWindows();

    qDebug() << "All windows closed, bye!";
}
