/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/func_test_case.hpp 
 *
 * @brief [LEVEL: beta] diagnostics::unittest::Func_Test_Case class and @ref
 * diagnostics::unittest::create_func_test_case builder.
 *
 * $Id: func_test_case.hpp,v 1.13 2005/06/23 09:54:24 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/func_test_case.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__FUNC_TEST_CASE_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__FUNC_TEST_CASE_HPP__INCLUDE_GUARD

// brief used as base class 
#include <diagnostics/unittest/std_test_case.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;


/**
 * @class Func_Test_Case diagnostics/unittest/func_test_case.hpp
 *
 * @brief The Func_Test_Case class is used to specify Test_Cases with
 * help of a functional objects.
 *
 * This class uses @ref Std_Test_Case as base and allows to add an
 * object which implements void operator()(Test_Data & test_data) (or
 * just a free function with this signature) to complete the @ref
 * Test_Case. A convenient way to create instances of this class is
 * provided by @ref create_func_test_case.
 *
 * @param FUNC must be copyable and must implement void
 * operator()(Test_Data & test_data) const
 *
 * @nosubgrouping
 */
template <typename FUNC>
class Func_Test_Case :
    public Std_Test_Case
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Func_Test_Case<FUNC> Self;
    typedef Std_Test_Case Super;
public:
    typedef FUNC Function_t;

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    /**
     * @brief no default cstr
     */
    Func_Test_Case();
    /**
     * @brief no copy cstr
     */
    Func_Test_Case(Self const & other);

    /**
     * @brief no assignement
     */
    Self & operator=(Self const & other);
    // @} 

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief the only way to construct a Func_Test_Case. 
     *
     * @a func is the functional object used by @ref run.  All other
     * parameters as well as the error conditions are the same as in
     * @ref Std_Test_Case::Std_Test_Case.
     */
    Func_Test_Case(Function_t const & func,
		   ::std::string const & name,
		   int const timeout,
		   bool const is_rerunable,
		   bool const testable_at_prod,
		   bool const testable_at_debug,
		   bool const testable_at_audit);

    /**
     * @throw never
     */
    virtual ~Func_Test_Case();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Hook Implementation
     * @{
     */
public:
    /**
     * @brief The run method is only delegating to @ref m_func.
     */
    virtual void run(Test_Data & test_data) const;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated State
     * @{
     */
private:
    Function_t const m_func;
    // @}
};

template<typename FUNC>
Func_Test_Case<FUNC>::Func_Test_Case(Function_t const & func,
				     ::std::string const & name,
				     int const timeout,
				     bool const is_rerunable,
				     bool const testable_at_prod,
				     bool const testable_at_debug,
				     bool const testable_at_audit)
    : Super(name,
	    timeout,
	    is_rerunable,
	    testable_at_prod,
	    testable_at_debug,
	    testable_at_audit),
      m_func(func)
{
}

template<typename FUNC>
void Func_Test_Case<FUNC>::run(Test_Data & test_data) const
{
    m_func(test_data);
}

template<typename FUNC>
Func_Test_Case<FUNC>::~Func_Test_Case()
{
}


/**
 * @brief wrapper for the cstr to get rid of explicit template
 * parameter specification.
 */
template<typename FUNC>
Test_Case * create_func_test_case(FUNC const & func,
				  ::std::string const & name,
				  int const timeout,
				  bool const is_rerunable,
				  bool const testable_at_prod,
				  bool const testable_at_debug,
				  bool const testable_at_audit)
{
    return new Func_Test_Case<FUNC>(func,
				    name,
				    timeout,
				    is_rerunable,
				    testable_at_prod,
				    testable_at_debug,
				    testable_at_audit);
}



UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif

// vim:ts=4:sw=4
