# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from os.path import join, isdir, isfile, basename
from os import makedirs, listdir, remove

from common.config import ConfigLoader
config = ConfigLoader()

class Playlists:
    """This class can create, delete and manager m3u-like playlists.
       Documentation: http://en.wikipedia.org/wiki/M3U
       This class handles playlists in unicode: m3u8"""

    def __init__(self):
        self.datadir = join(config.datadir, 'modules', 'explorer', 'playlists')
        if not isdir(self.datadir):
            makedirs(self.datadir)

    # This function returns the list of all Bluemindo's saved playlists
    def get_playlists_list(self):
        dir_list = listdir(self.datadir)

        files = []
        for filename in dir_list:
            if filename.endswith('.m3u8'):
                pretty = filename.split('.m3u8')
                files.append(pretty[0])

        return files

    # This function exports a playlist
    def export_playlist(self, playlist, new_playlist):
        songs = self.load_playlist(playlist)

        new_file = open(new_playlist, 'w')
        for song in songs:
            if isfile(song):
                new_file.write(song.encode('utf-8') + "\n")
        new_file.close()

    # This function imports a playlist and store it in the playlists folder
    def import_playlist(self, filename):
        # Get the real filename
        file_name = basename(filename)
        if file_name.endswith('.m3u'):
            file_name = file_name.split('.m3u')
            file_name = file_name[0]
        elif file_name.endswith('.m3u8'):
            file_name = file_name.split('.m3u8')
            file_name = file_name[0]

        # Create the playlist
        self.create_new_playlist(file_name)

        # Get all songs
        import_file = open(filename, 'r')
        import_songs = import_file.readlines()
        import_file.close()

        # Add the songs
        for song in import_songs:
            if isfile(song[:-1]):
                self.add_item_to_playlist(song[:-1], file_name)

        # Return the playlist name
        return file_name

    # This function creates a new playlist
    def create_new_playlist(self, name):
        playlist_file = open(join(self.datadir, str(name) + '.m3u8'), 'w')
        playlist_file.close()

    # This function deletes a playlist
    def delete_playlist(self, name):
        playlist_file = join(self.datadir, name + '.m3u8')

        if isfile(playlist_file):
            remove(playlist_file)

    # This function adds an item to a playlist
    def add_item_to_playlist(self, filename, name):
        playlist_file = open(join(self.datadir, name + '.m3u8'), 'a')
        playlist_file.write(filename.encode('utf-8') + '\n')
        playlist_file.close()

    # This function removes an item from the playlist
    def remove_item_from_playlist(self, filename, name):
        playlist_file = open(join(self.datadir, name + '.m3u8'), 'r')
        songs_list = playlist_file.readlines()
        playlist_file.close()

        songs = []
        for song in songs_list:
            if song[:-1] != filename:
                songs.append(song[:-1])

        remove(join(self.datadir, name + '.m3u8'))

        playlist_file = open(join(self.datadir, name + '.m3u8'), 'w')
        for song in songs:
            playlist_file.write(song.encode('utf-8') + '\n')
        playlist_file.close()

    # This function loads a playlist
    def load_playlist(self, name):
        playlist_file = open(join(self.datadir, name + '.m3u8'), 'r')
        songs = playlist_file.readlines()
        playlist_file.close()

        clean_songs = []
        for song in songs:
            clean_songs.append(song.replace('\n', ''))

        return clean_songs