C dftd3 program for computing the dispersion energy and forces from cartesian atomic coordinates
C and atomic numbers as described in
C
C S. Grimme, J. Antony, S. Ehrlich and H. Krieg
C A consistent and accurate ab initio parameterization of density functional dispersion correction
C (DFT-D) for the 94 elements H-Pu
C J. Chem. Phys, 132 (2010), 154104
C 
C if BJ-damping is used 
C S. Grimme, S. Ehrlich and L. Goerigk, J. Comput. Chem, 32 (2011), 1456-1465.
C
C should be cited as well.
C
C Copyright (C) 2009 - 2011 Stefan Grimme, University of Muenster, Germany
C
C This program is free software; you can redistribute it and/or modify
C it under the terms of the GNU General Public License as published by
C the Free Software Foundation; either version 1, or (at your option)
C any later version.

C This program is distributed in the hope that it will be useful,
C but WITHOUT ANY WARRANTY; without even the implied warranty of
C MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
C GNU General Public License for more details.

C For the GNU General Public License, see <http://www.gnu.org/licenses/>

 
      subroutine  nwxc_vdw3_dftd3(options,n,iz,xyz,lat,eout,g,g_lat)
      implicit none             
      character*(*) options
c number of atoms
      integer n
c cardinal numbers of elements
      integer iz(*)
c coordinates in au
      real*8 xyz(3,n)
c lattice in au
      real*8 lat(3,3)

      real*8 eout
c gradient
      real*8 g(3,n)
      real*8 g_lat(3,3)


      integer istart,iend
      integer maxat,max_elem,maxc                      
c conversion factors
      real*8 autoang,autokcal,c6conv,autoev
      parameter (maxat   =20000)
      parameter (max_elem=94)
c maximum coordination number references per element
      parameter (maxc    =5)
c coversion factors
      parameter (autoang =0.52917726d0)
      parameter (autokcal=627.509541d0)
      parameter (autoev  = 27.21138505)

c DFT-D version
      integer version
!c cardinal numbers of elements
      !integer,dimension(:), allocatable :: iz  
cc fixed atoms in geometry opt
c      logical fix(maxat)
c cut-off radii for all element pairs
      real*8 r0ab(max_elem,max_elem)
c C6 for all element pairs 
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)
c how many different C6 for one element
      integer mxc(max_elem)
c C6810 
      real*8 c6,c8,c10
c coordination numbers of the atoms
      real*8,dimension(:), allocatable :: cn  
c covalent radii
      real*8 rcov(max_elem)
c atomic <r^2>/<r^4> values
      real*8 r2r4(max_elem)
c energies
      real*8 e6, e8, e10, e12, disp, e6abc        
c THE PARAMETERS OF THE METHOD (not all a "free")
      real*8 rs6, rs8, rs10, s6, s18, alp6, alp8, alp10, s42, rs18, alp
c printout option
      logical echo
c grad ?
      logical grad
c analyse results ?
      logical anal
c third-order term?
      logical noabc
c gradient calctype
      logical numgrad
c special parameters
      logical tz
c periodic boundary conditions
      logical pbc
c repetitions of the unitcell to match the rthr and c_thr
      integer rep_vdw(3),rep_cn(3)
c R^2 distance neglect threshold (important for speed in case of large systems)
      real*8 rthr,rthr2
c R^2 distance to cutoff for CN_calculation
      real*8 cn_thr
c Integer for assigning only max/min cn C6 (0=normal, 1=min, 2=max)
c local and dummy variables
      character*80 atmp,btmp,ctmp,dtmp,etmp,ftmp,func
      character*2  esym 
      integer i,j,k,z,nn,iat,jat,i1,i2
      integer ida(max_elem),ipot
      real*8  x,y,dispr,displ,gdsp,dum,xx(10),dum6(86)
      real*8  dum1,dum2,dum3(3)
      logical ex,pot,fdum
      logical minc6list(max_elem),maxc6list(max_elem),minc6,maxc6

c PBE0/def2-QZVP atomic values 
!      data r2r4 /
!     .  8.0589,  3.4698, 29.0974, 14.8517, 11.8799,  7.8715,  5.5588,
!     .  4.7566,  3.8025,  3.1036, 26.1552, 17.2304, 17.7210, 12.7442,
!     .  9.5361,  8.1652,  6.7463,  5.6004, 29.2012, 22.3934, 19.0598,
!     . 16.8590, 15.4023, 12.5589, 13.4788, 12.2309, 11.2809, 10.5569,
!     . 10.1428,  9.4907, 13.4606, 10.8544,  8.9386,  8.1350,  7.1251,
!     .  6.1971, 30.0162, 24.4103, 20.3537, 17.4780, 13.5528, 11.8451,
!     . 11.0355, 10.1997,  9.5414,  9.0061,  8.6417,  8.9975, 14.0834,
!     . 11.8333, 10.0179,  9.3844,  8.4110,  7.5152, 32.7622, 27.5708,
!     . 23.1671, 21.6003, 20.9615, 20.4562, 20.1010, 19.7475, 19.4828,
!     . 15.6013, 19.2362, 17.4717, 17.8321, 17.4237, 17.1954, 17.1631,
!     . 14.5716, 15.8758, 13.8989, 12.4834, 11.4421, 10.2671,  8.3549,
!     .  7.8496,  7.3278,  7.4820, 13.5124, 11.6554, 10.0959,  9.7340,
!     .  8.8584,  8.0125, 29.8135, 26.3157, 19.1885, 15.8542, 16.1305,
!     . 15.6161, 15.1226, 16.1576 /                                       


c scale r4/r2 values of the atoms by sqrt(Z) 
c sqrt is also globally close to optimum
c together with the factor 1/2 this yield reasonable
c c8 for he, ne and ar. for larger Z, C8 becomes too large
c which effectively mimics higher R^n terms neglected due
c to stability reasons
      
!  r2r4 =sqrt(0.5*r2r4(i)*dfloat(i)**0.5 ) with i=elementnumber
!  the large number of digits is just to keep the results consistent
!  with older versions. They should not imply any higher accuracy than
!  the old values
      data r2r4 /
     . 2.00734898,  1.56637132,  5.01986934,  3.85379032,  3.64446594,
     . 3.10492822,  2.71175247,  2.59361680,  2.38825250,  2.21522516,
     . 6.58585536,  5.46295967,  5.65216669,  4.88284902,  4.29727576,
     . 4.04108902,  3.72932356,  3.44677275,  7.97762753,  7.07623947,
     . 6.60844053,  6.28791364,  6.07728703,  5.54643096,  5.80491167,
     . 5.58415602,  5.41374528,  5.28497229,  5.22592821,  5.09817141,
     . 6.12149689,  5.54083734,  5.06696878,  4.87005108,  4.59089647,
     . 4.31176304,  9.55461698,  8.67396077,  7.97210197,  7.43439917,
     . 6.58711862,  6.19536215,  6.01517290,  5.81623410,  5.65710424,
     . 5.52640661,  5.44263305,  5.58285373,  7.02081898,  6.46815523,
     . 5.98089120,  5.81686657,  5.53321815,  5.25477007, 11.02204549,
     .10.15679528,  9.35167836,  9.06926079,  8.97241155,  8.90092807,
     . 8.85984840,  8.81736827,  8.79317710,  7.89969626,  8.80588454,
     . 8.42439218,  8.54289262,  8.47583370,  8.45090888,  8.47339339,
     . 7.83525634,  8.20702843,  7.70559063,  7.32755997,  7.03887381,
     . 6.68978720,  6.05450052,  5.88752022,  5.70661499,  5.78450695,
     . 7.79780729,  7.26443867,  6.78151984,  6.67883169,  6.39024318,
     . 6.09527958, 11.79156076, 11.10997644,  9.51377795,  8.67197068,
     . 8.77140725,  8.65402716,  8.53923501,  8.85024712 /

c covalent radii (taken from Pyykko and Atsumi, Chem. Eur. J. 15, 2009, 188-197)
c values for metals decreased by 10 %
!      data rcov/
!     .  0.32, 0.46, 1.20, 0.94, 0.77, 0.75, 0.71, 0.63, 0.64, 0.67
!     ., 1.40, 1.25, 1.13, 1.04, 1.10, 1.02, 0.99, 0.96, 1.76, 1.54
!     ., 1.33, 1.22, 1.21, 1.10, 1.07, 1.04, 1.00, 0.99, 1.01, 1.09
!     ., 1.12, 1.09, 1.15, 1.10, 1.14, 1.17, 1.89, 1.67, 1.47, 1.39
!     ., 1.32, 1.24, 1.15, 1.13, 1.13, 1.08, 1.15, 1.23, 1.28, 1.26
!     ., 1.26, 1.23, 1.32, 1.31, 2.09, 1.76, 1.62, 1.47, 1.58, 1.57
!     ., 1.56, 1.55, 1.51, 1.52, 1.51, 1.50, 1.49, 1.49, 1.48, 1.53
!     ., 1.46, 1.37, 1.31, 1.23, 1.18, 1.16, 1.11, 1.12, 1.13, 1.32
!     ., 1.30, 1.30, 1.36, 1.31, 1.38, 1.42, 2.01, 1.81, 1.67, 1.58
!     ., 1.52, 1.53, 1.54, 1.55 /

! these new data are scaled with k2=4./3.  and converted a_0 via
! autoang=0.52917726d0
      data rcov/
     . 0.80628308, 1.15903197, 3.02356173, 2.36845659, 1.94011865,
     . 1.88972601, 1.78894056, 1.58736983, 1.61256616, 1.68815527,
     . 3.52748848, 3.14954334, 2.84718717, 2.62041997, 2.77159820,
     . 2.57002732, 2.49443835, 2.41884923, 4.43455700, 3.88023730,
     . 3.35111422, 3.07395437, 3.04875805, 2.77159820, 2.69600923,
     . 2.62041997, 2.51963467, 2.49443835, 2.54483100, 2.74640188,
     . 2.82199085, 2.74640188, 2.89757982, 2.77159820, 2.87238349,
     . 2.94797246, 4.76210950, 4.20778980, 3.70386304, 3.50229216,
     . 3.32591790, 3.12434702, 2.89757982, 2.84718717, 2.84718717,
     . 2.72120556, 2.89757982, 3.09915070, 3.22513231, 3.17473967,
     . 3.17473967, 3.09915070, 3.32591790, 3.30072128, 5.26603625,
     . 4.43455700, 4.08180818, 3.70386304, 3.98102289, 3.95582657,
     . 3.93062995, 3.90543362, 3.80464833, 3.82984466, 3.80464833,
     . 3.77945201, 3.75425569, 3.75425569, 3.72905937, 3.85504098,
     . 3.67866672, 3.45189952, 3.30072128, 3.09915070, 2.97316878,
     . 2.92277614, 2.79679452, 2.82199085, 2.84718717, 3.32591790,
     . 3.27552496, 3.27552496, 3.42670319, 3.30072128, 3.47709584,
     . 3.57788113, 5.06446567, 4.56053862, 4.20778980, 3.98102289,
     . 3.82984466, 3.85504098, 3.88023730, 3.90543362 /

c k1-k3
      !include 'param'

      real*8 k1,k2,k3

c global ad hoc parameters
      parameter (k1=16.0)
      parameter (k2=4./3.) 

c reasonable choices are between 3 and 5
c this gives smoth curves with maxima around the integer values
c k3=3 give for CN=0 a slightly smaller value than computed
c for the free atom. This also yields to larger CN for atoms
c in larger molecules but with the same chem. environment
c which is physically not right
c values >5 might lead to bumps in the potential
      parameter (k3=-4.) 

      logical  inp_strtok
      external inp_strtok



!      write(*,'(94(F12.8,'',''))')r2r4
!      stop

c scale and convert to au
!      rcov=k2*rcov/autoang
!      write(*,'(94(F11.8,'',''))')rcov
!      stop
!      do i=1,max_elem
!         dum    =0.5*r2r4(i)*dfloat(i)**0.5   
c store it as sqrt because the geom. av. is taken
!         r2r4(i)=sqrt(dum)                         
!      enddo
c init
      echo=.true. 
      grad=.false.
      pot =.false.
      anal=.false.
      noabc=.true. 
      numgrad=.false.
      tz=.false.
      func=' none (read from parameter file)'
      version=0
      pbc=.false.
      !fix=.false.
      minc6=.false.
      maxc6=.false.
      minc6list=.false.
      maxc6list=.false.
      fdum=.false.
c Cutoff r^2 thresholds for the gradient in bohr^2.
c rthr influences N^2 part of the gradient.
c rthr2 influences the N^3 part of the gradient. When using
c dftd3 in combination with semi-empirical methods or FFs, and large
c (>1000 atoms) systems, rthr2 is crucial for speed:
c Recommended values are 20^2 to 25^2 bohr.

      rthr=9000.0d0   ! UR, SE
      rthr2=1600.0d0
      cn_thr=1600.0d0

c J/mol nm^6 - > au
      c6conv=1.d-3/2625.4999d0/((0.052917726d0)**6)

c set radii
c     call rdab('~/.r0ab.dat',autoang,max_elem,r0ab)
      call setr0ab(max_elem,autoang,r0ab)

c read C6 file by default from $HOME     
c     btmp='~/.c6ab.dat'
c     inquire(file=btmp,exist=ex)
c Muenster directory as second default
c     if(.not.ex)btmp='/usr/qc/dftd3/c6ab.dat'
c     call loadc6(btmp,maxc,max_elem,c6ab,mxc)         

      

cc get coord filename
c      call getarg(1,etmp)
c      inquire(file=etmp,exist=ex)
c      if(.not.ex) call printoptions       
      ex=.false.
      ipot=0

c options
      istart = 0
      do while (inp_strtok(options,' ',istart,iend))
         ftmp = options(istart:iend)
         if(index(ftmp,'-h')      .ne.0) call printoptions
         if(index(ftmp,'-grad'   ).ne.0) grad=.true. 
         if(index(ftmp,'-anal'   ).ne.0) anal=.true. 
         if(index(ftmp,'-noprint').ne.0) echo=.false.
         if(index(ftmp,'-abc'    ).ne.0) noabc=.false.
         if(index(ftmp,'-pbc'    ).ne.0) pbc=.true.
         if(index(ftmp,'-num'    ).ne.0) numgrad=.true.
         if(index(ftmp,'-tz')     .ne.0) tz=.true.
         if(index(ftmp,'-old')    .ne.0) version=2
         if(index(ftmp,'-zero')   .ne.0) version=3
         if(index(ftmp,'-bj')     .ne.0) version=4
         if(index(ftmp,'-min')    .ne.0) then
           minc6=.true.
           j=0
           DO while (inp_strtok(options,' ',istart,iend))
             !call getarg(i+j+1,atmp)
             atmp = options(istart:iend)
  
             if (index(atmp,'-').eq.0.and.atmp.ne.'') then
               call elem(atmp,nn)
               if (nn.gt.max_elem.or.nn.lt.1)
     .           call stoprun('Could not recognize min Element')
               minc6list(nn)=.true.
               j=j+1
             else
               exit
             endif
           ENDDO
         endif
         if(index(ftmp,'-max')    .ne.0) then
           maxc6=.true.
           k=0
           DO while (inp_strtok(options,' ',istart,iend))
             !call getarg(i+k+1,atmp)
             atmp = options(istart:iend)
             if (index(atmp,'-').eq.0.and.atmp.ne.'') then
               call elem(atmp,nn)
               if (nn.gt.max_elem.or.nn.lt.1)
     .           call stoprun('Could not recognize max Element')
               maxc6list(nn)=.true.
               k=k+1
             else
               exit
             endif
           ENDDO
         endif
c         if(index(ftmp,'-pot')    .ne.0) then
c                                         pot=.true.
c                                         call getarg(i+1,atmp)
c                                         call readl(atmp,xx,nn)
c                                         ipot=idint(xx(1))
c                                         endif
c         if(index(ftmp,'-cnthr')   .ne.0) then
c                                         call getarg(i+1,atmp)
c                                         call readl(atmp,xx,nn)
c                                         rthr2=xx(1)
c                                         rthr2=rthr2**2
c                                         endif
         if(index(ftmp,'-func')  .ne.0)  then
            !call getarg(i+1,func)
            if (inp_strtok(options,' ',istart,iend)) then
               func = options(istart:iend)
            end if
            ex=.true.
         endif



c         if(index(ftmp,'-cutoff') .ne.0) then
c                                         call getarg(i+1,atmp)
c                                         call readl(atmp,xx,nn)
c                                         rthr=xx(1)**2
c                                         endif
      end do



c      do i=1,iargc()
c      call getarg(i,ftmp)
c      if(index(ftmp,'-h')      .ne.0) call printoptions
c      if(index(ftmp,'-grad'   ).ne.0) grad=.true. 
c      if(index(ftmp,'-anal'   ).ne.0) anal=.true. 
c      if(index(ftmp,'-noprint').ne.0) echo=.false.
c      if(index(ftmp,'-abc'    ).ne.0) noabc=.false.
c      if(index(ftmp,'-pbc'    ).ne.0) pbc=.true.
c      if(index(ftmp,'-num'    ).ne.0) numgrad=.true.
c      if(index(ftmp,'-tz')     .ne.0) tz=.true.
c      if(index(ftmp,'-old')    .ne.0) version=2
c      if(index(ftmp,'-zero')   .ne.0) version=3
c      if(index(ftmp,'-bj')     .ne.0) version=4
c      if(index(ftmp,'-min')    .ne.0) then
c        minc6=.true.
c        j=0
c        DO 
c          call getarg(i+j+1,atmp)
c          if (index(atmp,'-').eq.0.and.atmp.ne.'') then
c            call elem(atmp,nn)
c            if (nn.gt.max_elem.or.nn.lt.1) 
c     .        call stoprun('Could not recognize min Element')
c            minc6list(nn)=.true.
c            j=j+1
c          else
c            exit
c          endif
c        ENDDO
c      endif
c      if(index(ftmp,'-max')    .ne.0) then
c        maxc6=.true.
c        k=0
c        DO 
c          call getarg(i+k+1,atmp)
c          if (index(atmp,'-').eq.0.and.atmp.ne.'') then
c            call elem(atmp,nn)
c            if (nn.gt.max_elem.or.nn.lt.1) 
c     .        call stoprun('Could not recognize max Element')
c            maxc6list(nn)=.true.
c            k=k+1
c          else
c            exit
c          endif
c        ENDDO
c      endif
c      if(index(ftmp,'-pot')    .ne.0) then
c                                      pot=.true. 
c                                      call getarg(i+1,atmp)
c                                      call readl(atmp,xx,nn)
c                                      ipot=idint(xx(1))
c                                      endif
c      if(index(ftmp,'-cnthr')   .ne.0) then
c                                      call getarg(i+1,atmp)
c                                      call readl(atmp,xx,nn)
c                                      rthr2=xx(1)
c                                      rthr2=rthr2**2
c                                      endif
c      if(index(ftmp,'-func')  .ne.0)  then
c                                      call getarg(i+1,func)
c                                      ex=.true.
c                                      endif
c       
c
c
c      if(index(ftmp,'-cutoff') .ne.0) then
c                                      call getarg(i+1,atmp)
c                                      call readl(atmp,xx,nn)
c                                      rthr=xx(1)**2
c                                      endif
cc      if(index(ftmp,'-pot')    .ne.0) then
c      enddo
c

c     Check command line input


      if(minc6.and.j.lt.1)then
       call stoprun('No Element given for min/max')
      endif
      if(maxc6.and.k.lt.1)then
       call stoprun('No Element given for min/max')
      endif
      do i=1,max_elem

        if (minc6list(i).and.maxc6list(i)) 
     .    call stoprun('Unreasonable min/max input!')
!      if (minc6list(i)) write(*,*)'min:',i
!      if (maxc6list(i)) write(*,*)'max:',i
      enddo
c C6 hard-coded (c6ab.dat not used)
c this is alternative to loadc6
      call copyc6(btmp,maxc,max_elem,c6ab,mxc,
     .            minc6,minc6list,maxc6,maxc6list)   
       cn_thr=rthr2

!       write(*,*)'CN(P):',c6ab(15,15,mxc(15),1,2)
!       write(*,*)'mxc(P):',mxc(15)

c       if (pbc) then
c         call pbcrdatomnumber(etmp,n)
c       else
c         call rdatomnumber(etmp,n)
c       endif

!      allocations
      !allocate(xyz(3,n))
      !allocate(g(3,n))
      !allocate(iz(n))
      allocate(cn(n))

c reading coordinates and cell in VASP.5.2-format
c determing repetitions of unitcell
      if (pbc) then
              !call pbcrdcoord(etmp,lat,n,xyz,iz,autoang)
              call set_criteria(rthr,lat,dum3)
              rep_vdw=int(dum3)+1
              call set_criteria(cn_thr,lat,dum3)
              rep_cn=int(dum3)+1
c              write(*,*)'VDW-cutoff:',sqrt(rthr)*autoang
c              write(*,*)'CN-cutoff :',sqrt(cn_thr)*autoang
c              write(*,*)'repvdw:',rep_vdw
c              write(*,*)'repcn :',rep_cn
      else !no pbc
c read coordinates, either TM or xmol file
              !call rdcoord(etmp,n,xyz,iz,fix,fdum)
      endif !pbc
      if(n.lt.1)     call stoprun( 'no atoms' )
      if(n.gt.maxat) call stoprun( 'too many atoms' )


     
c the analytical E(3) grad is not available yet
      if(grad.and.(.not.noabc))numgrad=.true.
     
c set parameters for functionals
      if(ex) then
         call setfuncpar(func,version,tz,s6,rs6,s18,rs18,alp)
      else
         call rdpar     (dtmp,version,s6,s18,rs6,rs18,alp)
      endif

      if(echo)then
      write(*,*)' _________________________________'
      write(*,*)'                                  '
      write(*,*)'|         DFTD3 V3.0 Rev 1        |'
      write(*,*)'| S.Grimme, University Bonn       |'
      write(*,*)'|           May  2013             |'
      write(*,*)'|   see dftd3 -h for options      |'
      write(*,*)' _________________________________'
      write(*,*)
      write(*,*)'Please cite DFT-D3 work done with this code as:'
      write(*,*)'S. Grimme, J. Antony, S. Ehrlich and H. Krieg,'
      write(*,*)'J. Chem. Phys. 132 (2010), 154104'
      write(*,*)'If used with BJ-damping cite also'
      write(*,*)'S. Grimme, S. Ehrlich and L. Goerigk,'
      write(*,*)'J. Comput. Chem. 32 (2011), 1456-1465'
      write(*,*)'For DFT-D2 the reference is'
      write(*,*)'S. Grimme, J. Comput. Chem., 27 (2006), 1787-1799'
      write(*,*)
      write(*,*)' files read :     '
      !write(*,*)trim(etmp)       
      if(.not.ex)write(*,*)trim(dtmp)       
      endif

      if(version.lt.2.or.version.gt.4)stop 'inacceptable version number'

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C all calculations start here
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

c CNs for output
      if (pbc) then
          call pbcncoord(n,rcov,iz,xyz,cn,lat,rep_cn,cn_thr)
      else
          call ncoord(n,rcov,iz,xyz,cn,cn_thr)
      endif

      if(version.eq.2)then
          if(echo)write(*,'(''loading DFT-D2 parameters ...'')')
          call loadoldpar(autoang,max_elem,maxc,c6ab,r0ab,dum6)
c number of CNs for each element
          mxc=1
convert to au
          c6ab=c6ab*c6conv
      endif


c which atoms are present? (for printout)
      if(echo)then
      ida=0
      do i=1,n
         ida(iz(i))=ida(iz(i))+1
      enddo
      write(*,'(''C6 coefficients used:'')')
      do i=1,94
         if(ida(i).gt.0)then
            write(*,*) mxc(i),' C6 for element ',i
            do j=1,maxc
               if(c6ab(i,i,j,j,1).gt.0)then
               write(*,'(''Z='',i3,'' CN='',F6.3,5x,''C6(AA)='',F8.2)')
     .         i,c6ab(i,i,j,j,2),c6ab(i,i,j,j,1)
               endif
            enddo
         endif
      enddo
      endif
      
c output
      if (echo) then
          write(*,'(/''#               XYZ [au]  '',12x,
     .              '' R0(AA) [Ang.]''2x,
     .              ''CN'',7x,
     .              ''C6(AA)     C8(AA)   C10(AA) [au] '')
     .            ')
          x=0
          btmp=''
          do i=1,n
          z=iz(i)
          call getc6(maxc,max_elem,c6ab,mxc,iz(i),iz(i),cn(i),cn(i),c6)
          do j=1,n
          call getc6(maxc,max_elem,c6ab,mxc,iz(i),iz(j),cn(i),cn(j),dum)
          x=x+dum
          enddo
c compute C8/C10 for output
          c8 =r2r4(iz(i))**2*3.0d0*c6     
          c10=(49.0d0/40.0d0)*c8**2/c6
          dum=0.5*autoang*r0ab(z,z)
          if(version.eq.4)dum=rs6*0.5*autoang*sqrt(c8/c6)
          atmp=' '
          !if(fix(i)) then
          ! atmp='f'
          ! btmp='f'
          !endif
          write(*,'(i4,3F10.5,3x,a2,1x,a1,F7.3,2x,F7.3,3F12.1,L)')
     .    i,xyz(1:3,i),esym(z),atmp,
     .    dum,cn(i),
     .    c6,c8,c10
          enddo
          write(*,'(/''molecular C6(AA) [au] = '',F12.2)')x
          if(btmp.eq.'f') then
          write(*,*)'  '
           write(*,*)'Caution: Some coordinates fixed 
     .in gradient (marked f, see above).'
          write(*,*)'  '
          endif
          if(fdum)then
          write(*,*)'Caution: Dummy atoms found and ignored.'
          endif
   
      endif

      
c testoutput of radii
c     do i=1,94
c        call getc6(maxc,max_elem,c6ab,mxc,i,i,0.d0,0.0d0,c6)
c        c8 =r2r4(i)**2*3.0d0*c6     
c        write(22,*) i, sqrt(c8/c6)
c     enddo
c     write(22,*)
c     do i=1,94
c        write(22,*) i, r0ab(i,i)  
c     enddo
c     stop

c for global ad hoc parameters see
c k3 in subroutine getc6, k1 and k2 in subroutine ncoord*
c fixed or dependent ones:
      rs8  = rs18       
      rs10 = rs18
      alp6 = alp
      alp8 = alp+2.
      alp10= alp8+2. 
c note: if version=4 (Becke-Johnson), a1=rs6 and a2=rs18
c       and alp* have no meaning

c*********************************************************************
c*********************************************************************
c testing code
c output of C6=f(CN)
      if(pot.and.ipot.gt.100)then
      x=0
      do i=1,100
      call getc6(maxc,max_elem,c6ab,mxc,ipot-100,ipot-100,
     .                              x,x,C6)
      write(2,*) x,c6
      x=x+0.05
      enddo
      stop
      endif
c Edisp pot curve for testing. Caution: C6 is not constant along R!
      if(pot)then
      write(*,*) 'Computing Edisp potential curve for atom ',ipot
      xyz=0
      iz(1)=ipot
      iz(2)=ipot
      n=2
      xyz(3,2)=1.0/autoang
 142  if (pbc) then
       call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_vdw,cn_thr,rep_cn)
      else
        call edisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .     rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,rthr,cn_thr,
     .     e6,e8,e10,e12,e6abc)
      endif
      xyz(3,2)=xyz(3,2)+0.02
      disp=-s6*e6-s18*e8
      write(42,*) xyz(3,2)*autoang,disp*autokcal
      write(43,*) xyz(3,2)        ,disp*autokcal
      if (pbc) then
          call pbcncoord(n,rcov,iz,xyz,cn,lat,rep_cn,cn_thr)
      else
          call ncoord(n,rcov,iz,xyz,cn,cn_thr)
      endif
      call getc6(maxc,max_elem,c6ab,mxc,iz(1),iz(2),cn(1),cn(2),c6)
      write(2,*)xyz(3,2)*autoang,-autokcal*c6/xyz(3,2)**6
      if(xyz(3,2).lt.20) goto 142
      write(42,*)
      stop 'pot curve done'
      endif
c end testing code
c*********************************************************************
c*********************************************************************

c check if all parameters have been loaded and are resonable
      do iat=1,n-1
         do jat=iat+1,n
            if(r0ab(iz(jat),iz(iat)).lt.0.1) then
               write(*,*) iat,jat,iz(jat),iz(iat)
               call stoprun( 'radius missing' )
            endif
            if (version.eq.2)then
              c6=c6ab(iz(jat),iz(iat),1,1,1)
            else 
              call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                      cn(iat),cn(jat),c6)
            endif
            if(c6.lt.1.d-6) then
               write(*,*) iat,jat,iz(jat),iz(iat),cn(iat),cn(jat)
               call stoprun( 'C6 missing' )
            endif
         enddo
      enddo

c sanity check of read coordniates, based on covalnent radii. 
c Not omnipotent but better than nothing. S.E. 15.09.2011
c      call checkcn(n,iz,cn,c6ab,max_elem,maxc)
      if (pbc) then
        call pbccheckrcov(n,iz,rcov,xyz,lat)
      else
        call checkrcov(n,iz,rcov,xyz)
      endif

cccccccccccccc
c energy call
cccccccccccccc
      if (pbc) then
        call  pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .     rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .     e6,e8,e10,e12,e6abc,lat,rthr,rep_vdw,cn_thr,rep_cn)
 
      else
        call edisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .     rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,rthr,cn_thr,
     .     e6,e8,e10,e12,e6abc)
      endif

      e6   = e6   *s6

      e6abc= e6abc*s6

      e8   = e8   *s18

      disp =-e6-e8-e6abc

c e10 has been tested once again with BJ-damping but has no good effect
c     e10  = e10  *s18
c     disp =-e6-e8-e10-e6abc

      eout = disp
c output
      if (echo) then
      if(version.lt.4)then
      write(*,'(/10x,'' DFT-D V'',i1)') version       
      else
      write(*,'(/10x,'' DFT-D V3(BJ)'')') 
      endif
      write(*,'('' DF '',a50)') func          
      write(*,'('' parameters'')') 
      if(version.eq.2)then
         write(*,'('' s6       :'',f10.4)') s6            
         write(*,'('' alpha6   :'',f10.4)') alp6        
      endif
      if(version.eq.3)then
         write(*,'('' s6       :'',f10.4)') s6            
         write(*,'('' s8       :'',f10.4)') s18           
         write(*,'('' rs6      :'',f10.4)') rs6  
         write(*,'('' rs18     :'',f10.4)') rs18          
         write(*,'('' alpha6   :'',f10.4)') alp6        
         write(*,'('' alpha8   :'',f10.4)') alp8           
         write(*,'('' k1-k3    :'',3f10.4)') k1,k2,k3     
      endif
      if(version.eq.4)then
         write(*,'('' s6       :'',f10.4)') s6            
         write(*,'('' s8       :'',f10.4)') s18           
         write(*,'('' a1       :'',f10.4)') rs6           
         write(*,'('' a2       :'',f10.4)') rs18          
         write(*,'('' k1-k3    :'',3f10.4)') k1,k2,k3     
      endif
        write(*,'('' Cutoff   :'',f10.4,'' a.u.'')') sqrt(rthr) !*autoang
        write(*,'('' CN-Cutoff:'',f10.4,'' a.u.'')')sqrt(cn_thr)!*autoang
      if (pbc) then
        write(*,'('' Rep_vdw  :'',3I3)') rep_vdw
      endif
      write(*,*)
      if (pbc) then
      write(*,'('' Edisp /kcal,au,eV:'',f11.4,X,f12.8,X,f11.7)') 
     .  disp*autokcal,disp,disp*autoev
      else
        write(*,'('' Edisp /kcal,au:'',f11.4,f12.8)') disp*autokcal,disp
      endif
      write(*,'(/'' E6    /kcal :'',f11.4)')-e6*autokcal
      if(version.gt.2)then
      write(*,'('' E8    /kcal :'',f11.4)')-e8*autokcal 
c     write(*,'('' E10   /kcal :'',f11.4)')-e10*autokcal 
      if(.not.noabc)
     .write(*,'('' E6(ABC) "   :'',2f11.6,F16.12)')-e6abc*autokcal 
      write(*,'('' % E8        :'',f6.2)') -e8/disp/0.01         
      if(.not.noabc)
     .write(*,'('' % E6(ABC)   :'',f6.2)') -e6abc/disp/0.01        
      endif
      endif
c this file for tmer2 read tool
      open(unit=1,file='.EDISP')
      write(1,*) disp        
      close(1)

cccccccccccccccccccccccccc
c analyse Edisp pair terms
cccccccccccccccccccccccccc
      if(anal) then
       if (pbc) then

        call pbcadisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .           rs6,rs8,rs10,alp6,alp8,alp10,version,autokcal,autoang,
     .           rthr,rep_vdw,cn_thr,rep_cn,s6,s18,disp*autokcal,lat)
       else
        call adisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .           rs6,rs8,rs10,alp6,alp8,alp10,version,autokcal,
     .           autoang,rthr,cn_thr,s6,s18,disp*autokcal)
       endif !pbc
      endif !anal
 
cccccccccccccccccccccccccc
c gradient
cccccccccccccccccccccccccc
      if(grad)then
      g=0.0d0
      call cpu_time(dum1)
      if (pbc) then
      call pbcgdisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,s6,s18,rs6,rs8,rs10,alp6,alp8,alp10,noabc,numgrad,
     .                 version,g,gdsp,x,g_lat,lat,rep_vdw,rep_cn,
     .                 rthr,echo,cn_thr)
 
      else
c        call gdisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
c     .           s6,s18,rs6,rs8,rs10,alp6,alp8,alp10,noabc,rthr,
c     .           numgrad,version,echo,g,gdsp,x,rthr2,fix)
        call gdisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .           s6,s18,rs6,rs8,rs10,alp6,alp8,alp10,noabc,rthr,
     .           numgrad,version,echo,g,gdsp,x,rthr2)
      endif
      call cpu_time(dum2)
!      echo=.true. !Jonas
      if(echo)write(*,'('' time  '',f6.1)')dum2-dum1
!       if (pbc) call stresstensor(maxc,max_elem,autoev,s6,s18,xyz,n,iz,
!     .     lat,c6ab,mxc,version,numgrad,noabc,echo,r0ab,r2r4,
!     .     rcov,rs6,rs8,rs10,alp6,alp8,alp10,rthr,rep_vdw,cn_thr,rep_cn,
!     .     g_lat)
      if (echo.and.pbc) then
        write(*,*)'Stresstensor:'
        write(*,*)g_lat


      endif

 
c check if gdisp yields same energy as edisp
      if(abs((disp-gdsp)/disp).gt.1.d-8) then
         write(*,*) disp,gdsp
         call stoprun('internal error')
      endif !sanitycheck
c write to energy and gradient files in TM style
      if (pbc) then
        if (echo) then
          write(*,*)'Cartesian gradient written to file dftd3_gradient.'
          write(*,*)'Cartesian cellgradient written 
     . to file dftd3_cellgradient. (a.u.)'
          g_lat=g_lat!*autoev
          call pbcwregrad(n,g,g_lat)
        endif
      else
        if (echo) then
        write(*,*) 'Cartesian gradient written to file dftd3_gradient'
        call outg(n,g)
        end if

      endif
      endif !grad
      !deallocate(xyz,g,iz,cn)
      deallocate(cn)

      if(echo)write(*,*) 'normal termination of dftd3'

      return
      end subroutine

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine printoptions
      write(*,*) 'dftd3 <coord filename> [-options]'
      write(*,*) 'options:'
      write(*,*) '-func <functional name in TM style>'
      write(*,*) '-grad'
      write(*,*) '-anal (pair analysis)'
      write(*,*) '     file <fragemt> with atom numbers'
      write(*,*) '     is read for a fragement based   '
      write(*,*) '     analysis (one fragment per line,'
      write(*,*) '     atom ranges (e.g. 1-14 17-20) are allowed)'
      write(*,*) '-noprint'
      write(*,*) '-pbc (periodic boundaries; reads VASP-format)'
      write(*,*) '-abc (compute E(3))'
      write(*,*) '-cnthr (neglect threshold in Bohr for CN, default=40)'
      write(*,*) '-cutoff (neglect threshold in Bohr for E_disp, 
     . default=95)'
      write(*,*) '-old (DFT-D2)'         
      write(*,*) '-zero (DFT-D3 original zero-damping)' 
      write(*,*) '-bj   (DFT-D3 with Becke-Johnson finite-damping)' 
      write(*,*) '-tz (use special parameters for TZ-type calculations)'
      write(*,*) 'variable parameters read from ~/.dftd3par.<hostname>'
      write(*,*) 'if -func is used, -zero or -bj or -old is required!"'
      stop
      end subroutine printoptions

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C set parameters
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine setfuncpar(func,version,TZ,s6,rs6,s18,rs18,alp)
      implicit none  
      integer version
      real*8 s6,rs6,s18,alp,rs18
      character*(*) func     
      logical TZ
c double hybrid values revised according to procedure in the GMTKN30 paper

c DFT-D3 with Becke-Johnson finite-damping, variant 2 with their radii 
c SE: Alp is only used in 3-body calculations
      if(version.eq.4)then
      s6=1.0d0
      alp =14.0d0

      select case (func)
         case ("b-p")
              rs6 =0.3946
              s18 =3.2822
              rs18=4.8516
         case ("b-lyp")
              rs6 =0.4298
              s18 =2.6996
              rs18=4.2359
         case ("revpbe")
              rs6 =0.5238
              s18 =2.3550
              rs18=3.5016
         case ("rpbe")
              rs6 =0.8318
              s18 =0.1820
              rs18=4.0094
         case ("b97-d")
              rs6 =0.5545
              s18 =2.2609
              rs18=3.2297
         case ("pbe")
              rs6 =0.4289
              s18 =0.7875
              rs18=4.4407
         case ("rpw86-pbe")
              rs6 =0.4613
              s18 =1.3845
              rs18=4.5062
         case ("b3-lyp")
              rs6 =0.3981
              s18 =1.9889
              rs18=4.4211
         case ("tpss")
              rs6 =0.4535
              s18 =1.9435
              rs18=4.4752
         case ("hf")
              rs6 =0.3385
              s18 =0.9171
              rs18=2.8830
         case ("tpss0")
              rs6 =0.3768
              s18 =1.2576
              rs18=4.5865
         case ("pbe0")
              rs6 =0.4145
              s18 =1.2177
              rs18=4.8593
         case ("hse06")
              rs6 =0.383
              s18 =2.310
              rs18=5.685
         case ("revpbe38")
              rs6 =0.4309
              s18 =1.4760
              rs18=3.9446
         case ("pw6b95")
              rs6 =0.2076
              s18 =0.7257
              rs18=6.3750
         case ("b2-plyp")
              rs6 =0.3065
              s18 =0.9147
              rs18=5.0570
              s6=0.64d0
         case ("dsd-blyp")
              rs6 =0.0000
              s18 =0.2130
              rs18=6.0519
              s6=0.50d0
         case ("dsd-blyp-fc")
              rs6 =0.0009
              s18 =0.2112
              rs18=5.9807
              s6=0.50d0
         case ("bop")
              rs6 =0.4870
              s18 =3.2950
              rs18=3.5043
         case ("mpwlyp")
              rs6 =0.4831
              s18 =2.0077
              rs18=4.5323
         case ("o-lyp")
              rs6 =0.5299
              s18 =2.6205
              rs18=2.8065
         case ("pbesol")
              rs6 =0.4466
              s18 =2.9491
              rs18=6.1742
         case ("bpbe")
              rs6 =0.4567
              s18 =4.0728
              rs18=4.3908
         case ("opbe")
              rs6 =0.5512
              s18 =3.3816
              rs18=2.9444
         case ("ssb")
              rs6 =-0.0952
              s18 =-0.1744
              rs18=5.2170
         case ("revssb")
              rs6 =0.4720
              s18 =0.4389
              rs18=4.0986
         case ("otpss")
              rs6 =0.4634
              s18 =2.7495
              rs18=4.3153
        case ("b3pw91")
              rs6 =0.4312
              s18 =2.8524
              rs18=4.4693
         case ("bh-lyp")
              rs6 =0.2793
              s18 =1.0354
              rs18=4.9615
         case ("revpbe0")
              rs6 =0.4679
              s18 =1.7588
              rs18=3.7619
         case ("tpssh")
              rs6 =0.4529
              s18 =2.2382
              rs18=4.6550
         case ("mpw1b95")
              rs6 =0.1955
              s18 =1.0508
              rs18=6.4177
         case ("pwb6k")
              rs6 =0.1805
              s18 =0.9383
              rs18=7.7627
         case ("b1b95")
              rs6 =0.2092
              s18 =1.4507
              rs18=5.5545
         case ("bmk")
              rs6 =0.1940
              s18 =2.0860
              rs18=5.9197
         case ("cam-b3lyp")
              rs6 =0.3708
              s18 =2.0674
              rs18=5.4743
         case ("lc-wpbe")
              rs6 =0.3919
              s18 =1.8541
              rs18=5.0897
         case ("b2gp-plyp")
              rs6 =0.0000
              s18 =0.2597
              rs18=6.3332
                s6=0.560
         case ("ptpss")
              rs6 =0.0000
              s18 =0.2804
              rs18=6.5745
                s6=0.750
         case ("pwpb95")
              rs6 =0.0000
              s18 =0.2904
              rs18=7.3141
                s6=0.820
c special HF/DFT with eBSSE correction
         case ("hf/mixed")
              rs6 =0.5607  
              s18 =3.9027  
              rs18=4.5622  
         case ("hf/sv")
              rs6 =0.4249  
              s18 =2.1849  
              rs18=4.2783   
         case ("hf/minis")
              rs6 =0.1702  
              s18 =0.9841   
              rs18=3.8506  
         case ("b3-lyp/6-31gd")
              rs6 =0.5014  
              s18 =4.0672   
              rs18=4.8409
         case ("hcth120")
              rs6=0.3563
              s18=1.0821
              rs18=4.3359
c DFTB3 (zeta=4.0)   
         case ("dftb3")
              rs6=0.7461
              s18=3.209 
              rs18=4.1906
         case DEFAULT
              call stoprun( 'functional name unknown' )
      end select
      endif

c DFT-D3
      if(version.eq.3)then
      s6  =1.0d0
      alp =14.0d0
      rs18=1.0d0
c default def2-QZVP (almost basis set limit)
      if(.not.TZ) then
      select case (func)
         case ("slater-dirac-exchange")
              rs6 =0.999
              s18 =-1.957
              rs18=0.697
         case ("b-lyp")
              rs6=1.094
              s18=1.682
         case ("b-p")
              rs6=1.139
              s18=1.683
         case ("b97-d")
              rs6=0.892
              s18=0.909
         case ("revpbe")
              rs6=0.923
              s18=1.010
         case ("pbe")
              rs6=1.217
              s18=0.722
         case ("pbesol")
              rs6=1.345
              s18=0.612
         case ("rpw86-pbe")
              rs6=1.224
              s18=0.901
         case ("rpbe")
              rs6=0.872
              s18=0.514
         case ("tpss")
              rs6=1.166
              s18=1.105
         case ("b3-lyp")
              rs6=1.261
              s18=1.703
         case ("pbe0")
              rs6=1.287
              s18=0.928

         case ("hse06")
              rs6=1.129
              s18=0.109
         case ("revpbe38")
              rs6=1.021
              s18=0.862
         case ("pw6b95")
              rs6=1.532
              s18=0.862
         case ("tpss0")
              rs6=1.252
              s18=1.242
         case ("b2-plyp")
              rs6=1.427
              s18=1.022
              s6=0.64
         case ("pwpb95")
              rs6=1.557
              s18=0.705
              s6=0.82
         case ("b2gp-plyp")
              rs6=1.586
              s18=0.760
              s6=0.56
         case ("ptpss")
              rs6=1.541
              s18=0.879
              s6=0.75
         case ("hf")
              rs6=1.158
              s18=1.746
         case ("mpwlyp")
              rs6=1.239
              s18=1.098
         case ("bpbe")
              rs6=1.087
              s18=2.033
         case ("bh-lyp")
              rs6=1.370
              s18=1.442
         case ("tpssh")
              rs6=1.223
              s18=1.219
         case ("pwb6k")
              rs6=1.660
              s18=0.550
         case ("b1b95")
              rs6=1.613
              s18=1.868
         case ("bop")
              rs6=0.929
              s18=1.975
         case ("o-lyp")
              rs6=0.806
              s18=1.764
         case ("o-pbe")
              rs6=0.837
              s18=2.055
         case ("ssb")
              rs6=1.215
              s18=0.663
         case ("revssb")
              rs6=1.221
              s18=0.560
         case ("otpss")
              rs6=1.128
              s18=1.494
         case ("b3pw91")
              rs6=1.176
              s18=1.775
         case ("revpbe0")
              rs6=0.949
              s18=0.792
         case ("pbe38")
              rs6=1.333
              s18=0.998
         case ("mpw1b95")
              rs6=1.605
              s18=1.118
         case ("mpwb1k")
              rs6=1.671
              s18=1.061
         case ("bmk")
              rs6=1.931
              s18=2.168
         case ("cam-b3lyp")
              rs6=1.378
              s18=1.217
         case ("lc-wpbe")
              rs6=1.355
              s18=1.279
         case ("m05")
              rs6=1.373
              s18=0.595
         case ("m052x")
              rs6=1.417
              s18=0.000
         case ("m06l")
              rs6=1.581
              s18=0.000
         case ("m06")
              rs6=1.325
              s18=0.000
         case ("m062x")
              rs6=1.619
              s18=0.000
         case ("m06hf")
              rs6=1.446
              s18=0.000
c DFTB3 (zeta=4.0)   
         case ("dftb3")
              rs6=1.235
              s18=0.673
         case ("hcth120")
              rs6=1.221
              s18=1.206
         case DEFAULT
              call stoprun( 'functional name unknown' )
      end select
      else
c special TZVPP parameter
      select case (func)
         case ("b-lyp")
              rs6=1.243
              s18=2.022
         case ("b-p")
              rs6=1.221
              s18=1.838
         case ("b97-d")
              rs6=0.921
              s18=0.894
         case ("revpbe")
              rs6=0.953
              s18=0.989
         case ("pbe")
              rs6=1.277
              s18=0.777
         case ("tpss")
              rs6=1.213
              s18=1.176
         case ("b3-lyp")
              rs6=1.314
              s18=1.706
         case ("pbe0")
              rs6=1.328
              s18=0.926
         case ("pw6b95")
              rs6=1.562
              s18=0.821
         case ("tpss0")
              rs6=1.282
              s18=1.250
         case ("b2-plyp")
              rs6=1.551
              s18=1.109
              s6=0.5
         case DEFAULT
              call stoprun( 'functional name unknown (TZ case)' )
      end select
      endif
      endif
c DFT-D2
      if(version.eq.2)then
      rs6=1.1d0
      s18=0.0d0
      alp=20.0d0
      select case (func)
         case ("b-lyp")
              s6=1.2  
         case ("b-p")
              s6=1.05  
         case ("b97-d")
              s6=1.25 
         case ("revpbe")
              s6=1.25 
         case ("pbe")
              s6=0.75 
         case ("tpss")
              s6=1.0  
         case ("b3-lyp")
              s6=1.05 
         case ("pbe0")
              s6=0.6   
         case ("pw6b95")
              s6=0.5   
         case ("tpss0")
              s6=0.85  
         case ("b2-plyp")
              s6=0.55 
         case ("b2gp-plyp")
              s6=0.4
         case ("dsd-blyp")
              s6=0.41
              alp=60.0d0
         case DEFAULT
              call stoprun( 'functional name unknown' )
      end select

      endif

      end subroutine setfuncpar

      subroutine rdpar(dtmp,version,s6,s18,rs6,rs18,alp)
      implicit none
      real*8 s6,s18,rs6,rs18,alp
      integer version
      character*(*) dtmp
      character*80  ftmp
      logical ex
      real*8 xx(10)
      integer nn
c read parameter file
      call system('hostname > .tmpx')
      open(unit=43,file='.tmpx')
      read(43,'(a)')ftmp
      close(43,status='delete')
      write(dtmp,'(''~/.dftd3par.'',a)')trim(ftmp)
      inquire(file=dtmp,exist=ex)
      s6 =0
      s18=0
      rs6=0
      rs18=0
      alp =0
      if(ex)then
         open(unit=42,file=dtmp)
         read(42,'(a)',end=9)ftmp 
         call readl(ftmp,xx,nn)
         if(nn.eq.6) then
                     s6  =xx(1)
                     rs6 =xx(2)
                     s18 =xx(3)
                     rs18=xx(4)
                     alp =xx(5)
            version=idint(xx(6))
         endif
  9      close(42)
      endif

      end subroutine rdpar

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C compute energy
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine edisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .           rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,rthr,cn_thr,
     .           e6,e8,e10,e12,e63)
      implicit none  
      integer n,iz(*),max_elem,maxc,version,mxc(max_elem) 
      real*8 xyz(3,*),r0ab(max_elem,max_elem),r2r4(*)
      real*8 rs6,rs8,rs10,alp6,alp8,alp10,rcov(max_elem)
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)
      real*8 e6, e8, e10, e12, e63        
      logical noabc
 
      integer iat,jat,kat
      real*8 r,r2,r6,r8,tmp,alp,dx,dy,dz,c6,c8,c10,ang,rav
      real*8 damp6,damp8,damp10,rr,thr,c9,r42,c12,r10,c14,rthr,cn_thr
      real*8 cn(n)                             
      real*8 r2ab(n*n),cc6ab(n*n),dmp(n*n),d2(3),t1,t2,t3,a1,a2
      integer*2 icomp(n*n)
      integer ij,ik,jk
      integer,external :: lin

      e6 =0
      e8 =0
      e10=0
      e12=0
      e63=0

c     Becke-Johnson parameters
      a1=rs6
      a2=rs8

C DFT-D2
      if(version.eq.2)then

      do iat=1,n-1
         do jat=iat+1,n
            dx=xyz(1,iat)-xyz(1,jat)
            dy=xyz(2,iat)-xyz(2,jat)
            dz=xyz(3,iat)-xyz(3,jat)
            r2=dx*dx+dy*dy+dz*dz
c           if(r2.gt.rthr) cycle
            r=sqrt(r2)
            c6=c6ab(iz(jat),iz(iat),1,1,1)
            damp6=1./(1.+exp(-alp6*(r/(rs6*r0ab(iz(jat),iz(iat)))-1.)))
            r6=r2**3      
            e6 =e6+c6*damp6/r6
         enddo
      enddo

      else
C DFT-D3
      call ncoord(n,rcov,iz,xyz,cn,cn_thr)

      icomp=0
      do iat=1,n-1
         do jat=iat+1,n
            dx=xyz(1,iat)-xyz(1,jat)
            dy=xyz(2,iat)-xyz(2,jat)
            dz=xyz(3,iat)-xyz(3,jat)
            r2=dx*dx+dy*dy+dz*dz
CTHR        
            if(r2.gt.rthr) cycle
            r =sqrt(r2)
            rr=r0ab(iz(jat),iz(iat))/r
c damping
            tmp=rs6*rr   
            damp6 =1.d0/( 1.d0+6.d0*tmp**alp6 )
            tmp=rs8*rr     
            damp8 =1.d0/( 1.d0+6.d0*tmp**alp8 )
c get C6
            call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                    cn(iat),cn(jat),c6)
            if(.not.noabc)then
            ij=lin(jat,iat)
            icomp(ij)=1
c store C6 for C9, calc as sqrt
            cc6ab(ij)=sqrt(c6)
c store R^2 for abc
            r2ab(ij)=r2
c store for abc damping
            dmp(ij)=(1./rr)**(1./3.) 
            endif

            r6=r2**3      
            r8=r6*r2
c r2r4 stored in main as sqrt
            c8 =3.0d0*c6*r2r4(iz(iat))*r2r4(iz(jat))

C DFT-D3 zero-damp
            if(version.eq.3)then
               e6=e6+c6*damp6/r6
               e8=e8+c8*damp8/r8
            endif
C DFT-D3(BJ)          
            if(version.eq.4)then
c use BJ radius
               tmp=sqrt(c8/c6)              
               e6=e6+  c6/(r6+(a1*tmp+a2)**6)
               e8=e8+  c8/(r8+(a1*tmp+a2)**8)
c              c10=(49.0d0/40.0d0)*c8**2/c6
c             e10=e10+c10/(r8*r2+(a1*tmp+a2)**10)
            endif
         enddo
      enddo

      if(noabc)return

C compute non-additive third-order energy using averaged C6
      do iat=1,n-1
         do jat=iat+1,n
            ij=lin(jat,iat)
            if(icomp(ij).eq.1)then
            do kat=jat+1,n
            ik=lin(kat,iat)
            jk=lin(kat,jat)
            if(icomp(ik).eq.0.or.icomp(jk).eq.0) cycle
c damping func product
c           tmp=dmp(ik)*dmp(jk)*dmp(ij)
            rav=(4./3.)/(dmp(ik)*dmp(jk)*dmp(ij))
            tmp=1.d0/( 1.d0+6.d0*rav**alp8 )
c triple C6 coefficient (stored as sqrt)
            c9=cc6ab(ij)*cc6ab(ik)*cc6ab(jk)
c           write(*,*) 'C9', c9
c angular terms
c d is r^2
            d2(1)=r2ab(ij)
            d2(2)=r2ab(jk)
            d2(3)=r2ab(ik)
            t1 = (d2(1)+d2(2)-d2(3))/sqrt(d2(1)*d2(2))
            t2 = (d2(1)+d2(3)-d2(2))/sqrt(d2(1)*d2(3))
            t3 = (d2(3)+d2(2)-d2(1))/sqrt(d2(2)*d2(3))
            ang=0.375d0*t1*t2*t3+1.0d0

c C9 has negative sign
            e63=e63-tmp*c9*ang/(d2(1)*d2(2)*d2(3))**1.50d0

            enddo
            endif
         enddo
      enddo

      endif

      end subroutine edisp

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C analyse all pairs
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine adisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .                rs6,rs8,rs10,alp6,alp8,alp10,version,autokcal,
     .                autoang,rthr,cn_thr,s6,s18,etot)
      implicit none  
      integer n,iz(*),max_elem,maxc,version,mxc(max_elem) 
      real*8 xyz(3,*),r0ab(max_elem,max_elem),r2r4(*),s6
      real*8 rs6,rs8,rs10,alp6,alp8,alp10,autokcal,etot,s18,autoang
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3),rcov(max_elem)
 
      integer iat,jat,i,j,k,nbin
      real*8 R0,r,r2,r6,r8,tmp,alp,dx,dy,dz,c6,c8,c10
      real*8 damp6,damp8,damp10,r42,rr,check,rthr,cn_thr,rvdw
      real*8 cn(n),i6,e6,e8,e10,edisp                   
      real*8 dist(0:15),li(0:15,2)
      real*8 xx(500),eg(10000)
      integer grplist(500,20)
      integer grpn(20),at(n)
      integer ngrp,dash
      integer lin, iiii, jjjj, iii, jjj, ii, jj, ni, nj 
      integer iout(500)
      logical ex
      character*80 atmp
 
      real*8,dimension(:,:), allocatable :: ed
      allocate(ed(n,n))


c distance bins
      li(0,1)=0   
      li(0,2)=1.5 
      li(1,1)=1.5
      li(1,2)=2
      li(2,1)=2
      li(2,2)=2.3333333333
      li(3,1)=2.3333333333
      li(3,2)=2.6666666666
      li(4,1)=2.6666666666
      li(4,2)=3.0
      li(5,1)=3.0          
      li(5,2)=3.3333333333
      li(6,1)=3.3333333333
      li(6,2)=3.6666666666
      li(7,1)=3.6666666666
      li(7,2)=4.0
      li(8,1)=4.0
      li(8,2)=4.5
      li(9,1)=4.5
      li(9,2)=5.0
      li(10,1)=5.0
      li(10,2)=5.5
      li(11,1)=5.5
      li(11,2)=6.0
      li(12,1)=6.0
      li(12,2)=7.0           
      li(13,1)=7.0           
      li(13,2)=8.0           
      li(14,1)=8.0           
      li(14,2)=9.0           
      li(15,1)=9.0           
      li(15,2)=10.0          
      nbin=15

      call ncoord(n,rcov,iz,xyz,cn,cn_thr)

      write(*,*)
      write(*,*)'analysis of pair-wise terms (in kcal/mol)'
      write(*,'(''pair'',2x,''atoms'',9x,''C6'',14x,''C8'',12x,
     .''E6'',7x,''E8'',7x,''Edisp'')')
      e8=0
      ed=0
      dist=0
      check=0
      do iat=1,n-1
         do jat=iat+1,n

            dx=xyz(1,iat)-xyz(1,jat)
            dy=xyz(2,iat)-xyz(2,jat)
            dz=xyz(3,iat)-xyz(3,jat)
            r2=(dx*dx+dy*dy+dz*dz)
CTHR
            if(r2.gt.rthr) cycle
            r =sqrt(r2)
            R0=r0ab(iz(jat),iz(iat))
            rr=R0/r
            r6=r2**3       

            tmp=rs6*rr   
            damp6 =1.d0/( 1.d0+6.d0*tmp**alp6 )
            tmp=rs8*rr     
            damp8 =1.d0/( 1.d0+6.d0*tmp**alp8 )

            if (version.eq.2)then
              c6=c6ab(iz(jat),iz(iat),1,1,1)
              damp6=1.d0/(1.d0+exp(-alp6*(r/(rs6*R0)-1.0d0)))
              e6 =s6*autokcal*c6*damp6/r6
              e8=0.0
            else
              call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                      cn(iat),cn(jat),c6)
            endif

            if(version.eq.3)then
            e6 =s6*autokcal*c6*damp6/r6
            r8 =r6*r2
            r42=r2r4(iz(iat))*r2r4(iz(jat))
            c8 =3.0d0*c6*r42
            e8 =s18*autokcal*c8*damp8/r8
            endif

            if(version.eq.4)then
            r42=r2r4(iz(iat))*r2r4(iz(jat))
            c8 =3.0d0*c6*r42
c use BJ radius
            R0=sqrt(c8/c6)              
            rvdw=rs6*R0+rs8
            e6 =s6*autokcal*c6/(r6+rvdw**6)
            r8 =r6*r2
            e8 =s18*autokcal*c8/(r8+rvdw**8)
            endif

            edisp=-(e6+e8)
            ed(iat,jat)=edisp
            ed(jat,iat)=edisp

           write(*,'(2i4,2x,2i3,2D16.6,2F9.4,F10.5)')
     .     iat,jat,iz(iat),iz(jat),c6,c8,
     .    -e6,-e8,edisp

            check=check+edisp
            rr=r*autoang
            do i=0,nbin
               if(rr.gt.li(i,1).and.rr.le.li(i,2)) dist(i)=dist(i)+edisp
            enddo
         enddo
      enddo

      write(*,'(/''distance range (Angstroem) analysis'')')
      write(*,'( ''writing histogram data to <histo.dat>'')')
      open(unit=11,file='histo.dat')
      do i=0,nbin
         write(*,'(''R(low,high), Edisp, %tot :'',2f4.1,F12.5,F8.2)')
     .   li(i,1),li(i,2),dist(i),100.*dist(i)/etot
         write(11,*)(li(i,1)+li(i,2))*0.5,dist(i)
      enddo
      close(11)

      write(*,*) 'checksum (Edisp) ',check
      if(abs(check-etot).gt.1.d-3)stop'something is weired in adisp'

      inquire(file='fragment',exist=ex)
      if(.not.ex) return
      write(*,'(/''fragment based analysis'')')
      write(*,'( ''reading file <fragment> ...'')')
      open(unit=55,file='fragment')
      i=0
      at=0
 111  read(55,'(a)',end=222) atmp
      call readfrag(atmp,iout,j)
      if(j.gt.0)then
         i=i+1
         grpn(i)=j
         do k=1,j
            grplist(k,i)=iout(k)      
            at(grplist(k,i))=at(grplist(k,i))+1
         enddo
      endif
      goto 111
 222  continue
      ngrp=i  
      k=0
      do i=1,n
         if(at(i).gt.1) stop 'something is weird in file <fragment>'
         if(at(i).eq.0)then
            k=k+1
            grplist(k,ngrp+1)=i
         endif
      enddo
      if(k.gt.0) then
         ngrp=ngrp+1
         grpn(ngrp)=k
      endif
c Implemented display of atom ranges instead of whole list of atoms
      write(*,*)'group #        atoms '
      dash=0
      do i=1,ngrp
       write(*,'(i4,3x,i4)',advance='no')i,grplist(1,i)
       do j=2,grpn(i)
        if(grplist(j,i).eq.(grplist(j-1,i)+1)) then
         if(dash.eq.0)then
          write(*,'(A1)',advance='no')'-'
          dash=1
         endif
        else
         if(dash.eq.1)then
          write(*,'(i4)',advance='no') grplist(j-1,i)
          dash=0
         endif
         write(*,'(i4)',advance='no') grplist(j,i)
        endif
       enddo 
       if(dash.eq.1)then
        write(*,'(i4)',advance='no') grplist(j-1,i)
        dash=0
       endif
      write(*,*)''
      enddo

c old display list code
c      write(*,*)'group #        atoms '
c      do i=1,ngrp      
c         write(*,'(i4,3x,100i3)')i,(grplist(j,i),j=1,grpn(i))
c      enddo

      eg=0
      iii=0
      do i=1,ngrp
         ni=grpn(i)
         iii=iii+1
         jjj=0
         do j=1,ngrp
            nj=grpn(j)
            jjj=jjj+1
            do ii=1,ni
               iiii=grplist(ii,i)
               do jj=1,nj
                  jjjj=grplist(jj,j)
                  if(jjjj.lt.iiii)cycle
                  eg(lin(iii,jjj))=eg(lin(iii,jjj))+ed(iiii,jjjj)
               enddo
            enddo
         enddo
      enddo

c     call prmat(6,eg,ngrp,0,'intra- + inter-group dispersion energies')
      write(*,*)' group i      j     Edisp'
      k=0
      check=0
      do i=1,ngrp
      do j=1,i    
      k=k+1
      check=check+eg(k) 
      write(*,'(5x,i4,'' --'',i4,F8.2)')i,j,eg(k)
      enddo
      enddo
      write(*,*) 'checksum (Edisp) ',check

      end subroutine adisp

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C compute gradient
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
c      subroutine gdisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
c     .                 s6,s18,rs6,rs8,rs10,alp6,alp8,alp10,noabc,rthr,
c     .                 num,version,echo,g,disp,gnorm,cn_thr,fix)
c
      subroutine gdisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .                 s6,s18,rs6,rs8,rs10,alp6,alp8,alp10,noabc,rthr,
     .                 num,version,echo,g,disp,gnorm,cn_thr)
      implicit none  
      !include  'param'

      real*8 k1,k2,k3

c global ad hoc parameters
      parameter (k1=16.0)
      parameter (k2=4./3.) 

c reasonable choices are between 3 and 5
c this gives smoth curves with maxima around the integer values
c k3=3 give for CN=0 a slightly smaller value than computed
c for the free atom. This also yields to larger CN for atoms
c in larger molecules but with the same chem. environment
c which is physically not right
c values >5 might lead to bumps in the potential
      parameter (k3=-4.) 


      integer n,iz(*),max_elem,maxc,version,mxc(max_elem)
      real*8 xyz(3,*),r0ab(max_elem,max_elem),r2r4(*)
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)
      real*8 g(3,*),s6,s18,rcov(max_elem)
      real*8 rs6,rs8,rs10,alp10,alp8,alp6,a1,a2,r2ik        
      !logical noabc,num,echo,fix(n)
      logical noabc,num,echo
 
      integer iat,jat,i,j,kat
      real*8 R0,C6,alp,R42,disp,x1,y1,z1,x2,y2,z2,rr,e6abc  
      real*8 dx,dy,dz,r2,r,r4,r6,r8,r10,r12,t6,t8,t10,damp1
      real*8 damp6,damp8,damp10,e6,e8,e10,e12,gnorm,tmp1
      real*8 s10,s8,gC6(3),term,step,dispr,displ,r235,tmp2
      real*8 cn(n),gx1,gy1,gz1,gx2,gy2,gz2,rthr,c8,cn_thr
      real*8 rthr3

      real*8 rij(3),rik(3),rjk(3),r7,r9
      real*8 rik_dist,rjk_dist
      real*8 drij(n*(n+1)/2)  !d(E)/d(r_ij) derivative wrt. dist. iat-jat
      real*8 drik,drjk
      real*8 rcovij
      real*8 dc6,c6chk !d(C6ij)/d(r_ij)
      real*8 expterm,dcni
      real*8 dcn(n*(n+1)/2)    !dCN(iat)/d(r_ij) is equal to
                               !dCN(jat)/d(r_ij)     
      real*8 dc6_rest(n*(n+1)/2) ! saves (1/r^6*f_dmp + 3*r4r2/r^8*f_dmp) for kat loop
      integer,external :: lin
      integer  linij,linik,linjk
      real*8 vec(3),vec2(3)
      real*8 dc6ij(n,n)       !dC6(iat,jat)/dCN(iat) in dc6ij(i,j)
                              !dC6(iat,jat)/cCN(jat) in dc6ij(j,i)
      logical skip(n*(n+1)/2)                        


      dc6ij=0.0d0



      

c this is the crucial threshold to reduce the N^3 to an
c effective N^2. 

c      write(*,*)'rthr=',rthr,'rthr2=',rthr2,'rthr3=',rthr3

      if(echo)write(*,*) 
c 2222222222222222222222222222222222222222222222222222222222222222222222222
      if(version.eq.2)then
      if(echo)write(*,*) 'doing analytical gradient O(N^2) ...'
      disp=0
      do iat=1,n-1
         do jat=iat+1,n
            R0=r0ab(iz(jat),iz(iat))*rs6
            dx=(xyz(1,iat)-xyz(1,jat))
            dy=(xyz(2,iat)-xyz(2,jat))
            dz=(xyz(3,iat)-xyz(3,jat))
            r2  =dx*dx+dy*dy+dz*dz             
c           if(r2.gt.rthr) cycle
            r235=r2**3.5                       
            r   =sqrt(r2)
            damp6=exp(-alp6*(r/R0-1.0d0))
            damp1=1.+damp6           
            c6=c6ab(iz(jat),iz(iat),1,1,1)*s6
            tmp1=damp6/(damp1*damp1*r235*R0)
            tmp2=6./(damp1*r*r235)
            gx1=alp6* dx*tmp1-tmp2*dx
            gx2=alp6*(-dx)*tmp1+tmp2*dx
            gy1=alp6* dy*tmp1-tmp2*dy
            gy2=alp6*(-dy)*tmp1+tmp2*dy
            gz1=alp6* dz*tmp1-tmp2*dz
            gz2=alp6*(-dz)*tmp1+tmp2*dz
            g(1,iat)=g(1,iat)-gx1*c6
            g(2,iat)=g(2,iat)-gy1*c6
            g(3,iat)=g(3,iat)-gz1*c6
            g(1,jat)=g(1,jat)-gx2*c6  
            g(2,jat)=g(2,jat)-gy2*c6      
            g(3,jat)=g(3,jat)-gz2*c6      
            disp=disp+c6*(1./damp1)/r2**3
         enddo
      enddo
      disp=-disp
      goto 999
      endif

cNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN
      if(num) then
      if(echo)write(*,*) 'doing numerical gradient O(N^3) ...'

      call edisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .     rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,rthr,cn_thr,
     .     e6,e8,e10,e12,e6abc)
      disp=-s6*e6-s18*e8-s6*e6abc

      step=2.d-5     

      do i=1,n
      do j=1,3
      xyz(j,i)=xyz(j,i)+step        
      call edisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .     rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,rthr,cn_thr,
     .     e6,e8,e10,e12,e6abc)
      dispr=-s6*e6-s18*e8-s6*e6abc
      xyz(j,i)=xyz(j,i)-2*step      
      call edisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,rcov,
     .     rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,rthr,cn_thr,
     .     e6,e8,e10,e12,e6abc)
      displ=-s6*e6-s18*e8-s6*e6abc
      g(j,i)=0.5*(dispr-displ)/step  
      xyz(j,i)=xyz(j,i)+step        
      enddo
      enddo

      else

      if(echo)write(*,*) 'doing analytical gradient O(N^3) ...'
c precompute for analytical part

c 333333333333333333333333333333333333333333333333333333333333333333333333333
c standard correction
      if (version.eq.3) then
      call ncoord(n,rcov,iz,xyz,cn,cn_thr)
      s8 =s18
      s10=s18

      disp=0

      drij=0.0d0
      dc6_rest=0.0d0
      dcn=0.0d0
      kat=0
      skip=.true.


      do iat=1,n
        do jat=1,iat-1
          rij=xyz(:,jat)-xyz(:,iat)
          r2=sum(rij*rij)
          if (r2.gt.rthr) cycle
          linij=lin(iat,jat)
          skip(linij)=.false.

          r0=r0ab(iz(jat),iz(iat))
          r42=r2r4(iz(iat))*r2r4(iz(jat))
          rcovij=rcov(iz(iat))+rcov(iz(jat))
!
!      get_dC6_dCNij calculates the derivative dC6(iat,jat)/dCN(iat) and
!      dC6(iat,jat)/dCN(jat). these are saved in dC6ij for the kat loop
!
          call get_dC6_dCNij(maxc,max_elem,c6ab,mxc(iz(iat)),
     .          mxc(iz(jat)),cn(iat),cn(jat),iz(iat),iz(jat),iat,jat,
     .          c6,dc6ij(iat,jat),dc6ij(jat,iat))


          r=dsqrt(r2)
          r6=r2*r2*r2
          r7=r6*r
          r8=r6*r2
          r9=r8*r

!
!  Calculates damping functions:
          t6 = (r/(rs6*R0))**(-alp6)
          damp6 =1.d0/( 1.d0+6.d0*t6 )
          t8 = (r/(rs8*R0))**(-alp8)
          damp8 =1.d0/( 1.d0+6.d0*t8 )

          drij(linij)=drij(linij)-s6*(6.0/(r7)*C6*damp6)  ! d(r^(-6))/d(r_ij)
     .        -s8*(24.0/(r9)*C6*r42*damp8)


          drij(linij)=drij(linij)
     .        +s6*C6/r7*6.d0*alp6*t6*damp6*damp6     !d(f_dmp)/d(r_ij)
     .        +s8*C6*r42/r9*18.d0*alp8*t8*damp8*damp8


!
!      in dC6_rest all terms BUT C6-term is saved for the kat-loop
!          
          dc6_rest(linij)=s6/r6*damp6+3.d0*s8*r42/r8*damp8

          disp=disp-dc6_rest(linij)*c6  ! calculate E_disp for sanity check
!
!         Calculate dCN(iat)/dr_ij which is identical to dCN(jat)/dr_ij
!          this is needed for dC6/dr_ij          
!         saved in dcn for the kat-loop
!          

          if (r2.lt.cn_thr) then
          expterm=exp(-k1*(rcovij/r-1.d0))
          dcn(linij)=-k1*rcovij*expterm/
     .                 (r*r*(expterm+1.d0)*(expterm+1.d0))
!     
!     Combine dC6/dCN * dCN/dr_ij to get dC6/dr_ij          
          dc6=(dc6ij(iat,jat)+dc6ij(jat,iat))*dcn(linij)

          drij(linij)=drij(linij)
     .         + dc6_rest(linij)*dc6            !d(C6(ij))/d(r_ij)

          else
            dc6=0.0d0
            skip(linij)=.true.
          endif

!
!  The kat loop calculates the contributions of dC6(i,k)/dr_ij,
!  dC6(j,k)/dr_ij, dC6(ij)/dr_ik, dC6(ij)/dr_jk
!  Basically all term that depend on the coordinates of
!  3 atoms
!  This is the reason, why the gradient scales N^3            
!            
          DO kat=1,jat-1
            linik=lin(iat,kat)
            linjk=lin(jat,kat)


!              dc6=dc6ij(iat,kat)*dcn(linij)

            if (.not.skip(linij)) then
              drij(linij)=drij(linij)+
     .            dc6_rest(linik)*dc6ij(iat,kat)*dcn(linij)             

              drij(linij)=drij(linij)+
     .         dc6_rest(linjk)*dc6ij(jat,kat)*dcn(linij)
            endif


            if (.not.skip(linjk)) then
              drij(linjk)=drij(linjk)+
     .           dc6_rest(linik)*dc6ij(kat,iat)*dcn(linjk)

              drij(linjk)=drij(linjk)+
     .           dc6_rest(linij)*dc6ij(jat,iat)*dcn(linjk)
            endif
             
          
 

            
            if (.not.skip(linik)) then
              drij(linik)=drij(linik)+
     .        dc6_rest(linjk)*dc6ij(kat,jat)*dcn(linik) 


              drij(linik)=drij(linik)+
     .          dc6_rest(linij)*dc6ij(iat,jat)*dcn(linik)
            endif


          ENDDO !kat
!          g(:,iat)=g(:,iat)+drij*rij/r
!          g(:,jat)=g(:,jat)-drij*rij/r
        enddo !jat
      enddo !iat

! After calculating all derivatives dE/dr_ij w.r.t. distances,
! the grad w.r.t. the coordinates is calculated dE/dr_ij * dr_ij/dxyz_i       
      do iat=2,n
        do jat=1,iat-1
          rij=xyz(:,jat)-xyz(:,iat)
          r=sqrt(sum(rij*rij))
          g(:,iat)=g(:,iat)+drij(lin(iat,jat))*rij/r
          g(:,jat)=g(:,jat)-drij(lin(iat,jat))*rij/r


        enddo !iat
      enddo !jat

!      stop('Bis hier und nicht weiter!')
      endif !version 3

c BJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJBJ 
c Becke-Johnson finite damping 
      if (version.eq.4) then 
      a1 =rs6
      a2 =rs8
      s8 =s18

      disp=0
      call ncoord(n,rcov,iz,xyz,cn,cn_thr)

      drij=0.0d0
      dc6_rest=0.0d0
      dcn=0.0d0
      kat=0
      skip=.true.


      do iat=1,n
        do jat=1,iat-1
          rij=xyz(:,jat)-xyz(:,iat)
          r2=sum(rij*rij)
          if (r2.gt.rthr) cycle

          linij=lin(iat,jat)
          skip(linij)=.false.
          r0=r0ab(iz(jat),iz(iat))
          r42=r2r4(iz(iat))*r2r4(iz(jat))
          rcovij=rcov(iz(iat))+rcov(iz(jat))
!
!      get_dC6_dCNij calculates the derivative dC6(iat,jat)/dCN(iat) and
!      dC6(iat,jat)/dCN(jat). these are saved in dC6ij for the kat loop
!
          call get_dC6_dCNij(maxc,max_elem,c6ab,mxc(iz(iat)),
     .          mxc(iz(jat)),cn(iat),cn(jat),iz(iat),iz(jat),iat,jat,
     .          c6,dc6ij(iat,jat),dc6ij(jat,iat))


          r=dsqrt(r2)
          r4=r2*r2
          r6=r4*r2
          r7=r6*r
          r8=r6*r2
          r9=r8*r

c use BJ radius
          R0=a1*sqrt(3.0d0*r42)+a2
 
          t6=(r6+R0**6)
          t8=(r8+R0**8)

          drij(linij)=drij(linij)
     .        -s6*C6*6.0d0*r4*r/(t6*t6)
     .        -s8*C6*24.0d0*r42*r6*r/(t8*t8)

          dc6_rest(linij)=s6/t6+3.d0*s8*r42/t8
          disp=disp-dc6_rest(linij)*c6  ! calculate E_disp for sanity check
!
!         Calculate dCN(iat)/dr_ij which is identical to dCN(jat)/dr_ij
!          this is needed for dC6/dr_ij          
!         saved in dcn for the kat-loop
!          

          if (r2.lt.cn_thr) then 
            expterm=exp(-k1*(rcovij/r-1.d0))
            dcn(linij)=-k1*rcovij*expterm/
     .                 (r*r*(expterm+1.d0)*(expterm+1.d0))
!     
!     Combine dC6/dCN * dCN/dr_ij to get dC6/dr_ij          
            dc6=(dc6ij(iat,jat)+dc6ij(jat,iat))*dcn(linij)
!
!      in dC6_rest all terms BUT C6-term is saved for the kat-loop
!          

            drij(linij)=drij(linij)
     .         + dc6_rest(linij)*dc6            !d(C6(ij))/d(r_ij)


          else 
            dc6=0.0d0
            dcn(linij)=0.0d0
            skip(linij)=.true.
          endif

!
!  The kat loop calculates the contributions of dC6(i,k)/dr_ij,
!  dC6(j,k)/dr_ij, dC6(ij)/dr_ik, dC6(ij)/dr_jk
!  Basically all term that depend on the coordinates of
!  3 atoms
!  This is the reason, why the gradient scales N^3            
!            
          DO kat=1,jat-1

            linik=lin(iat,kat)
            linjk=lin(jat,kat)
              dc6=dc6ij(iat,kat)*dcn(linij)

              drij(linij)=drij(linij)+
     .            dc6_rest(linik)*dc6              

              dc6=dc6ij(jat,kat)*dcn(linij)
 
!              if (c6-c6chk.gt.1.d-9) stop('error in C6-grad')

              drij(linij)=drij(linij)+
     .         dc6_rest(linjk)*dc6

!            if (kat.eq.iat.or.kat.eq.jat) cycle
!            rik=xyz(:,kat)-xyz(:,iat)
!            rik_dist=sum(rik*rik)

!            rjk=xyz(:,kat)-xyz(:,jat)
!            rjk_dist=sum(rjk*rjk)
 
!            if (rik_dist.le.rthr) then
          

!              if (c6-c6chk.gt.1.d-9) stop('error in C6-grad')




            if(.not.skip(linjk)) then 
              drij(linjk)=drij(linjk)+
     .           dc6_rest(linik)*dc6ij(kat,iat)*dcn(linjk)


              drij(linjk)=drij(linjk)+
     .           dc6_rest(linij)*dc6ij(jat,iat)*dcn(linjk)
            endif 
!              g(:,kat)=g(:,kat)-drjk*rjk/rjk_dist
!              g(:,jat)=g(:,jat)+drjk*rjk/rjk_dist
!            endif !rik<rthr



!           if (rjk_dist.le.rthr) then
          

            if (.not.skip(linik)) then 
              drij(linik)=drij(linik)+
     .          dc6_rest(linjk)*dc6ij(kat,jat)*dcn(linik) 


              drij(linik)=drij(linik)+
     .          dc6_rest(linij)*dc6ij(iat,jat)*dcn(linik)
            endif 

!              g(:,kat)=g(:,kat)-drik*rik/rik_dist
!              g(:,iat)=g(:,iat)+drjk*rik/rik_dist
!            endif !rik<rthr

          ENDDO !kat



        enddo !jat

      enddo !iat
! After calculating all derivatives dE/dr_ij w.r.t. distances,
! the grad w.r.t. the coordinates is calculated dE/dr_ij * dr_ij/dxyz_i       
      do iat=2,n
        do jat=1,iat-1
          rij=xyz(:,jat)-xyz(:,iat)
          r=sqrt(sum(rij*rij))
          g(:,iat)=g(:,iat)+drij(lin(iat,jat))*rij/r
          g(:,jat)=g(:,jat)-drij(lin(iat,jat))*rij/r


        enddo !iat
      enddo !jat


      endif !version=4 (BJ)


      endif

 999  continue
      gnorm=sum(abs(g(1:3,1:n)))
      if(echo)then
      write(*,*)
      write(*,*)'|G|=',gnorm
      endif

c      do i=1,n                 !fixed atoms have no gradient
c       if(fix(i))g(:,i)=0.0
c      enddo

       

      end subroutine gdisp


CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C      The   N E W   gradC6 routine    C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
!
      subroutine get_dC6_dCNij(maxc,max_elem,c6ab,mxci,mxcj,cni,cnj,
     .           izi,izj,iat,jat,c6check,dc6i,dc6j)

      IMPLICIT NONE
      !include  'param'

      real*8 k1,k2,k3

c global ad hoc parameters
      parameter (k1=16.0)
      parameter (k2=4./3.) 

c reasonable choices are between 3 and 5
c this gives smoth curves with maxima around the integer values
c k3=3 give for CN=0 a slightly smaller value than computed
c for the free atom. This also yields to larger CN for atoms
c in larger molecules but with the same chem. environment
c which is physically not right
c values >5 might lead to bumps in the potential
      parameter (k3=-4.) 


      integer maxc,max_elem
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)
      integer mxci,mxcj   !mxc(iz(iat))
      real*8 cni,cnj
      integer iat,jat,izi,izj
      real*8  dc6i,dc6j,c6check


      integer i,j,a,b
      real*8 zaehler,nenner,dzaehler_i,dnenner_i,dzaehler_j,dnenner_j
      real*8 expterm,cn_refi,cn_refj,c6ref,r
      real*8 c6mem,r_save



      c6mem=-1.d99
      r_save=9999.0
      zaehler=0.0d0
      nenner=0.0d0

      dzaehler_i=0.d0
      dnenner_i=0.d0
      dzaehler_j=0.d0
      dnenner_j=0.d0


      DO a=1,mxci
        DO b=1,mxcj
          c6ref=c6ab(izi,izj,a,b,1)
          if (c6ref.gt.0) then
!            c6mem=c6ref
            cn_refi=c6ab(izi,izj,a,b,2)
            cn_refj=c6ab(izi,izj,a,b,3)
            r=(cn_refi-cni)*(cn_refi-cni)+(cn_refj-cnj)*(cn_refj-cnj)
            if (r.lt.r_save) then
               r_save=r
               c6mem=c6ref
            endif
            expterm=exp(k3*r)
            zaehler=zaehler+c6ref*expterm
            nenner=nenner+expterm
            dzaehler_i=dzaehler_i+c6ref*expterm*
     .             2.d0*k3*(cni-cn_refi)
            dnenner_i=dnenner_i+expterm*
     .             2.d0*k3*(cni-cn_refi)

            dzaehler_j=dzaehler_j+c6ref*expterm*
     .             2.d0*k3*(cnj-cn_refj)
            dnenner_j=dnenner_j+expterm*
     .             2.d0*k3*(cnj-cn_refj)
          endif
        ENDDO !b
      ENDDO !a

      if (nenner.gt.1.0d-99) then
        c6check=zaehler/nenner
        dc6i=((dzaehler_i*nenner)-(dnenner_i*zaehler))
     .    /(nenner*nenner)
        dc6j=((dzaehler_j*nenner)-(dnenner_j*zaehler))
     .    /(nenner*nenner)
      else
        c6check=c6mem
        dc6i=0.0d0
        dc6j=0.0d0
      endif
      end subroutine get_dC6_dCNij



CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C interpolate c6  
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine getc6(maxc,max_elem,c6ab,mxc,iat,jat,nci,ncj,c6)
      implicit none
      integer maxc,max_elem
      integer iat,jat,i,j,mxc(max_elem)
      real*8  nci,ncj,c6,c6mem
      real*8  c6ab(max_elem,max_elem,maxc,maxc,3)
c the exponential is sensitive to numerics
c when nci or ncj is much larger than cn1/cn2
      real*8  cn1,cn2,r,rsum,csum,tmp,tmp1   
      real*8  r_save
      !include 'param'

      real*8 k1,k2,k3

c global ad hoc parameters
      parameter (k1=16.0)
      parameter (k2=4./3.) 

c reasonable choices are between 3 and 5
c this gives smoth curves with maxima around the integer values
c k3=3 give for CN=0 a slightly smaller value than computed
c for the free atom. This also yields to larger CN for atoms
c in larger molecules but with the same chem. environment
c which is physically not right
c values >5 might lead to bumps in the potential
      parameter (k3=-4.) 


      c6mem=-1.d+99
      rsum=0.0
      csum=0.0
      c6  =0.0
      r_save=1.0d99
      do i=1,mxc(iat)
      do j=1,mxc(jat)
         c6=c6ab(iat,jat,i,j,1)
         if(c6.gt.0)then
!            c6mem=c6
            cn1=c6ab(iat,jat,i,j,2)
            cn2=c6ab(iat,jat,i,j,3)
c distance
            r=(cn1-nci)**2+(cn2-ncj)**2
            if (r.lt.r_save) then
               r_save=r
               c6mem=c6
            endif
            tmp1=exp(k3*r)
            rsum=rsum+tmp1     
            csum=csum+tmp1*c6
         endif
      enddo
      enddo

      if(rsum.gt.1.0d-99)then
         c6=csum/rsum
      else
         c6=c6mem
      endif

      end subroutine getc6

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C compute coordination numbers by adding an inverse damping function
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine ncoord(natoms,rcov,iz,xyz,cn,cn_thr)
      implicit none  
      !include 'param'

      real*8 k1,k2,k3

c global ad hoc parameters
      parameter (k1=16.0)
      parameter (k2=4./3.) 

c reasonable choices are between 3 and 5
c this gives smoth curves with maxima around the integer values
c k3=3 give for CN=0 a slightly smaller value than computed
c for the free atom. This also yields to larger CN for atoms
c in larger molecules but with the same chem. environment
c which is physically not right
c values >5 might lead to bumps in the potential
      parameter (k3=-4.) 

      integer iz(*),natoms,i,max_elem
      real*8 xyz(3,*),cn(*),rcov(94),input
      real*8 cn_thr

      integer iat    
      real*8 dx,dy,dz,r,damp,xn,rr,rco,r2

      do i=1,natoms
      xn=0.0d0
      do iat=1,natoms
         if(iat.ne.i)then
            dx=xyz(1,iat)-xyz(1,i)
            dy=xyz(2,iat)-xyz(2,i)
            dz=xyz(3,iat)-xyz(3,i)
            r2=dx*dx+dy*dy+dz*dz 
            if (r2.gt.cn_thr) cycle 
            r=sqrt(r2)                  
c covalent distance in Bohr
            rco=rcov(iz(i))+rcov(iz(iat))
            rr=rco/r
c counting function exponential has a better long-range behavior than MHGs inverse damping
            damp=1.d0/(1.d0+exp(-k1*(rr-1.0d0)))
            xn=xn+damp
         endif
      enddo
c     if (iz(i).eq.19) then
c        write(*,*) "Input CN of Kalium"
c        read(*,*),input
c         cn(i)=input
c      else
         cn(i)=xn
c      endif  
      enddo

      end subroutine ncoord

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C load C6 coefficients from file
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine loadc6(fname,maxc,max_elem,c6ab,maxci)
      implicit none
      integer maxc,max_elem,maxci(max_elem)
      real*8  c6ab(max_elem,max_elem,maxc,maxc,3)
      character*(*) fname
      character*1  atmp 
      character*80 btmp 

      real*8  x,y,f,cn1,cn2,cmax,xx(10)
      integer iat,jat,i,n,l,j,k,il,iadr,jadr,nn

      c6ab=-1
      maxci=0

c read file
      open(unit=1,file=fname)
      read(1,'(a)')btmp
 10   read(1,*,end=11) y,iat,jat,cn1,cn2  
      call limit(iat,jat,iadr,jadr)
      maxci(iat)=max(maxci(iat),iadr)
      maxci(jat)=max(maxci(jat),jadr)
      c6ab(iat,jat,iadr,jadr,1)=y  
      c6ab(iat,jat,iadr,jadr,2)=cn1
      c6ab(iat,jat,iadr,jadr,3)=cn2

      c6ab(jat,iat,jadr,iadr,1)=y  
      c6ab(jat,iat,jadr,iadr,2)=cn2
      c6ab(jat,iat,jadr,iadr,3)=cn1
c     endif
      goto 10
 11   continue
      close(1)

      end subroutine loadc6

      integer function lin(i1,i2)
      integer i1,i2,idum1,idum2
      idum1=max(i1,i2)
      idum2=min(i1,i2)
      lin=idum2+idum1*(idum1-1)/2
      return
      end function lin

      subroutine limit(iat,jat,iadr,jadr)
      implicit none
      integer iat,jat,iadr,jadr,i
      iadr=1
      jadr=1
      i=100
 10   if(iat.gt.100) then
         iat=iat-100
         iadr=iadr+1
         goto 10
      endif

      i=100
 20   if(jat.gt.100) then
         jat=jat-100
         jadr=jadr+1
         goto 20
      endif

      end subroutine limit

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C set cut-off radii
C in parts due to INTEL compiler bug
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine setr0ab(max_elem,autoang,r)
      implicit none  
      integer max_elem,i,j,k
      real*8 r(max_elem,max_elem),autoang
      real*8 r0ab(4465)
      r0ab(   1:  70)=(/
     .   2.1823,  1.8547,  1.7347,  2.9086,  2.5732,  3.4956,  2.3550
     .,  2.5095,  2.9802,  3.0982,  2.5141,  2.3917,  2.9977,  2.9484
     .,  3.2160,  2.4492,  2.2527,  3.1933,  3.0214,  2.9531,  2.9103
     .,  2.3667,  2.1328,  2.8784,  2.7660,  2.7776,  2.7063,  2.6225
     .,  2.1768,  2.0625,  2.6395,  2.6648,  2.6482,  2.5697,  2.4846
     .,  2.4817,  2.0646,  1.9891,  2.5086,  2.6908,  2.6233,  2.4770
     .,  2.3885,  2.3511,  2.2996,  1.9892,  1.9251,  2.4190,  2.5473
     .,  2.4994,  2.4091,  2.3176,  2.2571,  2.1946,  2.1374,  2.9898
     .,  2.6397,  3.6031,  3.1219,  3.7620,  3.2485,  2.9357,  2.7093
     .,  2.5781,  2.4839,  3.7082,  2.5129,  2.7321,  3.1052,  3.2962
     ./)
      r0ab(  71: 140)=(/
     .   3.1331,  3.2000,  2.9586,  3.0822,  2.8582,  2.7120,  3.2570
     .,  3.4839,  2.8766,  2.7427,  3.2776,  3.2363,  3.5929,  3.2826
     .,  3.0911,  2.9369,  2.9030,  2.7789,  3.3921,  3.3970,  4.0106
     .,  2.8884,  2.6605,  3.7513,  3.1613,  3.3605,  3.3325,  3.0991
     .,  2.9297,  2.8674,  2.7571,  3.8129,  3.3266,  3.7105,  3.7917
     .,  2.8304,  2.5538,  3.3932,  3.1193,  3.1866,  3.1245,  3.0465
     .,  2.8727,  2.7664,  2.6926,  3.4608,  3.2984,  3.5142,  3.5418
     .,  3.5017,  2.6190,  2.4797,  3.1331,  3.0540,  3.0651,  2.9879
     .,  2.9054,  2.8805,  2.7330,  2.6331,  3.2096,  3.5668,  3.3684
     .,  3.3686,  3.3180,  3.3107,  2.4757,  2.4019,  2.9789,  3.1468
     ./)
      r0ab( 141: 210)=(/
     .   2.9768,  2.8848,  2.7952,  2.7457,  2.6881,  2.5728,  3.0574
     .,  3.3264,  3.3562,  3.2529,  3.1916,  3.1523,  3.1046,  2.3725
     .,  2.3289,  2.8760,  2.9804,  2.9093,  2.8040,  2.7071,  2.6386
     .,  2.5720,  2.5139,  2.9517,  3.1606,  3.2085,  3.1692,  3.0982
     .,  3.0352,  2.9730,  2.9148,  3.2147,  2.8315,  3.8724,  3.4621
     .,  3.8823,  3.3760,  3.0746,  2.8817,  2.7552,  2.6605,  3.9740
     .,  3.6192,  3.6569,  3.9586,  3.6188,  3.3917,  3.2479,  3.1434
     .,  4.2411,  2.7597,  3.0588,  3.3474,  3.6214,  3.4353,  3.4729
     .,  3.2487,  3.3200,  3.0914,  2.9403,  3.4972,  3.7993,  3.6773
     .,  3.8678,  3.5808,  3.8243,  3.5826,  3.4156,  3.8765,  4.1035
     ./)
      r0ab( 211: 280)=(/
     .   2.7361,  2.9765,  3.2475,  3.5004,  3.4185,  3.4378,  3.2084
     .,  3.2787,  3.0604,  2.9187,  3.4037,  3.6759,  3.6586,  3.8327
     .,  3.5372,  3.7665,  3.5310,  3.3700,  3.7788,  3.9804,  3.8903
     .,  2.6832,  2.9060,  3.2613,  3.4359,  3.3538,  3.3860,  3.1550
     .,  3.2300,  3.0133,  2.8736,  3.4024,  3.6142,  3.5979,  3.5295
     .,  3.4834,  3.7140,  3.4782,  3.3170,  3.7434,  3.9623,  3.8181
     .,  3.7642,  2.6379,  2.8494,  3.1840,  3.4225,  3.2771,  3.3401
     .,  3.1072,  3.1885,  2.9714,  2.8319,  3.3315,  3.5979,  3.5256
     .,  3.4980,  3.4376,  3.6714,  3.4346,  3.2723,  3.6859,  3.8985
     .,  3.7918,  3.7372,  3.7211,  2.9230,  2.6223,  3.4161,  2.8999
     ./)
      r0ab( 281: 350)=(/
     .   3.0557,  3.3308,  3.0555,  2.8508,  2.7385,  2.6640,  3.5263
     .,  3.0277,  3.2990,  3.7721,  3.5017,  3.2751,  3.1368,  3.0435
     .,  3.7873,  3.2858,  3.2140,  3.1727,  3.2178,  3.4414,  2.5490
     .,  2.7623,  3.0991,  3.3252,  3.1836,  3.2428,  3.0259,  3.1225
     .,  2.9032,  2.7621,  3.2490,  3.5110,  3.4429,  3.3845,  3.3574
     .,  3.6045,  3.3658,  3.2013,  3.6110,  3.8241,  3.7090,  3.6496
     .,  3.6333,  3.0896,  3.5462,  2.4926,  2.7136,  3.0693,  3.2699
     .,  3.1272,  3.1893,  2.9658,  3.0972,  2.8778,  2.7358,  3.2206
     .,  3.4566,  3.3896,  3.3257,  3.2946,  3.5693,  3.3312,  3.1670
     .,  3.5805,  3.7711,  3.6536,  3.5927,  3.5775,  3.0411,  3.4885
     ./)
      r0ab( 351: 420)=(/
     .   3.4421,  2.4667,  2.6709,  3.0575,  3.2357,  3.0908,  3.1537
     .,  2.9235,  3.0669,  2.8476,  2.7054,  3.2064,  3.4519,  3.3593
     .,  3.2921,  3.2577,  3.2161,  3.2982,  3.1339,  3.5606,  3.7582
     .,  3.6432,  3.5833,  3.5691,  3.0161,  3.4812,  3.4339,  3.4327
     .,  2.4515,  2.6338,  3.0511,  3.2229,  3.0630,  3.1265,  2.8909
     .,  3.0253,  2.8184,  2.6764,  3.1968,  3.4114,  3.3492,  3.2691
     .,  3.2320,  3.1786,  3.2680,  3.1036,  3.5453,  3.7259,  3.6090
     .,  3.5473,  3.5327,  3.0018,  3.4413,  3.3907,  3.3593,  3.3462
     .,  2.4413,  2.6006,  3.0540,  3.1987,  3.0490,  3.1058,  2.8643
     .,  2.9948,  2.7908,  2.6491,  3.1950,  3.3922,  3.3316,  3.2585
     ./)
      r0ab( 421: 490)=(/
     .   3.2136,  3.1516,  3.2364,  3.0752,  3.5368,  3.7117,  3.5941
     .,  3.5313,  3.5164,  2.9962,  3.4225,  3.3699,  3.3370,  3.3234
     .,  3.3008,  2.4318,  2.5729,  3.0416,  3.1639,  3.0196,  3.0843
     .,  2.8413,  2.7436,  2.7608,  2.6271,  3.1811,  3.3591,  3.3045
     .,  3.2349,  3.1942,  3.1291,  3.2111,  3.0534,  3.5189,  3.6809
     .,  3.5635,  3.5001,  3.4854,  2.9857,  3.3897,  3.3363,  3.3027
     .,  3.2890,  3.2655,  3.2309,  2.8502,  2.6934,  3.2467,  3.1921
     .,  3.5663,  3.2541,  3.0571,  2.9048,  2.8657,  2.7438,  3.3547
     .,  3.3510,  3.9837,  3.6871,  3.4862,  3.3389,  3.2413,  3.1708
     .,  3.6096,  3.6280,  3.6860,  3.5568,  3.4836,  3.2868,  3.3994
     ./)
      r0ab( 491: 560)=(/
     .   3.3476,  3.3170,  3.2950,  3.2874,  3.2606,  3.9579,  2.9226
     .,  2.6838,  3.7867,  3.1732,  3.3872,  3.3643,  3.1267,  2.9541
     .,  2.8505,  2.7781,  3.8475,  3.3336,  3.7359,  3.8266,  3.5733
     .,  3.3959,  3.2775,  3.1915,  3.9878,  3.8816,  3.5810,  3.5364
     .,  3.5060,  3.8097,  3.3925,  3.3348,  3.3019,  3.2796,  3.2662
     .,  3.2464,  3.7136,  3.8619,  2.9140,  2.6271,  3.4771,  3.1774
     .,  3.2560,  3.1970,  3.1207,  2.9406,  2.8322,  2.7571,  3.5455
     .,  3.3514,  3.5837,  3.6177,  3.5816,  3.3902,  3.2604,  3.1652
     .,  3.7037,  3.6283,  3.5858,  3.5330,  3.4884,  3.5789,  3.4094
     .,  3.3473,  3.3118,  3.2876,  3.2707,  3.2521,  3.5570,  3.6496
     ./)
      r0ab( 561: 630)=(/
     .   3.6625,  2.7300,  2.5870,  3.2471,  3.1487,  3.1667,  3.0914
     .,  3.0107,  2.9812,  2.8300,  2.7284,  3.3259,  3.3182,  3.4707
     .,  3.4748,  3.4279,  3.4182,  3.2547,  3.1353,  3.5116,  3.9432
     .,  3.8828,  3.8303,  3.7880,  3.3760,  3.7218,  3.3408,  3.3059
     .,  3.2698,  3.2446,  3.2229,  3.4422,  3.5023,  3.5009,  3.5268
     .,  2.6026,  2.5355,  3.1129,  3.2863,  3.1029,  3.0108,  2.9227
     .,  2.8694,  2.8109,  2.6929,  3.1958,  3.4670,  3.4018,  3.3805
     .,  3.3218,  3.2815,  3.2346,  3.0994,  3.3937,  3.7266,  3.6697
     .,  3.6164,  3.5730,  3.2522,  3.5051,  3.4686,  3.4355,  3.4084
     .,  3.3748,  3.3496,  3.3692,  3.4052,  3.3910,  3.3849,  3.3662
     ./)
      r0ab( 631: 700)=(/
     .   2.5087,  2.4814,  3.0239,  3.1312,  3.0535,  2.9457,  2.8496
     .,  2.7780,  2.7828,  2.6532,  3.1063,  3.3143,  3.3549,  3.3120
     .,  3.2421,  3.1787,  3.1176,  3.0613,  3.3082,  3.5755,  3.5222
     .,  3.4678,  3.4231,  3.1684,  3.3528,  3.3162,  3.2827,  3.2527
     .,  3.2308,  3.2029,  3.3173,  3.3343,  3.3092,  3.2795,  3.2452
     .,  3.2096,  3.2893,  2.8991,  4.0388,  3.6100,  3.9388,  3.4475
     .,  3.1590,  2.9812,  2.8586,  2.7683,  4.1428,  3.7911,  3.8225
     .,  4.0372,  3.7059,  3.4935,  3.3529,  3.2492,  4.4352,  4.0826
     .,  3.9733,  3.9254,  3.8646,  3.9315,  3.7837,  3.7465,  3.7211
     .,  3.7012,  3.6893,  3.6676,  3.7736,  4.0660,  3.7926,  3.6158
     ./)
      r0ab( 701: 770)=(/
     .   3.5017,  3.4166,  4.6176,  2.8786,  3.1658,  3.5823,  3.7689
     .,  3.5762,  3.5789,  3.3552,  3.4004,  3.1722,  3.0212,  3.7241
     .,  3.9604,  3.8500,  3.9844,  3.7035,  3.9161,  3.6751,  3.5075
     .,  4.1151,  4.2877,  4.1579,  4.1247,  4.0617,  3.4874,  3.9848
     .,  3.9280,  3.9079,  3.8751,  3.8604,  3.8277,  3.8002,  3.9981
     .,  3.7544,  4.0371,  3.8225,  3.6718,  4.3092,  4.4764,  2.8997
     .,  3.0953,  3.4524,  3.6107,  3.6062,  3.5783,  3.3463,  3.3855
     .,  3.1746,  3.0381,  3.6019,  3.7938,  3.8697,  3.9781,  3.6877
     .,  3.8736,  3.6451,  3.4890,  3.9858,  4.1179,  4.0430,  3.9563
     .,  3.9182,  3.4002,  3.8310,  3.7716,  3.7543,  3.7203,  3.7053
     ./)
      r0ab( 771: 840)=(/
     .   3.6742,  3.8318,  3.7631,  3.7392,  3.9892,  3.7832,  3.6406
     .,  4.1701,  4.3016,  4.2196,  2.8535,  3.0167,  3.3978,  3.5363
     .,  3.5393,  3.5301,  3.2960,  3.3352,  3.1287,  2.9967,  3.6659
     .,  3.7239,  3.8070,  3.7165,  3.6368,  3.8162,  3.5885,  3.4336
     .,  3.9829,  4.0529,  3.9584,  3.9025,  3.8607,  3.3673,  3.7658
     .,  3.7035,  3.6866,  3.6504,  3.6339,  3.6024,  3.7708,  3.7283
     .,  3.6896,  3.9315,  3.7250,  3.5819,  4.1457,  4.2280,  4.1130
     .,  4.0597,  3.0905,  2.7998,  3.6448,  3.0739,  3.2996,  3.5262
     .,  3.2559,  3.0518,  2.9394,  2.8658,  3.7514,  3.2295,  3.5643
     .,  3.7808,  3.6931,  3.4723,  3.3357,  3.2429,  4.0280,  3.5589
     ./)
      r0ab( 841: 910)=(/
     .   3.4636,  3.4994,  3.4309,  3.6177,  3.2946,  3.2376,  3.2050
     .,  3.1847,  3.1715,  3.1599,  3.5555,  3.8111,  3.7693,  3.5718
     .,  3.4498,  3.3662,  4.1608,  3.7417,  3.6536,  3.6154,  3.8596
     .,  3.0301,  2.7312,  3.5821,  3.0473,  3.2137,  3.4679,  3.1975
     .,  2.9969,  2.8847,  2.8110,  3.6931,  3.2076,  3.4943,  3.5956
     .,  3.6379,  3.4190,  3.2808,  3.1860,  3.9850,  3.5105,  3.4330
     .,  3.3797,  3.4155,  3.6033,  3.2737,  3.2145,  3.1807,  3.1596
     .,  3.1461,  3.1337,  3.4812,  3.6251,  3.7152,  3.5201,  3.3966
     .,  3.3107,  4.1128,  3.6899,  3.6082,  3.5604,  3.7834,  3.7543
     .,  2.9189,  2.6777,  3.4925,  2.9648,  3.1216,  3.2940,  3.0975
     ./)
      r0ab( 911: 980)=(/
     .   2.9757,  2.8493,  2.7638,  3.6085,  3.1214,  3.4006,  3.4793
     .,  3.5147,  3.3806,  3.2356,  3.1335,  3.9144,  3.4183,  3.3369
     .,  3.2803,  3.2679,  3.4871,  3.1714,  3.1521,  3.1101,  3.0843
     .,  3.0670,  3.0539,  3.3890,  3.5086,  3.5895,  3.4783,  3.3484
     .,  3.2559,  4.0422,  3.5967,  3.5113,  3.4576,  3.6594,  3.6313
     .,  3.5690,  2.8578,  2.6334,  3.4673,  2.9245,  3.0732,  3.2435
     .,  3.0338,  2.9462,  2.8143,  2.7240,  3.5832,  3.0789,  3.3617
     .,  3.4246,  3.4505,  3.3443,  3.1964,  3.0913,  3.8921,  3.3713
     .,  3.2873,  3.2281,  3.2165,  3.4386,  3.1164,  3.1220,  3.0761
     .,  3.0480,  3.0295,  3.0155,  3.3495,  3.4543,  3.5260,  3.4413
     ./)
      r0ab( 981:1050)=(/
     .   3.3085,  3.2134,  4.0170,  3.5464,  3.4587,  3.4006,  3.6027
     .,  3.5730,  3.4945,  3.4623,  2.8240,  2.5960,  3.4635,  2.9032
     .,  3.0431,  3.2115,  2.9892,  2.9148,  2.7801,  2.6873,  3.5776
     .,  3.0568,  3.3433,  3.3949,  3.4132,  3.3116,  3.1616,  3.0548
     .,  3.8859,  3.3719,  3.2917,  3.2345,  3.2274,  3.4171,  3.1293
     .,  3.0567,  3.0565,  3.0274,  3.0087,  2.9939,  3.3293,  3.4249
     .,  3.4902,  3.4091,  3.2744,  3.1776,  4.0078,  3.5374,  3.4537
     .,  3.3956,  3.5747,  3.5430,  3.4522,  3.4160,  3.3975,  2.8004
     .,  2.5621,  3.4617,  2.9154,  3.0203,  3.1875,  2.9548,  2.8038
     .,  2.7472,  2.6530,  3.5736,  3.0584,  3.3304,  3.3748,  3.3871
     ./)
      r0ab(1051:1120)=(/
     .   3.2028,  3.1296,  3.0214,  3.8796,  3.3337,  3.2492,  3.1883
     .,  3.1802,  3.4050,  3.0756,  3.0478,  3.0322,  3.0323,  3.0163
     .,  3.0019,  3.3145,  3.4050,  3.4656,  3.3021,  3.2433,  3.1453
     .,  3.9991,  3.5017,  3.4141,  3.3520,  3.5583,  3.5251,  3.4243
     .,  3.3851,  3.3662,  3.3525,  2.7846,  2.5324,  3.4652,  2.8759
     .,  3.0051,  3.1692,  2.9273,  2.7615,  2.7164,  2.6212,  3.5744
     .,  3.0275,  3.3249,  3.3627,  3.3686,  3.1669,  3.0584,  2.9915
     .,  3.8773,  3.3099,  3.2231,  3.1600,  3.1520,  3.4023,  3.0426
     .,  3.0099,  2.9920,  2.9809,  2.9800,  2.9646,  3.3068,  3.3930
     .,  3.4486,  3.2682,  3.1729,  3.1168,  3.9952,  3.4796,  3.3901
     ./)
      r0ab(1121:1190)=(/
     .   3.3255,  3.5530,  3.5183,  3.4097,  3.3683,  3.3492,  3.3360
     .,  3.3308,  2.5424,  2.6601,  3.2555,  3.2807,  3.1384,  3.1737
     .,  2.9397,  2.8429,  2.8492,  2.7225,  3.3875,  3.4910,  3.4520
     .,  3.3608,  3.3036,  3.2345,  3.2999,  3.1487,  3.7409,  3.8392
     .,  3.7148,  3.6439,  3.6182,  3.1753,  3.5210,  3.4639,  3.4265
     .,  3.4075,  3.3828,  3.3474,  3.4071,  3.3754,  3.3646,  3.3308
     .,  3.4393,  3.2993,  3.8768,  3.9891,  3.8310,  3.7483,  3.3417
     .,  3.3019,  3.2250,  3.1832,  3.1578,  3.1564,  3.1224,  3.4620
     .,  2.9743,  2.8058,  3.4830,  3.3474,  3.6863,  3.3617,  3.1608
     .,  3.0069,  2.9640,  2.8427,  3.5885,  3.5219,  4.1314,  3.8120
     ./)
      r0ab(1191:1260)=(/
     .   3.6015,  3.4502,  3.3498,  3.2777,  3.8635,  3.8232,  3.8486
     .,  3.7215,  3.6487,  3.4724,  3.5627,  3.5087,  3.4757,  3.4517
     .,  3.4423,  3.4139,  4.1028,  3.8388,  3.6745,  3.5562,  3.4806
     .,  3.4272,  4.0182,  3.9991,  4.0007,  3.9282,  3.7238,  3.6498
     .,  3.5605,  3.5211,  3.5009,  3.4859,  3.4785,  3.5621,  4.2623
     .,  3.0775,  2.8275,  4.0181,  3.3385,  3.5379,  3.5036,  3.2589
     .,  3.0804,  3.0094,  2.9003,  4.0869,  3.5088,  3.9105,  3.9833
     .,  3.7176,  3.5323,  3.4102,  3.3227,  4.2702,  4.0888,  3.7560
     .,  3.7687,  3.6681,  3.6405,  3.5569,  3.4990,  3.4659,  3.4433
     .,  3.4330,  3.4092,  3.8867,  4.0190,  3.7961,  3.6412,  3.5405
     ./)
      r0ab(1261:1330)=(/
     .   3.4681,  4.3538,  4.2136,  3.9381,  3.8912,  3.9681,  3.7909
     .,  3.6774,  3.6262,  3.5999,  3.5823,  3.5727,  3.5419,  4.0245
     .,  4.1874,  3.0893,  2.7917,  3.7262,  3.3518,  3.4241,  3.5433
     .,  3.2773,  3.0890,  2.9775,  2.9010,  3.8048,  3.5362,  3.7746
     .,  3.7911,  3.7511,  3.5495,  3.4149,  3.3177,  4.0129,  3.8370
     .,  3.7739,  3.7125,  3.7152,  3.7701,  3.5813,  3.5187,  3.4835
     .,  3.4595,  3.4439,  3.4242,  3.7476,  3.8239,  3.8346,  3.6627
     .,  3.5479,  3.4639,  4.1026,  3.9733,  3.9292,  3.8667,  3.9513
     .,  3.8959,  3.7698,  3.7089,  3.6765,  3.6548,  3.6409,  3.5398
     .,  3.8759,  3.9804,  4.0150,  2.9091,  2.7638,  3.5066,  3.3377
     ./)
      r0ab(1331:1400)=(/
     .   3.3481,  3.2633,  3.1810,  3.1428,  2.9872,  2.8837,  3.5929
     .,  3.5183,  3.6729,  3.6596,  3.6082,  3.5927,  3.4224,  3.2997
     .,  3.8190,  4.1865,  4.1114,  4.0540,  3.6325,  3.5697,  3.5561
     .,  3.5259,  3.4901,  3.4552,  3.4315,  3.4091,  3.6438,  3.6879
     .,  3.6832,  3.7043,  3.5557,  3.4466,  3.9203,  4.2919,  4.2196
     .,  4.1542,  3.7573,  3.7039,  3.6546,  3.6151,  3.5293,  3.4849
     .,  3.4552,  3.5192,  3.7673,  3.8359,  3.8525,  3.8901,  2.7806
     .,  2.7209,  3.3812,  3.4958,  3.2913,  3.1888,  3.0990,  3.0394
     .,  2.9789,  2.8582,  3.4716,  3.6883,  3.6105,  3.5704,  3.5059
     .,  3.4619,  3.4138,  3.2742,  3.7080,  3.9773,  3.9010,  3.8409
     ./)
      r0ab(1401:1470)=(/
     .   3.7944,  3.4465,  3.7235,  3.6808,  3.6453,  3.6168,  3.5844
     .,  3.5576,  3.5772,  3.5959,  3.5768,  3.5678,  3.5486,  3.4228
     .,  3.8107,  4.0866,  4.0169,  3.9476,  3.6358,  3.5800,  3.5260
     .,  3.4838,  3.4501,  3.4204,  3.3553,  3.6487,  3.6973,  3.7398
     .,  3.7405,  3.7459,  3.7380,  2.6848,  2.6740,  3.2925,  3.3386
     .,  3.2473,  3.1284,  3.0301,  2.9531,  2.9602,  2.8272,  3.3830
     .,  3.5358,  3.5672,  3.5049,  3.4284,  3.3621,  3.3001,  3.2451
     .,  3.6209,  3.8299,  3.7543,  3.6920,  3.6436,  3.3598,  3.5701
     .,  3.5266,  3.4904,  3.4590,  3.4364,  3.4077,  3.5287,  3.5280
     .,  3.4969,  3.4650,  3.4304,  3.3963,  3.7229,  3.9402,  3.8753
     ./)
      r0ab(1471:1540)=(/
     .   3.8035,  3.5499,  3.4913,  3.4319,  3.3873,  3.3520,  3.3209
     .,  3.2948,  3.5052,  3.6465,  3.6696,  3.6577,  3.6388,  3.6142
     .,  3.5889,  3.3968,  3.0122,  4.2241,  3.7887,  4.0049,  3.5384
     .,  3.2698,  3.1083,  2.9917,  2.9057,  4.3340,  3.9900,  4.6588
     .,  4.1278,  3.8125,  3.6189,  3.4851,  3.3859,  4.6531,  4.3134
     .,  4.2258,  4.1309,  4.0692,  4.0944,  3.9850,  3.9416,  3.9112
     .,  3.8873,  3.8736,  3.8473,  4.6027,  4.1538,  3.8994,  3.7419
     .,  3.6356,  3.5548,  4.8353,  4.5413,  4.3891,  4.3416,  4.3243
     .,  4.2753,  4.2053,  4.1790,  4.1685,  4.1585,  4.1536,  4.0579
     .,  4.1980,  4.4564,  4.2192,  4.0528,  3.9489,  3.8642,  5.0567
     ./)
      r0ab(1541:1610)=(/
     .   3.0630,  3.3271,  4.0432,  4.0046,  4.1555,  3.7426,  3.5130
     .,  3.5174,  3.2884,  3.1378,  4.1894,  4.2321,  4.1725,  4.1833
     .,  3.8929,  4.0544,  3.8118,  3.6414,  4.6373,  4.6268,  4.4750
     .,  4.4134,  4.3458,  3.8582,  4.2583,  4.1898,  4.1562,  4.1191
     .,  4.1069,  4.0639,  4.1257,  4.1974,  3.9532,  4.1794,  3.9660
     .,  3.8130,  4.8160,  4.8272,  4.6294,  4.5840,  4.0770,  4.0088
     .,  3.9103,  3.8536,  3.8324,  3.7995,  3.7826,  4.2294,  4.3380
     .,  4.4352,  4.1933,  4.4580,  4.2554,  4.1072,  5.0454,  5.1814
     .,  3.0632,  3.2662,  3.6432,  3.8088,  3.7910,  3.7381,  3.5093
     .,  3.5155,  3.3047,  3.1681,  3.7871,  3.9924,  4.0637,  4.1382
     ./)
      r0ab(1611:1680)=(/
     .   3.8591,  4.0164,  3.7878,  3.6316,  4.1741,  4.3166,  4.2395
     .,  4.1831,  4.1107,  3.5857,  4.0270,  3.9676,  3.9463,  3.9150
     .,  3.9021,  3.8708,  4.0240,  4.1551,  3.9108,  4.1337,  3.9289
     .,  3.7873,  4.3666,  4.5080,  4.4232,  4.3155,  3.8461,  3.8007
     .,  3.6991,  3.6447,  3.6308,  3.5959,  3.5749,  4.0359,  4.3124
     .,  4.3539,  4.1122,  4.3772,  4.1785,  4.0386,  4.7004,  4.8604
     .,  4.6261,  2.9455,  3.2470,  3.6108,  3.8522,  3.6625,  3.6598
     .,  3.4411,  3.4660,  3.2415,  3.0944,  3.7514,  4.0397,  3.9231
     .,  4.0561,  3.7860,  3.9845,  3.7454,  3.5802,  4.1366,  4.3581
     .,  4.2351,  4.2011,  4.1402,  3.5381,  4.0653,  4.0093,  3.9883
     ./)
      r0ab(1681:1750)=(/
     .   3.9570,  3.9429,  3.9112,  3.8728,  4.0682,  3.8351,  4.1054
     .,  3.8928,  3.7445,  4.3415,  4.5497,  4.3833,  4.3122,  3.8051
     .,  3.7583,  3.6622,  3.6108,  3.5971,  3.5628,  3.5408,  4.0780
     .,  4.0727,  4.2836,  4.0553,  4.3647,  4.1622,  4.0178,  4.5802
     .,  4.9125,  4.5861,  4.6201,  2.9244,  3.2241,  3.5848,  3.8293
     .,  3.6395,  3.6400,  3.4204,  3.4499,  3.2253,  3.0779,  3.7257
     .,  4.0170,  3.9003,  4.0372,  3.7653,  3.9672,  3.7283,  3.5630
     .,  4.1092,  4.3347,  4.2117,  4.1793,  4.1179,  3.5139,  4.0426
     .,  3.9867,  3.9661,  3.9345,  3.9200,  3.8883,  3.8498,  4.0496
     .,  3.8145,  4.0881,  3.8756,  3.7271,  4.3128,  4.5242,  4.3578
     ./)
      r0ab(1751:1820)=(/
     .   4.2870,  3.7796,  3.7318,  3.6364,  3.5854,  3.5726,  3.5378
     .,  3.5155,  4.0527,  4.0478,  4.2630,  4.0322,  4.3449,  4.1421
     .,  3.9975,  4.5499,  4.8825,  4.5601,  4.5950,  4.5702,  2.9046
     .,  3.2044,  3.5621,  3.8078,  3.6185,  3.6220,  3.4019,  3.4359
     .,  3.2110,  3.0635,  3.7037,  3.9958,  3.8792,  4.0194,  3.7460
     .,  3.9517,  3.7128,  3.5474,  4.0872,  4.3138,  4.1906,  4.1593
     .,  4.0973,  3.4919,  4.0216,  3.9657,  3.9454,  3.9134,  3.8986
     .,  3.8669,  3.8289,  4.0323,  3.7954,  4.0725,  3.8598,  3.7113
     .,  4.2896,  4.5021,  4.3325,  4.2645,  3.7571,  3.7083,  3.6136
     .,  3.5628,  3.5507,  3.5155,  3.4929,  4.0297,  4.0234,  4.2442
     ./)
      r0ab(1821:1890)=(/
     .   4.0112,  4.3274,  4.1240,  3.9793,  4.5257,  4.8568,  4.5353
     .,  4.5733,  4.5485,  4.5271,  2.8878,  3.1890,  3.5412,  3.7908
     .,  3.5974,  3.6078,  3.3871,  3.4243,  3.1992,  3.0513,  3.6831
     .,  3.9784,  3.8579,  4.0049,  3.7304,  3.9392,  3.7002,  3.5347
     .,  4.0657,  4.2955,  4.1705,  4.1424,  4.0800,  3.4717,  4.0043
     .,  3.9485,  3.9286,  3.8965,  3.8815,  3.8500,  3.8073,  4.0180
     .,  3.7796,  4.0598,  3.8470,  3.6983,  4.2678,  4.4830,  4.3132
     .,  4.2444,  3.7370,  3.6876,  3.5935,  3.5428,  3.5314,  3.4958
     .,  3.4730,  4.0117,  4.0043,  4.2287,  3.9939,  4.3134,  4.1096
     .,  3.9646,  4.5032,  4.8356,  4.5156,  4.5544,  4.5297,  4.5083
     ./)
      r0ab(1891:1960)=(/
     .   4.4896,  2.8709,  3.1737,  3.5199,  3.7734,  3.5802,  3.5934
     .,  3.3724,  3.4128,  3.1877,  3.0396,  3.6624,  3.9608,  3.8397
     .,  3.9893,  3.7145,  3.9266,  3.6877,  3.5222,  4.0448,  4.2771
     .,  4.1523,  4.1247,  4.0626,  3.4530,  3.9866,  3.9310,  3.9115
     .,  3.8792,  3.8641,  3.8326,  3.7892,  4.0025,  3.7636,  4.0471
     .,  3.8343,  3.6854,  4.2464,  4.4635,  4.2939,  4.2252,  3.7169
     .,  3.6675,  3.5739,  3.5235,  3.5126,  3.4768,  3.4537,  3.9932
     .,  3.9854,  4.2123,  3.9765,  4.2992,  4.0951,  3.9500,  4.4811
     .,  4.8135,  4.4959,  4.5351,  4.5105,  4.4891,  4.4705,  4.4515
     .,  2.8568,  3.1608,  3.5050,  3.7598,  3.5665,  3.5803,  3.3601
     ./)
      r0ab(1961:2030)=(/
     .   3.4031,  3.1779,  3.0296,  3.6479,  3.9471,  3.8262,  3.9773
     .,  3.7015,  3.9162,  3.6771,  3.5115,  4.0306,  4.2634,  4.1385
     .,  4.1116,  4.0489,  3.4366,  3.9732,  3.9176,  3.8983,  3.8659
     .,  3.8507,  3.8191,  3.7757,  3.9907,  3.7506,  4.0365,  3.8235
     .,  3.6745,  4.2314,  4.4490,  4.2792,  4.2105,  3.7003,  3.6510
     .,  3.5578,  3.5075,  3.4971,  3.4609,  3.4377,  3.9788,  3.9712
     .,  4.1997,  3.9624,  4.2877,  4.0831,  3.9378,  4.4655,  4.7974
     .,  4.4813,  4.5209,  4.4964,  4.4750,  4.4565,  4.4375,  4.4234
     .,  2.6798,  3.0151,  3.2586,  3.5292,  3.5391,  3.4902,  3.2887
     .,  3.3322,  3.1228,  2.9888,  3.4012,  3.7145,  3.7830,  3.6665
     ./)
      r0ab(2031:2100)=(/
     .   3.5898,  3.8077,  3.5810,  3.4265,  3.7726,  4.0307,  3.9763
     .,  3.8890,  3.8489,  3.2706,  3.7595,  3.6984,  3.6772,  3.6428
     .,  3.6243,  3.5951,  3.7497,  3.6775,  3.6364,  3.9203,  3.7157
     .,  3.5746,  3.9494,  4.2076,  4.1563,  4.0508,  3.5329,  3.4780
     .,  3.3731,  3.3126,  3.2846,  3.2426,  3.2135,  3.7491,  3.9006
     .,  3.8332,  3.8029,  4.1436,  3.9407,  3.7998,  4.1663,  4.5309
     .,  4.3481,  4.2911,  4.2671,  4.2415,  4.2230,  4.2047,  4.1908
     .,  4.1243,  2.5189,  2.9703,  3.3063,  3.6235,  3.4517,  3.3989
     .,  3.2107,  3.2434,  3.0094,  2.8580,  3.4253,  3.8157,  3.7258
     .,  3.6132,  3.5297,  3.7566,  3.5095,  3.3368,  3.7890,  4.1298
     ./)
      r0ab(2101:2170)=(/
     .   4.0190,  3.9573,  3.9237,  3.2677,  3.8480,  3.8157,  3.7656
     .,  3.7317,  3.7126,  3.6814,  3.6793,  3.6218,  3.5788,  3.8763
     .,  3.6572,  3.5022,  3.9737,  4.3255,  4.1828,  4.1158,  3.5078
     .,  3.4595,  3.3600,  3.3088,  3.2575,  3.2164,  3.1856,  3.8522
     .,  3.8665,  3.8075,  3.7772,  4.1391,  3.9296,  3.7772,  4.2134
     .,  4.7308,  4.3787,  4.3894,  4.3649,  4.3441,  4.3257,  4.3073
     .,  4.2941,  4.1252,  4.2427,  3.0481,  2.9584,  3.6919,  3.5990
     .,  3.8881,  3.4209,  3.1606,  3.1938,  2.9975,  2.8646,  3.8138
     .,  3.7935,  3.7081,  3.9155,  3.5910,  3.4808,  3.4886,  3.3397
     .,  4.1336,  4.1122,  3.9888,  3.9543,  3.8917,  3.5894,  3.8131
     ./)
      r0ab(2171:2240)=(/
     .   3.7635,  3.7419,  3.7071,  3.6880,  3.6574,  3.6546,  3.9375
     .,  3.6579,  3.5870,  3.6361,  3.5039,  4.3149,  4.2978,  4.1321
     .,  4.1298,  3.8164,  3.7680,  3.7154,  3.6858,  3.6709,  3.6666
     .,  3.6517,  3.8174,  3.8608,  4.1805,  3.9102,  3.8394,  3.8968
     .,  3.7673,  4.5274,  4.6682,  4.3344,  4.3639,  4.3384,  4.3162
     .,  4.2972,  4.2779,  4.2636,  4.0253,  4.1168,  4.1541,  2.8136
     .,  3.0951,  3.4635,  3.6875,  3.4987,  3.5183,  3.2937,  3.3580
     .,  3.1325,  2.9832,  3.6078,  3.8757,  3.7616,  3.9222,  3.6370
     .,  3.8647,  3.6256,  3.4595,  3.9874,  4.1938,  4.0679,  4.0430
     .,  3.9781,  3.3886,  3.9008,  3.8463,  3.8288,  3.7950,  3.7790
     ./)
      r0ab(2241:2310)=(/
     .   3.7472,  3.7117,  3.9371,  3.6873,  3.9846,  3.7709,  3.6210
     .,  4.1812,  4.3750,  4.2044,  4.1340,  3.6459,  3.5929,  3.5036
     .,  3.4577,  3.4528,  3.4146,  3.3904,  3.9014,  3.9031,  4.1443
     .,  3.8961,  4.2295,  4.0227,  3.8763,  4.4086,  4.7097,  4.4064
     .,  4.4488,  4.4243,  4.4029,  4.3842,  4.3655,  4.3514,  4.1162
     .,  4.2205,  4.1953,  4.2794,  2.8032,  3.0805,  3.4519,  3.6700
     .,  3.4827,  3.5050,  3.2799,  3.3482,  3.1233,  2.9747,  3.5971
     .,  3.8586,  3.7461,  3.9100,  3.6228,  3.8535,  3.6147,  3.4490
     .,  3.9764,  4.1773,  4.0511,  4.0270,  3.9614,  3.3754,  3.8836
     .,  3.8291,  3.8121,  3.7780,  3.7619,  3.7300,  3.6965,  3.9253
     ./)
      r0ab(2311:2380)=(/
     .   3.6734,  3.9733,  3.7597,  3.6099,  4.1683,  4.3572,  4.1862
     .,  4.1153,  3.6312,  3.5772,  3.4881,  3.4429,  3.4395,  3.4009
     .,  3.3766,  3.8827,  3.8868,  4.1316,  3.8807,  4.2164,  4.0092
     .,  3.8627,  4.3936,  4.6871,  4.3882,  4.4316,  4.4073,  4.3858
     .,  4.3672,  4.3485,  4.3344,  4.0984,  4.2036,  4.1791,  4.2622
     .,  4.2450,  2.7967,  3.0689,  3.4445,  3.6581,  3.4717,  3.4951
     .,  3.2694,  3.3397,  3.1147,  2.9661,  3.5898,  3.8468,  3.7358
     .,  3.9014,  3.6129,  3.8443,  3.6054,  3.4396,  3.9683,  4.1656
     .,  4.0394,  4.0158,  3.9498,  3.3677,  3.8718,  3.8164,  3.8005
     .,  3.7662,  3.7500,  3.7181,  3.6863,  3.9170,  3.6637,  3.9641
     ./)
      r0ab(2381:2450)=(/
     .   3.7503,  3.6004,  4.1590,  4.3448,  4.1739,  4.1029,  3.6224
     .,  3.5677,  3.4785,  3.4314,  3.4313,  3.3923,  3.3680,  3.8698
     .,  3.8758,  4.1229,  3.8704,  4.2063,  3.9987,  3.8519,  4.3832
     .,  4.6728,  4.3759,  4.4195,  4.3952,  4.3737,  4.3551,  4.3364
     .,  4.3223,  4.0861,  4.1911,  4.1676,  4.2501,  4.2329,  4.2208
     .,  2.7897,  3.0636,  3.4344,  3.6480,  3.4626,  3.4892,  3.2626
     .,  3.3344,  3.1088,  2.9597,  3.5804,  3.8359,  3.7251,  3.8940
     .,  3.6047,  3.8375,  3.5990,  3.4329,  3.9597,  4.1542,  4.0278
     .,  4.0048,  3.9390,  3.3571,  3.8608,  3.8056,  3.7899,  3.7560
     .,  3.7400,  3.7081,  3.6758,  3.9095,  3.6552,  3.9572,  3.7436
     ./)
      r0ab(2451:2520)=(/
     .   3.5933,  4.1508,  4.3337,  4.1624,  4.0916,  3.6126,  3.5582
     .,  3.4684,  3.4212,  3.4207,  3.3829,  3.3586,  3.8604,  3.8658
     .,  4.1156,  3.8620,  4.1994,  3.9917,  3.8446,  4.3750,  4.6617
     .,  4.3644,  4.4083,  4.3840,  4.3625,  4.3439,  4.3253,  4.3112
     .,  4.0745,  4.1807,  4.1578,  4.2390,  4.2218,  4.2097,  4.1986
     .,  2.8395,  3.0081,  3.3171,  3.4878,  3.5360,  3.5145,  3.2809
     .,  3.3307,  3.1260,  2.9940,  3.4741,  3.6675,  3.7832,  3.6787
     .,  3.6156,  3.8041,  3.5813,  3.4301,  3.8480,  3.9849,  3.9314
     .,  3.8405,  3.8029,  3.2962,  3.7104,  3.6515,  3.6378,  3.6020
     .,  3.5849,  3.5550,  3.7494,  3.6893,  3.6666,  3.9170,  3.7150
     ./)
      r0ab(2521:2590)=(/
     .   3.5760,  4.0268,  4.1596,  4.1107,  3.9995,  3.5574,  3.5103
     .,  3.4163,  3.3655,  3.3677,  3.3243,  3.2975,  3.7071,  3.9047
     .,  3.8514,  3.8422,  3.8022,  3.9323,  3.7932,  4.2343,  4.4583
     .,  4.3115,  4.2457,  4.2213,  4.1945,  4.1756,  4.1569,  4.1424
     .,  4.0620,  4.0494,  3.9953,  4.0694,  4.0516,  4.0396,  4.0280
     .,  4.0130,  2.9007,  2.9674,  3.8174,  3.5856,  3.6486,  3.5339
     .,  3.2832,  3.3154,  3.1144,  2.9866,  3.9618,  3.8430,  3.9980
     .,  3.8134,  3.6652,  3.7985,  3.5756,  3.4207,  4.4061,  4.2817
     .,  4.1477,  4.0616,  3.9979,  3.6492,  3.8833,  3.8027,  3.7660
     .,  3.7183,  3.6954,  3.6525,  3.9669,  3.8371,  3.7325,  3.9160
     ./)
      r0ab(2591:2660)=(/
     .   3.7156,  3.5714,  4.6036,  4.4620,  4.3092,  4.2122,  3.8478
     .,  3.7572,  3.6597,  3.5969,  3.5575,  3.5386,  3.5153,  3.7818
     .,  4.1335,  4.0153,  3.9177,  3.8603,  3.9365,  3.7906,  4.7936
     .,  4.7410,  4.5461,  4.5662,  4.5340,  4.5059,  4.4832,  4.4604
     .,  4.4429,  4.2346,  4.4204,  4.3119,  4.3450,  4.3193,  4.3035
     .,  4.2933,  4.1582,  4.2450,  2.8559,  2.9050,  3.8325,  3.5442
     .,  3.5077,  3.4905,  3.2396,  3.2720,  3.0726,  2.9467,  3.9644
     .,  3.8050,  3.8981,  3.7762,  3.6216,  3.7531,  3.5297,  3.3742
     .,  4.3814,  4.2818,  4.1026,  4.0294,  3.9640,  3.6208,  3.8464
     .,  3.7648,  3.7281,  3.6790,  3.6542,  3.6117,  3.8650,  3.8010
     ./)
      r0ab(2661:2730)=(/
     .   3.6894,  3.8713,  3.6699,  3.5244,  4.5151,  4.4517,  4.2538
     .,  4.1483,  3.8641,  3.7244,  3.6243,  3.5589,  3.5172,  3.4973
     .,  3.4715,  3.7340,  4.0316,  3.9958,  3.8687,  3.8115,  3.8862
     .,  3.7379,  4.7091,  4.7156,  4.5199,  4.5542,  4.5230,  4.4959
     .,  4.4750,  4.4529,  4.4361,  4.1774,  4.3774,  4.2963,  4.3406
     .,  4.3159,  4.3006,  4.2910,  4.1008,  4.1568,  4.0980,  2.8110
     .,  2.8520,  3.7480,  3.5105,  3.4346,  3.3461,  3.1971,  3.2326
     .,  3.0329,  2.9070,  3.8823,  3.7928,  3.8264,  3.7006,  3.5797
     .,  3.7141,  3.4894,  3.3326,  4.3048,  4.2217,  4.0786,  3.9900
     .,  3.9357,  3.6331,  3.8333,  3.7317,  3.6957,  3.6460,  3.6197
     ./)
      r0ab(2731:2800)=(/
     .   3.5779,  3.7909,  3.7257,  3.6476,  3.5729,  3.6304,  3.4834
     .,  4.4368,  4.3921,  4.2207,  4.1133,  3.8067,  3.7421,  3.6140
     .,  3.5491,  3.5077,  3.4887,  3.4623,  3.6956,  3.9568,  3.8976
     .,  3.8240,  3.7684,  3.8451,  3.6949,  4.6318,  4.6559,  4.4533
     .,  4.4956,  4.4641,  4.4366,  4.4155,  4.3936,  4.3764,  4.1302
     .,  4.3398,  4.2283,  4.2796,  4.2547,  4.2391,  4.2296,  4.0699
     .,  4.1083,  4.0319,  3.9855,  2.7676,  2.8078,  3.6725,  3.4804
     .,  3.3775,  3.2411,  3.1581,  3.1983,  2.9973,  2.8705,  3.8070
     .,  3.7392,  3.7668,  3.6263,  3.5402,  3.6807,  3.4545,  3.2962
     .,  4.2283,  4.1698,  4.0240,  3.9341,  3.8711,  3.5489,  3.7798
     ./)
      r0ab(2801:2870)=(/
     .   3.7000,  3.6654,  3.6154,  3.5882,  3.5472,  3.7289,  3.6510
     .,  3.6078,  3.5355,  3.5963,  3.4480,  4.3587,  4.3390,  4.1635
     .,  4.0536,  3.7193,  3.6529,  3.5512,  3.4837,  3.4400,  3.4191
     .,  3.3891,  3.6622,  3.8934,  3.8235,  3.7823,  3.7292,  3.8106
     .,  3.6589,  4.5535,  4.6013,  4.3961,  4.4423,  4.4109,  4.3835
     .,  4.3625,  4.3407,  4.3237,  4.0863,  4.2835,  4.1675,  4.2272
     .,  4.2025,  4.1869,  4.1774,  4.0126,  4.0460,  3.9815,  3.9340
     .,  3.8955,  2.6912,  2.7604,  3.6037,  3.4194,  3.3094,  3.1710
     .,  3.0862,  3.1789,  2.9738,  2.8427,  3.7378,  3.6742,  3.6928
     .,  3.5512,  3.4614,  3.4087,  3.4201,  3.2607,  4.1527,  4.0977
     ./)
      r0ab(2871:2940)=(/
     .   3.9523,  3.8628,  3.8002,  3.4759,  3.7102,  3.6466,  3.6106
     .,  3.5580,  3.5282,  3.4878,  3.6547,  3.5763,  3.5289,  3.5086
     .,  3.5593,  3.4099,  4.2788,  4.2624,  4.0873,  3.9770,  3.6407
     .,  3.5743,  3.5178,  3.4753,  3.3931,  3.3694,  3.3339,  3.6002
     .,  3.8164,  3.7478,  3.7028,  3.6952,  3.7669,  3.6137,  4.4698
     .,  4.5488,  4.3168,  4.3646,  4.3338,  4.3067,  4.2860,  4.2645
     .,  4.2478,  4.0067,  4.2349,  4.0958,  4.1543,  4.1302,  4.1141
     .,  4.1048,  3.9410,  3.9595,  3.8941,  3.8465,  3.8089,  3.7490
     .,  2.7895,  2.5849,  3.6484,  3.0162,  3.1267,  3.2125,  3.0043
     .,  2.9572,  2.8197,  2.7261,  3.7701,  3.2446,  3.5239,  3.4696
     ./)
      r0ab(2941:3010)=(/
     .   3.4261,  3.3508,  3.1968,  3.0848,  4.1496,  3.6598,  3.5111
     .,  3.4199,  3.3809,  3.5382,  3.2572,  3.2100,  3.1917,  3.1519
     .,  3.1198,  3.1005,  3.5071,  3.5086,  3.5073,  3.4509,  3.3120
     .,  3.2082,  4.2611,  3.8117,  3.6988,  3.5646,  3.6925,  3.6295
     .,  3.5383,  3.4910,  3.4625,  3.4233,  3.4007,  3.2329,  3.6723
     .,  3.6845,  3.6876,  3.6197,  3.4799,  3.3737,  4.4341,  4.0525
     .,  3.9011,  3.8945,  3.8635,  3.8368,  3.8153,  3.7936,  3.7758
     .,  3.4944,  3.4873,  3.9040,  3.7110,  3.6922,  3.6799,  3.6724
     .,  3.5622,  3.6081,  3.5426,  3.4922,  3.4498,  3.3984,  3.4456
     .,  2.7522,  2.5524,  3.5742,  2.9508,  3.0751,  3.0158,  2.9644
     ./)
      r0ab(3011:3080)=(/
     .   2.8338,  2.7891,  2.6933,  3.6926,  3.1814,  3.4528,  3.4186
     .,  3.3836,  3.2213,  3.1626,  3.0507,  4.0548,  3.5312,  3.4244
     .,  3.3409,  3.2810,  3.4782,  3.1905,  3.1494,  3.1221,  3.1128
     .,  3.0853,  3.0384,  3.4366,  3.4562,  3.4638,  3.3211,  3.2762
     .,  3.1730,  4.1632,  3.6825,  3.5822,  3.4870,  3.6325,  3.5740
     .,  3.4733,  3.4247,  3.3969,  3.3764,  3.3525,  3.1984,  3.5989
     .,  3.6299,  3.6433,  3.4937,  3.4417,  3.3365,  4.3304,  3.9242
     .,  3.7793,  3.7623,  3.7327,  3.7071,  3.6860,  3.6650,  3.6476
     .,  3.3849,  3.3534,  3.8216,  3.5870,  3.5695,  3.5584,  3.5508
     .,  3.4856,  3.5523,  3.4934,  3.4464,  3.4055,  3.3551,  3.3888
     ./)
      r0ab(3081:3150)=(/
     .   3.3525,  2.7202,  2.5183,  3.4947,  2.8731,  3.0198,  3.1457
     .,  2.9276,  2.7826,  2.7574,  2.6606,  3.6090,  3.0581,  3.3747
     .,  3.3677,  3.3450,  3.1651,  3.1259,  3.0147,  3.9498,  3.3857
     .,  3.2917,  3.2154,  3.1604,  3.4174,  3.0735,  3.0342,  3.0096
     .,  3.0136,  2.9855,  2.9680,  3.3604,  3.4037,  3.4243,  3.2633
     .,  3.1810,  3.1351,  4.0557,  3.5368,  3.4526,  3.3699,  3.5707
     .,  3.5184,  3.4085,  3.3595,  3.3333,  3.3143,  3.3041,  3.1094
     .,  3.5193,  3.5745,  3.6025,  3.4338,  3.3448,  3.2952,  4.2158
     .,  3.7802,  3.6431,  3.6129,  3.5853,  3.5610,  3.5406,  3.5204
     .,  3.5036,  3.2679,  3.2162,  3.7068,  3.4483,  3.4323,  3.4221
     ./)
      r0ab(3151:3220)=(/
     .   3.4138,  3.3652,  3.4576,  3.4053,  3.3618,  3.3224,  3.2711
     .,  3.3326,  3.2950,  3.2564,  2.5315,  2.6104,  3.2734,  3.2299
     .,  3.1090,  2.9942,  2.9159,  2.8324,  2.8350,  2.7216,  3.3994
     .,  3.4475,  3.4354,  3.3438,  3.2807,  3.2169,  3.2677,  3.1296
     .,  3.7493,  3.8075,  3.6846,  3.6104,  3.5577,  3.2052,  3.4803
     .,  3.4236,  3.3845,  3.3640,  3.3365,  3.3010,  3.3938,  3.3624
     .,  3.3440,  3.3132,  3.4035,  3.2754,  3.8701,  3.9523,  3.8018
     .,  3.7149,  3.3673,  3.3199,  3.2483,  3.2069,  3.1793,  3.1558
     .,  3.1395,  3.4097,  3.5410,  3.5228,  3.5116,  3.4921,  3.4781
     .,  3.4690,  4.0420,  4.1759,  4.0078,  4.0450,  4.0189,  3.9952
     ./)
      r0ab(3221:3290)=(/
     .   3.9770,  3.9583,  3.9434,  3.7217,  3.8228,  3.7826,  3.8640
     .,  3.8446,  3.8314,  3.8225,  3.6817,  3.7068,  3.6555,  3.6159
     .,  3.5831,  3.5257,  3.2133,  3.1689,  3.1196,  3.3599,  2.9852
     .,  2.7881,  3.5284,  3.3493,  3.6958,  3.3642,  3.1568,  3.0055
     .,  2.9558,  2.8393,  3.6287,  3.5283,  4.1511,  3.8259,  3.6066
     .,  3.4527,  3.3480,  3.2713,  3.9037,  3.8361,  3.8579,  3.7311
     .,  3.6575,  3.5176,  3.5693,  3.5157,  3.4814,  3.4559,  3.4445
     .,  3.4160,  4.1231,  3.8543,  3.6816,  3.5602,  3.4798,  3.4208
     .,  4.0542,  4.0139,  4.0165,  3.9412,  3.7698,  3.6915,  3.6043
     .,  3.5639,  3.5416,  3.5247,  3.5153,  3.5654,  4.2862,  4.0437
     ./)
      r0ab(3291:3360)=(/
     .   3.8871,  3.7741,  3.6985,  3.6413,  4.2345,  4.3663,  4.3257
     .,  4.0869,  4.0612,  4.0364,  4.0170,  3.9978,  3.9834,  3.9137
     .,  3.8825,  3.8758,  3.9143,  3.8976,  3.8864,  3.8768,  3.9190
     .,  4.1613,  4.0566,  3.9784,  3.9116,  3.8326,  3.7122,  3.6378
     .,  3.5576,  3.5457,  4.3127,  3.1160,  2.8482,  4.0739,  3.3599
     .,  3.5698,  3.5366,  3.2854,  3.1039,  2.9953,  2.9192,  4.1432
     .,  3.5320,  3.9478,  4.0231,  3.7509,  3.5604,  3.4340,  3.3426
     .,  4.3328,  3.8288,  3.7822,  3.7909,  3.6907,  3.6864,  3.5793
     .,  3.5221,  3.4883,  3.4649,  3.4514,  3.4301,  3.9256,  4.0596
     .,  3.8307,  3.6702,  3.5651,  3.4884,  4.4182,  4.2516,  3.9687
     ./)
      r0ab(3361:3430)=(/
     .   3.9186,  3.9485,  3.8370,  3.7255,  3.6744,  3.6476,  3.6295
     .,  3.6193,  3.5659,  4.0663,  4.2309,  4.0183,  3.8680,  3.7672
     .,  3.6923,  4.5240,  4.4834,  4.1570,  4.3204,  4.2993,  4.2804
     .,  4.2647,  4.2481,  4.2354,  3.8626,  3.8448,  4.2267,  4.1799
     .,  4.1670,  3.8738,  3.8643,  3.8796,  4.0575,  4.0354,  3.9365
     .,  3.8611,  3.7847,  3.7388,  3.6826,  3.6251,  3.5492,  4.0889
     .,  4.2764,  3.1416,  2.8325,  3.7735,  3.3787,  3.4632,  3.5923
     .,  3.3214,  3.1285,  3.0147,  2.9366,  3.8527,  3.5602,  3.8131
     .,  3.8349,  3.7995,  3.5919,  3.4539,  3.3540,  4.0654,  3.8603
     .,  3.7972,  3.7358,  3.7392,  3.8157,  3.6055,  3.5438,  3.5089
     ./)
      r0ab(3431:3500)=(/
     .   3.4853,  3.4698,  3.4508,  3.7882,  3.8682,  3.8837,  3.7055
     .,  3.5870,  3.5000,  4.1573,  4.0005,  3.9568,  3.8936,  3.9990
     .,  3.9433,  3.8172,  3.7566,  3.7246,  3.7033,  3.6900,  3.5697
     .,  3.9183,  4.0262,  4.0659,  3.8969,  3.7809,  3.6949,  4.2765
     .,  4.2312,  4.1401,  4.0815,  4.0580,  4.0369,  4.0194,  4.0017
     .,  3.9874,  3.8312,  3.8120,  3.9454,  3.9210,  3.9055,  3.8951
     .,  3.8866,  3.8689,  3.9603,  3.9109,  3.9122,  3.8233,  3.7438
     .,  3.7436,  3.6981,  3.6555,  3.5452,  3.9327,  4.0658,  4.1175
     .,  2.9664,  2.8209,  3.5547,  3.3796,  3.3985,  3.3164,  3.2364
     .,  3.1956,  3.0370,  2.9313,  3.6425,  3.5565,  3.7209,  3.7108
     ./)
      r0ab(3501:3570)=(/
     .   3.6639,  3.6484,  3.4745,  3.3492,  3.8755,  4.2457,  3.7758
     .,  3.7161,  3.6693,  3.6155,  3.5941,  3.5643,  3.5292,  3.4950
     .,  3.4720,  3.4503,  3.6936,  3.7392,  3.7388,  3.7602,  3.6078
     .,  3.4960,  3.9800,  4.3518,  4.2802,  3.8580,  3.8056,  3.7527
     .,  3.7019,  3.6615,  3.5768,  3.5330,  3.5038,  3.5639,  3.8192
     .,  3.8883,  3.9092,  3.9478,  3.7995,  3.6896,  4.1165,  4.5232
     .,  4.4357,  4.4226,  4.4031,  4.3860,  4.3721,  4.3580,  4.3466
     .,  4.2036,  4.2037,  3.8867,  4.2895,  4.2766,  4.2662,  4.2598
     .,  3.8408,  3.9169,  3.8681,  3.8250,  3.7855,  3.7501,  3.6753
     .,  3.5499,  3.4872,  3.5401,  3.8288,  3.9217,  3.9538,  4.0054
     ./)
      r0ab(3571:3640)=(/
     .   2.8388,  2.7890,  3.4329,  3.5593,  3.3488,  3.2486,  3.1615
     .,  3.1000,  3.0394,  2.9165,  3.5267,  3.7479,  3.6650,  3.6263
     .,  3.5658,  3.5224,  3.4762,  3.3342,  3.7738,  4.0333,  3.9568
     .,  3.8975,  3.8521,  3.4929,  3.7830,  3.7409,  3.7062,  3.6786
     .,  3.6471,  3.6208,  3.6337,  3.6519,  3.6363,  3.6278,  3.6110
     .,  3.4825,  3.8795,  4.1448,  4.0736,  4.0045,  3.6843,  3.6291
     .,  3.5741,  3.5312,  3.4974,  3.4472,  3.4034,  3.7131,  3.7557
     .,  3.7966,  3.8005,  3.8068,  3.8015,  3.6747,  4.0222,  4.3207
     .,  4.2347,  4.2191,  4.1990,  4.1811,  4.1666,  4.1521,  4.1401
     .,  3.9970,  3.9943,  3.9592,  4.0800,  4.0664,  4.0559,  4.0488
     ./)
      r0ab(3641:3710)=(/
     .   3.9882,  4.0035,  3.9539,  3.9138,  3.8798,  3.8355,  3.5359
     .,  3.4954,  3.3962,  3.5339,  3.7595,  3.8250,  3.8408,  3.8600
     .,  3.8644,  2.7412,  2.7489,  3.3374,  3.3950,  3.3076,  3.1910
     .,  3.0961,  3.0175,  3.0280,  2.8929,  3.4328,  3.5883,  3.6227
     .,  3.5616,  3.4894,  3.4241,  3.3641,  3.3120,  3.6815,  3.8789
     .,  3.8031,  3.7413,  3.6939,  3.4010,  3.6225,  3.5797,  3.5443
     .,  3.5139,  3.4923,  3.4642,  3.5860,  3.5849,  3.5570,  3.5257
     .,  3.4936,  3.4628,  3.7874,  3.9916,  3.9249,  3.8530,  3.5932
     .,  3.5355,  3.4757,  3.4306,  3.3953,  3.3646,  3.3390,  3.5637
     .,  3.7053,  3.7266,  3.7177,  3.6996,  3.6775,  3.6558,  3.9331
     ./)
      r0ab(3711:3780)=(/
     .   4.1655,  4.0879,  4.0681,  4.0479,  4.0299,  4.0152,  4.0006
     .,  3.9883,  3.8500,  3.8359,  3.8249,  3.9269,  3.9133,  3.9025
     .,  3.8948,  3.8422,  3.8509,  3.7990,  3.7570,  3.7219,  3.6762
     .,  3.4260,  3.3866,  3.3425,  3.5294,  3.7022,  3.7497,  3.7542
     .,  3.7494,  3.7370,  3.7216,  3.4155,  3.0522,  4.2541,  3.8218
     .,  4.0438,  3.5875,  3.3286,  3.1682,  3.0566,  2.9746,  4.3627
     .,  4.0249,  4.6947,  4.1718,  3.8639,  3.6735,  3.5435,  3.4479
     .,  4.6806,  4.3485,  4.2668,  4.1690,  4.1061,  4.1245,  4.0206
     .,  3.9765,  3.9458,  3.9217,  3.9075,  3.8813,  3.9947,  4.1989
     .,  3.9507,  3.7960,  3.6925,  3.6150,  4.8535,  4.5642,  4.4134
     ./)
      r0ab(3781:3850)=(/
     .   4.3688,  4.3396,  4.2879,  4.2166,  4.1888,  4.1768,  4.1660
     .,  4.1608,  4.0745,  4.2289,  4.4863,  4.2513,  4.0897,  3.9876
     .,  3.9061,  5.0690,  5.0446,  4.6186,  4.6078,  4.5780,  4.5538
     .,  4.5319,  4.5101,  4.4945,  4.1912,  4.2315,  4.5534,  4.4373
     .,  4.4224,  4.4120,  4.4040,  4.2634,  4.7770,  4.6890,  4.6107
     .,  4.5331,  4.4496,  4.4082,  4.3095,  4.2023,  4.0501,  4.2595
     .,  4.5497,  4.3056,  4.1506,  4.0574,  3.9725,  5.0796,  3.0548
     .,  3.3206,  3.8132,  3.9720,  3.7675,  3.7351,  3.5167,  3.5274
     .,  3.3085,  3.1653,  3.9500,  4.1730,  4.0613,  4.1493,  3.8823
     .,  4.0537,  3.8200,  3.6582,  4.3422,  4.5111,  4.3795,  4.3362
     ./)
      r0ab(3851:3920)=(/
     .   4.2751,  3.7103,  4.1973,  4.1385,  4.1129,  4.0800,  4.0647
     .,  4.0308,  4.0096,  4.1619,  3.9360,  4.1766,  3.9705,  3.8262
     .,  4.5348,  4.7025,  4.5268,  4.5076,  3.9562,  3.9065,  3.8119
     .,  3.7605,  3.7447,  3.7119,  3.6916,  4.1950,  4.2110,  4.3843
     .,  4.1631,  4.4427,  4.2463,  4.1054,  4.7693,  5.0649,  4.7365
     .,  4.7761,  4.7498,  4.7272,  4.7076,  4.6877,  4.6730,  4.4274
     .,  4.5473,  4.5169,  4.5975,  4.5793,  4.5667,  4.5559,  4.3804
     .,  4.6920,  4.6731,  4.6142,  4.5600,  4.4801,  4.0149,  3.8856
     .,  3.7407,  4.1545,  4.2253,  4.4229,  4.1923,  4.5022,  4.3059
     .,  4.1591,  4.7883,  4.9294,  3.3850,  3.4208,  3.7004,  3.8800
     ./)
      r0ab(3921:3990)=(/
     .   3.9886,  3.9040,  3.6719,  3.6547,  3.4625,  3.3370,  3.8394
     .,  4.0335,  4.2373,  4.3023,  4.0306,  4.1408,  3.9297,  3.7857
     .,  4.1907,  4.3230,  4.2664,  4.2173,  4.1482,  3.6823,  4.0711
     .,  4.0180,  4.0017,  3.9747,  3.9634,  3.9383,  4.1993,  4.3205
     .,  4.0821,  4.2547,  4.0659,  3.9359,  4.3952,  4.5176,  4.3888
     .,  4.3607,  3.9583,  3.9280,  3.8390,  3.7971,  3.7955,  3.7674
     .,  3.7521,  4.1062,  4.3633,  4.2991,  4.2767,  4.4857,  4.3039
     .,  4.1762,  4.6197,  4.8654,  4.6633,  4.5878,  4.5640,  4.5422
     .,  4.5231,  4.5042,  4.4901,  4.3282,  4.3978,  4.3483,  4.4202
     .,  4.4039,  4.3926,  4.3807,  4.2649,  4.6135,  4.5605,  4.5232
     ./)
      r0ab(3991:4060)=(/
     .   4.4676,  4.3948,  4.0989,  3.9864,  3.8596,  4.0942,  4.2720
     .,  4.3270,  4.3022,  4.5410,  4.3576,  4.2235,  4.6545,  4.7447
     .,  4.7043,  3.0942,  3.2075,  3.5152,  3.6659,  3.8289,  3.7459
     .,  3.5156,  3.5197,  3.3290,  3.2069,  3.6702,  3.8448,  4.0340
     .,  3.9509,  3.8585,  3.9894,  3.7787,  3.6365,  4.1425,  4.1618
     .,  4.0940,  4.0466,  3.9941,  3.5426,  3.8952,  3.8327,  3.8126
     .,  3.7796,  3.7635,  3.7356,  4.0047,  3.9655,  3.9116,  4.1010
     .,  3.9102,  3.7800,  4.2964,  4.3330,  4.2622,  4.2254,  3.8195
     .,  3.7560,  3.6513,  3.5941,  3.5810,  3.5420,  3.5178,  3.8861
     .,  4.1459,  4.1147,  4.0772,  4.3120,  4.1207,  3.9900,  4.4733
     ./)
      r0ab(4061:4130)=(/
     .   4.6157,  4.4580,  4.4194,  4.3954,  4.3739,  4.3531,  4.3343
     .,  4.3196,  4.2140,  4.2339,  4.1738,  4.2458,  4.2278,  4.2158
     .,  4.2039,  4.1658,  4.3595,  4.2857,  4.2444,  4.1855,  4.1122
     .,  3.7839,  3.6879,  3.5816,  3.8633,  4.1585,  4.1402,  4.1036
     .,  4.3694,  4.1735,  4.0368,  4.5095,  4.5538,  4.5240,  4.4252
     .,  3.0187,  3.1918,  3.5127,  3.6875,  3.7404,  3.6943,  3.4702
     .,  3.4888,  3.2914,  3.1643,  3.6669,  3.8724,  3.9940,  4.0816
     .,  3.8054,  3.9661,  3.7492,  3.6024,  4.0428,  4.1951,  4.1466
     .,  4.0515,  4.0075,  3.5020,  3.9158,  3.8546,  3.8342,  3.8008
     .,  3.7845,  3.7549,  3.9602,  3.8872,  3.8564,  4.0793,  3.8835
     ./)
      r0ab(4131:4200)=(/
     .   3.7495,  4.2213,  4.3704,  4.3300,  4.2121,  3.7643,  3.7130
     .,  3.6144,  3.5599,  3.5474,  3.5093,  3.4853,  3.9075,  4.1115
     .,  4.0473,  4.0318,  4.2999,  4.1050,  3.9710,  4.4320,  4.6706
     .,  4.5273,  4.4581,  4.4332,  4.4064,  4.3873,  4.3684,  4.3537
     .,  4.2728,  4.2549,  4.2032,  4.2794,  4.2613,  4.2491,  4.2375
     .,  4.2322,  4.3665,  4.3061,  4.2714,  4.2155,  4.1416,  3.7660
     .,  3.6628,  3.5476,  3.8790,  4.1233,  4.0738,  4.0575,  4.3575
     .,  4.1586,  4.0183,  4.4593,  4.5927,  4.4865,  4.3813,  4.4594
     .,  2.9875,  3.1674,  3.4971,  3.6715,  3.7114,  3.6692,  3.4446
     .,  3.4676,  3.2685,  3.1405,  3.6546,  3.8579,  3.9637,  4.0581
     ./)
      r0ab(4201:4270)=(/
     .   3.7796,  3.9463,  3.7275,  3.5792,  4.0295,  4.1824,  4.1247
     .,  4.0357,  3.9926,  3.4827,  3.9007,  3.8392,  3.8191,  3.7851
     .,  3.7687,  3.7387,  3.9290,  3.8606,  3.8306,  4.0601,  3.8625
     .,  3.7269,  4.2062,  4.3566,  4.3022,  4.1929,  3.7401,  3.6888
     .,  3.5900,  3.5350,  3.5226,  3.4838,  3.4594,  3.8888,  4.0813
     .,  4.0209,  4.0059,  4.2810,  4.0843,  3.9486,  4.4162,  4.6542
     .,  4.5005,  4.4444,  4.4196,  4.3933,  4.3741,  4.3552,  4.3406
     .,  4.2484,  4.2413,  4.1907,  4.2656,  4.2474,  4.2352,  4.2236
     .,  4.2068,  4.3410,  4.2817,  4.2479,  4.1921,  4.1182,  3.7346
     .,  3.6314,  3.5168,  3.8582,  4.0927,  4.0469,  4.0313,  4.3391
     ./)
      r0ab(4271:4340)=(/
     .   4.1381,  3.9962,  4.4429,  4.5787,  4.4731,  4.3588,  4.4270
     .,  4.3957,  2.9659,  3.1442,  3.4795,  3.6503,  3.6814,  3.6476
     .,  3.4222,  3.4491,  3.2494,  3.1209,  3.6324,  3.8375,  3.9397
     .,  3.8311,  3.7581,  3.9274,  3.7085,  3.5598,  4.0080,  4.1641
     .,  4.1057,  4.0158,  3.9726,  3.4667,  3.8802,  3.8188,  3.7989
     .,  3.7644,  3.7474,  3.7173,  3.9049,  3.8424,  3.8095,  4.0412
     .,  3.8436,  3.7077,  4.1837,  4.3366,  4.2816,  4.1686,  3.7293
     .,  3.6709,  3.5700,  3.5153,  3.5039,  3.4684,  3.4437,  3.8663
     .,  4.0575,  4.0020,  3.9842,  4.2612,  4.0643,  3.9285,  4.3928
     .,  4.6308,  4.4799,  4.4244,  4.3996,  4.3737,  4.3547,  4.3358
     ./)
      r0ab(4341:4410)=(/
     .   4.3212,  4.2275,  4.2216,  4.1676,  4.2465,  4.2283,  4.2161
     .,  4.2045,  4.1841,  4.3135,  4.2562,  4.2226,  4.1667,  4.0932
     .,  3.7134,  3.6109,  3.4962,  3.8352,  4.0688,  4.0281,  4.0099
     .,  4.3199,  4.1188,  3.9768,  4.4192,  4.5577,  4.4516,  4.3365
     .,  4.4058,  4.3745,  4.3539,  2.8763,  3.1294,  3.5598,  3.7465
     .,  3.5659,  3.5816,  3.3599,  3.4024,  3.1877,  3.0484,  3.7009
     .,  3.9451,  3.8465,  3.9873,  3.7079,  3.9083,  3.6756,  3.5150
     .,  4.0829,  4.2780,  4.1511,  4.1260,  4.0571,  3.4865,  3.9744
     .,  3.9150,  3.8930,  3.8578,  3.8402,  3.8073,  3.7977,  4.0036
     .,  3.7604,  4.0288,  3.8210,  3.6757,  4.2646,  4.4558,  4.2862
     ./)
      r0ab(4411:4465)=(/
     .   4.2122,  3.7088,  3.6729,  3.5800,  3.5276,  3.5165,  3.4783
     .,  3.4539,  3.9553,  3.9818,  4.2040,  3.9604,  4.2718,  4.0689
     .,  3.9253,  4.4869,  4.7792,  4.4918,  4.5342,  4.5090,  4.4868
     .,  4.4680,  4.4486,  4.4341,  4.2023,  4.3122,  4.2710,  4.3587
     .,  4.3407,  4.3281,  4.3174,  4.1499,  4.3940,  4.3895,  4.3260
     .,  4.2725,  4.1961,  3.7361,  3.6193,  3.4916,  3.9115,  3.9914
     .,  3.9809,  3.9866,  4.3329,  4.1276,  3.9782,  4.5097,  4.6769
     .,  4.5158,  4.3291,  4.3609,  4.3462,  4.3265,  4.4341
     ./)
      
      k=0
      do i=1,max_elem
         do j=1,i
            k=k+1
            r(i,j)=r0ab(k)/autoang
            r(j,i)=r0ab(k)/autoang
         enddo
      enddo

      end subroutine setr0ab

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C load DFT-D2 data
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine loadoldpar(autoang,max_elem,maxc,c6ab,r0ab,c6)
      implicit none  
      integer max_elem,maxc
      real*8 r0ab(max_elem,max_elem)
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)
      real*8 autoang           

      real*8 c6(86),r0(86)
      integer i,j

c the published radii in S.Grimme, J.Comput.Chem. 27, (2006), 1787-1799 (tab 1)
c refer to the following values multiplied by 1.1 (rs6 in this code)
c H, He
         r0(1:86) = (/ 0.91d0,0.92d0,
c Li-Ne
     .      0.75d0,1.28d0,1.35d0,1.32d0,1.27d0,1.22d0,1.17d0,1.13d0,
c Na-Ar
     .      1.04d0,1.24d0,1.49d0,1.56d0,1.55d0,1.53d0,1.49d0,1.45d0,
c K, Ca old
     .      1.35d0,1.34d0,
c Sc-Zn
     .      1.42d0,1.42d0,1.42d0,1.42d0,1.42d0,
     .      1.42d0,1.42d0,1.42d0,1.42d0,1.42d0,
c Ga-Kr
     .      1.50d0,1.57d0,1.60d0,1.61d0,1.59d0,1.57d0,
c Rb, Sr
     .      1.48d0,1.46d0,
c Y-Cd
     .      1.49d0,1.49d0,1.49d0,1.49d0,1.49d0,
     .      1.49d0,1.49d0,1.49d0,1.49d0,1.49d0,
c In, Sn, Sb, Te, I, Xe
     .      1.52d0,1.64d0,1.71d0,1.72d0,1.72d0,1.71d0,
c Cs,Ba,La,Ce-Lu
     .      1.638d0,1.602d0,1.564d0,1.594d0,1.594d0,1.594d0,1.594d0,
     .      1.594d0,1.594d0,1.594d0,1.594d0,1.594d0,1.594d0,1.594d0,
     .      1.594d0,1.594d0,1.594d0,
c Hf, Ta-Au
     .      1.625d0,1.611d0,1.611d0,1.611d0,1.611d0,1.611d0,1.611d0,
     .      1.611d0,
c Hg,Tl,Pb,Bi,Po,At,Rn
     .      1.598d0,1.805d0,1.767d0,1.725d0,1.823d0,1.810d0,1.749d0/)

       c6(1:86) = (/0.14d0,0.08d0,
     .   1.61d0,1.61d0,3.13d0,1.75d0,1.23d0,0.70d0,0.75d0,0.63d0,
     .   5.71d0,5.71d0,10.79d0,9.23d0,7.84d0,5.57d0,5.07d0,4.61d0,
     .   10.8d0,10.8d0,10.8d0,10.8d0,10.8d0,
     .   10.8d0,10.8d0,10.8d0,10.8d0,10.8d0,10.8d0,10.8d0,16.99d0,
     .   17.10d0,16.37d0,12.64d0,12.47d0,12.01d0,24.67d0,24.67d0,
     .   24.67d0,24.67d0,24.67d0,24.67d0,24.67d0,24.67d0,24.67d0,
     .   24.67d0,24.67d0,24.67d0,37.32d0,38.71d0,38.44d0,31.74d0,
     .   31.50d0,29.99d0,315.275d0,226.994d0,176.252d0,
     .  140.68d0,140.68d0,140.68d0,140.68d0,140.68d0,140.68d0,140.68d0,
     .  140.68d0,140.68d0,140.68d0,140.68d0,140.68d0,140.68d0,140.68d0,
     .  105.112d0,
     .  81.24d0,81.24d0,81.24d0,81.24d0,81.24d0,81.24d0,81.24d0,
     .  57.364d0,57.254d0,63.162d0,63.540d0,55.283d0,57.171d0,56.64d0 /)

      c6ab = -1
      do i=1,86
         do j=1,i
            r0ab(i,j)=(r0(i)+r0(j))/autoang
            r0ab(j,i)=(r0(i)+r0(j))/autoang
            c6ab(i,j,1,1,1)=dsqrt(c6(i)*c6(j))
            c6ab(j,i,1,1,1)=dsqrt(c6(i)*c6(j))
         enddo
      enddo

      end subroutine loadoldpar

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C read atomic data (radii, r4/r2)
C not used
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine rdatpar(fname,max_elem,val)
      implicit none             
      integer max_elem
      real*8 val(max_elem)
      character*(*) fname
 
      integer i
      real*8 dum1

      val = 0

      open(unit=142,file=fname)
 502  read(142,*,end=602) i,dum1
      if(i.gt.0)then
        if(i.gt.max_elem)call stoprun('wrong cardinal number (rdatpar)')
        val(i)=dum1         
      endif
      goto 502
 602  close(142)

      do i=1,max_elem
         if(val(i).lt.1.d-6)then
         write(*,*)'atom ',i
         call stoprun( 'atomic value missing' )
         endif
      enddo

      end subroutine rdatpar

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C read radii
C not used
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine rdab(fname,autoang,max_elem,ab)
      implicit none             
      real*8 autoang
      integer max_elem
      real*8 ab(max_elem,max_elem)
      character*(*) fname
 
      integer i,j
      real*8 dum1

      ab = 0

      open(unit=142,file=fname)
 502  read(142,*,end=602) dum1,i,j
      if(i.gt.0.and.j.gt.0)then
        if(i.gt.max_elem) call stoprun( 'wrong cardinal number (rdab)')
        if(j.gt.max_elem) call stoprun( 'wrong cardinal number (rdab)')
        ab(i,j)=dum1/autoang      
        ab(j,i)=dum1/autoang     
      endif
      goto 502
 602  close(142)

      end subroutine rdab

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
c read coordinates in au or Ang. if its a xmol file
c redone by S.E. to avoid some input errors. Looks for $coord, ang, bohr or number (xmol) in the first line
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

c      subroutine rdcoord(fname,n,xyz,iat,fix,fdum)
c      implicit none             
c      interface
c        subroutine parse(str,delims,args,nargs)
c          character(len=*),intent(inout) :: str
c          character(len=*),intent(in)  :: delims
c          character(len=*),dimension(:),intent(inout) :: args
c          integer, intent(out) :: nargs
c        end subroutine parse
c      end interface
c
c      real*8 xyz(3,*)
c      integer iat(*),n 
c      character*(*) fname
c      logical fix(n),fdum !fix:array of fixed coordinates, fdum: whether dummy atoms exist or not, they are ignored
c
c      real*8 floats(3),f
c      character*80 line
c      character*80 strings(3)
c      integer j,ich,cs,cf,ncheck
c
c      f=0.0d0
c      ich=142
c      open(unit=ich,file=fname)
c      ncheck=0
c      rewind(ich)
c      DO
c        read(ich,'(a)',end=200)line
c        if(line.ne."") exit
c      ENDDO
c
c      call readline(line,floats,strings,cs,cf)
c      if(cf.eq.1.and.floats(1).gt.0) then
c         f=1./0.52917726d0
c         read(ich,'(A)',end=200)line
c      else if (index(line,'$coord').ne.0) then
c         f=1.0d0
c      else if (index(line,'ang').ne.0) then
c         f=1./0.52917726d0
c      else if (index(line,'bohr').ne.0) then
c         f=1.0d0
c      endif
c      if(f.lt.1.0d0) then
c       call stoprun('Coordinate format not recognized!')
c      endif       
c      DO
c        read(ich,'(a)',end=200)line
c         if(index(line,'$redu').ne.0) exit
c         if(index(line,'$user').ne.0) exit
c         if(index(line,'$end' ).ne.0) exit
c         call readline(line,floats,strings,cs,cf)
c         if(cf.ne.3) cycle
c         call elem(strings(1),j)         
c         if(j.eq.0) then
c          fdum=.true.
c          cycle !ignores dummies and unknown elements
c         endif
c         ncheck=ncheck+1
c         xyz(1,ncheck)=floats(1)*f
c         xyz(2,ncheck)=floats(2)*f
c         xyz(3,ncheck)=floats(3)*f
c         iat(ncheck)=j
c         if(strings(2).ne.'')fix(ncheck)=.true.      !fixes coordinate if any other string than element spec is found
cc      write(*,321)floats(1:3),strings(1:3),j,fix(ncheck)   !debug printout
c
c      ENDDO
c
c 200  continue
c
c      if (n.ne.ncheck) then
c          write(*,*)n,'/=',ncheck
c          call stoprun('error reading coord file')
c      endif
c      close(ich)
c
cc  321 FORMAT(F20.10,F20.10,F20.10,1X,A3,1X,A3,1X,A3,I3,L) !debug output
c      end subroutine rdcoord


c      subroutine rdatomnumber(fname,n)
c      implicit none             
c      integer n 
c      character*(*) fname
c
c      real*8 floats(3),f
c      character*80 line
c      character*80 strings(3)
c      integer j,ich,cs,cf
c
c      f=0.0d0
c      ich=53
c      open(unit=ich,file=fname)
c      n=0
c 300  read(ich,'(a)',end=200)line
c      if(line.eq."") goto 300
c      call readline(line,floats,strings,cs,cf)
c      if(cf.eq.1.and.floats(1).gt.0.and.cs.eq.0) then
c         f=1./0.52917726d0
c!         write(*,*)floats(1)
c         n=int(floats(1))
c         close(ich)
c         return
c      else if (index(line,'$coord').ne.0) then
c         f=1.0d0
c      else if (index(line,'ang').ne.0) then
c         f=1./0.52917726d0
c      else if (index(line,'bohr').ne.0) then
c         f=1.0d0
c      endif
c      if(f.lt.1.0d0) then
c       call stoprun('Coordinate format not recognized!')
c      endif       
c      DO
c         read(ich,'(a)',end=200)line
c         if(index(line,'$redu').ne.0) exit
c         if(index(line,'$user').ne.0) exit
c         if(index(line,'$end' ).ne.0) exit
c         call readline(line,floats,strings,cs,cf)
c         if(cf.ne.3) exit
c         call elem(strings(1),j)         
c         if(j.eq.0) cycle
c         
c        
c         n=n+1
c      ENDDO
c
c 200  continue
c
c      close(ich)
c
cc  321 FORMAT(F20.10,F20.10,F20.10,1X,A3,1X,A3,1X,A3,I3,L) !debug output
c      end subroutine rdatomnumber




CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine outg(nat,g)
      implicit none
      integer nat,i
      real*8 g(3,nat)

      open(unit=142,file='dftd3_gradient')

!      write(*,*)'Gradient:' !Jonas
!      write(*,*)            !Jonas
      do i=1,nat
         write(142,'(3E22.14)')g(1:3,i)
!         write(*,'(3D22.14)')g(1:3,i) !Jonas
      enddo

      close(142)

      end subroutine outg

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine stoprun(s)
      character*(*) s
      write(*,*)'program stopped due to: ',s
      call system('touch dscf_problem')
      stop 'must stop!'
      end subroutine stoprun

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C Returns the number of a given element string (h-pu, 1-94)
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      SUBROUTINE ELEM(KEY1, NAT)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      CHARACTER*(*) KEY1
      CHARACTER*2 ELEMNT(94),E

      DATA ELEMNT/'h ','he',
     1 'li','be','b ','c ','n ','o ','f ','ne',
     2 'na','mg','al','si','p ','s ','cl','ar',
     3 'k ','ca','sc','ti','v ','cr','mn','fe','co','ni','cu',
     4 'zn','ga','ge','as','se','br','kr',
     5 'rb','sr','y ','zr','nb','mo','tc','ru','rh','pd','ag',
     6 'cd','in','sn','sb','te','i ','xe',
     7 'cs','ba','la','ce','pr','nd','pm','sm','eu','gd','tb','dy',
     8 'ho','er','tm','yb','lu','hf','ta','w ','re','os','ir','pt',
     9 'au','hg','tl','pb','bi','po','at','rn',
     1 'fr','ra','ac','th','pa','u ','np','pu'/

      nat=0
      e='  '
      k=1
      DO J=1,len(key1)
         if (k.gt.2)exit       
         N=ICHAR(key1(J:J))
         if(n.ge.ichar('A') .and. n.le.ichar('Z') )then
            e(k:k)=char(n+ICHAR('a')-ICHAR('A'))
            k=k+1
         endif
         if(n.ge.ichar('a') .and. n.le.ichar('z') )then
            e(k:k)=key1(j:j)
            k=k+1
         endif
      enddo

      DO I=1,94
         if(e.eq.elemnt(i))then
            NAT=I
            RETURN
         ENDIF
      ENDDO

    
      end SUBROUTINE ELEM

C     *****************************************************************         

      FUNCTION ESYM(I)
      CHARACTER*2 ESYM
      CHARACTER*2 ELEMNT(94)
      DATA ELEMNT/'h ','he',
     1 'li','be','b ','c ','n ','o ','f ','ne',
     2 'na','mg','al','si','p ','s ','cl','ar',
     3 'k ','ca','sc','ti','v ','cr','mn','fe','co','ni','cu',
     4 'zn','ga','ge','as','se','br','kr',
     5 'rb','sr','y ','zr','nb','mo','tc','ru','rh','pd','ag',
     6 'cd','in','sn','sb','te','i ','xe',
     7 'cs','ba','la','ce','pr','nd','pm','sm','eu','gd','tb','dy',
     8 'ho','er','tm','yb','lu','hf','ta','w ','re','os','ir','pt',
     9 'au','hg','tl','pb','bi','po','at','rn',
     1 'fr','ra','ac','th','pa','u ','np','pu'/
      ESYM=ELEMNT(I)
      RETURN
      END FUNCTION ESYM


C     *****************************************************************        
C     Reads a given line
C     ***************************************************************** 
                                                                                
      SUBROUTINE READL(A1,X,N)                                               
      IMPLICIT REAL*8 (A-H,O-Z)                                                 
      CHARACTER*(*) A1                                                      
      DIMENSION X(*)                                                            
      I=0                                                                       
      IS=1                                                                      
  10  I=I+1                                                                     
      X(I)=READAA(A1,IS,IB,IE)                                               
      IF(IB.GT.0 .AND. IE.GT.0) THEN                                            
                                IS=IE                                           
                                GOTO 10                                         
      ENDIF                                                                     
      N=I-1                                                                     
      RETURN                                                                    
      END SUBROUTINE READL                                                                       
                                                                                
C     *****************************************************************         
C     this seems to be part of an old MOPAC code, extracts strings and numbers
C     *****************************************************************         
                                                                                
      FUNCTION READAA(A,ISTART,IEND,IEND2)                                   
      IMPLICIT REAL*8 (A-H,O-Z)                                                 
      REAL*8 READAA                                                             
      CHARACTER*(*) A                                                      
      NINE=ICHAR('9')                                                           
      IZERO=ICHAR('0')                                                          
      MINUS=ICHAR('-')                                                          
      IDOT=ICHAR('.')                                                           
      ND=ICHAR('D')                                                             
      NE=ICHAR('E')                                                             
      IBL=ICHAR(' ')                                                            
      IEND=0                                                                    
      IEND2=0                                                                   
      IDIG=0                                                                    
      C1=0                                                                      
      C2=0                                                                      
      ONE=1.D0                                                                  
      X = 1.D0                                                                  
      NL=LEN(A) 
      DO 10 J=ISTART,NL-1                                                       
         N=ICHAR(A(J:J))                                                          
         M=ICHAR(A(J+1:J+1)) 
         IF(N.LE.NINE.AND.N.GE.IZERO .OR.N.EQ.IDOT)GOTO 20                      
         IF(N.EQ.MINUS.AND.(M.LE.NINE.AND.M.GE.IZERO                            
     1 .OR. M.EQ.IDOT)) GOTO 20                                                 
   10 CONTINUE                                                                  
      READAA=0.D0                                                               
      RETURN                                                                    
   20 CONTINUE                                                                  
      IEND=J                                                                    
      DO 30 I=J,NL                                                              
         N=ICHAR(A(I:I))                                                          
         IF(N.LE.NINE.AND.N.GE.IZERO) THEN                                      
            IDIG=IDIG+1                                                         
            IF (IDIG.GT.10) GOTO 60                                             
            C1=C1*10+N-IZERO                                                    
         ELSEIF(N.EQ.MINUS.AND.I.EQ.J) THEN                                     
            ONE=-1.D0                                                           
         ELSEIF(N.EQ.IDOT) THEN                                                 
            GOTO 40                                                             
         ELSE                                                                   
            GOTO 60                                                             
         ENDIF                                                                  
   30 CONTINUE                                                                  
   40 CONTINUE                                                                  
      IDIG=0                                                                    
      DO 50 II=I+1,NL                                                           
         N=ICHAR(A(II:II))                                                         
         IF(N.LE.NINE.AND.N.GE.IZERO) THEN                                      
            IDIG=IDIG+1                                                         
            IF (IDIG.GT.10) GOTO 60                                             
            C2=C2*10+N-IZERO                                                    
            X = X /10                                                           
         ELSEIF(N.EQ.MINUS.AND.II.EQ.I) THEN                                    
            X=-X                                                                
         ELSE                                                                   
            GOTO 60                                                             
         ENDIF                                                                  
   50 CONTINUE                                                                  
C                                                                               
C PUT THE PIECES TOGETHER                                                       
C                                                                               
   60 CONTINUE                                                                  
      READAA= ONE * ( C1 + C2 * X)                                              
      DO 55 J=IEND,NL                                                           
         N=ICHAR(A(J:J))                                                          
         IEND2=J                                                                
         IF(N.EQ.IBL)RETURN                                                     
   55 IF(N.EQ.ND .OR. N.EQ.NE)GOTO 57                                           
      RETURN                                                                    
                                                                                
   57 C1=0.0D0                                                                  
      ONE=1.0D0                                                                 
      DO 31 I=J+1,NL                                                            
         N=ICHAR(A(I:I))                                                          
         IEND2=I                                                                
         IF(N.EQ.IBL)GOTO 70                                                    
         IF(N.LE.NINE.AND.N.GE.IZERO) C1=C1*10.0D0+N-IZERO                      
         IF(N.EQ.MINUS)ONE=-1.0D0                                               
   31 CONTINUE                                                                  
   61 CONTINUE                                                                  
   70 READAA=READAA*10**(ONE*C1)                                                
      RETURN                                                                    
      END FUNCTION READAA                                                                    

      subroutine prmat(iuout,r,n,m,head)
      real*8 r
      character*(*) head
      dimension r(*)
c     subroutine prints matrix r,which is supposed
c     to have dimension n,m  when m is nonzero and
c     ((n+1)*n)/2 when m is zero

      write(iuout,1001) head
      nkpb=10
      if(m)10,10,80
c
   10 continue
      ibl=n/nkpb
      ir=n-ibl*nkpb
      j1=1
      k1s=1
      kd=0
      if(ibl.eq.0) go to 50
      j2=nkpb
      do 40 i=1,ibl
      write(iuout,1002)(j,j=j1,j2)
      k1=k1s
      k2=k1
      kk=0
      do 20 j=j1,j2
      write(iuout,1003)j,(r(k),k=k1,k2)
      kk=kk+1
      k1=k1+kd+kk
   20 k2=k1+kk
      j1=j1+nkpb
      if(j1.gt.n) return
      j2=j2+nkpb
      k2=k1-1
      k1=k2+1
      k2=k1+(nkpb-1)
      k1s=k2+1
      kk=kd+nkpb
      do 30 j=j1,n
      write(iuout,1003)j,(r(k),k=k1,k2)
      kk=kk+1
      k1=k1+kk
   30 k2=k2+kk
   40 kd=kd+nkpb
   50 if(ir.eq.0) go to 70
      k1=k1s
      j2=j1+ir-1
      kk=0
      k2=k1
      write(iuout,1002)(j,j=j1,j2)
      write(iuout,1003)
      do 60 j=j1,j2
      write(iuout,1003)j,(r(k),k=k1,k2)
      kk=kk+1
      k1=k1+kd+kk
   60 k2=k1+kk
   70 return
   80 ibl=m/nkpb
      ir=m-ibl*nkpb
      i2=0
      k2=0
      if(ibl.eq.0) go to 100
      do 90 i=1,ibl
      i1=(i-1)*n*nkpb+1
      i2=i1+(nkpb-1)*n
      k1=k2+1
      k2=k1+(nkpb-1)
      write(iuout,1002)(k,k=k1,k2)
      do 90 j=1,n
      write(iuout,1003)j,(r(ij),ij=i1,i2,n)
      i1=i1+1
   90 i2=i1+(nkpb-1)*n
  100 if(ir.eq.0) go to 120
      i1=ibl*n*nkpb+1
      i2=i1+(ir-1)*n
      k1=k2+1
      k2=m
      write(iuout,1002)(k,k=k1,k2)
      write(iuout,1003)
      do 110 j=1,n
      write(iuout,1003)j,(r(ij),ij=i1,i2,n)
      i1=i1+1
      i2=i1+(ir-1)*n
  110 continue
  120 write(iuout,1003)
      return
 1001 format(/,a)
 1002 format(/,' ',4x,10(3x,i4,3x),/)
 1003 format(' ',i4,10f10.3)
      end subroutine prmat

ccccccccccccccccccccccccccccccccccccccccccccccc
c   readfrag: will read a list of numbers     c
c             from character.                 c
c        line: string containing the numbers  c
c        iout: array which returns integer    c
c              numbers                        c
c           n: number of returned integers    c
c           S.E., 17.02.2011                  c
ccccccccccccccccccccccccccccccccccccccccccccccc

      subroutine readfrag(line,iout,n)
      implicit none
      
      character*80 line
      character*12 str1,str2
      integer iout(500)
      integer*4 n,i,j,k,sta,sto
      character*11 nums
      
c      write(*,*) 'In readfrag:'
c      write(*,*) 'Line reads: ',line
      n=0
      iout=0
      str1=''
      str2=''
      nums='0123456789-'
      do i=1,80
c Check for allowed characters (part of nums) and add to number-string str1.
c If char NOT allowed, its a separator and str1 will be processed

       if(index(nums,line(i:i)).ne.0) then 
        str1=trim(str1)//line(i:i)
       else
       
        if(str1.ne.'') then

c If str1 is simple number, cast to integer and put in iout. increase n.
         if(index(str1,'-').eq.0) then  
          n=n+1
          read(str1,'(I4)') iout(n)
          str1=''
          
         endif
c If str1 contains '-', its treated as a number range.
         if(index(str1,'-').ne.0) then

          do k=1,12
c Determine beginning number          
           if(str1(k:k).ne.'-') then           
            str2=trim(str2)//str1(k:k)            
           endif
c '-' Marks end of beginning number. cast to integer and store in sta
           if(str1(k:k).eq.'-') then           
            read(str2,'(I4)') sta
            str2=''            
           endif
           
          enddo
c Get the rest, store in sto
          read(str2,'(I4)') sto
          str2=''
c Write all numbers between sta and sto to iout and increase n
          do k=sta,sto
           n=n+1
           iout(n)=k
          enddo
         str1='' 
         endif
        endif
       endif

      enddo

      end subroutine readfrag

c Input Geometry sanity check via CNs, not used
      subroutine checkcn(n,iz,cn,c6ab,max_elem,maxc)
      implicit none
      
      integer iz(*),i,n
      logical check
      real*8 cn(*),maxcn
      integer max_elem,maxc
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)

      check=.false.
      do i=1,n
       if (iz(i).lt.10) then
        if(iz(i).ne.2) then
         maxcn=maxval(c6ab(iz(i),1,1:5,1,2))
         if (cn(i).gt.maxcn*2.0) then
          check=.true.
         endif
        endif
       endif
      enddo
       if (check) then
        write(*,*)'----------------------------------------------------'
        write(*,*)'!!!! SOME CN ARE VERY LARGE. CHECK COORDINATES !!!!'
        write(*,*)'----------------------------------------------------'
       endif
      end subroutine checkcn

c     Input Geometry sanity check (to avoid au/Angtstrom mixups) S.E. 16.3.2012
      subroutine checkrcov(n,iz,rcov,xyz)
      implicit none
      logical check
      integer iz(*),n,i,j
      real*8 rcov(94),dist,dx,dy,dz,thr,xyz(3,*),r
      check=.false.
      do i=1,n-1
       do j=i+1,n
         dx=xyz(1,i)-xyz(1,j)
         dy=xyz(2,i)-xyz(2,j)
         dz=xyz(3,i)-xyz(3,j)
         r=sqrt(dx*dx+dy*dy+dz*dz)
         thr=0.6*(rcov(iz(i))+rcov(iz(j)))
         if (r.lt.thr) then
           check=.true.
         endif
       enddo
      enddo
      if (check) then
          write(*,*)'--------------------------------------------------'
          write(*,*)'!! SOME DISTANCES VERY SHORT. CHECK COORDINATES !!'
          write(*,*)'--------------------------------------------------'
      endif
      end subroutine checkrcov



    
!reads a line cuts the at blanks and tabstops and returns all floats and strings in order of occurence
      subroutine readline(line,floats,strings,cs,cf)  
      implicit none
      real*8 floats(3)
      character*80 line
      character*80 strings(3)

      real*8 num
      character*80 stmp,str
      character*1 digit
      integer i,ty,cs,cf
      
      stmp=''
      cs=1
      cf=1
      strings(:)=''
      do i=1,len(trim(line))
       digit=line(i:i)
       if(digit.ne.' '.and.digit.ne.char(9)) then  !should exclude tabstops and blanks, 9 is ascii code for tab
        stmp=trim(stmp)//trim(digit)
       elseif(stmp.ne.'')then
        call checktype(stmp,num,str,ty)      !get type of string, 0=number, 1=character
        if(ty.eq.0) then
         floats(cf)=num
         cf=cf+1
        elseif(ty.eq.1) then
         strings(cs)=str
         cs=cs+1
        else
          write(*,*)'Problem in checktype, must abort'
          exit
        endif
        stmp=''
       endif
       if(i.eq.len(trim(line))) then  !special case: end of line
        call checktype(stmp,num,str,ty)
        if(ty.eq.0) then
         floats(cf)=num
         cf=cf+1
        elseif(ty.eq.1) then
         strings(cs)=str
         cs=cs+1
        else
          write(*,*)'Problem in checktype, must abort'
          exit
        endif
        stmp=''
       endif
      enddo
      cs=cs-1
      cf=cf-1
      end subroutine readline        


!this checks the type of the string and returns it cast to real or as string.
      subroutine checktype(field,num,str,ty) 
      implicit none
      character*80 field,str,tstr
      real*8 num
      integer ty

      integer i,e
      logical is_num

      ty=99
      str=''
      is_num=.false.
      read(field,'(F20.10)',IOSTAT=e)num !cast string on real and get error code; 0 means success. Will return 0 for Q,E,D
      if(e.eq.0)is_num=.true.
      if(field.eq.'q'.or.field.eq.'Q')is_num=.false.  !handle exceptions the quick way..
      if(field.eq.'e'.or.field.eq.'E')is_num=.false.  
      if(field.eq.'d'.or.field.eq.'D')is_num=.false.  
      if(is_num)then
       if(index(field,'.').ne.0) then  !check for integer/real
        read(field,'(F20.10)')num
        ty=0
       else                       !if integer, add .0 to string; otherwise cast to real does not work
        str=trim(field)//'.0'
        read(str,'(F20.10)')num
        str=''
        ty=0
       endif
      else
       str=field
       ty=1
      endif
      
      end subroutine checktype







CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         B E G I N   O F   P B C   P A R T 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
c read coordinates in Angst and converts them to au 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

c      subroutine pbcrdcoord(fname,lattice,n,xyz,iat,autoang)
c      implicit none             
c      interface
c        subroutine parse(str,delims,args,nargs)
c        character(len=*),intent(inout) :: str
c        character(len=*),intent(in)  :: delims
c        character(len=*),dimension(:),intent(inout) :: args
c        integer, intent(out) :: nargs
c        end subroutine parse
c      end interface
c       
c      real*8                :: xyz(3,*)
c      real*8, INTENT(OUT)   ::lattice(3,3)
c      integer, INTENT(out)               :: iat(*) 
c      integer, INTENT(in)               :: n 
c      character*(*), INTENT(IN)          :: fname
c      logical              :: selective=.FALSE. ! Selective dynamics
c      logical              :: cartesian=.TRUE.  ! Cartesian or direct
c      real*8, INTENT(IN)   ::autoang
c
c      real*8 xx(10),scalar
c      character*80 line,args(90),args2(90)
c      
c      integer i,j,ich,nn,ntype,ntype2,atnum,i_dummy1,i_dummy2,ncheck
c
c
c      lattice=0
c      
c      ich=142
c      open(unit=ich,file=fname)
c      rewind(ich)
c      ncheck=0
c      ntype=0
c      read(ich,'(a)',end=200)line !first line must contain Element Info
c      call parse(line,' ',args,ntype)
c      read(ich,'(a)',end=200)line !second line contains global scaling factor
c      call readl(line,xx,nn)
c      scalar=xx(1)/autoang        !the Ang->au conversion is included in the scaling factor
cc      write(*,'(F8.6)')scalar
c      DO i=1,3            ! reading the lattice constants
c        read(ich,'(a)',end=200)line
c        call readl(line,xx,nn)
c        IF (nn < 3) call stoprun( 'Error reading unit cell vectors' )
c        lattice(1,i)=xx(1)*scalar
c        lattice(2,i)=xx(2)*scalar
c        lattice(3,i)=xx(3)*scalar
c      !  write(*,'(3F6.2)')lattice(1,i),lattice(2,i),lattice(3,i)
c      ENDDO
c      read(ich,'(a)',end=200)line !Ether here are the numbers of each element, or (>vasp.5.1) here are the element symbols
c      line=adjustl(line)
c      call readl(line,xx,nn)
c      IF (nn.eq.0) then      ! CONTCAR files have additional Element line here since vasp.5.1
c        call parse(line,' ',args,ntype)
c        read(ich,'(a)',end=200)line
c        line=adjustl(line)
c        call readl(line,xx,nn)
c      ENDIF
c!       call elem(args(1),i_dummy2)
c!       IF (i_dummy2<1 .OR. i_dummy2>94) THEN
c!          args=args2
c!       ENDIF
c      IF (nn.NE.ntype ) THEN
c!         IF(nn.NE.ntype2) THEN
c        call stoprun( 'Error reading number of atomtypes')
c!         ELSE
c!           ntype=ntype2
c!         ENDIF
c      ENDIF
c      ncheck=0
c      DO i=1,nn
c        i_dummy1=INT(xx(i))
c        call elem(args(i),i_dummy2)
c        IF (i_dummy2<1 .OR. i_dummy2>94) 
c     .   call stoprun( 'Error: unknown element.')
c        DO j=1,i_dummy1
c          ncheck=ncheck+1
c          iat(ncheck)=i_dummy2
c        ENDDO
c      ENDDO
c      if (n.ne.ncheck) call stoprun('Error reading Number of Atoms')
c
c      read(ich,'(a)',end=200)line
c      line=adjustl(line)
c      IF (line(:1).EQ.'s' .OR. line(:1).EQ.'S') THEN
c        selective=.TRUE.
c        read(ich,'(a)',end=200)line
c        line=adjustl(line)
c      ENDIF
c
cc      write(*,*)line(:1)
c      cartesian=(line(:1).EQ.'c' .OR. line(:1).EQ.'C' .OR. 
c     .line(:1).EQ.'k' .OR. line(:1).EQ.'K')
c      DO i=1,n
c        read(ich,'(a)',end=200)line
c        call readl(line,xx,nn)
c        IF (nn.NE.3) call stoprun( 'Error reading coordinates.')
c
c        IF (cartesian) THEN
c          xyz(1,i)=xx(1)*scalar
c          xyz(2,i)=xx(2)*scalar
c          xyz(3,i)=xx(3)*scalar
c        ELSE
c          xyz(1,i)=lattice(1,1)*xx(1)+lattice(1,2)*
c     .    xx(2)+lattice(1,3)*xx(3)
c          xyz(2,i)=lattice(2,1)*xx(1)+lattice(2,2)*xx(2)+lattice(2,3)*
c     .    xx(3)
c          xyz(3,i)=lattice(3,1)*xx(1)+lattice(3,2)*xx(2)+lattice(3,3)*
c     .    xx(3)
c        ENDIF
c        
cc      write(*,321)xyz(:,i),iat(i)   !debug printout
c      
c      ENDDO
c
c      
cc      IF line(1)
c
cc      lattice(1,*)=xx
c      
c
cc      call readl(line,xx,nn)
c      
c 200  continue
c
c      close(ich)
cc  321 FORMAT(3F20.10,1X,I3) !debug output
c      end subroutine pbcrdcoord


CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C compute coordination numbers by adding an inverse damping function
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine pbcncoord(natoms,rcov,iz,xyz,cn,lat,rep_cn,crit_cn)
      implicit none  
      !include 'param'

#include "global.fh"
#include "msgtypesf.h"

      real*8 k1,k2,k3

c global ad hoc parameters
      parameter (k1=16.0)
      parameter (k2=4./3.) 

c reasonable choices are between 3 and 5
c this gives smoth curves with maxima around the integer values
c k3=3 give for CN=0 a slightly smaller value than computed
c for the free atom. This also yields to larger CN for atoms
c in larger molecules but with the same chem. environment
c which is physically not right
c values >5 might lead to bumps in the potential
      parameter (k3=-4.) 


      integer,intent(in) :: natoms,iz(*)
      real*8,intent(in)  :: rcov(94)

      integer i,max_elem,rep_cn(3)
      real*8 xyz(3,*),cn(*),lat(3,3)

      integer iat,taux,tauy,tauz    
      real*8 dx,dy,dz,r,damp,xn,rr,rco,tau(3)
      real*8, INTENT(IN) :: crit_cn
      integer taskid,np,pcount

      call dcopy(natoms,0.0d0,0,cn,1)
      np     = GA_Nnodes()
      taskid = GA_Nodeid()
      pcount = 0

      do i=1,natoms
      xn=0.0d0
      do iat=1,natoms
        if (mod(pcount,np).eq.taskid) then
        do taux=-rep_cn(1),rep_cn(1)
         do tauy=-rep_cn(2),rep_cn(2)
          do tauz=-rep_cn(3),rep_cn(3)
            if(iat.eq.i .and. taux.eq.0 .and. tauy.eq.0 .and. 
     .       tauz.eq.0)        cycle
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
            dx=xyz(1,iat)-xyz(1,i)+tau(1)
            dy=xyz(2,iat)-xyz(2,i)+tau(2)
            dz=xyz(3,iat)-xyz(3,i)+tau(3)
            r=(dx*dx+dy*dy+dz*dz)
            if (r.gt.crit_cn) cycle
            r=sqrt(r)
c covalent distance in Bohr
            rco=rcov(iz(i))+rcov(iz(iat))
            rr=rco/r
c counting function exponential has a better long-range behavior than MHGs inverse damping
            damp=1.d0/(1.d0+exp(-k1*(rr-1.0d0)))
            xn=xn+damp
c            print '("cn(",I2,I2,"): ",E14.8)',i,iat,damp

          enddo !tauz
         enddo !tauy
        enddo !taux
        end if
        pcount = pcount + 1
      enddo !iat
      cn(i)=xn  
      enddo !i

      if (np.gt.1) call GA_DGOP(9+MSGDBL,cn,natoms,'+')

      end subroutine pbcncoord

c      subroutine pbcrdatomnumber(fname,n)
c      implicit none             
c      interface
c        subroutine parse(str,delims,args,nargs)
c        character(len=*),intent(inout) :: str
c        character(len=*),intent(in)  :: delims
c        character(len=*),dimension(:),intent(inout) :: args
c        integer, intent(out) :: nargs
c        end subroutine parse
c      end interface
c       
c      integer, INTENT(out)               :: n 
c      character*(*), INTENT(IN)          :: fname
c      logical              :: selective=.FALSE. ! Selective dynamics
c      logical              :: cartesian=.TRUE.  ! Cartesian or direct
c
c      real*8 xx(10),scalar,fdum
c      character*80 line,args(90),args2(90)
c      
c      integer i,j,ich,nn,ntype,ntype2,atnum,i_dummy1,i_dummy2
c      
c      ich=142
c      open(unit=ich,file=fname)
c      n=0
c      ntype=0
c      read(ich,'(a)',end=200)line !first line must contain Element Info
c      call parse(line,' ',args,ntype)
c      read(ich,'(a)',end=200)line !second line contains global scaling factor
c      call readl(line,xx,nn)
cc      write(*,'(F8.6)')scalar
c      DO i=1,3            ! reading the lattice constants
c        read(ich,'(a)',end=200)line
c        call readl(line,xx,nn)
c        IF (nn < 3) call stoprun( 'Error reading unit cell vectors' )
c      !  write(*,'(3F6.2)')lattice(1,i),lattice(2,i),lattice(3,i)
c      ENDDO
c      read(ich,'(a)',end=200)line !Ether here are the numbers of each element, or (>vasp.5.1) here are the element symbols
c      line=adjustl(line)
c      call readl(line,xx,nn)
c      IF (nn.eq.0) then      ! CONTCAR files have additional Element line here since vasp.5.1
c        call parse(line,' ',args,ntype)
c        read(ich,'(a)',end=200)line
c        line=adjustl(line)
c        call readl(line,xx,nn)
c      ENDIF
c!       call elem(args(1),i_dummy2)
c!       IF (i_dummy2<1 .OR. i_dummy2>94) THEN
c!          args=args2
c!       ENDIF
c      IF (nn.NE.ntype ) THEN
c!         IF(nn.NE.ntype2) THEN
c        call stoprun( 'Error reading number of atomtypes')
c!         ELSE
c!           ntype=ntype2
c!         ENDIF
c      ENDIF
c      n=0
c      DO i=1,nn
c        i_dummy1=INT(xx(i))
c          n=n+i_dummy1
c      ENDDO
c
c 200  continue
c
c      close(ich)
c      end subroutine pbcrdatomnumber



CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C compute energy
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
 
      subroutine pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e63,lat,rthr,rep_vdw,cn_thr,rep_cn)
      implicit none  
      integer max_elem,maxc
      real*8 r2r4(max_elem),rcov(max_elem)
      real*8 rs6,rs8,rs10,alp6,alp8,alp10
      real*8 rthr,cn_thr,crit_cn
      integer rep_vdw(3),rep_cn(3)
      integer n,iz(*),version,mxc(max_elem)
!      integer rep_v(3)=rep_vdw!,rep_cn(3)
      real*8 xyz(3,*),r0ab(max_elem,max_elem),lat(3,3)!,r2r4(*)
!      real*8 rs6,rs8,rs10,alp6,alp8,alp10,rcov(max_elem)
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)
      real*8 e6, e8, e10, e12, e63!,crit_vdw,crit_cn
      logical noabc
 
      integer iat,jat,kat
      real*8 r,r2,r6,r8,tmp,dx,dy,dz,c6,c8,c10,ang,rav
      real*8 damp6,damp8,damp10,rr,thr,c9,r42,c12,r10,c14
      real*8 cn(n),rxyz(3),dxyz(3)
      real*8 r2ab(n*n),cc6ab(n*n),dmp(n*n),d2(3),t1,t2,t3,tau(3)
      integer*2 icomp(n*n)
      integer lin,ij,ik,jk
      integer taux,tauy,tauz,counter
      real*8 a1,a2  !BJ-parameter
      real*8 bj_dmp6,bj_dmp8


      e6 =0
      e8 =0
      e10=0
      e12=0
      e63=0
      tau=(/0.0,0.0,0.0/)
      counter=0
      crit_cn=cn_thr
c Becke-Johnson parameters
      a1=rs6
      a2=rs8      
      


C DFT-D2
      if(version.eq.2)then


      do iat=1,n-1
         do jat=iat+1,n
           c6=c6ab(iz(jat),iz(iat),1,1,1)
           do taux=-rep_vdw(1),rep_vdw(1)
           do tauy=-rep_vdw(2),rep_vdw(2)
           do tauz=-rep_vdw(3),rep_vdw(3)
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
            dx=xyz(1,iat)-xyz(1,jat)+tau(1)
            dy=xyz(2,iat)-xyz(2,jat)+tau(2)
            dz=xyz(3,iat)-xyz(3,jat)+tau(3)
            r2=dx*dx+dy*dy+dz*dz
           if(r2.gt.rthr) cycle
            r=sqrt(r2)
            damp6=1./(1.+exp(-alp6*(r/(rs6*r0ab(iz(jat),iz(iat)))-1.)))
            r6=r2**3      
            e6 =e6+c6*damp6/r6
            counter=counter+1
           enddo !taux
           enddo !tauy
           enddo !tauz
         enddo
      enddo
      
      do iat=1,n
        jat=iat
        c6=c6ab(iz(jat),iz(iat),1,1,1)
        do taux=-rep_vdw(1),rep_vdw(1)
        do tauy=-rep_vdw(2),rep_vdw(2)
        do tauz=-rep_vdw(3),rep_vdw(3)
          if (taux.eq.0 .and. tauy.eq.0 .and. tauz.eq.0) cycle
          tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
          dx=tau(1)
          dy=tau(2)
          dz=tau(3)
          r2=dx*dx+dy*dy+dz*dz
           if(r2.gt.rthr) cycle
          r=sqrt(r2)
          damp6=1./(1.+exp(-alp6*(r/(rs6*r0ab(iz(jat),iz(iat)))-1.)))
          r6=r2**3      
          e6 =e6+c6*damp6/r6*0.50d0
          counter=counter+1
        enddo
        enddo
        enddo
      enddo !iat
!      write(*,*)'counter: ',counter
      
      

      else if (version.eq.3) then
C DFT-D3(zero-damping)

      call pbcncoord(n,rcov,iz,xyz,cn,lat,rep_cn,crit_cn)

      icomp=0
      do iat=1,n-1
         do jat=iat+1,n
c get C6
          call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                  cn(iat),cn(jat),c6)

           do taux=-rep_vdw(1),rep_vdw(1)
           do tauy=-rep_vdw(2),rep_vdw(2)
           do tauz=-rep_vdw(3),rep_vdw(3)
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)

            dx=xyz(1,iat)-xyz(1,jat)+tau(1)
            dy=xyz(2,iat)-xyz(2,jat)+tau(2)
            dz=xyz(3,iat)-xyz(3,jat)+tau(3)
            r2=dx*dx+dy*dy+dz*dz
c cutoff

           if(r2.gt.rthr) cycle
            r =sqrt(r2)
            rr=r0ab(iz(jat),iz(iat))/r
c damping
            tmp=rs6*rr   
            damp6 =1.d0/( 1.d0+6.d0*tmp**alp6 )
            tmp=rs8*rr     
            damp8 =1.d0/( 1.d0+6.d0*tmp**alp8 )

            if(.not.noabc)then
              ij=lin(jat,iat)
              icomp(ij)=1
c store C6 for C9, calc as sqrt
              cc6ab(ij)=sqrt(c6)
            endif

            r6=r2**3      
            e6 =e6+c6*damp6/r6
c             write(*,*)'e6: ',c6*damp6/r6*autokcal

c stored in main as sqrt
            c8 =3.0d0*c6*r2r4(iz(iat))*r2r4(iz(jat))
            r8 =r6*r2

            e8 =e8+c8*damp8/r8

            counter=counter+1

           enddo !tauz
           enddo !tauy
           enddo !taux
         enddo !jat
      enddo !iat
      
      do iat=1,n
        jat=iat
c get C6
        call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                  cn(iat),cn(jat),c6)
         
        do taux=-rep_vdw(1),rep_vdw(1)
         do tauy=-rep_vdw(2),rep_vdw(2)
          do tauz=-rep_vdw(3),rep_vdw(3)
            if (taux.eq.0 .and. tauy.eq.0 .and. tauz.eq.0) cycle
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)

            dx=tau(1)
            dy=tau(2)
            dz=tau(3)
            r2=dx*dx+dy*dy+dz*dz
c cutoff
           if(r2.gt.rthr) cycle
            r =sqrt(r2)
            rr=r0ab(iz(jat),iz(iat))/r
c damping
            tmp=rs6*rr   
            damp6 =1.d0/( 1.d0+6.d0*tmp**alp6 )
            tmp=rs8*rr     
            damp8 =1.d0/( 1.d0+6.d0*tmp**alp8 )

            if(.not.noabc)then
              ij=lin(jat,iat)
              icomp(ij)=1
c store C6 for C9, calc as sqrt
              cc6ab(ij)=sqrt(c6)
            endif

            r6=r2**3      

            e6 =e6+c6*damp6/r6*0.50d0

c stored in main as sqrt
            c8 =3.0d0*c6*r2r4(iz(iat))*r2r4(iz(jat))
            r8 =r6*r2

            e8 =e8+c8*damp8/r8*0.50d0
            counter=counter+1

         enddo !tauz
        enddo !tauy
       enddo !taux
      enddo !iat
!      write(*,*)'counter(edisp): ',counter
      else if (version.eq.4) then


C DFT-D3(BJ-damping)
      call pbcncoord(n,rcov,iz,xyz,cn,lat,rep_cn,crit_cn)

      icomp=0
      do iat=1,n
         do jat=iat+1,n
c get C6
           call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                  cn(iat),cn(jat),c6)

           rxyz=xyz(:,iat)-xyz(:,jat)
           r42=r2r4(iz(iat))*r2r4(iz(jat))
           bj_dmp6=(a1*dsqrt(3.0d0*r42)+a2)**6
           bj_dmp8=(a1*dsqrt(3.0d0*r42)+a2)**8

           do taux=-rep_vdw(1),rep_vdw(1)
           do tauy=-rep_vdw(2),rep_vdw(2)
           do tauz=-rep_vdw(3),rep_vdw(3)
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
            
            dxyz=rxyz+tau

            r2=sum(dxyz*dxyz)
c cutoff
           if(r2.gt.rthr) cycle
            r =sqrt(r2)
            rr=r0ab(iz(jat),iz(iat))/r

            if(.not.noabc)then
              ij=lin(jat,iat)
              icomp(ij)=1
c store C6 for C9, calc as sqrt
              cc6ab(ij)=sqrt(c6)
            endif

            r6=r2**3      

            e6 =e6+c6/(r6+bj_dmp6)
c            write(*,*)'e6: ',e6

c stored in main as sqrt
            c8 =3.0d0*c6*r42
            r8 =r6*r2

            e8 =e8+c8/(r8+bj_dmp8)

            counter=counter+1

           enddo !tauz
           enddo !tauy
           enddo !taux
         enddo !jat

! Now the self interaction
        jat=iat
c get C6
        call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                  cn(iat),cn(jat),c6)
        r42=r2r4(iz(iat))*r2r4(iz(iat))
        bj_dmp6=(a1*dsqrt(3.0d0*r42)+a2)**6
        bj_dmp8=(a1*dsqrt(3.0d0*r42)+a2)**8
         
        do taux=-rep_vdw(1),rep_vdw(1)
         do tauy=-rep_vdw(2),rep_vdw(2)
          do tauz=-rep_vdw(3),rep_vdw(3)
            if (taux.eq.0 .and. tauy.eq.0 .and. tauz.eq.0) cycle
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)

            r2=sum(tau*tau)
c cutoff
           if(r2.gt.rthr) cycle
            r =sqrt(r2)
            rr=r0ab(iz(jat),iz(iat))/r

            if(.not.noabc)then
              ij=lin(jat,iat)
              icomp(ij)=1
c store C6 for C9, calc as sqrt
              cc6ab(ij)=sqrt(c6)
            endif

            r6=r2**3      

            e6 =e6+c6/(r6+bj_dmp6)*0.50d0

c stored in main as sqrt
            c8 =3.0d0*c6*r42
            r8 =r6*r2

            e8 =e8+c8/(r8+bj_dmp8)*0.50d0
            counter=counter+1


c           r10=r8*r2
c           c10=(49.0d0/40.0d0)*c8*c8/c6
c           e10=e10+c10*damp8 /r10       
c           c12=c6*(c10/c8)**3 
c           e12=e12+c12*damp8 /(r10*r2)
c           c14=c8*(c12/c10)**3 
c           e12=e12+c14*damp8 /(r10*r2*r2)
         enddo !tauz
        enddo !tauy
       enddo !taux
      enddo !iat


      endif !version


      if(noabc)return

C compute non-additive third-order energy using averaged C6
!      call stoprun( 'Threebodyterm not jet  implemented' )
       call pbcthreebody(max_elem,xyz,lat,n,iz,rep_cn,cc6ab,r0ab,e63)

      end subroutine pbcedisp


      SUBROUTINE pbcthreebody(max_elem,xyz,lat,n,iz,repv,cc6ab,r0ab,
     .           eabc)
      IMPLICIT NONE
      integer max_elem
      INTEGER         :: n,i,j,k,jtaux,jtauy,jtauz,iat,jat,kat
      INTEGER         :: ktaux,ktauy,ktauz,counter,ij,ik,jk
      REAL*8          :: dx,dy,dz,rij2,rik2,rjk2,c9,rr0ij,rr0ik
      REAL*8          :: rr0jk,geomean,fdamp,rik,rjk,rij,tmp
      REAL*8,INTENT(OUT)::eabc
      REAL*8          :: cosij,cosik,cosjk !cosine of the triangular by "law of cosine"
                                           ! cosij is the angle opposite to the side ij
      REAL*8 ,DIMENSION(3,3),INTENT(IN)::lat
      REAL*8 ,DIMENSION(3,*),INTENT(IN) :: xyz
      INTEGER,DIMENSION(*),INTENT(IN)::iz
      REAL*8,DIMENSION(3):: jtau,ktau,jxyz,kxyz,ijvec,ikvec,jkvec,dumvec
      INTEGER,DIMENSION(3):: repv
      REAL*8,DIMENSION(n*n),INTENT(IN)::cc6ab
      REAL*8,DIMENSION(max_elem,max_elem),INTENT(IN):: r0ab
      REAL*8,PARAMETER::sr9=0.75d0    !reciprocal radii scaling parameter for damping function (s_r=4/3)
      REAL*8,PARAMETER::alp9=-16.0d0  !alpha saved with "-" sign
      INTEGER,EXTERNAL :: lin

      counter=0
      eabc=0.0d0

      do iat=1,n-2
        do jat=iat+1,n-1
          ijvec=xyz(:,jat)-xyz(:,iat)
          ij=lin(iat,jat)
          do kat=jat+1,n
!         write(*,*)'i:',iat,'j:',jat,'k:',kat
            ik=lin(iat,kat)
            jk=lin(jat,kat)
            ikvec=xyz(:,kat)-xyz(:,iat)
            jkvec=xyz(:,kat)-xyz(:,jat)
            c9=-1.0d0*(cc6ab(ij)*cc6ab(ik)*cc6ab(jk))
!            write(*,*)'c9: ',c9



            do jtaux=-repv(1),repv(1)
            do jtauy=-repv(2),repv(2)
            do jtauz=-repv(3),repv(3)
              jtau=jtaux*lat(:,1)+jtauy*lat(:,2)+jtauz*lat(:,3)
              dumvec=ijvec+jtau
              dumvec=dumvec*dumvec
              rij2=SUM(dumvec)
              rij=SQRT(rij2)

              rr0ij=rij/r0ab(iz(iat),iz(jat))
!              write(*,*)'r0ij:',r0ab(iz(iat),iz(jat))
!              write(*,*)'rr0ij:',rr0ij**1./3.


              do ktaux=-repv(1),repv(1)
              do ktauy=-repv(2),repv(2)
              do ktauz=-repv(3),repv(3)
                ktau=ktaux*lat(:,1)+ktauy*lat(:,2)+ktauz*lat(:,3)
                dumvec=ikvec+ktau
                dumvec=dumvec*dumvec
                rik2=SUM(dumvec)
                rik=DSQRT(rik2)
                rr0ik=rik/r0ab(iz(iat),iz(kat))

                dumvec=jkvec+ktau-jtau
                dumvec=dumvec*dumvec
                rjk2=SUM(dumvec)
                rjk=DSQRT(rjk2)
                rr0jk=rjk/r0ab(iz(jat),iz(kat))
!                write(*,*)'rr:',1.0/rr0jk

                geomean=(rr0ij*rr0ik*rr0jk)**(1.0d0/3.0d0)
!               write(*,*)'geomean:',geomean
                fdamp=1./(1.+6.*(sr9*geomean)**alp9)  !alp9 is already saved with "-"
                cosij=(rik2+rjk2-rij2)/(2.*rik*rjk)
                cosik=(rij2+rjk2-rik2)/(2.*rij*rjk)
                cosjk=(rij2+rik2-rjk2)/(2.*rik*rij)
                tmp=c9*(3.*cosij*cosik*cosjk+1)/
     .                 (rij*rik*rjk*rij2*rik2*rjk2)
!      write(*,*)'fdmp:',fdamp
!       write(*,*)'ang:',3.*cosij*cosik*cosjk+1 
                eabc=eabc+fdamp*tmp
!                write(*,'(''ktau'',3I2)'),ktaux,ktauy,ktauz
                counter=counter+1
              ENDDO !ktauz
              ENDDO !ktauy
              ENDDO !ktaux

            ENDDO !jtauz
            ENDDO !jtauy
            ENDDO !jtaux

          ENDDO !kat
        ENDDO !jat
      ENDDO !iat
!      write(*,*)'counter ijk: ',counter

      ! And now jat=iat, but cycling throug all imagecells without (0,0,0). and kat>iat going though all cells
      ! But this counts only 1/2

      DO iat=1,n-1
      jat=iat
      ijvec=0.0d0
      ij=lin(iat,jat)
        DO kat=iat+1,n
          ik=lin(iat,kat)
          jk=lin(jat,kat)
          ikvec=xyz(:,kat)-xyz(:,iat)
          jkvec=ikvec
          c9=-(cc6ab(ij)*cc6ab(ik)*cc6ab(jk))
          do jtaux=-repv(1),repv(1)
          do jtauy=-repv(2),repv(2)
          do jtauz=-repv(3),repv(3)
            IF (jtaux.eq.0 .and. jtauy.eq.0 .and. jtauz.eq.0) cycle
            jtau=jtaux*lat(:,1)+jtauy*lat(:,2)+jtauz*lat(:,3)
            dumvec=jtau
            dumvec=dumvec*dumvec
            rij2=SUM(dumvec)
            rij=SQRT(rij2)

            rr0ij=rij/r0ab(iz(iat),iz(jat))
       
            do ktaux=-repv(1),repv(1)
            do ktauy=-repv(2),repv(2)
            do ktauz=-repv(3),repv(3)
! every result * 0.5
              ktau=ktaux*lat(:,1)+ktauy*lat(:,2)+ktauz*lat(:,3)
              dumvec=ikvec+ktau
              dumvec=dumvec*dumvec
              rik2=SUM(dumvec)
              rik=SQRT(rik2)
              rr0ik=rik/r0ab(iz(iat),iz(kat))

              dumvec=jkvec+ktau-jtau
              dumvec=dumvec*dumvec
              rjk2=SUM(dumvec)
              rjk=SQRT(rjk2)
              rr0jk=rjk/r0ab(iz(jat),iz(kat))

              geomean=(rr0ij*rr0ik*rr0jk)**(1./3.)
              fdamp=1./(1.+6.*(sr9*geomean)**alp9)
              cosij=(rik2+rjk2-rij2)/(2.*rik*rjk)
              cosik=(rij2+rjk2-rik2)/(2.*rij*rjk)
              cosjk=(rij2+rik2-rjk2)/(2.*rik*rij)
              tmp=c9*(3.*cosij*cosik*cosjk+1)/
     .                 (rij*rik*rjk*rij2*rik2*rjk2)

              eabc=eabc+fdamp*tmp*0.5d0
              counter=counter+1
            ENDDO !ktauz
            ENDDO !ktauy
            ENDDO !ktaux
 
          ENDDO !jtauz
          ENDDO !jtauy
          ENDDO !jtaux
        ENDDO !kat
      ENDDO !iat

!      write(*,*)'counter iik: ',counter

! And finally the self interaction iat=jat=kat all 

      DO iat=1,n
      jat=iat
      kat=iat
      ijvec=0.0d0
      ij=lin(iat,jat)
      ik=lin(iat,kat)
      jk=lin(jat,kat)
      ikvec=ijvec
      jkvec=ikvec
          c9=-(cc6ab(ij)*cc6ab(ik)*cc6ab(jk))

        do jtaux=-repv(1),repv(1)
        do jtauy=-repv(2),repv(2)
        do jtauz=-repv(3),repv(3)
          IF (jtaux.eq.0 .and. jtauy.eq.0 .and. jtauz.eq.0) cycle
          jtau=jtaux*lat(:,1)+jtauy*lat(:,2)+jtauz*lat(:,3)
          dumvec=jtau
          dumvec=dumvec*dumvec
          rij2=SUM(dumvec)
          rij=SQRT(rij2)
          rr0ij=rij/r0ab(iz(iat),iz(jat))

          do ktaux=-repv(1),repv(1)
          do ktauy=-repv(2),repv(2)
          do ktauz=-repv(3),repv(3)
            IF (ktaux.eq.0 .and. ktauy.eq.0 .and. ktauz.eq.0) cycle !IF iat and kat are the same then cycle
            IF (ktaux.eq.jtaux .and. ktauy.eq.jtauy 
     .         .and. ktaux.eq.jtaux) cycle      !If kat and jat are the same then cycle
! every result * 1/6 becaues every triple is counted twice due to the two loops jtau and ktau going from -repv to repv -> *1/2
! 
!plus 1/3 becaues every triple is three times in each unitcell
              ktau=ktaux*lat(:,1)+ktauy*lat(:,2)+ktauz*lat(:,3)
              dumvec=ktau
              dumvec=dumvec*dumvec
              rik2=SUM(dumvec)
              rik=SQRT(rik2)
              rr0ik=rik/r0ab(iz(iat),iz(kat))

              dumvec=jkvec+ktau-jtau
              dumvec=dumvec*dumvec
              rjk2=SUM(dumvec)
              rjk=SQRT(rjk2)
              rr0jk=rjk/r0ab(iz(jat),iz(kat))

              geomean=(rr0ij*rr0ik*rr0jk)**(1./3.)
              fdamp=1./(1.+6.*(sr9*geomean)**alp9)
              cosij=(rik2+rjk2-rij2)/(2.*rik*rjk)
              cosik=(rij2+rjk2-rik2)/(2.*rij*rjk)
              cosjk=(rij2+rik2-rjk2)/(2.*rik*rij)
              tmp=c9*(3.*cosij*cosik*cosjk+1)/
     .                 (rij*rik*rjk*rij2*rik2*rjk2)
              eabc=eabc+fdamp*tmp/6.0d0
 
            counter=counter+1
          ENDDO !ktauz
          ENDDO !ktauy
          ENDDO !ktaux
        ENDDO !jtauz
        ENDDO !jtauy
        ENDDO !jtaux


      ENDDO !iat
c      write(*,*)'counter iii: ',counter

      END SUBROUTINE pbcthreebody


c     Input Geometry sanity check for pbc (to avoid au/Angtstrom mixups) J.M. 
      subroutine pbccheckrcov(n,iz,rcov,xyz,lat)
      implicit none
      logical check
      integer iz(*),n,i,j,taux,tauy,tauz
      real*8 rcov(94),dist,dx,dy,dz,thr,xyz(3,*),r,lat(3,3),tau(3)
      check=.false.
      do i=1,n-1
       do j=i+1,n
         do taux=-1,1
         do tauy=-1,1
         do tauz=-1,1
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)

            dx=xyz(1,i)-xyz(1,j)+tau(1)
            dy=xyz(2,i)-xyz(2,j)+tau(2)
            dz=xyz(3,i)-xyz(3,j)+tau(3)
 
           r=sqrt(dx*dx+dy*dy+dz*dz)
           thr=0.6*(rcov(iz(i))+rcov(iz(j)))
           if (r.lt.thr) then
             check=.true.
           endif
         enddo !tauz
         enddo !tauy
         enddo !taux
       enddo !j
      enddo !i
      if (check) then
          write(*,*)'--------------------------------------------------'
          write(*,*)'!! SOME DISTANCES VERY SHORT. CHECK COORDINATES !!'
          write(*,*)'--------------------------------------------------'
      endif
      end subroutine pbccheckrcov

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C compute gradient
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine pbcgdisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .            rcov,s6,s18,rs6,rs8,rs10,alp6,alp8,alp10,noabc,num,
     .                 version,g,disp,gnorm,stress,lat,rep_v,rep_cn,
     .                 crit_vdw,echo,crit_cn)
      
      implicit none  
      !include  'param'

      real*8 k1,k2,k3

c global ad hoc parameters
      parameter (k1=16.0)
      parameter (k2=4./3.) 

c reasonable choices are between 3 and 5
c this gives smoth curves with maxima around the integer values
c k3=3 give for CN=0 a slightly smaller value than computed
c for the free atom. This also yields to larger CN for atoms
c in larger molecules but with the same chem. environment
c which is physically not right
c values >5 might lead to bumps in the potential
      parameter (k3=-4.) 

      integer n,iz(*),max_elem,maxc,version,mxc(max_elem)
      real*8 xyz(3,*),r0ab(max_elem,max_elem),r2r4(*)
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3)
      real*8 g(3,*),s6,s18,rcov(max_elem)
      real*8 rs6,rs8,rs10,alp10,alp8,alp6        
      real*8 a1,a2 !BJ-parameters
      real*8 bj_dmp6,bj_dmp8 ! precalculated dampingterms
      logical noabc,num,echo
c coversion factors
      REAL*8, parameter ::autoang =0.52917726d0
      REAL*8, parameter ::autokcal=627.509541d0
      REAL*8, parameter ::autoev=27.211652d0   

      integer iat,jat,i,j,kat,my,ny,a,b,idum,tau2
      real*8 R0,C6,alp,R42,disp,x1,y1,z1,x2,y2,z2,rr,e6abc,fdum  
      real*8 dx,dy,dz,r2,r,r4,r6,r8,r10,r12,t6,t8,t10,damp1
      real*8 damp6,damp8,damp10,e6,e8,e10,e12,gnorm,tmp1
      real*8 s10,s8,gC6(3),term,step,dispr,displ,r235,tmp2
      real*8 cn(n),gx1,gy1,gz1,gx2,gy2,gz2,rthr,testsum
      real*8,  DIMENSION(3,3) :: lat,stress,sigma,virialstress,lat_1
      real*8,  DIMENSION(3,3) :: gC6_stress
      integer, DIMENSION(3)   :: rep_v,rep_cn
      real*8 crit_vdw,crit_cn
      integer taux,tauy,tauz,counter
      real*8, DIMENSION(3) :: tau,vec12,dxyz,dxyz0
      real*8,external  ::volume
      real*8           ::outpr(3,3)
      real*8, DIMENSION(3,3):: outerprod

      real*8 rij(3),rik(3),rjk(3),r7,r9
      real*8 rik_dist,rjk_dist
      real*8 drik,drjk
      real*8 rcovij
      real*8 dc6,c6chk !d(C6ij)/d(r_ij)
      real*8 expterm,dcni
      real*8, allocatable,dimension(:,:,:,:) ::  drij  !d(E)/d(r_ij) derivative wrt. dist. iat-jat
      real*8, allocatable,dimension(:,:,:,:) :: dcn    !dCN(iat)/d(r_ij) is equal to
                                                       !dCN(jat)/d(r_ij)     
      real*8, allocatable,dimension(:,:,:,:) :: dc6_rest ! saves (1/r^6*f_dmp + 3*r4r2/r^8*f_dmp) for kat loop
      integer,external :: lin
      real*8,external ::vectorsize
      real*8 vec(3),vec2(3),dummy
      real*8 dc6ij(n,n)       !dC6(iat,jat)/dCN(iat) in dc6ij(i,j)
                              !dC6(iat,jat)/cCN(jat) in dc6ij(j,i)
      real*8 dc6_rest_sum(n*(n+1)/2)
      logical, allocatable,dimension(:,:,:,:) ::  skip  !d(E)/d(r_ij) derivative wrt. dist. iat-jat
      integer linij,linik,linjk
      real*8 abc(3,n)



c R^2 cut-off 
      rthr=crit_vdw
      counter=0
      sigma=0.0d0
      virialstress=0.0d0
      
c      testsum=0.0d0

      if(echo)write(*,*) 

      if(num) then
      if (echo)
     .  write(*,*) 'doing numerical gradient O(N^3) ...'

      call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_v,crit_cn,rep_cn)
 

          disp=-s6*e6-s18*e8-s6*e6abc

      step=2.d-5

      do i=1,n
        do j=1,3
          xyz(j,i)=xyz(j,i)+step        
          call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_v,crit_cn,rep_cn)
 
          dispr=-s6*e6-s18*e8-s6*e6abc
          xyz(j,i)=xyz(j,i)-2*step      
          call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_v,crit_cn,rep_cn)
 
          displ=-s6*e6-s18*e8-s6*e6abc
          g(j,i)=0.5*(dispr-displ)/step  
          xyz(j,i)=xyz(j,i)+step        
        enddo !jat
      enddo   !iat
      IF (echo) write(*,*)'Doing numerical stresstensor...'

      call xyz_to_abc(xyz,abc,lat,n)
      step=2.d-5
      if (echo) write(*,*)'step: ',step
      do i=1,3
        do j=1,3
          lat(j,i)=lat(j,i)+step
          call abc_to_xyz(abc,xyz,lat,n)
          !call edisp...dum1
          call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_v,crit_cn,rep_cn)

         dispr=-s6*e6-s18*e8-s6*e6abc


          lat(j,i)=lat(j,i)-2*step
          call abc_to_xyz(abc,xyz,lat,n)
          !call edisp...dum2
          call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_v,crit_cn,rep_cn)

         displ=-s6*e6-s18*e8-s6*e6abc
          stress(j,i)=(dispr-displ)/(step*2.0)

          lat(j,i)=lat(j,i)+step
          call abc_to_xyz(abc,xyz,lat,n)
          
        enddo !j
      enddo !i

      sigma=0.0d0
      call inv_cell(lat,lat_1)
      do a=1,3
        do b=1,3
           do my=1,3
              sigma(a,b)=sigma(a,b)-stress(a,my)*lat(b,my)
           enddo
        enddo !b
      enddo !a

      goto 999

      endif !num


      if(version.eq.2)then
      if(echo)write(*,*) 'doing analytical gradient D-old O(N^2) ...'
      disp=0
      stress=0.0d0
      do iat=1,n-1
         do jat=iat+1,n
           R0=r0ab(iz(jat),iz(iat))*rs6
           c6=c6ab(iz(jat),iz(iat),1,1,1)*s6
           do taux=-rep_v(1),rep_v(1)
           do tauy=-rep_v(2),rep_v(2)
           do tauz=-rep_v(3),rep_v(3)
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
              dxyz=xyz(:,iat)-xyz(:,jat)+tau            
            r2  =sum(dxyz*dxyz)
           if(r2.gt.rthr) cycle
            r235=r2**3.5                       
            r   =dsqrt(r2)
            damp6=exp(-alp6*(r/R0-1.0d0))
            damp1=1.+damp6           
            tmp1=damp6/(damp1*damp1*r235*R0)
            tmp2=6./(damp1*r*r235)

            term=alp6*tmp1-tmp2
              g(:,iat)=g(:,iat)-term*dxyz*c6
              g(:,jat)=g(:,jat)+term*dxyz*c6
            disp=disp+c6*(1./damp1)/r2**3

            do ny=1,3
            do my=1,3
              sigma(my,ny)=sigma(my,ny)+term*dxyz(ny)*dxyz(my)*c6
!              stress(my,ny)=stress(my,ny)-term*c6*dxyz(my)*tau(ny)
            enddo !my
            enddo !ny
           enddo !tauz
           enddo !tauy
           enddo !taux
         enddo !jat
      enddo !iat
c and now the self interaction, only for convenient energy in dispersion
      do iat=1,n
         jat=iat
           R0=r0ab(iz(jat),iz(iat))*rs6
           c6=c6ab(iz(jat),iz(iat),1,1,1)*s6
           do taux=-rep_v(1),rep_v(1)
           do tauy=-rep_v(2),rep_v(2)
           do tauz=-rep_v(3),rep_v(3)
            if (taux.eq.0 .and. tauy.eq.0 .and. tauz.eq.0) cycle
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)

            dxyz=tau
!             vec12=(/ dx,dy,dz /)
            r2  =sum(dxyz*dxyz)
            if(r2.gt.rthr) cycle
            r235=r2**3.5                       
            r   =dsqrt(r2)
            damp6=exp(-alp6*(r/R0-1.0d0))
            damp1=1.+damp6           
            tmp1=damp6/(damp1*damp1*r235*R0)
            tmp2=6./(damp1*r*r235)
            disp=disp+(c6*(1./damp1)/r2**3)*0.50d0
            term=alp6*tmp1-tmp2
            do ny=1,3
            do my=1,3
             sigma(my,ny)=sigma(my,ny)+term*dxyz(ny)*dxyz(my)*c6*0.5d0
!              stress(my,ny)=stress(my,ny)-term*c6*dxyz(my)*tau(ny)*0.5d0
            enddo !my
            enddo !ny
            

           enddo !tauz
           enddo !tauy
           enddo !taux
      enddo !iat
      
      call inv_cell(lat,lat_1)
      do a=1,3
        do b=1,3
           do my=1,3
              stress(a,b)=stress(a,b)-sigma(a,my)*lat_1(b,my)
           enddo
        enddo !b
      enddo !a

      disp=-disp
!       sigma=virialstress
      goto 999
      endif !version==2

      if (version.eq.3) then
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!          
!    begin ZERO DAMPING GRADIENT         
!          
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      if (echo)   
     . write(*,*) 'doing analytical gradient O(N^3) ...'
c precompute for analytical part
      call pbcncoord(n,rcov,iz,xyz,cn,lat,rep_cn,crit_cn)


      s8 =s18
      s10=s18
      allocate(drij(-rep_v(3):rep_v(3),-rep_v(2):rep_v(2),
     .              -rep_v(1):rep_v(1),n*(n+1)/2))
      allocate(dc6_rest(-rep_v(3):rep_v(3),-rep_v(2):rep_v(2),
     .                  -rep_v(1):rep_v(1),n*(n+1)/2))
      allocate(dcn(-rep_cn(3):rep_cn(3),-rep_cn(2):rep_cn(2),
     .             -rep_cn(1):rep_cn(1),n*(n+1)/2))
      allocate(skip(-rep_v(3):rep_v(3),-rep_v(2):rep_v(2),
     .              -rep_v(1):rep_v(1),n*(n+1)/2))

      disp=0

      drij=0.0d0
      dc6_rest=0.0d0
      dc6_rest_sum=0.0d0
      dcn=0.0d0
      kat=0


      do iat=1,n
        call get_dC6_dCNij(maxc,max_elem,c6ab,mxc(iz(iat)),
     .          mxc(iz(iat)),cn(iat),cn(iat),iz(iat),iz(iat),iat,iat,
     .          c6,dc6ij(iat,iat),fdum)

        r0=r0ab(iz(iat),iz(iat))
        r42=r2r4(iz(iat))*r2r4(iz(iat))
        rcovij=rcov(iz(iat))+rcov(iz(iat))


      counter=0
        do taux=-rep_v(1),rep_v(1)
        do tauy=-rep_v(2),rep_v(2)
        do tauz=-rep_v(3),rep_v(3)
          tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
          counter=counter+1


!first dE/d(tau) saved in drij(i,i,counter)
          rij=tau
          r2=sum(rij*rij)
!          if (r2.gt.rthr) cycle

!          if (r2.gt.0.1) then
          if (r2.gt.0.1.and.r2.lt.rthr) then


          r=dsqrt(r2)
          r6=r2*r2*r2
          r7=r6*r
          r8=r6*r2
          r9=r8*r

!
!  Calculates damping functions:
          t6 = (r/(rs6*R0))**(-alp6)
          damp6 =1.d0/( 1.d0+6.d0*t6 )
          t8 = (r/(rs8*R0))**(-alp8)
          damp8 =1.d0/( 1.d0+6.d0*t8 )

          drij(tauz,tauy,taux,lin(iat,iat))=drij(tauz,tauy,taux,lin(iat,
     .     iat))
     .        +(-s6*(6.0/(r7)*C6*damp6)  ! d(r^(-6))/d(tau)
     .        -s8*(24.0/(r9)*C6*r42*damp8))*0.5d0


          drij(tauz,tauy,taux,lin(iat,iat))=drij(tauz,tauy,taux,lin(iat,
     .     iat))
     .        +(s6*C6/r7*6.d0*alp6*t6*damp6*damp6     !d(f_dmp)/d(tau)
     .        +s8*C6*r42/r9*18.d0*alp8*t8*damp8*damp8)*0.5d0
!
!      in dC6_rest all terms BUT C6-term is saved for the kat-loop
!          
          dc6_rest(tauz,tauy,taux,lin(iat,iat))=
     .        (s6/r6*damp6+3.d0*s8*r42/r8*damp8)*0.50d0


          disp=disp-dc6_rest(tauz,tauy,taux,lin(iat,iat))*c6  ! calculate E_disp for sanity check

!          if (r2.lt.crit_cn)
          dc6_rest_sum(lin(iat,iat))=dc6_rest_sum(lin(iat,iat))+
     .     (dc6_rest(tauz,tauy,taux,lin(iat,iat)))


          else !r2 < 0.1>rthr
             drij(tauz,tauy,taux,lin(iat,iat))=0.0d0
          endif


        ENDDO !tauz
        ENDDO !tauy
        ENDDO !taux

!!!!!!!!!!!!!!!!!!!!!!!!!!
! B E G I N   jat  L O O P 
!!!!!!!!!!!!!!!!!!!!!!!!!!         
        do jat=1,iat-1
!
!      get_dC6_dCNij calculates the derivative dC6(iat,jat)/dCN(iat) and
!      dC6(iat,jat)/dCN(jat). these are saved in dC6ij for the kat loop
!
          call get_dC6_dCNij(maxc,max_elem,c6ab,mxc(iz(iat)),
     .          mxc(iz(jat)),cn(iat),cn(jat),iz(iat),iz(jat),iat,jat,
     .          c6,dc6ij(iat,jat),dc6ij(jat,iat))

          r0=r0ab(iz(jat),iz(iat))
          r42=r2r4(iz(iat))*r2r4(iz(jat))
          rcovij=rcov(iz(iat))+rcov(iz(jat))
          linij=lin(iat,jat)
 
          counter=0
            do taux=-rep_v(1),rep_v(1)
            do tauy=-rep_v(2),rep_v(2)
            do tauz=-rep_v(3),rep_v(3)
              tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
              counter=counter+1
  
  
            rij=xyz(:,jat)-xyz(:,iat)+tau
            r2=sum(rij*rij)
            if (r2.gt.rthr) cycle
  
            skip(tauz,tauy,taux,linij)=.false.
 
            r=dsqrt(r2)
            r6=r2*r2*r2
            r7=r6*r
            r8=r6*r2
            r9=r8*r
  
!
!  Calculates damping functions:
            t6 = (r/(rs6*R0))**(-alp6)
            damp6 =1.d0/( 1.d0+6.d0*t6 )
            t8 = (r/(rs8*R0))**(-alp8)
            damp8 =1.d0/( 1.d0+6.d0*t8 )
  
            drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .           linij)
     .        -s6*(6.0/(r7)*C6*damp6)  ! d(r^(-6))/d(r_ij)
     .        -s8*(24.0/(r9)*C6*r42*damp8)
  
            drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .           linij)
     .        +s6*C6/r7*6.d0*alp6*t6*damp6*damp6     !d(f_dmp)/d(r_ij)
     .        +s8*C6*r42/r9*18.d0*alp8*t8*damp8*damp8
!
!      in dC6_rest all terms BUT C6-term is saved for the kat-loop
!          
          dc6_rest(tauz,tauy,taux,linij)=
     .        (s6/r6*damp6+3.d0*s8*r42/r8*damp8)

 
            disp=disp-dc6_rest(tauz,tauy,taux,linij)*c6  ! calculate E_disp for sanity check

!            if (r2.lt.crit_cn) 
             dc6_rest_sum(linij)=dc6_rest_sum(linij)
     .        +dc6_rest(tauz,tauy,taux,linij) 


          enddo !tauz
          enddo !tauy
          enddo !taux
  
        enddo !jat

      enddo !iat

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!      !B E G I N   d(C6)/dr
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      skip=.true.

      DO iat=1,n
        r0=r0ab(iz(iat),iz(iat))
        r42=r2r4(iz(iat))*r2r4(iz(iat))
        rcovij=rcov(iz(iat))+rcov(iz(iat))

        do taux=-rep_cn(1),rep_cn(1)
        do tauy=-rep_cn(2),rep_cn(2)
        do tauz=-rep_cn(3),rep_cn(3)
          tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
          r2=sum(tau*tau)
          if (r2.gt.0.1.and.r2.lt.crit_cn) then
            r=dsqrt(r2)

            skip(tauz,tauy,taux,lin(iat,iat))=.false.
!
!         Calculate dCN(iat)/dr_ij which is identical to dCN(iat)/d(tau)
!          this is needed for dC6/dr_ij          
!         saved in dcn for the kat-loop
!          
          
            expterm=exp(-k1*(rcovij/r-1.d0))
            dcn(tauz,tauy,taux,lin(iat,iat))=-k1*rcovij*expterm/
     .                 (r2*(expterm+1.d0)*(expterm+1.d0))

!     
!     Combine dC6/dCN * dCN/dr_ij to get dC6/dr_ij          
          dc6=(dc6ij(iat,iat)+dc6ij(iat,iat))*              !dC6(iat,iat)/dCN(iat) * dCN(iat)/dr(ii)
     .         dcn(tauz,tauy,taux,lin(iat,iat))

          drij(tauz,tauy,taux,lin(iat,iat))=drij(tauz,tauy,taux,
     .         lin(iat,iat))
     .         +  dc6_rest_sum(lin(iat,iat))*dc6            !d(C6(ii))/d(tau)


          endif ! r2<crit_cn
        enddo !tauz
        enddo !tauy
        enddo !taux

        DO jat=1,iat-1

          rcovij=rcov(iz(iat))+rcov(iz(jat))

          linij=lin(iat,jat)

!        write(*,*)'iat,jat:',iat,jat
            do taux=-rep_cn(1),rep_cn(1)
            do tauy=-rep_cn(2),rep_cn(2)
            do tauz=-rep_cn(3),rep_cn(3)
  

              if (.not.skip(tauz,tauy,taux,lin(iat,iat))) then
                dc6=(dc6ij(iat,jat))*                             !dC6(iat,jat)/dCN(iat) * dCN(iat)/dr(ii)
     .           dcn(tauz,tauy,taux,lin(iat,iat))

                drij(tauz,tauy,taux,lin(iat,iat))=drij(tauz,tauy,taux,
     .             lin(iat,iat))
     .             +  dc6_rest_sum(lin(iat,jat))*dc6            !d(C6(ij))/d(tau)
              endif


              if (.not.skip(tauz,tauy,taux,lin(jat,jat))) then
                dc6=(dc6ij(jat,iat))*                             !dC6(iat,jat)/dCN(iat) * dCN(iat)/dr(ii)
     .           dcn(tauz,tauy,taux,lin(jat,jat))

                drij(tauz,tauy,taux,lin(jat,jat))=drij(tauz,tauy,taux,
     .           lin(jat,jat))
     .           +  dc6_rest_sum(lin(jat,iat))*dc6            !d(C6(ij))/d(tau)
              endif

              tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
              rij=xyz(:,jat)-xyz(:,iat)+tau
              r2=sum(rij*rij)
              if (r2.gt.crit_cn) cycle
              r=dsqrt(r2)
              skip(tauz,tauy,taux,linij)=.false.

!           Calculate dCN(iat)/dr_ij which is identical to dCN(iat)/d(tau)
!           this is needed for dC6/dr_ij          
!           saved in dcn for the kat-loop
!          
          
              expterm=exp(-k1*(rcovij/r-1.d0))
              dcn(tauz,tauy,taux,linij)=-k1*rcovij*expterm/
     .                 (r2*(expterm+1.d0)*(expterm+1.d0))

              dc6=(dc6ij(iat,jat)+dc6ij(jat,iat))*  !dC6(iat,jat)/dCN(iat) * dCN(iat)/dr(ij)
     .             dcn(tauz,tauy,taux,linij)        !dC6(iat,jat)/dCN(jat) * dCN(jat)/dr(ij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .          linij)
     .          +dc6_rest_sum(linij)*dc6 

               dc6=(dc6ij(iat,iat)+dc6ij(iat,iat))* !dC6(iat,iat)/dCN(iat) * dCN(iat)/dr(ij)
     .             dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .          linij)
     .          +dc6_rest_sum(lin(iat,iat))*dc6 

              dc6=(dc6ij(jat,jat)+dc6ij(jat,jat))* !dC6(jat,jat)/dCN(jat) * dCN(jat)/dr(ij)
     .             dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .          linij)
     .          +dc6_rest_sum(lin(jat,jat))*dc6 



            enddo !tauz
            enddo !tauy
            enddo !taux
!
! In the kat loop all the 3rd atom contributions are calculated:
!            1/r_ij^6*f_dmp(ij)  *  d(C6(ij))/d r_ik
!            -------V----------     -------V--------
!             dc6_rest_sum(ij)   *  dc6ij(i,j)*dcn(ik)
!
!   To reduce the kat-loop to only jat-1, on gets 6 contributions
!   
!
          do kat=1,jat-1
            linik=lin(iat,kat)
            linjk=lin(jat,kat)
            do taux=-rep_cn(1),rep_cn(1)
            do tauy=-rep_cn(2),rep_cn(2)
            do tauz=-rep_cn(3),rep_cn(3)
!              tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)




            if (.not.skip(tauz,tauy,taux,linij)) then
              dc6=dc6ij(iat,kat)*dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .             linij)
     .             +dc6_rest_sum(linik)*dc6

              dc6=dc6ij(jat,kat)*dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .             linij)
     .             +dc6_rest_sum(linjk)*dc6

            endif


            if (.not.skip(tauz,tauy,taux,linjk)) then
              dc6=dc6ij(kat,iat)*dcn(tauz,tauy,taux,linjk)

              drij(tauz,tauy,taux,linjk)=drij(tauz,tauy,taux,
     .             linjk)
     .             +dc6_rest_sum(linik)*dc6

              dc6=dc6ij(jat,iat)*dcn(tauz,tauy,taux,linjk)

              drij(tauz,tauy,taux,linjk)=drij(tauz,tauy,taux,
     .             linjk)
     .             +dc6_rest_sum(linij)*dc6

            endif


            if (.not.skip(tauz,tauy,taux,linik)) then
              dc6=dc6ij(kat,jat)*dcn(tauz,tauy,taux,linik)

              drij(tauz,tauy,taux,linik)=drij(tauz,tauy,taux,
     .             linik)
     .             +dc6_rest_sum(linjk)*dc6

              dc6=dc6ij(iat,jat)*dcn(tauz,tauy,taux,linik)

              drij(tauz,tauy,taux,linik)=drij(tauz,tauy,taux,
     .             linik)
     .             +dc6_rest_sum(linij)*dc6
            endif



          


            enddo !tauz
            enddo !tauy
            enddo !taux
          enddo !kat
        ENDDO !jat


      ENDDO !iat






! After calculating all derivatives dE/dr_ij w.r.t. distances,
! the grad w.r.t. the coordinates is calculated dE/dr_ij * dr_ij/dxyz_i       
      do iat=2,n
        do jat=1,iat-1
          do taux=-rep_v(1),rep_v(1)
          do tauy=-rep_v(2),rep_v(2)
          do tauz=-rep_v(3),rep_v(3)
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)

          rij=xyz(:,jat)-xyz(:,iat)+tau
          r2=sum(rij*rij)  
          if (r2.gt.rthr.or.r2.lt.0.5) cycle

          r=dsqrt(r2)
          vec=drij(tauz,tauy,taux,lin(iat,jat))*rij/r
          g(:,iat)=g(:,iat)+vec
          g(:,jat)=g(:,jat)-vec
          do i=1,3
          do j=1,3
          sigma(j,i)=sigma(j,i)+vec(j)*rij(i)
          enddo !j
          enddo !i


          enddo !tauz
          enddo !tauy
          enddo !taux
        enddo !jat
      enddo !iat

      do iat=1,n
          do taux=-rep_v(1),rep_v(1)
          do tauy=-rep_v(2),rep_v(2)
          do tauz=-rep_v(3),rep_v(3)
!          if (taux.eq.0.and.tauy.eq.0.and.tauz.eq.0) cycle
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
          r2=sum(tau*tau)  
          if (r2.gt.rthr.or.r2.lt.0.5) cycle
          r=dsqrt(r2)
          vec=drij(tauz,tauy,taux,lin(iat,iat))*tau/r
          do i=1,3
          do j=1,3
          sigma(j,i)=sigma(j,i)+vec(j)*tau(i)
          enddo !j
          enddo !i




          enddo !tauz
          enddo !tauy
          enddo !taux

      enddo

      stress=0.0d0
      call inv_cell(lat,lat_1)
      do a=1,3
        do b=1,3
           do my=1,3
              stress(a,b)=stress(a,b)-sigma(a,my)*lat_1(b,my)
           enddo
        enddo !b
      enddo !a



!          write(*,*)'drij:',drij(lin(iat,jat),:)
!          write(*,*)'g:',g(1,1:3)
!          write(*,*)'dcn:',sum(dcn(lin(2,1),:))

      deallocate(drij,dc6_rest,dcn)

      elseif (version.eq.4)  then



!!!!!!!!!!!!!!!!!!!!!!!
! NOW THE BJ Gradient !
!!!!!!!!!!!!!!!!!!!!!!!


      if (echo) write(*,*) 'doing analytical gradient O(N^3) ...'
      call pbcncoord(n,rcov,iz,xyz,cn,lat,rep_cn,crit_cn)

      a1 =rs6
      a2 =rs8
      s8 =s18

      allocate(drij(-rep_v(3):rep_v(3),-rep_v(2):rep_v(2),
     .              -rep_v(1):rep_v(1),n*(n+1)/2))
      allocate(dc6_rest(-rep_v(3):rep_v(3),-rep_v(2):rep_v(2),
     .                  -rep_v(1):rep_v(1),n*(n+1)/2))
      allocate(dcn(-rep_cn(3):rep_cn(3),-rep_cn(2):rep_cn(2),
     .             -rep_cn(1):rep_cn(1),n*(n+1)/2))

      allocate(skip(-rep_v(3):rep_v(3),-rep_v(2):rep_v(2),
     .              -rep_v(1):rep_v(1),n*(n+1)/2))
      disp=0
      drij=0.0d0
      dc6_rest=0.0d0
      dc6_rest_sum=0.0d0
      dcn=0.0d0
      kat=0

      do iat=1,n
        call get_dC6_dCNij(maxc,max_elem,c6ab,mxc(iz(iat)),
     .          mxc(iz(iat)),cn(iat),cn(iat),iz(iat),iz(iat),iat,iat,
     .          c6,dc6ij(iat,iat),fdum)

        r42=r2r4(iz(iat))*r2r4(iz(iat))
        rcovij=rcov(iz(iat))+rcov(iz(iat))

        R0=a1*sqrt(3.0d0*r42)+a2

        do taux=-rep_v(1),rep_v(1)
        do tauy=-rep_v(2),rep_v(2)
        do tauz=-rep_v(3),rep_v(3)
          tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
          counter=counter+1

!first dE/d(tau) saved in drij(i,i,counter)
          rij=tau
          r2=sum(rij*rij)
!          if (r2.gt.rthr) cycle

!          if (r2.gt.0.1) then
          if (r2.gt.0.1.and.r2.lt.rthr) then
!
!      get_dC6_dCNij calculates the derivative dC6(iat,jat)/dCN(iat) and
!      dC6(iat,jat)/dCN(jat). these are saved in dC6ij for the kat loop
!
          r=dsqrt(r2)
          r4=r2*r2
          r6=r4*r2
          r7=r6*r
          r8=r6*r2
          r9=r8*r

!
!  Calculates damping functions:
 
          t6=(r6+R0**6)
          t8=(r8+R0**8)

          drij(tauz,tauy,taux,lin(iat,iat))=drij(tauz,tauy,taux,lin(iat,
     .     iat))
     .        -s6*C6*6.0d0*r4*r/(t6*t6)*0.5d0 ! d(1/(r^(6)+R0^6)/d(r)
     .        -s8*C6*24.0d0*r42*r7/(t8*t8)*0.5d0


!
!      in dC6_rest all terms BUT C6-term is saved for the kat-loop
!          
          dc6_rest(tauz,tauy,taux,lin(iat,iat))=
     .        (s6/t6+3.d0*s8*r42/t8)*0.50d0


          disp=disp-dc6_rest(tauz,tauy,taux,lin(iat,iat))*c6  ! calculate E_disp for sanity check

!          if (r2.lt.crit_cn)
          dc6_rest_sum(lin(iat,iat))=dc6_rest_sum(lin(iat,iat))+
     .     (dc6_rest(tauz,tauy,taux,lin(iat,iat)))


          else !r2 < 0.1>rthr
             drij(tauz,tauy,taux,lin(iat,iat))=0.0d0
          endif


        ENDDO !tauz
        ENDDO !tauy
        ENDDO !taux

!!!!!!!!!!!!!!!!!!!!!!!!!!
! B E G I N   jat  L O O P 
!!!!!!!!!!!!!!!!!!!!!!!!!!         
        do jat=1,iat-1
!
!      get_dC6_dCNij calculates the derivative dC6(iat,jat)/dCN(iat) and
!      dC6(iat,jat)/dCN(jat). these are saved in dC6ij for the kat loop
!
          call get_dC6_dCNij(maxc,max_elem,c6ab,mxc(iz(iat)),
     .          mxc(iz(jat)),cn(iat),cn(jat),iz(iat),iz(jat),iat,jat,
     .          c6,dc6ij(iat,jat),dc6ij(jat,iat))

          r42=r2r4(iz(iat))*r2r4(iz(jat))
          rcovij=rcov(iz(iat))+rcov(iz(jat))
 
          R0=a1*dsqrt(3.0d0*r42)+a2

            do taux=-rep_v(1),rep_v(1)
            do tauy=-rep_v(2),rep_v(2)
            do tauz=-rep_v(3),rep_v(3)
              tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
  
  
            rij=xyz(:,jat)-xyz(:,iat)+tau
            r2=sum(rij*rij)
            if (r2.gt.rthr) cycle
  
 
            r=dsqrt(r2)
            r4=r2*r2
            r6=r4*r2
            r7=r6*r
            r8=r6*r2
            r9=r8*r
  
!
!  Calculates damping functions:
            t6=(r6+R0**6)
            t8=(r8+R0**8)

 
            drij(tauz,tauy,taux,lin(iat,jat))=drij(tauz,tauy,taux,
     .           lin(iat,jat))
     .        -s6*C6*6.0d0*r4*r/(t6*t6)
     .        -s8*C6*24.0d0*r42*r7/(t8*t8)

!
!      in dC6_rest all terms BUT C6-term is saved for the kat-loop
!          
            dc6_rest(tauz,tauy,taux,lin(iat,jat))=
     .        (s6/t6+3.d0*s8*r42/t8)

 
            disp=disp-dc6_rest(tauz,tauy,taux,lin(iat,jat))*c6  ! calculate E_disp for sanity check

!            if (r2.lt.crit_cn) 
            dc6_rest_sum(lin(iat,jat))=dc6_rest_sum(lin(iat,jat))
     .        +dc6_rest(tauz,tauy,taux,lin(iat,jat)) 


          enddo !tauz
          enddo !tauy
          enddo !taux
  
        enddo !jat

      enddo !iat

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!      !B E G I N   d(C6)/dr
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      skip=.true.

      DO iat=1,n
        r0=r0ab(iz(iat),iz(iat))
        r42=r2r4(iz(iat))*r2r4(iz(iat))
        rcovij=rcov(iz(iat))+rcov(iz(iat))

        do taux=-rep_cn(1),rep_cn(1)
        do tauy=-rep_cn(2),rep_cn(2)
        do tauz=-rep_cn(3),rep_cn(3)
          tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
          r2=sum(tau*tau)
          if (r2.gt.0.1.and.r2.lt.crit_cn) then
            r=dsqrt(r2)
            skip(tauz,tauy,taux,lin(iat,iat))=.false.
!
!         Calculate dCN(iat)/dr_ij which is identical to dCN(iat)/d(tau)
!          this is needed for dC6/dr_ij          
!         saved in dcn for the kat-loop
!          
          
            expterm=exp(-k1*(rcovij/r-1.d0))
            dcn(tauz,tauy,taux,lin(iat,iat))=-k1*rcovij*expterm/
     .                 (r2*(expterm+1.d0)*(expterm+1.d0))

!     
!     Combine dC6/dCN * dCN/dr_ij to get dC6/dr_ij          
          dc6=(dc6ij(iat,iat)+dc6ij(iat,iat))*
     .         dcn(tauz,tauy,taux,lin(iat,iat))

          drij(tauz,tauy,taux,lin(iat,iat))=drij(tauz,tauy,taux,
     .         lin(iat,iat))
     .         +  dc6_rest_sum(lin(iat,iat))*dc6            !d(C6(ij))/d(tau)


          endif ! r2<crit_cn
        enddo !tauz
        enddo !tauy
        enddo !taux

        DO jat=1,iat-1

          rcovij=rcov(iz(iat))+rcov(iz(jat))
          linij=lin(iat,jat)


            do taux=-rep_cn(1),rep_cn(1)
            do tauy=-rep_cn(2),rep_cn(2)
            do tauz=-rep_cn(3),rep_cn(3)

              if (.not.skip(tauz,tauy,taux,lin(iat,iat))) then
                dc6=(dc6ij(iat,jat))*                             !dC6(iat,jat)/dCN(iat) * dCN(iat)/dr(ii)
     .           dcn(tauz,tauy,taux,lin(iat,iat))

                drij(tauz,tauy,taux,lin(iat,iat))=drij(tauz,tauy,taux,
     .             lin(iat,iat))
     .             +  dc6_rest_sum(lin(iat,jat))*dc6            !d(C6(ij))/d(tau)
              endif


              if (.not.skip(tauz,tauy,taux,lin(jat,jat))) then
                dc6=(dc6ij(jat,iat))*                             !dC6(iat,jat)/dCN(iat) * dCN(iat)/dr(ii)
     .           dcn(tauz,tauy,taux,lin(jat,jat))

                drij(tauz,tauy,taux,lin(jat,jat))=drij(tauz,tauy,taux,
     .           lin(jat,jat))
     .           +  dc6_rest_sum(lin(jat,iat))*dc6            !d(C6(ij))/d(tau)
              endif

              tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
  
              rij=xyz(:,jat)-xyz(:,iat)+tau
              r2=sum(rij*rij)
              if (r2.gt.crit_cn) cycle
              r=dsqrt(r2)
              skip(tauz,tauy,taux,linij)=.false.
!           Calculate dCN(iat)/dr_ij which is identical to dCN(iat)/d(tau)
!           this is needed for dC6/dr_ij          
!           saved in dcn for the kat-loop
!          
          
              expterm=exp(-k1*(rcovij/r-1.d0))
              dcn(tauz,tauy,taux,linij)=-k1*rcovij*expterm/
     .                 (r2*(expterm+1.d0)*(expterm+1.d0))

              dc6=(dc6ij(iat,jat)+dc6ij(jat,iat))*
     .             dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .          linij)
     .          +dc6_rest_sum(linij)*dc6 

               dc6=(dc6ij(iat,iat)+dc6ij(iat,iat))*
     .             dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .          linij)
     .          +dc6_rest_sum(lin(iat,iat))*dc6 

              dc6=(dc6ij(jat,jat)+dc6ij(jat,jat))*
     .             dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .          linij)
     .          +dc6_rest_sum(lin(jat,jat))*dc6 



            enddo !tauz
            enddo !tauy
            enddo !taux
!
! In the kat loop all the 3rd atom contributions are calculated:
!            1/r_ij^6*f_dmp(ij)  *  d(C6(ij))/d r_ik
!            -------V----------     -------V--------
!             dc6_rest_sum(ij)   *  dc6ij(i,j)*dcn(ik)
!
!   To reduce the kat-loop to only jat-1, on gets 6 contributions
!   
!
          do kat=1,jat-1
            linik=lin(iat,kat)
            linjk=lin(jat,kat)
            do taux=-rep_cn(1),rep_cn(1)
            do tauy=-rep_cn(2),rep_cn(2)
            do tauz=-rep_cn(3),rep_cn(3)


              if (.not.skip(tauz,tauy,taux,linij)) then
              dc6=dc6ij(iat,kat)*dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .             linij)
     .             +dc6_rest_sum(linik)*dc6

              dc6=dc6ij(jat,kat)*dcn(tauz,tauy,taux,linij)

              drij(tauz,tauy,taux,linij)=drij(tauz,tauy,taux,
     .             linij)
     .             +dc6_rest_sum(linjk)*dc6
              endif

              if (.not.skip(tauz,tauy,taux,linjk)) then
              dc6=dc6ij(kat,iat)*dcn(tauz,tauy,taux,linjk)

              drij(tauz,tauy,taux,linjk)=drij(tauz,tauy,taux,
     .             linjk)
     .             +dc6_rest_sum(linik)*dc6

              dc6=dc6ij(jat,iat)*dcn(tauz,tauy,taux,linjk)

              drij(tauz,tauy,taux,linjk)=drij(tauz,tauy,taux,
     .             linjk)
     .             +dc6_rest_sum(linij)*dc6
              endif




              if (.not.skip(tauz,tauy,taux,linik)) then
              dc6=dc6ij(kat,jat)*dcn(tauz,tauy,taux,linik)

              drij(tauz,tauy,taux,linik)=drij(tauz,tauy,taux,
     .             linik)
     .             +dc6_rest_sum(linjk)*dc6

              dc6=dc6ij(iat,jat)*dcn(tauz,tauy,taux,linik)

              drij(tauz,tauy,taux,linik)=drij(tauz,tauy,taux,
     .             linik)
     .             +dc6_rest_sum(linij)*dc6
              endif

            enddo !tauz
            enddo !tauy
            enddo !taux
          enddo !kat
        ENDDO !jat
      ENDDO !iat

! After calculating all derivatives dE/dr_ij w.r.t. distances,
! the grad w.r.t. the coordinates is calculated dE/dr_ij * dr_ij/dxyz_i       
      do iat=2,n
        do jat=1,iat-1
          do taux=-rep_v(1),rep_v(1)
          do tauy=-rep_v(2),rep_v(2)
          do tauz=-rep_v(3),rep_v(3)
            tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)

          rij=xyz(:,jat)-xyz(:,iat)+tau
          r=dsqrt(sum(rij*rij))
          vec=drij(tauz,tauy,taux,lin(iat,jat))*rij/r
          vec2(1)=taux
          vec2(2)=tauy
          vec2(3)=tauz
          g(:,iat)=g(:,iat)+vec
          g(:,jat)=g(:,jat)-vec
          do i=1,3
          do j=1,3
          sigma(j,i)=sigma(j,i)+vec(j)*rij(i)
          enddo !j
          enddo !i



          enddo !tauz
          enddo !tauy
          enddo !taux
        enddo !jat
      enddo !iat

      do iat=1,n
          do taux=-rep_v(1),rep_v(1)
          do tauy=-rep_v(2),rep_v(2)
          do tauz=-rep_v(3),rep_v(3)
          if (taux.eq.0.and.tauy.eq.0.and.tauz.eq.0) cycle

          tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
          r=dsqrt(sum(tau*tau))
          vec=drij(tauz,tauy,taux,lin(iat,iat))*tau/r
          vec2(1)=taux
          vec2(2)=tauy
          vec2(3)=tauz
          do i=1,3
          do j=1,3
            sigma(j,i)=sigma(j,i)+vec(j)*tau(i)
          enddo !j
          enddo !i


          enddo !tauz
          enddo !tauy
          enddo !taux



      enddo

      stress=0.0d0
      call inv_cell(lat,lat_1)
      do a=1,3
        do b=1,3
           do my=1,3
              stress(a,b)=stress(a,b)-sigma(a,my)*lat_1(b,my)
           enddo
        enddo !b
      enddo !a



!          write(*,*)'drij:',drij(lin(iat,jat),:)
!          write(*,*)'g:',g(1,1:3)
!          write(*,*)'dcn:',sum(dcn(lin(2,1),:))

      deallocate(drij,dc6_rest,dcn)



      endif ! version


 999  continue
!      do i=1,n
!        write(*,'(83F17.12)') g(1:3,i)
!      enddo
      gnorm=sum(abs(g(1:3,1:n)))
      if(echo)then
c      write(*,*)'testsum:',testsum*autoev/autoang
      write(*,*)'|G(force)| =',gnorm
      gnorm=sum(abs(stress(1:3,1:3)))
      write(*,*)'|G(stress)|=',gnorm
      endif

      end subroutine pbcgdisp


      subroutine pbcwregrad(nat,g,g_lat)
      implicit none
      integer nat,i
      real*8 g(3,nat)
      real*8 g_lat(3,3)

      open(unit=142,file='dftd3_gradient')

!      write(*,*)'Gradient:' !Jonas
!      write(*,*)            !Jonas
      do i=1,nat
         write(142,'(3E22.14)')g(1:3,i)
!         write(*,'(3D22.14)')g(1:3,i) !Jonas
      enddo

      close(142)

      open(unit=143,file='dftd3_cellgradient')

!      write(*,*)'Gradient:' !Jonas
!      write(*,*)            !Jonas
      do i=1,3
         write(143,'(3E22.14)')g_lat(1:3,i)
!         write(*,'(3D22.14)')g(1:3,i) !Jonas
      enddo

      close(143)
      end subroutine pbcwregrad

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C analyse all pairs
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine pbcadisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,autokcal,
     .           autoang,rthr,rep_v,cn_thr,rep_cn,s6,s18,etot,lat)
      implicit none  
      integer n,iz(*),max_elem,maxc,version,mxc(max_elem) 
      real*8 xyz(3,*),r0ab(max_elem,max_elem),r2r4(*),s6
      real*8 rs6,rs8,rs10,alp6,alp8,alp10,autokcal,etot,s18,autoang
      real*8 c6ab(max_elem,max_elem,maxc,maxc,3),rcov(max_elem)
      real*8 lat(3,3)
      integer rep_v(3),rep_cn(3)
 
      integer iat,jat,i,j,k,nbin,taux,tauy,tauz
      real*8 R0,r,r2,r6,r8,tmp,alp,dx,dy,dz,c6,c8,c10
      real*8 damp6,damp8,damp10,r42,rr,check,rthr,cn_thr,rvdw
      real*8 cn(n),i6,e6,e8,e10,edisp                   
      real*8,allocatable ::  dist(:),li(:,:)
      real*8 xx(500),eg(10000)
      integer grplist(500,20)
      integer grpn(20),at(n)
      integer ngrp,dash
      integer lin, iiii, jjjj, iii, jjj, ii, jj, ni, nj 
      integer iout(500)
      logical ex
      character*80 atmp
      real*8 tau(3)
 
      real*8,dimension(:,:), allocatable :: ed
      allocate(ed(n,n))


c distance bins
      nbin=17
      allocate(dist(0:nbin))
      allocate(li(0:nbin,2))

      li(0,1)=0   
      li(0,2)=1.5 
      li(1,1)=1.5
      li(1,2)=2
      li(2,1)=2
      li(2,2)=2.3333333333
      li(3,1)=2.3333333333
      li(3,2)=2.6666666666
      li(4,1)=2.6666666666
      li(4,2)=3.0
      li(5,1)=3.0          
      li(5,2)=3.3333333333
      li(6,1)=3.3333333333
      li(6,2)=3.6666666666
      li(7,1)=3.6666666666
      li(7,2)=4.0
      li(8,1)=4.0
      li(8,2)=4.5
      li(9,1)=4.5
      li(9,2)=5.0
      li(10,1)=5.0
      li(10,2)=5.5
      li(11,1)=5.5
      li(11,2)=6.0
      li(12,1)=6.0
      li(12,2)=7.0           
      li(13,1)=7.0           
      li(13,2)=8.0           
      li(14,1)=8.0           
      li(14,2)=9.0           
      li(15,1)=9.0           
      li(15,2)=10.0          
      li(16,1)=10.0           
      li(16,2)=20.0          
      li(17,1)=20.0           
      li(17,2)=dsqrt(rthr)*autoang


      call pbcncoord(n,rcov,iz,xyz,cn,lat,rep_cn,cn_thr)

      write(*,*)
      write(*,*)'analysis of pair-wise terms (in kcal/mol)'
      write(*,'(''pair'',2x,''atoms'',9x,''C6'',14x,''C8'',12x,
     .''E6'',7x,''E8'',7x,''Edisp'')')
      e8=0
      ed=0
      dist=0
      check=0
      do iat=1,n
         do jat=iat,n

             do taux=-rep_v(1),rep_v(1)
             do tauy=-rep_v(2),rep_v(2)
             do tauz=-rep_v(3),rep_v(3)
              tau=taux*lat(:,1)+tauy*lat(:,2)+tauz*lat(:,3)
            dx=xyz(1,iat)-xyz(1,jat)+tau(1)
            dy=xyz(2,iat)-xyz(2,jat)+tau(2)
            dz=xyz(3,iat)-xyz(3,jat)+tau(3)
            r2=(dx*dx+dy*dy+dz*dz)
CTHR
            if(r2.gt.rthr.or.r2.lt.0.5) cycle
            r =sqrt(r2)
            R0=r0ab(iz(jat),iz(iat))
            rr=R0/r
            r6=r2**3       

            tmp=rs6*rr   
            damp6 =1.d0/( 1.d0+6.d0*tmp**alp6 )
            tmp=rs8*rr     
            damp8 =1.d0/( 1.d0+6.d0*tmp**alp8 )

            if (version.eq.2)then
              c6=c6ab(iz(jat),iz(iat),1,1,1)
              damp6=1.d0/(1.d0+exp(-alp6*(r/(rs6*R0)-1.0d0)))
              if (iat.eq.jat) then
                e6 =s6*autokcal*c6*damp6/r6
              else
                e6 =s6*autokcal*c6*damp6/r6
              endif
              e8=0.0d0
            else
              call getc6(maxc,max_elem,c6ab,mxc,iz(iat),iz(jat),
     .                                      cn(iat),cn(jat),c6)
            endif

            if(version.eq.3)then
              r8 =r6*r2
              r42=r2r4(iz(iat))*r2r4(iz(jat))
              c8 =3.0d0*c6*r42
              if (iat.eq.jat) then
                e6 =s6*autokcal*c6*damp6/r6*0.5
                e8 =s18*autokcal*c8*damp8/r8*0.5
              else
                e6 =s6*autokcal*c6*damp6/r6
                e8 =s18*autokcal*c8*damp8/r8
              endif
            endif

            if(version.eq.4)then
              r42=r2r4(iz(iat))*r2r4(iz(jat))
              c8 =3.0d0*c6*r42
c use BJ radius
              R0=dsqrt(c8/c6)              
              rvdw=rs6*R0+rs8
              r8 =r6*r2
              if (iat.eq.jat) then
                e6 =s6*autokcal*c6/(r6+rvdw**6)*0.5
                e8 =s18*autokcal*c8/(r8+rvdw**8)*0.5
              else
                e6 =s6*autokcal*c6/(r6+rvdw**6)
                e8 =s18*autokcal*c8/(r8+rvdw**8)
              endif
            endif

            edisp=-(e6+e8)
            ed(iat,jat)=edisp
            ed(jat,iat)=edisp

!           write(*,'(2i4,2x,2i3,2D16.6,2F9.4,F10.5)')
!     .     iat,jat,iz(iat),iz(jat),c6,c8,
!     .    -e6,-e8,edisp

            check=check+edisp
            rr=r*autoang
            do i=0,nbin
               if(rr.gt.li(i,1).and.rr.le.li(i,2)) dist(i)=dist(i)+edisp
            enddo
          enddo !tauz
          enddo !tauy
          enddo !taux
         enddo !jat
      enddo !iat

      write(*,'(/''distance range (Angstroem) analysis'')')
      write(*,'( ''writing histogram data to <histo.dat>'')')
      open(unit=11,file='histo.dat')
      do i=0,nbin
         write(*,'(''R(low,high), Edisp, %tot :'',2f5.1,F12.5,F8.2)')
     .   li(i,1),li(i,2),dist(i),100.*dist(i)/etot
         write(11,*)(li(i,1)+li(i,2))*0.5,dist(i)
      enddo
      close(11)

      write(*,*) 'checksum (Edisp) ',check
      if(abs(check-etot).gt.1.d-3)stop'something is weired in adisp'

      deallocate(dist,li)
      return








      inquire(file='fragment',exist=ex)
      if(ex) return
      write(*,'(/''fragment based analysis'')')
      write(*,'( ''reading file <fragment> ...'')')
      open(unit=55,file='fragment')
      i=0
      at=0
 111  read(55,'(a)',end=222) atmp
      call readfrag(atmp,iout,j)
      if(j.gt.0)then
         i=i+1
         grpn(i)=j
         do k=1,j
            grplist(k,i)=iout(k)      
            at(grplist(k,i))=at(grplist(k,i))+1
         enddo
      endif
      goto 111
 222  continue
      ngrp=i  
      k=0
      do i=1,n
         if(at(i).gt.1) stop 'something is weird in file <fragment>'
         if(at(i).eq.0)then
            k=k+1
            grplist(k,ngrp+1)=i
         endif
      enddo
      if(k.gt.0) then
         ngrp=ngrp+1
         grpn(ngrp)=k
      endif
c Implemented display of atom ranges instead of whole list of atoms
      write(*,*)'group #        atoms '
      dash=0
      do i=1,ngrp
       write(*,'(i4,3x,i4)',advance='no')i,grplist(1,i)
       do j=2,grpn(i)
        if(grplist(j,i).eq.(grplist(j-1,i)+1)) then
         if(dash.eq.0)then
          write(*,'(A1)',advance='no')'-'
          dash=1
         endif
        else
         if(dash.eq.1)then
          write(*,'(i4)',advance='no') grplist(j-1,i)
          dash=0
         endif
         write(*,'(i4)',advance='no') grplist(j,i)
        endif
       enddo 
       if(dash.eq.1)then
        write(*,'(i4)',advance='no') grplist(j-1,i)
        dash=0
       endif
      write(*,*)''
      enddo

c old display list code
c      write(*,*)'group #        atoms '
c      do i=1,ngrp      
c         write(*,'(i4,3x,100i3)')i,(grplist(j,i),j=1,grpn(i))
c      enddo

      eg=0
      iii=0
      do i=1,ngrp
         ni=grpn(i)
         iii=iii+1
         jjj=0
         do j=1,ngrp
            nj=grpn(j)
            jjj=jjj+1
            do ii=1,ni
               iiii=grplist(ii,i)
               do jj=1,nj
                  jjjj=grplist(jj,j)
                  if(jjjj.lt.iiii)cycle
                  eg(lin(iii,jjj))=eg(lin(iii,jjj))+ed(iiii,jjjj)
               enddo
            enddo
         enddo
      enddo

c     call prmat(6,eg,ngrp,0,'intra- + inter-group dispersion energies')
      write(*,*)' group i      j     Edisp'
      k=0
      check=0
      do i=1,ngrp
      do j=1,i    
      k=k+1
      check=check+eg(k) 
      write(*,'(5x,i4,'' --'',i4,F8.2)')i,j,eg(k)
      enddo
      enddo
      write(*,*) 'checksum (Edisp) ',check

      deallocate(dist,li)
      end subroutine pbcadisp
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      SUBROUTINE SET_CRITERIA(rthr,lat,tau_max)

        REAL*8 :: r_cutoff,rthr
        REAL*8 :: lat(3,3)
        REAL*8 :: tau_max(3)
        REAL*8 :: norm1(3),norm2(3),norm3(3)
        REAL*8 :: cos10,cos21,cos32
        real*8,external :: vectorsize

        r_cutoff=sqrt(rthr)
c          write(*,*) 'lat',lat
          !c find normal to the plane...
        call kreuzprodukt(lat(:,2),lat(:,3),norm1)
        call kreuzprodukt(lat(:,3),lat(:,1),norm2)
        call kreuzprodukt(lat(:,1),lat(:,2),norm3)
c        write(*,*) 'norm2',norm2
        !c ...normalize it...
        norm1=norm1/VECTORSIZE(norm1)
        norm2=norm2/VECTORSIZE(norm2)
        norm3=norm3/VECTORSIZE(norm3)
c        write(*,*) 'norm2_',norm2
          !c cos angles between normals and lattice vectors
        cos10=SUM(norm1*lat(:,1))
        cos21=SUM(norm2*lat(:,2))
        cos32=SUM(norm3*lat(:,3))
          !write(*,*) 'cos32',cos32
          !tau_max(1)=abs(2*r_cutoff/cos10)
          !tau_max(2)=abs(2*r_cutoff/cos21)
          !tau_max(3)=abs(2*r_cutoff/cos32)
          !write(*,*) 'r_cutoff',r_cutoff
        tau_max(1)=abs(r_cutoff/cos10)
        tau_max(2)=abs(r_cutoff/cos21)
        tau_max(3)=abs(r_cutoff/cos32)
c        write(*,'(3f8.4)')tau_max(1),tau_max(2),tau_max(3)
      END SUBROUTINE SET_CRITERIA


      SUBROUTINE kreuzprodukt(A,B,C)
        IMPLICIT NONE
  
        REAL*8 :: A(3),B(3)
        REAL*8 :: X,Y,Z
        REAL*8 :: C(3)
        
        X=A(2)*B(3)-B(2)*A(3)
        Y=A(3)*B(1)-B(3)*A(1)
        Z=A(1)*B(2)-B(1)*A(2)
        C=(/X,Y,Z/)
      END SUBROUTINE kreuzprodukt

       FUNCTION VECTORSIZE(VECT)

         REAL*8 :: VECT(3)
         REAL*8 :: SVECT(3)
         REAL*8 :: VECTORSIZE

         SVECT=VECT*VECT
         VECTORSIZE=SUM(SVECT)
         VECTORSIZE=VECTORSIZE**(0.5)
       END FUNCTION VECTORSIZE

      SUBROUTINE stresstensor(maxc,max_elem,autoev,s6,s18,xyz,n,iz,
     .     latice,c6ab,mxc,version,num_ex,noabc,echo,r0ab,r2r4,
     .     rcov,rs6,rs8,rs10,alp6,alp8,alp10,rthr,rep_vdw,cn_thr,rep_cn,
     .     stress)
      IMPLICIT NONE
!!! all values in au !!!       
      INTEGER      :: maxc,max_elem
      real*8,intent(in)   :: autoev
      REAL*8,intent(inout):: xyz(3,*) !cartesian coordinates of atoms
      INTEGER,intent(in)  :: n !number of atoms
      REAL*8,intent(in)   :: latice(3,3) !unitcell vectors in cartesian
      INTEGER,intent(in) :: version !2=D2,3=D3-zero, 4=D3-BJ
      real*8, intent(out) :: stress(3,3) !stresstensor
      logical,intent(in)  :: num_ex !true, if numerical grad-routine
      logical,INTENT(in)  :: noabc !true if no threebodycontirbution
      logical,INTENT(in)  :: echo !true if printout on screen is desired
      real*8,INTENT(in)   :: c6ab(max_elem,max_elem,maxc,maxc,3)! C6 for all element pairs 
      integer,INTENT(in)  :: mxc(max_elem)! how many different C6 for one element
      real*8,intent(in)   :: r0ab(max_elem,max_elem)! cut-off radii for all element pairs
      integer,INTENT(in)  :: iz(*) !cardinal number of atoms
      real*8,intent(in)   :: s6,s18 !scalingparameters
      real*8              :: r2r4(max_elem),rcov(max_elem)
      real*8              :: rs6,rs8,rs10,alp6,alp8,alp10 
      real*8              :: rthr,cn_thr
      integer             :: rep_vdw(3),rep_cn(3)

      
      INTEGER        ::i,j,k,iat,my,ny,a,b
      REAL*8,DIMENSION(3,10000)::abc
      REAL*8                ::e6,e8,e10,e12,e6abc,disp
      real*8                ::step,dum1,dum2
      real*8,dimension(3,3) ::lat,lat_1,lat_1t
      real*8,dimension(3,3) :: num_stresstensor,sigma
      real*8                ::grad(3,n),gradnorm
      logical               :: echo2
      real*8,external       :: volume
      logical             :: num !true, if numerical grad-routine
      real*8               :: time1,time2
      
      
      echo2=echo
      lat=latice
      call inv_cell(lat,lat_1)
      lat_1t=transpose(lat_1)
      
      
      sigma=0.0d0
      num=.true.

      call cpu_time(time1)
      call xyz_to_abc(xyz,abc,lat,n)
      IF (num) THEN
      IF (echo2) write(*,*)'Doing numerical stresstensor...'
      step=2.d-5
      do i=1,3
        do j=1,3
          lat(j,i)=lat(j,i)+step
          call abc_to_xyz(abc,xyz,lat,n)
          !call edisp...dum1
       call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_vdw,cn_thr,rep_cn)
         dum1=-s6*e6-s18*e8-s6*e6abc


          lat(j,i)=lat(j,i)-2*step
          call abc_to_xyz(abc,xyz,lat,n)
          !call edisp...dum2
        call pbcedisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
     .           rcov,rs6,rs8,rs10,alp6,alp8,alp10,version,noabc,
     .           e6,e8,e10,e12,e6abc,lat,rthr,rep_vdw,cn_thr,rep_cn)
          dum2=-s6*e6-s18*e8-s6*e6abc
          num_stresstensor(j,i)=(dum1-dum2)/(step*2.0)

          lat(j,i)=lat(j,i)+step
          call abc_to_xyz(abc,xyz,lat,n)
          
        enddo !j
      enddo !i
      stress=num_stresstensor
      
      
      ELSE !NOT NUM analytic gradient
      IF (echo2) write(*,*)'Doing analytical stresstensor...'
      
!      write(*,*)'Analytical gradient for cellstress Not jet done!'
!      return
      
!      call pbcgdisp(max_elem,maxc,n,xyz,iz,c6ab,mxc,r2r4,r0ab,
!     .            rcov,s6,s18,rs6,rs8,rs10,alp6,alp8,alp10,noabc,num,
!     .                 version,g,disp,gnorm,sigma,lat,rep_v,rep_cn,
!     .                 crit_vdw,echo,crit_cn)


      sigma=sigma!/(volume(lat))

      do a=1,3
        do b=1,3
          do my=1,3
            stress(a,b)=stress(a,b)+sigma(a,my)*lat_1(b,my)
          enddo          
        enddo !b
      enddo !a
      stress=-stress!*volume(lat)

      ENDIF !num or anal
      call cpu_time(time2)
    

      IF (echo2) THEN 
        write(*,*)'Done!'
        write(*,'('' time  '',f4.1)')time2-time1
!        write(*,*)'Virialstress(eV):'
!        write(*,'(3F12.8)')sigma*autoev
!        write(*,*)''
        write(*,*)'Stress(a.u.):'
        write(*,'(3F12.8)')stress
      ENDIF !echo2
      
      END SUBROUTINE stresstensor
      
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      subroutine inv_cell(x,a) !x is normal lat, a is lat^(-1)
      IMPLICIT NONE
      real*8, intent(in)   :: x(3,3) !unitcell vectors in direct space
      real*8, intent(out)  :: a(3,3) !unitcell vectors in reciprocal space
      integer i
      real*8 det
      
      a=0.0
      det=x(1,1)*x(2,2)*x(3,3)+x(1,2)*x(2,3)*x(3,1)+x(1,3)*x(2,1)*
     .    x(3,2)-x(1,3)*x(2,2)*x(3,1)-x(1,2)*x(2,1)*x(3,3)-x(1,1)*
     .    x(2,3)*x(3,2)
!      write(*,*)'Det:',det
      a(1,1)=x(2,2)*x(3,3)-x(2,3)*x(3,2)
      a(2,1)=x(2,3)*x(3,1)-x(2,1)*x(3,3)
      a(3,1)=x(2,1)*x(3,2)-x(2,2)*x(3,1)
      a(1,2)=x(1,3)*x(3,2)-x(1,2)*x(3,3)
      a(2,2)=x(1,1)*x(3,3)-x(1,3)*x(3,1)
      a(3,2)=x(1,2)*x(3,1)-x(1,1)*x(3,2)
      a(1,3)=x(1,2)*x(2,3)-x(1,3)*x(2,2)
      a(2,3)=x(1,3)*x(2,1)-x(1,1)*x(2,3)
      a(3,3)=x(1,1)*x(2,2)-x(1,2)*x(2,1)
      a=a/det
      end subroutine inv_cell

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      SUBROUTINE xyz_to_abc(xyz,abc,lat,n)
      IMPLICIT NONE
      real*8, INTENT(in) :: xyz(3,*)
      real*8, intent(in) :: lat(3,3)
      real*8, intent(out) :: abc(3,*)
      integer,intent(in) :: n
      
      real*8 lat_1(3,3)
      integer i,j,k

      call inv_cell(lat,lat_1)
      
      abc(:,:n)=0.0d0
      do i=1,n
        do j=1,3
          do k=1,3
            abc(j,i)=abc(j,i)+lat_1(j,k)*xyz(k,i)            
          enddo !k
          abc(j,i)=dmod(abc(j,i),1.0d0)
        enddo !j
      enddo !i
  
      END SUBROUTINE xyz_to_abc

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      SUBROUTINE abc_to_xyz(abc,xyz,lat,n)
      IMPLICIT NONE
      real*8, INTENT(in) :: abc(3,*)
      real*8, intent(in) :: lat(3,3)
      real*8, intent(out) :: xyz(3,*)
      integer,intent(in) :: n
      
      integer i,j,k

      xyz(:,:n)=0.0d0
      do i=1,n
        do j=1,3
          do k=1,3
            xyz(j,i)=xyz(j,i)+lat(j,k)*abc(k,i)            
          enddo !k
        enddo !j
      enddo !i
  
      END SUBROUTINE abc_to_xyz

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      REAL*8 FUNCTION volume(lat)
        IMPLICIT NONE
        REAL*8, INTENT(in) ::lat(3,3)
        REAL*8 zwerg
        
        zwerg=lat(1,1)*lat(2,2)*lat(3,3)+lat(1,2)*lat(2,3)*lat(3,1)+
     .       lat(1,3)*lat(2,1)*lat(3,2)-lat(1,3)*lat(2,2)*lat(3,1)-
     .       lat(1,2)*lat(2,1)*lat(3,3)-lat(1,1)*lat(2,3)*lat(3,2)
        volume=abs(zwerg)
      END FUNCTION volume



ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c       
c            string pars procedures
c
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

c      subroutine parse(str,delims,args,nargs)
c
c! Parses the string 'str' into arguments args(1), ..., args(nargs) based on
c! the delimiters contained in the string 'delims'. Preceding a delimiter in
c! 'str' by a backslash (\) makes this particular instance not a delimiter.
c! The integer output variable nargs contains the number of arguments found.
c      interface 
c        subroutine split(str,delims,before,sep)
c         character(len=*),intent(inout) :: str,before
c         character(len=*),intent(in) :: delims
c         character,optional,intent(inout) :: sep
c       end subroutine split
c      end interface
c
c      character(len=*),intent(inout) :: str
c      character(len=*),intent(in) :: delims
c      character(len=len_trim(str)) :: strsav
c      character(len=*),dimension(:),intent(inout) :: args
c      integer, intent(out) :: nargs
c      
c      strsav=str
c      call compact(str)
c      na=size(args)
c      do i=1,na
c        args(i)=' '
c      end do  
c      nargs=0
c      lenstr=len_trim(str)
c      if(lenstr==0) return
c      k=0
c
c      do
c         if(len_trim(str) == 0) exit
c         nargs=nargs+1
c         call split(str,delims,args(nargs))
c         call removebksl(args(nargs))
c      end do   
c      str=strsav
c
c      end subroutine parse

!**********************************************************************

      subroutine compact(str)

! Converts multiple spaces and tabs to single spaces; deletes control characters;
! removes initial spaces.

      character(len=*):: str
      character(len=1):: ch
      character(len=len_trim(str)):: outstr
      
      str=adjustl(str)
      lenstr=len_trim(str)
      outstr=' '
      isp=0
      k=0

      do i=1,lenstr
        ch=str(i:i)
        ich=iachar(ch)
  
        select case(ich)
  
          case(9,32)     ! space or tab character
            if(isp==0) then
              k=k+1
              outstr(k:k)=' '
            end if
            isp=1
            
          case(33:)      ! not a space, quote, or control character
            k=k+1
            outstr(k:k)=ch
            isp=0
      
        end select
        
      end do

      str=adjustl(outstr)

      end subroutine compact

!**********************************************************************

      subroutine removesp(str)

      ! Removes spaces, tabs, and control characters in string str

      character(len=*):: str
      character(len=1):: ch
      character(len=len_trim(str))::outstr

      str=adjustl(str)
      lenstr=len_trim(str)
      outstr=' '
      k=0

      do i=1,lenstr
        ch=str(i:i)
        ich=iachar(ch)
        select case(ich)    
          case(0:32)  ! space, tab, or control character
               cycle       
          case(33:)  
            k=k+1
            outstr(k:k)=ch
        end select
      end do
      
      str=adjustl(outstr)
      
      end subroutine removesp



      subroutine split(str,delims,before,sep)

! Routine finds the first instance of a character from 'delims' in the
! the string 'str'. The characters before the found delimiter are
! output in 'before'. The characters after the found delimiter are
! output in 'str'. The optional output character 'sep' contains the 
! found delimiter. A delimiter in 'str' is treated like an ordinary 
! character if it is preceded by a backslash (\). If the backslash 
! character is desired in 'str', then precede it with another backslash.

      character(len=*),intent(inout) :: str,before
      character(len=*),intent(in) :: delims
      character,optional :: sep
      logical :: pres
      character :: ch,cha

      pres=present(sep)
      str=adjustl(str)
      call compact(str)
      lenstr=len_trim(str)
      if(lenstr == 0) return        ! string str is empty
      k=0
      ibsl=0                        ! backslash initially inactive
      before=' '
      do i=1,lenstr
         ch=str(i:i)
         if(ibsl == 1) then          ! backslash active
            k=k+1
            before(k:k)=ch
            ibsl=0
            cycle
         end if
         if(ch == '\') then          ! backslash with backslash inactive
            k=k+1
            before(k:k)=ch
            ibsl=1
            cycle
         end if
         ipos=index(delims,ch)         
         if(ipos == 0) then          ! character is not a delimiter
            k=k+1
            before(k:k)=ch
            cycle
         end if
         if(ch /= ' ') then          ! character is a delimiter that is not a space
            str=str(i+1:)
            if(pres) sep=ch
            exit
         end if
         cha=str(i+1:i+1)            ! character is a space delimiter
         iposa=index(delims,cha)
         if(iposa > 0) then          ! next character is a delimiter
            str=str(i+2:)
            if(pres) sep=cha
            exit
         else
            str=str(i+1:)
            if(pres) sep=ch
            exit
         end if
      end do
      if(i >= lenstr) str=''
      str=adjustl(str)              ! remove initial spaces
      return

      end subroutine split

!**********************************************************************

      subroutine removebksl(str)

! Removes backslash (\) characters. Double backslashes (\\) are replaced
! by a single backslash.

      character(len=*):: str
      character(len=1):: ch
      character(len=len_trim(str))::outstr

      str=adjustl(str)
      lenstr=len_trim(str)
      outstr=' '
      k=0
      ibsl=0                        ! backslash initially inactive
      
      do i=1,lenstr
        ch=str(i:i)
        if(ibsl == 1) then          ! backslash active
         k=k+1
         outstr(k:k)=ch
         ibsl=0
         cycle
        end if
        if(ch == '\') then          ! backslash with backslash inactive
         ibsl=1
         cycle
        end if
        k=k+1
        outstr(k:k)=ch              ! non-backslash with backslash inactive
      end do
      
      str=adjustl(outstr)
      
      end subroutine removebksl



CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
c copy from machine generated data statements inside pars.f
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      subroutine copyc6(fname,maxc,max_elem,c6ab,maxci,
     .                  minc6,minc6list,maxc6,maxc6list) 
      implicit none
      integer maxc,max_elem,maxci(max_elem),nlines,mima
      real*8  c6ab(max_elem,max_elem,maxc,maxc,3)
      character*(*) fname
      logical minc6,maxc6,minc6list(max_elem),maxc6list(max_elem)

      character*1  atmp 
      character*80 btmp 
      real*8  x,y,f,cn1,cn2,cmax,xx(10)
      integer iat,jat,i,n,l,j,k,il,iadr,jadr,nn,kk
      logical special

      !include 'pars.f'

      real*8 pars( 161925)
      nlines=  32385
      pars(      1:    100)=(/
     . 0.30267000D+01,0.100D+01,0.100D+01,0.91180000D+00,0.91180000D+00
     .,0.20835000D+01,0.200D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.15583000D+01,0.200D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.38944800D+02,0.300D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.22150800D+02,0.300D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.11634454D+04,0.300D+01,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.24441500D+02,0.400D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.14824600D+02,0.400D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.49461900D+03,0.400D+01,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.25748630D+03,0.400D+01,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.17314300D+02,0.500D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.11097500D+02,0.500D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.28373080D+03,0.500D+01,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.16159710D+03,0.500D+01,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.10717770D+03,0.500D+01,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.12140200D+02,0.600D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.81841000D+01,0.600D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.16990300D+03,0.600D+01,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.10295600D+03,0.600D+01,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.71279400D+02,0.600D+01,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    101:    200)=(/
     . 0.49113000D+02,0.600D+01,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.87171000D+01,0.700D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.61380000D+01,0.700D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.10848540D+03,0.700D+01,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.68645800D+02,0.700D+01,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.49113200D+02,0.700D+01,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.34814600D+02,0.700D+01,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.25268500D+02,0.700D+01,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.67180000D+01,0.800D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.48949000D+01,0.800D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.76961300D+02,0.800D+01,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.50125200D+02,0.800D+01,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.36724700D+02,0.800D+01,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.26592900D+02,0.800D+01,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.19654600D+02,0.800D+01,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.15505900D+02,0.800D+01,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.51616000D+01,0.900D+01,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.38825000D+01,0.900D+01,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.55093300D+02,0.900D+01,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.36745300D+02,0.900D+01,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    201:    300)=(/
     . 0.27482100D+02,0.900D+01,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.20282700D+02,0.900D+01,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.15241800D+02,0.900D+01,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.12183400D+02,0.900D+01,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.96916000D+01,0.900D+01,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.40112000D+01,0.100D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.31025000D+01,0.100D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.40473100D+02,0.100D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.27486700D+02,0.100D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.20902200D+02,0.100D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.15674000D+02,0.100D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.11947900D+02,0.100D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.96606000D+01,0.100D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.77691000D+01,0.100D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.62896000D+01,0.100D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.46823200D+02,0.110D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.26862800D+02,0.110D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.13673272D+04,0.110D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.58745630D+03,0.110D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.33872120D+03,0.110D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    301:    400)=(/
     . 0.20376310D+03,0.110D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.13065630D+03,0.110D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.93026300D+02,0.110D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.66842300D+02,0.110D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.49279900D+02,0.110D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.16080286D+04,0.110D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.38353100D+02,0.120D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.23032000D+02,0.120D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.83081560D+03,0.120D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.41821640D+03,0.120D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.25813030D+03,0.120D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.16260820D+03,0.120D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10761500D+03,0.120D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.78225000D+02,0.120D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.57160500D+02,0.120D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.42677100D+02,0.120D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.98516970D+03,0.120D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.68337580D+03,0.120D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.36290900D+02,0.130D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.22322400D+02,0.130D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    401:    500)=(/
     . 0.70582540D+03,0.130D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.37263020D+03,0.130D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.23647800D+03,0.130D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.15209340D+03,0.130D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10220000D+03,0.130D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.75075500D+02,0.130D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.55341200D+02,0.130D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.41596600D+02,0.130D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.83896480D+03,0.130D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.60346890D+03,0.130D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.54054060D+03,0.130D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.29594700D+02,0.140D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.18850000D+02,0.140D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.49534490D+03,0.140D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.27978630D+03,0.140D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.18451110D+03,0.140D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.12213870D+03,0.140D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.83849800D+02,0.140D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.62534900D+02,0.140D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.46693600D+02,0.140D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    501:    600)=(/
     . 0.35455000D+02,0.140D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.59104580D+03,0.140D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.44764230D+03,0.140D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.40896060D+03,0.140D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.31785740D+03,0.140D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.23760400D+02,0.150D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.15668900D+02,0.150D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.35080300D+03,0.150D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.20873310D+03,0.150D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.14234810D+03,0.150D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.96750300D+02,0.150D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.67787300D+02,0.150D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.51309800D+02,0.150D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.38808400D+02,0.150D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.29776700D+02,0.150D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.42000640D+03,0.150D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.33078010D+03,0.150D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.30729650D+03,0.150D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.24435460D+03,0.150D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.19168870D+03,0.150D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(    601:    700)=(/
     . 0.20094800D+02,0.160D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.13610800D+02,0.160D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.27378670D+03,0.160D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.16795130D+03,0.160D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.11711210D+03,0.160D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.81091900D+02,0.160D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.57673400D+02,0.160D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.44147000D+02,0.160D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.33726400D+02,0.160D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.26094000D+02,0.160D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.32859900D+03,0.160D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.26466650D+03,0.160D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.24850080D+03,0.160D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.20053740D+03,0.160D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.15948980D+03,0.160D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.13400660D+03,0.160D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.16705200D+02,0.170D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.11630200D+02,0.170D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.21066260D+03,0.170D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.13298080D+03,0.170D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    701:    800)=(/
     . 0.94761200D+02,0.170D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.66840700D+02,0.170D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.48262400D+02,0.170D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.37368800D+02,0.170D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.28844500D+02,0.170D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.22512100D+02,0.170D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.25351360D+03,0.170D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.20849780D+03,0.170D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.19775940D+03,0.170D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.16186860D+03,0.170D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.13047250D+03,0.170D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.11070060D+03,0.170D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.92346000D+02,0.170D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.13870000D+02,0.180D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.99130000D+01,0.180D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.16354970D+03,0.180D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.10572290D+03,0.180D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.76794900D+02,0.180D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.55089800D+02,0.180D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.40343500D+02,0.180D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    801:    900)=(/
     . 0.31578300D+02,0.180D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.24617800D+02,0.180D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.19377400D+02,0.180D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.19734400D+03,0.180D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.16510060D+03,0.180D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.15795950D+03,0.180D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.13089270D+03,0.180D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.10676980D+03,0.180D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.91401400D+02,0.180D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.76938300D+02,0.180D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.64646200D+02,0.180D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.76237600D+02,0.190D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.44041100D+02,0.190D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.23871574D+04,0.190D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.97231970D+03,0.190D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.55419800D+03,0.190D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.33226940D+03,0.190D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.21320010D+03,0.190D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.15214200D+03,0.190D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10964390D+03,0.190D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(    901:   1000)=(/
     . 0.81086100D+02,0.190D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.27986124D+04,0.190D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.16420587D+04,0.190D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.13879361D+04,0.190D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.96762830D+03,0.190D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.68449680D+03,0.190D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.53523840D+03,0.190D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.41315120D+03,0.190D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.32211550D+03,0.190D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.49835009D+04,0.190D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.65818000D+02,0.200D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.39070100D+02,0.200D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.16144719D+04,0.200D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.75790700D+03,0.200D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.45484390D+03,0.200D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.28170350D+03,0.200D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.18454980D+03,0.200D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.13338470D+03,0.200D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.97082400D+02,0.200D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.72307200D+02,0.200D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1001:   1100)=(/
     . 0.19077081D+04,0.200D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.12525932D+04,0.200D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.10888420D+04,0.200D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.79063470D+03,0.200D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.57546270D+03,0.200D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.45684430D+03,0.200D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.35747370D+03,0.200D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.28168230D+03,0.200D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.32404393D+04,0.200D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.23526862D+04,0.200D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.54966900D+02,0.210D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.32899500D+02,0.210D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.12781183D+04,0.210D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.61733580D+03,0.210D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.37504180D+03,0.210D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.23419500D+03,0.210D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.15429470D+03,0.210D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.11193590D+03,0.210D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.81724200D+02,0.210D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.61015400D+02,0.210D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1101:   1200)=(/
     . 0.15125338D+04,0.210D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.10155454D+04,0.210D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.88860250D+03,0.210D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.65120400D+03,0.210D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.47727970D+03,0.210D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.38041070D+03,0.210D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.29876860D+03,0.210D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.23614170D+03,0.210D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.25499412D+04,0.210D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.18887902D+04,0.210D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.15224676D+04,0.210D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.53687500D+02,0.220D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.32531600D+02,0.220D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.11929128D+04,0.220D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.58748980D+03,0.220D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.36108650D+03,0.220D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.22757230D+03,0.220D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.15101990D+03,0.220D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.11014520D+03,0.220D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.80797500D+02,0.220D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1201:   1300)=(/
     . 0.60559400D+02,0.220D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.14131570D+04,0.220D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.96298310D+03,0.220D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.84746150D+03,0.220D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.62623530D+03,0.220D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.46232470D+03,0.220D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.37026310D+03,0.220D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.29218530D+03,0.220D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.23192080D+03,0.220D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.23746690D+04,0.220D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.17795162D+04,0.220D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.14382841D+04,0.220D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.13619185D+04,0.220D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.49481900D+02,0.230D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.30185100D+02,0.230D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.10690426D+04,0.230D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.53334990D+03,0.230D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.33010230D+03,0.230D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.20914500D+03,0.230D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.13935120D+03,0.230D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1301:   1400)=(/
     . 0.10193250D+03,0.230D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.74967100D+02,0.230D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.56310800D+02,0.230D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.12673139D+04,0.230D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.87225790D+03,0.230D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.77034320D+03,0.230D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.57211290D+03,0.230D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.42414700D+03,0.230D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.34060200D+03,0.230D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.26948630D+03,0.230D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.21440020D+03,0.230D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.21241985D+04,0.230D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.16049178D+04,0.230D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.12995197D+04,0.230D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.12323235D+04,0.230D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.11160984D+04,0.230D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.39122100D+02,0.240D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.24146300D+02,0.240D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.84409360D+03,0.240D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.41816350D+03,0.240D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1401:   1500)=(/
     . 0.25930210D+03,0.240D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.16498820D+03,0.240D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.11048610D+03,0.240D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.81200000D+02,0.240D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.60009900D+02,0.240D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.45282000D+02,0.240D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.10004652D+04,0.240D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.68440150D+03,0.240D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.60440710D+03,0.240D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.44922160D+03,0.240D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.33383800D+03,0.240D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.26880290D+03,0.240D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.21336680D+03,0.240D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.17034650D+03,0.240D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.16836701D+04,0.240D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.12630809D+04,0.240D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.10216871D+04,0.240D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.96885650D+03,0.240D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.87738550D+03,0.240D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.69074250D+03,0.240D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1501:   1600)=(/
     . 0.43002800D+02,0.250D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.26497800D+02,0.250D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.89197560D+03,0.250D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.45336860D+03,0.250D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.28347440D+03,0.250D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.18100350D+03,0.250D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.12132330D+03,0.250D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.89134600D+02,0.250D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.65811200D+02,0.250D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.49595900D+02,0.250D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.10585103D+04,0.250D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.73904110D+03,0.250D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.65607380D+03,0.250D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.49081560D+03,0.250D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.36612040D+03,0.250D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.29517370D+03,0.250D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.23445330D+03,0.250D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.18717280D+03,0.250D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.17680322D+04,0.250D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.13514785D+04,0.250D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1601:   1700)=(/
     . 0.10971539D+04,0.250D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.10426625D+04,0.250D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.94560720D+03,0.250D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.74331790D+03,0.250D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.80274840D+03,0.250D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.33911000D+02,0.260D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.21175000D+02,0.260D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.69845390D+03,0.260D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.35353580D+03,0.260D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.22175470D+03,0.260D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.14234670D+03,0.260D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.95978200D+02,0.260D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.70894200D+02,0.260D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.52631000D+02,0.260D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.39865900D+02,0.260D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.82889360D+03,0.260D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.57651210D+03,0.260D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.51208230D+03,0.260D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.38374450D+03,0.260D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.28716060D+03,0.260D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1701:   1800)=(/
     . 0.23226250D+03,0.260D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.18518230D+03,0.260D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.14843190D+03,0.260D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.13889522D+04,0.260D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.10565193D+04,0.260D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.85717300D+03,0.260D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.81484020D+03,0.260D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.73905420D+03,0.260D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.58181140D+03,0.260D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.62754030D+03,0.260D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.49133490D+03,0.260D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.36323400D+02,0.270D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.22723300D+02,0.270D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.70153240D+03,0.270D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.36919280D+03,0.270D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.23486930D+03,0.270D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.15187030D+03,0.270D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10275030D+03,0.270D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.75991200D+02,0.270D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.56430700D+02,0.270D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1801:   1900)=(/
     . 0.42727000D+02,0.270D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.83418260D+03,0.270D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.59833020D+03,0.270D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.53597060D+03,0.270D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.40599070D+03,0.270D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.30592560D+03,0.270D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.24821160D+03,0.270D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.19835700D+03,0.270D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.15919570D+03,0.270D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.13829771D+04,0.270D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.10817690D+04,0.270D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.88245030D+03,0.270D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.84183350D+03,0.270D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.76531780D+03,0.270D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.60141780D+03,0.270D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.65196880D+03,0.270D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.50977640D+03,0.270D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.53277940D+03,0.270D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.37159600D+02,0.280D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.23079200D+02,0.280D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   1901:   2000)=(/
     . 0.73885800D+03,0.280D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.38391400D+03,0.280D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.24241740D+03,0.280D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.15585070D+03,0.280D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10498340D+03,0.280D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.77401000D+02,0.280D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.57324800D+02,0.280D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.43312900D+02,0.280D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.87793540D+03,0.280D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.62363050D+03,0.280D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.55653890D+03,0.280D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.41935930D+03,0.280D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.31456980D+03,0.280D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.25446830D+03,0.280D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.20277050D+03,0.280D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.16232640D+03,0.280D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.14584501D+04,0.280D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.11321074D+04,0.280D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.92189900D+03,0.280D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.87809390D+03,0.280D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2001:   2100)=(/
     . 0.79751950D+03,0.280D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.62664720D+03,0.280D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.67845310D+03,0.280D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.53032210D+03,0.280D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.55308340D+03,0.280D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.57474360D+03,0.280D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.28594000D+02,0.290D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.18021600D+02,0.290D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.56945260D+03,0.290D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.29279550D+03,0.290D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.18517410D+03,0.290D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.11963500D+03,0.290D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.81085500D+02,0.290D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.60133500D+02,0.290D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.44809800D+02,0.290D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.34054400D+02,0.290D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.67647730D+03,0.290D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.47624260D+03,0.290D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.42477230D+03,0.290D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.32019130D+03,0.290D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2101:   2200)=(/
     . 0.24079740D+03,0.290D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.19540750D+03,0.290D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.15631610D+03,0.290D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.12567760D+03,0.290D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.11299518D+04,0.290D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.86836740D+03,0.290D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.70607200D+03,0.290D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.67241280D+03,0.290D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.61057000D+03,0.290D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.48069380D+03,0.290D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.51930780D+03,0.290D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.40672680D+03,0.290D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.42308770D+03,0.290D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.43963650D+03,0.290D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.33718080D+03,0.290D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.29868900D+02,0.300D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.18969500D+02,0.300D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.54091600D+03,0.300D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.29331060D+03,0.300D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.18962930D+03,0.300D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2201:   2300)=(/
     . 0.12410320D+03,0.300D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.84731100D+02,0.300D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.63077400D+02,0.300D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.47112200D+02,0.300D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.35842700D+02,0.300D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.64434080D+03,0.300D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.47291430D+03,0.300D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.42716100D+03,0.300D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.32728100D+03,0.300D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.24897430D+03,0.300D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.20324270D+03,0.300D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.16338480D+03,0.300D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.13181190D+03,0.300D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.10625928D+04,0.300D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.84685270D+03,0.300D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.69367870D+03,0.300D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.66410090D+03,0.300D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.60506230D+03,0.300D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.47555940D+03,0.300D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.51709100D+03,0.300D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2301:   2400)=(/
     . 0.40455580D+03,0.300D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.42487900D+03,0.300D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.44009000D+03,0.300D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.33664930D+03,0.300D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.34052130D+03,0.300D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.35169700D+02,0.310D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.22045800D+02,0.310D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.65674410D+03,0.310D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.35111410D+03,0.310D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.22539330D+03,0.310D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.14655440D+03,0.310D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.99440700D+02,0.310D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.73625300D+02,0.310D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.54678200D+02,0.310D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.41368100D+02,0.310D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.78135240D+03,0.310D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.56716380D+03,0.310D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.51058540D+03,0.310D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.38926560D+03,0.310D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.29479250D+03,0.310D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2401:   2500)=(/
     . 0.23981830D+03,0.310D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.19205780D+03,0.310D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.15434230D+03,0.310D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.12928121D+04,0.310D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.10198835D+04,0.310D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.83374240D+03,0.310D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.79686170D+03,0.310D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.72525140D+03,0.310D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.56976450D+03,0.310D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.61883870D+03,0.310D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.48378150D+03,0.310D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.50713160D+03,0.310D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.52581620D+03,0.310D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.40196130D+03,0.310D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.40545650D+03,0.310D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.48375160D+03,0.310D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.31817000D+02,0.320D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.20431800D+02,0.320D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.52792680D+03,0.320D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.29842200D+03,0.320D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2501:   2600)=(/
     . 0.19734480D+03,0.320D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.13109540D+03,0.320D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.90332800D+02,0.320D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.67593200D+02,0.320D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.50641200D+02,0.320D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.38574600D+02,0.320D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.63010410D+03,0.320D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.47737440D+03,0.320D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.43650600D+03,0.320D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.33983280D+03,0.320D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.26182350D+03,0.320D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.21531010D+03,0.320D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.17419690D+03,0.320D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.14120630D+03,0.320D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.10326651D+04,0.320D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.84336440D+03,0.320D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.69470240D+03,0.320D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.66837860D+03,0.320D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.61076390D+03,0.320D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.47996390D+03,0.320D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2601:   2700)=(/
     . 0.52418480D+03,0.320D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.41021060D+03,0.320D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.43385080D+03,0.320D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.44798680D+03,0.320D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.34242610D+03,0.320D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.34998850D+03,0.320D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.41595300D+03,0.320D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.36354740D+03,0.320D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.27788400D+02,0.330D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.18328400D+02,0.330D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.41515320D+03,0.330D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.24548310D+03,0.330D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.16690600D+03,0.330D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.11326160D+03,0.330D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.79308600D+02,0.330D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.60028900D+02,0.330D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.45418000D+02,0.330D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.34868600D+02,0.330D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.49693380D+03,0.330D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.38949270D+03,0.330D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2701:   2800)=(/
     . 0.36119130D+03,0.330D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.28659090D+03,0.330D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.22447740D+03,0.330D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.18663480D+03,0.330D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.15260350D+03,0.330D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.12485610D+03,0.330D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.81065540D+03,0.330D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.67900440D+03,0.330D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.56269470D+03,0.330D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.54467970D+03,0.330D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.49949700D+03,0.330D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.39319220D+03,0.330D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.43091460D+03,0.330D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.33800960D+03,0.330D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.35971500D+03,0.330D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.37003590D+03,0.330D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.28333230D+03,0.330D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.29250300D+03,0.330D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.34635080D+03,0.330D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.30710100D+03,0.330D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2801:   2900)=(/
     . 0.26294980D+03,0.330D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.25309800D+02,0.340D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.17024300D+02,0.340D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.35454290D+03,0.340D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.21502380D+03,0.340D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.14880080D+03,0.340D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.10243460D+03,0.340D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.72540300D+02,0.340D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.55362700D+02,0.340D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.42192300D+02,0.340D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.32584300D+02,0.340D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.42518640D+03,0.340D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.33957420D+03,0.340D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.31761820D+03,0.340D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.25500990D+03,0.340D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.20190190D+03,0.340D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.16913080D+03,0.340D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.13931630D+03,0.340D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.11474490D+03,0.340D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.69291160D+03,0.340D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   2901:   3000)=(/
     . 0.58796470D+03,0.340D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.48887150D+03,0.340D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.47503480D+03,0.340D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.43657180D+03,0.340D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.34429560D+03,0.340D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.37782760D+03,0.340D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.29703960D+03,0.340D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.31701740D+03,0.340D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.32534190D+03,0.340D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.24963850D+03,0.340D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.25904590D+03,0.340D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.30595010D+03,0.340D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.27365300D+03,0.340D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.23634730D+03,0.340D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.21367380D+03,0.340D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.22483400D+02,0.350D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.15455300D+02,0.350D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.29478970D+03,0.350D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.18337420D+03,0.350D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.12924950D+03,0.350D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3001:   3100)=(/
     . 0.90342400D+02,0.350D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.64763200D+02,0.350D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.49879500D+02,0.350D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.38322500D+02,0.350D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.29795800D+02,0.350D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.35428690D+03,0.350D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.28827360D+03,0.350D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.27200980D+03,0.350D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.22105370D+03,0.350D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.17699300D+03,0.350D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.14945630D+03,0.350D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.12409090D+03,0.350D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.10294710D+03,0.350D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.57723180D+03,0.350D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.49597540D+03,0.350D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.41374680D+03,0.350D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.40365760D+03,0.350D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.37180710D+03,0.350D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.29391850D+03,0.350D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.32284330D+03,0.350D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3101:   3200)=(/
     . 0.25453060D+03,0.350D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.27230530D+03,0.350D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.27877050D+03,0.350D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.21450590D+03,0.350D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.22363480D+03,0.350D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.26334580D+03,0.350D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.23762950D+03,0.350D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.20707720D+03,0.350D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.18836050D+03,0.350D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.16712970D+03,0.350D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.19818200D+02,0.360D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.13921100D+02,0.360D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.24455540D+03,0.360D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.15556160D+03,0.360D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.11153710D+03,0.360D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.79108000D+02,0.360D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.57389500D+02,0.360D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.44600300D+02,0.360D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.34545200D+02,0.360D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.27043000D+02,0.360D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3201:   3300)=(/
     . 0.29455740D+03,0.360D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.24359650D+03,0.360D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.23169040D+03,0.360D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.19039340D+03,0.360D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.15405750D+03,0.360D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.13109280D+03,0.360D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.10968390D+03,0.360D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.91643000D+02,0.360D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.48024720D+03,0.360D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.41697770D+03,0.360D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.34884740D+03,0.360D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.34162730D+03,0.360D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.31532510D+03,0.360D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.24994130D+03,0.360D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.27464190D+03,0.360D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.21720710D+03,0.360D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.23275940D+03,0.360D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.23774470D+03,0.360D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.18353470D+03,0.360D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.19204840D+03,0.360D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3301:   3400)=(/
     . 0.22545370D+03,0.360D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.20505980D+03,0.360D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.18018180D+03,0.360D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.16484940D+03,0.360D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.14718300D+03,0.360D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.13040170D+03,0.360D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.85949900D+02,0.370D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.50049100D+02,0.370D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.26473331D+04,0.370D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.10827052D+04,0.370D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.62008780D+03,0.370D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.37353840D+03,0.370D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.24069400D+03,0.370D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.17233980D+03,0.370D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12458790D+03,0.370D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.92382700D+02,0.370D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.31041016D+04,0.370D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.18264648D+04,0.370D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.15469229D+04,0.370D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.10820563D+04,0.370D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3401:   3500)=(/
     . 0.76813350D+03,0.370D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.60222210D+03,0.370D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.46615650D+03,0.370D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.36440670D+03,0.370D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.55302806D+04,0.370D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.35994349D+04,0.370D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.28340920D+04,0.370D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.26412435D+04,0.370D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.23637175D+04,0.370D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.18740326D+04,0.370D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.19687631D+04,0.370D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.15471982D+04,0.370D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.15418997D+04,0.370D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.16251034D+04,0.370D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.12594427D+04,0.370D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.11862631D+04,0.370D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.14423807D+04,0.370D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.11551526D+04,0.370D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.90942570D+03,0.370D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.77892040D+03,0.370D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3501:   3600)=(/
     . 0.65036060D+03,0.370D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.54231600D+03,0.370D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.61387755D+04,0.370D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.78390100D+02,0.380D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.46665800D+02,0.380D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.19442260D+04,0.380D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.90392650D+03,0.380D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.54157290D+03,0.380D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.33545970D+03,0.380D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.21996080D+03,0.380D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.15914600D+03,0.380D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.11596470D+03,0.380D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.86462700D+02,0.380D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.22961906D+04,0.380D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.14958141D+04,0.380D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.12985356D+04,0.380D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.94141210D+03,0.380D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.68496320D+03,0.380D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.54396270D+03,0.380D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.42589970D+03,0.380D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3601:   3700)=(/
     . 0.33586900D+03,0.380D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.39153529D+04,0.380D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.28194935D+04,0.380D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.22604217D+04,0.380D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.21283951D+04,0.380D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.19186941D+04,0.380D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.15113043D+04,0.380D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.16147190D+04,0.380D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.12632327D+04,0.380D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.12908718D+04,0.380D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.13513702D+04,0.380D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.10377284D+04,0.380D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.10096809D+04,0.380D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.12165138D+04,0.380D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.10044824D+04,0.380D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.80833700D+03,0.380D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.70005320D+03,0.380D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.59073260D+03,0.380D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.49691200D+03,0.380D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.43492577D+04,0.380D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3701:   3800)=(/
     . 0.33813672D+04,0.380D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.70057000D+02,0.390D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.42419500D+02,0.390D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.15853950D+04,0.390D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.77148440D+03,0.390D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.47243640D+03,0.390D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.29721710D+03,0.390D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.19706110D+03,0.390D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.14366020D+03,0.390D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10534470D+03,0.390D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.78931600D+02,0.390D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.18768634D+04,0.390D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.12667602D+04,0.390D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.11122980D+04,0.390D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.81954180D+03,0.390D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.60399160D+03,0.390D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.48337970D+03,0.390D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.38124620D+03,0.390D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.30251430D+03,0.390D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.31660625D+04,0.390D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3801:   3900)=(/
     . 0.23508018D+04,0.390D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.18967809D+04,0.390D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.17943180D+04,0.390D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.16224689D+04,0.390D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.12764352D+04,0.390D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.13714492D+04,0.390D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.10723265D+04,0.390D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11053016D+04,0.390D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.11535737D+04,0.390D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.88416480D+03,0.390D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.87069350D+03,0.390D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.10455505D+04,0.390D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.87478360D+03,0.390D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.71175060D+03,0.390D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.62033340D+03,0.390D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.52682990D+03,0.390D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.44570130D+03,0.390D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.35209466D+04,0.390D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.28137007D+04,0.390D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.23658925D+04,0.390D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   3901:   4000)=(/
     . 0.63780100D+02,0.400D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.39108800D+02,0.400D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.13632842D+04,0.400D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.68152830D+03,0.400D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.42323020D+03,0.400D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.26904900D+03,0.400D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.17977770D+03,0.400D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.13178520D+03,0.400D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.97097700D+02,0.400D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.73031500D+02,0.400D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.16162863D+04,0.400D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.11138339D+04,0.400D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.98504940D+03,0.400D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.73318070D+03,0.400D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.54488820D+03,0.400D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.43838900D+03,0.400D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.34753340D+03,0.400D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.27699080D+03,0.400D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.27111542D+04,0.400D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.20485251D+04,0.400D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4001:   4100)=(/
     . 0.16591156D+04,0.400D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.15741401D+04,0.400D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.14260687D+04,0.400D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.11215489D+04,0.400D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.12087451D+04,0.400D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.94517130D+03,0.400D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.97897670D+03,0.400D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.10197589D+04,0.400D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.78111330D+03,0.400D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.77458710D+03,0.400D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.92819820D+03,0.400D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.78294420D+03,0.400D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.64155640D+03,0.400D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.56154740D+03,0.400D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.47900400D+03,0.400D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.40687270D+03,0.400D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.30176856D+04,0.400D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.24494723D+04,0.400D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.20726170D+04,0.400D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.18227181D+04,0.400D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4101:   4200)=(/
     . 0.58676800D+02,0.410D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.36341800D+02,0.410D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.12090304D+04,0.410D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.61398500D+03,0.410D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.38490720D+03,0.410D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.24652410D+03,0.410D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.16569430D+03,0.410D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.12198680D+03,0.410D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.90223400D+02,0.410D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.68076900D+02,0.410D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.14346641D+04,0.410D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.10005663D+04,0.410D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.88903020D+03,0.410D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.66615070D+03,0.410D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.49797030D+03,0.410D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.40219740D+03,0.410D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.32006760D+03,0.410D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.25597690D+03,0.410D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.24006918D+04,0.410D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.18308523D+04,0.410D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4201:   4300)=(/
     . 0.14860620D+04,0.410D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.14126818D+04,0.410D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.12813349D+04,0.410D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.10078571D+04,0.410D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.10879837D+04,0.410D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.85106340D+03,0.410D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.88388900D+03,0.410D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.91953540D+03,0.410D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.70437890D+03,0.410D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.70135490D+03,0.410D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.83928280D+03,0.410D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.71168230D+03,0.410D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.58600840D+03,0.410D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.51450770D+03,0.410D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.44029990D+03,0.410D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.37513390D+03,0.410D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.26739344D+04,0.410D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.21882864D+04,0.410D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.18586377D+04,0.410D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.16384938D+04,0.410D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4301:   4400)=(/
     . 0.14752500D+04,0.410D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.46040600D+02,0.420D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.29161700D+02,0.420D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.89051260D+03,0.420D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.46330650D+03,0.420D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.29535920D+03,0.420D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.19195280D+03,0.420D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.13060450D+03,0.420D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.97067500D+02,0.420D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.72418600D+02,0.420D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.55048700D+02,0.420D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.10583899D+04,0.420D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.75166280D+03,0.420D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.67311930D+03,0.420D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.51023520D+03,0.420D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.38558950D+03,0.420D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.31386570D+03,0.420D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.25177230D+03,0.420D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.20285720D+03,0.420D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.17658946D+04,0.420D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4401:   4500)=(/
     . 0.13654837D+04,0.420D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.11120098D+04,0.420D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.10606255D+04,0.420D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.96394340D+03,0.420D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.75899550D+03,0.420D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.82093130D+03,0.420D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.64312470D+03,0.420D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.67034900D+03,0.420D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.69583480D+03,0.420D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.53365350D+03,0.420D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.53452070D+03,0.420D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.63792330D+03,0.420D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.54578450D+03,0.420D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.45341860D+03,0.420D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.40049980D+03,0.420D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.34498260D+03,0.420D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.29579940D+03,0.420D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.19694796D+04,0.420D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.16315424D+04,0.420D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.13940976D+04,0.420D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4501:   4600)=(/
     . 0.12339095D+04,0.420D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.11140924D+04,0.420D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.84589720D+03,0.420D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.51052700D+02,0.430D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.32058800D+02,0.430D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.10116099D+04,0.430D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.52142490D+03,0.430D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.33032570D+03,0.430D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.21347990D+03,0.430D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.14456970D+03,0.430D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.10705880D+03,0.430D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.79612100D+02,0.430D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.60352100D+02,0.430D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.12015234D+04,0.430D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.84734970D+03,0.430D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.75661950D+03,0.430D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.57105010D+03,0.430D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.42976770D+03,0.430D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.34877440D+03,0.430D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.27891230D+03,0.430D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4601:   4700)=(/
     . 0.22407940D+03,0.430D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.20076137D+04,0.430D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.15434258D+04,0.430D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.12553446D+04,0.430D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.11958434D+04,0.430D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.10860231D+04,0.430D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.85474740D+03,0.430D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.92387430D+03,0.430D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.72332020D+03,0.430D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.75296780D+03,0.430D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.78224830D+03,0.430D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.59961920D+03,0.430D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.59931410D+03,0.430D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.71598110D+03,0.430D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.61053270D+03,0.430D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.50550910D+03,0.430D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.44547780D+03,0.430D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.38275910D+03,0.430D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.32738270D+03,0.430D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.22379665D+04,0.430D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4701:   4800)=(/
     . 0.18443774D+04,0.430D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.15723759D+04,0.430D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.13895979D+04,0.430D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.12533465D+04,0.430D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.94964900D+03,0.430D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.10670169D+04,0.430D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.39574500D+02,0.440D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.25462200D+02,0.440D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.73055330D+03,0.440D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.38726810D+03,0.440D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.24993820D+03,0.440D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.16412790D+03,0.440D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.11263780D+03,0.440D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.84272500D+02,0.440D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.63259900D+02,0.440D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.48342900D+02,0.440D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.86937740D+03,0.440D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.62621040D+03,0.440D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.56407160D+03,0.440D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.43121320D+03,0.440D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4801:   4900)=(/
     . 0.32840720D+03,0.440D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.26878240D+03,0.440D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.21680700D+03,0.440D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.17558760D+03,0.440D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.14473048D+04,0.440D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.11311769D+04,0.440D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.92358110D+03,0.440D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.88314920D+03,0.440D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.80388070D+03,0.440D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.63343940D+03,0.440D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.68617720D+03,0.440D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.53814350D+03,0.440D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.56248000D+03,0.440D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.58290140D+03,0.440D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.44742590D+03,0.440D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.45015430D+03,0.440D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.53610930D+03,0.440D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.46167950D+03,0.440D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.38598830D+03,0.440D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.34238140D+03,0.440D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   4901:   5000)=(/
     . 0.29626680D+03,0.440D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.25515160D+03,0.440D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.16158387D+04,0.440D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.13511988D+04,0.440D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.11599071D+04,0.440D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.10297665D+04,0.440D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.93174450D+03,0.440D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.71028370D+03,0.440D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.79619510D+03,0.440D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.59819880D+03,0.440D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.43166100D+02,0.450D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.27574500D+02,0.450D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.79420060D+03,0.450D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.42441200D+03,0.450D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.27371950D+03,0.450D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.17927080D+03,0.450D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.12264260D+03,0.450D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.91495900D+02,0.450D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.68488800D+02,0.450D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.52207200D+02,0.450D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5001:   5100)=(/
     . 0.94542210D+03,0.450D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.68566280D+03,0.450D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.61792440D+03,0.450D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.47237380D+03,0.450D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.35925470D+03,0.450D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.29352140D+03,0.450D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.23627500D+03,0.450D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.19093520D+03,0.450D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.15674954D+04,0.450D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.12346013D+04,0.450D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.10091933D+04,0.450D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.96523080D+03,0.450D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.87880270D+03,0.450D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.69163140D+03,0.450D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.75032510D+03,0.450D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.58775440D+03,0.450D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.61543640D+03,0.450D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.63776340D+03,0.450D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.48875010D+03,0.450D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.49263190D+03,0.450D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5101:   5200)=(/
     . 0.58677320D+03,0.450D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.50544520D+03,0.450D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.42219710D+03,0.450D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.37406320D+03,0.450D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.32319410D+03,0.450D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.27786800D+03,0.450D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.17497615D+04,0.450D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.14735487D+04,0.450D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.12668436D+04,0.450D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.11253403D+04,0.450D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.10182915D+04,0.450D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.77583530D+03,0.450D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.86989620D+03,0.450D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.65317750D+03,0.450D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.71394270D+03,0.450D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.40237200D+02,0.460D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.25871600D+02,0.460D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.72242940D+03,0.460D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.39043100D+03,0.460D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.25334250D+03,0.460D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5201:   5300)=(/
     . 0.16671110D+03,0.460D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.11447630D+03,0.460D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.85642700D+02,0.460D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.64270700D+02,0.460D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.49099000D+02,0.460D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.86062090D+03,0.460D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.62959120D+03,0.460D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.56914100D+03,0.460D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.43694630D+03,0.460D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.33352190D+03,0.460D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.27315680D+03,0.460D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.22040990D+03,0.460D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.17850210D+03,0.460D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.14240271D+04,0.460D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.11296670D+04,0.460D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.92485560D+03,0.460D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.88576570D+03,0.460D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.80712580D+03,0.460D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.63532570D+03,0.460D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.68996470D+03,0.460D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5301:   5400)=(/
     . 0.54065990D+03,0.460D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.56710440D+03,0.460D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.58718190D+03,0.460D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.45005590D+03,0.460D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.45480310D+03,0.460D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.54115010D+03,0.460D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.46770420D+03,0.460D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.39185280D+03,0.460D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.34783930D+03,0.460D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.30113900D+03,0.460D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.25939640D+03,0.460D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.15904415D+04,0.460D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.13479007D+04,0.460D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.11619289D+04,0.460D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.10338735D+04,0.460D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.93655280D+03,0.460D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.71494950D+03,0.460D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.80104420D+03,0.460D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.60279560D+03,0.460D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.65887530D+03,0.460D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5401:   5500)=(/
     . 0.60850410D+03,0.460D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.33541300D+02,0.470D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.21789600D+02,0.470D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.61414770D+03,0.470D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.32557750D+03,0.470D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.21068110D+03,0.470D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.13887580D+03,0.470D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.95713700D+02,0.470D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.71892600D+02,0.470D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.54190300D+02,0.470D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.41578000D+02,0.470D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.73108320D+03,0.470D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.52647760D+03,0.470D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.47457130D+03,0.470D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.36335850D+03,0.470D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.27735770D+03,0.470D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.22750240D+03,0.470D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.18398760D+03,0.470D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.14942690D+03,0.470D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.12184699D+04,0.470D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5501:   5600)=(/
     . 0.95149350D+03,0.470D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.77689140D+03,0.470D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.74320410D+03,0.470D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.67665530D+03,0.470D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.53368780D+03,0.470D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.57781270D+03,0.470D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.45363270D+03,0.470D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.47392440D+03,0.470D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.49097830D+03,0.470D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.37735310D+03,0.470D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.37956550D+03,0.470D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.45158870D+03,0.470D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.38930750D+03,0.470D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.32603710D+03,0.470D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.28964690D+03,0.470D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.25111890D+03,0.470D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.21673360D+03,0.470D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.13607517D+04,0.470D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.11369099D+04,0.470D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.97620250D+03,0.470D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5601:   5700)=(/
     . 0.86700560D+03,0.470D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.78481880D+03,0.470D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.59905300D+03,0.470D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.67117170D+03,0.470D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.50502690D+03,0.470D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.55108020D+03,0.470D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.50877420D+03,0.470D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.42674500D+03,0.470D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.35687900D+02,0.480D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.23156300D+02,0.480D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.62105360D+03,0.480D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.34041220D+03,0.480D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.22262730D+03,0.480D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.14741380D+03,0.480D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10173770D+03,0.480D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.76409200D+02,0.480D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.57550000D+02,0.480D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.44106200D+02,0.480D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.74057870D+03,0.480D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.54767210D+03,0.480D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5701:   5800)=(/
     . 0.49703370D+03,0.480D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.38367900D+03,0.480D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.29424650D+03,0.480D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.24176140D+03,0.480D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.19570440D+03,0.480D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.15896500D+03,0.480D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.12225553D+04,0.480D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.97842510D+03,0.480D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.80260200D+03,0.480D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.77003030D+03,0.480D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.70242130D+03,0.480D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.55309190D+03,0.480D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.60140530D+03,0.480D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.47153290D+03,0.480D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.49563130D+03,0.480D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.51262200D+03,0.480D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.39305100D+03,0.480D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.39845720D+03,0.480D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.47342030D+03,0.480D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.41090580D+03,0.480D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5801:   5900)=(/
     . 0.34560850D+03,0.480D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.30755390D+03,0.480D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.26697000D+03,0.480D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.23054930D+03,0.480D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.13663938D+04,0.480D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.11670471D+04,0.480D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.10094418D+04,0.480D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.90011540D+03,0.480D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.81654950D+03,0.480D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.62495060D+03,0.480D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.69953820D+03,0.480D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.52793870D+03,0.480D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.57701070D+03,0.480D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.53341340D+03,0.480D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.44587010D+03,0.480D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.46819000D+03,0.480D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.44041700D+02,0.490D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.27958600D+02,0.490D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.82896730D+03,0.490D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.43833960D+03,0.490D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   5901:   6000)=(/
     . 0.28117370D+03,0.490D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.18333680D+03,0.490D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.12497980D+03,0.490D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.92996800D+02,0.490D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.69455800D+02,0.490D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.52851700D+02,0.490D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.98609990D+03,0.490D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.70935650D+03,0.490D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.63758320D+03,0.490D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.48553920D+03,0.490D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.36802750D+03,0.490D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.29998530D+03,0.490D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.24091040D+03,0.490D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.19426460D+03,0.490D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.16388153D+04,0.490D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.12812612D+04,0.490D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.10458401D+04,0.490D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.99909710D+03,0.490D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.90897530D+03,0.490D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.71527990D+03,0.490D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6001:   6100)=(/
     . 0.77527360D+03,0.490D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.60709640D+03,0.490D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.63472630D+03,0.490D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.65825260D+03,0.490D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.50438020D+03,0.490D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.50724520D+03,0.490D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.60471830D+03,0.490D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.51935010D+03,0.490D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.43261670D+03,0.490D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.38259680D+03,0.490D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.32992500D+03,0.490D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.28312690D+03,0.490D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.18285696D+04,0.490D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.15296862D+04,0.490D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.13118922D+04,0.490D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.11635963D+04,0.490D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.10518917D+04,0.490D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.79999940D+03,0.490D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.89764510D+03,0.490D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.67265160D+03,0.490D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6101:   6200)=(/
     . 0.73525250D+03,0.490D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.67809960D+03,0.490D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.56733030D+03,0.490D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.59335740D+03,0.490D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.75773970D+03,0.490D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.41533400D+02,0.500D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.26786200D+02,0.500D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.70845970D+03,0.500D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.39379910D+03,0.500D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.25871460D+03,0.500D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.17143670D+03,0.500D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.11815090D+03,0.500D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.88540800D+02,0.500D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.66499600D+02,0.500D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.50811000D+02,0.500D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.84513480D+03,0.500D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.63194340D+03,0.500D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.57540180D+03,0.500D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.44576230D+03,0.500D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.34241690D+03,0.500D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6201:   6300)=(/
     . 0.28132550D+03,0.500D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.22756350D+03,0.500D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.18458450D+03,0.500D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.13902912D+04,0.500D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.11230616D+04,0.500D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.92301820D+03,0.500D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.88663180D+03,0.500D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.80941340D+03,0.500D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.63665610D+03,0.500D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.69374420D+03,0.500D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.54336290D+03,0.500D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.57281690D+03,0.500D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.59204640D+03,0.500D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.45320890D+03,0.500D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.46119710D+03,0.500D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.54805490D+03,0.500D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.47710340D+03,0.500D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.40194150D+03,0.500D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.35778710D+03,0.500D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.31050930D+03,0.500D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6301:   6400)=(/
     . 0.26795320D+03,0.500D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.15543698D+04,0.500D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.13385665D+04,0.500D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.11613674D+04,0.500D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.10372812D+04,0.500D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.94177920D+03,0.500D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.72129060D+03,0.500D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.80723580D+03,0.500D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.60961520D+03,0.500D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.66689540D+03,0.500D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.61675220D+03,0.500D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.51444770D+03,0.500D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.54154900D+03,0.500D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.68561380D+03,0.500D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.62756770D+03,0.500D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.37768100D+02,0.510D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.24858300D+02,0.510D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.58668880D+03,0.510D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.34030300D+03,0.510D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.22904210D+03,0.510D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6401:   6500)=(/
     . 0.15448510D+03,0.510D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10784360D+03,0.510D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.81535800D+02,0.510D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.61691200D+02,0.510D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.47407300D+02,0.510D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.70169030D+03,0.510D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.54198260D+03,0.510D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.49972920D+03,0.510D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.39368990D+03,0.510D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.30669120D+03,0.510D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.25424180D+03,0.510D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.20740430D+03,0.510D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.16945330D+03,0.510D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.11476700D+04,0.510D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.95065430D+03,0.510D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.78583230D+03,0.510D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.75894060D+03,0.510D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.69506830D+03,0.510D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.54719810D+03,0.510D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.59851680D+03,0.510D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6501:   6600)=(/
     . 0.46945690D+03,0.510D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.49804720D+03,0.510D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.51305850D+03,0.510D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.39302490D+03,0.510D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.40386210D+03,0.510D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.47839930D+03,0.510D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.42186160D+03,0.510D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.35955640D+03,0.510D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.32236080D+03,0.510D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.28180500D+03,0.510D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.24479020D+03,0.510D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.12861750D+04,0.510D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.11321713D+04,0.510D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.99240010D+03,0.510D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.89208770D+03,0.510D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.81343450D+03,0.510D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.62768490D+03,0.510D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.70051630D+03,0.510D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.53338040D+03,0.510D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.58333460D+03,0.510D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6601:   6700)=(/
     . 0.54091250D+03,0.510D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.45064270D+03,0.510D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.47657830D+03,0.510D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.59828910D+03,0.510D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.55335920D+03,0.510D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.49293790D+03,0.510D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.35484100D+02,0.520D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.23691100D+02,0.520D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.52125110D+03,0.520D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.30954390D+03,0.520D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.21148980D+03,0.520D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.14430320D+03,0.520D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10161240D+03,0.520D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.77297000D+02,0.520D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.58788100D+02,0.520D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.45361300D+02,0.520D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.62440760D+03,0.520D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.49087600D+03,0.520D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.45604000D+03,0.520D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.36295180D+03,0.520D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6701:   6800)=(/
     . 0.28527790D+03,0.520D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.23790180D+03,0.520D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.19518610D+03,0.520D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.16026740D+03,0.520D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.10193121D+04,0.520D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.85518820D+03,0.520D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.70913030D+03,0.520D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.68712350D+03,0.520D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.63049010D+03,0.520D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.49691430D+03,0.520D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.54441560D+03,0.520D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.42764390D+03,0.520D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.45508360D+03,0.520D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.46784930D+03,0.520D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.35881670D+03,0.520D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.37059360D+03,0.520D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.43811020D+03,0.520D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.38929970D+03,0.520D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.33422340D+03,0.520D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.30105450D+03,0.520D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6801:   6900)=(/
     . 0.26445160D+03,0.520D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.23074260D+03,0.520D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.11441913D+04,0.520D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.10183413D+04,0.520D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.89780430D+03,0.520D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.81010280D+03,0.520D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.74062110D+03,0.520D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.57429320D+03,0.520D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.63974650D+03,0.520D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.48969560D+03,0.520D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.53521890D+03,0.520D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.49709590D+03,0.520D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.41415120D+03,0.520D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.43888610D+03,0.520D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.54812670D+03,0.520D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.50996980D+03,0.520D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.45710570D+03,0.520D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.42553550D+03,0.520D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.32517100D+02,0.530D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.22081300D+02,0.530D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   6901:   7000)=(/
     . 0.44985120D+03,0.530D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.27373980D+03,0.530D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.19010940D+03,0.530D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.13140510D+03,0.530D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.93456200D+02,0.530D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.71605900D+02,0.530D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.54799000D+02,0.530D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.42495400D+02,0.530D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.53985360D+03,0.530D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.43217920D+03,0.530D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.40476310D+03,0.530D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.32569120D+03,0.530D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.25851210D+03,0.530D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.21703370D+03,0.530D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.17923060D+03,0.530D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.14802160D+03,0.530D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.88017410D+03,0.530D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.74793060D+03,0.530D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.62218410D+03,0.530D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.60505080D+03,0.530D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7001:   7100)=(/
     . 0.55631660D+03,0.530D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.43916600D+03,0.530D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.48180770D+03,0.530D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.37922120D+03,0.530D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.40469290D+03,0.530D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.41513030D+03,0.530D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.31896420D+03,0.530D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.33106510D+03,0.530D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.39045420D+03,0.530D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.34977600D+03,0.530D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.30267430D+03,0.530D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.27406300D+03,0.530D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.24205560D+03,0.530D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.21228200D+03,0.530D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.98989210D+03,0.530D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.89067230D+03,0.530D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.79002570D+03,0.530D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.71573180D+03,0.530D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.65622760D+03,0.530D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.51167720D+03,0.530D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7101:   7200)=(/
     . 0.56879090D+03,0.530D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.43799920D+03,0.530D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.47824810D+03,0.530D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.44496620D+03,0.530D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.37093440D+03,0.530D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.39376550D+03,0.530D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.48897090D+03,0.530D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.45771560D+03,0.530D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.41298520D+03,0.530D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.38609970D+03,0.530D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.35196670D+03,0.530D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.29605500D+02,0.540D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.20459400D+02,0.540D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.38699690D+03,0.540D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.24076890D+03,0.540D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.16985780D+03,0.540D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.11891780D+03,0.540D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.85429300D+02,0.540D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.65940300D+02,0.540D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.50790000D+02,0.540D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7201:   7300)=(/
     . 0.39595100D+02,0.540D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.46528170D+03,0.540D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.37861340D+03,0.540D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.35729600D+03,0.540D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.29049850D+03,0.540D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.23277860D+03,0.540D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.19673740D+03,0.540D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.16353680D+03,0.540D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.13585920D+03,0.540D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.75832530D+03,0.540D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.65153920D+03,0.540D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.54356310D+03,0.540D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.53043120D+03,0.540D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.48865220D+03,0.540D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.38650670D+03,0.540D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.42441250D+03,0.540D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.33482590D+03,0.540D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.35809540D+03,0.540D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.36655970D+03,0.540D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.28228880D+03,0.540D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7301:   7400)=(/
     . 0.29421380D+03,0.540D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.34613450D+03,0.540D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.31242650D+03,0.540D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.27241100D+03,0.540D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.24792750D+03,0.540D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.22015810D+03,0.540D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.19406990D+03,0.540D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.85452230D+03,0.540D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.77608590D+03,0.540D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.69223860D+03,0.540D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.62952300D+03,0.540D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.57878760D+03,0.540D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.45379980D+03,0.540D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.50338110D+03,0.540D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.38995320D+03,0.540D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.42526840D+03,0.540D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.39634680D+03,0.540D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.33076460D+03,0.540D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.35152850D+03,0.540D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.43409320D+03,0.540D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7401:   7500)=(/
     . 0.40859520D+03,0.540D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.37095720D+03,0.540D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.34822390D+03,0.540D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.31889050D+03,0.540D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.29022230D+03,0.540D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.10500960D+03,0.550D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.61318400D+02,0.550D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.32422404D+04,0.550D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.13162189D+04,0.550D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.75492430D+03,0.550D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.45569600D+03,0.550D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.29414830D+03,0.550D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.21086720D+03,0.550D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.15256720D+03,0.550D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.11317640D+03,0.550D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.37989439D+04,0.550D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.22209652D+04,0.550D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.18817164D+04,0.550D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.13170056D+04,0.550D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.93630690D+03,0.550D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7501:   7600)=(/
     . 0.73500610D+03,0.550D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.56967010D+03,0.550D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.44581710D+03,0.550D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.68083900D+04,0.550D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.43871669D+04,0.550D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.34508369D+04,0.550D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.32158544D+04,0.550D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.28775279D+04,0.550D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.22830674D+04,0.550D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.23963445D+04,0.550D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.18842824D+04,0.550D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.18757209D+04,0.550D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.19768609D+04,0.550D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.15334063D+04,0.550D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.14430004D+04,0.550D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.17555105D+04,0.550D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.14062206D+04,0.550D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.11083145D+04,0.550D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.95019750D+03,0.550D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.79421840D+03,0.550D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7601:   7700)=(/
     . 0.66295560D+03,0.550D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.75604361D+04,0.550D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.53051627D+04,0.550D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.42896097D+04,0.550D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.36748556D+04,0.550D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.32563633D+04,0.550D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.23987520D+04,0.550D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.27260486D+04,0.550D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.19683066D+04,0.550D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.21299275D+04,0.550D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.19358392D+04,0.550D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.16577829D+04,0.550D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.16630447D+04,0.550D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.22264867D+04,0.550D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.18919207D+04,0.550D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.15665042D+04,0.550D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.13945709D+04,0.550D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.12075672D+04,0.550D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.10433955D+04,0.550D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.93307294D+04,0.550D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7701:   7800)=(/
     . 0.99457900D+02,0.560D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.59114100D+02,0.560D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.25515260D+04,0.560D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.11604419D+04,0.560D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.69068520D+03,0.560D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.42636670D+03,0.560D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.27907420D+03,0.560D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.20174110D+03,0.560D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.14691740D+03,0.560D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.10950000D+03,0.560D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.30097233D+04,0.560D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.19263504D+04,0.560D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.16657662D+04,0.560D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.12012021D+04,0.560D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.87116470D+03,0.560D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.69086510D+03,0.560D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.54031600D+03,0.560D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.42579090D+03,0.560D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.51712978D+04,0.560D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.36591655D+04,0.560D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7801:   7900)=(/
     . 0.29244395D+04,0.560D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.27489797D+04,0.560D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.24751427D+04,0.560D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.19521126D+04,0.560D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.20794950D+04,0.560D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.16284040D+04,0.560D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.16569384D+04,0.560D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.17363987D+04,0.560D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.13357319D+04,0.560D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.12926630D+04,0.560D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.15597012D+04,0.560D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.12819152D+04,0.560D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.10285424D+04,0.560D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.88961470D+03,0.560D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.74984750D+03,0.560D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.63024360D+03,0.560D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.57435673D+04,0.560D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.43943607D+04,0.560D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.36394735D+04,0.560D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.31601776D+04,0.560D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   7901:   8000)=(/
     . 0.28195147D+04,0.560D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.20984813D+04,0.560D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.23739131D+04,0.560D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.17354827D+04,0.560D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.18901371D+04,0.560D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.17272738D+04,0.560D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.14605858D+04,0.560D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.14937369D+04,0.560D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.19640913D+04,0.560D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.17108832D+04,0.560D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.14425135D+04,0.560D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.12956354D+04,0.560D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.11317221D+04,0.560D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.98512660D+03,0.560D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.70164917D+04,0.560D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.57269887D+04,0.560D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.89101800D+02,0.570D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.53798200D+02,0.570D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.20850661D+04,0.570D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.99403160D+03,0.570D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   8001:   8100)=(/
     . 0.60449150D+03,0.570D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.37879280D+03,0.570D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.25055770D+03,0.570D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.18240750D+03,0.570D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.13361230D+03,0.570D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.10002770D+03,0.570D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.24655965D+04,0.570D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.16371586D+04,0.570D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.14317496D+04,0.570D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.10491934D+04,0.570D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.77049240D+03,0.570D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.61555590D+03,0.570D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.48476910D+03,0.570D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.38423060D+03,0.570D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.41858968D+04,0.570D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.30598941D+04,0.570D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.24617619D+04,0.570D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.23247454D+04,0.570D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.20995853D+04,0.570D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.16534435D+04,0.570D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   8101:   8200)=(/
     . 0.17717653D+04,0.570D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.13863132D+04,0.570D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.14234023D+04,0.570D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.14871454D+04,0.570D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.11414078D+04,0.570D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.11183973D+04,0.570D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.13447968D+04,0.570D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.11200021D+04,0.570D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.90836700D+03,0.570D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.79048550D+03,0.570D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.67038430D+03,0.570D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.56651010D+03,0.570D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.46537454D+04,0.570D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.36666556D+04,0.570D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.30693037D+04,0.570D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.26820879D+04,0.570D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.24019326D+04,0.570D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.17981970D+04,0.570D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.20296163D+04,0.570D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.14938895D+04,0.570D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   8201:   8300)=(/
     . 0.16299126D+04,0.570D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.14934855D+04,0.570D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.12574297D+04,0.570D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.12959364D+04,0.570D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.16894085D+04,0.570D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.14890440D+04,0.570D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.12682622D+04,0.570D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.11455595D+04,0.570D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.10065104D+04,0.570D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.88082020D+03,0.570D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.56757434D+04,0.570D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.47544826D+04,0.570D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.39906172D+04,0.570D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.44510400D+02,0.580D+02,0.100D+01,0.27991000D+01,0.91180000D+00
     .,0.29308100D+02,0.580D+02,0.200D+01,0.27991000D+01,0.00000000D+00
     .,0.70433450D+03,0.580D+02,0.300D+01,0.27991000D+01,0.00000000D+00
     .,0.40374270D+03,0.580D+02,0.400D+01,0.27991000D+01,0.00000000D+00
     .,0.27061870D+03,0.580D+02,0.500D+01,0.27991000D+01,0.00000000D+00
     .,0.18221640D+03,0.580D+02,0.600D+01,0.27991000D+01,0.00000000D+00
     .,0.12713550D+03,0.580D+02,0.700D+01,0.27991000D+01,0.00000000D+00
     ./)
      pars(   8301:   8400)=(/
     . 0.96118900D+02,0.580D+02,0.800D+01,0.27991000D+01,0.00000000D+00
     .,0.72735900D+02,0.580D+02,0.900D+01,0.27991000D+01,0.00000000D+00
     .,0.55904800D+02,0.580D+02,0.100D+02,0.27991000D+01,0.00000000D+00
     .,0.84189860D+03,0.580D+02,0.110D+02,0.27991000D+01,0.00000000D+00
     .,0.64422200D+03,0.580D+02,0.120D+02,0.27991000D+01,0.00000000D+00
     .,0.59240470D+03,0.580D+02,0.130D+02,0.27991000D+01,0.00000000D+00
     .,0.46526310D+03,0.580D+02,0.140D+02,0.27991000D+01,0.00000000D+00
     .,0.36180150D+03,0.580D+02,0.150D+02,0.27991000D+01,0.00000000D+00
     .,0.29974470D+03,0.580D+02,0.160D+02,0.27991000D+01,0.00000000D+00
     .,0.24444680D+03,0.580D+02,0.170D+02,0.27991000D+01,0.00000000D+00
     .,0.19971380D+03,0.580D+02,0.180D+02,0.27991000D+01,0.00000000D+00
     .,0.13810183D+04,0.580D+02,0.190D+02,0.27991000D+01,0.00000000D+00
     .,0.11346748D+04,0.580D+02,0.200D+02,0.27991000D+01,0.00000000D+00
     .,0.93640680D+03,0.580D+02,0.210D+02,0.27991000D+01,0.00000000D+00
     .,0.90333060D+03,0.580D+02,0.220D+02,0.27991000D+01,0.00000000D+00
     .,0.82670470D+03,0.580D+02,0.230D+02,0.27991000D+01,0.00000000D+00
     .,0.65114400D+03,0.580D+02,0.240D+02,0.27991000D+01,0.00000000D+00
     .,0.71114230D+03,0.580D+02,0.250D+02,0.27991000D+01,0.00000000D+00
     .,0.55801770D+03,0.580D+02,0.260D+02,0.27991000D+01,0.00000000D+00
     .,0.59073870D+03,0.580D+02,0.270D+02,0.27991000D+01,0.00000000D+00
     ./)
      pars(   8401:   8500)=(/
     . 0.60892420D+03,0.580D+02,0.280D+02,0.27991000D+01,0.00000000D+00
     .,0.46681100D+03,0.580D+02,0.290D+02,0.27991000D+01,0.00000000D+00
     .,0.47833500D+03,0.580D+02,0.300D+02,0.27991000D+01,0.00000000D+00
     .,0.56687830D+03,0.580D+02,0.310D+02,0.27991000D+01,0.00000000D+00
     .,0.49862840D+03,0.580D+02,0.320D+02,0.27991000D+01,0.00000000D+00
     .,0.42429060D+03,0.580D+02,0.330D+02,0.27991000D+01,0.00000000D+00
     .,0.38016060D+03,0.580D+02,0.340D+02,0.27991000D+01,0.00000000D+00
     .,0.33218840D+03,0.580D+02,0.350D+02,0.27991000D+01,0.00000000D+00
     .,0.28850100D+03,0.580D+02,0.360D+02,0.27991000D+01,0.00000000D+00
     .,0.15470780D+04,0.580D+02,0.370D+02,0.27991000D+01,0.00000000D+00
     .,0.13520803D+04,0.580D+02,0.380D+02,0.27991000D+01,0.00000000D+00
     .,0.11820124D+04,0.580D+02,0.390D+02,0.27991000D+01,0.00000000D+00
     .,0.10609721D+04,0.580D+02,0.400D+02,0.27991000D+01,0.00000000D+00
     .,0.96662010D+03,0.580D+02,0.410D+02,0.27991000D+01,0.00000000D+00
     .,0.74511760D+03,0.580D+02,0.420D+02,0.27991000D+01,0.00000000D+00
     .,0.83183600D+03,0.580D+02,0.430D+02,0.27991000D+01,0.00000000D+00
     .,0.63266390D+03,0.580D+02,0.440D+02,0.27991000D+01,0.00000000D+00
     .,0.69156460D+03,0.580D+02,0.450D+02,0.27991000D+01,0.00000000D+00
     .,0.64096110D+03,0.580D+02,0.460D+02,0.27991000D+01,0.00000000D+00
     .,0.53459930D+03,0.580D+02,0.470D+02,0.27991000D+01,0.00000000D+00
     ./)
      pars(   8501:   8600)=(/
     . 0.56439100D+03,0.580D+02,0.480D+02,0.27991000D+01,0.00000000D+00
     .,0.70950090D+03,0.580D+02,0.490D+02,0.27991000D+01,0.00000000D+00
     .,0.65467230D+03,0.580D+02,0.500D+02,0.27991000D+01,0.00000000D+00
     .,0.58216180D+03,0.580D+02,0.510D+02,0.27991000D+01,0.00000000D+00
     .,0.53941670D+03,0.580D+02,0.520D+02,0.27991000D+01,0.00000000D+00
     .,0.48703330D+03,0.580D+02,0.530D+02,0.27991000D+01,0.00000000D+00
     .,0.43728340D+03,0.580D+02,0.540D+02,0.27991000D+01,0.00000000D+00
     .,0.18844554D+04,0.580D+02,0.550D+02,0.27991000D+01,0.00000000D+00
     .,0.17247415D+04,0.580D+02,0.560D+02,0.27991000D+01,0.00000000D+00
     .,0.15122504D+04,0.580D+02,0.570D+02,0.27991000D+01,0.00000000D+00
     .,0.68803530D+03,0.580D+02,0.580D+02,0.27991000D+01,0.27991000D+01
     .,0.88687700D+02,0.590D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.53152300D+02,0.590D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.22103168D+04,0.590D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.10189540D+04,0.590D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.61076150D+03,0.590D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.37910380D+03,0.590D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.24926330D+03,0.590D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.18083830D+03,0.590D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.13215840D+03,0.590D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   8601:   8700)=(/
     . 0.98822000D+02,0.590D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.26090440D+04,0.590D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.16875419D+04,0.590D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.14644446D+04,0.590D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.10615213D+04,0.590D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.77317210D+03,0.590D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.61486520D+03,0.590D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.48225090D+03,0.590D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.38105030D+03,0.590D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.44737784D+04,0.590D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.31902086D+04,0.590D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.25547353D+04,0.590D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.24050561D+04,0.590D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.21676325D+04,0.590D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.17091137D+04,0.590D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.18237858D+04,0.590D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.14281039D+04,0.590D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.14570358D+04,0.590D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.15254141D+04,0.590D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.11729549D+04,0.590D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   8701:   8800)=(/
     . 0.11394352D+04,0.590D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.13728649D+04,0.590D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.11331627D+04,0.590D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.91252630D+03,0.590D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.79102140D+03,0.590D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.66833280D+03,0.590D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.56301560D+03,0.590D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.49710778D+04,0.590D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.38289626D+04,0.590D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.31814405D+04,0.590D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.27679894D+04,0.590D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.24726905D+04,0.590D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.18440715D+04,0.590D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.20845975D+04,0.590D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.15276056D+04,0.590D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.16644640D+04,0.590D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.15224806D+04,0.590D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.12860441D+04,0.590D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.13182791D+04,0.590D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.17282162D+04,0.590D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   8801:   8900)=(/
     . 0.15110459D+04,0.590D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.12783406D+04,0.590D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.11504200D+04,0.590D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.10070198D+04,0.590D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.87837980D+03,0.590D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.60738935D+04,0.590D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.49833126D+04,0.590D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.41503203D+04,0.590D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.15271461D+04,0.590D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.43422386D+04,0.590D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.85408000D+02,0.600D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.51295200D+02,0.600D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.20915068D+04,0.600D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.97461670D+03,0.600D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.58618960D+03,0.600D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.36465190D+03,0.600D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.24011390D+03,0.600D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.17436780D+03,0.600D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12753080D+03,0.600D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.95420200D+02,0.600D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   8901:   9000)=(/
     . 0.24706020D+04,0.600D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.16117831D+04,0.600D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.14013191D+04,0.600D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.10185098D+04,0.600D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.74324070D+03,0.600D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.59167520D+03,0.600D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.46450620D+03,0.600D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.36731630D+03,0.600D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.42163998D+04,0.600D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.30363662D+04,0.600D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.24350889D+04,0.600D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.22942857D+04,0.600D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.20689809D+04,0.600D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.16306018D+04,0.600D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.17421942D+04,0.600D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.13638456D+04,0.600D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.13940822D+04,0.600D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.14587399D+04,0.600D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.11209931D+04,0.600D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.10915722D+04,0.600D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9001:   9100)=(/
     . 0.13142306D+04,0.600D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.10872533D+04,0.600D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.87700660D+03,0.600D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.76089490D+03,0.600D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.64343050D+03,0.600D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.54243830D+03,0.600D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.46850846D+04,0.600D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.36420492D+04,0.600D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.30329183D+04,0.600D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.26420526D+04,0.600D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.23616974D+04,0.600D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.17631139D+04,0.600D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.19921534D+04,0.600D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.14616458D+04,0.600D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.15933696D+04,0.600D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.14581671D+04,0.600D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.12304969D+04,0.600D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.12633482D+04,0.600D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.16535039D+04,0.600D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.14488711D+04,0.600D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9101:   9200)=(/
     . 0.12277455D+04,0.600D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.11058143D+04,0.600D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.96879470D+03,0.600D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.84567030D+03,0.600D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.57167999D+04,0.600D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.47334185D+04,0.600D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.39519956D+04,0.600D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.14659695D+04,0.600D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.41259023D+04,0.600D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.39244211D+04,0.600D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.83331000D+02,0.610D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.50074800D+02,0.610D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.20312740D+04,0.610D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.94912550D+03,0.610D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.57141130D+03,0.610D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.35567120D+03,0.610D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.23429130D+03,0.610D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.17018170D+03,0.610D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12449460D+03,0.610D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.93163500D+02,0.610D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9201:   9300)=(/
     . 0.23998207D+04,0.610D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.15689885D+04,0.610D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.13648494D+04,0.610D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.99275070D+03,0.610D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.72481970D+03,0.610D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.57716960D+03,0.610D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.45323130D+03,0.610D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.35847210D+03,0.610D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.40920575D+04,0.610D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.29529794D+04,0.610D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.23691244D+04,0.610D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.22326503D+04,0.610D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.20137194D+04,0.610D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.15868512D+04,0.610D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.16960441D+04,0.610D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.13276064D+04,0.610D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.13577395D+04,0.610D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.14205045D+04,0.610D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.10914214D+04,0.610D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.10634898D+04,0.610D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9301:   9400)=(/
     . 0.12801762D+04,0.610D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.10597536D+04,0.610D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.85521770D+03,0.610D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.74216490D+03,0.610D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.62773420D+03,0.610D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.52930700D+03,0.610D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.45470720D+04,0.610D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.35414930D+04,0.610D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.29509303D+04,0.610D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.25714964D+04,0.610D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.22990481D+04,0.610D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.17168038D+04,0.610D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.19396215D+04,0.610D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.14235531D+04,0.610D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.15520522D+04,0.610D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.14205469D+04,0.610D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.11984194D+04,0.610D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.12309598D+04,0.610D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.16104142D+04,0.610D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.14119588D+04,0.610D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9401:   9500)=(/
     . 0.11970157D+04,0.610D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.10783854D+04,0.610D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.94498140D+03,0.610D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.82504730D+03,0.610D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.55474900D+04,0.610D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.46012171D+04,0.610D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.38440367D+04,0.610D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.14290690D+04,0.610D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.40113635D+04,0.610D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.38161118D+04,0.610D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.37109375D+04,0.610D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.81412300D+02,0.620D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.48940600D+02,0.620D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.19766955D+04,0.620D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.92581580D+03,0.620D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.55783800D+03,0.620D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.34739210D+03,0.620D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.22890650D+03,0.620D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.16630040D+03,0.620D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12167270D+03,0.620D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9501:   9600)=(/
     . 0.91061400D+02,0.620D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.23356474D+04,0.620D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.15299120D+04,0.620D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.13314786D+04,0.620D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.96910260D+03,0.620D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.70786040D+03,0.620D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.56378900D+03,0.620D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.44280950D+03,0.620D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.35028120D+03,0.620D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.39796309D+04,0.620D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.28770680D+04,0.620D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.23089962D+04,0.620D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.21764184D+04,0.620D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.19632723D+04,0.620D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.15469195D+04,0.620D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.16538773D+04,0.620D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.12944951D+04,0.620D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.13244770D+04,0.620D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.13855313D+04,0.620D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.10643798D+04,0.620D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9601:   9700)=(/
     . 0.10377490D+04,0.620D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.12489915D+04,0.620D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.10345024D+04,0.620D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.83516220D+03,0.620D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.72489870D+03,0.620D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.61324050D+03,0.620D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.51716170D+03,0.620D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.44222696D+04,0.620D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.34499906D+04,0.620D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.28761716D+04,0.620D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.25070805D+04,0.620D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.22418085D+04,0.620D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.16744360D+04,0.620D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.18915883D+04,0.620D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.13886668D+04,0.620D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.15142016D+04,0.620D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.13860622D+04,0.620D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.11690360D+04,0.620D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.12012473D+04,0.620D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.15709643D+04,0.620D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9701:   9800)=(/
     . 0.13780830D+04,0.620D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.11687524D+04,0.620D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.10531255D+04,0.620D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.92301940D+03,0.620D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.80600020D+03,0.620D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.53944846D+04,0.620D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.44810449D+04,0.620D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.37456921D+04,0.620D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.13951471D+04,0.620D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.39071923D+04,0.620D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.37175407D+04,0.620D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.36152067D+04,0.620D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.35220508D+04,0.620D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.79713800D+02,0.630D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.47936500D+02,0.630D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.19280810D+04,0.630D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.90511010D+03,0.630D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.54580150D+03,0.630D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.34005800D+03,0.630D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.22413830D+03,0.630D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9801:   9900)=(/
     . 0.16286380D+03,0.630D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.11917340D+03,0.630D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.89198800D+02,0.630D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.22784898D+04,0.630D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.14951792D+04,0.630D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.13018438D+04,0.630D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.94812880D+03,0.630D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.69283380D+03,0.630D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.55193900D+03,0.630D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.43358310D+03,0.630D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.34303070D+03,0.630D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.38794729D+04,0.630D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.28095233D+04,0.630D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.22555157D+04,0.630D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.21264190D+04,0.630D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.19184254D+04,0.630D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.15114156D+04,0.630D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.16164023D+04,0.630D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.12650643D+04,0.630D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.12949313D+04,0.630D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(   9901:  10000)=(/
     . 0.13544595D+04,0.630D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.10403490D+04,0.630D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.10148959D+04,0.630D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.12213030D+04,0.630D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.10121053D+04,0.630D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.81738880D+03,0.630D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.70960390D+03,0.630D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.60040640D+03,0.630D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.50640980D+03,0.630D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.43110961D+04,0.630D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.33685646D+04,0.630D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.28096885D+04,0.630D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.24498188D+04,0.630D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.21909387D+04,0.630D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.16367956D+04,0.630D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.18489097D+04,0.630D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.13576810D+04,0.630D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.14805878D+04,0.630D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.13554425D+04,0.630D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.11429352D+04,0.630D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10001:  10100)=(/
     . 0.11748699D+04,0.630D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.15359254D+04,0.630D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.13480216D+04,0.630D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.11436919D+04,0.630D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.10307380D+04,0.630D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.90356300D+03,0.630D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.78913200D+03,0.630D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.52582068D+04,0.630D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.43740826D+04,0.630D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.36582120D+04,0.630D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.13650615D+04,0.630D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.38144921D+04,0.630D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.36298268D+04,0.630D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.35300228D+04,0.630D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.34391601D+04,0.630D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.33583122D+04,0.630D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.64188000D+02,0.640D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.39578600D+02,0.640D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.14077335D+04,0.640D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.68992870D+03,0.640D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10101:  10200)=(/
     . 0.42648970D+03,0.640D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.27091830D+03,0.640D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.18125810D+03,0.640D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.13314380D+03,0.640D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.98357300D+02,0.640D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.74191200D+02,0.640D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.16673895D+04,0.640D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.11308198D+04,0.640D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.99679240D+03,0.640D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.73902000D+03,0.640D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.54836330D+03,0.640D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.44123030D+03,0.640D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.35003690D+03,0.640D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.27935150D+03,0.640D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.28176333D+04,0.640D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.20945326D+04,0.640D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.16917052D+04,0.640D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.16028927D+04,0.640D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.14507174D+04,0.640D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.11427015D+04,0.640D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10201:  10300)=(/
     . 0.12280426D+04,0.640D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.96156000D+03,0.640D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.99206730D+03,0.640D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.10341932D+04,0.640D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.79387630D+03,0.640D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.78351140D+03,0.640D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.93936520D+03,0.640D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.78963690D+03,0.640D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.64598490D+03,0.640D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.56529910D+03,0.640D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.48231720D+03,0.640D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.40998030D+03,0.640D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.31359242D+04,0.640D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.25077589D+04,0.640D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.21132076D+04,0.640D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.18544837D+04,0.640D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.16654372D+04,0.640D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.12530730D+04,0.640D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.14116759D+04,0.640D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.10450617D+04,0.640D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10301:  10400)=(/
     . 0.11404525D+04,0.640D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.10471279D+04,0.640D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.88053130D+03,0.640D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.91107960D+03,0.640D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.11799720D+04,0.640D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.10480976D+04,0.640D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.89951620D+03,0.640D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.81630470D+03,0.640D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.72098470D+03,0.640D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.63421630D+03,0.640D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.38238426D+04,0.640D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.32436561D+04,0.640D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.27405478D+04,0.640D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.10709597D+04,0.640D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.28381357D+04,0.640D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.27056826D+04,0.640D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.26326657D+04,0.640D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.25660607D+04,0.640D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.25068352D+04,0.640D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.18916719D+04,0.640D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10401:  10500)=(/
     . 0.70179400D+02,0.650D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.42367900D+02,0.650D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.18023852D+04,0.650D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.80848660D+03,0.650D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.48279770D+03,0.650D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.29984690D+03,0.650D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.19762340D+03,0.650D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.14377540D+03,0.650D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10539700D+03,0.650D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.79048400D+02,0.650D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.21240831D+04,0.650D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.13433651D+04,0.650D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.11621560D+04,0.650D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.83917760D+03,0.650D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.61083180D+03,0.650D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.48623190D+03,0.650D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.38195840D+03,0.650D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.30241490D+03,0.650D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.36876305D+04,0.650D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.25641443D+04,0.650D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10501:  10600)=(/
     . 0.20454927D+04,0.650D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.19228302D+04,0.650D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.17309890D+04,0.650D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.13680049D+04,0.650D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.14541614D+04,0.650D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.11408713D+04,0.650D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11579011D+04,0.650D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.12132014D+04,0.650D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.93575440D+03,0.650D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.90359910D+03,0.650D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.10900939D+04,0.650D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.89646510D+03,0.650D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.72120420D+03,0.650D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.62544800D+03,0.650D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.52893810D+03,0.650D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.44620840D+03,0.650D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.40985678D+04,0.650D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.30839899D+04,0.650D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.25485911D+04,0.650D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.22114907D+04,0.650D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10601:  10700)=(/
     . 0.19735648D+04,0.650D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.14705265D+04,0.650D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.16631347D+04,0.650D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.12173511D+04,0.650D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.13235223D+04,0.650D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.12096847D+04,0.650D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.10256835D+04,0.650D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.10465789D+04,0.650D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.13756067D+04,0.650D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.11975597D+04,0.650D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.10112975D+04,0.650D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.90991270D+03,0.650D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.79669950D+03,0.650D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.69543500D+03,0.650D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.50226967D+04,0.650D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.40300910D+04,0.650D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.33357435D+04,0.650D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.12096287D+04,0.650D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.35071716D+04,0.650D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.33250797D+04,0.650D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10701:  10800)=(/
     . 0.32313241D+04,0.650D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.31461740D+04,0.650D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.30703853D+04,0.650D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.22762532D+04,0.650D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.28516677D+04,0.650D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.67958000D+02,0.660D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.41162200D+02,0.660D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.17207353D+04,0.660D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.77577180D+03,0.660D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.46524390D+03,0.660D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.28984550D+03,0.660D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.19145500D+03,0.660D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.13949370D+03,0.660D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10238110D+03,0.660D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.76855700D+02,0.660D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.20278568D+04,0.660D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.12872471D+04,0.660D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.11160341D+04,0.660D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.80833220D+03,0.660D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.58989410D+03,0.660D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10801:  10900)=(/
     . 0.47029770D+03,0.660D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.36998330D+03,0.660D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.29329050D+03,0.660D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.35269705D+04,0.660D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.24517734D+04,0.660D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.19574581D+04,0.660D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.18415883D+04,0.660D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.16587019D+04,0.660D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.13107538D+04,0.660D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.13944853D+04,0.660D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.10940016D+04,0.660D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11118174D+04,0.660D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.11642755D+04,0.660D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.89783730D+03,0.660D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.86875770D+03,0.660D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.10475640D+04,0.660D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.86358600D+03,0.660D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.69627600D+03,0.660D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.60460480D+03,0.660D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.51197160D+03,0.660D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  10901:  11000)=(/
     . 0.43238930D+03,0.660D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.39216962D+04,0.660D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.29485688D+04,0.660D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.24401762D+04,0.660D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.21194908D+04,0.660D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.18927858D+04,0.660D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.14117363D+04,0.660D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.15962074D+04,0.660D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.11696785D+04,0.660D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.12718937D+04,0.660D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.11630374D+04,0.660D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.98556970D+03,0.660D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.10068382D+04,0.660D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.13215257D+04,0.660D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.11527975D+04,0.660D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.97544350D+03,0.660D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.87867530D+03,0.660D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.77029080D+03,0.660D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.67313940D+03,0.660D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.48123323D+04,0.660D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11001:  11100)=(/
     . 0.38522949D+04,0.660D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.31924039D+04,0.660D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.11661633D+04,0.660D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.33558765D+04,0.660D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.31799629D+04,0.660D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.30904072D+04,0.660D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.30090778D+04,0.660D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.29367065D+04,0.660D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.21810675D+04,0.660D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.27299739D+04,0.660D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.26173310D+04,0.660D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.72220300D+02,0.670D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.43571100D+02,0.670D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.17049679D+04,0.670D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.81136880D+03,0.670D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.49188750D+03,0.670D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.30751980D+03,0.670D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.20315540D+03,0.670D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.14783550D+03,0.670D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10830760D+03,0.670D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11101:  11200)=(/
     . 0.81141600D+02,0.670D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.20163392D+04,0.670D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.13374783D+04,0.670D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.11679358D+04,0.670D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.85407400D+03,0.670D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.62593940D+03,0.670D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.49945760D+03,0.670D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.39293850D+03,0.670D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.31124870D+03,0.670D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.34195160D+04,0.670D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.25014164D+04,0.670D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.20120352D+04,0.670D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.18992044D+04,0.670D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.17148635D+04,0.670D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.13502932D+04,0.670D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.14466020D+04,0.670D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.11317710D+04,0.670D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11614905D+04,0.670D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.12139348D+04,0.670D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.93167930D+03,0.670D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11201:  11300)=(/
     . 0.91200820D+03,0.670D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.10964450D+04,0.670D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.91172230D+03,0.670D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.73821290D+03,0.670D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.64174520D+03,0.670D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.54371620D+03,0.670D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.45912050D+03,0.670D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.38008304D+04,0.670D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.29970212D+04,0.670D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.25073572D+04,0.670D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.21900272D+04,0.670D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.19605194D+04,0.670D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.14668017D+04,0.670D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.16559466D+04,0.670D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.12180651D+04,0.670D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.13291426D+04,0.670D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.12176582D+04,0.670D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.10253902D+04,0.670D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.10563718D+04,0.670D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.13778901D+04,0.670D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11301:  11400)=(/
     . 0.12131043D+04,0.670D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.10318071D+04,0.670D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.93112390D+03,0.670D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.81732200D+03,0.670D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.71464770D+03,0.670D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.46328428D+04,0.670D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.38855093D+04,0.670D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.32599132D+04,0.670D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.12305989D+04,0.670D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.33915411D+04,0.670D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.32298168D+04,0.670D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.31416098D+04,0.670D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.30612537D+04,0.670D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.29897660D+04,0.670D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.22379026D+04,0.670D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.27243602D+04,0.670D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.26065136D+04,0.670D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.26641668D+04,0.670D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.70715400D+02,0.680D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.42665800D+02,0.680D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11401:  11500)=(/
     . 0.16647700D+04,0.680D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.79368770D+03,0.680D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.48143850D+03,0.680D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.30106920D+03,0.680D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.19891850D+03,0.680D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.14475870D+03,0.680D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10605520D+03,0.680D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.79453900D+02,0.680D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.19689928D+04,0.680D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.13079814D+04,0.680D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.11425656D+04,0.680D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.83589690D+03,0.680D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.61278090D+03,0.680D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.48901070D+03,0.680D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.38474960D+03,0.680D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.30477480D+03,0.680D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.33372227D+04,0.680D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.24446758D+04,0.680D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.19669050D+04,0.680D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.18568743D+04,0.680D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11501:  11600)=(/
     . 0.16768129D+04,0.680D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.13201912D+04,0.680D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.14147048D+04,0.680D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.11067246D+04,0.680D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11361897D+04,0.680D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.11873873D+04,0.680D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.91117070D+03,0.680D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.89233320D+03,0.680D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.10726768D+04,0.680D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.89230170D+03,0.680D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.72266540D+03,0.680D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.62829340D+03,0.680D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.53236410D+03,0.680D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.44955920D+03,0.680D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.37094297D+04,0.680D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.29287164D+04,0.680D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.24511775D+04,0.680D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.21414202D+04,0.680D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.19172203D+04,0.680D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.14346130D+04,0.680D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11601:  11700)=(/
     . 0.16195186D+04,0.680D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.11914716D+04,0.680D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.13002637D+04,0.680D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.11912962D+04,0.680D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.10029797D+04,0.680D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.10336009D+04,0.680D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.13478497D+04,0.680D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.11871040D+04,0.680D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.10099583D+04,0.680D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.91151370D+03,0.680D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.80019180D+03,0.680D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.69972310D+03,0.680D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.45209645D+04,0.680D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.37960924D+04,0.680D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.31862327D+04,0.680D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.12044185D+04,0.680D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.33138879D+04,0.680D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.31561992D+04,0.680D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.30700842D+04,0.680D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.29916275D+04,0.680D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11701:  11800)=(/
     . 0.29218310D+04,0.680D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.21877803D+04,0.680D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.26611461D+04,0.680D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.25461249D+04,0.680D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.26039653D+04,0.680D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.25451713D+04,0.680D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.69379100D+02,0.690D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.41872700D+02,0.690D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.16270951D+04,0.690D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.77746020D+03,0.690D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.47198600D+03,0.690D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.29530140D+03,0.690D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.19516300D+03,0.690D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.14204780D+03,0.690D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10408000D+03,0.690D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.77978800D+02,0.690D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.19246593D+04,0.690D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.12807911D+04,0.690D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.11193391D+04,0.690D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.81942800D+03,0.690D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11801:  11900)=(/
     . 0.60097010D+03,0.690D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.47969090D+03,0.690D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.37748730D+03,0.690D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.29906220D+03,0.690D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.32599368D+04,0.690D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.23919954D+04,0.690D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.19251263D+04,0.690D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.18177881D+04,0.690D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.16417354D+04,0.690D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.12924384D+04,0.690D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.13853712D+04,0.690D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.10836957D+04,0.690D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11130259D+04,0.690D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.11630369D+04,0.690D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.89235160D+03,0.690D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.87439610D+03,0.690D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.10509705D+04,0.690D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.87471320D+03,0.690D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.70869500D+03,0.690D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.61626580D+03,0.690D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  11901:  12000)=(/
     . 0.52226550D+03,0.690D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.44109320D+03,0.690D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.36236511D+04,0.690D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.28652613D+04,0.690D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.23992487D+04,0.690D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.20966400D+04,0.690D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.18774174D+04,0.690D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.14051352D+04,0.690D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.15861100D+04,0.690D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.11671878D+04,0.690D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.12739026D+04,0.690D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.11672707D+04,0.690D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.98252100D+03,0.690D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.10128904D+04,0.690D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.13203887D+04,0.690D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.11634988D+04,0.690D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.99025880D+03,0.690D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.89390800D+03,0.690D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.78488410D+03,0.690D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.68644520D+03,0.690D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12001:  12100)=(/
     . 0.44159694D+04,0.690D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.37128812D+04,0.690D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.31180001D+04,0.690D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.11807776D+04,0.690D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.32417266D+04,0.690D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.30878489D+04,0.690D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.30036922D+04,0.690D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.29270118D+04,0.690D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.28587977D+04,0.690D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.21415264D+04,0.690D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.26023122D+04,0.690D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.24899643D+04,0.690D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.25481802D+04,0.690D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.24906989D+04,0.690D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.24374539D+04,0.690D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.68494900D+02,0.700D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.41286600D+02,0.700D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.16127581D+04,0.700D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.76939830D+03,0.700D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.46660990D+03,0.700D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12101:  12200)=(/
     . 0.29168170D+03,0.700D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.19263240D+03,0.700D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.14013000D+03,0.700D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.10262510D+03,0.700D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.76858100D+02,0.700D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.19075332D+04,0.700D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.12678866D+04,0.700D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.11075215D+04,0.700D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.81018450D+03,0.700D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.59379110D+03,0.700D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.47373900D+03,0.700D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.37262670D+03,0.700D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.29508360D+03,0.700D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.32317117D+04,0.700D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.23690958D+04,0.700D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.19062802D+04,0.700D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.17996377D+04,0.700D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.16251430D+04,0.700D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.12793493D+04,0.700D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.13711198D+04,0.700D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12201:  12300)=(/
     . 0.10724959D+04,0.700D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11012182D+04,0.700D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.11508553D+04,0.700D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.88299190D+03,0.700D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.86485460D+03,0.700D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.10396601D+04,0.700D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.86479960D+03,0.700D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.70027180D+03,0.700D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.60871740D+03,0.700D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.51566490D+03,0.700D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.43535350D+03,0.700D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.35920465D+04,0.700D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.28379423D+04,0.700D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.23754679D+04,0.700D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.20753417D+04,0.700D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.18580353D+04,0.700D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.13902059D+04,0.700D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.15694426D+04,0.700D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.11545209D+04,0.700D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.12600764D+04,0.700D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12301:  12400)=(/
     . 0.11544639D+04,0.700D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.97180930D+03,0.700D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.10016214D+04,0.700D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.13062023D+04,0.700D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.11504571D+04,0.700D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.97868710D+03,0.700D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.88319190D+03,0.700D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.77520910D+03,0.700D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.67775440D+03,0.700D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.43774766D+04,0.700D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.36779606D+04,0.700D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.30875154D+04,0.700D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.11670799D+04,0.700D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.32108489D+04,0.700D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.30582277D+04,0.700D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.29748238D+04,0.700D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.28988349D+04,0.700D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.28312344D+04,0.700D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.21199794D+04,0.700D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.25778005D+04,0.700D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12401:  12500)=(/
     . 0.24663583D+04,0.700D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.25233630D+04,0.700D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.24664143D+04,0.700D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.24136514D+04,0.700D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.23901227D+04,0.700D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.58651900D+02,0.710D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.35951600D+02,0.710D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.12947583D+04,0.710D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.63458810D+03,0.710D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.39146520D+03,0.710D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.24795570D+03,0.710D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.16541840D+03,0.710D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.12121480D+03,0.710D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.89340500D+02,0.710D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.67257500D+02,0.710D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.15332675D+04,0.710D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.10401825D+04,0.710D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.91638240D+03,0.710D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.67855410D+03,0.710D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.50255170D+03,0.710D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12501:  12600)=(/
     . 0.40367180D+03,0.710D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.31962860D+03,0.710D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.25459790D+03,0.710D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.25921299D+04,0.710D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.19255453D+04,0.710D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.15553939D+04,0.710D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.14734091D+04,0.710D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.13334027D+04,0.710D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.10496142D+04,0.710D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.11285309D+04,0.710D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.88302170D+03,0.710D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.91141500D+03,0.710D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.95032780D+03,0.710D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.72889260D+03,0.710D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.71952410D+03,0.710D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.86291390D+03,0.710D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.72475130D+03,0.710D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.59205870D+03,0.710D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.51745760D+03,0.710D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.44084280D+03,0.710D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12601:  12700)=(/
     . 0.37414020D+03,0.710D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.28847585D+04,0.710D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.23048558D+04,0.710D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.19421387D+04,0.710D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.17040440D+04,0.710D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.15299721D+04,0.710D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.11501604D+04,0.710D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.12962701D+04,0.710D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.95869920D+03,0.710D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.10467597D+04,0.710D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.96091280D+03,0.710D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.80744060D+03,0.710D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.83583100D+03,0.710D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.10833008D+04,0.710D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.96184020D+03,0.710D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.82469640D+03,0.710D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.74775790D+03,0.710D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.65970490D+03,0.710D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.57959110D+03,0.710D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.35195236D+04,0.710D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12701:  12800)=(/
     . 0.29806436D+04,0.710D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.25182192D+04,0.710D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.98175310D+03,0.710D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.26089193D+04,0.710D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.24861931D+04,0.710D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.24191164D+04,0.710D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.23579439D+04,0.710D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.23035540D+04,0.710D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.17376565D+04,0.710D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.20919380D+04,0.710D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.20059243D+04,0.710D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.20565310D+04,0.710D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.20105203D+04,0.710D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.19680474D+04,0.710D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.19483102D+04,0.710D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.15974796D+04,0.710D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.58850400D+02,0.720D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.36574900D+02,0.720D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.11772268D+04,0.720D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.60758590D+03,0.720D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12801:  12900)=(/
     . 0.38350840D+03,0.720D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.24668260D+03,0.720D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.16624720D+03,0.720D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.12259130D+03,0.720D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.90778400D+02,0.720D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.68550500D+02,0.720D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.13981254D+04,0.720D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.98752390D+03,0.720D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.88079020D+03,0.720D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.66333140D+03,0.720D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.49770150D+03,0.720D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.40279470D+03,0.720D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.32111450D+03,0.720D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.25716510D+03,0.720D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.23300562D+04,0.720D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.17971024D+04,0.720D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.14619589D+04,0.720D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.13919848D+04,0.720D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.12638660D+04,0.720D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.99362820D+03,0.720D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  12901:  13000)=(/
     . 0.10747287D+04,0.720D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.84045370D+03,0.720D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.87543360D+03,0.720D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.90984070D+03,0.720D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.69642830D+03,0.720D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.69621350D+03,0.720D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.83232490D+03,0.720D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.70869710D+03,0.720D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.58542500D+03,0.720D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.51487840D+03,0.720D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.44133770D+03,0.720D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.37652870D+03,0.720D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.25964033D+04,0.720D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.21464040D+04,0.720D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.18297515D+04,0.720D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.16164737D+04,0.720D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.14572492D+04,0.720D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.11025669D+04,0.720D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.12395201D+04,0.720D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.92341980D+03,0.720D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13001:  13100)=(/
     . 0.10097527D+04,0.720D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.92946760D+03,0.720D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.77779950D+03,0.720D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.81120400D+03,0.720D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.10422993D+04,0.720D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.93663610D+03,0.720D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.81148690D+03,0.720D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.74006690D+03,0.720D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.65681710D+03,0.720D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.58013770D+03,0.720D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.31606446D+04,0.720D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.27609094D+04,0.720D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.23609186D+04,0.720D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.96347730D+03,0.720D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.24240491D+04,0.720D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.23170514D+04,0.720D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.22560584D+04,0.720D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.22002906D+04,0.720D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.21507417D+04,0.720D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.16406567D+04,0.720D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13101:  13200)=(/
     . 0.19309949D+04,0.720D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.18529581D+04,0.720D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.19265925D+04,0.720D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.18843048D+04,0.720D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.18455050D+04,0.720D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.18262149D+04,0.720D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.15072441D+04,0.720D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.14412394D+04,0.720D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.54401500D+02,0.730D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.34283400D+02,0.730D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.10273577D+04,0.730D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.54443550D+03,0.730D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.34862850D+03,0.730D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.22671590D+03,0.730D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.15406960D+03,0.730D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.11429670D+03,0.730D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.85081400D+02,0.730D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.64523800D+02,0.730D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.12220335D+04,0.730D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.88084690D+03,0.730D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13201:  13300)=(/
     . 0.79141350D+03,0.730D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.60213470D+03,0.730D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.45569700D+03,0.730D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.37087230D+03,0.730D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.29728390D+03,0.730D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.23922910D+03,0.730D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.20266698D+04,0.730D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.15894883D+04,0.730D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.12977826D+04,0.730D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.12395033D+04,0.730D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.11275812D+04,0.730D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.88658800D+03,0.730D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.96151940D+03,0.730D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.75231040D+03,0.730D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.78703270D+03,0.730D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.81634550D+03,0.730D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.62484080D+03,0.730D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.62867410D+03,0.730D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.74997760D+03,0.730D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.64373010D+03,0.730D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13301:  13400)=(/
     . 0.53560150D+03,0.730D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.47316850D+03,0.730D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.40746910D+03,0.730D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.34913050D+03,0.730D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.22608400D+04,0.730D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.18969799D+04,0.730D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.16272670D+04,0.730D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.14431973D+04,0.730D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.13043213D+04,0.730D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.99116570D+03,0.730D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.11124511D+04,0.730D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.83281910D+03,0.730D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.91084260D+03,0.730D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.83983510D+03,0.730D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.70192840D+03,0.730D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.73456590D+03,0.730D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.93877020D+03,0.730D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.84925790D+03,0.730D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.74051010D+03,0.730D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.67794300D+03,0.730D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13401:  13500)=(/
     . 0.60419180D+03,0.730D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.53577730D+03,0.730D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.27516509D+04,0.730D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.24344207D+04,0.730D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.20948531D+04,0.730D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.87810980D+03,0.730D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.21417667D+04,0.730D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.20496125D+04,0.730D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.19962879D+04,0.730D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.19474669D+04,0.730D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.19041085D+04,0.730D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.14623131D+04,0.730D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.17026705D+04,0.730D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.16355752D+04,0.730D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.17084912D+04,0.730D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.16713110D+04,0.730D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.16373225D+04,0.730D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.16197710D+04,0.730D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.13428697D+04,0.730D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.12925836D+04,0.730D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13501:  13600)=(/
     . 0.11638241D+04,0.730D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.46481300D+02,0.740D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.29820600D+02,0.740D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.84241230D+03,0.740D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.45198610D+03,0.740D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.29290150D+03,0.740D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.19259840D+03,0.740D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.13214100D+03,0.740D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.98766600D+02,0.740D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.74030900D+02,0.740D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.56476700D+02,0.740D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.10029508D+04,0.740D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.72933520D+03,0.740D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.65874710D+03,0.740D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.50518900D+03,0.740D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.38541090D+03,0.740D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.31556320D+03,0.740D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.25452870D+03,0.740D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.20602860D+03,0.740D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.16644321D+04,0.740D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13601:  13700)=(/
     . 0.13117641D+04,0.740D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.10728050D+04,0.740D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.10269214D+04,0.740D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.93538590D+03,0.740D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.73650630D+03,0.740D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.79917580D+03,0.740D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.62633800D+03,0.740D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.65622600D+03,0.740D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.67963140D+03,0.740D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.52108220D+03,0.740D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.52588940D+03,0.740D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.62622030D+03,0.740D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.54069840D+03,0.740D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.45278370D+03,0.740D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.40184290D+03,0.740D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.34779630D+03,0.740D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.29948800D+03,0.740D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.18587683D+04,0.740D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.15659413D+04,0.740D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.13478163D+04,0.740D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13701:  13800)=(/
     . 0.11983202D+04,0.740D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.10850878D+04,0.740D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.82786250D+03,0.740D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.92774070D+03,0.740D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.69763460D+03,0.740D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.76226600D+03,0.740D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.70376190D+03,0.740D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.58870290D+03,0.740D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.61664100D+03,0.740D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.78470960D+03,0.740D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.71294940D+03,0.740D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.62490100D+03,0.740D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.57415840D+03,0.740D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.51383150D+03,0.740D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.45758530D+03,0.740D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.22639013D+04,0.740D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.20087751D+04,0.740D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.17339129D+04,0.740D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.74068630D+03,0.740D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.17694139D+04,0.740D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13801:  13900)=(/
     . 0.16938988D+04,0.740D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.16500004D+04,0.740D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.16097798D+04,0.740D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.15740661D+04,0.740D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.12144448D+04,0.740D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.14078390D+04,0.740D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.13533433D+04,0.740D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.14132821D+04,0.740D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.13825761D+04,0.740D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.13545779D+04,0.740D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.13397711D+04,0.740D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.11143056D+04,0.740D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.10763807D+04,0.740D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.97195240D+03,0.740D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.81436220D+03,0.740D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.47481400D+02,0.750D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.30549200D+02,0.750D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.84030290D+03,0.750D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.45686870D+03,0.750D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.29769360D+03,0.750D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  13901:  14000)=(/
     . 0.19640730D+03,0.750D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.13504060D+03,0.750D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.10106690D+03,0.750D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.75833700D+02,0.750D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.57897000D+02,0.750D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.10011833D+04,0.750D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.73565530D+03,0.750D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.66653080D+03,0.750D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.51321450D+03,0.750D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.39266040D+03,0.750D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.32200220D+03,0.750D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.26008050D+03,0.750D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.21074970D+03,0.750D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.16563829D+04,0.750D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.13172961D+04,0.750D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.10793168D+04,0.750D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.10345489D+04,0.750D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.94314230D+03,0.750D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.74238390D+03,0.750D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.80678450D+03,0.750D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14001:  14100)=(/
     . 0.63220880D+03,0.750D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.66388710D+03,0.750D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.68701430D+03,0.750D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.52648590D+03,0.750D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.53299620D+03,0.750D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.63416620D+03,0.750D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.54932810D+03,0.750D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.46115310D+03,0.750D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.40981080D+03,0.750D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.35513780D+03,0.750D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.30613060D+03,0.750D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.18506131D+04,0.750D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.15716863D+04,0.750D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.13568533D+04,0.750D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.12084737D+04,0.750D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.10954282D+04,0.750D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.83707080D+03,0.750D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.93753010D+03,0.750D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.70623580D+03,0.750D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.77195670D+03,0.750D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14101:  14200)=(/
     . 0.71318460D+03,0.750D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.59599800D+03,0.750D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.62541990D+03,0.750D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.79423010D+03,0.750D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.72367260D+03,0.750D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.63581610D+03,0.750D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.58492700D+03,0.750D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.52412750D+03,0.750D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.46726420D+03,0.750D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.22535037D+04,0.750D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.20134740D+04,0.750D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.17433791D+04,0.750D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.75313690D+03,0.750D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.17753357D+04,0.750D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.17005706D+04,0.750D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.16567768D+04,0.750D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.16166266D+04,0.750D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.15809835D+04,0.750D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.12234224D+04,0.750D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.14105904D+04,0.750D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14201:  14300)=(/
     . 0.13567115D+04,0.750D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.14207081D+04,0.750D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.13899934D+04,0.750D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.13620334D+04,0.750D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.13469951D+04,0.750D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.11225706D+04,0.750D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.10876931D+04,0.750D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.98375910D+03,0.750D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.82496150D+03,0.750D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.83633100D+03,0.750D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.43502700D+02,0.760D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.28354600D+02,0.760D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.73192340D+03,0.760D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.40716460D+03,0.760D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.26873040D+03,0.760D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.17905630D+03,0.760D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.12405100D+03,0.760D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.93359700D+02,0.760D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.70395400D+02,0.760D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.53964200D+02,0.760D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14301:  14400)=(/
     . 0.87335890D+03,0.760D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.65307500D+03,0.760D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.59556280D+03,0.760D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.46268400D+03,0.760D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.35670880D+03,0.760D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.29400210D+03,0.760D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.23864150D+03,0.760D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.19423140D+03,0.760D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.14396620D+04,0.760D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.11612618D+04,0.760D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.95445330D+03,0.760D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.91746900D+03,0.760D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.83784550D+03,0.760D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.65982040D+03,0.760D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.71850590D+03,0.760D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.56350590D+03,0.760D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.59375330D+03,0.760D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.61335130D+03,0.760D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.47025530D+03,0.760D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.47853750D+03,0.760D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14401:  14500)=(/
     . 0.56822510D+03,0.760D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.49561460D+03,0.760D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.41868920D+03,0.760D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.37355900D+03,0.760D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.32506940D+03,0.760D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.28130190D+03,0.760D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.16103754D+04,0.760D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.13848254D+04,0.760D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.12021044D+04,0.760D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.10743491D+04,0.760D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.97608420D+03,0.760D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.74894180D+03,0.760D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.83752750D+03,0.760D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.63378860D+03,0.760D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.69267540D+03,0.760D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.64089680D+03,0.760D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.53528690D+03,0.760D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.56312050D+03,0.760D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.71169730D+03,0.760D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.65211350D+03,0.760D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14501:  14600)=(/
     . 0.57610750D+03,0.760D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.53178230D+03,0.760D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.47825570D+03,0.760D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.42787240D+03,0.760D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.19610192D+04,0.760D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.17707984D+04,0.760D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.15416747D+04,0.760D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.68178680D+03,0.760D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.15643082D+04,0.760D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.14998732D+04,0.760D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.14616315D+04,0.760D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.14265281D+04,0.760D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.13953763D+04,0.760D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.10864242D+04,0.760D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.12413205D+04,0.760D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.11950783D+04,0.760D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.12556728D+04,0.760D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.12287111D+04,0.760D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.12042523D+04,0.760D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.11906528D+04,0.760D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14601:  14700)=(/
     . 0.99636440D+03,0.760D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.97080130D+03,0.760D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.88109380D+03,0.760D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.74095590D+03,0.760D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.75215710D+03,0.760D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.67852780D+03,0.760D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.40192800D+02,0.770D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.26493600D+02,0.770D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.64922250D+03,0.770D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.36768190D+03,0.770D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.24523810D+03,0.770D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.16475860D+03,0.770D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.11488740D+03,0.770D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.86879300D+02,0.770D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.65790400D+02,0.770D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.50615900D+02,0.770D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.77563370D+03,0.770D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.58794870D+03,0.770D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.53899920D+03,0.770D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.42178390D+03,0.770D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14701:  14800)=(/
     . 0.32723040D+03,0.770D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.27085510D+03,0.770D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.22077630D+03,0.770D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.18037110D+03,0.770D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.12755198D+04,0.770D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.10398993D+04,0.770D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.85678860D+03,0.770D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.82551250D+03,0.770D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.75491930D+03,0.770D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.59487280D+03,0.770D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.64870850D+03,0.770D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.50921810D+03,0.770D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.53789460D+03,0.770D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.55484790D+03,0.770D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.42567880D+03,0.770D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.43487920D+03,0.770D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.51550510D+03,0.770D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.45212560D+03,0.770D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.38392650D+03,0.770D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.34368330D+03,0.770D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14801:  14900)=(/
     . 0.30011780D+03,0.770D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.26056720D+03,0.770D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.14282493D+04,0.770D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.12397351D+04,0.770D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.10808340D+04,0.770D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.96864880D+03,0.770D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.88171010D+03,0.770D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.67887180D+03,0.770D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.75818340D+03,0.770D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.57594520D+03,0.770D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.62929090D+03,0.770D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.58296220D+03,0.770D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.48681310D+03,0.770D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.51303760D+03,0.770D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.64586290D+03,0.770D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.59440490D+03,0.770D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.52746810D+03,0.770D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.48823190D+03,0.770D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.44042470D+03,0.770D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.39518890D+03,0.770D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  14901:  15000)=(/
     . 0.17394992D+04,0.770D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.15831333D+04,0.770D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.13842225D+04,0.770D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.62381140D+03,0.770D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.14006813D+04,0.770D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.13439514D+04,0.770D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.13099454D+04,0.770D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.12786986D+04,0.770D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.12509767D+04,0.770D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.97887040D+03,0.770D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.11106969D+04,0.770D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.10701912D+04,0.770D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.11269388D+04,0.770D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.11028614D+04,0.770D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.10810811D+04,0.770D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.10686460D+04,0.770D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.89729150D+03,0.770D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.87805750D+03,0.770D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.79916860D+03,0.770D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.67370750D+03,0.770D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15001:  15100)=(/
     . 0.68458550D+03,0.770D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.61911480D+03,0.770D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.56606600D+03,0.770D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.33737400D+02,0.780D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.22655000D+02,0.780D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.52329220D+03,0.780D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.30016240D+03,0.780D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.20254080D+03,0.780D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.13757620D+03,0.780D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.96865800D+02,0.780D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.73818300D+02,0.780D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.56305300D+02,0.780D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.43593000D+02,0.780D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.62596250D+03,0.780D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.47885350D+03,0.780D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.44120140D+03,0.780D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.34788170D+03,0.780D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.27198720D+03,0.780D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.22647100D+03,0.780D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.18574640D+03,0.780D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15101:  15200)=(/
     . 0.15265670D+03,0.780D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.10299899D+04,0.780D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.84443870D+03,0.780D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.69692720D+03,0.780D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.67305270D+03,0.780D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.61630660D+03,0.780D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.48652160D+03,0.780D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.53065270D+03,0.780D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.41743140D+03,0.780D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.44140210D+03,0.780D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.45462950D+03,0.780D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.34958030D+03,0.780D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.35801260D+03,0.780D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.42342920D+03,0.780D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.37342980D+03,0.780D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.31904090D+03,0.780D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.28687030D+03,0.780D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.25174820D+03,0.780D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.21965520D+03,0.780D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.11547638D+04,0.780D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15201:  15300)=(/
     . 0.10070623D+04,0.780D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.88099930D+03,0.780D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.79155440D+03,0.780D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.72193400D+03,0.780D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.55823890D+03,0.780D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.62242220D+03,0.780D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.47507150D+03,0.780D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.51845950D+03,0.780D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.48093890D+03,0.780D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.40219150D+03,0.780D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.42403900D+03,0.780D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.53144420D+03,0.780D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.49101090D+03,0.780D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.43782090D+03,0.780D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.40661850D+03,0.780D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.36825920D+03,0.780D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.33179040D+03,0.780D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.14071724D+04,0.780D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.12854322D+04,0.780D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.11275157D+04,0.780D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15301:  15400)=(/
     . 0.51768570D+03,0.780D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.11388994D+04,0.780D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.10932462D+04,0.780D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.10657050D+04,0.780D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.10403742D+04,0.780D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.10179032D+04,0.780D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.80035340D+03,0.780D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.90395940D+03,0.780D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.87158450D+03,0.780D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.91760410D+03,0.780D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.89802720D+03,0.780D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.88036430D+03,0.780D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.87003670D+03,0.780D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.73291850D+03,0.780D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.71957520D+03,0.780D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.65682120D+03,0.780D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.55562970D+03,0.780D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.56502640D+03,0.780D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.51242690D+03,0.780D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.46966640D+03,0.780D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15401:  15500)=(/
     . 0.39114480D+03,0.780D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.31643100D+02,0.790D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.21405700D+02,0.790D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.48271200D+03,0.790D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.27848740D+03,0.790D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.18878590D+03,0.790D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.12878430D+03,0.790D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.91021000D+02,0.790D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.69577200D+02,0.790D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.53225800D+02,0.790D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.41316600D+02,0.790D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.57775500D+03,0.790D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.44384910D+03,0.790D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.40978640D+03,0.790D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.32409250D+03,0.790D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.25414960D+03,0.790D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.21210390D+03,0.790D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.17438080D+03,0.790D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.14364860D+03,0.790D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.95062180D+03,0.790D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15501:  15600)=(/
     . 0.78160440D+03,0.790D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.64557500D+03,0.790D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.62406930D+03,0.790D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.57177460D+03,0.790D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.45167990D+03,0.790D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.49272680D+03,0.790D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.38791960D+03,0.790D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.41040680D+03,0.790D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.42244780D+03,0.790D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.32511980D+03,0.790D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.33331770D+03,0.790D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.39382270D+03,0.790D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.34809220D+03,0.790D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.29810130D+03,0.790D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.26850000D+03,0.790D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.23607650D+03,0.790D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.20637560D+03,0.790D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.10663289D+04,0.790D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.93220550D+03,0.790D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.81674650D+03,0.790D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15601:  15700)=(/
     . 0.73461520D+03,0.790D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.67055170D+03,0.790D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.51941000D+03,0.790D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.57874610D+03,0.790D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.44259210D+03,0.790D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.48280890D+03,0.790D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.44812160D+03,0.790D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.37494860D+03,0.790D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.39541490D+03,0.790D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.49465360D+03,0.790D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.45775290D+03,0.790D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.40894270D+03,0.790D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.38029110D+03,0.790D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.34494080D+03,0.790D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.31127000D+03,0.790D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.12995975D+04,0.790D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.11895405D+04,0.790D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.10448996D+04,0.790D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.48348310D+03,0.790D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.10546339D+04,0.790D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15701:  15800)=(/
     . 0.10125735D+04,0.790D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.98712160D+03,0.790D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.96370210D+03,0.790D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.94292740D+03,0.790D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.74291920D+03,0.790D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.83729210D+03,0.790D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.80754210D+03,0.790D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.85029540D+03,0.790D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.83217180D+03,0.790D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.81583760D+03,0.790D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.80619030D+03,0.790D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.68007310D+03,0.790D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.66862000D+03,0.790D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.61103860D+03,0.790D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.51760830D+03,0.790D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.52653700D+03,0.790D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.47806720D+03,0.790D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.43860830D+03,0.790D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.36582470D+03,0.790D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.34235260D+03,0.790D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15801:  15900)=(/
     . 0.32647500D+02,0.800D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.22083700D+02,0.800D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.48740200D+03,0.800D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.28506170D+03,0.800D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.19416150D+03,0.800D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.13273170D+03,0.800D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.93883600D+02,0.800D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.71776000D+02,0.800D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.54902300D+02,0.800D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.42608500D+02,0.800D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.58379600D+03,0.800D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.45335960D+03,0.800D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.41983840D+03,0.800D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.33321670D+03,0.800D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.26185840D+03,0.800D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.21871850D+03,0.800D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.17991240D+03,0.800D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.14823250D+03,0.800D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.95736170D+03,0.800D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.79465850D+03,0.800D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  15901:  16000)=(/
     . 0.65759170D+03,0.800D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.63652100D+03,0.800D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.58366570D+03,0.800D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.46085090D+03,0.800D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.50355400D+03,0.800D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.39629990D+03,0.800D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.42025520D+03,0.800D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.43227410D+03,0.800D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.33243400D+03,0.800D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.34187100D+03,0.800D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.40370590D+03,0.800D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.35784550D+03,0.800D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.30703710D+03,0.800D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.27676970D+03,0.800D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.24349600D+03,0.800D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.21293560D+03,0.800D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.10744128D+04,0.800D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.94720110D+03,0.800D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.83242220D+03,0.800D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.74998240D+03,0.800D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16001:  16100)=(/
     . 0.68523850D+03,0.800D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.53145620D+03,0.800D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.59192160D+03,0.800D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.45328020D+03,0.800D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.49471860D+03,0.800D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.45942820D+03,0.800D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.38395180D+03,0.800D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.40566070D+03,0.800D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.50664510D+03,0.800D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.47008500D+03,0.800D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.42079320D+03,0.800D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.39167400D+03,0.800D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.35554970D+03,0.800D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.32102510D+03,0.800D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.13092921D+04,0.800D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.12070341D+04,0.800D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.10636276D+04,0.800D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.49715300D+03,0.800D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.10712093D+04,0.800D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.10290835D+04,0.800D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16101:  16200)=(/
     . 0.10033846D+04,0.800D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.97972320D+03,0.800D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.95873970D+03,0.800D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.75756310D+03,0.800D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.84922660D+03,0.800D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.81952610D+03,0.800D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.86529630D+03,0.800D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.84694850D+03,0.800D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.83044190D+03,0.800D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.82053560D+03,0.800D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.69354790D+03,0.800D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.68391950D+03,0.800D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.62592010D+03,0.800D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.53051690D+03,0.800D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.54005040D+03,0.800D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.49085310D+03,0.800D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.45068210D+03,0.800D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.37600960D+03,0.800D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.35193500D+03,0.800D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.36207550D+03,0.800D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16201:  16300)=(/
     . 0.45565900D+02,0.810D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.29383200D+02,0.810D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.84156200D+03,0.810D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.44509900D+03,0.810D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.28743460D+03,0.810D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.18891370D+03,0.810D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.12977590D+03,0.810D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.97188700D+02,0.810D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.73035300D+02,0.810D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.55878200D+02,0.810D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.10013499D+04,0.810D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.71984280D+03,0.810D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.64850370D+03,0.810D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.49588920D+03,0.810D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.37786060D+03,0.810D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.30940570D+03,0.810D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.24971520D+03,0.810D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.20236390D+03,0.810D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.16699732D+04,0.810D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.13012325D+04,0.810D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16301:  16400)=(/
     . 0.10621257D+04,0.810D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.10156676D+04,0.810D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.92450540D+03,0.810D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.72875260D+03,0.810D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.78917040D+03,0.810D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.61912140D+03,0.810D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.64688450D+03,0.810D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.67034360D+03,0.810D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.51478030D+03,0.810D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.51777130D+03,0.810D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.61653920D+03,0.810D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.53101480D+03,0.810D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.44413160D+03,0.810D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.39408930D+03,0.810D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.34115420D+03,0.810D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.29394640D+03,0.810D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.18646164D+04,0.810D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.15547036D+04,0.810D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.13341432D+04,0.810D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.11843558D+04,0.810D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16401:  16500)=(/
     . 0.10716858D+04,0.810D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.81713880D+03,0.810D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.91594930D+03,0.810D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.68833610D+03,0.810D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.75141170D+03,0.810D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.69349420D+03,0.810D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.58127400D+03,0.810D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.60745600D+03,0.810D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.77388330D+03,0.810D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.70133540D+03,0.810D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.61379650D+03,0.810D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.56366330D+03,0.810D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.50431850D+03,0.810D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.44915730D+03,0.810D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.22724078D+04,0.810D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.19977665D+04,0.810D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.17187930D+04,0.810D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.72802070D+03,0.810D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.17584731D+04,0.810D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.16821281D+04,0.810D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16501:  16600)=(/
     . 0.16382173D+04,0.810D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.15980083D+04,0.810D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.15622939D+04,0.810D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.12025229D+04,0.810D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.14026627D+04,0.810D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.13479037D+04,0.810D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.14013832D+04,0.810D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.13707449D+04,0.810D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.13427654D+04,0.810D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.13281907D+04,0.810D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.11030932D+04,0.810D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.10619960D+04,0.810D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.95783300D+03,0.810D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.80262850D+03,0.810D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.81249710D+03,0.810D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.72924020D+03,0.810D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.66279090D+03,0.810D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.54691720D+03,0.810D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.50961460D+03,0.810D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.52190910D+03,0.810D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16601:  16700)=(/
     . 0.79223780D+03,0.810D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.45178600D+02,0.820D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.29396300D+02,0.820D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.77049850D+03,0.820D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.42598770D+03,0.820D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.28017140D+03,0.820D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.18622350D+03,0.820D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.12882430D+03,0.820D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.96880100D+02,0.820D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.73028500D+02,0.820D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.55990900D+02,0.820D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.91912570D+03,0.820D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.68410110D+03,0.820D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.62271340D+03,0.820D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.48258810D+03,0.820D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.37129390D+03,0.820D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.30563890D+03,0.820D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.24781410D+03,0.820D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.20153300D+03,0.820D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.15164193D+04,0.820D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16701:  16800)=(/
     . 0.12187009D+04,0.820D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.10008602D+04,0.820D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.96137180D+03,0.820D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.87756550D+03,0.820D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.69108340D+03,0.820D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.75211560D+03,0.820D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.58980850D+03,0.820D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.62087440D+03,0.820D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.64167640D+03,0.820D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.49199760D+03,0.820D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.49993220D+03,0.820D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.59373850D+03,0.820D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.51689320D+03,0.820D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.43592810D+03,0.820D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.38853730D+03,0.820D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.33776830D+03,0.820D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.29204720D+03,0.820D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.16956909D+04,0.820D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.14534745D+04,0.820D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.12598691D+04,0.820D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16801:  16900)=(/
     . 0.11249617D+04,0.820D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.10214733D+04,0.820D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.78299060D+03,0.820D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.87595820D+03,0.820D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.66216750D+03,0.820D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.72370460D+03,0.820D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.66938820D+03,0.820D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.55927100D+03,0.820D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.58793680D+03,0.820D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.74387970D+03,0.820D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.68057520D+03,0.820D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.60037140D+03,0.820D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.55368260D+03,0.820D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.49748720D+03,0.820D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.44470070D+03,0.820D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.20646687D+04,0.820D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.18594085D+04,0.820D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.16164677D+04,0.820D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.71062890D+03,0.820D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.16419885D+04,0.820D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  16901:  17000)=(/
     . 0.15739763D+04,0.820D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.15337469D+04,0.820D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.14968291D+04,0.820D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.14640624D+04,0.820D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.11380979D+04,0.820D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.13034316D+04,0.820D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.12545335D+04,0.820D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.13170177D+04,0.820D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.12886907D+04,0.820D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.12629662D+04,0.820D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.12487891D+04,0.820D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.10439034D+04,0.820D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.10154887D+04,0.820D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.92083050D+03,0.820D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.77385260D+03,0.820D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.78529450D+03,0.820D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.70790090D+03,0.820D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.64554700D+03,0.820D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.53402070D+03,0.820D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.49813010D+03,0.820D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17001:  17100)=(/
     . 0.51131450D+03,0.820D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.76196480D+03,0.820D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.73881560D+03,0.820D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.42241900D+02,0.830D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.27932900D+02,0.830D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.66152780D+03,0.830D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.38103070D+03,0.830D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.25605370D+03,0.830D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.17278310D+03,0.830D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.12079930D+03,0.830D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.91493400D+02,0.830D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.69368900D+02,0.830D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.53420400D+02,0.830D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.79108680D+03,0.830D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.60760080D+03,0.830D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.55943360D+03,0.830D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.44013430D+03,0.830D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.34277880D+03,0.830D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.28429480D+03,0.830D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.23212190D+03,0.830D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17101:  17200)=(/
     . 0.18987560D+03,0.830D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.12966253D+04,0.830D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.10686331D+04,0.830D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.88251530D+03,0.830D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.85188550D+03,0.830D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.77993360D+03,0.830D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.61445090D+03,0.830D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.67130550D+03,0.830D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.52693210D+03,0.830D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.55817400D+03,0.830D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.57514870D+03,0.830D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.44105510D+03,0.830D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.45236780D+03,0.830D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.53570130D+03,0.830D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.47183760D+03,0.830D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.40199200D+03,0.830D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.36047350D+03,0.830D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.31527740D+03,0.830D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.27407550D+03,0.830D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.14529418D+04,0.830D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17201:  17300)=(/
     . 0.12732370D+04,0.830D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.11143977D+04,0.830D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.10010315D+04,0.830D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.91248200D+03,0.830D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.70407130D+03,0.830D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.78575170D+03,0.830D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.59827920D+03,0.830D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.65393370D+03,0.830D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.60631280D+03,0.830D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.50575740D+03,0.830D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.53417140D+03,0.830D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.67073540D+03,0.830D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.61958270D+03,0.830D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.55156050D+03,0.830D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.51139760D+03,0.830D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.46208340D+03,0.830D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.41520110D+03,0.830D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.17696390D+04,0.830D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.16234740D+04,0.830D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.14251223D+04,0.830D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17301:  17400)=(/
     . 0.65171260D+03,0.830D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.14382615D+04,0.830D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.13809451D+04,0.830D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.13462674D+04,0.830D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.13143784D+04,0.830D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.12860963D+04,0.830D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.10103860D+04,0.830D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.11390166D+04,0.830D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.10983110D+04,0.830D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.11597774D+04,0.830D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.11351424D+04,0.830D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.11129109D+04,0.830D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.10999410D+04,0.830D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.92612910D+03,0.830D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.90981540D+03,0.830D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.82982850D+03,0.830D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.70043950D+03,0.830D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.71241700D+03,0.830D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.64537650D+03,0.830D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.59085050D+03,0.830D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17401:  17500)=(/
     . 0.49072340D+03,0.830D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.45847260D+03,0.830D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.47152310D+03,0.830D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.68856550D+03,0.830D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.67269040D+03,0.830D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.61752960D+03,0.830D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.40710600D+02,0.840D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.27211800D+02,0.840D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.60731510D+03,0.840D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.35746030D+03,0.840D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.24330570D+03,0.840D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.16573270D+03,0.840D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.11666150D+03,0.840D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.88773600D+02,0.840D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.67567200D+02,0.840D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.52188100D+02,0.840D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.72726310D+03,0.840D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.56780720D+03,0.840D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.52626000D+03,0.840D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.41768440D+03,0.840D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17501:  17600)=(/
     . 0.32771360D+03,0.840D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.27309950D+03,0.840D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.22398500D+03,0.840D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.18392230D+03,0.840D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.11891902D+04,0.840D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.99221250D+03,0.840D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.82178250D+03,0.840D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.79553840D+03,0.840D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.72956570D+03,0.840D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.57517310D+03,0.840D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.62948100D+03,0.840D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.49459140D+03,0.840D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.52549790D+03,0.840D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.54053270D+03,0.840D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.41478690D+03,0.840D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.42746160D+03,0.840D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.50536000D+03,0.840D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.44808220D+03,0.840D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.38408530D+03,0.840D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.34572660D+03,0.840D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17601:  17700)=(/
     . 0.30354480D+03,0.840D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.26479780D+03,0.840D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.13343692D+04,0.840D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.11818604D+04,0.840D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.10398622D+04,0.840D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.93720360D+03,0.840D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.85623980D+03,0.840D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.66336080D+03,0.840D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.73919570D+03,0.840D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.56531410D+03,0.840D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.61769980D+03,0.840D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.57350990D+03,0.840D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.47823360D+03,0.840D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.50616860D+03,0.840D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.63278330D+03,0.840D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.58760970D+03,0.840D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.52587280D+03,0.840D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.48916400D+03,0.840D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.44353000D+03,0.840D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.39983350D+03,0.840D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17701:  17800)=(/
     . 0.16260419D+04,0.840D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.15047607D+04,0.840D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.13277397D+04,0.840D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.62085480D+03,0.840D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.13354309D+04,0.840D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.12832336D+04,0.840D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.12512879D+04,0.840D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.12218775D+04,0.840D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.11958046D+04,0.840D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.94515700D+03,0.840D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.10569864D+04,0.840D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.10202790D+04,0.840D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.10796889D+04,0.840D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.10568831D+04,0.840D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.10363788D+04,0.840D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.10240338D+04,0.840D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.86580300D+03,0.840D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.85504390D+03,0.840D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.78248020D+03,0.840D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.66238200D+03,0.840D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17801:  17900)=(/
     . 0.67449800D+03,0.840D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.61277720D+03,0.840D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.56231000D+03,0.840D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.46825250D+03,0.840D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.43792710D+03,0.840D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.45080740D+03,0.840D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.65074290D+03,0.840D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.63822510D+03,0.840D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.58861880D+03,0.840D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.56260110D+03,0.840D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.38046800D+02,0.850D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.25807100D+02,0.850D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.53565210D+03,0.850D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.32315400D+03,0.850D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.22339710D+03,0.850D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.15398830D+03,0.850D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10936180D+03,0.850D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.83743700D+02,0.850D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.64079700D+02,0.850D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.49704300D+02,0.850D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  17901:  18000)=(/
     . 0.64255990D+03,0.850D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.51104720D+03,0.850D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.47738160D+03,0.850D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.38289140D+03,0.850D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.30317170D+03,0.850D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.25418990D+03,0.850D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.20969510D+03,0.850D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.17306680D+03,0.850D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.10486754D+04,0.850D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.88679380D+03,0.850D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.73687680D+03,0.850D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.71582420D+03,0.850D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.65777000D+03,0.850D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.51922800D+03,0.850D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.56918500D+03,0.850D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.44794120D+03,0.850D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.47740460D+03,0.850D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.49002960D+03,0.850D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.37654500D+03,0.850D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.39005450D+03,0.850D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18001:  18100)=(/
     . 0.46013230D+03,0.850D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.41119140D+03,0.850D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.35508990D+03,0.850D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.32115720D+03,0.850D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.28336260D+03,0.850D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.24831690D+03,0.850D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.11787843D+04,0.850D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.10561781D+04,0.850D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.93492800D+03,0.850D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.84595890D+03,0.850D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.77500770D+03,0.850D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.60351860D+03,0.850D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.67120920D+03,0.850D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.51618070D+03,0.850D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.56361340D+03,0.850D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.52417160D+03,0.850D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.43716280D+03,0.850D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.46363160D+03,0.850D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.57648690D+03,0.850D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.53857130D+03,0.850D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18101:  18200)=(/
     . 0.48503660D+03,0.850D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.45297260D+03,0.850D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.41249290D+03,0.850D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.37339600D+03,0.850D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.14375237D+04,0.850D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.13427557D+04,0.850D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.11918004D+04,0.850D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.57219880D+03,0.850D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.11941552D+04,0.850D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.11484879D+04,0.850D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.11201674D+04,0.850D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.10940580D+04,0.850D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.10709219D+04,0.850D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.85263060D+03,0.850D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.94504500D+03,0.850D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.91333710D+03,0.850D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.96825870D+03,0.850D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.94792020D+03,0.850D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.92971630D+03,0.850D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.91834420D+03,0.850D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18201:  18300)=(/
     . 0.78031510D+03,0.850D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.77523370D+03,0.850D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.71228880D+03,0.850D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.60524130D+03,0.850D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.61710680D+03,0.850D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.56258800D+03,0.850D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.51772850D+03,0.850D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.43264880D+03,0.850D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.40518020D+03,0.850D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.41747370D+03,0.850D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.59434970D+03,0.850D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.58543140D+03,0.850D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.54287440D+03,0.850D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.52058720D+03,0.850D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.48365360D+03,0.850D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.35275000D+02,0.860D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.24288600D+02,0.860D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.47129940D+03,0.860D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.29040790D+03,0.860D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.20366210D+03,0.860D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18301:  18400)=(/
     . 0.14199460D+03,0.860D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.10173400D+03,0.860D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.78399300D+02,0.860D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.60320800D+02,0.860D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.46997000D+02,0.860D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.56630470D+03,0.860D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.45752070D+03,0.860D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.43040700D+03,0.860D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.34853240D+03,0.860D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.27834410D+03,0.860D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.23475630D+03,0.860D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.19477780D+03,0.860D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.16157780D+03,0.860D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.92348960D+03,0.860D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.78946510D+03,0.860D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.65782090D+03,0.860D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.64106890D+03,0.860D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.59013650D+03,0.860D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.46657860D+03,0.860D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.51200690D+03,0.860D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18401:  18500)=(/
     . 0.40371620D+03,0.860D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.43125430D+03,0.860D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.44180520D+03,0.860D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.34010290D+03,0.860D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.35375690D+03,0.860D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.41640720D+03,0.860D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.37473780D+03,0.860D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.32584680D+03,0.860D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.29605910D+03,0.860D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.26246780D+03,0.860D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.23103920D+03,0.860D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.10398788D+04,0.860D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.94037660D+03,0.860D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.83682440D+03,0.860D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.75986860D+03,0.860D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.69790750D+03,0.860D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.54617620D+03,0.860D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.60628810D+03,0.860D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.46875270D+03,0.860D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.51133700D+03,0.860D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18501:  18600)=(/
     . 0.47629130D+03,0.860D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.39751230D+03,0.860D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.42213930D+03,0.860D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.52225240D+03,0.860D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.49045740D+03,0.860D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.44423510D+03,0.860D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.41640380D+03,0.860D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.38074630D+03,0.860D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.34603400D+03,0.860D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.12691514D+04,0.860D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.11942553D+04,0.860D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.10653886D+04,0.860D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.52379990D+03,0.860D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.10641113D+04,0.860D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.10241601D+04,0.860D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.99910040D+03,0.860D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.97596710D+03,0.860D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.95547530D+03,0.860D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.76577540D+03,0.860D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.84248080D+03,0.860D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18601:  18700)=(/
     . 0.81508100D+03,0.860D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.86486000D+03,0.860D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.84676410D+03,0.860D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.83063370D+03,0.860D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.82022230D+03,0.860D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.70009390D+03,0.860D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.69908500D+03,0.860D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.64467130D+03,0.860D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.54983920D+03,0.860D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.56121370D+03,0.860D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.51327610D+03,0.860D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.47360870D+03,0.860D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.39719060D+03,0.860D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.37248250D+03,0.860D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.38402410D+03,0.860D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.53990550D+03,0.860D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.53369840D+03,0.860D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.49729720D+03,0.860D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.47831130D+03,0.860D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.44604470D+03,0.860D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18701:  18800)=(/
     . 0.41282750D+03,0.860D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.10074940D+03,0.870D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.59659400D+02,0.870D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.28945966D+04,0.870D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.12189348D+04,0.870D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.71065850D+03,0.870D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.43419670D+03,0.870D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.28277630D+03,0.870D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.20397230D+03,0.870D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.14835840D+03,0.870D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.11051200D+03,0.870D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.33986461D+04,0.870D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.20446296D+04,0.870D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.17465418D+04,0.870D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.12378269D+04,0.870D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.88882250D+03,0.870D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.70197610D+03,0.870D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.54727120D+03,0.870D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.43044680D+03,0.870D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.60209123D+04,0.870D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18801:  18900)=(/
     . 0.39885535D+04,0.870D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.31533933D+04,0.870D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.29483214D+04,0.870D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.26441348D+04,0.870D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.20951563D+04,0.870D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.22092516D+04,0.870D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.17359767D+04,0.870D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.17406013D+04,0.870D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.18302286D+04,0.870D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.14168737D+04,0.870D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.13464975D+04,0.870D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.16330869D+04,0.870D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.13219669D+04,0.870D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.10509064D+04,0.870D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.90547240D+03,0.870D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.76070860D+03,0.870D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.63792100D+03,0.870D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.66879551D+04,0.870D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.48140613D+04,0.870D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.39238630D+04,0.870D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  18901:  19000)=(/
     . 0.33777348D+04,0.870D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.30012009D+04,0.870D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.22208404D+04,0.870D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.25189457D+04,0.870D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.18286037D+04,0.870D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.19819822D+04,0.870D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.18052792D+04,0.870D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.15403020D+04,0.870D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.15550964D+04,0.870D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.20671731D+04,0.870D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.17732620D+04,0.870D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.14801255D+04,0.870D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.13236329D+04,0.870D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.11516386D+04,0.870D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.99949930D+03,0.870D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.82315280D+04,0.870D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.63397235D+04,0.870D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.51715628D+04,0.870D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.17765919D+04,0.870D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.54969601D+04,0.870D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19001:  19100)=(/
     . 0.51886826D+04,0.870D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.50376830D+04,0.870D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.49009739D+04,0.870D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.47792385D+04,0.870D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.35004650D+04,0.870D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.45180145D+04,0.870D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.43256882D+04,0.870D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.42216216D+04,0.870D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.41211013D+04,0.870D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.40270614D+04,0.870D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.39908778D+04,0.870D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.32187704D+04,0.870D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.29218119D+04,0.870D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.25563295D+04,0.870D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.21076991D+04,0.870D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.21032454D+04,0.870D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.18383882D+04,0.870D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.16363884D+04,0.870D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.13271181D+04,0.870D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.12270914D+04,0.870D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19101:  19200)=(/
     . 0.12394986D+04,0.870D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.21094753D+04,0.870D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.19333025D+04,0.870D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.16699965D+04,0.870D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.15408317D+04,0.870D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.13687363D+04,0.870D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.12134959D+04,0.870D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.73147398D+04,0.870D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.99413100D+02,0.880D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.59674000D+02,0.880D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.24340697D+04,0.880D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.11320344D+04,0.880D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.68137890D+03,0.880D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.42416130D+03,0.880D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.27936020D+03,0.880D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.20282600D+03,0.880D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.14825160D+03,0.880D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.11081260D+03,0.880D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.28745717D+04,0.880D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.18719600D+04,0.880D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19201:  19300)=(/
     . 0.16279493D+04,0.880D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.11837103D+04,0.880D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.86434290D+03,0.880D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.68837560D+03,0.880D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.54058680D+03,0.880D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.42751470D+03,0.880D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.49128581D+04,0.880D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.35289590D+04,0.880D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.28292273D+04,0.880D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.26655950D+04,0.880D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.24036878D+04,0.880D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.18947707D+04,0.880D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.20238919D+04,0.880D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.15845838D+04,0.880D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.16192221D+04,0.880D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.16942469D+04,0.880D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.13022136D+04,0.880D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.12677966D+04,0.880D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.15270149D+04,0.880D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.12635730D+04,0.880D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19301:  19400)=(/
     . 0.10197251D+04,0.880D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.88504330D+03,0.880D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.74865020D+03,0.880D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.63127570D+03,0.880D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.54594039D+04,0.880D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.42339736D+04,0.880D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.35246180D+04,0.880D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.30700206D+04,0.880D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.27442675D+04,0.880D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.20488350D+04,0.880D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.23149349D+04,0.880D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.16984672D+04,0.880D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.18511077D+04,0.880D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.16939362D+04,0.880D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.14296950D+04,0.880D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.14674624D+04,0.880D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.19210215D+04,0.880D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.16832537D+04,0.880D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.14267831D+04,0.880D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.12854608D+04,0.880D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19401:  19500)=(/
     . 0.11265319D+04,0.880D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.98363370D+03,0.880D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.66646049D+04,0.880D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.55052582D+04,0.880D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.45943095D+04,0.880D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.17037104D+04,0.880D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.47976061D+04,0.880D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.45623865D+04,0.880D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.44362592D+04,0.880D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.43214914D+04,0.880D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.42193690D+04,0.880D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.31447184D+04,0.880D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.38693475D+04,0.880D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.37006978D+04,0.880D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.37536534D+04,0.880D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.36679746D+04,0.880D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.35884351D+04,0.880D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.35540067D+04,0.880D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.28892983D+04,0.880D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.26921229D+04,0.880D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19501:  19600)=(/
     . 0.23813318D+04,0.880D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.19684730D+04,0.880D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.19760672D+04,0.880D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.17428522D+04,0.880D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.15616887D+04,0.880D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.12704258D+04,0.880D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.11766222D+04,0.880D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.11957573D+04,0.880D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.19547292D+04,0.880D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.18286515D+04,0.880D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.16046349D+04,0.880D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.14914089D+04,0.880D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.13351632D+04,0.880D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.11909286D+04,0.880D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.60457168D+04,0.880D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.53054399D+04,0.880D+02,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.89406000D+02,0.890D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.54461300D+02,0.890D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.20082591D+04,0.890D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.97721480D+03,0.890D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19601:  19700)=(/
     . 0.60005330D+03,0.890D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.37866720D+03,0.890D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.25180860D+03,0.890D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.18403180D+03,0.890D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.13527920D+03,0.890D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.10158570D+03,0.890D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.23774617D+04,0.890D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.16040300D+04,0.890D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.14098711D+04,0.890D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.10405681D+04,0.890D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.76851320D+03,0.890D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.61613190D+03,0.890D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.48688350D+03,0.890D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.38707200D+03,0.890D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.40174782D+04,0.890D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.29771478D+04,0.890D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.24021540D+04,0.890D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.22732739D+04,0.890D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.20559589D+04,0.890D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.16183634D+04,0.890D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19701:  19800)=(/
     . 0.17384495D+04,0.890D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.13600663D+04,0.890D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.14017622D+04,0.890D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.14625357D+04,0.890D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.11217466D+04,0.890D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.11049523D+04,0.890D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.13263601D+04,0.890D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.11110958D+04,0.890D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.90553420D+03,0.890D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.79026450D+03,0.890D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.67216580D+03,0.890D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.56954790D+03,0.890D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.44690053D+04,0.890D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.35643510D+04,0.890D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.29977722D+04,0.890D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.26270446D+04,0.890D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.23567083D+04,0.890D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.17692538D+04,0.890D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.19948729D+04,0.890D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.14730457D+04,0.890D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19801:  19900)=(/
     . 0.16081722D+04,0.890D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.14753924D+04,0.890D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.12402274D+04,0.890D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.12822835D+04,0.890D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.16650502D+04,0.890D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.14751121D+04,0.890D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.12620917D+04,0.890D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.11429090D+04,0.890D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.10069141D+04,0.890D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.88341400D+03,0.890D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.54478929D+04,0.890D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.46120247D+04,0.890D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.38897052D+04,0.890D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.15031569D+04,0.890D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.40323992D+04,0.890D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.38433536D+04,0.890D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.37393646D+04,0.890D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.36445469D+04,0.890D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.35602278D+04,0.890D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.26796802D+04,0.890D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  19901:  20000)=(/
     . 0.32333681D+04,0.890D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.30968078D+04,0.890D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.31769406D+04,0.890D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.31056847D+04,0.890D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.30398461D+04,0.890D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.30095999D+04,0.890D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.24623774D+04,0.890D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.23202277D+04,0.890D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.20645670D+04,0.890D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.17115570D+04,0.890D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.17232231D+04,0.890D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.15276535D+04,0.890D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.13743640D+04,0.890D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.11213924D+04,0.890D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.10400238D+04,0.890D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.10600607D+04,0.890D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.16946605D+04,0.890D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.16008616D+04,0.890D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.14174346D+04,0.890D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.13236360D+04,0.890D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20001:  20100)=(/
     . 0.11913272D+04,0.890D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.10675065D+04,0.890D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.49814961D+04,0.890D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.44669584D+04,0.890D+02,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.37996565D+04,0.890D+02,0.890D+02,0.00000000D+00,0.00000000D+00
     .,0.82083600D+02,0.900D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.51110300D+02,0.900D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.16724768D+04,0.900D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.85252490D+03,0.900D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.53608580D+03,0.900D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.34435760D+03,0.900D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.23207230D+03,0.900D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.17124180D+03,0.900D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12693930D+03,0.900D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.95984500D+02,0.900D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.19852103D+04,0.900D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.13883756D+04,0.900D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.12352681D+04,0.900D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.92751070D+03,0.900D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.69479830D+03,0.900D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20101:  20200)=(/
     . 0.56206360D+03,0.900D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.44804860D+03,0.900D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.35892730D+03,0.900D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.33208171D+04,0.900D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.25377308D+04,0.900D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.20608901D+04,0.900D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.19602827D+04,0.900D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.17786546D+04,0.900D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.13994778D+04,0.900D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.15110816D+04,0.900D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.11825093D+04,0.900D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.12287380D+04,0.900D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.12777896D+04,0.900D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.97919840D+03,0.900D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.97588560D+03,0.900D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.11670583D+04,0.900D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.99120150D+03,0.900D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.81755520D+03,0.900D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.71866790D+03,0.900D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.61585050D+03,0.900D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20201:  20300)=(/
     . 0.52542400D+03,0.900D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.36996637D+04,0.900D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.30331273D+04,0.900D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.25787074D+04,0.900D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.22748325D+04,0.900D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.20492299D+04,0.900D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.15491780D+04,0.900D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.17421202D+04,0.900D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.12966551D+04,0.900D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.14168423D+04,0.900D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.13036060D+04,0.900D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.10925939D+04,0.900D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.11371745D+04,0.900D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.14631580D+04,0.900D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.13115880D+04,0.900D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.11344385D+04,0.900D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.10338769D+04,0.900D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.91709200D+03,0.900D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.80980180D+03,0.900D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.45055681D+04,0.900D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20301:  20400)=(/
     . 0.39071262D+04,0.900D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.33315473D+04,0.900D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.13478223D+04,0.900D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.34277894D+04,0.900D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.32744792D+04,0.900D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.31877428D+04,0.900D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.31084769D+04,0.900D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.30380345D+04,0.900D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.23122086D+04,0.900D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.27358113D+04,0.900D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.26241922D+04,0.900D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.27191319D+04,0.900D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.26591268D+04,0.900D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.26040009D+04,0.900D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.25769767D+04,0.900D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.21236819D+04,0.900D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.20248239D+04,0.900D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.18137564D+04,0.900D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.15101361D+04,0.900D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.15249297D+04,0.900D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20401:  20500)=(/
     . 0.13598576D+04,0.900D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.12292164D+04,0.900D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.10074225D+04,0.900D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.93609580D+03,0.900D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.95680470D+03,0.900D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.14915698D+04,0.900D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.14229431D+04,0.900D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.12726601D+04,0.900D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.11951492D+04,0.900D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.10828719D+04,0.900D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.97614170D+03,0.900D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.41555932D+04,0.900D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.38047750D+04,0.900D+02,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.32704258D+04,0.900D+02,0.890D+02,0.00000000D+00,0.00000000D+00
     .,0.28472704D+04,0.900D+02,0.900D+02,0.00000000D+00,0.00000000D+00
     .,0.80726000D+02,0.910D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.50005200D+02,0.910D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.17306038D+04,0.910D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.85763790D+03,0.910D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.53308180D+03,0.910D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20501:  20600)=(/
     . 0.33998630D+03,0.910D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.22813330D+03,0.910D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.16791870D+03,0.910D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12426420D+03,0.910D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.93866300D+02,0.910D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.20511151D+04,0.910D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.14030924D+04,0.910D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.12403071D+04,0.910D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.92324960D+03,0.910D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.68729140D+03,0.910D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.55412740D+03,0.910D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.44044180D+03,0.910D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.35207430D+03,0.910D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.34570434D+04,0.910D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.25892888D+04,0.910D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.20945389D+04,0.910D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.19869265D+04,0.910D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.17996577D+04,0.910D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.14173421D+04,0.910D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.15251022D+04,0.910D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20601:  20700)=(/
     . 0.11941778D+04,0.910D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.12344914D+04,0.910D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.12859300D+04,0.910D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.98684670D+03,0.910D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.97668650D+03,0.910D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.11699494D+04,0.910D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.98664000D+03,0.910D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.80937540D+03,0.910D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.70943800D+03,0.910D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.60630260D+03,0.910D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.51614250D+03,0.910D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.38488310D+04,0.910D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.30987868D+04,0.910D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.26179361D+04,0.910D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.23009943D+04,0.910D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.20684111D+04,0.910D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.15587481D+04,0.910D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.17549706D+04,0.910D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.13015598D+04,0.910D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.14207211D+04,0.910D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20701:  20800)=(/
     . 0.13053320D+04,0.910D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.10968175D+04,0.910D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.11367042D+04,0.910D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.14690715D+04,0.910D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.13085146D+04,0.910D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.11258498D+04,0.910D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.10231942D+04,0.910D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.90511410D+03,0.910D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.79733130D+03,0.910D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.46916964D+04,0.910D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.40037089D+04,0.910D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.33915606D+04,0.910D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.13396248D+04,0.910D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.35059336D+04,0.910D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.33442162D+04,0.910D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.32544268D+04,0.910D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.31724783D+04,0.910D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.30996207D+04,0.910D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.23450320D+04,0.910D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.28084101D+04,0.910D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20801:  20900)=(/
     . 0.26917985D+04,0.910D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.27690943D+04,0.910D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.27073188D+04,0.910D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.26503867D+04,0.910D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.26234579D+04,0.910D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.21538516D+04,0.910D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.20393959D+04,0.910D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.18205331D+04,0.910D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.15133881D+04,0.910D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.15256409D+04,0.910D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.13566458D+04,0.910D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.12236676D+04,0.910D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.10014754D+04,0.910D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.92999430D+03,0.910D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.94896980D+03,0.910D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.14976018D+04,0.910D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.14206996D+04,0.910D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.12642423D+04,0.910D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.11841627D+04,0.910D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.10698651D+04,0.910D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  20901:  21000)=(/
     . 0.96217680D+03,0.910D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.43039645D+04,0.910D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.38865716D+04,0.910D+02,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.33200764D+04,0.910D+02,0.890D+02,0.00000000D+00,0.00000000D+00
     .,0.28725136D+04,0.910D+02,0.900D+02,0.00000000D+00,0.00000000D+00
     .,0.29089206D+04,0.910D+02,0.910D+02,0.00000000D+00,0.00000000D+00
     .,0.78203400D+02,0.920D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.48498000D+02,0.920D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.16716156D+04,0.920D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.82971320D+03,0.920D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.51601950D+03,0.920D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.32928020D+03,0.920D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.22106590D+03,0.920D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.16279260D+03,0.920D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12052900D+03,0.920D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.91086900D+02,0.920D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.19814675D+04,0.920D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.13571628D+04,0.920D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.12000275D+04,0.920D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.89364440D+03,0.920D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21001:  21100)=(/
     . 0.66549650D+03,0.920D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.53670890D+03,0.920D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.42673540D+03,0.920D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.34123400D+03,0.920D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.33376961D+04,0.920D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.25033576D+04,0.920D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.20254740D+04,0.920D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.19216819D+04,0.920D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.17407294D+04,0.920D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.13709264D+04,0.920D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.14753669D+04,0.920D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.11552737D+04,0.920D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.11945389D+04,0.920D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.12442037D+04,0.920D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.95483090D+03,0.920D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.94527750D+03,0.920D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.11321107D+04,0.920D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.95506070D+03,0.920D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.78370690D+03,0.920D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.68708390D+03,0.920D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21101:  21200)=(/
     . 0.58734500D+03,0.920D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.50013680D+03,0.920D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.37160678D+04,0.920D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.29957186D+04,0.920D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.25317373D+04,0.920D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.22256807D+04,0.920D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.20009439D+04,0.920D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.15082494D+04,0.920D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.16979484D+04,0.920D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.12596142D+04,0.920D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.13749665D+04,0.920D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.12634136D+04,0.920D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.10615457D+04,0.920D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.11003399D+04,0.920D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.14216099D+04,0.920D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.12666329D+04,0.920D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.10901053D+04,0.920D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.99086960D+03,0.920D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.87669020D+03,0.920D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.77245190D+03,0.920D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21201:  21300)=(/
     . 0.45291727D+04,0.920D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.38697611D+04,0.920D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.32793360D+04,0.920D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.12970461D+04,0.920D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.33890395D+04,0.920D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.32330975D+04,0.920D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.31463690D+04,0.920D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.30672053D+04,0.920D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.29968237D+04,0.920D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.22680138D+04,0.920D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.27140296D+04,0.920D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.26013226D+04,0.920D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.26775672D+04,0.920D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.26178718D+04,0.920D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.25628645D+04,0.920D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.25367915D+04,0.920D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.20830385D+04,0.920D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.19730821D+04,0.920D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.17616844D+04,0.920D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.14646529D+04,0.920D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21301:  21400)=(/
     . 0.14766337D+04,0.920D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.13133066D+04,0.920D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.11847530D+04,0.920D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.96980220D+03,0.920D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.90065620D+03,0.920D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.91908240D+03,0.920D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.14493061D+04,0.920D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.13752752D+04,0.920D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.12241322D+04,0.920D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.11467509D+04,0.920D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.10362496D+04,0.920D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.93211210D+03,0.920D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.41565121D+04,0.920D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.37572430D+04,0.920D+02,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.32107560D+04,0.920D+02,0.890D+02,0.00000000D+00,0.00000000D+00
     .,0.27789432D+04,0.920D+02,0.900D+02,0.00000000D+00,0.00000000D+00
     .,0.28136269D+04,0.920D+02,0.910D+02,0.00000000D+00,0.00000000D+00
     .,0.27215209D+04,0.920D+02,0.920D+02,0.00000000D+00,0.00000000D+00
     .,0.79439100D+02,0.930D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.48724100D+02,0.930D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21401:  21500)=(/
     . 0.18005066D+04,0.930D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.86956410D+03,0.930D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.53289880D+03,0.930D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.33646270D+03,0.930D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.22420940D+03,0.930D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.16429590D+03,0.930D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.12116670D+03,0.930D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.91309000D+02,0.930D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.21313007D+04,0.930D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.14294308D+04,0.930D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.12543604D+04,0.930D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.92419700D+03,0.930D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.68227500D+03,0.930D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.54731890D+03,0.930D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.43301050D+03,0.930D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.34483050D+03,0.930D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.36085602D+04,0.930D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.26611178D+04,0.930D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.21449232D+04,0.930D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.20287846D+04,0.930D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21501:  21600)=(/
     . 0.18341800D+04,0.930D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.14450682D+04,0.930D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.15501899D+04,0.930D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.12138708D+04,0.930D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.12487909D+04,0.930D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.13033334D+04,0.930D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.10009549D+04,0.930D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.98372150D+03,0.930D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.11804499D+04,0.930D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.98737510D+03,0.930D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.80425620D+03,0.930D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.70203110D+03,0.930D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.59750350D+03,0.930D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.50682160D+03,0.930D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.40138973D+04,0.930D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.31875734D+04,0.930D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.26766472D+04,0.930D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.23437586D+04,0.930D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.21018417D+04,0.930D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.15778736D+04,0.930D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21601:  21700)=(/
     . 0.17790323D+04,0.930D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.13137158D+04,0.930D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.14331563D+04,0.930D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.13146561D+04,0.930D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.11068617D+04,0.930D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.11425196D+04,0.930D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.14840150D+04,0.930D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.13126653D+04,0.930D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.11221440D+04,0.930D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.10160025D+04,0.930D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.89523210D+03,0.930D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.78580380D+03,0.930D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.48930436D+04,0.930D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.41279330D+04,0.930D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.34757124D+04,0.930D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.13372850D+04,0.930D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.36078825D+04,0.930D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.34378544D+04,0.930D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.33445279D+04,0.930D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.32594431D+04,0.930D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21701:  21800)=(/
     . 0.31837627D+04,0.930D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.23935584D+04,0.930D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.28964080D+04,0.930D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.27730351D+04,0.930D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.28396469D+04,0.930D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.27757424D+04,0.930D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.27166522D+04,0.930D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.26897033D+04,0.930D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.21986632D+04,0.930D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.20680029D+04,0.930D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.18390464D+04,0.930D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.15251115D+04,0.930D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.15347972D+04,0.930D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.13602378D+04,0.930D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.12236448D+04,0.930D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.99920960D+03,0.930D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.92702850D+03,0.930D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.94428620D+03,0.930D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.15116714D+04,0.930D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.14258266D+04,0.930D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21801:  21900)=(/
     . 0.12612415D+04,0.930D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.11774038D+04,0.930D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.10596377D+04,0.930D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.94975540D+03,0.930D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.44697774D+04,0.930D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.39952606D+04,0.930D+02,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.33932601D+04,0.930D+02,0.890D+02,0.00000000D+00,0.00000000D+00
     .,0.29170992D+04,0.930D+02,0.900D+02,0.00000000D+00,0.00000000D+00
     .,0.29646668D+04,0.930D+02,0.910D+02,0.00000000D+00,0.00000000D+00
     .,0.28671025D+04,0.930D+02,0.920D+02,0.00000000D+00,0.00000000D+00
     .,0.30329760D+04,0.930D+02,0.930D+02,0.00000000D+00,0.00000000D+00
     .,0.77102100D+02,0.940D+02,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.47379800D+02,0.940D+02,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.17278140D+04,0.940D+02,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.83944780D+03,0.940D+02,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.51579770D+03,0.940D+02,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.32625000D+03,0.940D+02,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.21767650D+03,0.940D+02,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.15964350D+03,0.940D+02,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.11781930D+03,0.940D+02,0.900D+01,0.00000000D+00,0.00000000D+00
     ./)
      pars(  21901:  22000)=(/
     . 0.88836500D+02,0.940D+02,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.20459284D+04,0.940D+02,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.13785876D+04,0.940D+02,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.12114443D+04,0.940D+02,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.89432550D+03,0.940D+02,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.66120920D+03,0.940D+02,0.150D+02,0.00000000D+00,0.00000000D+00
     .,0.53088280D+03,0.940D+02,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.42034980D+03,0.940D+02,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.33497590D+03,0.940D+02,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.34586883D+04,0.940D+02,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.25612750D+04,0.940D+02,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.20661768D+04,0.940D+02,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.19554505D+04,0.940D+02,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.17685649D+04,0.940D+02,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.13931385D+04,0.940D+02,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.14955657D+04,0.940D+02,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.11710013D+04,0.940D+02,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.12060158D+04,0.940D+02,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.12582170D+04,0.940D+02,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.96605990D+03,0.940D+02,0.290D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22001:  22100)=(/
     . 0.95086030D+03,0.940D+02,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.11405197D+04,0.940D+02,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.95550390D+03,0.940D+02,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.77929760D+03,0.940D+02,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.68073550D+03,0.940D+02,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.57979560D+03,0.940D+02,0.350D+02,0.00000000D+00,0.00000000D+00
     .,0.49211280D+03,0.940D+02,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.38477324D+04,0.940D+02,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.30671463D+04,0.940D+02,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.25790022D+04,0.940D+02,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.22600639D+04,0.940D+02,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.20277440D+04,0.940D+02,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.15233742D+04,0.940D+02,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.17171041D+04,0.940D+02,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.12690647D+04,0.940D+02,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.13847315D+04,0.940D+02,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.12706578D+04,0.940D+02,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.10692791D+04,0.940D+02,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.11047467D+04,0.940D+02,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.14334370D+04,0.940D+02,0.490D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22101:  22200)=(/
     . 0.12697368D+04,0.940D+02,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.10867671D+04,0.940D+02,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.98462910D+03,0.940D+02,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.86818680D+03,0.940D+02,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.76254070D+03,0.940D+02,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.46894909D+04,0.940D+02,0.550D+02,0.00000000D+00,0.00000000D+00
     .,0.39694488D+04,0.940D+02,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.33469278D+04,0.940D+02,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.12946997D+04,0.940D+02,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.34708103D+04,0.940D+02,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.33082782D+04,0.940D+02,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.32187253D+04,0.940D+02,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.31370576D+04,0.940D+02,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.30644224D+04,0.940D+02,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.23068366D+04,0.940D+02,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.27842586D+04,0.940D+02,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.26660816D+04,0.940D+02,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.27342974D+04,0.940D+02,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.26729042D+04,0.940D+02,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.26161724D+04,0.940D+02,0.690D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22201:  22300)=(/
     . 0.25900957D+04,0.940D+02,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.21189811D+04,0.940D+02,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.19960266D+04,0.940D+02,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.17764187D+04,0.940D+02,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.14737544D+04,0.940D+02,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.14836747D+04,0.940D+02,0.750D+02,0.00000000D+00,0.00000000D+00
     .,0.13158080D+04,0.940D+02,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.11842955D+04,0.940D+02,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.96746430D+03,0.940D+02,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.89773860D+03,0.940D+02,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.91479120D+03,0.940D+02,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.14602247D+04,0.940D+02,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.13790263D+04,0.940D+02,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.12212647D+04,0.940D+02,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.11407827D+04,0.940D+02,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.10273948D+04,0.940D+02,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.92140470D+03,0.940D+02,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.42885858D+04,0.940D+02,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.38444325D+04,0.940D+02,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.32695288D+04,0.940D+02,0.890D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22301:  22400)=(/
     . 0.28146033D+04,0.940D+02,0.900D+02,0.00000000D+00,0.00000000D+00
     .,0.28581875D+04,0.940D+02,0.910D+02,0.00000000D+00,0.00000000D+00
     .,0.27642585D+04,0.940D+02,0.920D+02,0.00000000D+00,0.00000000D+00
     .,0.29218223D+04,0.940D+02,0.930D+02,0.00000000D+00,0.00000000D+00
     .,0.28152366D+04,0.940D+02,0.940D+02,0.00000000D+00,0.00000000D+00
     .,0.47379000D+01,0.101D+03,0.100D+01,0.00000000D+00,0.91180000D+00
     .,0.31287000D+01,0.101D+03,0.200D+01,0.00000000D+00,0.00000000D+00
     .,0.68939100D+02,0.101D+03,0.300D+01,0.00000000D+00,0.00000000D+00
     .,0.41307800D+02,0.101D+03,0.400D+01,0.00000000D+00,0.00000000D+00
     .,0.28276700D+02,0.101D+03,0.500D+01,0.00000000D+00,0.00000000D+00
     .,0.19265300D+02,0.101D+03,0.600D+01,0.00000000D+00,0.00000000D+00
     .,0.13516400D+02,0.101D+03,0.700D+01,0.00000000D+00,0.00000000D+00
     .,0.10237100D+02,0.101D+03,0.800D+01,0.00000000D+00,0.00000000D+00
     .,0.77441000D+01,0.101D+03,0.900D+01,0.00000000D+00,0.00000000D+00
     .,0.59403000D+01,0.101D+03,0.100D+02,0.00000000D+00,0.00000000D+00
     .,0.82564100D+02,0.101D+03,0.110D+02,0.00000000D+00,0.00000000D+00
     .,0.65370300D+02,0.101D+03,0.120D+02,0.00000000D+00,0.00000000D+00
     .,0.60855800D+02,0.101D+03,0.130D+02,0.00000000D+00,0.00000000D+00
     .,0.48518200D+02,0.101D+03,0.140D+02,0.00000000D+00,0.00000000D+00
     .,0.38139700D+02,0.101D+03,0.150D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22401:  22500)=(/
     . 0.31771300D+02,0.101D+03,0.160D+02,0.00000000D+00,0.00000000D+00
     .,0.26017000D+02,0.101D+03,0.170D+02,0.00000000D+00,0.00000000D+00
     .,0.21305500D+02,0.101D+03,0.180D+02,0.00000000D+00,0.00000000D+00
     .,0.13444410D+03,0.101D+03,0.190D+02,0.00000000D+00,0.00000000D+00
     .,0.11348570D+03,0.101D+03,0.200D+02,0.00000000D+00,0.00000000D+00
     .,0.94208300D+02,0.101D+03,0.210D+02,0.00000000D+00,0.00000000D+00
     .,0.91333400D+02,0.101D+03,0.220D+02,0.00000000D+00,0.00000000D+00
     .,0.83831300D+02,0.101D+03,0.230D+02,0.00000000D+00,0.00000000D+00
     .,0.65983700D+02,0.101D+03,0.240D+02,0.00000000D+00,0.00000000D+00
     .,0.72411100D+02,0.101D+03,0.250D+02,0.00000000D+00,0.00000000D+00
     .,0.56799300D+02,0.101D+03,0.260D+02,0.00000000D+00,0.00000000D+00
     .,0.60574600D+02,0.101D+03,0.270D+02,0.00000000D+00,0.00000000D+00
     .,0.62253200D+02,0.101D+03,0.280D+02,0.00000000D+00,0.00000000D+00
     .,0.47649900D+02,0.101D+03,0.290D+02,0.00000000D+00,0.00000000D+00
     .,0.49346900D+02,0.101D+03,0.300D+02,0.00000000D+00,0.00000000D+00
     .,0.58420700D+02,0.101D+03,0.310D+02,0.00000000D+00,0.00000000D+00
     .,0.51990500D+02,0.101D+03,0.320D+02,0.00000000D+00,0.00000000D+00
     .,0.44651200D+02,0.101D+03,0.330D+02,0.00000000D+00,0.00000000D+00
     .,0.40201000D+02,0.101D+03,0.340D+02,0.00000000D+00,0.00000000D+00
     .,0.35274000D+02,0.101D+03,0.350D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22501:  22600)=(/
     . 0.30726100D+02,0.101D+03,0.360D+02,0.00000000D+00,0.00000000D+00
     .,0.15093400D+03,0.101D+03,0.370D+02,0.00000000D+00,0.00000000D+00
     .,0.13506640D+03,0.101D+03,0.380D+02,0.00000000D+00,0.00000000D+00
     .,0.11929610D+03,0.101D+03,0.390D+02,0.00000000D+00,0.00000000D+00
     .,0.10773060D+03,0.101D+03,0.400D+02,0.00000000D+00,0.00000000D+00
     .,0.98518300D+02,0.101D+03,0.410D+02,0.00000000D+00,0.00000000D+00
     .,0.76366100D+02,0.101D+03,0.420D+02,0.00000000D+00,0.00000000D+00
     .,0.85080000D+02,0.101D+03,0.430D+02,0.00000000D+00,0.00000000D+00
     .,0.65086300D+02,0.101D+03,0.440D+02,0.00000000D+00,0.00000000D+00
     .,0.71198500D+02,0.101D+03,0.450D+02,0.00000000D+00,0.00000000D+00
     .,0.66121100D+02,0.101D+03,0.460D+02,0.00000000D+00,0.00000000D+00
     .,0.54965800D+02,0.101D+03,0.470D+02,0.00000000D+00,0.00000000D+00
     .,0.58356800D+02,0.101D+03,0.480D+02,0.00000000D+00,0.00000000D+00
     .,0.72907600D+02,0.101D+03,0.490D+02,0.00000000D+00,0.00000000D+00
     .,0.67943000D+02,0.101D+03,0.500D+02,0.00000000D+00,0.00000000D+00
     .,0.60947000D+02,0.101D+03,0.510D+02,0.00000000D+00,0.00000000D+00
     .,0.56742700D+02,0.101D+03,0.520D+02,0.00000000D+00,0.00000000D+00
     .,0.51465600D+02,0.101D+03,0.530D+02,0.00000000D+00,0.00000000D+00
     .,0.46379400D+02,0.101D+03,0.540D+02,0.00000000D+00,0.00000000D+00
     .,0.18402230D+03,0.101D+03,0.550D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22601:  22700)=(/
     . 0.17170330D+03,0.101D+03,0.560D+02,0.00000000D+00,0.00000000D+00
     .,0.15211060D+03,0.101D+03,0.570D+02,0.00000000D+00,0.00000000D+00
     .,0.71883000D+02,0.101D+03,0.580D+02,0.00000000D+00,0.27991000D+01
     .,0.15245020D+03,0.101D+03,0.590D+02,0.00000000D+00,0.00000000D+00
     .,0.14658510D+03,0.101D+03,0.600D+02,0.00000000D+00,0.00000000D+00
     .,0.14296200D+03,0.101D+03,0.610D+02,0.00000000D+00,0.00000000D+00
     .,0.13962540D+03,0.101D+03,0.620D+02,0.00000000D+00,0.00000000D+00
     .,0.13666950D+03,0.101D+03,0.630D+02,0.00000000D+00,0.00000000D+00
     .,0.10836620D+03,0.101D+03,0.640D+02,0.00000000D+00,0.00000000D+00
     .,0.12040560D+03,0.101D+03,0.650D+02,0.00000000D+00,0.00000000D+00
     .,0.11631780D+03,0.101D+03,0.660D+02,0.00000000D+00,0.00000000D+00
     .,0.12352200D+03,0.101D+03,0.670D+02,0.00000000D+00,0.00000000D+00
     .,0.12093020D+03,0.101D+03,0.680D+02,0.00000000D+00,0.00000000D+00
     .,0.11860700D+03,0.101D+03,0.690D+02,0.00000000D+00,0.00000000D+00
     .,0.11718110D+03,0.101D+03,0.700D+02,0.00000000D+00,0.00000000D+00
     .,0.99298400D+02,0.101D+03,0.710D+02,0.00000000D+00,0.00000000D+00
     .,0.98513600D+02,0.101D+03,0.720D+02,0.00000000D+00,0.00000000D+00
     .,0.90285800D+02,0.101D+03,0.730D+02,0.00000000D+00,0.00000000D+00
     .,0.76415000D+02,0.101D+03,0.740D+02,0.00000000D+00,0.00000000D+00
     .,0.77879300D+02,0.101D+03,0.750D+02,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22701:  22800)=(/
     . 0.70802400D+02,0.101D+03,0.760D+02,0.00000000D+00,0.00000000D+00
     .,0.64989000D+02,0.101D+03,0.770D+02,0.00000000D+00,0.00000000D+00
     .,0.54045100D+02,0.101D+03,0.780D+02,0.00000000D+00,0.00000000D+00
     .,0.50508600D+02,0.101D+03,0.790D+02,0.00000000D+00,0.00000000D+00
     .,0.52054700D+02,0.101D+03,0.800D+02,0.00000000D+00,0.00000000D+00
     .,0.74881700D+02,0.101D+03,0.810D+02,0.00000000D+00,0.00000000D+00
     .,0.73669200D+02,0.101D+03,0.820D+02,0.00000000D+00,0.00000000D+00
     .,0.68103100D+02,0.101D+03,0.830D+02,0.00000000D+00,0.00000000D+00
     .,0.65160500D+02,0.101D+03,0.840D+02,0.00000000D+00,0.00000000D+00
     .,0.60333800D+02,0.101D+03,0.850D+02,0.00000000D+00,0.00000000D+00
     .,0.55434500D+02,0.101D+03,0.860D+02,0.00000000D+00,0.00000000D+00
     .,0.17492120D+03,0.101D+03,0.870D+02,0.00000000D+00,0.00000000D+00
     .,0.17050820D+03,0.101D+03,0.880D+02,0.00000000D+00,0.00000000D+00
     .,0.15181390D+03,0.101D+03,0.890D+02,0.00000000D+00,0.00000000D+00
     .,0.13747240D+03,0.101D+03,0.900D+02,0.00000000D+00,0.00000000D+00
     .,0.13587010D+03,0.101D+03,0.910D+02,0.00000000D+00,0.00000000D+00
     .,0.13156440D+03,0.101D+03,0.920D+02,0.00000000D+00,0.00000000D+00
     .,0.13473040D+03,0.101D+03,0.930D+02,0.00000000D+00,0.00000000D+00
     .,0.13059460D+03,0.101D+03,0.940D+02,0.00000000D+00,0.00000000D+00
     .,0.75916000D+01,0.101D+03,0.101D+03,0.00000000D+00,0.00000000D+00
     ./)
      pars(  22801:  22900)=(/
     . 0.14316500D+02,0.103D+03,0.100D+01,0.98650000D+00,0.91180000D+00
     .,0.87773000D+01,0.103D+03,0.200D+01,0.98650000D+00,0.00000000D+00
     .,0.28221060D+03,0.103D+03,0.300D+01,0.98650000D+00,0.00000000D+00
     .,0.14813230D+03,0.103D+03,0.400D+01,0.98650000D+00,0.00000000D+00
     .,0.93672900D+02,0.103D+03,0.500D+01,0.98650000D+00,0.00000000D+00
     .,0.60086000D+02,0.103D+03,0.600D+01,0.98650000D+00,0.00000000D+00
     .,0.40294400D+02,0.103D+03,0.700D+01,0.98650000D+00,0.00000000D+00
     .,0.29557500D+02,0.103D+03,0.800D+01,0.98650000D+00,0.00000000D+00
     .,0.21760500D+02,0.103D+03,0.900D+01,0.98650000D+00,0.00000000D+00
     .,0.16338800D+02,0.103D+03,0.100D+02,0.98650000D+00,0.00000000D+00
     .,0.33534500D+03,0.103D+03,0.110D+02,0.98650000D+00,0.00000000D+00
     .,0.24016270D+03,0.103D+03,0.120D+02,0.98650000D+00,0.00000000D+00
     .,0.21472650D+03,0.103D+03,0.130D+02,0.98650000D+00,0.00000000D+00
     .,0.16205080D+03,0.103D+03,0.140D+02,0.98650000D+00,0.00000000D+00
     .,0.12150670D+03,0.103D+03,0.150D+02,0.98650000D+00,0.00000000D+00
     .,0.98124900D+02,0.103D+03,0.160D+02,0.98650000D+00,0.00000000D+00
     .,0.77986500D+02,0.103D+03,0.170D+02,0.98650000D+00,0.00000000D+00
     .,0.62220500D+02,0.103D+03,0.180D+02,0.98650000D+00,0.00000000D+00
     .,0.55515160D+03,0.103D+03,0.190D+02,0.98650000D+00,0.00000000D+00
     .,0.43415760D+03,0.103D+03,0.200D+02,0.98650000D+00,0.00000000D+00
     ./)
      pars(  22901:  23000)=(/
     . 0.35403170D+03,0.103D+03,0.210D+02,0.98650000D+00,0.00000000D+00
     .,0.33738960D+03,0.103D+03,0.220D+02,0.98650000D+00,0.00000000D+00
     .,0.30654820D+03,0.103D+03,0.230D+02,0.98650000D+00,0.00000000D+00
     .,0.24050330D+03,0.103D+03,0.240D+02,0.98650000D+00,0.00000000D+00
     .,0.26090190D+03,0.103D+03,0.250D+02,0.98650000D+00,0.00000000D+00
     .,0.20361490D+03,0.103D+03,0.260D+02,0.98650000D+00,0.00000000D+00
     .,0.21289780D+03,0.103D+03,0.270D+02,0.98650000D+00,0.00000000D+00
     .,0.22117390D+03,0.103D+03,0.280D+02,0.98650000D+00,0.00000000D+00
     .,0.16880750D+03,0.103D+03,0.290D+02,0.98650000D+00,0.00000000D+00
     .,0.16949490D+03,0.103D+03,0.300D+02,0.98650000D+00,0.00000000D+00
     .,0.20269000D+03,0.103D+03,0.310D+02,0.98650000D+00,0.00000000D+00
     .,0.17294250D+03,0.103D+03,0.320D+02,0.98650000D+00,0.00000000D+00
     .,0.14284710D+03,0.103D+03,0.330D+02,0.98650000D+00,0.00000000D+00
     .,0.12547680D+03,0.103D+03,0.340D+02,0.98650000D+00,0.00000000D+00
     .,0.10733800D+03,0.103D+03,0.350D+02,0.98650000D+00,0.00000000D+00
     .,0.91333300D+02,0.103D+03,0.360D+02,0.98650000D+00,0.00000000D+00
     .,0.61856810D+03,0.103D+03,0.370D+02,0.98650000D+00,0.00000000D+00
     .,0.51783930D+03,0.103D+03,0.380D+02,0.98650000D+00,0.00000000D+00
     .,0.44294480D+03,0.103D+03,0.390D+02,0.98650000D+00,0.00000000D+00
     .,0.39191790D+03,0.103D+03,0.400D+02,0.98650000D+00,0.00000000D+00
     ./)
      pars(  23001:  23100)=(/
     . 0.35350080D+03,0.103D+03,0.410D+02,0.98650000D+00,0.00000000D+00
     .,0.26737610D+03,0.103D+03,0.420D+02,0.98650000D+00,0.00000000D+00
     .,0.30065330D+03,0.103D+03,0.430D+02,0.98650000D+00,0.00000000D+00
     .,0.22388710D+03,0.103D+03,0.440D+02,0.98650000D+00,0.00000000D+00
     .,0.24525610D+03,0.103D+03,0.450D+02,0.98650000D+00,0.00000000D+00
     .,0.22580420D+03,0.103D+03,0.460D+02,0.98650000D+00,0.00000000D+00
     .,0.18833740D+03,0.103D+03,0.470D+02,0.98650000D+00,0.00000000D+00
     .,0.19709400D+03,0.103D+03,0.480D+02,0.98650000D+00,0.00000000D+00
     .,0.25313700D+03,0.103D+03,0.490D+02,0.98650000D+00,0.00000000D+00
     .,0.22808140D+03,0.103D+03,0.500D+02,0.98650000D+00,0.00000000D+00
     .,0.19776980D+03,0.103D+03,0.510D+02,0.98650000D+00,0.00000000D+00
     .,0.18030560D+03,0.103D+03,0.520D+02,0.98650000D+00,0.00000000D+00
     .,0.15986700D+03,0.103D+03,0.530D+02,0.98650000D+00,0.00000000D+00
     .,0.14098180D+03,0.103D+03,0.540D+02,0.98650000D+00,0.00000000D+00
     .,0.75234180D+03,0.103D+03,0.550D+02,0.98650000D+00,0.00000000D+00
     .,0.66457190D+03,0.103D+03,0.560D+02,0.98650000D+00,0.00000000D+00
     .,0.57042660D+03,0.103D+03,0.570D+02,0.98650000D+00,0.00000000D+00
     .,0.23452710D+03,0.103D+03,0.580D+02,0.98650000D+00,0.27991000D+01
     .,0.58397970D+03,0.103D+03,0.590D+02,0.98650000D+00,0.00000000D+00
     .,0.55869720D+03,0.103D+03,0.600D+02,0.98650000D+00,0.00000000D+00
     ./)
      pars(  23101:  23200)=(/
     . 0.54412320D+03,0.103D+03,0.610D+02,0.98650000D+00,0.00000000D+00
     .,0.53079090D+03,0.103D+03,0.620D+02,0.98650000D+00,0.00000000D+00
     .,0.51894980D+03,0.103D+03,0.630D+02,0.98650000D+00,0.00000000D+00
     .,0.39672230D+03,0.103D+03,0.640D+02,0.98650000D+00,0.00000000D+00
     .,0.46354420D+03,0.103D+03,0.650D+02,0.98650000D+00,0.00000000D+00
     .,0.44501010D+03,0.103D+03,0.660D+02,0.98650000D+00,0.00000000D+00
     .,0.46541000D+03,0.103D+03,0.670D+02,0.98650000D+00,0.00000000D+00
     .,0.45528270D+03,0.103D+03,0.680D+02,0.98650000D+00,0.00000000D+00
     .,0.44600300D+03,0.103D+03,0.690D+02,0.98650000D+00,0.00000000D+00
     .,0.44132200D+03,0.103D+03,0.700D+02,0.98650000D+00,0.00000000D+00
     .,0.36474670D+03,0.103D+03,0.710D+02,0.98650000D+00,0.00000000D+00
     .,0.35006170D+03,0.103D+03,0.720D+02,0.98650000D+00,0.00000000D+00
     .,0.31424830D+03,0.103D+03,0.730D+02,0.98650000D+00,0.00000000D+00
     .,0.26138330D+03,0.103D+03,0.740D+02,0.98650000D+00,0.00000000D+00
     .,0.26436970D+03,0.103D+03,0.750D+02,0.98650000D+00,0.00000000D+00
     .,0.23602580D+03,0.103D+03,0.760D+02,0.98650000D+00,0.00000000D+00
     .,0.21346440D+03,0.103D+03,0.770D+02,0.98650000D+00,0.00000000D+00
     .,0.17461320D+03,0.103D+03,0.780D+02,0.98650000D+00,0.00000000D+00
     .,0.16213420D+03,0.103D+03,0.790D+02,0.98650000D+00,0.00000000D+00
     .,0.16604890D+03,0.103D+03,0.800D+02,0.98650000D+00,0.00000000D+00
     ./)
      pars(  23201:  23300)=(/
     . 0.25738230D+03,0.103D+03,0.810D+02,0.98650000D+00,0.00000000D+00
     .,0.24683550D+03,0.103D+03,0.820D+02,0.98650000D+00,0.00000000D+00
     .,0.22143250D+03,0.103D+03,0.830D+02,0.98650000D+00,0.00000000D+00
     .,0.20812690D+03,0.103D+03,0.840D+02,0.98650000D+00,0.00000000D+00
     .,0.18860730D+03,0.103D+03,0.850D+02,0.98650000D+00,0.00000000D+00
     .,0.16989430D+03,0.103D+03,0.860D+02,0.98650000D+00,0.00000000D+00
     .,0.69764490D+03,0.103D+03,0.870D+02,0.98650000D+00,0.00000000D+00
     .,0.64903250D+03,0.103D+03,0.880D+02,0.98650000D+00,0.00000000D+00
     .,0.56135190D+03,0.103D+03,0.890D+02,0.98650000D+00,0.00000000D+00
     .,0.49107770D+03,0.103D+03,0.900D+02,0.98650000D+00,0.00000000D+00
     .,0.49346510D+03,0.103D+03,0.910D+02,0.98650000D+00,0.00000000D+00
     .,0.47742540D+03,0.103D+03,0.920D+02,0.98650000D+00,0.00000000D+00
     .,0.49952020D+03,0.103D+03,0.930D+02,0.98650000D+00,0.00000000D+00
     .,0.48234660D+03,0.103D+03,0.940D+02,0.98650000D+00,0.00000000D+00
     .,0.24057300D+02,0.103D+03,0.101D+03,0.98650000D+00,0.00000000D+00
     .,0.85319700D+02,0.103D+03,0.103D+03,0.98650000D+00,0.98650000D+00
     .,0.18465600D+02,0.104D+03,0.100D+01,0.98080000D+00,0.91180000D+00
     .,0.11465500D+02,0.104D+03,0.200D+01,0.98080000D+00,0.00000000D+00
     .,0.35221600D+03,0.104D+03,0.300D+01,0.98080000D+00,0.00000000D+00
     .,0.18634530D+03,0.104D+03,0.400D+01,0.98080000D+00,0.00000000D+00
     ./)
      pars(  23301:  23400)=(/
     . 0.11908150D+03,0.104D+03,0.500D+01,0.98080000D+00,0.00000000D+00
     .,0.77095100D+02,0.104D+03,0.600D+01,0.98080000D+00,0.00000000D+00
     .,0.52085200D+02,0.104D+03,0.700D+01,0.98080000D+00,0.00000000D+00
     .,0.38413200D+02,0.104D+03,0.800D+01,0.98080000D+00,0.00000000D+00
     .,0.28409800D+02,0.104D+03,0.900D+01,0.98080000D+00,0.00000000D+00
     .,0.21406700D+02,0.104D+03,0.100D+02,0.98080000D+00,0.00000000D+00
     .,0.41864880D+03,0.104D+03,0.110D+02,0.98080000D+00,0.00000000D+00
     .,0.30137770D+03,0.104D+03,0.120D+02,0.98080000D+00,0.00000000D+00
     .,0.27074610D+03,0.104D+03,0.130D+02,0.98080000D+00,0.00000000D+00
     .,0.20575440D+03,0.104D+03,0.140D+02,0.98080000D+00,0.00000000D+00
     .,0.15536440D+03,0.104D+03,0.150D+02,0.98080000D+00,0.00000000D+00
     .,0.12609930D+03,0.104D+03,0.160D+02,0.98080000D+00,0.00000000D+00
     .,0.10072120D+03,0.104D+03,0.170D+02,0.98080000D+00,0.00000000D+00
     .,0.80717400D+02,0.104D+03,0.180D+02,0.98080000D+00,0.00000000D+00
     .,0.69485960D+03,0.104D+03,0.190D+02,0.98080000D+00,0.00000000D+00
     .,0.54358420D+03,0.104D+03,0.200D+02,0.98080000D+00,0.00000000D+00
     .,0.44371900D+03,0.104D+03,0.210D+02,0.98080000D+00,0.00000000D+00
     .,0.42362170D+03,0.104D+03,0.220D+02,0.98080000D+00,0.00000000D+00
     .,0.38527540D+03,0.104D+03,0.230D+02,0.98080000D+00,0.00000000D+00
     .,0.30260610D+03,0.104D+03,0.240D+02,0.98080000D+00,0.00000000D+00
     ./)
      pars(  23401:  23500)=(/
     . 0.32840120D+03,0.104D+03,0.250D+02,0.98080000D+00,0.00000000D+00
     .,0.25660430D+03,0.104D+03,0.260D+02,0.98080000D+00,0.00000000D+00
     .,0.26862840D+03,0.104D+03,0.270D+02,0.98080000D+00,0.00000000D+00
     .,0.27871160D+03,0.104D+03,0.280D+02,0.98080000D+00,0.00000000D+00
     .,0.21297970D+03,0.104D+03,0.290D+02,0.98080000D+00,0.00000000D+00
     .,0.21442560D+03,0.104D+03,0.300D+02,0.98080000D+00,0.00000000D+00
     .,0.25620490D+03,0.104D+03,0.310D+02,0.98080000D+00,0.00000000D+00
     .,0.21973750D+03,0.104D+03,0.320D+02,0.98080000D+00,0.00000000D+00
     .,0.18253380D+03,0.104D+03,0.330D+02,0.98080000D+00,0.00000000D+00
     .,0.16096870D+03,0.104D+03,0.340D+02,0.98080000D+00,0.00000000D+00
     .,0.13827470D+03,0.104D+03,0.350D+02,0.98080000D+00,0.00000000D+00
     .,0.11812400D+03,0.104D+03,0.360D+02,0.98080000D+00,0.00000000D+00
     .,0.77496560D+03,0.104D+03,0.370D+02,0.98080000D+00,0.00000000D+00
     .,0.64864220D+03,0.104D+03,0.380D+02,0.98080000D+00,0.00000000D+00
     .,0.55615030D+03,0.104D+03,0.390D+02,0.98080000D+00,0.00000000D+00
     .,0.49301000D+03,0.104D+03,0.400D+02,0.98080000D+00,0.00000000D+00
     .,0.44537360D+03,0.104D+03,0.410D+02,0.98080000D+00,0.00000000D+00
     .,0.33790350D+03,0.104D+03,0.420D+02,0.98080000D+00,0.00000000D+00
     .,0.37952540D+03,0.104D+03,0.430D+02,0.98080000D+00,0.00000000D+00
     .,0.28356950D+03,0.104D+03,0.440D+02,0.98080000D+00,0.00000000D+00
     ./)
      pars(  23501:  23600)=(/
     . 0.31037810D+03,0.104D+03,0.450D+02,0.98080000D+00,0.00000000D+00
     .,0.28603950D+03,0.104D+03,0.460D+02,0.98080000D+00,0.00000000D+00
     .,0.23869620D+03,0.104D+03,0.470D+02,0.98080000D+00,0.00000000D+00
     .,0.24999150D+03,0.104D+03,0.480D+02,0.98080000D+00,0.00000000D+00
     .,0.32009120D+03,0.104D+03,0.490D+02,0.98080000D+00,0.00000000D+00
     .,0.28947130D+03,0.104D+03,0.500D+02,0.98080000D+00,0.00000000D+00
     .,0.25216490D+03,0.104D+03,0.510D+02,0.98080000D+00,0.00000000D+00
     .,0.23063090D+03,0.104D+03,0.520D+02,0.98080000D+00,0.00000000D+00
     .,0.20522370D+03,0.104D+03,0.530D+02,0.98080000D+00,0.00000000D+00
     .,0.18162600D+03,0.104D+03,0.540D+02,0.98080000D+00,0.00000000D+00
     .,0.94385470D+03,0.104D+03,0.550D+02,0.98080000D+00,0.00000000D+00
     .,0.83262510D+03,0.104D+03,0.560D+02,0.98080000D+00,0.00000000D+00
     .,0.71607580D+03,0.104D+03,0.570D+02,0.98080000D+00,0.00000000D+00
     .,0.29887820D+03,0.104D+03,0.580D+02,0.98080000D+00,0.27991000D+01
     .,0.73216420D+03,0.104D+03,0.590D+02,0.98080000D+00,0.00000000D+00
     .,0.70044420D+03,0.104D+03,0.600D+02,0.98080000D+00,0.00000000D+00
     .,0.68219010D+03,0.104D+03,0.610D+02,0.98080000D+00,0.00000000D+00
     .,0.66548680D+03,0.104D+03,0.620D+02,0.98080000D+00,0.00000000D+00
     .,0.65065570D+03,0.104D+03,0.630D+02,0.98080000D+00,0.00000000D+00
     .,0.49918370D+03,0.104D+03,0.640D+02,0.98080000D+00,0.00000000D+00
     ./)
      pars(  23601:  23700)=(/
     . 0.58212240D+03,0.104D+03,0.650D+02,0.98080000D+00,0.00000000D+00
     .,0.55929170D+03,0.104D+03,0.660D+02,0.98080000D+00,0.00000000D+00
     .,0.58370670D+03,0.104D+03,0.670D+02,0.98080000D+00,0.00000000D+00
     .,0.57100390D+03,0.104D+03,0.680D+02,0.98080000D+00,0.00000000D+00
     .,0.55938980D+03,0.104D+03,0.690D+02,0.98080000D+00,0.00000000D+00
     .,0.55342660D+03,0.104D+03,0.700D+02,0.98080000D+00,0.00000000D+00
     .,0.45865820D+03,0.104D+03,0.710D+02,0.98080000D+00,0.00000000D+00
     .,0.44137890D+03,0.104D+03,0.720D+02,0.98080000D+00,0.00000000D+00
     .,0.39713120D+03,0.104D+03,0.730D+02,0.98080000D+00,0.00000000D+00
     .,0.33121240D+03,0.104D+03,0.740D+02,0.98080000D+00,0.00000000D+00
     .,0.33522610D+03,0.104D+03,0.750D+02,0.98080000D+00,0.00000000D+00
     .,0.29995650D+03,0.104D+03,0.760D+02,0.98080000D+00,0.00000000D+00
     .,0.27181190D+03,0.104D+03,0.770D+02,0.98080000D+00,0.00000000D+00
     .,0.22291570D+03,0.104D+03,0.780D+02,0.98080000D+00,0.00000000D+00
     .,0.20718310D+03,0.104D+03,0.790D+02,0.98080000D+00,0.00000000D+00
     .,0.21231320D+03,0.104D+03,0.800D+02,0.98080000D+00,0.00000000D+00
     .,0.32610680D+03,0.104D+03,0.810D+02,0.98080000D+00,0.00000000D+00
     .,0.31346750D+03,0.104D+03,0.820D+02,0.98080000D+00,0.00000000D+00
     .,0.28229350D+03,0.104D+03,0.830D+02,0.98080000D+00,0.00000000D+00
     .,0.26602000D+03,0.104D+03,0.840D+02,0.98080000D+00,0.00000000D+00
     ./)
      pars(  23701:  23800)=(/
     . 0.24186880D+03,0.104D+03,0.850D+02,0.98080000D+00,0.00000000D+00
     .,0.21856870D+03,0.104D+03,0.860D+02,0.98080000D+00,0.00000000D+00
     .,0.87585630D+03,0.104D+03,0.870D+02,0.98080000D+00,0.00000000D+00
     .,0.81413080D+03,0.104D+03,0.880D+02,0.98080000D+00,0.00000000D+00
     .,0.70540110D+03,0.104D+03,0.890D+02,0.98080000D+00,0.00000000D+00
     .,0.61906050D+03,0.104D+03,0.900D+02,0.98080000D+00,0.00000000D+00
     .,0.62133990D+03,0.104D+03,0.910D+02,0.98080000D+00,0.00000000D+00
     .,0.60114960D+03,0.104D+03,0.920D+02,0.98080000D+00,0.00000000D+00
     .,0.62763850D+03,0.104D+03,0.930D+02,0.98080000D+00,0.00000000D+00
     .,0.60621390D+03,0.104D+03,0.940D+02,0.98080000D+00,0.00000000D+00
     .,0.30786600D+02,0.104D+03,0.101D+03,0.98080000D+00,0.00000000D+00
     .,0.10750830D+03,0.104D+03,0.103D+03,0.98080000D+00,0.98650000D+00
     .,0.13584450D+03,0.104D+03,0.104D+03,0.98080000D+00,0.98080000D+00
     .,0.14723700D+02,0.105D+03,0.100D+01,0.97060000D+00,0.91180000D+00
     .,0.95836000D+01,0.105D+03,0.200D+01,0.97060000D+00,0.00000000D+00
     .,0.23142090D+03,0.105D+03,0.300D+01,0.97060000D+00,0.00000000D+00
     .,0.13372650D+03,0.105D+03,0.400D+01,0.97060000D+00,0.00000000D+00
     .,0.89742900D+02,0.105D+03,0.500D+01,0.97060000D+00,0.00000000D+00
     .,0.60299600D+02,0.105D+03,0.600D+01,0.97060000D+00,0.00000000D+00
     .,0.41901500D+02,0.105D+03,0.700D+01,0.97060000D+00,0.00000000D+00
     ./)
      pars(  23801:  23900)=(/
     . 0.31535400D+02,0.105D+03,0.800D+01,0.97060000D+00,0.00000000D+00
     .,0.23736900D+02,0.105D+03,0.900D+01,0.97060000D+00,0.00000000D+00
     .,0.18142400D+02,0.105D+03,0.100D+02,0.97060000D+00,0.00000000D+00
     .,0.27656710D+03,0.105D+03,0.110D+02,0.97060000D+00,0.00000000D+00
     .,0.21299790D+03,0.105D+03,0.120D+02,0.97060000D+00,0.00000000D+00
     .,0.19620140D+03,0.105D+03,0.130D+02,0.97060000D+00,0.00000000D+00
     .,0.15429010D+03,0.105D+03,0.140D+02,0.97060000D+00,0.00000000D+00
     .,0.11993530D+03,0.105D+03,0.150D+02,0.97060000D+00,0.00000000D+00
     .,0.99215900D+02,0.105D+03,0.160D+02,0.97060000D+00,0.00000000D+00
     .,0.80727200D+02,0.105D+03,0.170D+02,0.97060000D+00,0.00000000D+00
     .,0.65758000D+02,0.105D+03,0.180D+02,0.97060000D+00,0.00000000D+00
     .,0.45251930D+03,0.105D+03,0.190D+02,0.97060000D+00,0.00000000D+00
     .,0.37388150D+03,0.105D+03,0.200D+02,0.97060000D+00,0.00000000D+00
     .,0.30887210D+03,0.105D+03,0.210D+02,0.97060000D+00,0.00000000D+00
     .,0.29808710D+03,0.105D+03,0.220D+02,0.97060000D+00,0.00000000D+00
     .,0.27287470D+03,0.105D+03,0.230D+02,0.97060000D+00,0.00000000D+00
     .,0.21463600D+03,0.105D+03,0.240D+02,0.97060000D+00,0.00000000D+00
     .,0.23480000D+03,0.105D+03,0.250D+02,0.97060000D+00,0.00000000D+00
     .,0.18397370D+03,0.105D+03,0.260D+02,0.97060000D+00,0.00000000D+00
     .,0.19517310D+03,0.105D+03,0.270D+02,0.97060000D+00,0.00000000D+00
     ./)
      pars(  23901:  24000)=(/
     . 0.20113510D+03,0.105D+03,0.280D+02,0.97060000D+00,0.00000000D+00
     .,0.15388240D+03,0.105D+03,0.290D+02,0.97060000D+00,0.00000000D+00
     .,0.15808480D+03,0.105D+03,0.300D+02,0.97060000D+00,0.00000000D+00
     .,0.18757970D+03,0.105D+03,0.310D+02,0.97060000D+00,0.00000000D+00
     .,0.16519300D+03,0.105D+03,0.320D+02,0.97060000D+00,0.00000000D+00
     .,0.14056250D+03,0.105D+03,0.330D+02,0.97060000D+00,0.00000000D+00
     .,0.12584370D+03,0.105D+03,0.340D+02,0.97060000D+00,0.00000000D+00
     .,0.10980600D+03,0.105D+03,0.350D+02,0.97060000D+00,0.00000000D+00
     .,0.95176300D+02,0.105D+03,0.360D+02,0.97060000D+00,0.00000000D+00
     .,0.50694210D+03,0.105D+03,0.370D+02,0.97060000D+00,0.00000000D+00
     .,0.44526080D+03,0.105D+03,0.380D+02,0.97060000D+00,0.00000000D+00
     .,0.38988230D+03,0.105D+03,0.390D+02,0.97060000D+00,0.00000000D+00
     .,0.35019700D+03,0.105D+03,0.400D+02,0.97060000D+00,0.00000000D+00
     .,0.31911510D+03,0.105D+03,0.410D+02,0.97060000D+00,0.00000000D+00
     .,0.24585270D+03,0.105D+03,0.420D+02,0.97060000D+00,0.00000000D+00
     .,0.27453340D+03,0.105D+03,0.430D+02,0.97060000D+00,0.00000000D+00
     .,0.20863620D+03,0.105D+03,0.440D+02,0.97060000D+00,0.00000000D+00
     .,0.22828220D+03,0.105D+03,0.450D+02,0.97060000D+00,0.00000000D+00
     .,0.21155590D+03,0.105D+03,0.460D+02,0.97060000D+00,0.00000000D+00
     .,0.17606030D+03,0.105D+03,0.470D+02,0.97060000D+00,0.00000000D+00
     ./)
      pars(  24001:  24100)=(/
     . 0.18621990D+03,0.105D+03,0.480D+02,0.97060000D+00,0.00000000D+00
     .,0.23421710D+03,0.105D+03,0.490D+02,0.97060000D+00,0.00000000D+00
     .,0.21642760D+03,0.105D+03,0.500D+02,0.97060000D+00,0.00000000D+00
     .,0.19255420D+03,0.105D+03,0.510D+02,0.97060000D+00,0.00000000D+00
     .,0.17839190D+03,0.105D+03,0.520D+02,0.97060000D+00,0.00000000D+00
     .,0.16096790D+03,0.105D+03,0.530D+02,0.97060000D+00,0.00000000D+00
     .,0.14436970D+03,0.105D+03,0.540D+02,0.97060000D+00,0.00000000D+00
     .,0.61760580D+03,0.105D+03,0.550D+02,0.97060000D+00,0.00000000D+00
     .,0.56751360D+03,0.105D+03,0.560D+02,0.97060000D+00,0.00000000D+00
     .,0.49845340D+03,0.105D+03,0.570D+02,0.97060000D+00,0.00000000D+00
     .,0.22741890D+03,0.105D+03,0.580D+02,0.97060000D+00,0.27991000D+01
     .,0.50249670D+03,0.105D+03,0.590D+02,0.97060000D+00,0.00000000D+00
     .,0.48250770D+03,0.105D+03,0.600D+02,0.97060000D+00,0.00000000D+00
     .,0.47040290D+03,0.105D+03,0.610D+02,0.97060000D+00,0.00000000D+00
     .,0.45927530D+03,0.105D+03,0.620D+02,0.97060000D+00,0.00000000D+00
     .,0.44941020D+03,0.105D+03,0.630D+02,0.97060000D+00,0.00000000D+00
     .,0.35290870D+03,0.105D+03,0.640D+02,0.97060000D+00,0.00000000D+00
     .,0.39744990D+03,0.105D+03,0.650D+02,0.97060000D+00,0.00000000D+00
     .,0.38329520D+03,0.105D+03,0.660D+02,0.97060000D+00,0.00000000D+00
     .,0.40532850D+03,0.105D+03,0.670D+02,0.97060000D+00,0.00000000D+00
     ./)
      pars(  24101:  24200)=(/
     . 0.39673770D+03,0.105D+03,0.680D+02,0.97060000D+00,0.00000000D+00
     .,0.38898900D+03,0.105D+03,0.690D+02,0.97060000D+00,0.00000000D+00
     .,0.38447010D+03,0.105D+03,0.700D+02,0.97060000D+00,0.00000000D+00
     .,0.32364500D+03,0.105D+03,0.710D+02,0.97060000D+00,0.00000000D+00
     .,0.31828720D+03,0.105D+03,0.720D+02,0.97060000D+00,0.00000000D+00
     .,0.29018010D+03,0.105D+03,0.730D+02,0.97060000D+00,0.00000000D+00
     .,0.24456750D+03,0.105D+03,0.740D+02,0.97060000D+00,0.00000000D+00
     .,0.24877750D+03,0.105D+03,0.750D+02,0.97060000D+00,0.00000000D+00
     .,0.22518190D+03,0.105D+03,0.760D+02,0.97060000D+00,0.00000000D+00
     .,0.20597020D+03,0.105D+03,0.770D+02,0.97060000D+00,0.00000000D+00
     .,0.17065560D+03,0.105D+03,0.780D+02,0.97060000D+00,0.00000000D+00
     .,0.15926350D+03,0.105D+03,0.790D+02,0.97060000D+00,0.00000000D+00
     .,0.16388200D+03,0.105D+03,0.800D+02,0.97060000D+00,0.00000000D+00
     .,0.24000000D+03,0.105D+03,0.810D+02,0.97060000D+00,0.00000000D+00
     .,0.23460480D+03,0.105D+03,0.820D+02,0.97060000D+00,0.00000000D+00
     .,0.21530270D+03,0.105D+03,0.830D+02,0.97060000D+00,0.00000000D+00
     .,0.20513340D+03,0.105D+03,0.840D+02,0.97060000D+00,0.00000000D+00
     .,0.18899910D+03,0.105D+03,0.850D+02,0.97060000D+00,0.00000000D+00
     .,0.17288540D+03,0.105D+03,0.860D+02,0.97060000D+00,0.00000000D+00
     .,0.58298150D+03,0.105D+03,0.870D+02,0.97060000D+00,0.00000000D+00
     ./)
      pars(  24201:  24300)=(/
     . 0.56100860D+03,0.105D+03,0.880D+02,0.97060000D+00,0.00000000D+00
     .,0.49563890D+03,0.105D+03,0.890D+02,0.97060000D+00,0.00000000D+00
     .,0.44481020D+03,0.105D+03,0.900D+02,0.97060000D+00,0.00000000D+00
     .,0.44155780D+03,0.105D+03,0.910D+02,0.97060000D+00,0.00000000D+00
     .,0.42748070D+03,0.105D+03,0.920D+02,0.97060000D+00,0.00000000D+00
     .,0.44027100D+03,0.105D+03,0.930D+02,0.97060000D+00,0.00000000D+00
     .,0.42632550D+03,0.105D+03,0.940D+02,0.97060000D+00,0.00000000D+00
     .,0.23841200D+02,0.105D+03,0.101D+03,0.97060000D+00,0.00000000D+00
     .,0.77663300D+02,0.105D+03,0.103D+03,0.97060000D+00,0.98650000D+00
     .,0.98993800D+02,0.105D+03,0.104D+03,0.97060000D+00,0.98080000D+00
     .,0.75368600D+02,0.105D+03,0.105D+03,0.97060000D+00,0.97060000D+00
     .,0.11393200D+02,0.106D+03,0.100D+01,0.98680000D+00,0.91180000D+00
     .,0.77065000D+01,0.106D+03,0.200D+01,0.98680000D+00,0.00000000D+00
     .,0.16045900D+03,0.106D+03,0.300D+01,0.98680000D+00,0.00000000D+00
     .,0.96543100D+02,0.106D+03,0.400D+01,0.98680000D+00,0.00000000D+00
     .,0.66798600D+02,0.106D+03,0.500D+01,0.98680000D+00,0.00000000D+00
     .,0.46068100D+02,0.106D+03,0.600D+01,0.98680000D+00,0.00000000D+00
     .,0.32700900D+02,0.106D+03,0.700D+01,0.98680000D+00,0.00000000D+00
     .,0.25009700D+02,0.106D+03,0.800D+01,0.98680000D+00,0.00000000D+00
     .,0.19097400D+02,0.106D+03,0.900D+01,0.98680000D+00,0.00000000D+00
     ./)
      pars(  24301:  24400)=(/
     . 0.14771600D+02,0.106D+03,0.100D+02,0.98680000D+00,0.00000000D+00
     .,0.19237290D+03,0.106D+03,0.110D+02,0.98680000D+00,0.00000000D+00
     .,0.15261300D+03,0.106D+03,0.120D+02,0.98680000D+00,0.00000000D+00
     .,0.14260940D+03,0.106D+03,0.130D+02,0.98680000D+00,0.00000000D+00
     .,0.11444050D+03,0.106D+03,0.140D+02,0.98680000D+00,0.00000000D+00
     .,0.90676700D+02,0.106D+03,0.150D+02,0.98680000D+00,0.00000000D+00
     .,0.76055200D+02,0.106D+03,0.160D+02,0.98680000D+00,0.00000000D+00
     .,0.62746500D+02,0.106D+03,0.170D+02,0.98680000D+00,0.00000000D+00
     .,0.51768400D+02,0.106D+03,0.180D+02,0.98680000D+00,0.00000000D+00
     .,0.31458000D+03,0.106D+03,0.190D+02,0.98680000D+00,0.00000000D+00
     .,0.26512060D+03,0.106D+03,0.200D+02,0.98680000D+00,0.00000000D+00
     .,0.22018540D+03,0.106D+03,0.210D+02,0.98680000D+00,0.00000000D+00
     .,0.21386820D+03,0.106D+03,0.220D+02,0.98680000D+00,0.00000000D+00
     .,0.19648850D+03,0.106D+03,0.230D+02,0.98680000D+00,0.00000000D+00
     .,0.15511710D+03,0.106D+03,0.240D+02,0.98680000D+00,0.00000000D+00
     .,0.16997960D+03,0.106D+03,0.250D+02,0.98680000D+00,0.00000000D+00
     .,0.13377000D+03,0.106D+03,0.260D+02,0.98680000D+00,0.00000000D+00
     .,0.14251400D+03,0.106D+03,0.270D+02,0.98680000D+00,0.00000000D+00
     .,0.14627110D+03,0.106D+03,0.280D+02,0.98680000D+00,0.00000000D+00
     .,0.11239140D+03,0.106D+03,0.290D+02,0.98680000D+00,0.00000000D+00
     ./)
      pars(  24401:  24500)=(/
     . 0.11639870D+03,0.106D+03,0.300D+02,0.98680000D+00,0.00000000D+00
     .,0.13747230D+03,0.106D+03,0.310D+02,0.98680000D+00,0.00000000D+00
     .,0.12288050D+03,0.106D+03,0.320D+02,0.98680000D+00,0.00000000D+00
     .,0.10616860D+03,0.106D+03,0.330D+02,0.98680000D+00,0.00000000D+00
     .,0.96059900D+02,0.106D+03,0.340D+02,0.98680000D+00,0.00000000D+00
     .,0.84772300D+02,0.106D+03,0.350D+02,0.98680000D+00,0.00000000D+00
     .,0.74286500D+02,0.106D+03,0.360D+02,0.98680000D+00,0.00000000D+00
     .,0.35363020D+03,0.106D+03,0.370D+02,0.98680000D+00,0.00000000D+00
     .,0.31589230D+03,0.106D+03,0.380D+02,0.98680000D+00,0.00000000D+00
     .,0.27947270D+03,0.106D+03,0.390D+02,0.98680000D+00,0.00000000D+00
     .,0.25282130D+03,0.106D+03,0.400D+02,0.98680000D+00,0.00000000D+00
     .,0.23159760D+03,0.106D+03,0.410D+02,0.98680000D+00,0.00000000D+00
     .,0.18033900D+03,0.106D+03,0.420D+02,0.98680000D+00,0.00000000D+00
     .,0.20054640D+03,0.106D+03,0.430D+02,0.98680000D+00,0.00000000D+00
     .,0.15419380D+03,0.106D+03,0.440D+02,0.98680000D+00,0.00000000D+00
     .,0.16831150D+03,0.106D+03,0.450D+02,0.98680000D+00,0.00000000D+00
     .,0.15650060D+03,0.106D+03,0.460D+02,0.98680000D+00,0.00000000D+00
     .,0.13051440D+03,0.106D+03,0.470D+02,0.98680000D+00,0.00000000D+00
     .,0.13836720D+03,0.106D+03,0.480D+02,0.98680000D+00,0.00000000D+00
     .,0.17212290D+03,0.106D+03,0.490D+02,0.98680000D+00,0.00000000D+00
     ./)
      pars(  24501:  24600)=(/
     . 0.16079100D+03,0.106D+03,0.500D+02,0.98680000D+00,0.00000000D+00
     .,0.14484620D+03,0.106D+03,0.510D+02,0.98680000D+00,0.00000000D+00
     .,0.13531220D+03,0.106D+03,0.520D+02,0.98680000D+00,0.00000000D+00
     .,0.12325380D+03,0.106D+03,0.530D+02,0.98680000D+00,0.00000000D+00
     .,0.11159230D+03,0.106D+03,0.540D+02,0.98680000D+00,0.00000000D+00
     .,0.43151720D+03,0.106D+03,0.550D+02,0.98680000D+00,0.00000000D+00
     .,0.40187740D+03,0.106D+03,0.560D+02,0.98680000D+00,0.00000000D+00
     .,0.35646100D+03,0.106D+03,0.570D+02,0.98680000D+00,0.00000000D+00
     .,0.17092620D+03,0.106D+03,0.580D+02,0.98680000D+00,0.27991000D+01
     .,0.35713990D+03,0.106D+03,0.590D+02,0.98680000D+00,0.00000000D+00
     .,0.34341530D+03,0.106D+03,0.600D+02,0.98680000D+00,0.00000000D+00
     .,0.33492450D+03,0.106D+03,0.610D+02,0.98680000D+00,0.00000000D+00
     .,0.32709910D+03,0.106D+03,0.620D+02,0.98680000D+00,0.00000000D+00
     .,0.32016570D+03,0.106D+03,0.630D+02,0.98680000D+00,0.00000000D+00
     .,0.25481220D+03,0.106D+03,0.640D+02,0.98680000D+00,0.00000000D+00
     .,0.28285470D+03,0.106D+03,0.650D+02,0.98680000D+00,0.00000000D+00
     .,0.27333750D+03,0.106D+03,0.660D+02,0.98680000D+00,0.00000000D+00
     .,0.28939200D+03,0.106D+03,0.670D+02,0.98680000D+00,0.00000000D+00
     .,0.28330010D+03,0.106D+03,0.680D+02,0.98680000D+00,0.00000000D+00
     .,0.27784930D+03,0.106D+03,0.690D+02,0.98680000D+00,0.00000000D+00
     ./)
      pars(  24601:  24700)=(/
     . 0.27445000D+03,0.106D+03,0.700D+02,0.98680000D+00,0.00000000D+00
     .,0.23313940D+03,0.106D+03,0.710D+02,0.98680000D+00,0.00000000D+00
     .,0.23163340D+03,0.106D+03,0.720D+02,0.98680000D+00,0.00000000D+00
     .,0.21280410D+03,0.106D+03,0.730D+02,0.98680000D+00,0.00000000D+00
     .,0.18084620D+03,0.106D+03,0.740D+02,0.98680000D+00,0.00000000D+00
     .,0.18435860D+03,0.106D+03,0.750D+02,0.98680000D+00,0.00000000D+00
     .,0.16804770D+03,0.106D+03,0.760D+02,0.98680000D+00,0.00000000D+00
     .,0.15462480D+03,0.106D+03,0.770D+02,0.98680000D+00,0.00000000D+00
     .,0.12918350D+03,0.106D+03,0.780D+02,0.98680000D+00,0.00000000D+00
     .,0.12094850D+03,0.106D+03,0.790D+02,0.98680000D+00,0.00000000D+00
     .,0.12459510D+03,0.106D+03,0.800D+02,0.98680000D+00,0.00000000D+00
     .,0.17748530D+03,0.106D+03,0.810D+02,0.98680000D+00,0.00000000D+00
     .,0.17476750D+03,0.106D+03,0.820D+02,0.98680000D+00,0.00000000D+00
     .,0.16207400D+03,0.106D+03,0.830D+02,0.98680000D+00,0.00000000D+00
     .,0.15545060D+03,0.106D+03,0.840D+02,0.98680000D+00,0.00000000D+00
     .,0.14445320D+03,0.106D+03,0.850D+02,0.98680000D+00,0.00000000D+00
     .,0.13324320D+03,0.106D+03,0.860D+02,0.98680000D+00,0.00000000D+00
     .,0.41058600D+03,0.106D+03,0.870D+02,0.98680000D+00,0.00000000D+00
     .,0.39952250D+03,0.106D+03,0.880D+02,0.98680000D+00,0.00000000D+00
     .,0.35611270D+03,0.106D+03,0.890D+02,0.98680000D+00,0.00000000D+00
     ./)
      pars(  24701:  24800)=(/
     . 0.32349590D+03,0.106D+03,0.900D+02,0.98680000D+00,0.00000000D+00
     .,0.31967610D+03,0.106D+03,0.910D+02,0.98680000D+00,0.00000000D+00
     .,0.30960820D+03,0.106D+03,0.920D+02,0.98680000D+00,0.00000000D+00
     .,0.31659670D+03,0.106D+03,0.930D+02,0.98680000D+00,0.00000000D+00
     .,0.30693030D+03,0.106D+03,0.940D+02,0.98680000D+00,0.00000000D+00
     .,0.18057500D+02,0.106D+03,0.101D+03,0.98680000D+00,0.00000000D+00
     .,0.56346800D+02,0.106D+03,0.103D+03,0.98680000D+00,0.98650000D+00
     .,0.72316200D+02,0.106D+03,0.104D+03,0.98680000D+00,0.98080000D+00
     .,0.56529500D+02,0.106D+03,0.105D+03,0.98680000D+00,0.97060000D+00
     .,0.43245200D+02,0.106D+03,0.106D+03,0.98680000D+00,0.98680000D+00
     .,0.81417000D+01,0.107D+03,0.100D+01,0.99440000D+00,0.91180000D+00
     .,0.57601000D+01,0.107D+03,0.200D+01,0.99440000D+00,0.00000000D+00
     .,0.10127010D+03,0.107D+03,0.300D+01,0.99440000D+00,0.00000000D+00
     .,0.63858000D+02,0.107D+03,0.400D+01,0.99440000D+00,0.00000000D+00
     .,0.45731900D+02,0.107D+03,0.500D+01,0.99440000D+00,0.00000000D+00
     .,0.32484800D+02,0.107D+03,0.600D+01,0.99440000D+00,0.00000000D+00
     .,0.23629500D+02,0.107D+03,0.700D+01,0.99440000D+00,0.00000000D+00
     .,0.18412800D+02,0.107D+03,0.800D+01,0.99440000D+00,0.00000000D+00
     .,0.14301500D+02,0.107D+03,0.900D+01,0.99440000D+00,0.00000000D+00
     .,0.11224300D+02,0.107D+03,0.100D+02,0.99440000D+00,0.00000000D+00
     ./)
      pars(  24801:  24900)=(/
     . 0.12196010D+03,0.107D+03,0.110D+02,0.99440000D+00,0.00000000D+00
     .,0.10014430D+03,0.107D+03,0.120D+02,0.99440000D+00,0.00000000D+00
     .,0.95091600D+02,0.107D+03,0.130D+02,0.99440000D+00,0.00000000D+00
     .,0.78046600D+02,0.107D+03,0.140D+02,0.99440000D+00,0.00000000D+00
     .,0.63166900D+02,0.107D+03,0.150D+02,0.99440000D+00,0.00000000D+00
     .,0.53810700D+02,0.107D+03,0.160D+02,0.99440000D+00,0.00000000D+00
     .,0.45095200D+02,0.107D+03,0.170D+02,0.99440000D+00,0.00000000D+00
     .,0.37752200D+02,0.107D+03,0.180D+02,0.99440000D+00,0.00000000D+00
     .,0.19944050D+03,0.107D+03,0.190D+02,0.99440000D+00,0.00000000D+00
     .,0.17204550D+03,0.107D+03,0.200D+02,0.99440000D+00,0.00000000D+00
     .,0.14376580D+03,0.107D+03,0.210D+02,0.99440000D+00,0.00000000D+00
     .,0.14071150D+03,0.107D+03,0.220D+02,0.99440000D+00,0.00000000D+00
     .,0.12982970D+03,0.107D+03,0.230D+02,0.99440000D+00,0.00000000D+00
     .,0.10301890D+03,0.107D+03,0.240D+02,0.99440000D+00,0.00000000D+00
     .,0.11302480D+03,0.107D+03,0.250D+02,0.99440000D+00,0.00000000D+00
     .,0.89486800D+02,0.107D+03,0.260D+02,0.99440000D+00,0.00000000D+00
     .,0.95705100D+02,0.107D+03,0.270D+02,0.99440000D+00,0.00000000D+00
     .,0.97776100D+02,0.107D+03,0.280D+02,0.99440000D+00,0.00000000D+00
     .,0.75598700D+02,0.107D+03,0.290D+02,0.99440000D+00,0.00000000D+00
     .,0.78918300D+02,0.107D+03,0.300D+02,0.99440000D+00,0.00000000D+00
     ./)
      pars(  24901:  25000)=(/
     . 0.92602500D+02,0.107D+03,0.310D+02,0.99440000D+00,0.00000000D+00
     .,0.84127600D+02,0.107D+03,0.320D+02,0.99440000D+00,0.00000000D+00
     .,0.73914200D+02,0.107D+03,0.330D+02,0.99440000D+00,0.00000000D+00
     .,0.67663600D+02,0.107D+03,0.340D+02,0.99440000D+00,0.00000000D+00
     .,0.60472800D+02,0.107D+03,0.350D+02,0.99440000D+00,0.00000000D+00
     .,0.53649600D+02,0.107D+03,0.360D+02,0.99440000D+00,0.00000000D+00
     .,0.22519190D+03,0.107D+03,0.370D+02,0.99440000D+00,0.00000000D+00
     .,0.20515740D+03,0.107D+03,0.380D+02,0.99440000D+00,0.00000000D+00
     .,0.18368510D+03,0.107D+03,0.390D+02,0.99440000D+00,0.00000000D+00
     .,0.16755010D+03,0.107D+03,0.400D+02,0.99440000D+00,0.00000000D+00
     .,0.15443560D+03,0.107D+03,0.410D+02,0.99440000D+00,0.00000000D+00
     .,0.12180660D+03,0.107D+03,0.420D+02,0.99440000D+00,0.00000000D+00
     .,0.13478870D+03,0.107D+03,0.430D+02,0.99440000D+00,0.00000000D+00
     .,0.10509050D+03,0.107D+03,0.440D+02,0.99440000D+00,0.00000000D+00
     .,0.11434960D+03,0.107D+03,0.450D+02,0.99440000D+00,0.00000000D+00
     .,0.10674380D+03,0.107D+03,0.460D+02,0.99440000D+00,0.00000000D+00
     .,0.89341900D+02,0.107D+03,0.470D+02,0.99440000D+00,0.00000000D+00
     .,0.94877000D+02,0.107D+03,0.480D+02,0.99440000D+00,0.00000000D+00
     .,0.11649590D+03,0.107D+03,0.490D+02,0.99440000D+00,0.00000000D+00
     .,0.11009510D+03,0.107D+03,0.500D+02,0.99440000D+00,0.00000000D+00
     ./)
      pars(  25001:  25100)=(/
     . 0.10051820D+03,0.107D+03,0.510D+02,0.99440000D+00,0.00000000D+00
     .,0.94750800D+02,0.107D+03,0.520D+02,0.99440000D+00,0.00000000D+00
     .,0.87202800D+02,0.107D+03,0.530D+02,0.99440000D+00,0.00000000D+00
     .,0.79777300D+02,0.107D+03,0.540D+02,0.99440000D+00,0.00000000D+00
     .,0.27525670D+03,0.107D+03,0.550D+02,0.99440000D+00,0.00000000D+00
     .,0.26044810D+03,0.107D+03,0.560D+02,0.99440000D+00,0.00000000D+00
     .,0.23366190D+03,0.107D+03,0.570D+02,0.99440000D+00,0.00000000D+00
     .,0.11854700D+03,0.107D+03,0.580D+02,0.99440000D+00,0.27991000D+01
     .,0.23257830D+03,0.107D+03,0.590D+02,0.99440000D+00,0.00000000D+00
     .,0.22400400D+03,0.107D+03,0.600D+02,0.99440000D+00,0.00000000D+00
     .,0.21856110D+03,0.107D+03,0.610D+02,0.99440000D+00,0.00000000D+00
     .,0.21352770D+03,0.107D+03,0.620D+02,0.99440000D+00,0.00000000D+00
     .,0.20907000D+03,0.107D+03,0.630D+02,0.99440000D+00,0.00000000D+00
     .,0.16905560D+03,0.107D+03,0.640D+02,0.99440000D+00,0.00000000D+00
     .,0.18459040D+03,0.107D+03,0.650D+02,0.99440000D+00,0.00000000D+00
     .,0.17880740D+03,0.107D+03,0.660D+02,0.99440000D+00,0.00000000D+00
     .,0.18945310D+03,0.107D+03,0.670D+02,0.99440000D+00,0.00000000D+00
     .,0.18549310D+03,0.107D+03,0.680D+02,0.99440000D+00,0.00000000D+00
     .,0.18198230D+03,0.107D+03,0.690D+02,0.99440000D+00,0.00000000D+00
     .,0.17962020D+03,0.107D+03,0.700D+02,0.99440000D+00,0.00000000D+00
     ./)
      pars(  25101:  25200)=(/
     . 0.15422920D+03,0.107D+03,0.710D+02,0.99440000D+00,0.00000000D+00
     .,0.15491020D+03,0.107D+03,0.720D+02,0.99440000D+00,0.00000000D+00
     .,0.14357630D+03,0.107D+03,0.730D+02,0.99440000D+00,0.00000000D+00
     .,0.12323020D+03,0.107D+03,0.740D+02,0.99440000D+00,0.00000000D+00
     .,0.12591210D+03,0.107D+03,0.750D+02,0.99440000D+00,0.00000000D+00
     .,0.11569380D+03,0.107D+03,0.760D+02,0.99440000D+00,0.00000000D+00
     .,0.10717750D+03,0.107D+03,0.770D+02,0.99440000D+00,0.00000000D+00
     .,0.90445600D+02,0.107D+03,0.780D+02,0.99440000D+00,0.00000000D+00
     .,0.85014600D+02,0.107D+03,0.790D+02,0.99440000D+00,0.00000000D+00
     .,0.87649000D+02,0.107D+03,0.800D+02,0.99440000D+00,0.00000000D+00
     .,0.12108000D+03,0.107D+03,0.810D+02,0.99440000D+00,0.00000000D+00
     .,0.12012680D+03,0.107D+03,0.820D+02,0.99440000D+00,0.00000000D+00
     .,0.11264850D+03,0.107D+03,0.830D+02,0.99440000D+00,0.00000000D+00
     .,0.10881340D+03,0.107D+03,0.840D+02,0.99440000D+00,0.00000000D+00
     .,0.10205240D+03,0.107D+03,0.850D+02,0.99440000D+00,0.00000000D+00
     .,0.94992800D+02,0.107D+03,0.860D+02,0.99440000D+00,0.00000000D+00
     .,0.26444300D+03,0.107D+03,0.870D+02,0.99440000D+00,0.00000000D+00
     .,0.26065290D+03,0.107D+03,0.880D+02,0.99440000D+00,0.00000000D+00
     .,0.23476520D+03,0.107D+03,0.890D+02,0.99440000D+00,0.00000000D+00
     .,0.21630740D+03,0.107D+03,0.900D+02,0.99440000D+00,0.00000000D+00
     ./)
      pars(  25201:  25300)=(/
     . 0.21274560D+03,0.107D+03,0.910D+02,0.99440000D+00,0.00000000D+00
     .,0.20616310D+03,0.107D+03,0.920D+02,0.99440000D+00,0.00000000D+00
     .,0.20915240D+03,0.107D+03,0.930D+02,0.99440000D+00,0.00000000D+00
     .,0.20304180D+03,0.107D+03,0.940D+02,0.99440000D+00,0.00000000D+00
     .,0.12598000D+02,0.107D+03,0.101D+03,0.99440000D+00,0.00000000D+00
     .,0.37493900D+02,0.107D+03,0.103D+03,0.99440000D+00,0.98650000D+00
     .,0.48488500D+02,0.107D+03,0.104D+03,0.99440000D+00,0.98080000D+00
     .,0.39043300D+02,0.107D+03,0.105D+03,0.99440000D+00,0.97060000D+00
     .,0.30541000D+02,0.107D+03,0.106D+03,0.99440000D+00,0.98680000D+00
     .,0.22124100D+02,0.107D+03,0.107D+03,0.99440000D+00,0.99440000D+00
     .,0.60575000D+01,0.108D+03,0.100D+01,0.99250000D+00,0.91180000D+00
     .,0.44593000D+01,0.108D+03,0.200D+01,0.99250000D+00,0.00000000D+00
     .,0.67931200D+02,0.108D+03,0.300D+01,0.99250000D+00,0.00000000D+00
     .,0.44496800D+02,0.108D+03,0.400D+01,0.99250000D+00,0.00000000D+00
     .,0.32813600D+02,0.108D+03,0.500D+01,0.99250000D+00,0.00000000D+00
     .,0.23912000D+02,0.108D+03,0.600D+01,0.99250000D+00,0.00000000D+00
     .,0.17769800D+02,0.108D+03,0.700D+01,0.99250000D+00,0.00000000D+00
     .,0.14076400D+02,0.108D+03,0.800D+01,0.99250000D+00,0.00000000D+00
     .,0.11099400D+02,0.108D+03,0.900D+01,0.99250000D+00,0.00000000D+00
     .,0.88252000D+01,0.108D+03,0.100D+02,0.99250000D+00,0.00000000D+00
     ./)
      pars(  25301:  25400)=(/
     . 0.82182000D+02,0.108D+03,0.110D+02,0.99250000D+00,0.00000000D+00
     .,0.69367500D+02,0.108D+03,0.120D+02,0.99250000D+00,0.00000000D+00
     .,0.66745100D+02,0.108D+03,0.130D+02,0.99250000D+00,0.00000000D+00
     .,0.55817500D+02,0.108D+03,0.140D+02,0.99250000D+00,0.00000000D+00
     .,0.45993700D+02,0.108D+03,0.150D+02,0.99250000D+00,0.00000000D+00
     .,0.39709100D+02,0.108D+03,0.160D+02,0.99250000D+00,0.00000000D+00
     .,0.33731300D+02,0.108D+03,0.170D+02,0.99250000D+00,0.00000000D+00
     .,0.28600000D+02,0.108D+03,0.180D+02,0.99250000D+00,0.00000000D+00
     .,0.13465080D+03,0.108D+03,0.190D+02,0.99250000D+00,0.00000000D+00
     .,0.11824110D+03,0.108D+03,0.200D+02,0.99250000D+00,0.00000000D+00
     .,0.99293900D+02,0.108D+03,0.210D+02,0.99250000D+00,0.00000000D+00
     .,0.97834500D+02,0.108D+03,0.220D+02,0.99250000D+00,0.00000000D+00
     .,0.90600700D+02,0.108D+03,0.230D+02,0.99250000D+00,0.00000000D+00
     .,0.72287200D+02,0.108D+03,0.240D+02,0.99250000D+00,0.00000000D+00
     .,0.79304700D+02,0.108D+03,0.250D+02,0.99250000D+00,0.00000000D+00
     .,0.63186500D+02,0.108D+03,0.260D+02,0.99250000D+00,0.00000000D+00
     .,0.67711700D+02,0.108D+03,0.270D+02,0.99250000D+00,0.00000000D+00
     .,0.68908600D+02,0.108D+03,0.280D+02,0.99250000D+00,0.00000000D+00
     .,0.53642000D+02,0.108D+03,0.290D+02,0.99250000D+00,0.00000000D+00
     .,0.56288400D+02,0.108D+03,0.300D+02,0.99250000D+00,0.00000000D+00
     ./)
      pars(  25401:  25500)=(/
     . 0.65621400D+02,0.108D+03,0.310D+02,0.99250000D+00,0.00000000D+00
     .,0.60404100D+02,0.108D+03,0.320D+02,0.99250000D+00,0.00000000D+00
     .,0.53815300D+02,0.108D+03,0.330D+02,0.99250000D+00,0.00000000D+00
     .,0.49755200D+02,0.108D+03,0.340D+02,0.99250000D+00,0.00000000D+00
     .,0.44951000D+02,0.108D+03,0.350D+02,0.99250000D+00,0.00000000D+00
     .,0.40304300D+02,0.108D+03,0.360D+02,0.99250000D+00,0.00000000D+00
     .,0.15267570D+03,0.108D+03,0.370D+02,0.99250000D+00,0.00000000D+00
     .,0.14115920D+03,0.108D+03,0.380D+02,0.99250000D+00,0.00000000D+00
     .,0.12762820D+03,0.108D+03,0.390D+02,0.99250000D+00,0.00000000D+00
     .,0.11723280D+03,0.108D+03,0.400D+02,0.99250000D+00,0.00000000D+00
     .,0.10863780D+03,0.108D+03,0.410D+02,0.99250000D+00,0.00000000D+00
     .,0.86680300D+02,0.108D+03,0.420D+02,0.99250000D+00,0.00000000D+00
     .,0.95496600D+02,0.108D+03,0.430D+02,0.99250000D+00,0.00000000D+00
     .,0.75391000D+02,0.108D+03,0.440D+02,0.99250000D+00,0.00000000D+00
     .,0.81764000D+02,0.108D+03,0.450D+02,0.99250000D+00,0.00000000D+00
     .,0.76587100D+02,0.108D+03,0.460D+02,0.99250000D+00,0.00000000D+00
     .,0.64388000D+02,0.108D+03,0.470D+02,0.99250000D+00,0.00000000D+00
     .,0.68394300D+02,0.108D+03,0.480D+02,0.99250000D+00,0.00000000D+00
     .,0.83027800D+02,0.108D+03,0.490D+02,0.99250000D+00,0.00000000D+00
     .,0.79170400D+02,0.108D+03,0.500D+02,0.99250000D+00,0.00000000D+00
     ./)
      pars(  25501:  25600)=(/
     . 0.73071800D+02,0.108D+03,0.510D+02,0.99250000D+00,0.00000000D+00
     .,0.69390100D+02,0.108D+03,0.520D+02,0.99250000D+00,0.00000000D+00
     .,0.64414100D+02,0.108D+03,0.530D+02,0.99250000D+00,0.00000000D+00
     .,0.59446800D+02,0.108D+03,0.540D+02,0.99250000D+00,0.00000000D+00
     .,0.18687910D+03,0.108D+03,0.550D+02,0.99250000D+00,0.00000000D+00
     .,0.17895490D+03,0.108D+03,0.560D+02,0.99250000D+00,0.00000000D+00
     .,0.16203710D+03,0.108D+03,0.570D+02,0.99250000D+00,0.00000000D+00
     .,0.86166500D+02,0.108D+03,0.580D+02,0.99250000D+00,0.27991000D+01
     .,0.16052330D+03,0.108D+03,0.590D+02,0.99250000D+00,0.00000000D+00
     .,0.15480670D+03,0.108D+03,0.600D+02,0.99250000D+00,0.00000000D+00
     .,0.15109700D+03,0.108D+03,0.610D+02,0.99250000D+00,0.00000000D+00
     .,0.14765520D+03,0.108D+03,0.620D+02,0.99250000D+00,0.00000000D+00
     .,0.14460750D+03,0.108D+03,0.630D+02,0.99250000D+00,0.00000000D+00
     .,0.11853880D+03,0.108D+03,0.640D+02,0.99250000D+00,0.00000000D+00
     .,0.12778280D+03,0.108D+03,0.650D+02,0.99250000D+00,0.00000000D+00
     .,0.12401580D+03,0.108D+03,0.660D+02,0.99250000D+00,0.00000000D+00
     .,0.13130040D+03,0.108D+03,0.670D+02,0.99250000D+00,0.00000000D+00
     .,0.12856660D+03,0.108D+03,0.680D+02,0.99250000D+00,0.00000000D+00
     .,0.12616140D+03,0.108D+03,0.690D+02,0.99250000D+00,0.00000000D+00
     .,0.12444020D+03,0.108D+03,0.700D+02,0.99250000D+00,0.00000000D+00
     ./)
      pars(  25601:  25700)=(/
     . 0.10783510D+03,0.108D+03,0.710D+02,0.99250000D+00,0.00000000D+00
     .,0.10920420D+03,0.108D+03,0.720D+02,0.99250000D+00,0.00000000D+00
     .,0.10197340D+03,0.108D+03,0.730D+02,0.99250000D+00,0.00000000D+00
     .,0.88315600D+02,0.108D+03,0.740D+02,0.99250000D+00,0.00000000D+00
     .,0.90391800D+02,0.108D+03,0.750D+02,0.99250000D+00,0.00000000D+00
     .,0.83622700D+02,0.108D+03,0.760D+02,0.99250000D+00,0.00000000D+00
     .,0.77919400D+02,0.108D+03,0.770D+02,0.99250000D+00,0.00000000D+00
     .,0.66359100D+02,0.108D+03,0.780D+02,0.99250000D+00,0.00000000D+00
     .,0.62600800D+02,0.108D+03,0.790D+02,0.99250000D+00,0.00000000D+00
     .,0.64555200D+02,0.108D+03,0.800D+02,0.99250000D+00,0.00000000D+00
     .,0.86955200D+02,0.108D+03,0.810D+02,0.99250000D+00,0.00000000D+00
     .,0.86739200D+02,0.108D+03,0.820D+02,0.99250000D+00,0.00000000D+00
     .,0.82055900D+02,0.108D+03,0.830D+02,0.99250000D+00,0.00000000D+00
     .,0.79713300D+02,0.108D+03,0.840D+02,0.99250000D+00,0.00000000D+00
     .,0.75326800D+02,0.108D+03,0.850D+02,0.99250000D+00,0.00000000D+00
     .,0.70648000D+02,0.108D+03,0.860D+02,0.99250000D+00,0.00000000D+00
     .,0.18100550D+03,0.108D+03,0.870D+02,0.99250000D+00,0.00000000D+00
     .,0.18009880D+03,0.108D+03,0.880D+02,0.99250000D+00,0.00000000D+00
     .,0.16360420D+03,0.108D+03,0.890D+02,0.99250000D+00,0.00000000D+00
     .,0.15256500D+03,0.108D+03,0.900D+02,0.99250000D+00,0.00000000D+00
     ./)
      pars(  25701:  25800)=(/
     . 0.14955510D+03,0.108D+03,0.910D+02,0.99250000D+00,0.00000000D+00
     .,0.14501130D+03,0.108D+03,0.920D+02,0.99250000D+00,0.00000000D+00
     .,0.14619280D+03,0.108D+03,0.930D+02,0.99250000D+00,0.00000000D+00
     .,0.14207750D+03,0.108D+03,0.940D+02,0.99250000D+00,0.00000000D+00
     .,0.91812000D+01,0.108D+03,0.101D+03,0.99250000D+00,0.00000000D+00
     .,0.26270800D+02,0.108D+03,0.103D+03,0.99250000D+00,0.98650000D+00
     .,0.34198300D+02,0.108D+03,0.104D+03,0.99250000D+00,0.98080000D+00
     .,0.28232700D+02,0.108D+03,0.105D+03,0.99250000D+00,0.97060000D+00
     .,0.22517800D+02,0.108D+03,0.106D+03,0.99250000D+00,0.98680000D+00
     .,0.16677500D+02,0.108D+03,0.107D+03,0.99250000D+00,0.99440000D+00
     .,0.12816100D+02,0.108D+03,0.108D+03,0.99250000D+00,0.99250000D+00
     .,0.42672000D+01,0.109D+03,0.100D+01,0.99820000D+00,0.91180000D+00
     .,0.33077000D+01,0.109D+03,0.200D+01,0.99820000D+00,0.00000000D+00
     .,0.42062700D+02,0.109D+03,0.300D+01,0.99820000D+00,0.00000000D+00
     .,0.28861500D+02,0.109D+03,0.400D+01,0.99820000D+00,0.00000000D+00
     .,0.22093100D+02,0.109D+03,0.500D+01,0.99820000D+00,0.00000000D+00
     .,0.16637300D+02,0.109D+03,0.600D+01,0.99820000D+00,0.00000000D+00
     .,0.12710700D+02,0.109D+03,0.700D+01,0.99820000D+00,0.00000000D+00
     .,0.10286700D+02,0.109D+03,0.800D+01,0.99820000D+00,0.00000000D+00
     .,0.82739000D+01,0.109D+03,0.900D+01,0.99820000D+00,0.00000000D+00
     ./)
      pars(  25801:  25900)=(/
     . 0.66950000D+01,0.109D+03,0.100D+02,0.99820000D+00,0.00000000D+00
     .,0.51249800D+02,0.109D+03,0.110D+02,0.99820000D+00,0.00000000D+00
     .,0.44714100D+02,0.109D+03,0.120D+02,0.99820000D+00,0.00000000D+00
     .,0.43736400D+02,0.109D+03,0.130D+02,0.99820000D+00,0.00000000D+00
     .,0.37442100D+02,0.109D+03,0.140D+02,0.99820000D+00,0.00000000D+00
     .,0.31557700D+02,0.109D+03,0.150D+02,0.99820000D+00,0.00000000D+00
     .,0.27713700D+02,0.109D+03,0.160D+02,0.99820000D+00,0.00000000D+00
     .,0.23950500D+02,0.109D+03,0.170D+02,0.99820000D+00,0.00000000D+00
     .,0.20639400D+02,0.109D+03,0.180D+02,0.99820000D+00,0.00000000D+00
     .,0.84368800D+02,0.109D+03,0.190D+02,0.99820000D+00,0.00000000D+00
     .,0.75585000D+02,0.109D+03,0.200D+02,0.99820000D+00,0.00000000D+00
     .,0.63856400D+02,0.109D+03,0.210D+02,0.99820000D+00,0.00000000D+00
     .,0.63474800D+02,0.109D+03,0.220D+02,0.99820000D+00,0.00000000D+00
     .,0.59066500D+02,0.109D+03,0.230D+02,0.99820000D+00,0.00000000D+00
     .,0.47531800D+02,0.109D+03,0.240D+02,0.99820000D+00,0.00000000D+00
     .,0.52078000D+02,0.109D+03,0.250D+02,0.99820000D+00,0.00000000D+00
     .,0.41894200D+02,0.109D+03,0.260D+02,0.99820000D+00,0.00000000D+00
     .,0.44940700D+02,0.109D+03,0.270D+02,0.99820000D+00,0.00000000D+00
     .,0.45515500D+02,0.109D+03,0.280D+02,0.99820000D+00,0.00000000D+00
     .,0.35810000D+02,0.109D+03,0.290D+02,0.99820000D+00,0.00000000D+00
     ./)
      pars(  25901:  26000)=(/
     . 0.37755900D+02,0.109D+03,0.300D+02,0.99820000D+00,0.00000000D+00
     .,0.43568500D+02,0.109D+03,0.310D+02,0.99820000D+00,0.00000000D+00
     .,0.40750000D+02,0.109D+03,0.320D+02,0.99820000D+00,0.00000000D+00
     .,0.36938900D+02,0.109D+03,0.330D+02,0.99820000D+00,0.00000000D+00
     .,0.34579200D+02,0.109D+03,0.340D+02,0.99820000D+00,0.00000000D+00
     .,0.31669900D+02,0.109D+03,0.350D+02,0.99820000D+00,0.00000000D+00
     .,0.28780000D+02,0.109D+03,0.360D+02,0.99820000D+00,0.00000000D+00
     .,0.96223500D+02,0.109D+03,0.370D+02,0.99820000D+00,0.00000000D+00
     .,0.90408100D+02,0.109D+03,0.380D+02,0.99820000D+00,0.00000000D+00
     .,0.82733700D+02,0.109D+03,0.390D+02,0.99820000D+00,0.00000000D+00
     .,0.76671200D+02,0.109D+03,0.400D+02,0.99820000D+00,0.00000000D+00
     .,0.71549300D+02,0.109D+03,0.410D+02,0.99820000D+00,0.00000000D+00
     .,0.57978400D+02,0.109D+03,0.420D+02,0.99820000D+00,0.00000000D+00
     .,0.63507900D+02,0.109D+03,0.430D+02,0.99820000D+00,0.00000000D+00
     .,0.50977000D+02,0.109D+03,0.440D+02,0.99820000D+00,0.00000000D+00
     .,0.55024000D+02,0.109D+03,0.450D+02,0.99820000D+00,0.00000000D+00
     .,0.51774000D+02,0.109D+03,0.460D+02,0.99820000D+00,0.00000000D+00
     .,0.43848000D+02,0.109D+03,0.470D+02,0.99820000D+00,0.00000000D+00
     .,0.46534000D+02,0.109D+03,0.480D+02,0.99820000D+00,0.00000000D+00
     .,0.55658900D+02,0.109D+03,0.490D+02,0.99820000D+00,0.00000000D+00
     ./)
      pars(  26001:  26100)=(/
     . 0.53624300D+02,0.109D+03,0.500D+02,0.99820000D+00,0.00000000D+00
     .,0.50147300D+02,0.109D+03,0.510D+02,0.99820000D+00,0.00000000D+00
     .,0.48053500D+02,0.109D+03,0.520D+02,0.99820000D+00,0.00000000D+00
     .,0.45084800D+02,0.109D+03,0.530D+02,0.99820000D+00,0.00000000D+00
     .,0.42063400D+02,0.109D+03,0.540D+02,0.99820000D+00,0.00000000D+00
     .,0.11797180D+03,0.109D+03,0.550D+02,0.99820000D+00,0.00000000D+00
     .,0.11446760D+03,0.109D+03,0.560D+02,0.99820000D+00,0.00000000D+00
     .,0.10480950D+03,0.109D+03,0.570D+02,0.99820000D+00,0.00000000D+00
     .,0.59137900D+02,0.109D+03,0.580D+02,0.99820000D+00,0.27991000D+01
     .,0.10336160D+03,0.109D+03,0.590D+02,0.99820000D+00,0.00000000D+00
     .,0.99833100D+02,0.109D+03,0.600D+02,0.99820000D+00,0.00000000D+00
     .,0.97478900D+02,0.109D+03,0.610D+02,0.99820000D+00,0.00000000D+00
     .,0.95284600D+02,0.109D+03,0.620D+02,0.99820000D+00,0.00000000D+00
     .,0.93341000D+02,0.109D+03,0.630D+02,0.99820000D+00,0.00000000D+00
     .,0.77874200D+02,0.109D+03,0.640D+02,0.99820000D+00,0.00000000D+00
     .,0.82715400D+02,0.109D+03,0.650D+02,0.99820000D+00,0.00000000D+00
     .,0.80460700D+02,0.109D+03,0.660D+02,0.99820000D+00,0.00000000D+00
     .,0.84947800D+02,0.109D+03,0.670D+02,0.99820000D+00,0.00000000D+00
     .,0.83182700D+02,0.109D+03,0.680D+02,0.99820000D+00,0.00000000D+00
     .,0.81643700D+02,0.109D+03,0.690D+02,0.99820000D+00,0.00000000D+00
     ./)
      pars(  26101:  26200)=(/
     . 0.80457500D+02,0.109D+03,0.700D+02,0.99820000D+00,0.00000000D+00
     .,0.70552200D+02,0.109D+03,0.710D+02,0.99820000D+00,0.00000000D+00
     .,0.72088700D+02,0.109D+03,0.720D+02,0.99820000D+00,0.00000000D+00
     .,0.67959000D+02,0.109D+03,0.730D+02,0.99820000D+00,0.00000000D+00
     .,0.59575800D+02,0.109D+03,0.740D+02,0.99820000D+00,0.00000000D+00
     .,0.61094900D+02,0.109D+03,0.750D+02,0.99820000D+00,0.00000000D+00
     .,0.57011100D+02,0.109D+03,0.760D+02,0.99820000D+00,0.00000000D+00
     .,0.53521400D+02,0.109D+03,0.770D+02,0.99820000D+00,0.00000000D+00
     .,0.46144200D+02,0.109D+03,0.780D+02,0.99820000D+00,0.00000000D+00
     .,0.43746800D+02,0.109D+03,0.790D+02,0.99820000D+00,0.00000000D+00
     .,0.45120500D+02,0.109D+03,0.800D+02,0.99820000D+00,0.00000000D+00
     .,0.58912900D+02,0.109D+03,0.810D+02,0.99820000D+00,0.00000000D+00
     .,0.59109700D+02,0.109D+03,0.820D+02,0.99820000D+00,0.00000000D+00
     .,0.56500700D+02,0.109D+03,0.830D+02,0.99820000D+00,0.00000000D+00
     .,0.55261400D+02,0.109D+03,0.840D+02,0.99820000D+00,0.00000000D+00
     .,0.52703300D+02,0.109D+03,0.850D+02,0.99820000D+00,0.00000000D+00
     .,0.49892700D+02,0.109D+03,0.860D+02,0.99820000D+00,0.00000000D+00
     .,0.11543210D+03,0.109D+03,0.870D+02,0.99820000D+00,0.00000000D+00
     .,0.11600620D+03,0.109D+03,0.880D+02,0.99820000D+00,0.00000000D+00
     .,0.10653030D+03,0.109D+03,0.890D+02,0.99820000D+00,0.00000000D+00
     ./)
      pars(  26201:  26300)=(/
     . 0.10090290D+03,0.109D+03,0.900D+02,0.99820000D+00,0.00000000D+00
     .,0.98576800D+02,0.109D+03,0.910D+02,0.99820000D+00,0.00000000D+00
     .,0.95662500D+02,0.109D+03,0.920D+02,0.99820000D+00,0.00000000D+00
     .,0.95732200D+02,0.109D+03,0.930D+02,0.99820000D+00,0.00000000D+00
     .,0.93161400D+02,0.109D+03,0.940D+02,0.99820000D+00,0.00000000D+00
     .,0.63000000D+01,0.109D+03,0.101D+03,0.99820000D+00,0.00000000D+00
     .,0.17173300D+02,0.109D+03,0.103D+03,0.99820000D+00,0.98650000D+00
     .,0.22541300D+02,0.109D+03,0.104D+03,0.99820000D+00,0.98080000D+00
     .,0.19202900D+02,0.109D+03,0.105D+03,0.99820000D+00,0.97060000D+00
     .,0.15685700D+02,0.109D+03,0.106D+03,0.99820000D+00,0.98680000D+00
     .,0.11947000D+02,0.109D+03,0.107D+03,0.99820000D+00,0.99440000D+00
     .,0.94048000D+01,0.109D+03,0.108D+03,0.99820000D+00,0.99250000D+00
     .,0.71341000D+01,0.109D+03,0.109D+03,0.99820000D+00,0.99820000D+00
     .,0.20756700D+02,0.111D+03,0.100D+01,0.96840000D+00,0.91180000D+00
     .,0.12728700D+02,0.111D+03,0.200D+01,0.96840000D+00,0.00000000D+00
     .,0.42571610D+03,0.111D+03,0.300D+01,0.96840000D+00,0.00000000D+00
     .,0.21865720D+03,0.111D+03,0.400D+01,0.96840000D+00,0.00000000D+00
     .,0.13695360D+03,0.111D+03,0.500D+01,0.96840000D+00,0.00000000D+00
     .,0.87395600D+02,0.111D+03,0.600D+01,0.96840000D+00,0.00000000D+00
     .,0.58485300D+02,0.111D+03,0.700D+01,0.96840000D+00,0.00000000D+00
     ./)
      pars(  26301:  26400)=(/
     . 0.42891100D+02,0.111D+03,0.800D+01,0.96840000D+00,0.00000000D+00
     .,0.31606000D+02,0.111D+03,0.900D+01,0.96840000D+00,0.00000000D+00
     .,0.23774500D+02,0.111D+03,0.100D+02,0.96840000D+00,0.00000000D+00
     .,0.50543630D+03,0.111D+03,0.110D+02,0.96840000D+00,0.00000000D+00
     .,0.35591800D+03,0.111D+03,0.120D+02,0.96840000D+00,0.00000000D+00
     .,0.31645880D+03,0.111D+03,0.130D+02,0.96840000D+00,0.00000000D+00
     .,0.23713260D+03,0.111D+03,0.140D+02,0.96840000D+00,0.00000000D+00
     .,0.17692050D+03,0.111D+03,0.150D+02,0.96840000D+00,0.00000000D+00
     .,0.14254300D+03,0.111D+03,0.160D+02,0.96840000D+00,0.00000000D+00
     .,0.11310250D+03,0.111D+03,0.170D+02,0.96840000D+00,0.00000000D+00
     .,0.90173500D+02,0.111D+03,0.180D+02,0.96840000D+00,0.00000000D+00
     .,0.84082990D+03,0.111D+03,0.190D+02,0.96840000D+00,0.00000000D+00
     .,0.64831050D+03,0.111D+03,0.200D+02,0.96840000D+00,0.00000000D+00
     .,0.52709170D+03,0.111D+03,0.210D+02,0.96840000D+00,0.00000000D+00
     .,0.50123930D+03,0.111D+03,0.220D+02,0.96840000D+00,0.00000000D+00
     .,0.45480300D+03,0.111D+03,0.230D+02,0.96840000D+00,0.00000000D+00
     .,0.35715080D+03,0.111D+03,0.240D+02,0.96840000D+00,0.00000000D+00
     .,0.38634260D+03,0.111D+03,0.250D+02,0.96840000D+00,0.00000000D+00
     .,0.30174010D+03,0.111D+03,0.260D+02,0.96840000D+00,0.00000000D+00
     .,0.31417180D+03,0.111D+03,0.270D+02,0.96840000D+00,0.00000000D+00
     ./)
      pars(  26401:  26500)=(/
     . 0.32682710D+03,0.111D+03,0.280D+02,0.96840000D+00,0.00000000D+00
     .,0.24981970D+03,0.111D+03,0.290D+02,0.96840000D+00,0.00000000D+00
     .,0.24939250D+03,0.111D+03,0.300D+02,0.96840000D+00,0.00000000D+00
     .,0.29840070D+03,0.111D+03,0.310D+02,0.96840000D+00,0.00000000D+00
     .,0.25314330D+03,0.111D+03,0.320D+02,0.96840000D+00,0.00000000D+00
     .,0.20818020D+03,0.111D+03,0.330D+02,0.96840000D+00,0.00000000D+00
     .,0.18247440D+03,0.111D+03,0.340D+02,0.96840000D+00,0.00000000D+00
     .,0.15582020D+03,0.111D+03,0.350D+02,0.96840000D+00,0.00000000D+00
     .,0.13243480D+03,0.111D+03,0.360D+02,0.96840000D+00,0.00000000D+00
     .,0.93630060D+03,0.111D+03,0.370D+02,0.96840000D+00,0.00000000D+00
     .,0.77398540D+03,0.111D+03,0.380D+02,0.96840000D+00,0.00000000D+00
     .,0.65879630D+03,0.111D+03,0.390D+02,0.96840000D+00,0.00000000D+00
     .,0.58124800D+03,0.111D+03,0.400D+02,0.96840000D+00,0.00000000D+00
     .,0.52340580D+03,0.111D+03,0.410D+02,0.96840000D+00,0.00000000D+00
     .,0.39500580D+03,0.111D+03,0.420D+02,0.96840000D+00,0.00000000D+00
     .,0.44452690D+03,0.111D+03,0.430D+02,0.96840000D+00,0.00000000D+00
     .,0.33022730D+03,0.111D+03,0.440D+02,0.96840000D+00,0.00000000D+00
     .,0.36142870D+03,0.111D+03,0.450D+02,0.96840000D+00,0.00000000D+00
     .,0.33243920D+03,0.111D+03,0.460D+02,0.96840000D+00,0.00000000D+00
     .,0.27794280D+03,0.111D+03,0.470D+02,0.96840000D+00,0.00000000D+00
     ./)
      pars(  26501:  26600)=(/
     . 0.28984740D+03,0.111D+03,0.480D+02,0.96840000D+00,0.00000000D+00
     .,0.37337940D+03,0.111D+03,0.490D+02,0.96840000D+00,0.00000000D+00
     .,0.33470230D+03,0.111D+03,0.500D+02,0.96840000D+00,0.00000000D+00
     .,0.28898880D+03,0.111D+03,0.510D+02,0.96840000D+00,0.00000000D+00
     .,0.26288460D+03,0.111D+03,0.520D+02,0.96840000D+00,0.00000000D+00
     .,0.23260530D+03,0.111D+03,0.530D+02,0.96840000D+00,0.00000000D+00
     .,0.20479860D+03,0.111D+03,0.540D+02,0.96840000D+00,0.00000000D+00
     .,0.11389256D+04,0.111D+03,0.550D+02,0.96840000D+00,0.00000000D+00
     .,0.99536600D+03,0.111D+03,0.560D+02,0.96840000D+00,0.00000000D+00
     .,0.85007680D+03,0.111D+03,0.570D+02,0.96840000D+00,0.00000000D+00
     .,0.34313820D+03,0.111D+03,0.580D+02,0.96840000D+00,0.27991000D+01
     .,0.87351340D+03,0.111D+03,0.590D+02,0.96840000D+00,0.00000000D+00
     .,0.83492630D+03,0.111D+03,0.600D+02,0.96840000D+00,0.00000000D+00
     .,0.81293500D+03,0.111D+03,0.610D+02,0.96840000D+00,0.00000000D+00
     .,0.79283360D+03,0.111D+03,0.620D+02,0.96840000D+00,0.00000000D+00
     .,0.77497190D+03,0.111D+03,0.630D+02,0.96840000D+00,0.00000000D+00
     .,0.58968210D+03,0.111D+03,0.640D+02,0.96840000D+00,0.00000000D+00
     .,0.69502110D+03,0.111D+03,0.650D+02,0.96840000D+00,0.00000000D+00
     .,0.66665150D+03,0.111D+03,0.660D+02,0.96840000D+00,0.00000000D+00
     .,0.69407420D+03,0.111D+03,0.670D+02,0.96840000D+00,0.00000000D+00
     ./)
      pars(  26601:  26700)=(/
     . 0.67884860D+03,0.111D+03,0.680D+02,0.96840000D+00,0.00000000D+00
     .,0.66485800D+03,0.111D+03,0.690D+02,0.96840000D+00,0.00000000D+00
     .,0.65799220D+03,0.111D+03,0.700D+02,0.96840000D+00,0.00000000D+00
     .,0.54208720D+03,0.111D+03,0.710D+02,0.96840000D+00,0.00000000D+00
     .,0.51742950D+03,0.111D+03,0.720D+02,0.96840000D+00,0.00000000D+00
     .,0.46327240D+03,0.111D+03,0.730D+02,0.96840000D+00,0.00000000D+00
     .,0.38490910D+03,0.111D+03,0.740D+02,0.96840000D+00,0.00000000D+00
     .,0.38879710D+03,0.111D+03,0.750D+02,0.96840000D+00,0.00000000D+00
     .,0.34639970D+03,0.111D+03,0.760D+02,0.96840000D+00,0.00000000D+00
     .,0.31281320D+03,0.111D+03,0.770D+02,0.96840000D+00,0.00000000D+00
     .,0.25572890D+03,0.111D+03,0.780D+02,0.96840000D+00,0.00000000D+00
     .,0.23740440D+03,0.111D+03,0.790D+02,0.96840000D+00,0.00000000D+00
     .,0.24279190D+03,0.111D+03,0.800D+02,0.96840000D+00,0.00000000D+00
     .,0.37972090D+03,0.111D+03,0.810D+02,0.96840000D+00,0.00000000D+00
     .,0.36256490D+03,0.111D+03,0.820D+02,0.96840000D+00,0.00000000D+00
     .,0.32393500D+03,0.111D+03,0.830D+02,0.96840000D+00,0.00000000D+00
     .,0.30383030D+03,0.111D+03,0.840D+02,0.96840000D+00,0.00000000D+00
     .,0.27473150D+03,0.111D+03,0.850D+02,0.96840000D+00,0.00000000D+00
     .,0.24706350D+03,0.111D+03,0.860D+02,0.96840000D+00,0.00000000D+00
     .,0.10521318D+04,0.111D+03,0.870D+02,0.96840000D+00,0.00000000D+00
     ./)
      pars(  26701:  26800)=(/
     . 0.96974210D+03,0.111D+03,0.880D+02,0.96840000D+00,0.00000000D+00
     .,0.83486500D+03,0.111D+03,0.890D+02,0.96840000D+00,0.00000000D+00
     .,0.72693870D+03,0.111D+03,0.900D+02,0.96840000D+00,0.00000000D+00
     .,0.73265540D+03,0.111D+03,0.910D+02,0.96840000D+00,0.00000000D+00
     .,0.70879360D+03,0.111D+03,0.920D+02,0.96840000D+00,0.00000000D+00
     .,0.74391900D+03,0.111D+03,0.930D+02,0.96840000D+00,0.00000000D+00
     .,0.71791000D+03,0.111D+03,0.940D+02,0.96840000D+00,0.00000000D+00
     .,0.34995200D+02,0.111D+03,0.101D+03,0.96840000D+00,0.00000000D+00
     .,0.12582450D+03,0.111D+03,0.103D+03,0.96840000D+00,0.98650000D+00
     .,0.15832680D+03,0.111D+03,0.104D+03,0.96840000D+00,0.98080000D+00
     .,0.11340160D+03,0.111D+03,0.105D+03,0.96840000D+00,0.97060000D+00
     .,0.82014400D+02,0.111D+03,0.106D+03,0.96840000D+00,0.98680000D+00
     .,0.54445400D+02,0.111D+03,0.107D+03,0.96840000D+00,0.99440000D+00
     .,0.38122700D+02,0.111D+03,0.108D+03,0.96840000D+00,0.99250000D+00
     .,0.24956000D+02,0.111D+03,0.109D+03,0.96840000D+00,0.99820000D+00
     .,0.18610520D+03,0.111D+03,0.111D+03,0.96840000D+00,0.96840000D+00
     .,0.31991300D+02,0.112D+03,0.100D+01,0.96280000D+00,0.91180000D+00
     .,0.19464800D+02,0.112D+03,0.200D+01,0.96280000D+00,0.00000000D+00
     .,0.67003300D+03,0.112D+03,0.300D+01,0.96280000D+00,0.00000000D+00
     .,0.34026020D+03,0.112D+03,0.400D+01,0.96280000D+00,0.00000000D+00
     ./)
      pars(  26801:  26900)=(/
     . 0.21221800D+03,0.112D+03,0.500D+01,0.96280000D+00,0.00000000D+00
     .,0.13492270D+03,0.112D+03,0.600D+01,0.96280000D+00,0.00000000D+00
     .,0.89964600D+02,0.112D+03,0.700D+01,0.96280000D+00,0.00000000D+00
     .,0.65760900D+02,0.112D+03,0.800D+01,0.96280000D+00,0.00000000D+00
     .,0.48286500D+02,0.112D+03,0.900D+01,0.96280000D+00,0.00000000D+00
     .,0.36191100D+02,0.112D+03,0.100D+02,0.96280000D+00,0.00000000D+00
     .,0.79475970D+03,0.112D+03,0.110D+02,0.96280000D+00,0.00000000D+00
     .,0.55459670D+03,0.112D+03,0.120D+02,0.96280000D+00,0.00000000D+00
     .,0.49208290D+03,0.112D+03,0.130D+02,0.96280000D+00,0.00000000D+00
     .,0.36758110D+03,0.112D+03,0.140D+02,0.96280000D+00,0.00000000D+00
     .,0.27352680D+03,0.112D+03,0.150D+02,0.96280000D+00,0.00000000D+00
     .,0.21994890D+03,0.112D+03,0.160D+02,0.96280000D+00,0.00000000D+00
     .,0.17413990D+03,0.112D+03,0.170D+02,0.96280000D+00,0.00000000D+00
     .,0.13851900D+03,0.112D+03,0.180D+02,0.96280000D+00,0.00000000D+00
     .,0.13273176D+04,0.112D+03,0.190D+02,0.96280000D+00,0.00000000D+00
     .,0.10137695D+04,0.112D+03,0.200D+02,0.96280000D+00,0.00000000D+00
     .,0.82289570D+03,0.112D+03,0.210D+02,0.96280000D+00,0.00000000D+00
     .,0.78169930D+03,0.112D+03,0.220D+02,0.96280000D+00,0.00000000D+00
     .,0.70876840D+03,0.112D+03,0.230D+02,0.96280000D+00,0.00000000D+00
     .,0.55661270D+03,0.112D+03,0.240D+02,0.96280000D+00,0.00000000D+00
     ./)
      pars(  26901:  27000)=(/
     . 0.60144990D+03,0.112D+03,0.250D+02,0.96280000D+00,0.00000000D+00
     .,0.46964370D+03,0.112D+03,0.260D+02,0.96280000D+00,0.00000000D+00
     .,0.48817150D+03,0.112D+03,0.270D+02,0.96280000D+00,0.00000000D+00
     .,0.50815850D+03,0.112D+03,0.280D+02,0.96280000D+00,0.00000000D+00
     .,0.38842090D+03,0.112D+03,0.290D+02,0.96280000D+00,0.00000000D+00
     .,0.38688840D+03,0.112D+03,0.300D+02,0.96280000D+00,0.00000000D+00
     .,0.46356910D+03,0.112D+03,0.310D+02,0.96280000D+00,0.00000000D+00
     .,0.39225330D+03,0.112D+03,0.320D+02,0.96280000D+00,0.00000000D+00
     .,0.32187320D+03,0.112D+03,0.330D+02,0.96280000D+00,0.00000000D+00
     .,0.28171730D+03,0.112D+03,0.340D+02,0.96280000D+00,0.00000000D+00
     .,0.24015650D+03,0.112D+03,0.350D+02,0.96280000D+00,0.00000000D+00
     .,0.20374730D+03,0.112D+03,0.360D+02,0.96280000D+00,0.00000000D+00
     .,0.14776556D+04,0.112D+03,0.370D+02,0.96280000D+00,0.00000000D+00
     .,0.12109593D+04,0.112D+03,0.380D+02,0.96280000D+00,0.00000000D+00
     .,0.10281365D+04,0.112D+03,0.390D+02,0.96280000D+00,0.00000000D+00
     .,0.90577500D+03,0.112D+03,0.400D+02,0.96280000D+00,0.00000000D+00
     .,0.81492890D+03,0.112D+03,0.410D+02,0.96280000D+00,0.00000000D+00
     .,0.61402650D+03,0.112D+03,0.420D+02,0.96280000D+00,0.00000000D+00
     .,0.69144970D+03,0.112D+03,0.430D+02,0.96280000D+00,0.00000000D+00
     .,0.51267590D+03,0.112D+03,0.440D+02,0.96280000D+00,0.00000000D+00
     ./)
      pars(  27001:  27100)=(/
     . 0.56100100D+03,0.112D+03,0.450D+02,0.96280000D+00,0.00000000D+00
     .,0.51564570D+03,0.112D+03,0.460D+02,0.96280000D+00,0.00000000D+00
     .,0.43128070D+03,0.112D+03,0.470D+02,0.96280000D+00,0.00000000D+00
     .,0.44915670D+03,0.112D+03,0.480D+02,0.96280000D+00,0.00000000D+00
     .,0.57992340D+03,0.112D+03,0.490D+02,0.96280000D+00,0.00000000D+00
     .,0.51865180D+03,0.112D+03,0.500D+02,0.96280000D+00,0.00000000D+00
     .,0.44692300D+03,0.112D+03,0.510D+02,0.96280000D+00,0.00000000D+00
     .,0.40607280D+03,0.112D+03,0.520D+02,0.96280000D+00,0.00000000D+00
     .,0.35880520D+03,0.112D+03,0.530D+02,0.96280000D+00,0.00000000D+00
     .,0.31545810D+03,0.112D+03,0.540D+02,0.96280000D+00,0.00000000D+00
     .,0.17990180D+04,0.112D+03,0.550D+02,0.96280000D+00,0.00000000D+00
     .,0.15595050D+04,0.112D+03,0.560D+02,0.96280000D+00,0.00000000D+00
     .,0.13282510D+04,0.112D+03,0.570D+02,0.96280000D+00,0.00000000D+00
     .,0.53086500D+03,0.112D+03,0.580D+02,0.96280000D+00,0.27991000D+01
     .,0.13672714D+04,0.112D+03,0.590D+02,0.96280000D+00,0.00000000D+00
     .,0.13059000D+04,0.112D+03,0.600D+02,0.96280000D+00,0.00000000D+00
     .,0.12712883D+04,0.112D+03,0.610D+02,0.96280000D+00,0.00000000D+00
     .,0.12396749D+04,0.112D+03,0.620D+02,0.96280000D+00,0.00000000D+00
     .,0.12115825D+04,0.112D+03,0.630D+02,0.96280000D+00,0.00000000D+00
     .,0.91963290D+03,0.112D+03,0.640D+02,0.96280000D+00,0.00000000D+00
     ./)
      pars(  27101:  27200)=(/
     . 0.10898214D+04,0.112D+03,0.650D+02,0.96280000D+00,0.00000000D+00
     .,0.10451961D+04,0.112D+03,0.660D+02,0.96280000D+00,0.00000000D+00
     .,0.10842313D+04,0.112D+03,0.670D+02,0.96280000D+00,0.00000000D+00
     .,0.10603386D+04,0.112D+03,0.680D+02,0.96280000D+00,0.00000000D+00
     .,0.10383601D+04,0.112D+03,0.690D+02,0.96280000D+00,0.00000000D+00
     .,0.10277356D+04,0.112D+03,0.700D+02,0.96280000D+00,0.00000000D+00
     .,0.84553860D+03,0.112D+03,0.710D+02,0.96280000D+00,0.00000000D+00
     .,0.80500140D+03,0.112D+03,0.720D+02,0.96280000D+00,0.00000000D+00
     .,0.71971520D+03,0.112D+03,0.730D+02,0.96280000D+00,0.00000000D+00
     .,0.59743440D+03,0.112D+03,0.740D+02,0.96280000D+00,0.00000000D+00
     .,0.60309070D+03,0.112D+03,0.750D+02,0.96280000D+00,0.00000000D+00
     .,0.53662210D+03,0.112D+03,0.760D+02,0.96280000D+00,0.00000000D+00
     .,0.48407160D+03,0.112D+03,0.770D+02,0.96280000D+00,0.00000000D+00
     .,0.39524010D+03,0.112D+03,0.780D+02,0.96280000D+00,0.00000000D+00
     .,0.36670170D+03,0.112D+03,0.790D+02,0.96280000D+00,0.00000000D+00
     .,0.37485520D+03,0.112D+03,0.800D+02,0.96280000D+00,0.00000000D+00
     .,0.58953950D+03,0.112D+03,0.810D+02,0.96280000D+00,0.00000000D+00
     .,0.56172980D+03,0.112D+03,0.820D+02,0.96280000D+00,0.00000000D+00
     .,0.50092230D+03,0.112D+03,0.830D+02,0.96280000D+00,0.00000000D+00
     .,0.46935230D+03,0.112D+03,0.840D+02,0.96280000D+00,0.00000000D+00
     ./)
      pars(  27201:  27300)=(/
     . 0.42385000D+03,0.112D+03,0.850D+02,0.96280000D+00,0.00000000D+00
     .,0.38067930D+03,0.112D+03,0.860D+02,0.96280000D+00,0.00000000D+00
     .,0.16575070D+04,0.112D+03,0.870D+02,0.96280000D+00,0.00000000D+00
     .,0.15173457D+04,0.112D+03,0.880D+02,0.96280000D+00,0.00000000D+00
     .,0.13028376D+04,0.112D+03,0.890D+02,0.96280000D+00,0.00000000D+00
     .,0.11313898D+04,0.112D+03,0.900D+02,0.96280000D+00,0.00000000D+00
     .,0.11418668D+04,0.112D+03,0.910D+02,0.96280000D+00,0.00000000D+00
     .,0.11045083D+04,0.112D+03,0.920D+02,0.96280000D+00,0.00000000D+00
     .,0.11608135D+04,0.112D+03,0.930D+02,0.96280000D+00,0.00000000D+00
     .,0.11198466D+04,0.112D+03,0.940D+02,0.96280000D+00,0.00000000D+00
     .,0.54102100D+02,0.112D+03,0.101D+03,0.96280000D+00,0.00000000D+00
     .,0.19570570D+03,0.112D+03,0.103D+03,0.96280000D+00,0.98650000D+00
     .,0.24622270D+03,0.112D+03,0.104D+03,0.96280000D+00,0.98080000D+00
     .,0.17556460D+03,0.112D+03,0.105D+03,0.96280000D+00,0.97060000D+00
     .,0.12666570D+03,0.112D+03,0.106D+03,0.96280000D+00,0.98680000D+00
     .,0.83770300D+02,0.112D+03,0.107D+03,0.96280000D+00,0.99440000D+00
     .,0.58430300D+02,0.112D+03,0.108D+03,0.96280000D+00,0.99250000D+00
     .,0.37988400D+02,0.112D+03,0.109D+03,0.96280000D+00,0.99820000D+00
     .,0.28960820D+03,0.112D+03,0.111D+03,0.96280000D+00,0.96840000D+00
     .,0.45127080D+03,0.112D+03,0.112D+03,0.96280000D+00,0.96280000D+00
     ./)
      pars(  27301:  27400)=(/
     . 0.33104800D+02,0.113D+03,0.100D+01,0.96480000D+00,0.91180000D+00
     .,0.20515800D+02,0.113D+03,0.200D+01,0.96480000D+00,0.00000000D+00
     .,0.62753090D+03,0.113D+03,0.300D+01,0.96480000D+00,0.00000000D+00
     .,0.33473350D+03,0.113D+03,0.400D+01,0.96480000D+00,0.00000000D+00
     .,0.21388480D+03,0.113D+03,0.500D+01,0.96480000D+00,0.00000000D+00
     .,0.13832540D+03,0.113D+03,0.600D+01,0.96480000D+00,0.00000000D+00
     .,0.93354200D+02,0.113D+03,0.700D+01,0.96480000D+00,0.00000000D+00
     .,0.68798300D+02,0.113D+03,0.800D+01,0.96480000D+00,0.00000000D+00
     .,0.50857700D+02,0.113D+03,0.900D+01,0.96480000D+00,0.00000000D+00
     .,0.38315400D+02,0.113D+03,0.100D+02,0.96480000D+00,0.00000000D+00
     .,0.74634190D+03,0.113D+03,0.110D+02,0.96480000D+00,0.00000000D+00
     .,0.54102850D+03,0.113D+03,0.120D+02,0.96480000D+00,0.00000000D+00
     .,0.48623650D+03,0.113D+03,0.130D+02,0.96480000D+00,0.00000000D+00
     .,0.36962320D+03,0.113D+03,0.140D+02,0.96480000D+00,0.00000000D+00
     .,0.27892720D+03,0.113D+03,0.150D+02,0.96480000D+00,0.00000000D+00
     .,0.22621560D+03,0.113D+03,0.160D+02,0.96480000D+00,0.00000000D+00
     .,0.18054300D+03,0.113D+03,0.170D+02,0.96480000D+00,0.00000000D+00
     .,0.14457980D+03,0.113D+03,0.180D+02,0.96480000D+00,0.00000000D+00
     .,0.12332237D+04,0.113D+03,0.190D+02,0.96480000D+00,0.00000000D+00
     .,0.97304930D+03,0.113D+03,0.200D+02,0.96480000D+00,0.00000000D+00
     ./)
      pars(  27401:  27500)=(/
     . 0.79522060D+03,0.113D+03,0.210D+02,0.96480000D+00,0.00000000D+00
     .,0.75945320D+03,0.113D+03,0.220D+02,0.96480000D+00,0.00000000D+00
     .,0.69091560D+03,0.113D+03,0.230D+02,0.96480000D+00,0.00000000D+00
     .,0.54224590D+03,0.113D+03,0.240D+02,0.96480000D+00,0.00000000D+00
     .,0.58914750D+03,0.113D+03,0.250D+02,0.96480000D+00,0.00000000D+00
     .,0.46004960D+03,0.113D+03,0.260D+02,0.96480000D+00,0.00000000D+00
     .,0.48229900D+03,0.113D+03,0.270D+02,0.96480000D+00,0.00000000D+00
     .,0.50035250D+03,0.113D+03,0.280D+02,0.96480000D+00,0.00000000D+00
     .,0.38199230D+03,0.113D+03,0.290D+02,0.96480000D+00,0.00000000D+00
     .,0.38514490D+03,0.113D+03,0.300D+02,0.96480000D+00,0.00000000D+00
     .,0.45995440D+03,0.113D+03,0.310D+02,0.96480000D+00,0.00000000D+00
     .,0.39466500D+03,0.113D+03,0.320D+02,0.96480000D+00,0.00000000D+00
     .,0.32772290D+03,0.113D+03,0.330D+02,0.96480000D+00,0.00000000D+00
     .,0.28884850D+03,0.113D+03,0.340D+02,0.96480000D+00,0.00000000D+00
     .,0.24796920D+03,0.113D+03,0.350D+02,0.96480000D+00,0.00000000D+00
     .,0.21169490D+03,0.113D+03,0.360D+02,0.96480000D+00,0.00000000D+00
     .,0.13752558D+04,0.113D+03,0.370D+02,0.96480000D+00,0.00000000D+00
     .,0.11602539D+04,0.113D+03,0.380D+02,0.96480000D+00,0.00000000D+00
     .,0.99637570D+03,0.113D+03,0.390D+02,0.96480000D+00,0.00000000D+00
     .,0.88385410D+03,0.113D+03,0.400D+02,0.96480000D+00,0.00000000D+00
     ./)
      pars(  27501:  27600)=(/
     . 0.79860940D+03,0.113D+03,0.410D+02,0.96480000D+00,0.00000000D+00
     .,0.60591350D+03,0.113D+03,0.420D+02,0.96480000D+00,0.00000000D+00
     .,0.68054270D+03,0.113D+03,0.430D+02,0.96480000D+00,0.00000000D+00
     .,0.50852890D+03,0.113D+03,0.440D+02,0.96480000D+00,0.00000000D+00
     .,0.55700430D+03,0.113D+03,0.450D+02,0.96480000D+00,0.00000000D+00
     .,0.51341630D+03,0.113D+03,0.460D+02,0.96480000D+00,0.00000000D+00
     .,0.42799620D+03,0.113D+03,0.470D+02,0.96480000D+00,0.00000000D+00
     .,0.44880710D+03,0.113D+03,0.480D+02,0.96480000D+00,0.00000000D+00
     .,0.57434200D+03,0.113D+03,0.490D+02,0.96480000D+00,0.00000000D+00
     .,0.51984240D+03,0.113D+03,0.500D+02,0.96480000D+00,0.00000000D+00
     .,0.45284940D+03,0.113D+03,0.510D+02,0.96480000D+00,0.00000000D+00
     .,0.41405440D+03,0.113D+03,0.520D+02,0.96480000D+00,0.00000000D+00
     .,0.36827920D+03,0.113D+03,0.530D+02,0.96480000D+00,0.00000000D+00
     .,0.32576250D+03,0.113D+03,0.540D+02,0.96480000D+00,0.00000000D+00
     .,0.16731922D+04,0.113D+03,0.550D+02,0.96480000D+00,0.00000000D+00
     .,0.14872964D+04,0.113D+03,0.560D+02,0.96480000D+00,0.00000000D+00
     .,0.12815229D+04,0.113D+03,0.570D+02,0.96480000D+00,0.00000000D+00
     .,0.53656260D+03,0.113D+03,0.580D+02,0.96480000D+00,0.27991000D+01
     .,0.13086041D+04,0.113D+03,0.590D+02,0.96480000D+00,0.00000000D+00
     .,0.12526928D+04,0.113D+03,0.600D+02,0.96480000D+00,0.00000000D+00
     ./)
      pars(  27601:  27700)=(/
     . 0.12202246D+04,0.113D+03,0.610D+02,0.96480000D+00,0.00000000D+00
     .,0.11905003D+04,0.113D+03,0.620D+02,0.96480000D+00,0.00000000D+00
     .,0.11641089D+04,0.113D+03,0.630D+02,0.96480000D+00,0.00000000D+00
     .,0.89395150D+03,0.113D+03,0.640D+02,0.96480000D+00,0.00000000D+00
     .,0.10381047D+04,0.113D+03,0.650D+02,0.96480000D+00,0.00000000D+00
     .,0.99738370D+03,0.113D+03,0.660D+02,0.96480000D+00,0.00000000D+00
     .,0.10450038D+04,0.113D+03,0.670D+02,0.96480000D+00,0.00000000D+00
     .,0.10223674D+04,0.113D+03,0.680D+02,0.96480000D+00,0.00000000D+00
     .,0.10016784D+04,0.113D+03,0.690D+02,0.96480000D+00,0.00000000D+00
     .,0.99098080D+03,0.113D+03,0.700D+02,0.96480000D+00,0.00000000D+00
     .,0.82160050D+03,0.113D+03,0.710D+02,0.96480000D+00,0.00000000D+00
     .,0.79184560D+03,0.113D+03,0.720D+02,0.96480000D+00,0.00000000D+00
     .,0.71274220D+03,0.113D+03,0.730D+02,0.96480000D+00,0.00000000D+00
     .,0.59416160D+03,0.113D+03,0.740D+02,0.96480000D+00,0.00000000D+00
     .,0.60157800D+03,0.113D+03,0.750D+02,0.96480000D+00,0.00000000D+00
     .,0.53838550D+03,0.113D+03,0.760D+02,0.96480000D+00,0.00000000D+00
     .,0.48789840D+03,0.113D+03,0.770D+02,0.96480000D+00,0.00000000D+00
     .,0.39997680D+03,0.113D+03,0.780D+02,0.96480000D+00,0.00000000D+00
     .,0.37172100D+03,0.113D+03,0.790D+02,0.96480000D+00,0.00000000D+00
     .,0.38105900D+03,0.113D+03,0.800D+02,0.96480000D+00,0.00000000D+00
     ./)
      pars(  27701:  27800)=(/
     . 0.58469720D+03,0.113D+03,0.810D+02,0.96480000D+00,0.00000000D+00
     .,0.56268750D+03,0.113D+03,0.820D+02,0.96480000D+00,0.00000000D+00
     .,0.50686630D+03,0.113D+03,0.830D+02,0.96480000D+00,0.00000000D+00
     .,0.47758620D+03,0.113D+03,0.840D+02,0.96480000D+00,0.00000000D+00
     .,0.43409690D+03,0.113D+03,0.850D+02,0.96480000D+00,0.00000000D+00
     .,0.39211690D+03,0.113D+03,0.860D+02,0.96480000D+00,0.00000000D+00
     .,0.15558848D+04,0.113D+03,0.870D+02,0.96480000D+00,0.00000000D+00
     .,0.14554241D+04,0.113D+03,0.880D+02,0.96480000D+00,0.00000000D+00
     .,0.12633724D+04,0.113D+03,0.890D+02,0.96480000D+00,0.00000000D+00
     .,0.11100628D+04,0.113D+03,0.900D+02,0.96480000D+00,0.00000000D+00
     .,0.11130738D+04,0.113D+03,0.910D+02,0.96480000D+00,0.00000000D+00
     .,0.10770005D+04,0.113D+03,0.920D+02,0.96480000D+00,0.00000000D+00
     .,0.11238008D+04,0.113D+03,0.930D+02,0.96480000D+00,0.00000000D+00
     .,0.10856846D+04,0.113D+03,0.940D+02,0.96480000D+00,0.00000000D+00
     .,0.55263600D+02,0.113D+03,0.101D+03,0.96480000D+00,0.00000000D+00
     .,0.19307070D+03,0.113D+03,0.103D+03,0.96480000D+00,0.98650000D+00
     .,0.24376120D+03,0.113D+03,0.104D+03,0.96480000D+00,0.98080000D+00
     .,0.17772540D+03,0.113D+03,0.105D+03,0.96480000D+00,0.97060000D+00
     .,0.12969220D+03,0.113D+03,0.106D+03,0.96480000D+00,0.98680000D+00
     .,0.86872800D+02,0.113D+03,0.107D+03,0.96480000D+00,0.99440000D+00
     ./)
      pars(  27801:  27900)=(/
     . 0.61217500D+02,0.113D+03,0.108D+03,0.96480000D+00,0.99250000D+00
     .,0.40322000D+02,0.113D+03,0.109D+03,0.96480000D+00,0.99820000D+00
     .,0.28422510D+03,0.113D+03,0.111D+03,0.96480000D+00,0.96840000D+00
     .,0.44173220D+03,0.113D+03,0.112D+03,0.96480000D+00,0.96280000D+00
     .,0.43775750D+03,0.113D+03,0.113D+03,0.96480000D+00,0.96480000D+00
     .,0.27498900D+02,0.114D+03,0.100D+01,0.95070000D+00,0.91180000D+00
     .,0.17635600D+02,0.114D+03,0.200D+01,0.95070000D+00,0.00000000D+00
     .,0.45095830D+03,0.114D+03,0.300D+01,0.95070000D+00,0.00000000D+00
     .,0.25658810D+03,0.114D+03,0.400D+01,0.95070000D+00,0.00000000D+00
     .,0.17018910D+03,0.114D+03,0.500D+01,0.95070000D+00,0.00000000D+00
     .,0.11320460D+03,0.114D+03,0.600D+01,0.95070000D+00,0.00000000D+00
     .,0.78019000D+02,0.114D+03,0.700D+01,0.95070000D+00,0.00000000D+00
     .,0.58355700D+02,0.114D+03,0.800D+01,0.95070000D+00,0.00000000D+00
     .,0.43685500D+02,0.114D+03,0.900D+01,0.95070000D+00,0.00000000D+00
     .,0.33241600D+02,0.114D+03,0.100D+02,0.95070000D+00,0.00000000D+00
     .,0.53834120D+03,0.114D+03,0.110D+02,0.95070000D+00,0.00000000D+00
     .,0.40991740D+03,0.114D+03,0.120D+02,0.95070000D+00,0.00000000D+00
     .,0.37552330D+03,0.114D+03,0.130D+02,0.95070000D+00,0.00000000D+00
     .,0.29299940D+03,0.114D+03,0.140D+02,0.95070000D+00,0.00000000D+00
     .,0.22606570D+03,0.114D+03,0.150D+02,0.95070000D+00,0.00000000D+00
     ./)
      pars(  27901:  28000)=(/
     . 0.18600510D+03,0.114D+03,0.160D+02,0.95070000D+00,0.00000000D+00
     .,0.15052310D+03,0.114D+03,0.170D+02,0.95070000D+00,0.00000000D+00
     .,0.12200120D+03,0.114D+03,0.180D+02,0.95070000D+00,0.00000000D+00
     .,0.88126240D+03,0.114D+03,0.190D+02,0.95070000D+00,0.00000000D+00
     .,0.72251110D+03,0.114D+03,0.200D+02,0.95070000D+00,0.00000000D+00
     .,0.59567250D+03,0.114D+03,0.210D+02,0.95070000D+00,0.00000000D+00
     .,0.57349880D+03,0.114D+03,0.220D+02,0.95070000D+00,0.00000000D+00
     .,0.52428080D+03,0.114D+03,0.230D+02,0.95070000D+00,0.00000000D+00
     .,0.41188790D+03,0.114D+03,0.240D+02,0.95070000D+00,0.00000000D+00
     .,0.45022340D+03,0.114D+03,0.250D+02,0.95070000D+00,0.00000000D+00
     .,0.35224080D+03,0.114D+03,0.260D+02,0.95070000D+00,0.00000000D+00
     .,0.37301250D+03,0.114D+03,0.270D+02,0.95070000D+00,0.00000000D+00
     .,0.38500480D+03,0.114D+03,0.280D+02,0.95070000D+00,0.00000000D+00
     .,0.29414060D+03,0.114D+03,0.290D+02,0.95070000D+00,0.00000000D+00
     .,0.30117090D+03,0.114D+03,0.300D+02,0.95070000D+00,0.00000000D+00
     .,0.35794110D+03,0.114D+03,0.310D+02,0.95070000D+00,0.00000000D+00
     .,0.31339110D+03,0.114D+03,0.320D+02,0.95070000D+00,0.00000000D+00
     .,0.26507110D+03,0.114D+03,0.330D+02,0.95070000D+00,0.00000000D+00
     .,0.23633230D+03,0.114D+03,0.340D+02,0.95070000D+00,0.00000000D+00
     .,0.20529760D+03,0.114D+03,0.350D+02,0.95070000D+00,0.00000000D+00
     ./)
      pars(  28001:  28100)=(/
     . 0.17718110D+03,0.114D+03,0.360D+02,0.95070000D+00,0.00000000D+00
     .,0.98604380D+03,0.114D+03,0.370D+02,0.95070000D+00,0.00000000D+00
     .,0.86034210D+03,0.114D+03,0.380D+02,0.95070000D+00,0.00000000D+00
     .,0.75038170D+03,0.114D+03,0.390D+02,0.95070000D+00,0.00000000D+00
     .,0.67218340D+03,0.114D+03,0.400D+02,0.95070000D+00,0.00000000D+00
     .,0.61131470D+03,0.114D+03,0.410D+02,0.95070000D+00,0.00000000D+00
     .,0.46910920D+03,0.114D+03,0.420D+02,0.95070000D+00,0.00000000D+00
     .,0.52464790D+03,0.114D+03,0.430D+02,0.95070000D+00,0.00000000D+00
     .,0.39698620D+03,0.114D+03,0.440D+02,0.95070000D+00,0.00000000D+00
     .,0.43472510D+03,0.114D+03,0.450D+02,0.95070000D+00,0.00000000D+00
     .,0.40236620D+03,0.114D+03,0.460D+02,0.95070000D+00,0.00000000D+00
     .,0.33466730D+03,0.114D+03,0.470D+02,0.95070000D+00,0.00000000D+00
     .,0.35359760D+03,0.114D+03,0.480D+02,0.95070000D+00,0.00000000D+00
     .,0.44659370D+03,0.114D+03,0.490D+02,0.95070000D+00,0.00000000D+00
     .,0.41089600D+03,0.114D+03,0.500D+02,0.95070000D+00,0.00000000D+00
     .,0.36378460D+03,0.114D+03,0.510D+02,0.95070000D+00,0.00000000D+00
     .,0.33592400D+03,0.114D+03,0.520D+02,0.95070000D+00,0.00000000D+00
     .,0.30198520D+03,0.114D+03,0.530D+02,0.95070000D+00,0.00000000D+00
     .,0.26983720D+03,0.114D+03,0.540D+02,0.95070000D+00,0.00000000D+00
     .,0.12005803D+04,0.114D+03,0.550D+02,0.95070000D+00,0.00000000D+00
     ./)
      pars(  28101:  28200)=(/
     . 0.10973980D+04,0.114D+03,0.560D+02,0.95070000D+00,0.00000000D+00
     .,0.96023320D+03,0.114D+03,0.570D+02,0.95070000D+00,0.00000000D+00
     .,0.42980740D+03,0.114D+03,0.580D+02,0.95070000D+00,0.27991000D+01
     .,0.97039570D+03,0.114D+03,0.590D+02,0.95070000D+00,0.00000000D+00
     .,0.93129170D+03,0.114D+03,0.600D+02,0.95070000D+00,0.00000000D+00
     .,0.90779700D+03,0.114D+03,0.610D+02,0.95070000D+00,0.00000000D+00
     .,0.88621930D+03,0.114D+03,0.620D+02,0.95070000D+00,0.00000000D+00
     .,0.86708440D+03,0.114D+03,0.630D+02,0.95070000D+00,0.00000000D+00
     .,0.67748060D+03,0.114D+03,0.640D+02,0.95070000D+00,0.00000000D+00
     .,0.76730550D+03,0.114D+03,0.650D+02,0.95070000D+00,0.00000000D+00
     .,0.73939750D+03,0.114D+03,0.660D+02,0.95070000D+00,0.00000000D+00
     .,0.78137480D+03,0.114D+03,0.670D+02,0.95070000D+00,0.00000000D+00
     .,0.76476680D+03,0.114D+03,0.680D+02,0.95070000D+00,0.00000000D+00
     .,0.74974130D+03,0.114D+03,0.690D+02,0.95070000D+00,0.00000000D+00
     .,0.74120300D+03,0.114D+03,0.700D+02,0.95070000D+00,0.00000000D+00
     .,0.62181690D+03,0.114D+03,0.710D+02,0.95070000D+00,0.00000000D+00
     .,0.60906080D+03,0.114D+03,0.720D+02,0.95070000D+00,0.00000000D+00
     .,0.55365420D+03,0.114D+03,0.730D+02,0.95070000D+00,0.00000000D+00
     .,0.46519420D+03,0.114D+03,0.740D+02,0.95070000D+00,0.00000000D+00
     .,0.47279230D+03,0.114D+03,0.750D+02,0.95070000D+00,0.00000000D+00
     ./)
      pars(  28201:  28300)=(/
     . 0.42679800D+03,0.114D+03,0.760D+02,0.95070000D+00,0.00000000D+00
     .,0.38949880D+03,0.114D+03,0.770D+02,0.95070000D+00,0.00000000D+00
     .,0.32171830D+03,0.114D+03,0.780D+02,0.95070000D+00,0.00000000D+00
     .,0.29988580D+03,0.114D+03,0.790D+02,0.95070000D+00,0.00000000D+00
     .,0.30842630D+03,0.114D+03,0.800D+02,0.95070000D+00,0.00000000D+00
     .,0.45658720D+03,0.114D+03,0.810D+02,0.95070000D+00,0.00000000D+00
     .,0.44500360D+03,0.114D+03,0.820D+02,0.95070000D+00,0.00000000D+00
     .,0.40670170D+03,0.114D+03,0.830D+02,0.95070000D+00,0.00000000D+00
     .,0.38646750D+03,0.114D+03,0.840D+02,0.95070000D+00,0.00000000D+00
     .,0.35486660D+03,0.114D+03,0.850D+02,0.95070000D+00,0.00000000D+00
     .,0.32354110D+03,0.114D+03,0.860D+02,0.95070000D+00,0.00000000D+00
     .,0.11298915D+04,0.114D+03,0.870D+02,0.95070000D+00,0.00000000D+00
     .,0.10825018D+04,0.114D+03,0.880D+02,0.95070000D+00,0.00000000D+00
     .,0.95314260D+03,0.114D+03,0.890D+02,0.95070000D+00,0.00000000D+00
     .,0.85147580D+03,0.114D+03,0.900D+02,0.95070000D+00,0.00000000D+00
     .,0.84679590D+03,0.114D+03,0.910D+02,0.95070000D+00,0.00000000D+00
     .,0.81968560D+03,0.114D+03,0.920D+02,0.95070000D+00,0.00000000D+00
     .,0.84655430D+03,0.114D+03,0.930D+02,0.95070000D+00,0.00000000D+00
     .,0.81936980D+03,0.114D+03,0.940D+02,0.95070000D+00,0.00000000D+00
     .,0.44903000D+02,0.114D+03,0.101D+03,0.95070000D+00,0.00000000D+00
     ./)
      pars(  28301:  28400)=(/
     . 0.14874780D+03,0.114D+03,0.103D+03,0.95070000D+00,0.98650000D+00
     .,0.18910900D+03,0.114D+03,0.104D+03,0.95070000D+00,0.98080000D+00
     .,0.14251230D+03,0.114D+03,0.105D+03,0.95070000D+00,0.97060000D+00
     .,0.10608060D+03,0.114D+03,0.106D+03,0.95070000D+00,0.98680000D+00
     .,0.72637300D+02,0.114D+03,0.107D+03,0.95070000D+00,0.99440000D+00
     .,0.52131100D+02,0.114D+03,0.108D+03,0.95070000D+00,0.99250000D+00
     .,0.35129000D+02,0.114D+03,0.109D+03,0.95070000D+00,0.99820000D+00
     .,0.21750490D+03,0.114D+03,0.111D+03,0.95070000D+00,0.96840000D+00
     .,0.33702500D+03,0.114D+03,0.112D+03,0.95070000D+00,0.96280000D+00
     .,0.33965160D+03,0.114D+03,0.113D+03,0.95070000D+00,0.96480000D+00
     .,0.27026500D+03,0.114D+03,0.114D+03,0.95070000D+00,0.95070000D+00
     .,0.23090800D+02,0.115D+03,0.100D+01,0.99470000D+00,0.91180000D+00
     .,0.15261900D+02,0.115D+03,0.200D+01,0.99470000D+00,0.00000000D+00
     .,0.33966930D+03,0.115D+03,0.300D+01,0.99470000D+00,0.00000000D+00
     .,0.20222820D+03,0.115D+03,0.400D+01,0.99470000D+00,0.00000000D+00
     .,0.13807740D+03,0.115D+03,0.500D+01,0.99470000D+00,0.00000000D+00
     .,0.93966300D+02,0.115D+03,0.600D+01,0.99470000D+00,0.00000000D+00
     .,0.65912700D+02,0.115D+03,0.700D+01,0.99470000D+00,0.00000000D+00
     .,0.49937400D+02,0.115D+03,0.800D+01,0.99470000D+00,0.00000000D+00
     .,0.37803500D+02,0.115D+03,0.900D+01,0.99470000D+00,0.00000000D+00
     ./)
      pars(  28401:  28500)=(/
     . 0.29028000D+02,0.115D+03,0.100D+02,0.99470000D+00,0.00000000D+00
     .,0.40671500D+03,0.115D+03,0.110D+02,0.99470000D+00,0.00000000D+00
     .,0.32041760D+03,0.115D+03,0.120D+02,0.99470000D+00,0.00000000D+00
     .,0.29780760D+03,0.115D+03,0.130D+02,0.99470000D+00,0.00000000D+00
     .,0.23698640D+03,0.115D+03,0.140D+02,0.99470000D+00,0.00000000D+00
     .,0.18606840D+03,0.115D+03,0.150D+02,0.99470000D+00,0.00000000D+00
     .,0.15492290D+03,0.115D+03,0.160D+02,0.99470000D+00,0.00000000D+00
     .,0.12683120D+03,0.115D+03,0.170D+02,0.99470000D+00,0.00000000D+00
     .,0.10386530D+03,0.115D+03,0.180D+02,0.99470000D+00,0.00000000D+00
     .,0.66311840D+03,0.115D+03,0.190D+02,0.99470000D+00,0.00000000D+00
     .,0.55742550D+03,0.115D+03,0.200D+02,0.99470000D+00,0.00000000D+00
     .,0.46235110D+03,0.115D+03,0.210D+02,0.99470000D+00,0.00000000D+00
     .,0.44795900D+03,0.115D+03,0.220D+02,0.99470000D+00,0.00000000D+00
     .,0.41101210D+03,0.115D+03,0.230D+02,0.99470000D+00,0.00000000D+00
     .,0.32359140D+03,0.115D+03,0.240D+02,0.99470000D+00,0.00000000D+00
     .,0.35484320D+03,0.115D+03,0.250D+02,0.99470000D+00,0.00000000D+00
     .,0.27840200D+03,0.115D+03,0.260D+02,0.99470000D+00,0.00000000D+00
     .,0.29657890D+03,0.115D+03,0.270D+02,0.99470000D+00,0.00000000D+00
     .,0.30491450D+03,0.115D+03,0.280D+02,0.99470000D+00,0.00000000D+00
     .,0.23348980D+03,0.115D+03,0.290D+02,0.99470000D+00,0.00000000D+00
     ./)
      pars(  28501:  28600)=(/
     . 0.24143800D+03,0.115D+03,0.300D+02,0.99470000D+00,0.00000000D+00
     .,0.28580720D+03,0.115D+03,0.310D+02,0.99470000D+00,0.00000000D+00
     .,0.25397440D+03,0.115D+03,0.320D+02,0.99470000D+00,0.00000000D+00
     .,0.21789200D+03,0.115D+03,0.330D+02,0.99470000D+00,0.00000000D+00
     .,0.19608060D+03,0.115D+03,0.340D+02,0.99470000D+00,0.00000000D+00
     .,0.17199100D+03,0.115D+03,0.350D+02,0.99470000D+00,0.00000000D+00
     .,0.14979330D+03,0.115D+03,0.360D+02,0.99470000D+00,0.00000000D+00
     .,0.74424750D+03,0.115D+03,0.370D+02,0.99470000D+00,0.00000000D+00
     .,0.66356560D+03,0.115D+03,0.380D+02,0.99470000D+00,0.00000000D+00
     .,0.58524670D+03,0.115D+03,0.390D+02,0.99470000D+00,0.00000000D+00
     .,0.52808410D+03,0.115D+03,0.400D+02,0.99470000D+00,0.00000000D+00
     .,0.48270360D+03,0.115D+03,0.410D+02,0.99470000D+00,0.00000000D+00
     .,0.37394270D+03,0.115D+03,0.420D+02,0.99470000D+00,0.00000000D+00
     .,0.41670800D+03,0.115D+03,0.430D+02,0.99470000D+00,0.00000000D+00
     .,0.31859120D+03,0.115D+03,0.440D+02,0.99470000D+00,0.00000000D+00
     .,0.34844490D+03,0.115D+03,0.450D+02,0.99470000D+00,0.00000000D+00
     .,0.32352760D+03,0.115D+03,0.460D+02,0.99470000D+00,0.00000000D+00
     .,0.26912200D+03,0.115D+03,0.470D+02,0.99470000D+00,0.00000000D+00
     .,0.28548020D+03,0.115D+03,0.480D+02,0.99470000D+00,0.00000000D+00
     .,0.35690690D+03,0.115D+03,0.490D+02,0.99470000D+00,0.00000000D+00
     ./)
      pars(  28601:  28700)=(/
     . 0.33217220D+03,0.115D+03,0.500D+02,0.99470000D+00,0.00000000D+00
     .,0.29765980D+03,0.115D+03,0.510D+02,0.99470000D+00,0.00000000D+00
     .,0.27698090D+03,0.115D+03,0.520D+02,0.99470000D+00,0.00000000D+00
     .,0.25110810D+03,0.115D+03,0.530D+02,0.99470000D+00,0.00000000D+00
     .,0.22622110D+03,0.115D+03,0.540D+02,0.99470000D+00,0.00000000D+00
     .,0.90729930D+03,0.115D+03,0.550D+02,0.99470000D+00,0.00000000D+00
     .,0.84401200D+03,0.115D+03,0.560D+02,0.99470000D+00,0.00000000D+00
     .,0.74659930D+03,0.115D+03,0.570D+02,0.99470000D+00,0.00000000D+00
     .,0.35115850D+03,0.115D+03,0.580D+02,0.99470000D+00,0.27991000D+01
     .,0.74914380D+03,0.115D+03,0.590D+02,0.99470000D+00,0.00000000D+00
     .,0.72014500D+03,0.115D+03,0.600D+02,0.99470000D+00,0.00000000D+00
     .,0.70229590D+03,0.115D+03,0.610D+02,0.99470000D+00,0.00000000D+00
     .,0.68586200D+03,0.115D+03,0.620D+02,0.99470000D+00,0.00000000D+00
     .,0.67130090D+03,0.115D+03,0.630D+02,0.99470000D+00,0.00000000D+00
     .,0.53154240D+03,0.115D+03,0.640D+02,0.99470000D+00,0.00000000D+00
     .,0.59201350D+03,0.115D+03,0.650D+02,0.99470000D+00,0.00000000D+00
     .,0.57174810D+03,0.115D+03,0.660D+02,0.99470000D+00,0.00000000D+00
     .,0.60649400D+03,0.115D+03,0.670D+02,0.99470000D+00,0.00000000D+00
     .,0.59374030D+03,0.115D+03,0.680D+02,0.99470000D+00,0.00000000D+00
     .,0.58229560D+03,0.115D+03,0.690D+02,0.99470000D+00,0.00000000D+00
     ./)
      pars(  28701:  28800)=(/
     . 0.57532620D+03,0.115D+03,0.700D+02,0.99470000D+00,0.00000000D+00
     .,0.48706190D+03,0.115D+03,0.710D+02,0.99470000D+00,0.00000000D+00
     .,0.48246160D+03,0.115D+03,0.720D+02,0.99470000D+00,0.00000000D+00
     .,0.44186160D+03,0.115D+03,0.730D+02,0.99470000D+00,0.00000000D+00
     .,0.37386540D+03,0.115D+03,0.740D+02,0.99470000D+00,0.00000000D+00
     .,0.38091200D+03,0.115D+03,0.750D+02,0.99470000D+00,0.00000000D+00
     .,0.34613260D+03,0.115D+03,0.760D+02,0.99470000D+00,0.00000000D+00
     .,0.31760850D+03,0.115D+03,0.770D+02,0.99470000D+00,0.00000000D+00
     .,0.26410910D+03,0.115D+03,0.780D+02,0.99470000D+00,0.00000000D+00
     .,0.24683080D+03,0.115D+03,0.790D+02,0.99470000D+00,0.00000000D+00
     .,0.25431280D+03,0.115D+03,0.800D+02,0.99470000D+00,0.00000000D+00
     .,0.36658420D+03,0.115D+03,0.810D+02,0.99470000D+00,0.00000000D+00
     .,0.36026310D+03,0.115D+03,0.820D+02,0.99470000D+00,0.00000000D+00
     .,0.33271860D+03,0.115D+03,0.830D+02,0.99470000D+00,0.00000000D+00
     .,0.31818410D+03,0.115D+03,0.840D+02,0.99470000D+00,0.00000000D+00
     .,0.29447100D+03,0.115D+03,0.850D+02,0.99470000D+00,0.00000000D+00
     .,0.27046770D+03,0.115D+03,0.860D+02,0.99470000D+00,0.00000000D+00
     .,0.86138240D+03,0.115D+03,0.870D+02,0.99470000D+00,0.00000000D+00
     .,0.83751280D+03,0.115D+03,0.880D+02,0.99470000D+00,0.00000000D+00
     .,0.74476320D+03,0.115D+03,0.890D+02,0.99470000D+00,0.00000000D+00
     ./)
      pars(  28801:  28900)=(/
     . 0.67356880D+03,0.115D+03,0.900D+02,0.99470000D+00,0.00000000D+00
     .,0.66625820D+03,0.115D+03,0.910D+02,0.99470000D+00,0.00000000D+00
     .,0.64514050D+03,0.115D+03,0.920D+02,0.99470000D+00,0.00000000D+00
     .,0.66128050D+03,0.115D+03,0.930D+02,0.99470000D+00,0.00000000D+00
     .,0.64087610D+03,0.115D+03,0.940D+02,0.99470000D+00,0.00000000D+00
     .,0.37023600D+02,0.115D+03,0.101D+03,0.99470000D+00,0.00000000D+00
     .,0.11774720D+03,0.115D+03,0.103D+03,0.99470000D+00,0.98650000D+00
     .,0.15060680D+03,0.115D+03,0.104D+03,0.99470000D+00,0.98080000D+00
     .,0.11638180D+03,0.115D+03,0.105D+03,0.99470000D+00,0.97060000D+00
     .,0.88076900D+02,0.115D+03,0.106D+03,0.99470000D+00,0.98680000D+00
     .,0.61428600D+02,0.115D+03,0.107D+03,0.99470000D+00,0.99440000D+00
     .,0.44777000D+02,0.115D+03,0.108D+03,0.99470000D+00,0.99250000D+00
     .,0.30768800D+02,0.115D+03,0.109D+03,0.99470000D+00,0.99820000D+00
     .,0.17144690D+03,0.115D+03,0.111D+03,0.99470000D+00,0.96840000D+00
     .,0.26503780D+03,0.115D+03,0.112D+03,0.99470000D+00,0.96280000D+00
     .,0.27035800D+03,0.115D+03,0.113D+03,0.99470000D+00,0.96480000D+00
     .,0.21928630D+03,0.115D+03,0.114D+03,0.99470000D+00,0.95070000D+00
     .,0.18062380D+03,0.115D+03,0.115D+03,0.99470000D+00,0.99470000D+00
     .,0.19866700D+02,0.116D+03,0.100D+01,0.99480000D+00,0.91180000D+00
     .,0.13459800D+02,0.116D+03,0.200D+01,0.99480000D+00,0.00000000D+00
     ./)
      pars(  28901:  29000)=(/
     . 0.27130110D+03,0.116D+03,0.300D+01,0.99480000D+00,0.00000000D+00
     .,0.16618070D+03,0.116D+03,0.400D+01,0.99480000D+00,0.00000000D+00
     .,0.11581680D+03,0.116D+03,0.500D+01,0.99480000D+00,0.00000000D+00
     .,0.80180400D+02,0.116D+03,0.600D+01,0.99480000D+00,0.00000000D+00
     .,0.57025500D+02,0.116D+03,0.700D+01,0.99480000D+00,0.00000000D+00
     .,0.43654800D+02,0.116D+03,0.800D+01,0.99480000D+00,0.00000000D+00
     .,0.33355000D+02,0.116D+03,0.900D+01,0.99480000D+00,0.00000000D+00
     .,0.25810700D+02,0.116D+03,0.100D+02,0.99480000D+00,0.00000000D+00
     .,0.32559690D+03,0.116D+03,0.110D+02,0.99480000D+00,0.00000000D+00
     .,0.26194370D+03,0.116D+03,0.120D+02,0.99480000D+00,0.00000000D+00
     .,0.24585680D+03,0.116D+03,0.130D+02,0.99480000D+00,0.00000000D+00
     .,0.19832660D+03,0.116D+03,0.140D+02,0.99480000D+00,0.00000000D+00
     .,0.15769710D+03,0.116D+03,0.150D+02,0.99480000D+00,0.00000000D+00
     .,0.13249220D+03,0.116D+03,0.160D+02,0.99480000D+00,0.00000000D+00
     .,0.10944860D+03,0.116D+03,0.170D+02,0.99480000D+00,0.00000000D+00
     .,0.90371600D+02,0.116D+03,0.180D+02,0.99480000D+00,0.00000000D+00
     .,0.53051920D+03,0.116D+03,0.190D+02,0.99480000D+00,0.00000000D+00
     .,0.45237450D+03,0.116D+03,0.200D+02,0.99480000D+00,0.00000000D+00
     .,0.37661420D+03,0.116D+03,0.210D+02,0.99480000D+00,0.00000000D+00
     .,0.36651510D+03,0.116D+03,0.220D+02,0.99480000D+00,0.00000000D+00
     ./)
      pars(  29001:  29100)=(/
     . 0.33712500D+03,0.116D+03,0.230D+02,0.99480000D+00,0.00000000D+00
     .,0.26607620D+03,0.116D+03,0.240D+02,0.99480000D+00,0.00000000D+00
     .,0.29212570D+03,0.116D+03,0.250D+02,0.99480000D+00,0.00000000D+00
     .,0.22987760D+03,0.116D+03,0.260D+02,0.99480000D+00,0.00000000D+00
     .,0.24559880D+03,0.116D+03,0.270D+02,0.99480000D+00,0.00000000D+00
     .,0.25180970D+03,0.116D+03,0.280D+02,0.99480000D+00,0.00000000D+00
     .,0.19338650D+03,0.116D+03,0.290D+02,0.99480000D+00,0.00000000D+00
     .,0.20107040D+03,0.116D+03,0.300D+02,0.99480000D+00,0.00000000D+00
     .,0.23725630D+03,0.116D+03,0.310D+02,0.99480000D+00,0.00000000D+00
     .,0.21294370D+03,0.116D+03,0.320D+02,0.99480000D+00,0.00000000D+00
     .,0.18454670D+03,0.116D+03,0.330D+02,0.99480000D+00,0.00000000D+00
     .,0.16722640D+03,0.116D+03,0.340D+02,0.99480000D+00,0.00000000D+00
     .,0.14776790D+03,0.116D+03,0.350D+02,0.99480000D+00,0.00000000D+00
     .,0.12961230D+03,0.116D+03,0.360D+02,0.99480000D+00,0.00000000D+00
     .,0.59687630D+03,0.116D+03,0.370D+02,0.99480000D+00,0.00000000D+00
     .,0.53867240D+03,0.116D+03,0.380D+02,0.99480000D+00,0.00000000D+00
     .,0.47852040D+03,0.116D+03,0.390D+02,0.99480000D+00,0.00000000D+00
     .,0.43390360D+03,0.116D+03,0.400D+02,0.99480000D+00,0.00000000D+00
     .,0.39804190D+03,0.116D+03,0.410D+02,0.99480000D+00,0.00000000D+00
     .,0.31058610D+03,0.116D+03,0.420D+02,0.99480000D+00,0.00000000D+00
     ./)
      pars(  29101:  29200)=(/
     . 0.34514340D+03,0.116D+03,0.430D+02,0.99480000D+00,0.00000000D+00
     .,0.26595280D+03,0.116D+03,0.440D+02,0.99480000D+00,0.00000000D+00
     .,0.29041460D+03,0.116D+03,0.450D+02,0.99480000D+00,0.00000000D+00
     .,0.27025240D+03,0.116D+03,0.460D+02,0.99480000D+00,0.00000000D+00
     .,0.22511810D+03,0.116D+03,0.470D+02,0.99480000D+00,0.00000000D+00
     .,0.23917840D+03,0.116D+03,0.480D+02,0.99480000D+00,0.00000000D+00
     .,0.29682140D+03,0.116D+03,0.490D+02,0.99480000D+00,0.00000000D+00
     .,0.27827870D+03,0.116D+03,0.500D+02,0.99480000D+00,0.00000000D+00
     .,0.25143440D+03,0.116D+03,0.510D+02,0.99480000D+00,0.00000000D+00
     .,0.23525190D+03,0.116D+03,0.520D+02,0.99480000D+00,0.00000000D+00
     .,0.21460020D+03,0.116D+03,0.530D+02,0.99480000D+00,0.00000000D+00
     .,0.19452370D+03,0.116D+03,0.540D+02,0.99480000D+00,0.00000000D+00
     .,0.72846530D+03,0.116D+03,0.550D+02,0.99480000D+00,0.00000000D+00
     .,0.68423530D+03,0.116D+03,0.560D+02,0.99480000D+00,0.00000000D+00
     .,0.60944200D+03,0.116D+03,0.570D+02,0.99480000D+00,0.00000000D+00
     .,0.29645810D+03,0.116D+03,0.580D+02,0.99480000D+00,0.27991000D+01
     .,0.60891190D+03,0.116D+03,0.590D+02,0.99480000D+00,0.00000000D+00
     .,0.58591340D+03,0.116D+03,0.600D+02,0.99480000D+00,0.00000000D+00
     .,0.57153970D+03,0.116D+03,0.610D+02,0.99480000D+00,0.00000000D+00
     .,0.55828160D+03,0.116D+03,0.620D+02,0.99480000D+00,0.00000000D+00
     ./)
      pars(  29201:  29300)=(/
     . 0.54653950D+03,0.116D+03,0.630D+02,0.99480000D+00,0.00000000D+00
     .,0.43677850D+03,0.116D+03,0.640D+02,0.99480000D+00,0.00000000D+00
     .,0.48158940D+03,0.116D+03,0.650D+02,0.99480000D+00,0.00000000D+00
     .,0.46577580D+03,0.116D+03,0.660D+02,0.99480000D+00,0.00000000D+00
     .,0.49452850D+03,0.116D+03,0.670D+02,0.99480000D+00,0.00000000D+00
     .,0.48417910D+03,0.116D+03,0.680D+02,0.99480000D+00,0.00000000D+00
     .,0.47494430D+03,0.116D+03,0.690D+02,0.99480000D+00,0.00000000D+00
     .,0.46905670D+03,0.116D+03,0.700D+02,0.99480000D+00,0.00000000D+00
     .,0.39959520D+03,0.116D+03,0.710D+02,0.99480000D+00,0.00000000D+00
     .,0.39859150D+03,0.116D+03,0.720D+02,0.99480000D+00,0.00000000D+00
     .,0.36694770D+03,0.116D+03,0.730D+02,0.99480000D+00,0.00000000D+00
     .,0.31220810D+03,0.116D+03,0.740D+02,0.99480000D+00,0.00000000D+00
     .,0.31855560D+03,0.116D+03,0.750D+02,0.99480000D+00,0.00000000D+00
     .,0.29082630D+03,0.116D+03,0.760D+02,0.99480000D+00,0.00000000D+00
     .,0.26791140D+03,0.116D+03,0.770D+02,0.99480000D+00,0.00000000D+00
     .,0.22401000D+03,0.116D+03,0.780D+02,0.99480000D+00,0.00000000D+00
     .,0.20979980D+03,0.116D+03,0.790D+02,0.99480000D+00,0.00000000D+00
     .,0.21632530D+03,0.116D+03,0.800D+02,0.99480000D+00,0.00000000D+00
     .,0.30615090D+03,0.116D+03,0.810D+02,0.99480000D+00,0.00000000D+00
     .,0.30235050D+03,0.116D+03,0.820D+02,0.99480000D+00,0.00000000D+00
     ./)
      pars(  29301:  29400)=(/
     . 0.28117730D+03,0.116D+03,0.830D+02,0.99480000D+00,0.00000000D+00
     .,0.27007790D+03,0.116D+03,0.840D+02,0.99480000D+00,0.00000000D+00
     .,0.25135500D+03,0.116D+03,0.850D+02,0.99480000D+00,0.00000000D+00
     .,0.23212580D+03,0.116D+03,0.860D+02,0.99480000D+00,0.00000000D+00
     .,0.69553530D+03,0.116D+03,0.870D+02,0.99480000D+00,0.00000000D+00
     .,0.68165580D+03,0.116D+03,0.880D+02,0.99480000D+00,0.00000000D+00
     .,0.60993410D+03,0.116D+03,0.890D+02,0.99480000D+00,0.00000000D+00
     .,0.55624920D+03,0.116D+03,0.900D+02,0.99480000D+00,0.00000000D+00
     .,0.54849710D+03,0.116D+03,0.910D+02,0.99480000D+00,0.00000000D+00
     .,0.53125540D+03,0.116D+03,0.920D+02,0.99480000D+00,0.00000000D+00
     .,0.54187030D+03,0.116D+03,0.930D+02,0.99480000D+00,0.00000000D+00
     .,0.52557770D+03,0.116D+03,0.940D+02,0.99480000D+00,0.00000000D+00
     .,0.31412700D+02,0.116D+03,0.101D+03,0.99480000D+00,0.00000000D+00
     .,0.97084900D+02,0.116D+03,0.103D+03,0.99480000D+00,0.98650000D+00
     .,0.12475120D+03,0.116D+03,0.104D+03,0.99480000D+00,0.98080000D+00
     .,0.98113900D+02,0.116D+03,0.105D+03,0.99480000D+00,0.97060000D+00
     .,0.75203900D+02,0.116D+03,0.106D+03,0.99480000D+00,0.98680000D+00
     .,0.53208600D+02,0.116D+03,0.107D+03,0.99480000D+00,0.99440000D+00
     .,0.39268300D+02,0.116D+03,0.108D+03,0.99480000D+00,0.99250000D+00
     .,0.27411700D+02,0.116D+03,0.109D+03,0.99480000D+00,0.99820000D+00
     ./)
      pars(  29401:  29500)=(/
     . 0.14105990D+03,0.116D+03,0.111D+03,0.99480000D+00,0.96840000D+00
     .,0.21766290D+03,0.116D+03,0.112D+03,0.99480000D+00,0.96280000D+00
     .,0.22379420D+03,0.116D+03,0.113D+03,0.99480000D+00,0.96480000D+00
     .,0.18394830D+03,0.116D+03,0.114D+03,0.99480000D+00,0.95070000D+00
     .,0.15318230D+03,0.116D+03,0.115D+03,0.99480000D+00,0.99470000D+00
     .,0.13099650D+03,0.116D+03,0.116D+03,0.99480000D+00,0.99480000D+00
     .,0.16527300D+02,0.117D+03,0.100D+01,0.99720000D+00,0.91180000D+00
     .,0.11509200D+02,0.117D+03,0.200D+01,0.99720000D+00,0.00000000D+00
     .,0.20877430D+03,0.117D+03,0.300D+01,0.99720000D+00,0.00000000D+00
     .,0.13164980D+03,0.117D+03,0.400D+01,0.99720000D+00,0.00000000D+00
     .,0.93774300D+02,0.117D+03,0.500D+01,0.99720000D+00,0.00000000D+00
     .,0.66135400D+02,0.117D+03,0.600D+01,0.99720000D+00,0.00000000D+00
     .,0.47754000D+02,0.117D+03,0.700D+01,0.99720000D+00,0.00000000D+00
     .,0.36978300D+02,0.117D+03,0.800D+01,0.99720000D+00,0.00000000D+00
     .,0.28546800D+02,0.117D+03,0.900D+01,0.99720000D+00,0.00000000D+00
     .,0.22283200D+02,0.117D+03,0.100D+02,0.99720000D+00,0.00000000D+00
     .,0.25123260D+03,0.117D+03,0.110D+02,0.99720000D+00,0.00000000D+00
     .,0.20645210D+03,0.117D+03,0.120D+02,0.99720000D+00,0.00000000D+00
     .,0.19576470D+03,0.117D+03,0.130D+02,0.99720000D+00,0.00000000D+00
     .,0.16018790D+03,0.117D+03,0.140D+02,0.99720000D+00,0.00000000D+00
     ./)
      pars(  29501:  29600)=(/
     . 0.12909560D+03,0.117D+03,0.150D+02,0.99720000D+00,0.00000000D+00
     .,0.10952700D+03,0.117D+03,0.160D+02,0.99720000D+00,0.00000000D+00
     .,0.91366700D+02,0.117D+03,0.170D+02,0.99720000D+00,0.00000000D+00
     .,0.76125600D+02,0.117D+03,0.180D+02,0.99720000D+00,0.00000000D+00
     .,0.40951050D+03,0.117D+03,0.190D+02,0.99720000D+00,0.00000000D+00
     .,0.35409350D+03,0.117D+03,0.200D+02,0.99720000D+00,0.00000000D+00
     .,0.29590280D+03,0.117D+03,0.210D+02,0.99720000D+00,0.00000000D+00
     .,0.28935120D+03,0.117D+03,0.220D+02,0.99720000D+00,0.00000000D+00
     .,0.26685570D+03,0.117D+03,0.230D+02,0.99720000D+00,0.00000000D+00
     .,0.21129280D+03,0.117D+03,0.240D+02,0.99720000D+00,0.00000000D+00
     .,0.23214470D+03,0.117D+03,0.250D+02,0.99720000D+00,0.00000000D+00
     .,0.18336610D+03,0.117D+03,0.260D+02,0.99720000D+00,0.00000000D+00
     .,0.19637550D+03,0.117D+03,0.270D+02,0.99720000D+00,0.00000000D+00
     .,0.20075710D+03,0.117D+03,0.280D+02,0.99720000D+00,0.00000000D+00
     .,0.15477560D+03,0.117D+03,0.290D+02,0.99720000D+00,0.00000000D+00
     .,0.16173370D+03,0.117D+03,0.300D+02,0.99720000D+00,0.00000000D+00
     .,0.19011390D+03,0.117D+03,0.310D+02,0.99720000D+00,0.00000000D+00
     .,0.17239330D+03,0.117D+03,0.320D+02,0.99720000D+00,0.00000000D+00
     .,0.15099990D+03,0.117D+03,0.330D+02,0.99720000D+00,0.00000000D+00
     .,0.13784430D+03,0.117D+03,0.340D+02,0.99720000D+00,0.00000000D+00
     ./)
      pars(  29601:  29700)=(/
     . 0.12277630D+03,0.117D+03,0.350D+02,0.99720000D+00,0.00000000D+00
     .,0.10852260D+03,0.117D+03,0.360D+02,0.99720000D+00,0.00000000D+00
     .,0.46202520D+03,0.117D+03,0.370D+02,0.99720000D+00,0.00000000D+00
     .,0.42188670D+03,0.117D+03,0.380D+02,0.99720000D+00,0.00000000D+00
     .,0.37756420D+03,0.117D+03,0.390D+02,0.99720000D+00,0.00000000D+00
     .,0.34413180D+03,0.117D+03,0.400D+02,0.99720000D+00,0.00000000D+00
     .,0.31691100D+03,0.117D+03,0.410D+02,0.99720000D+00,0.00000000D+00
     .,0.24926710D+03,0.117D+03,0.420D+02,0.99720000D+00,0.00000000D+00
     .,0.27614550D+03,0.117D+03,0.430D+02,0.99720000D+00,0.00000000D+00
     .,0.21463800D+03,0.117D+03,0.440D+02,0.99720000D+00,0.00000000D+00
     .,0.23390310D+03,0.117D+03,0.450D+02,0.99720000D+00,0.00000000D+00
     .,0.21819030D+03,0.117D+03,0.460D+02,0.99720000D+00,0.00000000D+00
     .,0.18215470D+03,0.117D+03,0.470D+02,0.99720000D+00,0.00000000D+00
     .,0.19372730D+03,0.117D+03,0.480D+02,0.99720000D+00,0.00000000D+00
     .,0.23849880D+03,0.117D+03,0.490D+02,0.99720000D+00,0.00000000D+00
     .,0.22523810D+03,0.117D+03,0.500D+02,0.99720000D+00,0.00000000D+00
     .,0.20525120D+03,0.117D+03,0.510D+02,0.99720000D+00,0.00000000D+00
     .,0.19314490D+03,0.117D+03,0.520D+02,0.99720000D+00,0.00000000D+00
     .,0.17734580D+03,0.117D+03,0.530D+02,0.99720000D+00,0.00000000D+00
     .,0.16181190D+03,0.117D+03,0.540D+02,0.99720000D+00,0.00000000D+00
     ./)
      pars(  29701:  29800)=(/
     . 0.56460060D+03,0.117D+03,0.550D+02,0.99720000D+00,0.00000000D+00
     .,0.53526780D+03,0.117D+03,0.560D+02,0.99720000D+00,0.00000000D+00
     .,0.48012440D+03,0.117D+03,0.570D+02,0.99720000D+00,0.00000000D+00
     .,0.24192230D+03,0.117D+03,0.580D+02,0.99720000D+00,0.27991000D+01
     .,0.47771970D+03,0.117D+03,0.590D+02,0.99720000D+00,0.00000000D+00
     .,0.46012500D+03,0.117D+03,0.600D+02,0.99720000D+00,0.00000000D+00
     .,0.44895170D+03,0.117D+03,0.610D+02,0.99720000D+00,0.00000000D+00
     .,0.43862430D+03,0.117D+03,0.620D+02,0.99720000D+00,0.00000000D+00
     .,0.42948100D+03,0.117D+03,0.630D+02,0.99720000D+00,0.00000000D+00
     .,0.34664610D+03,0.117D+03,0.640D+02,0.99720000D+00,0.00000000D+00
     .,0.37839840D+03,0.117D+03,0.650D+02,0.99720000D+00,0.00000000D+00
     .,0.36651660D+03,0.117D+03,0.660D+02,0.99720000D+00,0.00000000D+00
     .,0.38919850D+03,0.117D+03,0.670D+02,0.99720000D+00,0.00000000D+00
     .,0.38108470D+03,0.117D+03,0.680D+02,0.99720000D+00,0.00000000D+00
     .,0.37388780D+03,0.117D+03,0.690D+02,0.99720000D+00,0.00000000D+00
     .,0.36907690D+03,0.117D+03,0.700D+02,0.99720000D+00,0.00000000D+00
     .,0.31653270D+03,0.117D+03,0.710D+02,0.99720000D+00,0.00000000D+00
     .,0.31792390D+03,0.117D+03,0.720D+02,0.99720000D+00,0.00000000D+00
     .,0.29429820D+03,0.117D+03,0.730D+02,0.99720000D+00,0.00000000D+00
     .,0.25196160D+03,0.117D+03,0.740D+02,0.99720000D+00,0.00000000D+00
     ./)
      pars(  29801:  29900)=(/
     . 0.25744470D+03,0.117D+03,0.750D+02,0.99720000D+00,0.00000000D+00
     .,0.23620650D+03,0.117D+03,0.760D+02,0.99720000D+00,0.00000000D+00
     .,0.21851340D+03,0.117D+03,0.770D+02,0.99720000D+00,0.00000000D+00
     .,0.18384350D+03,0.117D+03,0.780D+02,0.99720000D+00,0.00000000D+00
     .,0.17259550D+03,0.117D+03,0.790D+02,0.99720000D+00,0.00000000D+00
     .,0.17806050D+03,0.117D+03,0.800D+02,0.99720000D+00,0.00000000D+00
     .,0.24721900D+03,0.117D+03,0.810D+02,0.99720000D+00,0.00000000D+00
     .,0.24529500D+03,0.117D+03,0.820D+02,0.99720000D+00,0.00000000D+00
     .,0.22972670D+03,0.117D+03,0.830D+02,0.99720000D+00,0.00000000D+00
     .,0.22165670D+03,0.117D+03,0.840D+02,0.99720000D+00,0.00000000D+00
     .,0.20750080D+03,0.117D+03,0.850D+02,0.99720000D+00,0.00000000D+00
     .,0.19273160D+03,0.117D+03,0.860D+02,0.99720000D+00,0.00000000D+00
     .,0.54229530D+03,0.117D+03,0.870D+02,0.99720000D+00,0.00000000D+00
     .,0.53547060D+03,0.117D+03,0.880D+02,0.99720000D+00,0.00000000D+00
     .,0.48217720D+03,0.117D+03,0.890D+02,0.99720000D+00,0.00000000D+00
     .,0.44362620D+03,0.117D+03,0.900D+02,0.99720000D+00,0.00000000D+00
     .,0.43615260D+03,0.117D+03,0.910D+02,0.99720000D+00,0.00000000D+00
     .,0.42258040D+03,0.117D+03,0.920D+02,0.99720000D+00,0.00000000D+00
     .,0.42886190D+03,0.117D+03,0.930D+02,0.99720000D+00,0.00000000D+00
     .,0.41631150D+03,0.117D+03,0.940D+02,0.99720000D+00,0.00000000D+00
     ./)
      pars(  29901:  30000)=(/
     . 0.25741400D+02,0.117D+03,0.101D+03,0.99720000D+00,0.00000000D+00
     .,0.77202300D+02,0.117D+03,0.103D+03,0.99720000D+00,0.98650000D+00
     .,0.99699500D+02,0.117D+03,0.104D+03,0.99720000D+00,0.98080000D+00
     .,0.79883300D+02,0.117D+03,0.105D+03,0.99720000D+00,0.97060000D+00
     .,0.62086400D+02,0.117D+03,0.106D+03,0.99720000D+00,0.98680000D+00
     .,0.44621700D+02,0.117D+03,0.107D+03,0.99720000D+00,0.99440000D+00
     .,0.33380100D+02,0.117D+03,0.108D+03,0.99720000D+00,0.99250000D+00
     .,0.23705900D+02,0.117D+03,0.109D+03,0.99720000D+00,0.99820000D+00
     .,0.11198290D+03,0.117D+03,0.111D+03,0.99720000D+00,0.96840000D+00
     .,0.17241480D+03,0.117D+03,0.112D+03,0.99720000D+00,0.96280000D+00
     .,0.17871270D+03,0.117D+03,0.113D+03,0.99720000D+00,0.96480000D+00
     .,0.14895590D+03,0.117D+03,0.114D+03,0.99720000D+00,0.95070000D+00
     .,0.12549300D+03,0.117D+03,0.115D+03,0.99720000D+00,0.99470000D+00
     .,0.10828930D+03,0.117D+03,0.116D+03,0.99720000D+00,0.99480000D+00
     .,0.90398500D+02,0.117D+03,0.117D+03,0.99720000D+00,0.99720000D+00
     .,0.29300000D+02,0.119D+03,0.100D+01,0.97670000D+00,0.91180000D+00
     .,0.18665500D+02,0.119D+03,0.200D+01,0.97670000D+00,0.00000000D+00
     .,0.56029540D+03,0.119D+03,0.300D+01,0.97670000D+00,0.00000000D+00
     .,0.29222240D+03,0.119D+03,0.400D+01,0.97670000D+00,0.00000000D+00
     .,0.18695660D+03,0.119D+03,0.500D+01,0.97670000D+00,0.00000000D+00
     ./)
      pars(  30001:  30100)=(/
     . 0.12192920D+03,0.119D+03,0.600D+01,0.97670000D+00,0.00000000D+00
     .,0.83215300D+02,0.119D+03,0.700D+01,0.97670000D+00,0.00000000D+00
     .,0.61994800D+02,0.119D+03,0.800D+01,0.97670000D+00,0.00000000D+00
     .,0.46352000D+02,0.119D+03,0.900D+01,0.97670000D+00,0.00000000D+00
     .,0.35296900D+02,0.119D+03,0.100D+02,0.97670000D+00,0.00000000D+00
     .,0.66603570D+03,0.119D+03,0.110D+02,0.97670000D+00,0.00000000D+00
     .,0.47376610D+03,0.119D+03,0.120D+02,0.97670000D+00,0.00000000D+00
     .,0.42489260D+03,0.119D+03,0.130D+02,0.97670000D+00,0.00000000D+00
     .,0.32282330D+03,0.119D+03,0.140D+02,0.97670000D+00,0.00000000D+00
     .,0.24457290D+03,0.119D+03,0.150D+02,0.97670000D+00,0.00000000D+00
     .,0.19946720D+03,0.119D+03,0.160D+02,0.97670000D+00,0.00000000D+00
     .,0.16033040D+03,0.119D+03,0.170D+02,0.97670000D+00,0.00000000D+00
     .,0.12942830D+03,0.119D+03,0.180D+02,0.97670000D+00,0.00000000D+00
     .,0.11122277D+04,0.119D+03,0.190D+02,0.97670000D+00,0.00000000D+00
     .,0.86017710D+03,0.119D+03,0.200D+02,0.97670000D+00,0.00000000D+00
     .,0.70072810D+03,0.119D+03,0.210D+02,0.97670000D+00,0.00000000D+00
     .,0.66875460D+03,0.119D+03,0.220D+02,0.97670000D+00,0.00000000D+00
     .,0.60799510D+03,0.119D+03,0.230D+02,0.97670000D+00,0.00000000D+00
     .,0.47897440D+03,0.119D+03,0.240D+02,0.97670000D+00,0.00000000D+00
     .,0.51805830D+03,0.119D+03,0.250D+02,0.97670000D+00,0.00000000D+00
     ./)
      pars(  30101:  30200)=(/
     . 0.40608760D+03,0.119D+03,0.260D+02,0.97670000D+00,0.00000000D+00
     .,0.42338090D+03,0.119D+03,0.270D+02,0.97670000D+00,0.00000000D+00
     .,0.43928450D+03,0.119D+03,0.280D+02,0.97670000D+00,0.00000000D+00
     .,0.33711170D+03,0.119D+03,0.290D+02,0.97670000D+00,0.00000000D+00
     .,0.33789860D+03,0.119D+03,0.300D+02,0.97670000D+00,0.00000000D+00
     .,0.40307450D+03,0.119D+03,0.310D+02,0.97670000D+00,0.00000000D+00
     .,0.34544340D+03,0.119D+03,0.320D+02,0.97670000D+00,0.00000000D+00
     .,0.28755350D+03,0.119D+03,0.330D+02,0.97670000D+00,0.00000000D+00
     .,0.25436700D+03,0.119D+03,0.340D+02,0.97670000D+00,0.00000000D+00
     .,0.21946470D+03,0.119D+03,0.350D+02,0.97670000D+00,0.00000000D+00
     .,0.18848190D+03,0.119D+03,0.360D+02,0.97670000D+00,0.00000000D+00
     .,0.12408631D+04,0.119D+03,0.370D+02,0.97670000D+00,0.00000000D+00
     .,0.10279671D+04,0.119D+03,0.380D+02,0.97670000D+00,0.00000000D+00
     .,0.87903460D+03,0.119D+03,0.390D+02,0.97670000D+00,0.00000000D+00
     .,0.77852380D+03,0.119D+03,0.400D+02,0.97670000D+00,0.00000000D+00
     .,0.70330600D+03,0.119D+03,0.410D+02,0.97670000D+00,0.00000000D+00
     .,0.53463030D+03,0.119D+03,0.420D+02,0.97670000D+00,0.00000000D+00
     .,0.59993020D+03,0.119D+03,0.430D+02,0.97670000D+00,0.00000000D+00
     .,0.44930300D+03,0.119D+03,0.440D+02,0.97670000D+00,0.00000000D+00
     .,0.49058070D+03,0.119D+03,0.450D+02,0.97670000D+00,0.00000000D+00
     ./)
      pars(  30201:  30300)=(/
     . 0.45224640D+03,0.119D+03,0.460D+02,0.97670000D+00,0.00000000D+00
     .,0.37908360D+03,0.119D+03,0.470D+02,0.97670000D+00,0.00000000D+00
     .,0.39551350D+03,0.119D+03,0.480D+02,0.97670000D+00,0.00000000D+00
     .,0.50568380D+03,0.119D+03,0.490D+02,0.97670000D+00,0.00000000D+00
     .,0.45646130D+03,0.119D+03,0.500D+02,0.97670000D+00,0.00000000D+00
     .,0.39784160D+03,0.119D+03,0.510D+02,0.97670000D+00,0.00000000D+00
     .,0.36440900D+03,0.119D+03,0.520D+02,0.97670000D+00,0.00000000D+00
     .,0.32510680D+03,0.119D+03,0.530D+02,0.97670000D+00,0.00000000D+00
     .,0.28872750D+03,0.119D+03,0.540D+02,0.97670000D+00,0.00000000D+00
     .,0.15118898D+04,0.119D+03,0.550D+02,0.97670000D+00,0.00000000D+00
     .,0.13222619D+04,0.119D+03,0.560D+02,0.97670000D+00,0.00000000D+00
     .,0.11337749D+04,0.119D+03,0.570D+02,0.97670000D+00,0.00000000D+00
     .,0.47224550D+03,0.119D+03,0.580D+02,0.97670000D+00,0.27991000D+01
     .,0.11622835D+04,0.119D+03,0.590D+02,0.97670000D+00,0.00000000D+00
     .,0.11112638D+04,0.119D+03,0.600D+02,0.97670000D+00,0.00000000D+00
     .,0.10820865D+04,0.119D+03,0.610D+02,0.97670000D+00,0.00000000D+00
     .,0.10553891D+04,0.119D+03,0.620D+02,0.97670000D+00,0.00000000D+00
     .,0.10316716D+04,0.119D+03,0.630D+02,0.97670000D+00,0.00000000D+00
     .,0.79078540D+03,0.119D+03,0.640D+02,0.97670000D+00,0.00000000D+00
     .,0.92736720D+03,0.119D+03,0.650D+02,0.97670000D+00,0.00000000D+00
     ./)
      pars(  30301:  30400)=(/
     . 0.89050560D+03,0.119D+03,0.660D+02,0.97670000D+00,0.00000000D+00
     .,0.92461970D+03,0.119D+03,0.670D+02,0.97670000D+00,0.00000000D+00
     .,0.90432570D+03,0.119D+03,0.680D+02,0.97670000D+00,0.00000000D+00
     .,0.88575380D+03,0.119D+03,0.690D+02,0.97670000D+00,0.00000000D+00
     .,0.87628990D+03,0.119D+03,0.700D+02,0.97670000D+00,0.00000000D+00
     .,0.72564410D+03,0.119D+03,0.710D+02,0.97670000D+00,0.00000000D+00
     .,0.69618310D+03,0.119D+03,0.720D+02,0.97670000D+00,0.00000000D+00
     .,0.62634420D+03,0.119D+03,0.730D+02,0.97670000D+00,0.00000000D+00
     .,0.52368920D+03,0.119D+03,0.740D+02,0.97670000D+00,0.00000000D+00
     .,0.52962490D+03,0.119D+03,0.750D+02,0.97670000D+00,0.00000000D+00
     .,0.47425390D+03,0.119D+03,0.760D+02,0.97670000D+00,0.00000000D+00
     .,0.43019640D+03,0.119D+03,0.770D+02,0.97670000D+00,0.00000000D+00
     .,0.35414110D+03,0.119D+03,0.780D+02,0.97670000D+00,0.00000000D+00
     .,0.32964710D+03,0.119D+03,0.790D+02,0.97670000D+00,0.00000000D+00
     .,0.33733350D+03,0.119D+03,0.800D+02,0.97670000D+00,0.00000000D+00
     .,0.51695330D+03,0.119D+03,0.810D+02,0.97670000D+00,0.00000000D+00
     .,0.49569810D+03,0.119D+03,0.820D+02,0.97670000D+00,0.00000000D+00
     .,0.44630330D+03,0.119D+03,0.830D+02,0.97670000D+00,0.00000000D+00
     .,0.42086840D+03,0.119D+03,0.840D+02,0.97670000D+00,0.00000000D+00
     .,0.38335330D+03,0.119D+03,0.850D+02,0.97670000D+00,0.00000000D+00
     ./)
      pars(  30401:  30500)=(/
     . 0.34734370D+03,0.119D+03,0.860D+02,0.97670000D+00,0.00000000D+00
     .,0.14001686D+04,0.119D+03,0.870D+02,0.97670000D+00,0.00000000D+00
     .,0.12915073D+04,0.119D+03,0.880D+02,0.97670000D+00,0.00000000D+00
     .,0.11159288D+04,0.119D+03,0.890D+02,0.97670000D+00,0.00000000D+00
     .,0.97819280D+03,0.119D+03,0.900D+02,0.97670000D+00,0.00000000D+00
     .,0.98391310D+03,0.119D+03,0.910D+02,0.97670000D+00,0.00000000D+00
     .,0.95206270D+03,0.119D+03,0.920D+02,0.97670000D+00,0.00000000D+00
     .,0.99535270D+03,0.119D+03,0.930D+02,0.97670000D+00,0.00000000D+00
     .,0.96105430D+03,0.119D+03,0.940D+02,0.97670000D+00,0.00000000D+00
     .,0.48449900D+02,0.119D+03,0.101D+03,0.97670000D+00,0.00000000D+00
     .,0.16874120D+03,0.119D+03,0.103D+03,0.97670000D+00,0.98650000D+00
     .,0.21344440D+03,0.119D+03,0.104D+03,0.97670000D+00,0.98080000D+00
     .,0.15577810D+03,0.119D+03,0.105D+03,0.97670000D+00,0.97060000D+00
     .,0.11457230D+03,0.119D+03,0.106D+03,0.97670000D+00,0.98680000D+00
     .,0.77631700D+02,0.119D+03,0.107D+03,0.97670000D+00,0.99440000D+00
     .,0.55403200D+02,0.119D+03,0.108D+03,0.97670000D+00,0.99250000D+00
     .,0.37198200D+02,0.119D+03,0.109D+03,0.97670000D+00,0.99820000D+00
     .,0.24922000D+03,0.119D+03,0.111D+03,0.97670000D+00,0.96840000D+00
     .,0.38733490D+03,0.119D+03,0.112D+03,0.97670000D+00,0.96280000D+00
     .,0.38264220D+03,0.119D+03,0.113D+03,0.97670000D+00,0.96480000D+00
     ./)
      pars(  30501:  30600)=(/
     . 0.29694090D+03,0.119D+03,0.114D+03,0.97670000D+00,0.95070000D+00
     .,0.23721910D+03,0.119D+03,0.115D+03,0.97670000D+00,0.99470000D+00
     .,0.19738060D+03,0.119D+03,0.116D+03,0.97670000D+00,0.99480000D+00
     .,0.15873300D+03,0.119D+03,0.117D+03,0.97670000D+00,0.99720000D+00
     .,0.33802070D+03,0.119D+03,0.119D+03,0.97670000D+00,0.97670000D+00
     .,0.52987100D+02,0.120D+03,0.100D+01,0.98310000D+00,0.91180000D+00
     .,0.31879600D+02,0.120D+03,0.200D+01,0.98310000D+00,0.00000000D+00
     .,0.13100314D+04,0.120D+03,0.300D+01,0.98310000D+00,0.00000000D+00
     .,0.60256610D+03,0.120D+03,0.400D+01,0.98310000D+00,0.00000000D+00
     .,0.36268620D+03,0.120D+03,0.500D+01,0.98310000D+00,0.00000000D+00
     .,0.22596290D+03,0.120D+03,0.600D+01,0.98310000D+00,0.00000000D+00
     .,0.14897930D+03,0.120D+03,0.700D+01,0.98310000D+00,0.00000000D+00
     .,0.10827180D+03,0.120D+03,0.800D+01,0.98310000D+00,0.00000000D+00
     .,0.79221800D+02,0.120D+03,0.900D+01,0.98310000D+00,0.00000000D+00
     .,0.59277900D+02,0.120D+03,0.100D+02,0.98310000D+00,0.00000000D+00
     .,0.15457064D+04,0.120D+03,0.110D+02,0.98310000D+00,0.00000000D+00
     .,0.99724860D+03,0.120D+03,0.120D+02,0.98310000D+00,0.00000000D+00
     .,0.86695170D+03,0.120D+03,0.130D+02,0.98310000D+00,0.00000000D+00
     .,0.63004560D+03,0.120D+03,0.140D+02,0.98310000D+00,0.00000000D+00
     .,0.46026350D+03,0.120D+03,0.150D+02,0.98310000D+00,0.00000000D+00
     ./)
      pars(  30601:  30700)=(/
     . 0.36677110D+03,0.120D+03,0.160D+02,0.98310000D+00,0.00000000D+00
     .,0.28821820D+03,0.120D+03,0.170D+02,0.98310000D+00,0.00000000D+00
     .,0.22808920D+03,0.120D+03,0.180D+02,0.98310000D+00,0.00000000D+00
     .,0.26610830D+04,0.120D+03,0.190D+02,0.98310000D+00,0.00000000D+00
     .,0.18864221D+04,0.120D+03,0.200D+02,0.98310000D+00,0.00000000D+00
     .,0.15101427D+04,0.120D+03,0.210D+02,0.98310000D+00,0.00000000D+00
     .,0.14223244D+04,0.120D+03,0.220D+02,0.98310000D+00,0.00000000D+00
     .,0.12821610D+04,0.120D+03,0.230D+02,0.98310000D+00,0.00000000D+00
     .,0.10115955D+04,0.120D+03,0.240D+02,0.98310000D+00,0.00000000D+00
     .,0.10791532D+04,0.120D+03,0.250D+02,0.98310000D+00,0.00000000D+00
     .,0.84546860D+03,0.120D+03,0.260D+02,0.98310000D+00,0.00000000D+00
     .,0.86250360D+03,0.120D+03,0.270D+02,0.98310000D+00,0.00000000D+00
     .,0.90260140D+03,0.120D+03,0.280D+02,0.98310000D+00,0.00000000D+00
     .,0.69450580D+03,0.120D+03,0.290D+02,0.98310000D+00,0.00000000D+00
     .,0.67502990D+03,0.120D+03,0.300D+02,0.98310000D+00,0.00000000D+00
     .,0.81349430D+03,0.120D+03,0.310D+02,0.98310000D+00,0.00000000D+00
     .,0.67269170D+03,0.120D+03,0.320D+02,0.98310000D+00,0.00000000D+00
     .,0.54301230D+03,0.120D+03,0.330D+02,0.98310000D+00,0.00000000D+00
     .,0.47148220D+03,0.120D+03,0.340D+02,0.98310000D+00,0.00000000D+00
     .,0.39902260D+03,0.120D+03,0.350D+02,0.98310000D+00,0.00000000D+00
     ./)
      pars(  30701:  30800)=(/
     . 0.33664830D+03,0.120D+03,0.360D+02,0.98310000D+00,0.00000000D+00
     .,0.29580410D+04,0.120D+03,0.370D+02,0.98310000D+00,0.00000000D+00
     .,0.22654155D+04,0.120D+03,0.380D+02,0.98310000D+00,0.00000000D+00
     .,0.18821338D+04,0.120D+03,0.390D+02,0.98310000D+00,0.00000000D+00
     .,0.16379667D+04,0.120D+03,0.400D+02,0.98310000D+00,0.00000000D+00
     .,0.14638890D+04,0.120D+03,0.410D+02,0.98310000D+00,0.00000000D+00
     .,0.10926733D+04,0.120D+03,0.420D+02,0.98310000D+00,0.00000000D+00
     .,0.12349316D+04,0.120D+03,0.430D+02,0.98310000D+00,0.00000000D+00
     .,0.90575150D+03,0.120D+03,0.440D+02,0.98310000D+00,0.00000000D+00
     .,0.98637270D+03,0.120D+03,0.450D+02,0.98310000D+00,0.00000000D+00
     .,0.90244050D+03,0.120D+03,0.460D+02,0.98310000D+00,0.00000000D+00
     .,0.76261740D+03,0.120D+03,0.470D+02,0.98310000D+00,0.00000000D+00
     .,0.78166000D+03,0.120D+03,0.480D+02,0.98310000D+00,0.00000000D+00
     .,0.10241614D+04,0.120D+03,0.490D+02,0.98310000D+00,0.00000000D+00
     .,0.89641550D+03,0.120D+03,0.500D+02,0.98310000D+00,0.00000000D+00
     .,0.75979880D+03,0.120D+03,0.510D+02,0.98310000D+00,0.00000000D+00
     .,0.68470730D+03,0.120D+03,0.520D+02,0.98310000D+00,0.00000000D+00
     .,0.60026980D+03,0.120D+03,0.530D+02,0.98310000D+00,0.00000000D+00
     .,0.52435470D+03,0.120D+03,0.540D+02,0.98310000D+00,0.00000000D+00
     .,0.36192660D+04,0.120D+03,0.550D+02,0.98310000D+00,0.00000000D+00
     ./)
      pars(  30801:  30900)=(/
     . 0.29513362D+04,0.120D+03,0.560D+02,0.98310000D+00,0.00000000D+00
     .,0.24567601D+04,0.120D+03,0.570D+02,0.98310000D+00,0.00000000D+00
     .,0.90740120D+03,0.120D+03,0.580D+02,0.98310000D+00,0.27991000D+01
     .,0.25715319D+04,0.120D+03,0.590D+02,0.98310000D+00,0.00000000D+00
     .,0.24420495D+04,0.120D+03,0.600D+02,0.98310000D+00,0.00000000D+00
     .,0.23740445D+04,0.120D+03,0.610D+02,0.98310000D+00,0.00000000D+00
     .,0.23122147D+04,0.120D+03,0.620D+02,0.98310000D+00,0.00000000D+00
     .,0.22571993D+04,0.120D+03,0.630D+02,0.98310000D+00,0.00000000D+00
     .,0.16806338D+04,0.120D+03,0.640D+02,0.98310000D+00,0.00000000D+00
     .,0.20816173D+04,0.120D+03,0.650D+02,0.98310000D+00,0.00000000D+00
     .,0.19928728D+04,0.120D+03,0.660D+02,0.98310000D+00,0.00000000D+00
     .,0.20062958D+04,0.120D+03,0.670D+02,0.98310000D+00,0.00000000D+00
     .,0.19602391D+04,0.120D+03,0.680D+02,0.98310000D+00,0.00000000D+00
     .,0.19174640D+04,0.120D+03,0.690D+02,0.98310000D+00,0.00000000D+00
     .,0.18991306D+04,0.120D+03,0.700D+02,0.98310000D+00,0.00000000D+00
     .,0.15446852D+04,0.120D+03,0.710D+02,0.98310000D+00,0.00000000D+00
     .,0.14351503D+04,0.120D+03,0.720D+02,0.98310000D+00,0.00000000D+00
     .,0.12687518D+04,0.120D+03,0.730D+02,0.98310000D+00,0.00000000D+00
     .,0.10493296D+04,0.120D+03,0.740D+02,0.98310000D+00,0.00000000D+00
     .,0.10529922D+04,0.120D+03,0.750D+02,0.98310000D+00,0.00000000D+00
     ./)
      pars(  30901:  31000)=(/
     . 0.92845210D+03,0.120D+03,0.760D+02,0.98310000D+00,0.00000000D+00
     .,0.83187520D+03,0.120D+03,0.770D+02,0.98310000D+00,0.00000000D+00
     .,0.67699020D+03,0.120D+03,0.780D+02,0.98310000D+00,0.00000000D+00
     .,0.62708530D+03,0.120D+03,0.790D+02,0.98310000D+00,0.00000000D+00
     .,0.63711120D+03,0.120D+03,0.800D+02,0.98310000D+00,0.00000000D+00
     .,0.10430556D+04,0.120D+03,0.810D+02,0.98310000D+00,0.00000000D+00
     .,0.97428450D+03,0.120D+03,0.820D+02,0.98310000D+00,0.00000000D+00
     .,0.85466400D+03,0.120D+03,0.830D+02,0.98310000D+00,0.00000000D+00
     .,0.79444910D+03,0.120D+03,0.840D+02,0.98310000D+00,0.00000000D+00
     .,0.71141430D+03,0.120D+03,0.850D+02,0.98310000D+00,0.00000000D+00
     .,0.63479210D+03,0.120D+03,0.860D+02,0.98310000D+00,0.00000000D+00
     .,0.32712563D+04,0.120D+03,0.870D+02,0.98310000D+00,0.00000000D+00
     .,0.28409447D+04,0.120D+03,0.880D+02,0.98310000D+00,0.00000000D+00
     .,0.23867637D+04,0.120D+03,0.890D+02,0.98310000D+00,0.00000000D+00
     .,0.20296245D+04,0.120D+03,0.900D+02,0.98310000D+00,0.00000000D+00
     .,0.20761172D+04,0.120D+03,0.910D+02,0.98310000D+00,0.00000000D+00
     .,0.20066915D+04,0.120D+03,0.920D+02,0.98310000D+00,0.00000000D+00
     .,0.21352269D+04,0.120D+03,0.930D+02,0.98310000D+00,0.00000000D+00
     .,0.20540131D+04,0.120D+03,0.940D+02,0.98310000D+00,0.00000000D+00
     .,0.90792700D+02,0.120D+03,0.101D+03,0.98310000D+00,0.00000000D+00
     ./)
      pars(  31001:  31100)=(/
     . 0.34561200D+03,0.120D+03,0.103D+03,0.98310000D+00,0.98650000D+00
     .,0.43402840D+03,0.120D+03,0.104D+03,0.98310000D+00,0.98080000D+00
     .,0.29872640D+03,0.120D+03,0.105D+03,0.98310000D+00,0.97060000D+00
     .,0.21289570D+03,0.120D+03,0.106D+03,0.98310000D+00,0.98680000D+00
     .,0.13902190D+03,0.120D+03,0.107D+03,0.98310000D+00,0.99440000D+00
     .,0.96158400D+02,0.120D+03,0.108D+03,0.98310000D+00,0.99250000D+00
     .,0.62058000D+02,0.120D+03,0.109D+03,0.98310000D+00,0.99820000D+00
     .,0.51669170D+03,0.120D+03,0.111D+03,0.98310000D+00,0.96840000D+00
     .,0.80921820D+03,0.120D+03,0.112D+03,0.98310000D+00,0.96280000D+00
     .,0.77508320D+03,0.120D+03,0.113D+03,0.98310000D+00,0.96480000D+00
     .,0.57629620D+03,0.120D+03,0.114D+03,0.98310000D+00,0.95070000D+00
     .,0.44601700D+03,0.120D+03,0.115D+03,0.98310000D+00,0.99470000D+00
     .,0.36319510D+03,0.120D+03,0.116D+03,0.98310000D+00,0.99480000D+00
     .,0.28549050D+03,0.120D+03,0.117D+03,0.98310000D+00,0.99720000D+00
     .,0.68899110D+03,0.120D+03,0.119D+03,0.98310000D+00,0.97670000D+00
     .,0.15251891D+04,0.120D+03,0.120D+03,0.98310000D+00,0.98310000D+00
     .,0.29995300D+02,0.121D+03,0.100D+01,0.18627000D+01,0.91180000D+00
     .,0.19384100D+02,0.121D+03,0.200D+01,0.18627000D+01,0.00000000D+00
     .,0.50482680D+03,0.121D+03,0.300D+01,0.18627000D+01,0.00000000D+00
     .,0.28244200D+03,0.121D+03,0.400D+01,0.18627000D+01,0.00000000D+00
     ./)
      pars(  31101:  31200)=(/
     . 0.18614190D+03,0.121D+03,0.500D+01,0.18627000D+01,0.00000000D+00
     .,0.12363790D+03,0.121D+03,0.600D+01,0.18627000D+01,0.00000000D+00
     .,0.85340400D+02,0.121D+03,0.700D+01,0.18627000D+01,0.00000000D+00
     .,0.64007200D+02,0.121D+03,0.800D+01,0.18627000D+01,0.00000000D+00
     .,0.48093600D+02,0.121D+03,0.900D+01,0.18627000D+01,0.00000000D+00
     .,0.36746800D+02,0.121D+03,0.100D+02,0.18627000D+01,0.00000000D+00
     .,0.60249290D+03,0.121D+03,0.110D+02,0.18627000D+01,0.00000000D+00
     .,0.45276000D+03,0.121D+03,0.120D+02,0.18627000D+01,0.00000000D+00
     .,0.41290580D+03,0.121D+03,0.130D+02,0.18627000D+01,0.00000000D+00
     .,0.32058960D+03,0.121D+03,0.140D+02,0.18627000D+01,0.00000000D+00
     .,0.24673580D+03,0.121D+03,0.150D+02,0.18627000D+01,0.00000000D+00
     .,0.20296720D+03,0.121D+03,0.160D+02,0.18627000D+01,0.00000000D+00
     .,0.16436550D+03,0.121D+03,0.170D+02,0.18627000D+01,0.00000000D+00
     .,0.13344040D+03,0.121D+03,0.180D+02,0.18627000D+01,0.00000000D+00
     .,0.98858810D+03,0.121D+03,0.190D+02,0.18627000D+01,0.00000000D+00
     .,0.80323800D+03,0.121D+03,0.200D+02,0.18627000D+01,0.00000000D+00
     .,0.66067800D+03,0.121D+03,0.210D+02,0.18627000D+01,0.00000000D+00
     .,0.63504570D+03,0.121D+03,0.220D+02,0.18627000D+01,0.00000000D+00
     .,0.57995840D+03,0.121D+03,0.230D+02,0.18627000D+01,0.00000000D+00
     .,0.45620230D+03,0.121D+03,0.240D+02,0.18627000D+01,0.00000000D+00
     ./)
      pars(  31201:  31300)=(/
     . 0.49734740D+03,0.121D+03,0.250D+02,0.18627000D+01,0.00000000D+00
     .,0.38959740D+03,0.121D+03,0.260D+02,0.18627000D+01,0.00000000D+00
     .,0.41105720D+03,0.121D+03,0.270D+02,0.18627000D+01,0.00000000D+00
     .,0.42466930D+03,0.121D+03,0.280D+02,0.18627000D+01,0.00000000D+00
     .,0.32508500D+03,0.121D+03,0.290D+02,0.18627000D+01,0.00000000D+00
     .,0.33123100D+03,0.121D+03,0.300D+02,0.18627000D+01,0.00000000D+00
     .,0.39353000D+03,0.121D+03,0.310D+02,0.18627000D+01,0.00000000D+00
     .,0.34316790D+03,0.121D+03,0.320D+02,0.18627000D+01,0.00000000D+00
     .,0.28955820D+03,0.121D+03,0.330D+02,0.18627000D+01,0.00000000D+00
     .,0.25800840D+03,0.121D+03,0.340D+02,0.18627000D+01,0.00000000D+00
     .,0.22413750D+03,0.121D+03,0.350D+02,0.18627000D+01,0.00000000D+00
     .,0.19358680D+03,0.121D+03,0.360D+02,0.18627000D+01,0.00000000D+00
     .,0.11054608D+04,0.121D+03,0.370D+02,0.18627000D+01,0.00000000D+00
     .,0.95722710D+03,0.121D+03,0.380D+02,0.18627000D+01,0.00000000D+00
     .,0.83166790D+03,0.121D+03,0.390D+02,0.18627000D+01,0.00000000D+00
     .,0.74343180D+03,0.121D+03,0.400D+02,0.18627000D+01,0.00000000D+00
     .,0.67533140D+03,0.121D+03,0.410D+02,0.18627000D+01,0.00000000D+00
     .,0.51773400D+03,0.121D+03,0.420D+02,0.18627000D+01,0.00000000D+00
     .,0.57916230D+03,0.121D+03,0.430D+02,0.18627000D+01,0.00000000D+00
     .,0.43784320D+03,0.121D+03,0.440D+02,0.18627000D+01,0.00000000D+00
     ./)
      pars(  31301:  31400)=(/
     . 0.47895980D+03,0.121D+03,0.450D+02,0.18627000D+01,0.00000000D+00
     .,0.44308160D+03,0.121D+03,0.460D+02,0.18627000D+01,0.00000000D+00
     .,0.36949330D+03,0.121D+03,0.470D+02,0.18627000D+01,0.00000000D+00
     .,0.38918590D+03,0.121D+03,0.480D+02,0.18627000D+01,0.00000000D+00
     .,0.49219040D+03,0.121D+03,0.490D+02,0.18627000D+01,0.00000000D+00
     .,0.45113400D+03,0.121D+03,0.500D+02,0.18627000D+01,0.00000000D+00
     .,0.39831520D+03,0.121D+03,0.510D+02,0.18627000D+01,0.00000000D+00
     .,0.36739080D+03,0.121D+03,0.520D+02,0.18627000D+01,0.00000000D+00
     .,0.33004180D+03,0.121D+03,0.530D+02,0.18627000D+01,0.00000000D+00
     .,0.29487060D+03,0.121D+03,0.540D+02,0.18627000D+01,0.00000000D+00
     .,0.13445274D+04,0.121D+03,0.550D+02,0.18627000D+01,0.00000000D+00
     .,0.12227426D+04,0.121D+03,0.560D+02,0.18627000D+01,0.00000000D+00
     .,0.10658140D+04,0.121D+03,0.570D+02,0.18627000D+01,0.00000000D+00
     .,0.47118540D+03,0.121D+03,0.580D+02,0.18627000D+01,0.27991000D+01
     .,0.10799647D+04,0.121D+03,0.590D+02,0.18627000D+01,0.00000000D+00
     .,0.10362055D+04,0.121D+03,0.600D+02,0.18627000D+01,0.00000000D+00
     .,0.10098893D+04,0.121D+03,0.610D+02,0.18627000D+01,0.00000000D+00
     .,0.98572640D+03,0.121D+03,0.620D+02,0.18627000D+01,0.00000000D+00
     .,0.96428550D+03,0.121D+03,0.630D+02,0.18627000D+01,0.00000000D+00
     .,0.75078480D+03,0.121D+03,0.640D+02,0.18627000D+01,0.00000000D+00
     ./)
      pars(  31401:  31500)=(/
     . 0.85514710D+03,0.121D+03,0.650D+02,0.18627000D+01,0.00000000D+00
     .,0.82284200D+03,0.121D+03,0.660D+02,0.18627000D+01,0.00000000D+00
     .,0.86811040D+03,0.121D+03,0.670D+02,0.18627000D+01,0.00000000D+00
     .,0.84953940D+03,0.121D+03,0.680D+02,0.18627000D+01,0.00000000D+00
     .,0.83269610D+03,0.121D+03,0.690D+02,0.18627000D+01,0.00000000D+00
     .,0.82330890D+03,0.121D+03,0.700D+02,0.18627000D+01,0.00000000D+00
     .,0.68862420D+03,0.121D+03,0.710D+02,0.18627000D+01,0.00000000D+00
     .,0.67192580D+03,0.121D+03,0.720D+02,0.18627000D+01,0.00000000D+00
     .,0.60975150D+03,0.121D+03,0.730D+02,0.18627000D+01,0.00000000D+00
     .,0.51221050D+03,0.121D+03,0.740D+02,0.18627000D+01,0.00000000D+00
     .,0.52005780D+03,0.121D+03,0.750D+02,0.18627000D+01,0.00000000D+00
     .,0.46895650D+03,0.121D+03,0.760D+02,0.18627000D+01,0.00000000D+00
     .,0.42768450D+03,0.121D+03,0.770D+02,0.18627000D+01,0.00000000D+00
     .,0.35348450D+03,0.121D+03,0.780D+02,0.18627000D+01,0.00000000D+00
     .,0.32959850D+03,0.121D+03,0.790D+02,0.18627000D+01,0.00000000D+00
     .,0.33854780D+03,0.121D+03,0.800D+02,0.18627000D+01,0.00000000D+00
     .,0.50363930D+03,0.121D+03,0.810D+02,0.18627000D+01,0.00000000D+00
     .,0.48925410D+03,0.121D+03,0.820D+02,0.18627000D+01,0.00000000D+00
     .,0.44592310D+03,0.121D+03,0.830D+02,0.18627000D+01,0.00000000D+00
     .,0.42320630D+03,0.121D+03,0.840D+02,0.18627000D+01,0.00000000D+00
     ./)
      pars(  31501:  31600)=(/
     . 0.38821220D+03,0.121D+03,0.850D+02,0.18627000D+01,0.00000000D+00
     .,0.35380050D+03,0.121D+03,0.860D+02,0.18627000D+01,0.00000000D+00
     .,0.12630191D+04,0.121D+03,0.870D+02,0.18627000D+01,0.00000000D+00
     .,0.12039859D+04,0.121D+03,0.880D+02,0.18627000D+01,0.00000000D+00
     .,0.10563541D+04,0.121D+03,0.890D+02,0.18627000D+01,0.00000000D+00
     .,0.94062410D+03,0.121D+03,0.900D+02,0.18627000D+01,0.00000000D+00
     .,0.93765790D+03,0.121D+03,0.910D+02,0.18627000D+01,0.00000000D+00
     .,0.90769560D+03,0.121D+03,0.920D+02,0.18627000D+01,0.00000000D+00
     .,0.93986910D+03,0.121D+03,0.930D+02,0.18627000D+01,0.00000000D+00
     .,0.90928950D+03,0.121D+03,0.940D+02,0.18627000D+01,0.00000000D+00
     .,0.48975300D+02,0.121D+03,0.101D+03,0.18627000D+01,0.00000000D+00
     .,0.16364460D+03,0.121D+03,0.103D+03,0.18627000D+01,0.98650000D+00
     .,0.20781070D+03,0.121D+03,0.104D+03,0.18627000D+01,0.98080000D+00
     .,0.15582280D+03,0.121D+03,0.105D+03,0.18627000D+01,0.97060000D+00
     .,0.11597260D+03,0.121D+03,0.106D+03,0.18627000D+01,0.98680000D+00
     .,0.79538300D+02,0.121D+03,0.107D+03,0.18627000D+01,0.99440000D+00
     .,0.57261100D+02,0.121D+03,0.108D+03,0.18627000D+01,0.99250000D+00
     .,0.38809700D+02,0.121D+03,0.109D+03,0.18627000D+01,0.99820000D+00
     .,0.23997140D+03,0.121D+03,0.111D+03,0.18627000D+01,0.96840000D+00
     .,0.37177680D+03,0.121D+03,0.112D+03,0.18627000D+01,0.96280000D+00
     ./)
      pars(  31601:  31700)=(/
     . 0.37318450D+03,0.121D+03,0.113D+03,0.18627000D+01,0.96480000D+00
     .,0.29561010D+03,0.121D+03,0.114D+03,0.18627000D+01,0.95070000D+00
     .,0.23937010D+03,0.121D+03,0.115D+03,0.18627000D+01,0.99470000D+00
     .,0.20076220D+03,0.121D+03,0.116D+03,0.18627000D+01,0.99480000D+00
     .,0.16268060D+03,0.121D+03,0.117D+03,0.18627000D+01,0.99720000D+00
     .,0.32770920D+03,0.121D+03,0.119D+03,0.18627000D+01,0.97670000D+00
     .,0.64070760D+03,0.121D+03,0.120D+03,0.18627000D+01,0.98310000D+00
     .,0.32450680D+03,0.121D+03,0.121D+03,0.18627000D+01,0.18627000D+01
     .,0.29000900D+02,0.122D+03,0.100D+01,0.18299000D+01,0.91180000D+00
     .,0.18808400D+02,0.122D+03,0.200D+01,0.18299000D+01,0.00000000D+00
     .,0.48829310D+03,0.122D+03,0.300D+01,0.18299000D+01,0.00000000D+00
     .,0.27202010D+03,0.122D+03,0.400D+01,0.18299000D+01,0.00000000D+00
     .,0.17950770D+03,0.122D+03,0.500D+01,0.18299000D+01,0.00000000D+00
     .,0.11943760D+03,0.122D+03,0.600D+01,0.18299000D+01,0.00000000D+00
     .,0.82583400D+02,0.122D+03,0.700D+01,0.18299000D+01,0.00000000D+00
     .,0.62030900D+02,0.122D+03,0.800D+01,0.18299000D+01,0.00000000D+00
     .,0.46676500D+02,0.122D+03,0.900D+01,0.18299000D+01,0.00000000D+00
     .,0.35711700D+02,0.122D+03,0.100D+02,0.18299000D+01,0.00000000D+00
     .,0.58252700D+03,0.122D+03,0.110D+02,0.18299000D+01,0.00000000D+00
     .,0.43610910D+03,0.122D+03,0.120D+02,0.18299000D+01,0.00000000D+00
     ./)
      pars(  31701:  31800)=(/
     . 0.39786180D+03,0.122D+03,0.130D+02,0.18299000D+01,0.00000000D+00
     .,0.30910810D+03,0.122D+03,0.140D+02,0.18299000D+01,0.00000000D+00
     .,0.23816440D+03,0.122D+03,0.150D+02,0.18299000D+01,0.00000000D+00
     .,0.19611280D+03,0.122D+03,0.160D+02,0.18299000D+01,0.00000000D+00
     .,0.15898920D+03,0.122D+03,0.170D+02,0.18299000D+01,0.00000000D+00
     .,0.12921800D+03,0.122D+03,0.180D+02,0.18299000D+01,0.00000000D+00
     .,0.96033760D+03,0.122D+03,0.190D+02,0.18299000D+01,0.00000000D+00
     .,0.77468630D+03,0.122D+03,0.200D+02,0.18299000D+01,0.00000000D+00
     .,0.63687060D+03,0.122D+03,0.210D+02,0.18299000D+01,0.00000000D+00
     .,0.61222250D+03,0.122D+03,0.220D+02,0.18299000D+01,0.00000000D+00
     .,0.55911570D+03,0.122D+03,0.230D+02,0.18299000D+01,0.00000000D+00
     .,0.44007620D+03,0.122D+03,0.240D+02,0.18299000D+01,0.00000000D+00
     .,0.47949690D+03,0.122D+03,0.250D+02,0.18299000D+01,0.00000000D+00
     .,0.37582420D+03,0.122D+03,0.260D+02,0.18299000D+01,0.00000000D+00
     .,0.39627480D+03,0.122D+03,0.270D+02,0.18299000D+01,0.00000000D+00
     .,0.40935550D+03,0.122D+03,0.280D+02,0.18299000D+01,0.00000000D+00
     .,0.31360260D+03,0.122D+03,0.290D+02,0.18299000D+01,0.00000000D+00
     .,0.31938340D+03,0.122D+03,0.300D+02,0.18299000D+01,0.00000000D+00
     .,0.37941090D+03,0.122D+03,0.310D+02,0.18299000D+01,0.00000000D+00
     .,0.33097100D+03,0.122D+03,0.320D+02,0.18299000D+01,0.00000000D+00
     ./)
      pars(  31801:  31900)=(/
     . 0.27949900D+03,0.122D+03,0.330D+02,0.18299000D+01,0.00000000D+00
     .,0.24922580D+03,0.122D+03,0.340D+02,0.18299000D+01,0.00000000D+00
     .,0.21669210D+03,0.122D+03,0.350D+02,0.18299000D+01,0.00000000D+00
     .,0.18732310D+03,0.122D+03,0.360D+02,0.18299000D+01,0.00000000D+00
     .,0.10741973D+04,0.122D+03,0.370D+02,0.18299000D+01,0.00000000D+00
     .,0.92365570D+03,0.122D+03,0.380D+02,0.18299000D+01,0.00000000D+00
     .,0.80203240D+03,0.122D+03,0.390D+02,0.18299000D+01,0.00000000D+00
     .,0.71686090D+03,0.122D+03,0.400D+02,0.18299000D+01,0.00000000D+00
     .,0.65128470D+03,0.122D+03,0.410D+02,0.18299000D+01,0.00000000D+00
     .,0.49950110D+03,0.122D+03,0.420D+02,0.18299000D+01,0.00000000D+00
     .,0.55872110D+03,0.122D+03,0.430D+02,0.18299000D+01,0.00000000D+00
     .,0.42257080D+03,0.122D+03,0.440D+02,0.18299000D+01,0.00000000D+00
     .,0.46203140D+03,0.122D+03,0.450D+02,0.18299000D+01,0.00000000D+00
     .,0.42745770D+03,0.122D+03,0.460D+02,0.18299000D+01,0.00000000D+00
     .,0.35672450D+03,0.122D+03,0.470D+02,0.18299000D+01,0.00000000D+00
     .,0.37552610D+03,0.122D+03,0.480D+02,0.18299000D+01,0.00000000D+00
     .,0.47481120D+03,0.122D+03,0.490D+02,0.18299000D+01,0.00000000D+00
     .,0.43519480D+03,0.122D+03,0.500D+02,0.18299000D+01,0.00000000D+00
     .,0.38444870D+03,0.122D+03,0.510D+02,0.18299000D+01,0.00000000D+00
     .,0.35477930D+03,0.122D+03,0.520D+02,0.18299000D+01,0.00000000D+00
     ./)
      pars(  31901:  32000)=(/
     . 0.31891710D+03,0.122D+03,0.530D+02,0.18299000D+01,0.00000000D+00
     .,0.28513250D+03,0.122D+03,0.540D+02,0.18299000D+01,0.00000000D+00
     .,0.13086418D+04,0.122D+03,0.550D+02,0.18299000D+01,0.00000000D+00
     .,0.11809424D+04,0.122D+03,0.560D+02,0.18299000D+01,0.00000000D+00
     .,0.10284169D+04,0.122D+03,0.570D+02,0.18299000D+01,0.00000000D+00
     .,0.45482000D+03,0.122D+03,0.580D+02,0.18299000D+01,0.27991000D+01
     .,0.10433187D+04,0.122D+03,0.590D+02,0.18299000D+01,0.00000000D+00
     .,0.10002155D+04,0.122D+03,0.600D+02,0.18299000D+01,0.00000000D+00
     .,0.97472270D+03,0.122D+03,0.610D+02,0.18299000D+01,0.00000000D+00
     .,0.95132570D+03,0.122D+03,0.620D+02,0.18299000D+01,0.00000000D+00
     .,0.93056540D+03,0.122D+03,0.630D+02,0.18299000D+01,0.00000000D+00
     .,0.72458860D+03,0.122D+03,0.640D+02,0.18299000D+01,0.00000000D+00
     .,0.82779700D+03,0.122D+03,0.650D+02,0.18299000D+01,0.00000000D+00
     .,0.79723640D+03,0.122D+03,0.660D+02,0.18299000D+01,0.00000000D+00
     .,0.83746180D+03,0.122D+03,0.670D+02,0.18299000D+01,0.00000000D+00
     .,0.81949730D+03,0.122D+03,0.680D+02,0.18299000D+01,0.00000000D+00
     .,0.80320320D+03,0.122D+03,0.690D+02,0.18299000D+01,0.00000000D+00
     .,0.79414040D+03,0.122D+03,0.700D+02,0.18299000D+01,0.00000000D+00
     .,0.66477270D+03,0.122D+03,0.710D+02,0.18299000D+01,0.00000000D+00
     .,0.64787460D+03,0.122D+03,0.720D+02,0.18299000D+01,0.00000000D+00
     ./)
      pars(  32001:  32100)=(/
     . 0.58798130D+03,0.122D+03,0.730D+02,0.18299000D+01,0.00000000D+00
     .,0.49423570D+03,0.122D+03,0.740D+02,0.18299000D+01,0.00000000D+00
     .,0.50177330D+03,0.122D+03,0.750D+02,0.18299000D+01,0.00000000D+00
     .,0.45257260D+03,0.122D+03,0.760D+02,0.18299000D+01,0.00000000D+00
     .,0.41285580D+03,0.122D+03,0.770D+02,0.18299000D+01,0.00000000D+00
     .,0.34145400D+03,0.122D+03,0.780D+02,0.18299000D+01,0.00000000D+00
     .,0.31846190D+03,0.122D+03,0.790D+02,0.18299000D+01,0.00000000D+00
     .,0.32707700D+03,0.122D+03,0.800D+02,0.18299000D+01,0.00000000D+00
     .,0.48621160D+03,0.122D+03,0.810D+02,0.18299000D+01,0.00000000D+00
     .,0.47215520D+03,0.122D+03,0.820D+02,0.18299000D+01,0.00000000D+00
     .,0.43048050D+03,0.122D+03,0.830D+02,0.18299000D+01,0.00000000D+00
     .,0.40869510D+03,0.122D+03,0.840D+02,0.18299000D+01,0.00000000D+00
     .,0.37510290D+03,0.122D+03,0.850D+02,0.18299000D+01,0.00000000D+00
     .,0.34205750D+03,0.122D+03,0.860D+02,0.18299000D+01,0.00000000D+00
     .,0.12263970D+04,0.122D+03,0.870D+02,0.18299000D+01,0.00000000D+00
     .,0.11623660D+04,0.122D+03,0.880D+02,0.18299000D+01,0.00000000D+00
     .,0.10191422D+04,0.122D+03,0.890D+02,0.18299000D+01,0.00000000D+00
     .,0.90723670D+03,0.122D+03,0.900D+02,0.18299000D+01,0.00000000D+00
     .,0.90483310D+03,0.122D+03,0.910D+02,0.18299000D+01,0.00000000D+00
     .,0.87587480D+03,0.122D+03,0.920D+02,0.18299000D+01,0.00000000D+00
     ./)
      pars(  32101:  32200)=(/
     . 0.90696160D+03,0.122D+03,0.930D+02,0.18299000D+01,0.00000000D+00
     .,0.87736450D+03,0.122D+03,0.940D+02,0.18299000D+01,0.00000000D+00
     .,0.47275500D+02,0.122D+03,0.101D+03,0.18299000D+01,0.00000000D+00
     .,0.15766670D+03,0.122D+03,0.103D+03,0.18299000D+01,0.98650000D+00
     .,0.20036690D+03,0.122D+03,0.104D+03,0.18299000D+01,0.98080000D+00
     .,0.15035200D+03,0.122D+03,0.105D+03,0.18299000D+01,0.97060000D+00
     .,0.11205230D+03,0.122D+03,0.106D+03,0.18299000D+01,0.98680000D+00
     .,0.76983400D+02,0.122D+03,0.107D+03,0.18299000D+01,0.99440000D+00
     .,0.55516000D+02,0.122D+03,0.108D+03,0.18299000D+01,0.99250000D+00
     .,0.37722200D+02,0.122D+03,0.109D+03,0.18299000D+01,0.99820000D+00
     .,0.23126120D+03,0.122D+03,0.111D+03,0.18299000D+01,0.96840000D+00
     .,0.35836020D+03,0.122D+03,0.112D+03,0.18299000D+01,0.96280000D+00
     .,0.35964970D+03,0.122D+03,0.113D+03,0.18299000D+01,0.96480000D+00
     .,0.28509570D+03,0.122D+03,0.114D+03,0.18299000D+01,0.95070000D+00
     .,0.23107730D+03,0.122D+03,0.115D+03,0.18299000D+01,0.99470000D+00
     .,0.19398450D+03,0.122D+03,0.116D+03,0.18299000D+01,0.99480000D+00
     .,0.15736040D+03,0.122D+03,0.117D+03,0.18299000D+01,0.99720000D+00
     .,0.31625180D+03,0.122D+03,0.119D+03,0.18299000D+01,0.97670000D+00
     .,0.61943010D+03,0.122D+03,0.120D+03,0.18299000D+01,0.98310000D+00
     .,0.31292020D+03,0.122D+03,0.121D+03,0.18299000D+01,0.18627000D+01
     ./)
      pars(  32201:  32300)=(/
     . 0.30199880D+03,0.122D+03,0.122D+03,0.18299000D+01,0.18299000D+01
     .,0.28406700D+02,0.123D+03,0.100D+01,0.19138000D+01,0.91180000D+00
     .,0.18435900D+02,0.123D+03,0.200D+01,0.19138000D+01,0.00000000D+00
     .,0.47914020D+03,0.123D+03,0.300D+01,0.19138000D+01,0.00000000D+00
     .,0.26670500D+03,0.123D+03,0.400D+01,0.19138000D+01,0.00000000D+00
     .,0.17589090D+03,0.123D+03,0.500D+01,0.19138000D+01,0.00000000D+00
     .,0.11700980D+03,0.123D+03,0.600D+01,0.19138000D+01,0.00000000D+00
     .,0.80914400D+02,0.123D+03,0.700D+01,0.19138000D+01,0.00000000D+00
     .,0.60793300D+02,0.123D+03,0.800D+01,0.19138000D+01,0.00000000D+00
     .,0.45763000D+02,0.123D+03,0.900D+01,0.19138000D+01,0.00000000D+00
     .,0.35029000D+02,0.123D+03,0.100D+02,0.19138000D+01,0.00000000D+00
     .,0.57164550D+03,0.123D+03,0.110D+02,0.19138000D+01,0.00000000D+00
     .,0.42771560D+03,0.123D+03,0.120D+02,0.19138000D+01,0.00000000D+00
     .,0.39004320D+03,0.123D+03,0.130D+02,0.19138000D+01,0.00000000D+00
     .,0.30289850D+03,0.123D+03,0.140D+02,0.19138000D+01,0.00000000D+00
     .,0.23331760D+03,0.123D+03,0.150D+02,0.19138000D+01,0.00000000D+00
     .,0.19211130D+03,0.123D+03,0.160D+02,0.19138000D+01,0.00000000D+00
     .,0.15575070D+03,0.123D+03,0.170D+02,0.19138000D+01,0.00000000D+00
     .,0.12660270D+03,0.123D+03,0.180D+02,0.19138000D+01,0.00000000D+00
     .,0.94198370D+03,0.123D+03,0.190D+02,0.19138000D+01,0.00000000D+00
     ./)
      pars(  32301:  32400)=(/
     . 0.76007180D+03,0.123D+03,0.200D+02,0.19138000D+01,0.00000000D+00
     .,0.62478010D+03,0.123D+03,0.210D+02,0.19138000D+01,0.00000000D+00
     .,0.60052210D+03,0.123D+03,0.220D+02,0.19138000D+01,0.00000000D+00
     .,0.54839080D+03,0.123D+03,0.230D+02,0.19138000D+01,0.00000000D+00
     .,0.43166830D+03,0.123D+03,0.240D+02,0.19138000D+01,0.00000000D+00
     .,0.47025290D+03,0.123D+03,0.250D+02,0.19138000D+01,0.00000000D+00
     .,0.36861360D+03,0.123D+03,0.260D+02,0.19138000D+01,0.00000000D+00
     .,0.38857290D+03,0.123D+03,0.270D+02,0.19138000D+01,0.00000000D+00
     .,0.40143290D+03,0.123D+03,0.280D+02,0.19138000D+01,0.00000000D+00
     .,0.30757630D+03,0.123D+03,0.290D+02,0.19138000D+01,0.00000000D+00
     .,0.31312860D+03,0.123D+03,0.300D+02,0.19138000D+01,0.00000000D+00
     .,0.37194270D+03,0.123D+03,0.310D+02,0.19138000D+01,0.00000000D+00
     .,0.32434350D+03,0.123D+03,0.320D+02,0.19138000D+01,0.00000000D+00
     .,0.27383520D+03,0.123D+03,0.330D+02,0.19138000D+01,0.00000000D+00
     .,0.24415540D+03,0.123D+03,0.340D+02,0.19138000D+01,0.00000000D+00
     .,0.21227920D+03,0.123D+03,0.350D+02,0.19138000D+01,0.00000000D+00
     .,0.18351710D+03,0.123D+03,0.360D+02,0.19138000D+01,0.00000000D+00
     .,0.10535834D+04,0.123D+03,0.370D+02,0.19138000D+01,0.00000000D+00
     .,0.90623640D+03,0.123D+03,0.380D+02,0.19138000D+01,0.00000000D+00
     .,0.78672550D+03,0.123D+03,0.390D+02,0.19138000D+01,0.00000000D+00
     ./)
      pars(  32401:  32500)=(/
     . 0.70307840D+03,0.123D+03,0.400D+02,0.19138000D+01,0.00000000D+00
     .,0.63870140D+03,0.123D+03,0.410D+02,0.19138000D+01,0.00000000D+00
     .,0.48981600D+03,0.123D+03,0.420D+02,0.19138000D+01,0.00000000D+00
     .,0.54789060D+03,0.123D+03,0.430D+02,0.19138000D+01,0.00000000D+00
     .,0.41435750D+03,0.123D+03,0.440D+02,0.19138000D+01,0.00000000D+00
     .,0.45302620D+03,0.123D+03,0.450D+02,0.19138000D+01,0.00000000D+00
     .,0.41911340D+03,0.123D+03,0.460D+02,0.19138000D+01,0.00000000D+00
     .,0.34982880D+03,0.123D+03,0.470D+02,0.19138000D+01,0.00000000D+00
     .,0.36818710D+03,0.123D+03,0.480D+02,0.19138000D+01,0.00000000D+00
     .,0.46556470D+03,0.123D+03,0.490D+02,0.19138000D+01,0.00000000D+00
     .,0.42659330D+03,0.123D+03,0.500D+02,0.19138000D+01,0.00000000D+00
     .,0.37675440D+03,0.123D+03,0.510D+02,0.19138000D+01,0.00000000D+00
     .,0.34763740D+03,0.123D+03,0.520D+02,0.19138000D+01,0.00000000D+00
     .,0.31247140D+03,0.123D+03,0.530D+02,0.19138000D+01,0.00000000D+00
     .,0.27936120D+03,0.123D+03,0.540D+02,0.19138000D+01,0.00000000D+00
     .,0.12831193D+04,0.123D+03,0.550D+02,0.19138000D+01,0.00000000D+00
     .,0.11586662D+04,0.123D+03,0.560D+02,0.19138000D+01,0.00000000D+00
     .,0.10088336D+04,0.123D+03,0.570D+02,0.19138000D+01,0.00000000D+00
     .,0.44575700D+03,0.123D+03,0.580D+02,0.19138000D+01,0.27991000D+01
     .,0.10235395D+04,0.123D+03,0.590D+02,0.19138000D+01,0.00000000D+00
     ./)
      pars(  32501:  32600)=(/
     . 0.98134570D+03,0.123D+03,0.600D+02,0.19138000D+01,0.00000000D+00
     .,0.95633210D+03,0.123D+03,0.610D+02,0.19138000D+01,0.00000000D+00
     .,0.93337380D+03,0.123D+03,0.620D+02,0.19138000D+01,0.00000000D+00
     .,0.91300140D+03,0.123D+03,0.630D+02,0.19138000D+01,0.00000000D+00
     .,0.71072370D+03,0.123D+03,0.640D+02,0.19138000D+01,0.00000000D+00
     .,0.81199050D+03,0.123D+03,0.650D+02,0.19138000D+01,0.00000000D+00
     .,0.78182720D+03,0.123D+03,0.660D+02,0.19138000D+01,0.00000000D+00
     .,0.82162740D+03,0.123D+03,0.670D+02,0.19138000D+01,0.00000000D+00
     .,0.80399890D+03,0.123D+03,0.680D+02,0.19138000D+01,0.00000000D+00
     .,0.78800640D+03,0.123D+03,0.690D+02,0.19138000D+01,0.00000000D+00
     .,0.77912190D+03,0.123D+03,0.700D+02,0.19138000D+01,0.00000000D+00
     .,0.65197940D+03,0.123D+03,0.710D+02,0.19138000D+01,0.00000000D+00
     .,0.63529960D+03,0.123D+03,0.720D+02,0.19138000D+01,0.00000000D+00
     .,0.57649460D+03,0.123D+03,0.730D+02,0.19138000D+01,0.00000000D+00
     .,0.48455830D+03,0.123D+03,0.740D+02,0.19138000D+01,0.00000000D+00
     .,0.49191200D+03,0.123D+03,0.750D+02,0.19138000D+01,0.00000000D+00
     .,0.44364130D+03,0.123D+03,0.760D+02,0.19138000D+01,0.00000000D+00
     .,0.40468640D+03,0.123D+03,0.770D+02,0.19138000D+01,0.00000000D+00
     .,0.33471530D+03,0.123D+03,0.780D+02,0.19138000D+01,0.00000000D+00
     .,0.31218730D+03,0.123D+03,0.790D+02,0.19138000D+01,0.00000000D+00
     ./)
      pars(  32601:  32700)=(/
     . 0.32060120D+03,0.123D+03,0.800D+02,0.19138000D+01,0.00000000D+00
     .,0.47675960D+03,0.123D+03,0.810D+02,0.19138000D+01,0.00000000D+00
     .,0.46287410D+03,0.123D+03,0.820D+02,0.19138000D+01,0.00000000D+00
     .,0.42191810D+03,0.123D+03,0.830D+02,0.19138000D+01,0.00000000D+00
     .,0.40051720D+03,0.123D+03,0.840D+02,0.19138000D+01,0.00000000D+00
     .,0.36755880D+03,0.123D+03,0.850D+02,0.19138000D+01,0.00000000D+00
     .,0.33516100D+03,0.123D+03,0.860D+02,0.19138000D+01,0.00000000D+00
     .,0.12027024D+04,0.123D+03,0.870D+02,0.19138000D+01,0.00000000D+00
     .,0.11403430D+04,0.123D+03,0.880D+02,0.19138000D+01,0.00000000D+00
     .,0.99965520D+03,0.123D+03,0.890D+02,0.19138000D+01,0.00000000D+00
     .,0.88972590D+03,0.123D+03,0.900D+02,0.19138000D+01,0.00000000D+00
     .,0.88748800D+03,0.123D+03,0.910D+02,0.19138000D+01,0.00000000D+00
     .,0.85909970D+03,0.123D+03,0.920D+02,0.19138000D+01,0.00000000D+00
     .,0.88977190D+03,0.123D+03,0.930D+02,0.19138000D+01,0.00000000D+00
     .,0.86071780D+03,0.123D+03,0.940D+02,0.19138000D+01,0.00000000D+00
     .,0.46308900D+02,0.123D+03,0.101D+03,0.19138000D+01,0.00000000D+00
     .,0.15457620D+03,0.123D+03,0.103D+03,0.19138000D+01,0.98650000D+00
     .,0.19640270D+03,0.123D+03,0.104D+03,0.19138000D+01,0.98080000D+00
     .,0.14731530D+03,0.123D+03,0.105D+03,0.19138000D+01,0.97060000D+00
     .,0.10977970D+03,0.123D+03,0.106D+03,0.19138000D+01,0.98680000D+00
     ./)
      pars(  32701:  32800)=(/
     . 0.75430300D+02,0.123D+03,0.107D+03,0.19138000D+01,0.99440000D+00
     .,0.54410900D+02,0.123D+03,0.108D+03,0.19138000D+01,0.99250000D+00
     .,0.36996200D+02,0.123D+03,0.109D+03,0.19138000D+01,0.99820000D+00
     .,0.22678860D+03,0.123D+03,0.111D+03,0.19138000D+01,0.96840000D+00
     .,0.35139280D+03,0.123D+03,0.112D+03,0.19138000D+01,0.96280000D+00
     .,0.35255750D+03,0.123D+03,0.113D+03,0.19138000D+01,0.96480000D+00
     .,0.27935630D+03,0.123D+03,0.114D+03,0.19138000D+01,0.95070000D+00
     .,0.22637670D+03,0.123D+03,0.115D+03,0.19138000D+01,0.99470000D+00
     .,0.19002990D+03,0.123D+03,0.116D+03,0.19138000D+01,0.99480000D+00
     .,0.15415760D+03,0.123D+03,0.117D+03,0.19138000D+01,0.99720000D+00
     .,0.31011080D+03,0.123D+03,0.119D+03,0.19138000D+01,0.97670000D+00
     .,0.60755710D+03,0.123D+03,0.120D+03,0.19138000D+01,0.98310000D+00
     .,0.30673890D+03,0.123D+03,0.121D+03,0.19138000D+01,0.18627000D+01
     .,0.29600290D+03,0.123D+03,0.122D+03,0.19138000D+01,0.18299000D+01
     .,0.29014360D+03,0.123D+03,0.123D+03,0.19138000D+01,0.19138000D+01
     .,0.28056200D+02,0.124D+03,0.100D+01,0.18269000D+01,0.91180000D+00
     .,0.18175400D+02,0.124D+03,0.200D+01,0.18269000D+01,0.00000000D+00
     .,0.48059440D+03,0.124D+03,0.300D+01,0.18269000D+01,0.00000000D+00
     .,0.26540880D+03,0.124D+03,0.400D+01,0.18269000D+01,0.00000000D+00
     .,0.17437190D+03,0.124D+03,0.500D+01,0.18269000D+01,0.00000000D+00
     ./)
      pars(  32801:  32900)=(/
     . 0.11571820D+03,0.124D+03,0.600D+01,0.18269000D+01,0.00000000D+00
     .,0.79902400D+02,0.124D+03,0.700D+01,0.18269000D+01,0.00000000D+00
     .,0.59982900D+02,0.124D+03,0.800D+01,0.18269000D+01,0.00000000D+00
     .,0.45129000D+02,0.124D+03,0.900D+01,0.18269000D+01,0.00000000D+00
     .,0.34535000D+02,0.124D+03,0.100D+02,0.18269000D+01,0.00000000D+00
     .,0.57314360D+03,0.124D+03,0.110D+02,0.18269000D+01,0.00000000D+00
     .,0.42622760D+03,0.124D+03,0.120D+02,0.18269000D+01,0.00000000D+00
     .,0.38786950D+03,0.124D+03,0.130D+02,0.18269000D+01,0.00000000D+00
     .,0.30039280D+03,0.124D+03,0.140D+02,0.18269000D+01,0.00000000D+00
     .,0.23090900D+03,0.124D+03,0.150D+02,0.18269000D+01,0.00000000D+00
     .,0.18990520D+03,0.124D+03,0.160D+02,0.18269000D+01,0.00000000D+00
     .,0.15380570D+03,0.124D+03,0.170D+02,0.18269000D+01,0.00000000D+00
     .,0.12492650D+03,0.124D+03,0.180D+02,0.18269000D+01,0.00000000D+00
     .,0.94587000D+03,0.124D+03,0.190D+02,0.18269000D+01,0.00000000D+00
     .,0.75936840D+03,0.124D+03,0.200D+02,0.18269000D+01,0.00000000D+00
     .,0.62353320D+03,0.124D+03,0.210D+02,0.18269000D+01,0.00000000D+00
     .,0.59879810D+03,0.124D+03,0.220D+02,0.18269000D+01,0.00000000D+00
     .,0.54652440D+03,0.124D+03,0.230D+02,0.18269000D+01,0.00000000D+00
     .,0.43023530D+03,0.124D+03,0.240D+02,0.18269000D+01,0.00000000D+00
     .,0.46829610D+03,0.124D+03,0.250D+02,0.18269000D+01,0.00000000D+00
     ./)
      pars(  32901:  33000)=(/
     . 0.36707850D+03,0.124D+03,0.260D+02,0.18269000D+01,0.00000000D+00
     .,0.38644860D+03,0.124D+03,0.270D+02,0.18269000D+01,0.00000000D+00
     .,0.39945300D+03,0.124D+03,0.280D+02,0.18269000D+01,0.00000000D+00
     .,0.30611810D+03,0.124D+03,0.290D+02,0.18269000D+01,0.00000000D+00
     .,0.31106030D+03,0.124D+03,0.300D+02,0.18269000D+01,0.00000000D+00
     .,0.36963020D+03,0.124D+03,0.310D+02,0.18269000D+01,0.00000000D+00
     .,0.32164040D+03,0.124D+03,0.320D+02,0.18269000D+01,0.00000000D+00
     .,0.27107700D+03,0.124D+03,0.330D+02,0.18269000D+01,0.00000000D+00
     .,0.24145850D+03,0.124D+03,0.340D+02,0.18269000D+01,0.00000000D+00
     .,0.20973930D+03,0.124D+03,0.350D+02,0.18269000D+01,0.00000000D+00
     .,0.18118190D+03,0.124D+03,0.360D+02,0.18269000D+01,0.00000000D+00
     .,0.10575718D+04,0.124D+03,0.370D+02,0.18269000D+01,0.00000000D+00
     .,0.90561870D+03,0.124D+03,0.380D+02,0.18269000D+01,0.00000000D+00
     .,0.78474890D+03,0.124D+03,0.390D+02,0.18269000D+01,0.00000000D+00
     .,0.70054320D+03,0.124D+03,0.400D+02,0.18269000D+01,0.00000000D+00
     .,0.63596390D+03,0.124D+03,0.410D+02,0.18269000D+01,0.00000000D+00
     .,0.48719110D+03,0.124D+03,0.420D+02,0.18269000D+01,0.00000000D+00
     .,0.54517240D+03,0.124D+03,0.430D+02,0.18269000D+01,0.00000000D+00
     .,0.41181870D+03,0.124D+03,0.440D+02,0.18269000D+01,0.00000000D+00
     .,0.45019430D+03,0.124D+03,0.450D+02,0.18269000D+01,0.00000000D+00
     ./)
      pars(  33001:  33100)=(/
     . 0.41632300D+03,0.124D+03,0.460D+02,0.18269000D+01,0.00000000D+00
     .,0.34767860D+03,0.124D+03,0.470D+02,0.18269000D+01,0.00000000D+00
     .,0.36555310D+03,0.124D+03,0.480D+02,0.18269000D+01,0.00000000D+00
     .,0.46283040D+03,0.124D+03,0.490D+02,0.18269000D+01,0.00000000D+00
     .,0.42332080D+03,0.124D+03,0.500D+02,0.18269000D+01,0.00000000D+00
     .,0.37325910D+03,0.124D+03,0.510D+02,0.18269000D+01,0.00000000D+00
     .,0.34409820D+03,0.124D+03,0.520D+02,0.18269000D+01,0.00000000D+00
     .,0.30900630D+03,0.124D+03,0.530D+02,0.18269000D+01,0.00000000D+00
     .,0.27604060D+03,0.124D+03,0.540D+02,0.18269000D+01,0.00000000D+00
     .,0.12879616D+04,0.124D+03,0.550D+02,0.18269000D+01,0.00000000D+00
     .,0.11586672D+04,0.124D+03,0.560D+02,0.18269000D+01,0.00000000D+00
     .,0.10069636D+04,0.124D+03,0.570D+02,0.18269000D+01,0.00000000D+00
     .,0.44178090D+03,0.124D+03,0.580D+02,0.18269000D+01,0.27991000D+01
     .,0.10229773D+04,0.124D+03,0.590D+02,0.18269000D+01,0.00000000D+00
     .,0.98048290D+03,0.124D+03,0.600D+02,0.18269000D+01,0.00000000D+00
     .,0.95540420D+03,0.124D+03,0.610D+02,0.18269000D+01,0.00000000D+00
     .,0.93239460D+03,0.124D+03,0.620D+02,0.18269000D+01,0.00000000D+00
     .,0.91197350D+03,0.124D+03,0.630D+02,0.18269000D+01,0.00000000D+00
     .,0.70857690D+03,0.124D+03,0.640D+02,0.18269000D+01,0.00000000D+00
     .,0.81204950D+03,0.124D+03,0.650D+02,0.18269000D+01,0.00000000D+00
     ./)
      pars(  33101:  33200)=(/
     . 0.78162600D+03,0.124D+03,0.660D+02,0.18269000D+01,0.00000000D+00
     .,0.82030710D+03,0.124D+03,0.670D+02,0.18269000D+01,0.00000000D+00
     .,0.80266090D+03,0.124D+03,0.680D+02,0.18269000D+01,0.00000000D+00
     .,0.78663360D+03,0.124D+03,0.690D+02,0.18269000D+01,0.00000000D+00
     .,0.77782320D+03,0.124D+03,0.700D+02,0.18269000D+01,0.00000000D+00
     .,0.65005720D+03,0.124D+03,0.710D+02,0.18269000D+01,0.00000000D+00
     .,0.63220990D+03,0.124D+03,0.720D+02,0.18269000D+01,0.00000000D+00
     .,0.57309450D+03,0.124D+03,0.730D+02,0.18269000D+01,0.00000000D+00
     .,0.48138140D+03,0.124D+03,0.740D+02,0.18269000D+01,0.00000000D+00
     .,0.48847190D+03,0.124D+03,0.750D+02,0.18269000D+01,0.00000000D+00
     .,0.44016550D+03,0.124D+03,0.760D+02,0.18269000D+01,0.00000000D+00
     .,0.40125220D+03,0.124D+03,0.770D+02,0.18269000D+01,0.00000000D+00
     .,0.33169650D+03,0.124D+03,0.780D+02,0.18269000D+01,0.00000000D+00
     .,0.30930750D+03,0.124D+03,0.790D+02,0.18269000D+01,0.00000000D+00
     .,0.31751490D+03,0.124D+03,0.800D+02,0.18269000D+01,0.00000000D+00
     .,0.47384960D+03,0.124D+03,0.810D+02,0.18269000D+01,0.00000000D+00
     .,0.45937430D+03,0.124D+03,0.820D+02,0.18269000D+01,0.00000000D+00
     .,0.41810440D+03,0.124D+03,0.830D+02,0.18269000D+01,0.00000000D+00
     .,0.39657360D+03,0.124D+03,0.840D+02,0.18269000D+01,0.00000000D+00
     .,0.36360680D+03,0.124D+03,0.850D+02,0.18269000D+01,0.00000000D+00
     ./)
      pars(  33201:  33300)=(/
     . 0.33130140D+03,0.124D+03,0.860D+02,0.18269000D+01,0.00000000D+00
     .,0.12054436D+04,0.124D+03,0.870D+02,0.18269000D+01,0.00000000D+00
     .,0.11392761D+04,0.124D+03,0.880D+02,0.18269000D+01,0.00000000D+00
     .,0.99702490D+03,0.124D+03,0.890D+02,0.18269000D+01,0.00000000D+00
     .,0.88576610D+03,0.124D+03,0.900D+02,0.18269000D+01,0.00000000D+00
     .,0.88442800D+03,0.124D+03,0.910D+02,0.18269000D+01,0.00000000D+00
     .,0.85610390D+03,0.124D+03,0.920D+02,0.18269000D+01,0.00000000D+00
     .,0.88770660D+03,0.124D+03,0.930D+02,0.18269000D+01,0.00000000D+00
     .,0.85853020D+03,0.124D+03,0.940D+02,0.18269000D+01,0.00000000D+00
     .,0.45817300D+02,0.124D+03,0.101D+03,0.18269000D+01,0.00000000D+00
     .,0.15375390D+03,0.124D+03,0.103D+03,0.18269000D+01,0.98650000D+00
     .,0.19523170D+03,0.124D+03,0.104D+03,0.18269000D+01,0.98080000D+00
     .,0.14594770D+03,0.124D+03,0.105D+03,0.18269000D+01,0.97060000D+00
     .,0.10858200D+03,0.124D+03,0.106D+03,0.18269000D+01,0.98680000D+00
     .,0.74489000D+02,0.124D+03,0.107D+03,0.18269000D+01,0.99440000D+00
     .,0.53673300D+02,0.124D+03,0.108D+03,0.18269000D+01,0.99250000D+00
     .,0.36459400D+02,0.124D+03,0.109D+03,0.18269000D+01,0.99820000D+00
     .,0.22578370D+03,0.124D+03,0.111D+03,0.18269000D+01,0.96840000D+00
     .,0.34993280D+03,0.124D+03,0.112D+03,0.18269000D+01,0.96280000D+00
     .,0.35042960D+03,0.124D+03,0.113D+03,0.18269000D+01,0.96480000D+00
     ./)
      pars(  33301:  33400)=(/
     . 0.27695120D+03,0.124D+03,0.114D+03,0.18269000D+01,0.95070000D+00
     .,0.22403030D+03,0.124D+03,0.115D+03,0.18269000D+01,0.99470000D+00
     .,0.18785700D+03,0.124D+03,0.116D+03,0.18269000D+01,0.99480000D+00
     .,0.15223830D+03,0.124D+03,0.117D+03,0.18269000D+01,0.99720000D+00
     .,0.30839010D+03,0.124D+03,0.119D+03,0.18269000D+01,0.97670000D+00
     .,0.60707000D+03,0.124D+03,0.120D+03,0.18269000D+01,0.98310000D+00
     .,0.30430940D+03,0.124D+03,0.121D+03,0.18269000D+01,0.18627000D+01
     .,0.29365980D+03,0.124D+03,0.122D+03,0.18269000D+01,0.18299000D+01
     .,0.28786570D+03,0.124D+03,0.123D+03,0.18269000D+01,0.19138000D+01
     .,0.28569050D+03,0.124D+03,0.124D+03,0.18269000D+01,0.18269000D+01
     .,0.26181000D+02,0.125D+03,0.100D+01,0.16406000D+01,0.91180000D+00
     .,0.17152600D+02,0.125D+03,0.200D+01,0.16406000D+01,0.00000000D+00
     .,0.41608360D+03,0.125D+03,0.300D+01,0.16406000D+01,0.00000000D+00
     .,0.23896060D+03,0.125D+03,0.400D+01,0.16406000D+01,0.00000000D+00
     .,0.15984920D+03,0.125D+03,0.500D+01,0.16406000D+01,0.00000000D+00
     .,0.10733820D+03,0.125D+03,0.600D+01,0.16406000D+01,0.00000000D+00
     .,0.74699800D+02,0.125D+03,0.700D+01,0.16406000D+01,0.00000000D+00
     .,0.56362200D+02,0.125D+03,0.800D+01,0.16406000D+01,0.00000000D+00
     .,0.42576400D+02,0.125D+03,0.900D+01,0.16406000D+01,0.00000000D+00
     .,0.32680600D+02,0.125D+03,0.100D+02,0.16406000D+01,0.00000000D+00
     ./)
      pars(  33401:  33500)=(/
     . 0.49735820D+03,0.125D+03,0.110D+02,0.16406000D+01,0.00000000D+00
     .,0.38129680D+03,0.125D+03,0.120D+02,0.16406000D+01,0.00000000D+00
     .,0.35045220D+03,0.125D+03,0.130D+02,0.16406000D+01,0.00000000D+00
     .,0.27492570D+03,0.125D+03,0.140D+02,0.16406000D+01,0.00000000D+00
     .,0.21341010D+03,0.125D+03,0.150D+02,0.16406000D+01,0.00000000D+00
     .,0.17651650D+03,0.125D+03,0.160D+02,0.16406000D+01,0.00000000D+00
     .,0.14370010D+03,0.125D+03,0.170D+02,0.16406000D+01,0.00000000D+00
     .,0.11720570D+03,0.125D+03,0.180D+02,0.16406000D+01,0.00000000D+00
     .,0.81424680D+03,0.125D+03,0.190D+02,0.16406000D+01,0.00000000D+00
     .,0.67083370D+03,0.125D+03,0.200D+02,0.16406000D+01,0.00000000D+00
     .,0.55379310D+03,0.125D+03,0.210D+02,0.16406000D+01,0.00000000D+00
     .,0.53412660D+03,0.125D+03,0.220D+02,0.16406000D+01,0.00000000D+00
     .,0.48879470D+03,0.125D+03,0.230D+02,0.16406000D+01,0.00000000D+00
     .,0.38469310D+03,0.125D+03,0.240D+02,0.16406000D+01,0.00000000D+00
     .,0.42042070D+03,0.125D+03,0.250D+02,0.16406000D+01,0.00000000D+00
     .,0.32962730D+03,0.125D+03,0.260D+02,0.16406000D+01,0.00000000D+00
     .,0.34919410D+03,0.125D+03,0.270D+02,0.16406000D+01,0.00000000D+00
     .,0.36001720D+03,0.125D+03,0.280D+02,0.16406000D+01,0.00000000D+00
     .,0.27572060D+03,0.125D+03,0.290D+02,0.16406000D+01,0.00000000D+00
     .,0.28266670D+03,0.125D+03,0.300D+02,0.16406000D+01,0.00000000D+00
     ./)
      pars(  33501:  33600)=(/
     . 0.33506150D+03,0.125D+03,0.310D+02,0.16406000D+01,0.00000000D+00
     .,0.29451840D+03,0.125D+03,0.320D+02,0.16406000D+01,0.00000000D+00
     .,0.25028310D+03,0.125D+03,0.330D+02,0.16406000D+01,0.00000000D+00
     .,0.22398570D+03,0.125D+03,0.340D+02,0.16406000D+01,0.00000000D+00
     .,0.19545360D+03,0.125D+03,0.350D+02,0.16406000D+01,0.00000000D+00
     .,0.16950970D+03,0.125D+03,0.360D+02,0.16406000D+01,0.00000000D+00
     .,0.91194330D+03,0.125D+03,0.370D+02,0.16406000D+01,0.00000000D+00
     .,0.79901250D+03,0.125D+03,0.380D+02,0.16406000D+01,0.00000000D+00
     .,0.69867600D+03,0.125D+03,0.390D+02,0.16406000D+01,0.00000000D+00
     .,0.62707900D+03,0.125D+03,0.400D+02,0.16406000D+01,0.00000000D+00
     .,0.57118620D+03,0.125D+03,0.410D+02,0.16406000D+01,0.00000000D+00
     .,0.43993260D+03,0.125D+03,0.420D+02,0.16406000D+01,0.00000000D+00
     .,0.49132100D+03,0.125D+03,0.430D+02,0.16406000D+01,0.00000000D+00
     .,0.37334420D+03,0.125D+03,0.440D+02,0.16406000D+01,0.00000000D+00
     .,0.40836770D+03,0.125D+03,0.450D+02,0.16406000D+01,0.00000000D+00
     .,0.37842950D+03,0.125D+03,0.460D+02,0.16406000D+01,0.00000000D+00
     .,0.31534820D+03,0.125D+03,0.470D+02,0.16406000D+01,0.00000000D+00
     .,0.33315180D+03,0.125D+03,0.480D+02,0.16406000D+01,0.00000000D+00
     .,0.41907070D+03,0.125D+03,0.490D+02,0.16406000D+01,0.00000000D+00
     .,0.38661840D+03,0.125D+03,0.500D+02,0.16406000D+01,0.00000000D+00
     ./)
      pars(  33601:  33700)=(/
     . 0.34352050D+03,0.125D+03,0.510D+02,0.16406000D+01,0.00000000D+00
     .,0.31804850D+03,0.125D+03,0.520D+02,0.16406000D+01,0.00000000D+00
     .,0.28687200D+03,0.125D+03,0.530D+02,0.16406000D+01,0.00000000D+00
     .,0.25728030D+03,0.125D+03,0.540D+02,0.16406000D+01,0.00000000D+00
     .,0.11103588D+04,0.125D+03,0.550D+02,0.16406000D+01,0.00000000D+00
     .,0.10186688D+04,0.125D+03,0.560D+02,0.16406000D+01,0.00000000D+00
     .,0.89350560D+03,0.125D+03,0.570D+02,0.16406000D+01,0.00000000D+00
     .,0.40587370D+03,0.125D+03,0.580D+02,0.16406000D+01,0.27991000D+01
     .,0.90204820D+03,0.125D+03,0.590D+02,0.16406000D+01,0.00000000D+00
     .,0.86604420D+03,0.125D+03,0.600D+02,0.16406000D+01,0.00000000D+00
     .,0.84428560D+03,0.125D+03,0.610D+02,0.16406000D+01,0.00000000D+00
     .,0.82428300D+03,0.125D+03,0.620D+02,0.16406000D+01,0.00000000D+00
     .,0.80654320D+03,0.125D+03,0.630D+02,0.16406000D+01,0.00000000D+00
     .,0.63257430D+03,0.125D+03,0.640D+02,0.16406000D+01,0.00000000D+00
     .,0.71375520D+03,0.125D+03,0.650D+02,0.16406000D+01,0.00000000D+00
     .,0.68811250D+03,0.125D+03,0.660D+02,0.16406000D+01,0.00000000D+00
     .,0.72723780D+03,0.125D+03,0.670D+02,0.16406000D+01,0.00000000D+00
     .,0.71180010D+03,0.125D+03,0.680D+02,0.16406000D+01,0.00000000D+00
     .,0.69785650D+03,0.125D+03,0.690D+02,0.16406000D+01,0.00000000D+00
     .,0.68978470D+03,0.125D+03,0.700D+02,0.16406000D+01,0.00000000D+00
     ./)
      pars(  33701:  33800)=(/
     . 0.58012870D+03,0.125D+03,0.710D+02,0.16406000D+01,0.00000000D+00
     .,0.56940560D+03,0.125D+03,0.720D+02,0.16406000D+01,0.00000000D+00
     .,0.51879050D+03,0.125D+03,0.730D+02,0.16406000D+01,0.00000000D+00
     .,0.43720880D+03,0.125D+03,0.740D+02,0.16406000D+01,0.00000000D+00
     .,0.44459970D+03,0.125D+03,0.750D+02,0.16406000D+01,0.00000000D+00
     .,0.40230970D+03,0.125D+03,0.760D+02,0.16406000D+01,0.00000000D+00
     .,0.36794540D+03,0.125D+03,0.770D+02,0.16406000D+01,0.00000000D+00
     .,0.30505010D+03,0.125D+03,0.780D+02,0.16406000D+01,0.00000000D+00
     .,0.28480180D+03,0.125D+03,0.790D+02,0.16406000D+01,0.00000000D+00
     .,0.29291770D+03,0.125D+03,0.800D+02,0.16406000D+01,0.00000000D+00
     .,0.42959510D+03,0.125D+03,0.810D+02,0.16406000D+01,0.00000000D+00
     .,0.41941490D+03,0.125D+03,0.820D+02,0.16406000D+01,0.00000000D+00
     .,0.38444960D+03,0.125D+03,0.830D+02,0.16406000D+01,0.00000000D+00
     .,0.36605470D+03,0.125D+03,0.840D+02,0.16406000D+01,0.00000000D+00
     .,0.33709280D+03,0.125D+03,0.850D+02,0.16406000D+01,0.00000000D+00
     .,0.30829470D+03,0.125D+03,0.860D+02,0.16406000D+01,0.00000000D+00
     .,0.10472098D+04,0.125D+03,0.870D+02,0.16406000D+01,0.00000000D+00
     .,0.10062859D+04,0.125D+03,0.880D+02,0.16406000D+01,0.00000000D+00
     .,0.88823000D+03,0.125D+03,0.890D+02,0.16406000D+01,0.00000000D+00
     .,0.79631050D+03,0.125D+03,0.900D+02,0.16406000D+01,0.00000000D+00
     ./)
      pars(  33801:  33900)=(/
     . 0.79125770D+03,0.125D+03,0.910D+02,0.16406000D+01,0.00000000D+00
     .,0.76609840D+03,0.125D+03,0.920D+02,0.16406000D+01,0.00000000D+00
     .,0.78992350D+03,0.125D+03,0.930D+02,0.16406000D+01,0.00000000D+00
     .,0.76480610D+03,0.125D+03,0.940D+02,0.16406000D+01,0.00000000D+00
     .,0.42392700D+02,0.125D+03,0.101D+03,0.16406000D+01,0.00000000D+00
     .,0.13874910D+03,0.125D+03,0.103D+03,0.16406000D+01,0.98650000D+00
     .,0.17668170D+03,0.125D+03,0.104D+03,0.16406000D+01,0.98080000D+00
     .,0.13421360D+03,0.125D+03,0.105D+03,0.16406000D+01,0.97060000D+00
     .,0.10064630D+03,0.125D+03,0.106D+03,0.16406000D+01,0.98680000D+00
     .,0.69621500D+02,0.125D+03,0.107D+03,0.16406000D+01,0.99440000D+00
     .,0.50485800D+02,0.125D+03,0.108D+03,0.16406000D+01,0.99250000D+00
     .,0.34548100D+02,0.125D+03,0.109D+03,0.16406000D+01,0.99820000D+00
     .,0.20296760D+03,0.125D+03,0.111D+03,0.16406000D+01,0.96840000D+00
     .,0.31401190D+03,0.125D+03,0.112D+03,0.16406000D+01,0.96280000D+00
     .,0.31732870D+03,0.125D+03,0.113D+03,0.16406000D+01,0.96480000D+00
     .,0.25388000D+03,0.125D+03,0.114D+03,0.16406000D+01,0.95070000D+00
     .,0.20709950D+03,0.125D+03,0.115D+03,0.16406000D+01,0.99470000D+00
     .,0.17457750D+03,0.125D+03,0.116D+03,0.16406000D+01,0.99480000D+00
     .,0.14221510D+03,0.125D+03,0.117D+03,0.16406000D+01,0.99720000D+00
     .,0.27872590D+03,0.125D+03,0.119D+03,0.16406000D+01,0.97670000D+00
     ./)
      pars(  33901:  34000)=(/
     . 0.53581490D+03,0.125D+03,0.120D+03,0.16406000D+01,0.98310000D+00
     .,0.27819650D+03,0.125D+03,0.121D+03,0.16406000D+01,0.18627000D+01
     .,0.26846570D+03,0.125D+03,0.122D+03,0.16406000D+01,0.18299000D+01
     .,0.26311270D+03,0.125D+03,0.123D+03,0.16406000D+01,0.19138000D+01
     .,0.26077730D+03,0.125D+03,0.124D+03,0.16406000D+01,0.18269000D+01
     .,0.23955540D+03,0.125D+03,0.125D+03,0.16406000D+01,0.16406000D+01
     .,0.24345200D+02,0.126D+03,0.100D+01,0.16483000D+01,0.91180000D+00
     .,0.16056000D+02,0.126D+03,0.200D+01,0.16483000D+01,0.00000000D+00
     .,0.37921170D+03,0.126D+03,0.300D+01,0.16483000D+01,0.00000000D+00
     .,0.21947750D+03,0.126D+03,0.400D+01,0.16483000D+01,0.00000000D+00
     .,0.14762980D+03,0.126D+03,0.500D+01,0.16483000D+01,0.00000000D+00
     .,0.99588000D+02,0.126D+03,0.600D+01,0.16483000D+01,0.00000000D+00
     .,0.69563800D+02,0.126D+03,0.700D+01,0.16483000D+01,0.00000000D+00
     .,0.52635100D+02,0.126D+03,0.800D+01,0.16483000D+01,0.00000000D+00
     .,0.39862800D+02,0.126D+03,0.900D+01,0.16483000D+01,0.00000000D+00
     .,0.30664800D+02,0.126D+03,0.100D+02,0.16483000D+01,0.00000000D+00
     .,0.45354440D+03,0.126D+03,0.110D+02,0.16483000D+01,0.00000000D+00
     .,0.34971160D+03,0.126D+03,0.120D+02,0.16483000D+01,0.00000000D+00
     .,0.32227810D+03,0.126D+03,0.130D+02,0.16483000D+01,0.00000000D+00
     .,0.25376380D+03,0.126D+03,0.140D+02,0.16483000D+01,0.00000000D+00
     ./)
      pars(  34001:  34100)=(/
     . 0.19765690D+03,0.126D+03,0.150D+02,0.16483000D+01,0.00000000D+00
     .,0.16387940D+03,0.126D+03,0.160D+02,0.16483000D+01,0.00000000D+00
     .,0.13373240D+03,0.126D+03,0.170D+02,0.16483000D+01,0.00000000D+00
     .,0.10931550D+03,0.126D+03,0.180D+02,0.16483000D+01,0.00000000D+00
     .,0.74225200D+03,0.126D+03,0.190D+02,0.16483000D+01,0.00000000D+00
     .,0.61392330D+03,0.126D+03,0.200D+02,0.16483000D+01,0.00000000D+00
     .,0.50734010D+03,0.126D+03,0.210D+02,0.16483000D+01,0.00000000D+00
     .,0.48989960D+03,0.126D+03,0.220D+02,0.16483000D+01,0.00000000D+00
     .,0.44862620D+03,0.126D+03,0.230D+02,0.16483000D+01,0.00000000D+00
     .,0.35327090D+03,0.126D+03,0.240D+02,0.16483000D+01,0.00000000D+00
     .,0.38626020D+03,0.126D+03,0.250D+02,0.16483000D+01,0.00000000D+00
     .,0.30304800D+03,0.126D+03,0.260D+02,0.16483000D+01,0.00000000D+00
     .,0.32134480D+03,0.126D+03,0.270D+02,0.16483000D+01,0.00000000D+00
     .,0.33106140D+03,0.126D+03,0.280D+02,0.16483000D+01,0.00000000D+00
     .,0.25370380D+03,0.126D+03,0.290D+02,0.16483000D+01,0.00000000D+00
     .,0.26053410D+03,0.126D+03,0.300D+02,0.16483000D+01,0.00000000D+00
     .,0.30855290D+03,0.126D+03,0.310D+02,0.16483000D+01,0.00000000D+00
     .,0.27197110D+03,0.126D+03,0.320D+02,0.16483000D+01,0.00000000D+00
     .,0.23176180D+03,0.126D+03,0.330D+02,0.16483000D+01,0.00000000D+00
     .,0.20779480D+03,0.126D+03,0.340D+02,0.16483000D+01,0.00000000D+00
     ./)
      pars(  34101:  34200)=(/
     . 0.18168360D+03,0.126D+03,0.350D+02,0.16483000D+01,0.00000000D+00
     .,0.15786610D+03,0.126D+03,0.360D+02,0.16483000D+01,0.00000000D+00
     .,0.83179600D+03,0.126D+03,0.370D+02,0.16483000D+01,0.00000000D+00
     .,0.73123300D+03,0.126D+03,0.380D+02,0.16483000D+01,0.00000000D+00
     .,0.64066410D+03,0.126D+03,0.390D+02,0.16483000D+01,0.00000000D+00
     .,0.57577520D+03,0.126D+03,0.400D+02,0.16483000D+01,0.00000000D+00
     .,0.52495880D+03,0.126D+03,0.410D+02,0.16483000D+01,0.00000000D+00
     .,0.40508170D+03,0.126D+03,0.420D+02,0.16483000D+01,0.00000000D+00
     .,0.45208310D+03,0.126D+03,0.430D+02,0.16483000D+01,0.00000000D+00
     .,0.34423220D+03,0.126D+03,0.440D+02,0.16483000D+01,0.00000000D+00
     .,0.37640450D+03,0.126D+03,0.450D+02,0.16483000D+01,0.00000000D+00
     .,0.34902580D+03,0.126D+03,0.460D+02,0.16483000D+01,0.00000000D+00
     .,0.29090830D+03,0.126D+03,0.470D+02,0.16483000D+01,0.00000000D+00
     .,0.30752090D+03,0.126D+03,0.480D+02,0.16483000D+01,0.00000000D+00
     .,0.38606280D+03,0.126D+03,0.490D+02,0.16483000D+01,0.00000000D+00
     .,0.35691590D+03,0.126D+03,0.500D+02,0.16483000D+01,0.00000000D+00
     .,0.31786240D+03,0.126D+03,0.510D+02,0.16483000D+01,0.00000000D+00
     .,0.29473480D+03,0.126D+03,0.520D+02,0.16483000D+01,0.00000000D+00
     .,0.26629100D+03,0.126D+03,0.530D+02,0.16483000D+01,0.00000000D+00
     .,0.23921900D+03,0.126D+03,0.540D+02,0.16483000D+01,0.00000000D+00
     ./)
      pars(  34201:  34300)=(/
     . 0.10130459D+04,0.126D+03,0.550D+02,0.16483000D+01,0.00000000D+00
     .,0.93186100D+03,0.126D+03,0.560D+02,0.16483000D+01,0.00000000D+00
     .,0.81889940D+03,0.126D+03,0.570D+02,0.16483000D+01,0.00000000D+00
     .,0.37545680D+03,0.126D+03,0.580D+02,0.16483000D+01,0.27991000D+01
     .,0.82577350D+03,0.126D+03,0.590D+02,0.16483000D+01,0.00000000D+00
     .,0.79302390D+03,0.126D+03,0.600D+02,0.16483000D+01,0.00000000D+00
     .,0.77315750D+03,0.126D+03,0.610D+02,0.16483000D+01,0.00000000D+00
     .,0.75488610D+03,0.126D+03,0.620D+02,0.16483000D+01,0.00000000D+00
     .,0.73868360D+03,0.126D+03,0.630D+02,0.16483000D+01,0.00000000D+00
     .,0.58077730D+03,0.126D+03,0.640D+02,0.16483000D+01,0.00000000D+00
     .,0.65347520D+03,0.126D+03,0.650D+02,0.16483000D+01,0.00000000D+00
     .,0.63025330D+03,0.126D+03,0.660D+02,0.16483000D+01,0.00000000D+00
     .,0.66633780D+03,0.126D+03,0.670D+02,0.16483000D+01,0.00000000D+00
     .,0.65221550D+03,0.126D+03,0.680D+02,0.16483000D+01,0.00000000D+00
     .,0.63947820D+03,0.126D+03,0.690D+02,0.16483000D+01,0.00000000D+00
     .,0.63201230D+03,0.126D+03,0.700D+02,0.16483000D+01,0.00000000D+00
     .,0.53244100D+03,0.126D+03,0.710D+02,0.16483000D+01,0.00000000D+00
     .,0.52361220D+03,0.126D+03,0.720D+02,0.16483000D+01,0.00000000D+00
     .,0.47773970D+03,0.126D+03,0.730D+02,0.16483000D+01,0.00000000D+00
     .,0.40318600D+03,0.126D+03,0.740D+02,0.16483000D+01,0.00000000D+00
     ./)
      pars(  34301:  34400)=(/
     . 0.41018490D+03,0.126D+03,0.750D+02,0.16483000D+01,0.00000000D+00
     .,0.37164480D+03,0.126D+03,0.760D+02,0.16483000D+01,0.00000000D+00
     .,0.34026720D+03,0.126D+03,0.770D+02,0.16483000D+01,0.00000000D+00
     .,0.28250560D+03,0.126D+03,0.780D+02,0.16483000D+01,0.00000000D+00
     .,0.26390350D+03,0.126D+03,0.790D+02,0.16483000D+01,0.00000000D+00
     .,0.27150750D+03,0.126D+03,0.800D+02,0.16483000D+01,0.00000000D+00
     .,0.39615750D+03,0.126D+03,0.810D+02,0.16483000D+01,0.00000000D+00
     .,0.38734040D+03,0.126D+03,0.820D+02,0.16483000D+01,0.00000000D+00
     .,0.35575280D+03,0.126D+03,0.830D+02,0.16483000D+01,0.00000000D+00
     .,0.33914780D+03,0.126D+03,0.840D+02,0.16483000D+01,0.00000000D+00
     .,0.31279910D+03,0.126D+03,0.850D+02,0.16483000D+01,0.00000000D+00
     .,0.28650170D+03,0.126D+03,0.860D+02,0.16483000D+01,0.00000000D+00
     .,0.95682050D+03,0.126D+03,0.870D+02,0.16483000D+01,0.00000000D+00
     .,0.92149130D+03,0.126D+03,0.880D+02,0.16483000D+01,0.00000000D+00
     .,0.81480800D+03,0.126D+03,0.890D+02,0.16483000D+01,0.00000000D+00
     .,0.73215470D+03,0.126D+03,0.900D+02,0.16483000D+01,0.00000000D+00
     .,0.72684330D+03,0.126D+03,0.910D+02,0.16483000D+01,0.00000000D+00
     .,0.70377870D+03,0.126D+03,0.920D+02,0.16483000D+01,0.00000000D+00
     .,0.72468840D+03,0.126D+03,0.930D+02,0.16483000D+01,0.00000000D+00
     .,0.70180630D+03,0.126D+03,0.940D+02,0.16483000D+01,0.00000000D+00
     ./)
      pars(  34401:  34500)=(/
     . 0.39274200D+02,0.126D+03,0.101D+03,0.16483000D+01,0.00000000D+00
     .,0.12754920D+03,0.126D+03,0.103D+03,0.16483000D+01,0.98650000D+00
     .,0.16261080D+03,0.126D+03,0.104D+03,0.16483000D+01,0.98080000D+00
     .,0.12411700D+03,0.126D+03,0.105D+03,0.16483000D+01,0.97060000D+00
     .,0.93384800D+02,0.126D+03,0.106D+03,0.16483000D+01,0.98680000D+00
     .,0.64846800D+02,0.126D+03,0.107D+03,0.16483000D+01,0.99440000D+00
     .,0.47181600D+02,0.126D+03,0.108D+03,0.16483000D+01,0.99250000D+00
     .,0.32432100D+02,0.126D+03,0.109D+03,0.16483000D+01,0.99820000D+00
     .,0.18646690D+03,0.126D+03,0.111D+03,0.16483000D+01,0.96840000D+00
     .,0.28833710D+03,0.126D+03,0.112D+03,0.16483000D+01,0.96280000D+00
     .,0.29202470D+03,0.126D+03,0.113D+03,0.16483000D+01,0.96480000D+00
     .,0.23448430D+03,0.126D+03,0.114D+03,0.16483000D+01,0.95070000D+00
     .,0.19184290D+03,0.126D+03,0.115D+03,0.16483000D+01,0.99470000D+00
     .,0.16207500D+03,0.126D+03,0.116D+03,0.16483000D+01,0.99480000D+00
     .,0.13234800D+03,0.126D+03,0.117D+03,0.16483000D+01,0.99720000D+00
     .,0.25675790D+03,0.126D+03,0.119D+03,0.16483000D+01,0.97670000D+00
     .,0.49074810D+03,0.126D+03,0.120D+03,0.16483000D+01,0.98310000D+00
     .,0.25688890D+03,0.126D+03,0.121D+03,0.16483000D+01,0.18627000D+01
     .,0.24796360D+03,0.126D+03,0.122D+03,0.16483000D+01,0.18299000D+01
     .,0.24301480D+03,0.126D+03,0.123D+03,0.16483000D+01,0.19138000D+01
     ./)
      pars(  34501:  34600)=(/
     . 0.24078340D+03,0.126D+03,0.124D+03,0.16483000D+01,0.18269000D+01
     .,0.22153400D+03,0.126D+03,0.125D+03,0.16483000D+01,0.16406000D+01
     .,0.20499250D+03,0.126D+03,0.126D+03,0.16483000D+01,0.16483000D+01
     .,0.23231000D+02,0.127D+03,0.100D+01,0.17149000D+01,0.91180000D+00
     .,0.15354500D+02,0.127D+03,0.200D+01,0.17149000D+01,0.00000000D+00
     .,0.36237460D+03,0.127D+03,0.300D+01,0.17149000D+01,0.00000000D+00
     .,0.20922270D+03,0.127D+03,0.400D+01,0.17149000D+01,0.00000000D+00
     .,0.14074280D+03,0.127D+03,0.500D+01,0.17149000D+01,0.00000000D+00
     .,0.95005100D+02,0.127D+03,0.600D+01,0.17149000D+01,0.00000000D+00
     .,0.66421800D+02,0.127D+03,0.700D+01,0.17149000D+01,0.00000000D+00
     .,0.50301600D+02,0.127D+03,0.800D+01,0.17149000D+01,0.00000000D+00
     .,0.38131300D+02,0.127D+03,0.900D+01,0.17149000D+01,0.00000000D+00
     .,0.29359800D+02,0.127D+03,0.100D+02,0.17149000D+01,0.00000000D+00
     .,0.43337360D+03,0.127D+03,0.110D+02,0.17149000D+01,0.00000000D+00
     .,0.33347670D+03,0.127D+03,0.120D+02,0.17149000D+01,0.00000000D+00
     .,0.30725400D+03,0.127D+03,0.130D+02,0.17149000D+01,0.00000000D+00
     .,0.24191590D+03,0.127D+03,0.140D+02,0.17149000D+01,0.00000000D+00
     .,0.18848520D+03,0.127D+03,0.150D+02,0.17149000D+01,0.00000000D+00
     .,0.15634230D+03,0.127D+03,0.160D+02,0.17149000D+01,0.00000000D+00
     .,0.12765130D+03,0.127D+03,0.170D+02,0.17149000D+01,0.00000000D+00
     ./)
      pars(  34601:  34700)=(/
     . 0.10440910D+03,0.127D+03,0.180D+02,0.17149000D+01,0.00000000D+00
     .,0.71018180D+03,0.127D+03,0.190D+02,0.17149000D+01,0.00000000D+00
     .,0.58595780D+03,0.127D+03,0.200D+02,0.17149000D+01,0.00000000D+00
     .,0.48407230D+03,0.127D+03,0.210D+02,0.17149000D+01,0.00000000D+00
     .,0.46739840D+03,0.127D+03,0.220D+02,0.17149000D+01,0.00000000D+00
     .,0.42799360D+03,0.127D+03,0.230D+02,0.17149000D+01,0.00000000D+00
     .,0.33714360D+03,0.127D+03,0.240D+02,0.17149000D+01,0.00000000D+00
     .,0.36847190D+03,0.127D+03,0.250D+02,0.17149000D+01,0.00000000D+00
     .,0.28919370D+03,0.127D+03,0.260D+02,0.17149000D+01,0.00000000D+00
     .,0.30649680D+03,0.127D+03,0.270D+02,0.17149000D+01,0.00000000D+00
     .,0.31577130D+03,0.127D+03,0.280D+02,0.17149000D+01,0.00000000D+00
     .,0.24210300D+03,0.127D+03,0.290D+02,0.17149000D+01,0.00000000D+00
     .,0.24848550D+03,0.127D+03,0.300D+02,0.17149000D+01,0.00000000D+00
     .,0.29424080D+03,0.127D+03,0.310D+02,0.17149000D+01,0.00000000D+00
     .,0.25932240D+03,0.127D+03,0.320D+02,0.17149000D+01,0.00000000D+00
     .,0.22102330D+03,0.127D+03,0.330D+02,0.17149000D+01,0.00000000D+00
     .,0.19822130D+03,0.127D+03,0.340D+02,0.17149000D+01,0.00000000D+00
     .,0.17337930D+03,0.127D+03,0.350D+02,0.17149000D+01,0.00000000D+00
     .,0.15071900D+03,0.127D+03,0.360D+02,0.17149000D+01,0.00000000D+00
     .,0.79588590D+03,0.127D+03,0.370D+02,0.17149000D+01,0.00000000D+00
     ./)
      pars(  34701:  34800)=(/
     . 0.69807620D+03,0.127D+03,0.380D+02,0.17149000D+01,0.00000000D+00
     .,0.61134030D+03,0.127D+03,0.390D+02,0.17149000D+01,0.00000000D+00
     .,0.54932940D+03,0.127D+03,0.400D+02,0.17149000D+01,0.00000000D+00
     .,0.50083550D+03,0.127D+03,0.410D+02,0.17149000D+01,0.00000000D+00
     .,0.38652450D+03,0.127D+03,0.420D+02,0.17149000D+01,0.00000000D+00
     .,0.43134780D+03,0.127D+03,0.430D+02,0.17149000D+01,0.00000000D+00
     .,0.32850150D+03,0.127D+03,0.440D+02,0.17149000D+01,0.00000000D+00
     .,0.35910570D+03,0.127D+03,0.450D+02,0.17149000D+01,0.00000000D+00
     .,0.33298960D+03,0.127D+03,0.460D+02,0.17149000D+01,0.00000000D+00
     .,0.27768230D+03,0.127D+03,0.470D+02,0.17149000D+01,0.00000000D+00
     .,0.29340740D+03,0.127D+03,0.480D+02,0.17149000D+01,0.00000000D+00
     .,0.36832240D+03,0.127D+03,0.490D+02,0.17149000D+01,0.00000000D+00
     .,0.34042650D+03,0.127D+03,0.500D+02,0.17149000D+01,0.00000000D+00
     .,0.30318560D+03,0.127D+03,0.510D+02,0.17149000D+01,0.00000000D+00
     .,0.28116330D+03,0.127D+03,0.520D+02,0.17149000D+01,0.00000000D+00
     .,0.25408770D+03,0.127D+03,0.530D+02,0.17149000D+01,0.00000000D+00
     .,0.22832560D+03,0.127D+03,0.540D+02,0.17149000D+01,0.00000000D+00
     .,0.96954620D+03,0.127D+03,0.550D+02,0.17149000D+01,0.00000000D+00
     .,0.88993460D+03,0.127D+03,0.560D+02,0.17149000D+01,0.00000000D+00
     .,0.78163500D+03,0.127D+03,0.570D+02,0.17149000D+01,0.00000000D+00
     ./)
      pars(  34801:  34900)=(/
     . 0.35816770D+03,0.127D+03,0.580D+02,0.17149000D+01,0.27991000D+01
     .,0.78858090D+03,0.127D+03,0.590D+02,0.17149000D+01,0.00000000D+00
     .,0.75717110D+03,0.127D+03,0.600D+02,0.17149000D+01,0.00000000D+00
     .,0.73817480D+03,0.127D+03,0.610D+02,0.17149000D+01,0.00000000D+00
     .,0.72070520D+03,0.127D+03,0.620D+02,0.17149000D+01,0.00000000D+00
     .,0.70521290D+03,0.127D+03,0.630D+02,0.17149000D+01,0.00000000D+00
     .,0.55434620D+03,0.127D+03,0.640D+02,0.17149000D+01,0.00000000D+00
     .,0.62444810D+03,0.127D+03,0.650D+02,0.17149000D+01,0.00000000D+00
     .,0.60226720D+03,0.127D+03,0.660D+02,0.17149000D+01,0.00000000D+00
     .,0.63603570D+03,0.127D+03,0.670D+02,0.17149000D+01,0.00000000D+00
     .,0.62253720D+03,0.127D+03,0.680D+02,0.17149000D+01,0.00000000D+00
     .,0.61036000D+03,0.127D+03,0.690D+02,0.17149000D+01,0.00000000D+00
     .,0.60323520D+03,0.127D+03,0.700D+02,0.17149000D+01,0.00000000D+00
     .,0.50816310D+03,0.127D+03,0.710D+02,0.17149000D+01,0.00000000D+00
     .,0.49947010D+03,0.127D+03,0.720D+02,0.17149000D+01,0.00000000D+00
     .,0.45568660D+03,0.127D+03,0.730D+02,0.17149000D+01,0.00000000D+00
     .,0.38467230D+03,0.127D+03,0.740D+02,0.17149000D+01,0.00000000D+00
     .,0.39131100D+03,0.127D+03,0.750D+02,0.17149000D+01,0.00000000D+00
     .,0.35455820D+03,0.127D+03,0.760D+02,0.17149000D+01,0.00000000D+00
     .,0.32464910D+03,0.127D+03,0.770D+02,0.17149000D+01,0.00000000D+00
     ./)
      pars(  34901:  35000)=(/
     . 0.26963380D+03,0.127D+03,0.780D+02,0.17149000D+01,0.00000000D+00
     .,0.25191510D+03,0.127D+03,0.790D+02,0.17149000D+01,0.00000000D+00
     .,0.25913820D+03,0.127D+03,0.800D+02,0.17149000D+01,0.00000000D+00
     .,0.37809130D+03,0.127D+03,0.810D+02,0.17149000D+01,0.00000000D+00
     .,0.36955020D+03,0.127D+03,0.820D+02,0.17149000D+01,0.00000000D+00
     .,0.33939390D+03,0.127D+03,0.830D+02,0.17149000D+01,0.00000000D+00
     .,0.32357100D+03,0.127D+03,0.840D+02,0.17149000D+01,0.00000000D+00
     .,0.29847990D+03,0.127D+03,0.850D+02,0.17149000D+01,0.00000000D+00
     .,0.27345010D+03,0.127D+03,0.860D+02,0.17149000D+01,0.00000000D+00
     .,0.91516100D+03,0.127D+03,0.870D+02,0.17149000D+01,0.00000000D+00
     .,0.87983390D+03,0.127D+03,0.880D+02,0.17149000D+01,0.00000000D+00
     .,0.77761650D+03,0.127D+03,0.890D+02,0.17149000D+01,0.00000000D+00
     .,0.69856190D+03,0.127D+03,0.900D+02,0.17149000D+01,0.00000000D+00
     .,0.69371430D+03,0.127D+03,0.910D+02,0.17149000D+01,0.00000000D+00
     .,0.67170070D+03,0.127D+03,0.920D+02,0.17149000D+01,0.00000000D+00
     .,0.69179850D+03,0.127D+03,0.930D+02,0.17149000D+01,0.00000000D+00
     .,0.66991580D+03,0.127D+03,0.940D+02,0.17149000D+01,0.00000000D+00
     .,0.37449600D+02,0.127D+03,0.101D+03,0.17149000D+01,0.00000000D+00
     .,0.12160370D+03,0.127D+03,0.103D+03,0.17149000D+01,0.98650000D+00
     .,0.15505540D+03,0.127D+03,0.104D+03,0.17149000D+01,0.98080000D+00
     ./)
      pars(  35001:  35100)=(/
     . 0.11835300D+03,0.127D+03,0.105D+03,0.17149000D+01,0.97060000D+00
     .,0.89099200D+02,0.127D+03,0.106D+03,0.17149000D+01,0.98680000D+00
     .,0.61925600D+02,0.127D+03,0.107D+03,0.17149000D+01,0.99440000D+00
     .,0.45100200D+02,0.127D+03,0.108D+03,0.17149000D+01,0.99250000D+00
     .,0.31051600D+02,0.127D+03,0.109D+03,0.17149000D+01,0.99820000D+00
     .,0.17783550D+03,0.127D+03,0.111D+03,0.17149000D+01,0.96840000D+00
     .,0.27498400D+03,0.127D+03,0.112D+03,0.17149000D+01,0.96280000D+00
     .,0.27841730D+03,0.127D+03,0.113D+03,0.17149000D+01,0.96480000D+00
     .,0.22355480D+03,0.127D+03,0.114D+03,0.17149000D+01,0.95070000D+00
     .,0.18295110D+03,0.127D+03,0.115D+03,0.17149000D+01,0.99470000D+00
     .,0.15462430D+03,0.127D+03,0.116D+03,0.17149000D+01,0.99480000D+00
     .,0.12633190D+03,0.127D+03,0.117D+03,0.17149000D+01,0.99720000D+00
     .,0.24501880D+03,0.127D+03,0.119D+03,0.17149000D+01,0.97670000D+00
     .,0.46868990D+03,0.127D+03,0.120D+03,0.17149000D+01,0.98310000D+00
     .,0.24500350D+03,0.127D+03,0.121D+03,0.17149000D+01,0.18627000D+01
     .,0.23653250D+03,0.127D+03,0.122D+03,0.17149000D+01,0.18299000D+01
     .,0.23181800D+03,0.127D+03,0.123D+03,0.17149000D+01,0.19138000D+01
     .,0.22970220D+03,0.127D+03,0.124D+03,0.17149000D+01,0.18269000D+01
     .,0.21130250D+03,0.127D+03,0.125D+03,0.17149000D+01,0.16406000D+01
     .,0.19554300D+03,0.127D+03,0.126D+03,0.17149000D+01,0.16483000D+01
     ./)
      pars(  35101:  35200)=(/
     . 0.18654620D+03,0.127D+03,0.127D+03,0.17149000D+01,0.17149000D+01
     .,0.22682200D+02,0.128D+03,0.100D+01,0.17937000D+01,0.91180000D+00
     .,0.14983300D+02,0.128D+03,0.200D+01,0.17937000D+01,0.00000000D+00
     .,0.35596060D+03,0.128D+03,0.300D+01,0.17937000D+01,0.00000000D+00
     .,0.20490860D+03,0.128D+03,0.400D+01,0.17937000D+01,0.00000000D+00
     .,0.13762540D+03,0.128D+03,0.500D+01,0.17937000D+01,0.00000000D+00
     .,0.92810700D+02,0.128D+03,0.600D+01,0.17937000D+01,0.00000000D+00
     .,0.64851900D+02,0.128D+03,0.700D+01,0.17937000D+01,0.00000000D+00
     .,0.49099600D+02,0.128D+03,0.800D+01,0.17937000D+01,0.00000000D+00
     .,0.37215800D+02,0.128D+03,0.900D+01,0.17937000D+01,0.00000000D+00
     .,0.28655400D+02,0.128D+03,0.100D+02,0.17937000D+01,0.00000000D+00
     .,0.42564550D+03,0.128D+03,0.110D+02,0.17937000D+01,0.00000000D+00
     .,0.32678640D+03,0.128D+03,0.120D+02,0.17937000D+01,0.00000000D+00
     .,0.30082710D+03,0.128D+03,0.130D+02,0.17937000D+01,0.00000000D+00
     .,0.23659520D+03,0.128D+03,0.140D+02,0.17937000D+01,0.00000000D+00
     .,0.18418470D+03,0.128D+03,0.150D+02,0.17937000D+01,0.00000000D+00
     .,0.15270360D+03,0.128D+03,0.160D+02,0.17937000D+01,0.00000000D+00
     .,0.12463170D+03,0.128D+03,0.170D+02,0.17937000D+01,0.00000000D+00
     .,0.10191160D+03,0.128D+03,0.180D+02,0.17937000D+01,0.00000000D+00
     .,0.69776500D+03,0.128D+03,0.190D+02,0.17937000D+01,0.00000000D+00
     ./)
      pars(  35201:  35300)=(/
     . 0.57475220D+03,0.128D+03,0.200D+02,0.17937000D+01,0.00000000D+00
     .,0.47462540D+03,0.128D+03,0.210D+02,0.17937000D+01,0.00000000D+00
     .,0.45811360D+03,0.128D+03,0.220D+02,0.17937000D+01,0.00000000D+00
     .,0.41940370D+03,0.128D+03,0.230D+02,0.17937000D+01,0.00000000D+00
     .,0.33038080D+03,0.128D+03,0.240D+02,0.17937000D+01,0.00000000D+00
     .,0.36096920D+03,0.128D+03,0.250D+02,0.17937000D+01,0.00000000D+00
     .,0.28330110D+03,0.128D+03,0.260D+02,0.17937000D+01,0.00000000D+00
     .,0.30010550D+03,0.128D+03,0.270D+02,0.17937000D+01,0.00000000D+00
     .,0.30925430D+03,0.128D+03,0.280D+02,0.17937000D+01,0.00000000D+00
     .,0.23712000D+03,0.128D+03,0.290D+02,0.17937000D+01,0.00000000D+00
     .,0.24319600D+03,0.128D+03,0.300D+02,0.17937000D+01,0.00000000D+00
     .,0.28800730D+03,0.128D+03,0.310D+02,0.17937000D+01,0.00000000D+00
     .,0.25361340D+03,0.128D+03,0.320D+02,0.17937000D+01,0.00000000D+00
     .,0.21600430D+03,0.128D+03,0.330D+02,0.17937000D+01,0.00000000D+00
     .,0.19364320D+03,0.128D+03,0.340D+02,0.17937000D+01,0.00000000D+00
     .,0.16931330D+03,0.128D+03,0.350D+02,0.17937000D+01,0.00000000D+00
     .,0.14714170D+03,0.128D+03,0.360D+02,0.17937000D+01,0.00000000D+00
     .,0.78184410D+03,0.128D+03,0.370D+02,0.17937000D+01,0.00000000D+00
     .,0.68477150D+03,0.128D+03,0.380D+02,0.17937000D+01,0.00000000D+00
     .,0.59926520D+03,0.128D+03,0.390D+02,0.17937000D+01,0.00000000D+00
     ./)
      pars(  35301:  35400)=(/
     . 0.53824800D+03,0.128D+03,0.400D+02,0.17937000D+01,0.00000000D+00
     .,0.49059730D+03,0.128D+03,0.410D+02,0.17937000D+01,0.00000000D+00
     .,0.37845940D+03,0.128D+03,0.420D+02,0.17937000D+01,0.00000000D+00
     .,0.42241440D+03,0.128D+03,0.430D+02,0.17937000D+01,0.00000000D+00
     .,0.32155060D+03,0.128D+03,0.440D+02,0.17937000D+01,0.00000000D+00
     .,0.35149860D+03,0.128D+03,0.450D+02,0.17937000D+01,0.00000000D+00
     .,0.32588580D+03,0.128D+03,0.460D+02,0.17937000D+01,0.00000000D+00
     .,0.27180880D+03,0.128D+03,0.470D+02,0.17937000D+01,0.00000000D+00
     .,0.28709580D+03,0.128D+03,0.480D+02,0.17937000D+01,0.00000000D+00
     .,0.36057310D+03,0.128D+03,0.490D+02,0.17937000D+01,0.00000000D+00
     .,0.33303010D+03,0.128D+03,0.500D+02,0.17937000D+01,0.00000000D+00
     .,0.29640640D+03,0.128D+03,0.510D+02,0.17937000D+01,0.00000000D+00
     .,0.27477510D+03,0.128D+03,0.520D+02,0.17937000D+01,0.00000000D+00
     .,0.24822370D+03,0.128D+03,0.530D+02,0.17937000D+01,0.00000000D+00
     .,0.22298560D+03,0.128D+03,0.540D+02,0.17937000D+01,0.00000000D+00
     .,0.95234010D+03,0.128D+03,0.550D+02,0.17937000D+01,0.00000000D+00
     .,0.87316220D+03,0.128D+03,0.560D+02,0.17937000D+01,0.00000000D+00
     .,0.76636770D+03,0.128D+03,0.570D+02,0.17937000D+01,0.00000000D+00
     .,0.35020540D+03,0.128D+03,0.580D+02,0.17937000D+01,0.27991000D+01
     .,0.77355490D+03,0.128D+03,0.590D+02,0.17937000D+01,0.00000000D+00
     ./)
      pars(  35401:  35500)=(/
     . 0.74267570D+03,0.128D+03,0.600D+02,0.17937000D+01,0.00000000D+00
     .,0.72402070D+03,0.128D+03,0.610D+02,0.17937000D+01,0.00000000D+00
     .,0.70686710D+03,0.128D+03,0.620D+02,0.17937000D+01,0.00000000D+00
     .,0.69165380D+03,0.128D+03,0.630D+02,0.17937000D+01,0.00000000D+00
     .,0.54327420D+03,0.128D+03,0.640D+02,0.17937000D+01,0.00000000D+00
     .,0.61263290D+03,0.128D+03,0.650D+02,0.17937000D+01,0.00000000D+00
     .,0.59076940D+03,0.128D+03,0.660D+02,0.17937000D+01,0.00000000D+00
     .,0.62369960D+03,0.128D+03,0.670D+02,0.17937000D+01,0.00000000D+00
     .,0.61045110D+03,0.128D+03,0.680D+02,0.17937000D+01,0.00000000D+00
     .,0.59849350D+03,0.128D+03,0.690D+02,0.17937000D+01,0.00000000D+00
     .,0.59152550D+03,0.128D+03,0.700D+02,0.17937000D+01,0.00000000D+00
     .,0.49802530D+03,0.128D+03,0.710D+02,0.17937000D+01,0.00000000D+00
     .,0.48915250D+03,0.128D+03,0.720D+02,0.17937000D+01,0.00000000D+00
     .,0.44609160D+03,0.128D+03,0.730D+02,0.17937000D+01,0.00000000D+00
     .,0.37646650D+03,0.128D+03,0.740D+02,0.17937000D+01,0.00000000D+00
     .,0.38290120D+03,0.128D+03,0.750D+02,0.17937000D+01,0.00000000D+00
     .,0.34682510D+03,0.128D+03,0.760D+02,0.17937000D+01,0.00000000D+00
     .,0.31748840D+03,0.128D+03,0.770D+02,0.17937000D+01,0.00000000D+00
     .,0.26363220D+03,0.128D+03,0.780D+02,0.17937000D+01,0.00000000D+00
     .,0.24629020D+03,0.128D+03,0.790D+02,0.17937000D+01,0.00000000D+00
     ./)
      pars(  35501:  35600)=(/
     . 0.25331490D+03,0.128D+03,0.800D+02,0.17937000D+01,0.00000000D+00
     .,0.37009330D+03,0.128D+03,0.810D+02,0.17937000D+01,0.00000000D+00
     .,0.36153570D+03,0.128D+03,0.820D+02,0.17937000D+01,0.00000000D+00
     .,0.33183930D+03,0.128D+03,0.830D+02,0.17937000D+01,0.00000000D+00
     .,0.31626530D+03,0.128D+03,0.840D+02,0.17937000D+01,0.00000000D+00
     .,0.29163450D+03,0.128D+03,0.850D+02,0.17937000D+01,0.00000000D+00
     .,0.26709740D+03,0.128D+03,0.860D+02,0.17937000D+01,0.00000000D+00
     .,0.89847060D+03,0.128D+03,0.870D+02,0.17937000D+01,0.00000000D+00
     .,0.86294020D+03,0.128D+03,0.880D+02,0.17937000D+01,0.00000000D+00
     .,0.76220730D+03,0.128D+03,0.890D+02,0.17937000D+01,0.00000000D+00
     .,0.68424210D+03,0.128D+03,0.900D+02,0.17937000D+01,0.00000000D+00
     .,0.67974920D+03,0.128D+03,0.910D+02,0.17937000D+01,0.00000000D+00
     .,0.65817270D+03,0.128D+03,0.920D+02,0.17937000D+01,0.00000000D+00
     .,0.67818560D+03,0.128D+03,0.930D+02,0.17937000D+01,0.00000000D+00
     .,0.65668000D+03,0.128D+03,0.940D+02,0.17937000D+01,0.00000000D+00
     .,0.36590100D+02,0.128D+03,0.101D+03,0.17937000D+01,0.00000000D+00
     .,0.11907250D+03,0.128D+03,0.103D+03,0.17937000D+01,0.98650000D+00
     .,0.15177980D+03,0.128D+03,0.104D+03,0.17937000D+01,0.98080000D+00
     .,0.11570000D+03,0.128D+03,0.105D+03,0.17937000D+01,0.97060000D+00
     .,0.87043200D+02,0.128D+03,0.106D+03,0.17937000D+01,0.98680000D+00
     ./)
      pars(  35601:  35700)=(/
     . 0.60460800D+02,0.128D+03,0.107D+03,0.17937000D+01,0.99440000D+00
     .,0.44017300D+02,0.128D+03,0.108D+03,0.17937000D+01,0.99250000D+00
     .,0.30300400D+02,0.128D+03,0.109D+03,0.17937000D+01,0.99820000D+00
     .,0.17420010D+03,0.128D+03,0.111D+03,0.17937000D+01,0.96840000D+00
     .,0.26937790D+03,0.128D+03,0.112D+03,0.17937000D+01,0.96280000D+00
     .,0.27254090D+03,0.128D+03,0.113D+03,0.17937000D+01,0.96480000D+00
     .,0.21860660D+03,0.128D+03,0.114D+03,0.17937000D+01,0.95070000D+00
     .,0.17877350D+03,0.128D+03,0.115D+03,0.17937000D+01,0.99470000D+00
     .,0.15102860D+03,0.128D+03,0.116D+03,0.17937000D+01,0.99480000D+00
     .,0.12334570D+03,0.128D+03,0.117D+03,0.17937000D+01,0.99720000D+00
     .,0.23988490D+03,0.128D+03,0.119D+03,0.17937000D+01,0.97670000D+00
     .,0.45968120D+03,0.128D+03,0.120D+03,0.17937000D+01,0.98310000D+00
     .,0.23965630D+03,0.128D+03,0.121D+03,0.17937000D+01,0.18627000D+01
     .,0.23136310D+03,0.128D+03,0.122D+03,0.17937000D+01,0.18299000D+01
     .,0.22676010D+03,0.128D+03,0.123D+03,0.17937000D+01,0.19138000D+01
     .,0.22471760D+03,0.128D+03,0.124D+03,0.17937000D+01,0.18269000D+01
     .,0.20661260D+03,0.128D+03,0.125D+03,0.17937000D+01,0.16406000D+01
     .,0.19117980D+03,0.128D+03,0.126D+03,0.17937000D+01,0.16483000D+01
     .,0.18238760D+03,0.128D+03,0.127D+03,0.17937000D+01,0.17149000D+01
     .,0.17833110D+03,0.128D+03,0.128D+03,0.17937000D+01,0.17937000D+01
     ./)
      pars(  35701:  35800)=(/
     . 0.22155000D+02,0.129D+03,0.100D+01,0.95760000D+00,0.91180000D+00
     .,0.14514700D+02,0.129D+03,0.200D+01,0.95760000D+00,0.00000000D+00
     .,0.36743080D+03,0.129D+03,0.300D+01,0.95760000D+00,0.00000000D+00
     .,0.20593130D+03,0.129D+03,0.400D+01,0.95760000D+00,0.00000000D+00
     .,0.13637050D+03,0.129D+03,0.500D+01,0.95760000D+00,0.00000000D+00
     .,0.91102500D+02,0.129D+03,0.600D+01,0.95760000D+00,0.00000000D+00
     .,0.63270900D+02,0.129D+03,0.700D+01,0.95760000D+00,0.00000000D+00
     .,0.47723900D+02,0.129D+03,0.800D+01,0.95760000D+00,0.00000000D+00
     .,0.36074900D+02,0.129D+03,0.900D+01,0.95760000D+00,0.00000000D+00
     .,0.27728200D+02,0.129D+03,0.100D+02,0.95760000D+00,0.00000000D+00
     .,0.43871980D+03,0.129D+03,0.110D+02,0.95760000D+00,0.00000000D+00
     .,0.33000180D+03,0.129D+03,0.120D+02,0.95760000D+00,0.00000000D+00
     .,0.30146500D+03,0.129D+03,0.130D+02,0.95760000D+00,0.00000000D+00
     .,0.23475480D+03,0.129D+03,0.140D+02,0.95760000D+00,0.00000000D+00
     .,0.18132670D+03,0.129D+03,0.150D+02,0.95760000D+00,0.00000000D+00
     .,0.14964210D+03,0.129D+03,0.160D+02,0.95760000D+00,0.00000000D+00
     .,0.12163290D+03,0.129D+03,0.170D+02,0.95760000D+00,0.00000000D+00
     .,0.99139800D+02,0.129D+03,0.180D+02,0.95760000D+00,0.00000000D+00
     .,0.72200020D+03,0.129D+03,0.190D+02,0.95760000D+00,0.00000000D+00
     .,0.58537040D+03,0.129D+03,0.200D+02,0.95760000D+00,0.00000000D+00
     ./)
      pars(  35801:  35900)=(/
     . 0.48164720D+03,0.129D+03,0.210D+02,0.95760000D+00,0.00000000D+00
     .,0.46339240D+03,0.129D+03,0.220D+02,0.95760000D+00,0.00000000D+00
     .,0.42341930D+03,0.129D+03,0.230D+02,0.95760000D+00,0.00000000D+00
     .,0.33350140D+03,0.129D+03,0.240D+02,0.95760000D+00,0.00000000D+00
     .,0.36341710D+03,0.129D+03,0.250D+02,0.95760000D+00,0.00000000D+00
     .,0.28510180D+03,0.129D+03,0.260D+02,0.95760000D+00,0.00000000D+00
     .,0.30073560D+03,0.129D+03,0.270D+02,0.95760000D+00,0.00000000D+00
     .,0.31051330D+03,0.129D+03,0.280D+02,0.95760000D+00,0.00000000D+00
     .,0.23812070D+03,0.129D+03,0.290D+02,0.95760000D+00,0.00000000D+00
     .,0.24268930D+03,0.129D+03,0.300D+02,0.95760000D+00,0.00000000D+00
     .,0.28785200D+03,0.129D+03,0.310D+02,0.95760000D+00,0.00000000D+00
     .,0.25153860D+03,0.129D+03,0.320D+02,0.95760000D+00,0.00000000D+00
     .,0.21283460D+03,0.129D+03,0.330D+02,0.95760000D+00,0.00000000D+00
     .,0.19007710D+03,0.129D+03,0.340D+02,0.95760000D+00,0.00000000D+00
     .,0.16558600D+03,0.129D+03,0.350D+02,0.95760000D+00,0.00000000D+00
     .,0.14345250D+03,0.129D+03,0.360D+02,0.95760000D+00,0.00000000D+00
     .,0.80787860D+03,0.129D+03,0.370D+02,0.95760000D+00,0.00000000D+00
     .,0.69786230D+03,0.129D+03,0.380D+02,0.95760000D+00,0.00000000D+00
     .,0.60685850D+03,0.129D+03,0.390D+02,0.95760000D+00,0.00000000D+00
     .,0.54294990D+03,0.129D+03,0.400D+02,0.95760000D+00,0.00000000D+00
     ./)
      pars(  35901:  36000)=(/
     . 0.49363630D+03,0.129D+03,0.410D+02,0.95760000D+00,0.00000000D+00
     .,0.37922990D+03,0.129D+03,0.420D+02,0.95760000D+00,0.00000000D+00
     .,0.42391540D+03,0.129D+03,0.430D+02,0.95760000D+00,0.00000000D+00
     .,0.32125020D+03,0.129D+03,0.440D+02,0.95760000D+00,0.00000000D+00
     .,0.35112220D+03,0.129D+03,0.450D+02,0.95760000D+00,0.00000000D+00
     .,0.32504660D+03,0.129D+03,0.460D+02,0.95760000D+00,0.00000000D+00
     .,0.27146340D+03,0.129D+03,0.470D+02,0.95760000D+00,0.00000000D+00
     .,0.28581800D+03,0.129D+03,0.480D+02,0.95760000D+00,0.00000000D+00
     .,0.36066260D+03,0.129D+03,0.490D+02,0.95760000D+00,0.00000000D+00
     .,0.33100630D+03,0.129D+03,0.500D+02,0.95760000D+00,0.00000000D+00
     .,0.29285890D+03,0.129D+03,0.510D+02,0.95760000D+00,0.00000000D+00
     .,0.27055260D+03,0.129D+03,0.520D+02,0.95760000D+00,0.00000000D+00
     .,0.24354750D+03,0.129D+03,0.530D+02,0.95760000D+00,0.00000000D+00
     .,0.21809290D+03,0.129D+03,0.540D+02,0.95760000D+00,0.00000000D+00
     .,0.98359160D+03,0.129D+03,0.550D+02,0.95760000D+00,0.00000000D+00
     .,0.89168850D+03,0.129D+03,0.560D+02,0.95760000D+00,0.00000000D+00
     .,0.77770700D+03,0.129D+03,0.570D+02,0.95760000D+00,0.00000000D+00
     .,0.34643740D+03,0.129D+03,0.580D+02,0.95760000D+00,0.27991000D+01
     .,0.78837610D+03,0.129D+03,0.590D+02,0.95760000D+00,0.00000000D+00
     .,0.75615770D+03,0.129D+03,0.600D+02,0.95760000D+00,0.00000000D+00
     ./)
      pars(  36001:  36100)=(/
     . 0.73695090D+03,0.129D+03,0.610D+02,0.95760000D+00,0.00000000D+00
     .,0.71931180D+03,0.129D+03,0.620D+02,0.95760000D+00,0.00000000D+00
     .,0.70365920D+03,0.129D+03,0.630D+02,0.95760000D+00,0.00000000D+00
     .,0.54891190D+03,0.129D+03,0.640D+02,0.95760000D+00,0.00000000D+00
     .,0.62523960D+03,0.129D+03,0.650D+02,0.95760000D+00,0.00000000D+00
     .,0.60214160D+03,0.129D+03,0.660D+02,0.95760000D+00,0.00000000D+00
     .,0.63352600D+03,0.129D+03,0.670D+02,0.95760000D+00,0.00000000D+00
     .,0.61995970D+03,0.129D+03,0.680D+02,0.95760000D+00,0.00000000D+00
     .,0.60766330D+03,0.129D+03,0.690D+02,0.95760000D+00,0.00000000D+00
     .,0.60075720D+03,0.129D+03,0.700D+02,0.95760000D+00,0.00000000D+00
     .,0.50338410D+03,0.129D+03,0.710D+02,0.95760000D+00,0.00000000D+00
     .,0.49122120D+03,0.129D+03,0.720D+02,0.95760000D+00,0.00000000D+00
     .,0.44629550D+03,0.129D+03,0.730D+02,0.95760000D+00,0.00000000D+00
     .,0.37560840D+03,0.129D+03,0.740D+02,0.95760000D+00,0.00000000D+00
     .,0.38145840D+03,0.129D+03,0.750D+02,0.95760000D+00,0.00000000D+00
     .,0.34444290D+03,0.129D+03,0.760D+02,0.95760000D+00,0.00000000D+00
     .,0.31453250D+03,0.129D+03,0.770D+02,0.95760000D+00,0.00000000D+00
     .,0.26057850D+03,0.129D+03,0.780D+02,0.95760000D+00,0.00000000D+00
     .,0.24322040D+03,0.129D+03,0.790D+02,0.95760000D+00,0.00000000D+00
     .,0.24982310D+03,0.129D+03,0.800D+02,0.95760000D+00,0.00000000D+00
     ./)
      pars(  36101:  36200)=(/
     . 0.36971340D+03,0.129D+03,0.810D+02,0.95760000D+00,0.00000000D+00
     .,0.35938220D+03,0.129D+03,0.820D+02,0.95760000D+00,0.00000000D+00
     .,0.32809610D+03,0.129D+03,0.830D+02,0.95760000D+00,0.00000000D+00
     .,0.31175430D+03,0.129D+03,0.840D+02,0.95760000D+00,0.00000000D+00
     .,0.28648030D+03,0.129D+03,0.850D+02,0.95760000D+00,0.00000000D+00
     .,0.26159230D+03,0.129D+03,0.860D+02,0.95760000D+00,0.00000000D+00
     .,0.92344480D+03,0.129D+03,0.870D+02,0.95760000D+00,0.00000000D+00
     .,0.87838880D+03,0.129D+03,0.880D+02,0.95760000D+00,0.00000000D+00
     .,0.77137100D+03,0.129D+03,0.890D+02,0.95760000D+00,0.00000000D+00
     .,0.68798510D+03,0.129D+03,0.900D+02,0.95760000D+00,0.00000000D+00
     .,0.68576140D+03,0.129D+03,0.910D+02,0.95760000D+00,0.00000000D+00
     .,0.66390850D+03,0.129D+03,0.920D+02,0.95760000D+00,0.00000000D+00
     .,0.68694090D+03,0.129D+03,0.930D+02,0.95760000D+00,0.00000000D+00
     .,0.66465810D+03,0.129D+03,0.940D+02,0.95760000D+00,0.00000000D+00
     .,0.35987100D+02,0.129D+03,0.101D+03,0.95760000D+00,0.00000000D+00
     .,0.11943970D+03,0.129D+03,0.103D+03,0.95760000D+00,0.98650000D+00
     .,0.15184260D+03,0.129D+03,0.104D+03,0.95760000D+00,0.98080000D+00
     .,0.11434660D+03,0.129D+03,0.105D+03,0.95760000D+00,0.97060000D+00
     .,0.85473100D+02,0.129D+03,0.106D+03,0.95760000D+00,0.98680000D+00
     .,0.58990300D+02,0.129D+03,0.107D+03,0.95760000D+00,0.99440000D+00
     ./)
      pars(  36201:  36300)=(/
     . 0.42745800D+02,0.129D+03,0.108D+03,0.95760000D+00,0.99250000D+00
     .,0.29280800D+02,0.129D+03,0.109D+03,0.95760000D+00,0.99820000D+00
     .,0.17525310D+03,0.129D+03,0.111D+03,0.95760000D+00,0.96840000D+00
     .,0.27127550D+03,0.129D+03,0.112D+03,0.95760000D+00,0.96280000D+00
     .,0.27263840D+03,0.129D+03,0.113D+03,0.95760000D+00,0.96480000D+00
     .,0.21661540D+03,0.129D+03,0.114D+03,0.95760000D+00,0.95070000D+00
     .,0.17596280D+03,0.129D+03,0.115D+03,0.95760000D+00,0.99470000D+00
     .,0.14802440D+03,0.129D+03,0.116D+03,0.95760000D+00,0.99480000D+00
     .,0.12039270D+03,0.129D+03,0.117D+03,0.95760000D+00,0.99720000D+00
     .,0.24017910D+03,0.129D+03,0.119D+03,0.95760000D+00,0.97670000D+00
     .,0.46791560D+03,0.129D+03,0.120D+03,0.95760000D+00,0.98310000D+00
     .,0.23801280D+03,0.129D+03,0.121D+03,0.95760000D+00,0.18627000D+01
     .,0.22972260D+03,0.129D+03,0.122D+03,0.95760000D+00,0.18299000D+01
     .,0.22520200D+03,0.129D+03,0.123D+03,0.95760000D+00,0.19138000D+01
     .,0.22340090D+03,0.129D+03,0.124D+03,0.95760000D+00,0.18269000D+01
     .,0.20444800D+03,0.129D+03,0.125D+03,0.95760000D+00,0.16406000D+01
     .,0.18894450D+03,0.129D+03,0.126D+03,0.95760000D+00,0.16483000D+01
     .,0.18027190D+03,0.129D+03,0.127D+03,0.95760000D+00,0.17149000D+01
     .,0.17633580D+03,0.129D+03,0.128D+03,0.95760000D+00,0.17937000D+01
     .,0.17499690D+03,0.129D+03,0.129D+03,0.95760000D+00,0.95760000D+00
     ./)
      pars(  36301:  36400)=(/
     . 0.21213900D+02,0.130D+03,0.100D+01,0.19419000D+01,0.91180000D+00
     .,0.14118500D+02,0.130D+03,0.200D+01,0.19419000D+01,0.00000000D+00
     .,0.31788260D+03,0.130D+03,0.300D+01,0.19419000D+01,0.00000000D+00
     .,0.18730120D+03,0.130D+03,0.400D+01,0.19419000D+01,0.00000000D+00
     .,0.12724880D+03,0.130D+03,0.500D+01,0.19419000D+01,0.00000000D+00
     .,0.86470900D+02,0.130D+03,0.600D+01,0.19419000D+01,0.00000000D+00
     .,0.60733100D+02,0.130D+03,0.700D+01,0.19419000D+01,0.00000000D+00
     .,0.46135500D+02,0.130D+03,0.800D+01,0.19419000D+01,0.00000000D+00
     .,0.35062900D+02,0.130D+03,0.900D+01,0.19419000D+01,0.00000000D+00
     .,0.27052400D+02,0.130D+03,0.100D+02,0.19419000D+01,0.00000000D+00
     .,0.38065680D+03,0.130D+03,0.110D+02,0.19419000D+01,0.00000000D+00
     .,0.29754400D+03,0.130D+03,0.120D+02,0.19419000D+01,0.00000000D+00
     .,0.27562990D+03,0.130D+03,0.130D+02,0.19419000D+01,0.00000000D+00
     .,0.21852820D+03,0.130D+03,0.140D+02,0.19419000D+01,0.00000000D+00
     .,0.17118890D+03,0.130D+03,0.150D+02,0.19419000D+01,0.00000000D+00
     .,0.14245910D+03,0.130D+03,0.160D+02,0.19419000D+01,0.00000000D+00
     .,0.11666370D+03,0.130D+03,0.170D+02,0.19419000D+01,0.00000000D+00
     .,0.95658600D+02,0.130D+03,0.180D+02,0.19419000D+01,0.00000000D+00
     .,0.62148410D+03,0.130D+03,0.190D+02,0.19419000D+01,0.00000000D+00
     .,0.51951820D+03,0.130D+03,0.200D+02,0.19419000D+01,0.00000000D+00
     ./)
      pars(  36401:  36500)=(/
     . 0.43036750D+03,0.130D+03,0.210D+02,0.19419000D+01,0.00000000D+00
     .,0.41653280D+03,0.130D+03,0.220D+02,0.19419000D+01,0.00000000D+00
     .,0.38196290D+03,0.130D+03,0.230D+02,0.19419000D+01,0.00000000D+00
     .,0.30093230D+03,0.130D+03,0.240D+02,0.19419000D+01,0.00000000D+00
     .,0.32951960D+03,0.130D+03,0.250D+02,0.19419000D+01,0.00000000D+00
     .,0.25872900D+03,0.130D+03,0.260D+02,0.19419000D+01,0.00000000D+00
     .,0.27503810D+03,0.130D+03,0.270D+02,0.19419000D+01,0.00000000D+00
     .,0.28296560D+03,0.130D+03,0.280D+02,0.19419000D+01,0.00000000D+00
     .,0.21695700D+03,0.130D+03,0.290D+02,0.19419000D+01,0.00000000D+00
     .,0.22366220D+03,0.130D+03,0.300D+02,0.19419000D+01,0.00000000D+00
     .,0.26447240D+03,0.130D+03,0.310D+02,0.19419000D+01,0.00000000D+00
     .,0.23434070D+03,0.130D+03,0.320D+02,0.19419000D+01,0.00000000D+00
     .,0.20064080D+03,0.130D+03,0.330D+02,0.19419000D+01,0.00000000D+00
     .,0.18041890D+03,0.130D+03,0.340D+02,0.19419000D+01,0.00000000D+00
     .,0.15822130D+03,0.130D+03,0.350D+02,0.19419000D+01,0.00000000D+00
     .,0.13785900D+03,0.130D+03,0.360D+02,0.19419000D+01,0.00000000D+00
     .,0.69720120D+03,0.130D+03,0.370D+02,0.19419000D+01,0.00000000D+00
     .,0.61859280D+03,0.130D+03,0.380D+02,0.19419000D+01,0.00000000D+00
     .,0.54431400D+03,0.130D+03,0.390D+02,0.19419000D+01,0.00000000D+00
     .,0.49050970D+03,0.130D+03,0.400D+02,0.19419000D+01,0.00000000D+00
     ./)
      pars(  36501:  36600)=(/
     . 0.44803280D+03,0.130D+03,0.410D+02,0.19419000D+01,0.00000000D+00
     .,0.34683870D+03,0.130D+03,0.420D+02,0.19419000D+01,0.00000000D+00
     .,0.38662560D+03,0.130D+03,0.430D+02,0.19419000D+01,0.00000000D+00
     .,0.29543070D+03,0.130D+03,0.440D+02,0.19419000D+01,0.00000000D+00
     .,0.32298310D+03,0.130D+03,0.450D+02,0.19419000D+01,0.00000000D+00
     .,0.29983170D+03,0.130D+03,0.460D+02,0.19419000D+01,0.00000000D+00
     .,0.24983480D+03,0.130D+03,0.470D+02,0.19419000D+01,0.00000000D+00
     .,0.26457160D+03,0.130D+03,0.480D+02,0.19419000D+01,0.00000000D+00
     .,0.33095850D+03,0.130D+03,0.490D+02,0.19419000D+01,0.00000000D+00
     .,0.30726350D+03,0.130D+03,0.500D+02,0.19419000D+01,0.00000000D+00
     .,0.27478160D+03,0.130D+03,0.510D+02,0.19419000D+01,0.00000000D+00
     .,0.25542890D+03,0.130D+03,0.520D+02,0.19419000D+01,0.00000000D+00
     .,0.23140060D+03,0.130D+03,0.530D+02,0.19419000D+01,0.00000000D+00
     .,0.20840550D+03,0.130D+03,0.540D+02,0.19419000D+01,0.00000000D+00
     .,0.84934460D+03,0.130D+03,0.550D+02,0.19419000D+01,0.00000000D+00
     .,0.78727710D+03,0.130D+03,0.560D+02,0.19419000D+01,0.00000000D+00
     .,0.69479590D+03,0.130D+03,0.570D+02,0.19419000D+01,0.00000000D+00
     .,0.32433650D+03,0.130D+03,0.580D+02,0.19419000D+01,0.27991000D+01
     .,0.69872430D+03,0.130D+03,0.590D+02,0.19419000D+01,0.00000000D+00
     .,0.67147320D+03,0.130D+03,0.600D+02,0.19419000D+01,0.00000000D+00
     ./)
      pars(  36601:  36700)=(/
     . 0.65477760D+03,0.130D+03,0.610D+02,0.19419000D+01,0.00000000D+00
     .,0.63940750D+03,0.130D+03,0.620D+02,0.19419000D+01,0.00000000D+00
     .,0.62578200D+03,0.130D+03,0.630D+02,0.19419000D+01,0.00000000D+00
     .,0.49442430D+03,0.130D+03,0.640D+02,0.19419000D+01,0.00000000D+00
     .,0.55258030D+03,0.130D+03,0.650D+02,0.19419000D+01,0.00000000D+00
     .,0.53339380D+03,0.130D+03,0.660D+02,0.19419000D+01,0.00000000D+00
     .,0.56508580D+03,0.130D+03,0.670D+02,0.19419000D+01,0.00000000D+00
     .,0.55316870D+03,0.130D+03,0.680D+02,0.19419000D+01,0.00000000D+00
     .,0.54245160D+03,0.130D+03,0.690D+02,0.19419000D+01,0.00000000D+00
     .,0.53600770D+03,0.130D+03,0.700D+02,0.19419000D+01,0.00000000D+00
     .,0.45307620D+03,0.130D+03,0.710D+02,0.19419000D+01,0.00000000D+00
     .,0.44744320D+03,0.130D+03,0.720D+02,0.19419000D+01,0.00000000D+00
     .,0.40933690D+03,0.130D+03,0.730D+02,0.19419000D+01,0.00000000D+00
     .,0.34623600D+03,0.130D+03,0.740D+02,0.19419000D+01,0.00000000D+00
     .,0.35258880D+03,0.130D+03,0.750D+02,0.19419000D+01,0.00000000D+00
     .,0.32019590D+03,0.130D+03,0.760D+02,0.19419000D+01,0.00000000D+00
     .,0.29371130D+03,0.130D+03,0.770D+02,0.19419000D+01,0.00000000D+00
     .,0.24437210D+03,0.130D+03,0.780D+02,0.19419000D+01,0.00000000D+00
     .,0.22847850D+03,0.130D+03,0.790D+02,0.19419000D+01,0.00000000D+00
     .,0.23524690D+03,0.130D+03,0.800D+02,0.19419000D+01,0.00000000D+00
     ./)
      pars(  36701:  36800)=(/
     . 0.34006610D+03,0.130D+03,0.810D+02,0.19419000D+01,0.00000000D+00
     .,0.33355770D+03,0.130D+03,0.820D+02,0.19419000D+01,0.00000000D+00
     .,0.30748710D+03,0.130D+03,0.830D+02,0.19419000D+01,0.00000000D+00
     .,0.29376220D+03,0.130D+03,0.840D+02,0.19419000D+01,0.00000000D+00
     .,0.27163520D+03,0.130D+03,0.850D+02,0.19419000D+01,0.00000000D+00
     .,0.24938090D+03,0.130D+03,0.860D+02,0.19419000D+01,0.00000000D+00
     .,0.80499020D+03,0.130D+03,0.870D+02,0.19419000D+01,0.00000000D+00
     .,0.78026750D+03,0.130D+03,0.880D+02,0.19419000D+01,0.00000000D+00
     .,0.69268280D+03,0.130D+03,0.890D+02,0.19419000D+01,0.00000000D+00
     .,0.62529330D+03,0.130D+03,0.900D+02,0.19419000D+01,0.00000000D+00
     .,0.61943840D+03,0.130D+03,0.910D+02,0.19419000D+01,0.00000000D+00
     .,0.59985590D+03,0.130D+03,0.920D+02,0.19419000D+01,0.00000000D+00
     .,0.61596070D+03,0.130D+03,0.930D+02,0.19419000D+01,0.00000000D+00
     .,0.59681690D+03,0.130D+03,0.940D+02,0.19419000D+01,0.00000000D+00
     .,0.34031200D+02,0.130D+03,0.101D+03,0.19419000D+01,0.00000000D+00
     .,0.10901270D+03,0.130D+03,0.103D+03,0.19419000D+01,0.98650000D+00
     .,0.13924200D+03,0.130D+03,0.104D+03,0.19419000D+01,0.98080000D+00
     .,0.10720260D+03,0.130D+03,0.105D+03,0.19419000D+01,0.97060000D+00
     .,0.81068600D+02,0.130D+03,0.106D+03,0.19419000D+01,0.98680000D+00
     .,0.56614000D+02,0.130D+03,0.107D+03,0.19419000D+01,0.99440000D+00
     ./)
      pars(  36801:  36900)=(/
     . 0.41386200D+02,0.130D+03,0.108D+03,0.19419000D+01,0.99250000D+00
     .,0.28629100D+02,0.130D+03,0.109D+03,0.19419000D+01,0.99820000D+00
     .,0.15912280D+03,0.130D+03,0.111D+03,0.19419000D+01,0.96840000D+00
     .,0.24580540D+03,0.130D+03,0.112D+03,0.19419000D+01,0.96280000D+00
     .,0.25007300D+03,0.130D+03,0.113D+03,0.19419000D+01,0.96480000D+00
     .,0.20213060D+03,0.130D+03,0.114D+03,0.19419000D+01,0.95070000D+00
     .,0.16618870D+03,0.130D+03,0.115D+03,0.19419000D+01,0.99470000D+00
     .,0.14088040D+03,0.130D+03,0.116D+03,0.19419000D+01,0.99480000D+00
     .,0.11545010D+03,0.130D+03,0.117D+03,0.19419000D+01,0.99720000D+00
     .,0.21998520D+03,0.130D+03,0.119D+03,0.19419000D+01,0.97670000D+00
     .,0.41556380D+03,0.130D+03,0.120D+03,0.19419000D+01,0.98310000D+00
     .,0.22124670D+03,0.130D+03,0.121D+03,0.19419000D+01,0.18627000D+01
     .,0.21361910D+03,0.130D+03,0.122D+03,0.19419000D+01,0.18299000D+01
     .,0.20934280D+03,0.130D+03,0.123D+03,0.19419000D+01,0.19138000D+01
     .,0.20729260D+03,0.130D+03,0.124D+03,0.19419000D+01,0.18269000D+01
     .,0.19131250D+03,0.130D+03,0.125D+03,0.19419000D+01,0.16406000D+01
     .,0.17720340D+03,0.130D+03,0.126D+03,0.19419000D+01,0.16483000D+01
     .,0.16904610D+03,0.130D+03,0.127D+03,0.19419000D+01,0.17149000D+01
     .,0.16523610D+03,0.130D+03,0.128D+03,0.19419000D+01,0.17937000D+01
     .,0.16292360D+03,0.130D+03,0.129D+03,0.19419000D+01,0.95760000D+00
     ./)
      pars(  36901:  37000)=(/
     . 0.15345280D+03,0.130D+03,0.130D+03,0.19419000D+01,0.19419000D+01
     .,0.33251500D+02,0.131D+03,0.100D+01,0.96010000D+00,0.91180000D+00
     .,0.20987400D+02,0.131D+03,0.200D+01,0.96010000D+00,0.00000000D+00
     .,0.60281040D+03,0.131D+03,0.300D+01,0.96010000D+00,0.00000000D+00
     .,0.32666720D+03,0.131D+03,0.400D+01,0.96010000D+00,0.00000000D+00
     .,0.21127560D+03,0.131D+03,0.500D+01,0.96010000D+00,0.00000000D+00
     .,0.13814970D+03,0.131D+03,0.600D+01,0.96010000D+00,0.00000000D+00
     .,0.94132900D+02,0.131D+03,0.700D+01,0.96010000D+00,0.00000000D+00
     .,0.69903900D+02,0.131D+03,0.800D+01,0.96010000D+00,0.00000000D+00
     .,0.52047600D+02,0.131D+03,0.900D+01,0.96010000D+00,0.00000000D+00
     .,0.39458700D+02,0.131D+03,0.100D+02,0.96010000D+00,0.00000000D+00
     .,0.71775310D+03,0.131D+03,0.110D+02,0.96010000D+00,0.00000000D+00
     .,0.52642360D+03,0.131D+03,0.120D+02,0.96010000D+00,0.00000000D+00
     .,0.47573180D+03,0.131D+03,0.130D+02,0.96010000D+00,0.00000000D+00
     .,0.36461100D+03,0.131D+03,0.140D+02,0.96010000D+00,0.00000000D+00
     .,0.27735230D+03,0.131D+03,0.150D+02,0.96010000D+00,0.00000000D+00
     .,0.22627960D+03,0.131D+03,0.160D+02,0.96010000D+00,0.00000000D+00
     .,0.18171780D+03,0.131D+03,0.170D+02,0.96010000D+00,0.00000000D+00
     .,0.14638550D+03,0.131D+03,0.180D+02,0.96010000D+00,0.00000000D+00
     .,0.11848327D+04,0.131D+03,0.190D+02,0.96010000D+00,0.00000000D+00
     ./)
      pars(  37001:  37100)=(/
     . 0.94258280D+03,0.131D+03,0.200D+02,0.96010000D+00,0.00000000D+00
     .,0.77197630D+03,0.131D+03,0.210D+02,0.96010000D+00,0.00000000D+00
     .,0.73902810D+03,0.131D+03,0.220D+02,0.96010000D+00,0.00000000D+00
     .,0.67328230D+03,0.131D+03,0.230D+02,0.96010000D+00,0.00000000D+00
     .,0.52900060D+03,0.131D+03,0.240D+02,0.96010000D+00,0.00000000D+00
     .,0.57532290D+03,0.131D+03,0.250D+02,0.96010000D+00,0.00000000D+00
     .,0.44990400D+03,0.131D+03,0.260D+02,0.96010000D+00,0.00000000D+00
     .,0.47264010D+03,0.131D+03,0.270D+02,0.96010000D+00,0.00000000D+00
     .,0.48954990D+03,0.131D+03,0.280D+02,0.96010000D+00,0.00000000D+00
     .,0.37425370D+03,0.131D+03,0.290D+02,0.96010000D+00,0.00000000D+00
     .,0.37873290D+03,0.131D+03,0.300D+02,0.96010000D+00,0.00000000D+00
     .,0.45140030D+03,0.131D+03,0.310D+02,0.96010000D+00,0.00000000D+00
     .,0.38973700D+03,0.131D+03,0.320D+02,0.96010000D+00,0.00000000D+00
     .,0.32572680D+03,0.131D+03,0.330D+02,0.96010000D+00,0.00000000D+00
     .,0.28839340D+03,0.131D+03,0.340D+02,0.96010000D+00,0.00000000D+00
     .,0.24882110D+03,0.131D+03,0.350D+02,0.96010000D+00,0.00000000D+00
     .,0.21348210D+03,0.131D+03,0.360D+02,0.96010000D+00,0.00000000D+00
     .,0.13227322D+04,0.131D+03,0.370D+02,0.96010000D+00,0.00000000D+00
     .,0.11239167D+04,0.131D+03,0.380D+02,0.96010000D+00,0.00000000D+00
     .,0.96907770D+03,0.131D+03,0.390D+02,0.96010000D+00,0.00000000D+00
     ./)
      pars(  37101:  37200)=(/
     . 0.86203630D+03,0.131D+03,0.400D+02,0.96010000D+00,0.00000000D+00
     .,0.78048050D+03,0.131D+03,0.410D+02,0.96010000D+00,0.00000000D+00
     .,0.59457650D+03,0.131D+03,0.420D+02,0.96010000D+00,0.00000000D+00
     .,0.66676010D+03,0.131D+03,0.430D+02,0.96010000D+00,0.00000000D+00
     .,0.50051620D+03,0.131D+03,0.440D+02,0.96010000D+00,0.00000000D+00
     .,0.54783460D+03,0.131D+03,0.450D+02,0.96010000D+00,0.00000000D+00
     .,0.50566980D+03,0.131D+03,0.460D+02,0.96010000D+00,0.00000000D+00
     .,0.42174200D+03,0.131D+03,0.470D+02,0.96010000D+00,0.00000000D+00
     .,0.44286390D+03,0.131D+03,0.480D+02,0.96010000D+00,0.00000000D+00
     .,0.56415640D+03,0.131D+03,0.490D+02,0.96010000D+00,0.00000000D+00
     .,0.51303150D+03,0.131D+03,0.500D+02,0.96010000D+00,0.00000000D+00
     .,0.44930060D+03,0.131D+03,0.510D+02,0.96010000D+00,0.00000000D+00
     .,0.41227930D+03,0.131D+03,0.520D+02,0.96010000D+00,0.00000000D+00
     .,0.36821830D+03,0.131D+03,0.530D+02,0.96010000D+00,0.00000000D+00
     .,0.32708380D+03,0.131D+03,0.540D+02,0.96010000D+00,0.00000000D+00
     .,0.16098951D+04,0.131D+03,0.550D+02,0.96010000D+00,0.00000000D+00
     .,0.14393610D+04,0.131D+03,0.560D+02,0.96010000D+00,0.00000000D+00
     .,0.12450382D+04,0.131D+03,0.570D+02,0.96010000D+00,0.00000000D+00
     .,0.53206830D+03,0.131D+03,0.580D+02,0.96010000D+00,0.27991000D+01
     .,0.12682581D+04,0.131D+03,0.590D+02,0.96010000D+00,0.00000000D+00
     ./)
      pars(  37201:  37300)=(/
     . 0.12147867D+04,0.131D+03,0.600D+02,0.96010000D+00,0.00000000D+00
     .,0.11834931D+04,0.131D+03,0.610D+02,0.96010000D+00,0.00000000D+00
     .,0.11548172D+04,0.131D+03,0.620D+02,0.96010000D+00,0.00000000D+00
     .,0.11293623D+04,0.131D+03,0.630D+02,0.96010000D+00,0.00000000D+00
     .,0.87169660D+03,0.131D+03,0.640D+02,0.96010000D+00,0.00000000D+00
     .,0.10061273D+04,0.131D+03,0.650D+02,0.96010000D+00,0.00000000D+00
     .,0.96741930D+03,0.131D+03,0.660D+02,0.96010000D+00,0.00000000D+00
     .,0.10147443D+04,0.131D+03,0.670D+02,0.96010000D+00,0.00000000D+00
     .,0.99283980D+03,0.131D+03,0.680D+02,0.96010000D+00,0.00000000D+00
     .,0.97287530D+03,0.131D+03,0.690D+02,0.96010000D+00,0.00000000D+00
     .,0.96226810D+03,0.131D+03,0.700D+02,0.96010000D+00,0.00000000D+00
     .,0.80056070D+03,0.131D+03,0.710D+02,0.96010000D+00,0.00000000D+00
     .,0.77479490D+03,0.131D+03,0.720D+02,0.96010000D+00,0.00000000D+00
     .,0.69954820D+03,0.131D+03,0.730D+02,0.96010000D+00,0.00000000D+00
     .,0.58500690D+03,0.131D+03,0.740D+02,0.96010000D+00,0.00000000D+00
     .,0.59290660D+03,0.131D+03,0.750D+02,0.96010000D+00,0.00000000D+00
     .,0.53219500D+03,0.131D+03,0.760D+02,0.96010000D+00,0.00000000D+00
     .,0.48350900D+03,0.131D+03,0.770D+02,0.96010000D+00,0.00000000D+00
     .,0.39773910D+03,0.131D+03,0.780D+02,0.96010000D+00,0.00000000D+00
     .,0.37015170D+03,0.131D+03,0.790D+02,0.96010000D+00,0.00000000D+00
     ./)
      pars(  37301:  37400)=(/
     . 0.37970920D+03,0.131D+03,0.800D+02,0.96010000D+00,0.00000000D+00
     .,0.57562810D+03,0.131D+03,0.810D+02,0.96010000D+00,0.00000000D+00
     .,0.55583070D+03,0.131D+03,0.820D+02,0.96010000D+00,0.00000000D+00
     .,0.50298850D+03,0.131D+03,0.830D+02,0.96010000D+00,0.00000000D+00
     .,0.47531340D+03,0.131D+03,0.840D+02,0.96010000D+00,0.00000000D+00
     .,0.43366250D+03,0.131D+03,0.850D+02,0.96010000D+00,0.00000000D+00
     .,0.39318500D+03,0.131D+03,0.860D+02,0.96010000D+00,0.00000000D+00
     .,0.15014541D+04,0.131D+03,0.870D+02,0.96010000D+00,0.00000000D+00
     .,0.14115217D+04,0.131D+03,0.880D+02,0.96010000D+00,0.00000000D+00
     .,0.12297307D+04,0.131D+03,0.890D+02,0.96010000D+00,0.00000000D+00
     .,0.10857394D+04,0.131D+03,0.900D+02,0.96010000D+00,0.00000000D+00
     .,0.10865380D+04,0.131D+03,0.910D+02,0.96010000D+00,0.00000000D+00
     .,0.10514942D+04,0.131D+03,0.920D+02,0.96010000D+00,0.00000000D+00
     .,0.10941031D+04,0.131D+03,0.930D+02,0.96010000D+00,0.00000000D+00
     .,0.10575095D+04,0.131D+03,0.940D+02,0.96010000D+00,0.00000000D+00
     .,0.54991900D+02,0.131D+03,0.101D+03,0.96010000D+00,0.00000000D+00
     .,0.18876220D+03,0.131D+03,0.103D+03,0.96010000D+00,0.98650000D+00
     .,0.23890680D+03,0.131D+03,0.104D+03,0.96010000D+00,0.98080000D+00
     .,0.17609760D+03,0.131D+03,0.105D+03,0.96010000D+00,0.97060000D+00
     .,0.12956700D+03,0.131D+03,0.106D+03,0.96010000D+00,0.98680000D+00
     ./)
      pars(  37401:  37500)=(/
     . 0.87664500D+02,0.131D+03,0.107D+03,0.96010000D+00,0.99440000D+00
     .,0.62352200D+02,0.131D+03,0.108D+03,0.96010000D+00,0.99250000D+00
     .,0.41590900D+02,0.131D+03,0.109D+03,0.96010000D+00,0.99820000D+00
     .,0.27751430D+03,0.131D+03,0.111D+03,0.96010000D+00,0.96840000D+00
     .,0.43082970D+03,0.131D+03,0.112D+03,0.96010000D+00,0.96280000D+00
     .,0.42895210D+03,0.131D+03,0.113D+03,0.96010000D+00,0.96480000D+00
     .,0.33552060D+03,0.131D+03,0.114D+03,0.96010000D+00,0.95070000D+00
     .,0.26893700D+03,0.131D+03,0.115D+03,0.96010000D+00,0.99470000D+00
     .,0.22384530D+03,0.131D+03,0.116D+03,0.96010000D+00,0.99480000D+00
     .,0.17986830D+03,0.131D+03,0.117D+03,0.96010000D+00,0.99720000D+00
     .,0.37584950D+03,0.131D+03,0.119D+03,0.96010000D+00,0.97670000D+00
     .,0.75181440D+03,0.131D+03,0.120D+03,0.96010000D+00,0.98310000D+00
     .,0.36854170D+03,0.131D+03,0.121D+03,0.96010000D+00,0.18627000D+01
     .,0.35534980D+03,0.131D+03,0.122D+03,0.96010000D+00,0.18299000D+01
     .,0.34832930D+03,0.131D+03,0.123D+03,0.96010000D+00,0.19138000D+01
     .,0.34597960D+03,0.131D+03,0.124D+03,0.96010000D+00,0.18269000D+01
     .,0.31442440D+03,0.131D+03,0.125D+03,0.96010000D+00,0.16406000D+01
     .,0.28975880D+03,0.131D+03,0.126D+03,0.96010000D+00,0.16483000D+01
     .,0.27631300D+03,0.131D+03,0.127D+03,0.96010000D+00,0.17149000D+01
     .,0.27040230D+03,0.131D+03,0.128D+03,0.96010000D+00,0.17937000D+01
     ./)
      pars(  37501:  37600)=(/
     . 0.26973440D+03,0.131D+03,0.129D+03,0.96010000D+00,0.95760000D+00
     .,0.24870030D+03,0.131D+03,0.130D+03,0.96010000D+00,0.19419000D+01
     .,0.42164970D+03,0.131D+03,0.131D+03,0.96010000D+00,0.96010000D+00
     .,0.29952700D+02,0.132D+03,0.100D+01,0.94340000D+00,0.91180000D+00
     .,0.19376100D+02,0.132D+03,0.200D+01,0.94340000D+00,0.00000000D+00
     .,0.48401650D+03,0.132D+03,0.300D+01,0.94340000D+00,0.00000000D+00
     .,0.27657170D+03,0.132D+03,0.400D+01,0.94340000D+00,0.00000000D+00
     .,0.18420550D+03,0.132D+03,0.500D+01,0.94340000D+00,0.00000000D+00
     .,0.12305740D+03,0.132D+03,0.600D+01,0.94340000D+00,0.00000000D+00
     .,0.85162100D+02,0.132D+03,0.700D+01,0.94340000D+00,0.00000000D+00
     .,0.63924500D+02,0.132D+03,0.800D+01,0.94340000D+00,0.00000000D+00
     .,0.48023300D+02,0.132D+03,0.900D+01,0.94340000D+00,0.00000000D+00
     .,0.36661500D+02,0.132D+03,0.100D+02,0.94340000D+00,0.00000000D+00
     .,0.57808820D+03,0.132D+03,0.110D+02,0.94340000D+00,0.00000000D+00
     .,0.44152390D+03,0.132D+03,0.120D+02,0.94340000D+00,0.00000000D+00
     .,0.40516110D+03,0.132D+03,0.130D+02,0.94340000D+00,0.00000000D+00
     .,0.31697000D+03,0.132D+03,0.140D+02,0.94340000D+00,0.00000000D+00
     .,0.24526830D+03,0.132D+03,0.150D+02,0.94340000D+00,0.00000000D+00
     .,0.20228630D+03,0.132D+03,0.160D+02,0.94340000D+00,0.00000000D+00
     .,0.16412660D+03,0.132D+03,0.170D+02,0.94340000D+00,0.00000000D+00
     ./)
      pars(  37601:  37700)=(/
     . 0.13337940D+03,0.132D+03,0.180D+02,0.94340000D+00,0.00000000D+00
     .,0.94657560D+03,0.132D+03,0.190D+02,0.94340000D+00,0.00000000D+00
     .,0.77753610D+03,0.132D+03,0.200D+02,0.94340000D+00,0.00000000D+00
     .,0.64140640D+03,0.132D+03,0.210D+02,0.94340000D+00,0.00000000D+00
     .,0.61804010D+03,0.132D+03,0.220D+02,0.94340000D+00,0.00000000D+00
     .,0.56526390D+03,0.132D+03,0.230D+02,0.94340000D+00,0.00000000D+00
     .,0.44441960D+03,0.132D+03,0.240D+02,0.94340000D+00,0.00000000D+00
     .,0.48576480D+03,0.132D+03,0.250D+02,0.94340000D+00,0.00000000D+00
     .,0.38038530D+03,0.132D+03,0.260D+02,0.94340000D+00,0.00000000D+00
     .,0.40291460D+03,0.132D+03,0.270D+02,0.94340000D+00,0.00000000D+00
     .,0.41564390D+03,0.132D+03,0.280D+02,0.94340000D+00,0.00000000D+00
     .,0.31786240D+03,0.132D+03,0.290D+02,0.94340000D+00,0.00000000D+00
     .,0.32569280D+03,0.132D+03,0.300D+02,0.94340000D+00,0.00000000D+00
     .,0.38671290D+03,0.132D+03,0.310D+02,0.94340000D+00,0.00000000D+00
     .,0.33924040D+03,0.132D+03,0.320D+02,0.94340000D+00,0.00000000D+00
     .,0.28758470D+03,0.132D+03,0.330D+02,0.94340000D+00,0.00000000D+00
     .,0.25685170D+03,0.132D+03,0.340D+02,0.94340000D+00,0.00000000D+00
     .,0.22357440D+03,0.132D+03,0.350D+02,0.94340000D+00,0.00000000D+00
     .,0.19336340D+03,0.132D+03,0.360D+02,0.94340000D+00,0.00000000D+00
     .,0.10596044D+04,0.132D+03,0.370D+02,0.94340000D+00,0.00000000D+00
     ./)
      pars(  37701:  37800)=(/
     . 0.92600250D+03,0.132D+03,0.380D+02,0.94340000D+00,0.00000000D+00
     .,0.80859910D+03,0.132D+03,0.390D+02,0.94340000D+00,0.00000000D+00
     .,0.72498050D+03,0.132D+03,0.400D+02,0.94340000D+00,0.00000000D+00
     .,0.65980390D+03,0.132D+03,0.410D+02,0.94340000D+00,0.00000000D+00
     .,0.50715030D+03,0.132D+03,0.420D+02,0.94340000D+00,0.00000000D+00
     .,0.56682940D+03,0.132D+03,0.430D+02,0.94340000D+00,0.00000000D+00
     .,0.42969960D+03,0.132D+03,0.440D+02,0.94340000D+00,0.00000000D+00
     .,0.47030780D+03,0.132D+03,0.450D+02,0.94340000D+00,0.00000000D+00
     .,0.43552680D+03,0.132D+03,0.460D+02,0.94340000D+00,0.00000000D+00
     .,0.36250760D+03,0.132D+03,0.470D+02,0.94340000D+00,0.00000000D+00
     .,0.38302040D+03,0.132D+03,0.480D+02,0.94340000D+00,0.00000000D+00
     .,0.48291150D+03,0.132D+03,0.490D+02,0.94340000D+00,0.00000000D+00
     .,0.44490480D+03,0.132D+03,0.500D+02,0.94340000D+00,0.00000000D+00
     .,0.39458220D+03,0.132D+03,0.510D+02,0.94340000D+00,0.00000000D+00
     .,0.36482430D+03,0.132D+03,0.520D+02,0.94340000D+00,0.00000000D+00
     .,0.32847500D+03,0.132D+03,0.530D+02,0.94340000D+00,0.00000000D+00
     .,0.29399660D+03,0.132D+03,0.540D+02,0.94340000D+00,0.00000000D+00
     .,0.12903249D+04,0.132D+03,0.550D+02,0.94340000D+00,0.00000000D+00
     .,0.11809729D+04,0.132D+03,0.560D+02,0.94340000D+00,0.00000000D+00
     .,0.10344900D+04,0.132D+03,0.570D+02,0.94340000D+00,0.00000000D+00
     ./)
      pars(  37801:  37900)=(/
     . 0.46618800D+03,0.132D+03,0.580D+02,0.94340000D+00,0.27991000D+01
     .,0.10448624D+04,0.132D+03,0.590D+02,0.94340000D+00,0.00000000D+00
     .,0.10029138D+04,0.132D+03,0.600D+02,0.94340000D+00,0.00000000D+00
     .,0.97765230D+03,0.132D+03,0.610D+02,0.94340000D+00,0.00000000D+00
     .,0.95444310D+03,0.132D+03,0.620D+02,0.94340000D+00,0.00000000D+00
     .,0.93386150D+03,0.132D+03,0.630D+02,0.94340000D+00,0.00000000D+00
     .,0.73092270D+03,0.132D+03,0.640D+02,0.94340000D+00,0.00000000D+00
     .,0.82651040D+03,0.132D+03,0.650D+02,0.94340000D+00,0.00000000D+00
     .,0.79662990D+03,0.132D+03,0.660D+02,0.94340000D+00,0.00000000D+00
     .,0.84175260D+03,0.132D+03,0.670D+02,0.94340000D+00,0.00000000D+00
     .,0.82386870D+03,0.132D+03,0.680D+02,0.94340000D+00,0.00000000D+00
     .,0.80770310D+03,0.132D+03,0.690D+02,0.94340000D+00,0.00000000D+00
     .,0.79843800D+03,0.132D+03,0.700D+02,0.94340000D+00,0.00000000D+00
     .,0.67060970D+03,0.132D+03,0.710D+02,0.94340000D+00,0.00000000D+00
     .,0.65755650D+03,0.132D+03,0.720D+02,0.94340000D+00,0.00000000D+00
     .,0.59836780D+03,0.132D+03,0.730D+02,0.94340000D+00,0.00000000D+00
     .,0.50344310D+03,0.132D+03,0.740D+02,0.94340000D+00,0.00000000D+00
     .,0.51179750D+03,0.132D+03,0.750D+02,0.94340000D+00,0.00000000D+00
     .,0.46250420D+03,0.132D+03,0.760D+02,0.94340000D+00,0.00000000D+00
     .,0.42248760D+03,0.132D+03,0.770D+02,0.94340000D+00,0.00000000D+00
     ./)
      pars(  37901:  38000)=(/
     . 0.34951600D+03,0.132D+03,0.780D+02,0.94340000D+00,0.00000000D+00
     .,0.32600720D+03,0.132D+03,0.790D+02,0.94340000D+00,0.00000000D+00
     .,0.33530450D+03,0.132D+03,0.800D+02,0.94340000D+00,0.00000000D+00
     .,0.49429340D+03,0.132D+03,0.810D+02,0.94340000D+00,0.00000000D+00
     .,0.48215480D+03,0.132D+03,0.820D+02,0.94340000D+00,0.00000000D+00
     .,0.44128680D+03,0.132D+03,0.830D+02,0.94340000D+00,0.00000000D+00
     .,0.41974070D+03,0.132D+03,0.840D+02,0.94340000D+00,0.00000000D+00
     .,0.38594190D+03,0.132D+03,0.850D+02,0.94340000D+00,0.00000000D+00
     .,0.35237490D+03,0.132D+03,0.860D+02,0.94340000D+00,0.00000000D+00
     .,0.12154454D+04,0.132D+03,0.870D+02,0.94340000D+00,0.00000000D+00
     .,0.11657107D+04,0.132D+03,0.880D+02,0.94340000D+00,0.00000000D+00
     .,0.10274859D+04,0.132D+03,0.890D+02,0.94340000D+00,0.00000000D+00
     .,0.91935170D+03,0.132D+03,0.900D+02,0.94340000D+00,0.00000000D+00
     .,0.91390230D+03,0.132D+03,0.910D+02,0.94340000D+00,0.00000000D+00
     .,0.88471530D+03,0.132D+03,0.920D+02,0.94340000D+00,0.00000000D+00
     .,0.91297050D+03,0.132D+03,0.930D+02,0.94340000D+00,0.00000000D+00
     .,0.88377800D+03,0.132D+03,0.940D+02,0.94340000D+00,0.00000000D+00
     .,0.48724700D+02,0.132D+03,0.101D+03,0.94340000D+00,0.00000000D+00
     .,0.16045020D+03,0.132D+03,0.103D+03,0.94340000D+00,0.98650000D+00
     .,0.20416900D+03,0.132D+03,0.104D+03,0.94340000D+00,0.98080000D+00
     ./)
      pars(  38001:  38100)=(/
     . 0.15444090D+03,0.132D+03,0.105D+03,0.94340000D+00,0.97060000D+00
     .,0.11534880D+03,0.132D+03,0.106D+03,0.94340000D+00,0.98680000D+00
     .,0.79327800D+02,0.132D+03,0.107D+03,0.94340000D+00,0.99440000D+00
     .,0.57175200D+02,0.132D+03,0.108D+03,0.94340000D+00,0.99250000D+00
     .,0.38758300D+02,0.132D+03,0.109D+03,0.94340000D+00,0.99820000D+00
     .,0.23460480D+03,0.132D+03,0.111D+03,0.94340000D+00,0.96840000D+00
     .,0.36333200D+03,0.132D+03,0.112D+03,0.94340000D+00,0.96280000D+00
     .,0.36665610D+03,0.132D+03,0.113D+03,0.94340000D+00,0.96480000D+00
     .,0.29253350D+03,0.132D+03,0.114D+03,0.94340000D+00,0.95070000D+00
     .,0.23795800D+03,0.132D+03,0.115D+03,0.94340000D+00,0.99470000D+00
     .,0.20005260D+03,0.132D+03,0.116D+03,0.94340000D+00,0.99480000D+00
     .,0.16242060D+03,0.132D+03,0.117D+03,0.94340000D+00,0.99720000D+00
     .,0.32115740D+03,0.132D+03,0.119D+03,0.94340000D+00,0.97670000D+00
     .,0.62067350D+03,0.132D+03,0.120D+03,0.94340000D+00,0.98310000D+00
     .,0.32013420D+03,0.132D+03,0.121D+03,0.94340000D+00,0.18627000D+01
     .,0.30883040D+03,0.132D+03,0.122D+03,0.94340000D+00,0.18299000D+01
     .,0.30262720D+03,0.132D+03,0.123D+03,0.94340000D+00,0.19138000D+01
     .,0.29997060D+03,0.132D+03,0.124D+03,0.94340000D+00,0.18269000D+01
     .,0.27526070D+03,0.132D+03,0.125D+03,0.94340000D+00,0.16406000D+01
     .,0.25437350D+03,0.132D+03,0.126D+03,0.94340000D+00,0.16483000D+01
     ./)
      pars(  38101:  38200)=(/
     . 0.24255720D+03,0.132D+03,0.127D+03,0.94340000D+00,0.17149000D+01
     .,0.23717370D+03,0.132D+03,0.128D+03,0.94340000D+00,0.17937000D+01
     .,0.23483270D+03,0.132D+03,0.129D+03,0.94340000D+00,0.95760000D+00
     .,0.21945080D+03,0.132D+03,0.130D+03,0.94340000D+00,0.19419000D+01
     .,0.36268320D+03,0.132D+03,0.131D+03,0.94340000D+00,0.96010000D+00
     .,0.31685210D+03,0.132D+03,0.132D+03,0.94340000D+00,0.94340000D+00
     .,0.27357400D+02,0.133D+03,0.100D+01,0.98890000D+00,0.91180000D+00
     .,0.18081600D+02,0.133D+03,0.200D+01,0.98890000D+00,0.00000000D+00
     .,0.40634270D+03,0.133D+03,0.300D+01,0.98890000D+00,0.00000000D+00
     .,0.24075590D+03,0.133D+03,0.400D+01,0.98890000D+00,0.00000000D+00
     .,0.16396550D+03,0.133D+03,0.500D+01,0.98890000D+00,0.00000000D+00
     .,0.11142570D+03,0.133D+03,0.600D+01,0.98890000D+00,0.00000000D+00
     .,0.78113800D+02,0.133D+03,0.700D+01,0.98890000D+00,0.00000000D+00
     .,0.59176400D+02,0.133D+03,0.800D+01,0.98890000D+00,0.00000000D+00
     .,0.44807900D+02,0.133D+03,0.900D+01,0.98890000D+00,0.00000000D+00
     .,0.34422600D+02,0.133D+03,0.100D+02,0.98890000D+00,0.00000000D+00
     .,0.48646380D+03,0.133D+03,0.110D+02,0.98890000D+00,0.00000000D+00
     .,0.38183960D+03,0.133D+03,0.120D+02,0.98890000D+00,0.00000000D+00
     .,0.35437060D+03,0.133D+03,0.130D+02,0.98890000D+00,0.00000000D+00
     .,0.28148930D+03,0.133D+03,0.140D+02,0.98890000D+00,0.00000000D+00
     ./)
      pars(  38201:  38300)=(/
     . 0.22071460D+03,0.133D+03,0.150D+02,0.98890000D+00,0.00000000D+00
     .,0.18364620D+03,0.133D+03,0.160D+02,0.98890000D+00,0.00000000D+00
     .,0.15027420D+03,0.133D+03,0.170D+02,0.98890000D+00,0.00000000D+00
     .,0.12303580D+03,0.133D+03,0.180D+02,0.98890000D+00,0.00000000D+00
     .,0.79363060D+03,0.133D+03,0.190D+02,0.98890000D+00,0.00000000D+00
     .,0.66532210D+03,0.133D+03,0.200D+02,0.98890000D+00,0.00000000D+00
     .,0.55149990D+03,0.133D+03,0.210D+02,0.98890000D+00,0.00000000D+00
     .,0.53402660D+03,0.133D+03,0.220D+02,0.98890000D+00,0.00000000D+00
     .,0.48982100D+03,0.133D+03,0.230D+02,0.98890000D+00,0.00000000D+00
     .,0.38565770D+03,0.133D+03,0.240D+02,0.98890000D+00,0.00000000D+00
     .,0.42268740D+03,0.133D+03,0.250D+02,0.98890000D+00,0.00000000D+00
     .,0.33163890D+03,0.133D+03,0.260D+02,0.98890000D+00,0.00000000D+00
     .,0.35300740D+03,0.133D+03,0.270D+02,0.98890000D+00,0.00000000D+00
     .,0.36305660D+03,0.133D+03,0.280D+02,0.98890000D+00,0.00000000D+00
     .,0.27805780D+03,0.133D+03,0.290D+02,0.98890000D+00,0.00000000D+00
     .,0.28717780D+03,0.133D+03,0.300D+02,0.98890000D+00,0.00000000D+00
     .,0.33996250D+03,0.133D+03,0.310D+02,0.98890000D+00,0.00000000D+00
     .,0.30168000D+03,0.133D+03,0.320D+02,0.98890000D+00,0.00000000D+00
     .,0.25852630D+03,0.133D+03,0.330D+02,0.98890000D+00,0.00000000D+00
     .,0.23250720D+03,0.133D+03,0.340D+02,0.98890000D+00,0.00000000D+00
     ./)
      pars(  38301:  38400)=(/
     . 0.20384000D+03,0.133D+03,0.350D+02,0.98890000D+00,0.00000000D+00
     .,0.17747160D+03,0.133D+03,0.360D+02,0.98890000D+00,0.00000000D+00
     .,0.89049310D+03,0.133D+03,0.370D+02,0.98890000D+00,0.00000000D+00
     .,0.79208160D+03,0.133D+03,0.380D+02,0.98890000D+00,0.00000000D+00
     .,0.69780370D+03,0.133D+03,0.390D+02,0.98890000D+00,0.00000000D+00
     .,0.62921900D+03,0.133D+03,0.400D+02,0.98890000D+00,0.00000000D+00
     .,0.57490090D+03,0.133D+03,0.410D+02,0.98890000D+00,0.00000000D+00
     .,0.44507990D+03,0.133D+03,0.420D+02,0.98890000D+00,0.00000000D+00
     .,0.49610230D+03,0.133D+03,0.430D+02,0.98890000D+00,0.00000000D+00
     .,0.37904270D+03,0.133D+03,0.440D+02,0.98890000D+00,0.00000000D+00
     .,0.41454110D+03,0.133D+03,0.450D+02,0.98890000D+00,0.00000000D+00
     .,0.38481520D+03,0.133D+03,0.460D+02,0.98890000D+00,0.00000000D+00
     .,0.32022310D+03,0.133D+03,0.470D+02,0.98890000D+00,0.00000000D+00
     .,0.33948140D+03,0.133D+03,0.480D+02,0.98890000D+00,0.00000000D+00
     .,0.42470080D+03,0.133D+03,0.490D+02,0.98890000D+00,0.00000000D+00
     .,0.39481650D+03,0.133D+03,0.500D+02,0.98890000D+00,0.00000000D+00
     .,0.35342450D+03,0.133D+03,0.510D+02,0.98890000D+00,0.00000000D+00
     .,0.32867560D+03,0.133D+03,0.520D+02,0.98890000D+00,0.00000000D+00
     .,0.29780640D+03,0.133D+03,0.530D+02,0.98890000D+00,0.00000000D+00
     .,0.26817000D+03,0.133D+03,0.540D+02,0.98890000D+00,0.00000000D+00
     ./)
      pars(  38401:  38500)=(/
     . 0.10853677D+04,0.133D+03,0.550D+02,0.98890000D+00,0.00000000D+00
     .,0.10077913D+04,0.133D+03,0.560D+02,0.98890000D+00,0.00000000D+00
     .,0.89047640D+03,0.133D+03,0.570D+02,0.98890000D+00,0.00000000D+00
     .,0.41703500D+03,0.133D+03,0.580D+02,0.98890000D+00,0.27991000D+01
     .,0.89428150D+03,0.133D+03,0.590D+02,0.98890000D+00,0.00000000D+00
     .,0.85952480D+03,0.133D+03,0.600D+02,0.98890000D+00,0.00000000D+00
     .,0.83818370D+03,0.133D+03,0.610D+02,0.98890000D+00,0.00000000D+00
     .,0.81853810D+03,0.133D+03,0.620D+02,0.98890000D+00,0.00000000D+00
     .,0.80112860D+03,0.133D+03,0.630D+02,0.98890000D+00,0.00000000D+00
     .,0.63358090D+03,0.133D+03,0.640D+02,0.98890000D+00,0.00000000D+00
     .,0.70687070D+03,0.133D+03,0.650D+02,0.98890000D+00,0.00000000D+00
     .,0.68251260D+03,0.133D+03,0.660D+02,0.98890000D+00,0.00000000D+00
     .,0.72360130D+03,0.133D+03,0.670D+02,0.98890000D+00,0.00000000D+00
     .,0.70836530D+03,0.133D+03,0.680D+02,0.98890000D+00,0.00000000D+00
     .,0.69468070D+03,0.133D+03,0.690D+02,0.98890000D+00,0.00000000D+00
     .,0.68640110D+03,0.133D+03,0.700D+02,0.98890000D+00,0.00000000D+00
     .,0.58061170D+03,0.133D+03,0.710D+02,0.98890000D+00,0.00000000D+00
     .,0.57440900D+03,0.133D+03,0.720D+02,0.98890000D+00,0.00000000D+00
     .,0.52573750D+03,0.133D+03,0.730D+02,0.98890000D+00,0.00000000D+00
     .,0.44464860D+03,0.133D+03,0.740D+02,0.98890000D+00,0.00000000D+00
     ./)
      pars(  38501:  38600)=(/
     . 0.45291770D+03,0.133D+03,0.750D+02,0.98890000D+00,0.00000000D+00
     .,0.41136690D+03,0.133D+03,0.760D+02,0.98890000D+00,0.00000000D+00
     .,0.37733280D+03,0.133D+03,0.770D+02,0.98890000D+00,0.00000000D+00
     .,0.31370400D+03,0.133D+03,0.780D+02,0.98890000D+00,0.00000000D+00
     .,0.29316560D+03,0.133D+03,0.790D+02,0.98890000D+00,0.00000000D+00
     .,0.30197230D+03,0.133D+03,0.800D+02,0.98890000D+00,0.00000000D+00
     .,0.43615840D+03,0.133D+03,0.810D+02,0.98890000D+00,0.00000000D+00
     .,0.42826390D+03,0.133D+03,0.820D+02,0.98890000D+00,0.00000000D+00
     .,0.39514950D+03,0.133D+03,0.830D+02,0.98890000D+00,0.00000000D+00
     .,0.37768730D+03,0.133D+03,0.840D+02,0.98890000D+00,0.00000000D+00
     .,0.34934070D+03,0.133D+03,0.850D+02,0.98890000D+00,0.00000000D+00
     .,0.32071970D+03,0.133D+03,0.860D+02,0.98890000D+00,0.00000000D+00
     .,0.10295399D+04,0.133D+03,0.870D+02,0.98890000D+00,0.00000000D+00
     .,0.99943950D+03,0.133D+03,0.880D+02,0.98890000D+00,0.00000000D+00
     .,0.88791120D+03,0.133D+03,0.890D+02,0.98890000D+00,0.00000000D+00
     .,0.80215530D+03,0.133D+03,0.900D+02,0.98890000D+00,0.00000000D+00
     .,0.79394730D+03,0.133D+03,0.910D+02,0.98890000D+00,0.00000000D+00
     .,0.76878210D+03,0.133D+03,0.920D+02,0.98890000D+00,0.00000000D+00
     .,0.78864020D+03,0.133D+03,0.930D+02,0.98890000D+00,0.00000000D+00
     .,0.76421100D+03,0.133D+03,0.940D+02,0.98890000D+00,0.00000000D+00
     ./)
      pars(  38601:  38700)=(/
     . 0.43906700D+02,0.133D+03,0.101D+03,0.98890000D+00,0.00000000D+00
     .,0.14013530D+03,0.133D+03,0.103D+03,0.98890000D+00,0.98650000D+00
     .,0.17913900D+03,0.133D+03,0.104D+03,0.98890000D+00,0.98080000D+00
     .,0.13814460D+03,0.133D+03,0.105D+03,0.98890000D+00,0.97060000D+00
     .,0.10445320D+03,0.133D+03,0.106D+03,0.98890000D+00,0.98680000D+00
     .,0.72807300D+02,0.133D+03,0.107D+03,0.98890000D+00,0.99440000D+00
     .,0.53064500D+02,0.133D+03,0.108D+03,0.98890000D+00,0.99250000D+00
     .,0.36472800D+02,0.133D+03,0.109D+03,0.98890000D+00,0.99820000D+00
     .,0.20419370D+03,0.133D+03,0.111D+03,0.98890000D+00,0.96840000D+00
     .,0.31566880D+03,0.133D+03,0.112D+03,0.98890000D+00,0.96280000D+00
     .,0.32160430D+03,0.133D+03,0.113D+03,0.98890000D+00,0.96480000D+00
     .,0.26040440D+03,0.133D+03,0.114D+03,0.98890000D+00,0.95070000D+00
     .,0.21425140D+03,0.133D+03,0.115D+03,0.98890000D+00,0.99470000D+00
     .,0.18159050D+03,0.133D+03,0.116D+03,0.98890000D+00,0.99480000D+00
     .,0.14869430D+03,0.133D+03,0.117D+03,0.98890000D+00,0.99720000D+00
     .,0.28230700D+03,0.133D+03,0.119D+03,0.98890000D+00,0.97670000D+00
     .,0.53224610D+03,0.133D+03,0.120D+03,0.98890000D+00,0.98310000D+00
     .,0.28445160D+03,0.133D+03,0.121D+03,0.98890000D+00,0.18627000D+01
     .,0.27459290D+03,0.133D+03,0.122D+03,0.98890000D+00,0.18299000D+01
     .,0.26902700D+03,0.133D+03,0.123D+03,0.98890000D+00,0.19138000D+01
     ./)
      pars(  38701:  38800)=(/
     . 0.26629390D+03,0.133D+03,0.124D+03,0.98890000D+00,0.18269000D+01
     .,0.24597130D+03,0.133D+03,0.125D+03,0.98890000D+00,0.16406000D+01
     .,0.22781130D+03,0.133D+03,0.126D+03,0.98890000D+00,0.16483000D+01
     .,0.21726350D+03,0.133D+03,0.127D+03,0.98890000D+00,0.17149000D+01
     .,0.21232210D+03,0.133D+03,0.128D+03,0.98890000D+00,0.17937000D+01
     .,0.20913220D+03,0.133D+03,0.129D+03,0.98890000D+00,0.95760000D+00
     .,0.19727710D+03,0.133D+03,0.130D+03,0.98890000D+00,0.19419000D+01
     .,0.31978710D+03,0.133D+03,0.131D+03,0.98890000D+00,0.96010000D+00
     .,0.28257170D+03,0.133D+03,0.132D+03,0.98890000D+00,0.94340000D+00
     .,0.25419240D+03,0.133D+03,0.133D+03,0.98890000D+00,0.98890000D+00
     .,0.25314900D+02,0.134D+03,0.100D+01,0.99010000D+00,0.91180000D+00
     .,0.17028000D+02,0.134D+03,0.200D+01,0.99010000D+00,0.00000000D+00
     .,0.35504500D+03,0.134D+03,0.300D+01,0.99010000D+00,0.00000000D+00
     .,0.21516710D+03,0.134D+03,0.400D+01,0.99010000D+00,0.00000000D+00
     .,0.14885900D+03,0.134D+03,0.500D+01,0.99010000D+00,0.00000000D+00
     .,0.10246200D+03,0.134D+03,0.600D+01,0.99010000D+00,0.00000000D+00
     .,0.72556700D+02,0.134D+03,0.700D+01,0.99010000D+00,0.00000000D+00
     .,0.55375200D+02,0.134D+03,0.800D+01,0.99010000D+00,0.00000000D+00
     .,0.42202700D+02,0.134D+03,0.900D+01,0.99010000D+00,0.00000000D+00
     .,0.32593500D+02,0.134D+03,0.100D+02,0.99010000D+00,0.00000000D+00
     ./)
      pars(  38801:  38900)=(/
     . 0.42577330D+03,0.134D+03,0.110D+02,0.99010000D+00,0.00000000D+00
     .,0.33984270D+03,0.134D+03,0.120D+02,0.99010000D+00,0.00000000D+00
     .,0.31781330D+03,0.134D+03,0.130D+02,0.99010000D+00,0.00000000D+00
     .,0.25511530D+03,0.134D+03,0.140D+02,0.99010000D+00,0.00000000D+00
     .,0.20196030D+03,0.134D+03,0.150D+02,0.99010000D+00,0.00000000D+00
     .,0.16917120D+03,0.134D+03,0.160D+02,0.99010000D+00,0.00000000D+00
     .,0.13934530D+03,0.134D+03,0.170D+02,0.99010000D+00,0.00000000D+00
     .,0.11476770D+03,0.134D+03,0.180D+02,0.99010000D+00,0.00000000D+00
     .,0.69398180D+03,0.134D+03,0.190D+02,0.99010000D+00,0.00000000D+00
     .,0.58857630D+03,0.134D+03,0.200D+02,0.99010000D+00,0.00000000D+00
     .,0.48933050D+03,0.134D+03,0.210D+02,0.99010000D+00,0.00000000D+00
     .,0.47544530D+03,0.134D+03,0.220D+02,0.99010000D+00,0.00000000D+00
     .,0.43692930D+03,0.134D+03,0.230D+02,0.99010000D+00,0.00000000D+00
     .,0.34458580D+03,0.134D+03,0.240D+02,0.99010000D+00,0.00000000D+00
     .,0.37811330D+03,0.134D+03,0.250D+02,0.99010000D+00,0.00000000D+00
     .,0.29726970D+03,0.134D+03,0.260D+02,0.99010000D+00,0.00000000D+00
     .,0.31722330D+03,0.134D+03,0.270D+02,0.99010000D+00,0.00000000D+00
     .,0.32556670D+03,0.134D+03,0.280D+02,0.99010000D+00,0.00000000D+00
     .,0.24982100D+03,0.134D+03,0.290D+02,0.99010000D+00,0.00000000D+00
     .,0.25919160D+03,0.134D+03,0.300D+02,0.99010000D+00,0.00000000D+00
     ./)
      pars(  38901:  39000)=(/
     . 0.30612750D+03,0.134D+03,0.310D+02,0.99010000D+00,0.00000000D+00
     .,0.27376800D+03,0.134D+03,0.320D+02,0.99010000D+00,0.00000000D+00
     .,0.23642060D+03,0.134D+03,0.330D+02,0.99010000D+00,0.00000000D+00
     .,0.21372980D+03,0.134D+03,0.340D+02,0.99010000D+00,0.00000000D+00
     .,0.18840310D+03,0.134D+03,0.350D+02,0.99010000D+00,0.00000000D+00
     .,0.16488330D+03,0.134D+03,0.360D+02,0.99010000D+00,0.00000000D+00
     .,0.78009860D+03,0.134D+03,0.370D+02,0.99010000D+00,0.00000000D+00
     .,0.70080150D+03,0.134D+03,0.380D+02,0.99010000D+00,0.00000000D+00
     .,0.62089180D+03,0.134D+03,0.390D+02,0.99010000D+00,0.00000000D+00
     .,0.56200010D+03,0.134D+03,0.400D+02,0.99010000D+00,0.00000000D+00
     .,0.51489450D+03,0.134D+03,0.410D+02,0.99010000D+00,0.00000000D+00
     .,0.40077220D+03,0.134D+03,0.420D+02,0.99010000D+00,0.00000000D+00
     .,0.44579160D+03,0.134D+03,0.430D+02,0.99010000D+00,0.00000000D+00
     .,0.34259700D+03,0.134D+03,0.440D+02,0.99010000D+00,0.00000000D+00
     .,0.37429000D+03,0.134D+03,0.450D+02,0.99010000D+00,0.00000000D+00
     .,0.34804020D+03,0.134D+03,0.460D+02,0.99010000D+00,0.00000000D+00
     .,0.28983280D+03,0.134D+03,0.470D+02,0.99010000D+00,0.00000000D+00
     .,0.30772140D+03,0.134D+03,0.480D+02,0.99010000D+00,0.00000000D+00
     .,0.38283810D+03,0.134D+03,0.490D+02,0.99010000D+00,0.00000000D+00
     .,0.35796090D+03,0.134D+03,0.500D+02,0.99010000D+00,0.00000000D+00
     ./)
      pars(  39001:  39100)=(/
     . 0.32248120D+03,0.134D+03,0.510D+02,0.99010000D+00,0.00000000D+00
     .,0.30115060D+03,0.134D+03,0.520D+02,0.99010000D+00,0.00000000D+00
     .,0.27413770D+03,0.134D+03,0.530D+02,0.99010000D+00,0.00000000D+00
     .,0.24798670D+03,0.134D+03,0.540D+02,0.99010000D+00,0.00000000D+00
     .,0.95163050D+03,0.134D+03,0.550D+02,0.99010000D+00,0.00000000D+00
     .,0.89062640D+03,0.134D+03,0.560D+02,0.99010000D+00,0.00000000D+00
     .,0.79124710D+03,0.134D+03,0.570D+02,0.99010000D+00,0.00000000D+00
     .,0.38031600D+03,0.134D+03,0.580D+02,0.99010000D+00,0.27991000D+01
     .,0.79188170D+03,0.134D+03,0.590D+02,0.99010000D+00,0.00000000D+00
     .,0.76169980D+03,0.134D+03,0.600D+02,0.99010000D+00,0.00000000D+00
     .,0.74294350D+03,0.134D+03,0.610D+02,0.99010000D+00,0.00000000D+00
     .,0.72565360D+03,0.134D+03,0.620D+02,0.99010000D+00,0.00000000D+00
     .,0.71033760D+03,0.134D+03,0.630D+02,0.99010000D+00,0.00000000D+00
     .,0.56579140D+03,0.134D+03,0.640D+02,0.99010000D+00,0.00000000D+00
     .,0.62617050D+03,0.134D+03,0.650D+02,0.99010000D+00,0.00000000D+00
     .,0.60528280D+03,0.134D+03,0.660D+02,0.99010000D+00,0.00000000D+00
     .,0.64237950D+03,0.134D+03,0.670D+02,0.99010000D+00,0.00000000D+00
     .,0.62891080D+03,0.134D+03,0.680D+02,0.99010000D+00,0.00000000D+00
     .,0.61686660D+03,0.134D+03,0.690D+02,0.99010000D+00,0.00000000D+00
     .,0.60931460D+03,0.134D+03,0.700D+02,0.99010000D+00,0.00000000D+00
     ./)
      pars(  39101:  39200)=(/
     . 0.51790690D+03,0.134D+03,0.710D+02,0.99010000D+00,0.00000000D+00
     .,0.51523880D+03,0.134D+03,0.720D+02,0.99010000D+00,0.00000000D+00
     .,0.47346190D+03,0.134D+03,0.730D+02,0.99010000D+00,0.00000000D+00
     .,0.40207780D+03,0.134D+03,0.740D+02,0.99010000D+00,0.00000000D+00
     .,0.41003520D+03,0.134D+03,0.750D+02,0.99010000D+00,0.00000000D+00
     .,0.37374220D+03,0.134D+03,0.760D+02,0.99010000D+00,0.00000000D+00
     .,0.34383760D+03,0.134D+03,0.770D+02,0.99010000D+00,0.00000000D+00
     .,0.28699450D+03,0.134D+03,0.780D+02,0.99010000D+00,0.00000000D+00
     .,0.26861450D+03,0.134D+03,0.790D+02,0.99010000D+00,0.00000000D+00
     .,0.27687780D+03,0.134D+03,0.800D+02,0.99010000D+00,0.00000000D+00
     .,0.39433850D+03,0.134D+03,0.810D+02,0.99010000D+00,0.00000000D+00
     .,0.38873570D+03,0.134D+03,0.820D+02,0.99010000D+00,0.00000000D+00
     .,0.36061850D+03,0.134D+03,0.830D+02,0.99010000D+00,0.00000000D+00
     .,0.34584750D+03,0.134D+03,0.840D+02,0.99010000D+00,0.00000000D+00
     .,0.32125280D+03,0.134D+03,0.850D+02,0.99010000D+00,0.00000000D+00
     .,0.29613640D+03,0.134D+03,0.860D+02,0.99010000D+00,0.00000000D+00
     .,0.90670650D+03,0.134D+03,0.870D+02,0.99010000D+00,0.00000000D+00
     .,0.88597380D+03,0.134D+03,0.880D+02,0.99010000D+00,0.00000000D+00
     .,0.79097170D+03,0.134D+03,0.890D+02,0.99010000D+00,0.00000000D+00
     .,0.71920180D+03,0.134D+03,0.900D+02,0.99010000D+00,0.00000000D+00
     ./)
      pars(  39201:  39300)=(/
     . 0.71002990D+03,0.134D+03,0.910D+02,0.99010000D+00,0.00000000D+00
     .,0.68765590D+03,0.134D+03,0.920D+02,0.99010000D+00,0.00000000D+00
     .,0.70268800D+03,0.134D+03,0.930D+02,0.99010000D+00,0.00000000D+00
     .,0.68135890D+03,0.134D+03,0.940D+02,0.99010000D+00,0.00000000D+00
     .,0.40211700D+02,0.134D+03,0.101D+03,0.99010000D+00,0.00000000D+00
     .,0.12555640D+03,0.134D+03,0.103D+03,0.99010000D+00,0.98650000D+00
     .,0.16106370D+03,0.134D+03,0.104D+03,0.99010000D+00,0.98080000D+00
     .,0.12588880D+03,0.134D+03,0.105D+03,0.99010000D+00,0.97060000D+00
     .,0.96087200D+02,0.134D+03,0.106D+03,0.99010000D+00,0.98680000D+00
     .,0.67679600D+02,0.134D+03,0.107D+03,0.99010000D+00,0.99440000D+00
     .,0.49766500D+02,0.134D+03,0.108D+03,0.99010000D+00,0.99250000D+00
     .,0.34588200D+02,0.134D+03,0.109D+03,0.99010000D+00,0.99820000D+00
     .,0.18260620D+03,0.134D+03,0.111D+03,0.99010000D+00,0.96840000D+00
     .,0.28192490D+03,0.134D+03,0.112D+03,0.99010000D+00,0.96280000D+00
     .,0.28901630D+03,0.134D+03,0.113D+03,0.99010000D+00,0.96480000D+00
     .,0.23642530D+03,0.134D+03,0.114D+03,0.99010000D+00,0.95070000D+00
     .,0.19613740D+03,0.134D+03,0.115D+03,0.99010000D+00,0.99470000D+00
     .,0.16726720D+03,0.134D+03,0.116D+03,0.99010000D+00,0.99480000D+00
     .,0.13787350D+03,0.134D+03,0.117D+03,0.99010000D+00,0.99720000D+00
     .,0.25453760D+03,0.134D+03,0.119D+03,0.99010000D+00,0.97670000D+00
     ./)
      pars(  39301:  39400)=(/
     . 0.47198210D+03,0.134D+03,0.120D+03,0.99010000D+00,0.98310000D+00
     .,0.25812950D+03,0.134D+03,0.121D+03,0.99010000D+00,0.18627000D+01
     .,0.24934370D+03,0.134D+03,0.122D+03,0.99010000D+00,0.18299000D+01
     .,0.24427270D+03,0.134D+03,0.123D+03,0.99010000D+00,0.19138000D+01
     .,0.24158030D+03,0.134D+03,0.124D+03,0.99010000D+00,0.18269000D+01
     .,0.22407570D+03,0.134D+03,0.125D+03,0.99010000D+00,0.16406000D+01
     .,0.20787480D+03,0.134D+03,0.126D+03,0.99010000D+00,0.16483000D+01
     .,0.19829930D+03,0.134D+03,0.127D+03,0.99010000D+00,0.17149000D+01
     .,0.19372130D+03,0.134D+03,0.128D+03,0.99010000D+00,0.17937000D+01
     .,0.19016850D+03,0.134D+03,0.129D+03,0.99010000D+00,0.95760000D+00
     .,0.18048120D+03,0.134D+03,0.130D+03,0.99010000D+00,0.19419000D+01
     .,0.28854930D+03,0.134D+03,0.131D+03,0.99010000D+00,0.96010000D+00
     .,0.25695230D+03,0.134D+03,0.132D+03,0.99010000D+00,0.94340000D+00
     .,0.23257840D+03,0.134D+03,0.133D+03,0.99010000D+00,0.98890000D+00
     .,0.21378620D+03,0.134D+03,0.134D+03,0.99010000D+00,0.99010000D+00
     .,0.22623000D+02,0.135D+03,0.100D+01,0.99740000D+00,0.91180000D+00
     .,0.15538700D+02,0.135D+03,0.200D+01,0.99740000D+00,0.00000000D+00
     .,0.29768350D+03,0.135D+03,0.300D+01,0.99740000D+00,0.00000000D+00
     .,0.18488320D+03,0.135D+03,0.400D+01,0.99740000D+00,0.00000000D+00
     .,0.13018810D+03,0.135D+03,0.500D+01,0.99740000D+00,0.00000000D+00
     ./)
      pars(  39401:  39500)=(/
     . 0.90934300D+02,0.135D+03,0.600D+01,0.99740000D+00,0.00000000D+00
     .,0.65154300D+02,0.135D+03,0.700D+01,0.99740000D+00,0.00000000D+00
     .,0.50163200D+02,0.135D+03,0.800D+01,0.99740000D+00,0.00000000D+00
     .,0.38529400D+02,0.135D+03,0.900D+01,0.99740000D+00,0.00000000D+00
     .,0.29950100D+02,0.135D+03,0.100D+02,0.99740000D+00,0.00000000D+00
     .,0.35772460D+03,0.135D+03,0.110D+02,0.99740000D+00,0.00000000D+00
     .,0.29072760D+03,0.135D+03,0.120D+02,0.99740000D+00,0.00000000D+00
     .,0.27418940D+03,0.135D+03,0.130D+02,0.99740000D+00,0.00000000D+00
     .,0.22268120D+03,0.135D+03,0.140D+02,0.99740000D+00,0.00000000D+00
     .,0.17819840D+03,0.135D+03,0.150D+02,0.99740000D+00,0.00000000D+00
     .,0.15042030D+03,0.135D+03,0.160D+02,0.99740000D+00,0.00000000D+00
     .,0.12484950D+03,0.135D+03,0.170D+02,0.99740000D+00,0.00000000D+00
     .,0.10354670D+03,0.135D+03,0.180D+02,0.99740000D+00,0.00000000D+00
     .,0.58289140D+03,0.135D+03,0.190D+02,0.99740000D+00,0.00000000D+00
     .,0.50041520D+03,0.135D+03,0.200D+02,0.99740000D+00,0.00000000D+00
     .,0.41736480D+03,0.135D+03,0.210D+02,0.99740000D+00,0.00000000D+00
     .,0.40709700D+03,0.135D+03,0.220D+02,0.99740000D+00,0.00000000D+00
     .,0.37492830D+03,0.135D+03,0.230D+02,0.99740000D+00,0.00000000D+00
     .,0.29636070D+03,0.135D+03,0.240D+02,0.99740000D+00,0.00000000D+00
     .,0.32549450D+03,0.135D+03,0.250D+02,0.99740000D+00,0.00000000D+00
     ./)
      pars(  39501:  39600)=(/
     . 0.25659390D+03,0.135D+03,0.260D+02,0.99740000D+00,0.00000000D+00
     .,0.27446200D+03,0.135D+03,0.270D+02,0.99740000D+00,0.00000000D+00
     .,0.28101540D+03,0.135D+03,0.280D+02,0.99740000D+00,0.00000000D+00
     .,0.21621400D+03,0.135D+03,0.290D+02,0.99740000D+00,0.00000000D+00
     .,0.22534570D+03,0.135D+03,0.300D+02,0.99740000D+00,0.00000000D+00
     .,0.26539390D+03,0.135D+03,0.310D+02,0.99740000D+00,0.00000000D+00
     .,0.23936360D+03,0.135D+03,0.320D+02,0.99740000D+00,0.00000000D+00
     .,0.20849570D+03,0.135D+03,0.330D+02,0.99740000D+00,0.00000000D+00
     .,0.18959740D+03,0.135D+03,0.340D+02,0.99740000D+00,0.00000000D+00
     .,0.16817910D+03,0.135D+03,0.350D+02,0.99740000D+00,0.00000000D+00
     .,0.14806890D+03,0.135D+03,0.360D+02,0.99740000D+00,0.00000000D+00
     .,0.65665740D+03,0.135D+03,0.370D+02,0.99740000D+00,0.00000000D+00
     .,0.59602140D+03,0.135D+03,0.380D+02,0.99740000D+00,0.00000000D+00
     .,0.53134260D+03,0.135D+03,0.390D+02,0.99740000D+00,0.00000000D+00
     .,0.48298730D+03,0.135D+03,0.400D+02,0.99740000D+00,0.00000000D+00
     .,0.44388480D+03,0.135D+03,0.410D+02,0.99740000D+00,0.00000000D+00
     .,0.34768000D+03,0.135D+03,0.420D+02,0.99740000D+00,0.00000000D+00
     .,0.38579860D+03,0.135D+03,0.430D+02,0.99740000D+00,0.00000000D+00
     .,0.29851740D+03,0.135D+03,0.440D+02,0.99740000D+00,0.00000000D+00
     .,0.32566500D+03,0.135D+03,0.450D+02,0.99740000D+00,0.00000000D+00
     ./)
      pars(  39601:  39700)=(/
     . 0.30341070D+03,0.135D+03,0.460D+02,0.99740000D+00,0.00000000D+00
     .,0.25300950D+03,0.135D+03,0.470D+02,0.99740000D+00,0.00000000D+00
     .,0.26894910D+03,0.135D+03,0.480D+02,0.99740000D+00,0.00000000D+00
     .,0.33247910D+03,0.135D+03,0.490D+02,0.99740000D+00,0.00000000D+00
     .,0.31279940D+03,0.135D+03,0.500D+02,0.99740000D+00,0.00000000D+00
     .,0.28377610D+03,0.135D+03,0.510D+02,0.99740000D+00,0.00000000D+00
     .,0.26623860D+03,0.135D+03,0.520D+02,0.99740000D+00,0.00000000D+00
     .,0.24362960D+03,0.135D+03,0.530D+02,0.99740000D+00,0.00000000D+00
     .,0.22153670D+03,0.135D+03,0.540D+02,0.99740000D+00,0.00000000D+00
     .,0.80186140D+03,0.135D+03,0.550D+02,0.99740000D+00,0.00000000D+00
     .,0.75662710D+03,0.135D+03,0.560D+02,0.99740000D+00,0.00000000D+00
     .,0.67619300D+03,0.135D+03,0.570D+02,0.99740000D+00,0.00000000D+00
     .,0.33452620D+03,0.135D+03,0.580D+02,0.99740000D+00,0.27991000D+01
     .,0.67428760D+03,0.135D+03,0.590D+02,0.99740000D+00,0.00000000D+00
     .,0.64912820D+03,0.135D+03,0.600D+02,0.99740000D+00,0.00000000D+00
     .,0.63328350D+03,0.135D+03,0.610D+02,0.99740000D+00,0.00000000D+00
     .,0.61865400D+03,0.135D+03,0.620D+02,0.99740000D+00,0.00000000D+00
     .,0.60569940D+03,0.135D+03,0.630D+02,0.99740000D+00,0.00000000D+00
     .,0.48634440D+03,0.135D+03,0.640D+02,0.99740000D+00,0.00000000D+00
     .,0.53364850D+03,0.135D+03,0.650D+02,0.99740000D+00,0.00000000D+00
     ./)
      pars(  39701:  39800)=(/
     . 0.51649050D+03,0.135D+03,0.660D+02,0.99740000D+00,0.00000000D+00
     .,0.54846290D+03,0.135D+03,0.670D+02,0.99740000D+00,0.00000000D+00
     .,0.53700790D+03,0.135D+03,0.680D+02,0.99740000D+00,0.00000000D+00
     .,0.52681470D+03,0.135D+03,0.690D+02,0.99740000D+00,0.00000000D+00
     .,0.52016740D+03,0.135D+03,0.700D+02,0.99740000D+00,0.00000000D+00
     .,0.44455160D+03,0.135D+03,0.710D+02,0.99740000D+00,0.00000000D+00
     .,0.44488510D+03,0.135D+03,0.720D+02,0.99740000D+00,0.00000000D+00
     .,0.41064300D+03,0.135D+03,0.730D+02,0.99740000D+00,0.00000000D+00
     .,0.35042340D+03,0.135D+03,0.740D+02,0.99740000D+00,0.00000000D+00
     .,0.35779320D+03,0.135D+03,0.750D+02,0.99740000D+00,0.00000000D+00
     .,0.32743170D+03,0.135D+03,0.760D+02,0.99740000D+00,0.00000000D+00
     .,0.30224750D+03,0.135D+03,0.770D+02,0.99740000D+00,0.00000000D+00
     .,0.25348160D+03,0.135D+03,0.780D+02,0.99740000D+00,0.00000000D+00
     .,0.23768320D+03,0.135D+03,0.790D+02,0.99740000D+00,0.00000000D+00
     .,0.24514110D+03,0.135D+03,0.800D+02,0.99740000D+00,0.00000000D+00
     .,0.34374040D+03,0.135D+03,0.810D+02,0.99740000D+00,0.00000000D+00
     .,0.34024220D+03,0.135D+03,0.820D+02,0.99740000D+00,0.00000000D+00
     .,0.31748380D+03,0.135D+03,0.830D+02,0.99740000D+00,0.00000000D+00
     .,0.30560970D+03,0.135D+03,0.840D+02,0.99740000D+00,0.00000000D+00
     .,0.28522290D+03,0.135D+03,0.850D+02,0.99740000D+00,0.00000000D+00
     ./)
      pars(  39801:  39900)=(/
     . 0.26413340D+03,0.135D+03,0.860D+02,0.99740000D+00,0.00000000D+00
     .,0.76779060D+03,0.135D+03,0.870D+02,0.99740000D+00,0.00000000D+00
     .,0.75526170D+03,0.135D+03,0.880D+02,0.99740000D+00,0.00000000D+00
     .,0.67787670D+03,0.135D+03,0.890D+02,0.99740000D+00,0.00000000D+00
     .,0.62082600D+03,0.135D+03,0.900D+02,0.99740000D+00,0.00000000D+00
     .,0.61130750D+03,0.135D+03,0.910D+02,0.99740000D+00,0.00000000D+00
     .,0.59218710D+03,0.135D+03,0.920D+02,0.99740000D+00,0.00000000D+00
     .,0.60258350D+03,0.135D+03,0.930D+02,0.99740000D+00,0.00000000D+00
     .,0.58469980D+03,0.135D+03,0.940D+02,0.99740000D+00,0.00000000D+00
     .,0.35512300D+02,0.135D+03,0.101D+03,0.99740000D+00,0.00000000D+00
     .,0.10820480D+03,0.135D+03,0.103D+03,0.99740000D+00,0.98650000D+00
     .,0.13936170D+03,0.135D+03,0.104D+03,0.99740000D+00,0.98080000D+00
     .,0.11058020D+03,0.135D+03,0.105D+03,0.99740000D+00,0.97060000D+00
     .,0.85326500D+02,0.135D+03,0.106D+03,0.99740000D+00,0.98680000D+00
     .,0.60835900D+02,0.135D+03,0.107D+03,0.99740000D+00,0.99440000D+00
     .,0.45201900D+02,0.135D+03,0.108D+03,0.99740000D+00,0.99250000D+00
     .,0.31831300D+02,0.135D+03,0.109D+03,0.99740000D+00,0.99820000D+00
     .,0.15710050D+03,0.135D+03,0.111D+03,0.99740000D+00,0.96840000D+00
     .,0.24214760D+03,0.135D+03,0.112D+03,0.99740000D+00,0.96280000D+00
     .,0.24992480D+03,0.135D+03,0.113D+03,0.99740000D+00,0.96480000D+00
     ./)
      pars(  39901:  40000)=(/
     . 0.20678800D+03,0.135D+03,0.114D+03,0.99740000D+00,0.95070000D+00
     .,0.17315820D+03,0.135D+03,0.115D+03,0.99740000D+00,0.99470000D+00
     .,0.14872190D+03,0.135D+03,0.116D+03,0.99740000D+00,0.99480000D+00
     .,0.12352740D+03,0.135D+03,0.117D+03,0.99740000D+00,0.99720000D+00
     .,0.22116420D+03,0.135D+03,0.119D+03,0.99740000D+00,0.97670000D+00
     .,0.40253800D+03,0.135D+03,0.120D+03,0.99740000D+00,0.98310000D+00
     .,0.22577840D+03,0.135D+03,0.121D+03,0.99740000D+00,0.18627000D+01
     .,0.21827110D+03,0.135D+03,0.122D+03,0.99740000D+00,0.18299000D+01
     .,0.21382760D+03,0.135D+03,0.123D+03,0.99740000D+00,0.19138000D+01
     .,0.21128090D+03,0.135D+03,0.124D+03,0.99740000D+00,0.18269000D+01
     .,0.19683950D+03,0.135D+03,0.125D+03,0.99740000D+00,0.16406000D+01
     .,0.18295500D+03,0.135D+03,0.126D+03,0.99740000D+00,0.16483000D+01
     .,0.17459100D+03,0.135D+03,0.127D+03,0.99740000D+00,0.17149000D+01
     .,0.17050040D+03,0.135D+03,0.128D+03,0.99740000D+00,0.17937000D+01
     .,0.16678160D+03,0.135D+03,0.129D+03,0.99740000D+00,0.95760000D+00
     .,0.15930490D+03,0.135D+03,0.130D+03,0.99740000D+00,0.19419000D+01
     .,0.25072470D+03,0.135D+03,0.131D+03,0.99740000D+00,0.96010000D+00
     .,0.22517920D+03,0.135D+03,0.132D+03,0.99740000D+00,0.94340000D+00
     .,0.20523050D+03,0.135D+03,0.133D+03,0.99740000D+00,0.98890000D+00
     .,0.18964080D+03,0.135D+03,0.134D+03,0.99740000D+00,0.99010000D+00
     ./)
      pars(  40001:  40100)=(/
     . 0.16923750D+03,0.135D+03,0.135D+03,0.99740000D+00,0.99740000D+00
     .,0.35196300D+02,0.137D+03,0.100D+01,0.97380000D+00,0.91180000D+00
     .,0.22623100D+02,0.137D+03,0.200D+01,0.97380000D+00,0.00000000D+00
     .,0.65740760D+03,0.137D+03,0.300D+01,0.97380000D+00,0.00000000D+00
     .,0.34543780D+03,0.137D+03,0.400D+01,0.97380000D+00,0.00000000D+00
     .,0.22250810D+03,0.137D+03,0.500D+01,0.97380000D+00,0.00000000D+00
     .,0.14599740D+03,0.137D+03,0.600D+01,0.97380000D+00,0.00000000D+00
     .,0.10014180D+03,0.137D+03,0.700D+01,0.97380000D+00,0.00000000D+00
     .,0.74886800D+02,0.137D+03,0.800D+01,0.97380000D+00,0.00000000D+00
     .,0.56177900D+02,0.137D+03,0.900D+01,0.97380000D+00,0.00000000D+00
     .,0.42895700D+02,0.137D+03,0.100D+02,0.97380000D+00,0.00000000D+00
     .,0.78185270D+03,0.137D+03,0.110D+02,0.97380000D+00,0.00000000D+00
     .,0.55913370D+03,0.137D+03,0.120D+02,0.97380000D+00,0.00000000D+00
     .,0.50299330D+03,0.137D+03,0.130D+02,0.97380000D+00,0.00000000D+00
     .,0.38390840D+03,0.137D+03,0.140D+02,0.97380000D+00,0.00000000D+00
     .,0.29216040D+03,0.137D+03,0.150D+02,0.97380000D+00,0.00000000D+00
     .,0.23905720D+03,0.137D+03,0.160D+02,0.97380000D+00,0.00000000D+00
     .,0.19278770D+03,0.137D+03,0.170D+02,0.97380000D+00,0.00000000D+00
     .,0.15610100D+03,0.137D+03,0.180D+02,0.97380000D+00,0.00000000D+00
     .,0.13058840D+04,0.137D+03,0.190D+02,0.97380000D+00,0.00000000D+00
     ./)
      pars(  40101:  40200)=(/
     . 0.10130706D+04,0.137D+03,0.200D+02,0.97380000D+00,0.00000000D+00
     .,0.82610410D+03,0.137D+03,0.210D+02,0.97380000D+00,0.00000000D+00
     .,0.78941170D+03,0.137D+03,0.220D+02,0.97380000D+00,0.00000000D+00
     .,0.71821250D+03,0.137D+03,0.230D+02,0.97380000D+00,0.00000000D+00
     .,0.56618560D+03,0.137D+03,0.240D+02,0.97380000D+00,0.00000000D+00
     .,0.61264440D+03,0.137D+03,0.250D+02,0.97380000D+00,0.00000000D+00
     .,0.48062360D+03,0.137D+03,0.260D+02,0.97380000D+00,0.00000000D+00
     .,0.50159400D+03,0.137D+03,0.270D+02,0.97380000D+00,0.00000000D+00
     .,0.51998260D+03,0.137D+03,0.280D+02,0.97380000D+00,0.00000000D+00
     .,0.39935340D+03,0.137D+03,0.290D+02,0.97380000D+00,0.00000000D+00
     .,0.40105480D+03,0.137D+03,0.300D+02,0.97380000D+00,0.00000000D+00
     .,0.47797370D+03,0.137D+03,0.310D+02,0.97380000D+00,0.00000000D+00
     .,0.41103890D+03,0.137D+03,0.320D+02,0.97380000D+00,0.00000000D+00
     .,0.34339780D+03,0.137D+03,0.330D+02,0.97380000D+00,0.00000000D+00
     .,0.30453120D+03,0.137D+03,0.340D+02,0.97380000D+00,0.00000000D+00
     .,0.26345920D+03,0.137D+03,0.350D+02,0.97380000D+00,0.00000000D+00
     .,0.22685900D+03,0.137D+03,0.360D+02,0.97380000D+00,0.00000000D+00
     .,0.14577817D+04,0.137D+03,0.370D+02,0.97380000D+00,0.00000000D+00
     .,0.12108108D+04,0.137D+03,0.380D+02,0.97380000D+00,0.00000000D+00
     .,0.10374434D+04,0.137D+03,0.390D+02,0.97380000D+00,0.00000000D+00
     ./)
      pars(  40201:  40300)=(/
     . 0.92013580D+03,0.137D+03,0.400D+02,0.97380000D+00,0.00000000D+00
     .,0.83213680D+03,0.137D+03,0.410D+02,0.97380000D+00,0.00000000D+00
     .,0.63394710D+03,0.137D+03,0.420D+02,0.97380000D+00,0.00000000D+00
     .,0.71077790D+03,0.137D+03,0.430D+02,0.97380000D+00,0.00000000D+00
     .,0.53361350D+03,0.137D+03,0.440D+02,0.97380000D+00,0.00000000D+00
     .,0.58237420D+03,0.137D+03,0.450D+02,0.97380000D+00,0.00000000D+00
     .,0.53725650D+03,0.137D+03,0.460D+02,0.97380000D+00,0.00000000D+00
     .,0.45047140D+03,0.137D+03,0.470D+02,0.97380000D+00,0.00000000D+00
     .,0.47031190D+03,0.137D+03,0.480D+02,0.97380000D+00,0.00000000D+00
     .,0.59989870D+03,0.137D+03,0.490D+02,0.97380000D+00,0.00000000D+00
     .,0.54287770D+03,0.137D+03,0.500D+02,0.97380000D+00,0.00000000D+00
     .,0.47456410D+03,0.137D+03,0.510D+02,0.97380000D+00,0.00000000D+00
     .,0.43555550D+03,0.137D+03,0.520D+02,0.97380000D+00,0.00000000D+00
     .,0.38946840D+03,0.137D+03,0.530D+02,0.97380000D+00,0.00000000D+00
     .,0.34667910D+03,0.137D+03,0.540D+02,0.97380000D+00,0.00000000D+00
     .,0.17768880D+04,0.137D+03,0.550D+02,0.97380000D+00,0.00000000D+00
     .,0.15569958D+04,0.137D+03,0.560D+02,0.97380000D+00,0.00000000D+00
     .,0.13375010D+04,0.137D+03,0.570D+02,0.97380000D+00,0.00000000D+00
     .,0.56314800D+03,0.137D+03,0.580D+02,0.97380000D+00,0.27991000D+01
     .,0.13695245D+04,0.137D+03,0.590D+02,0.97380000D+00,0.00000000D+00
     ./)
      pars(  40301:  40400)=(/
     . 0.13097014D+04,0.137D+03,0.600D+02,0.97380000D+00,0.00000000D+00
     .,0.12753968D+04,0.137D+03,0.610D+02,0.97380000D+00,0.00000000D+00
     .,0.12439950D+04,0.137D+03,0.620D+02,0.97380000D+00,0.00000000D+00
     .,0.12161018D+04,0.137D+03,0.630D+02,0.97380000D+00,0.00000000D+00
     .,0.93460820D+03,0.137D+03,0.640D+02,0.97380000D+00,0.00000000D+00
     .,0.10930934D+04,0.137D+03,0.650D+02,0.97380000D+00,0.00000000D+00
     .,0.10500925D+04,0.137D+03,0.660D+02,0.97380000D+00,0.00000000D+00
     .,0.10903500D+04,0.137D+03,0.670D+02,0.97380000D+00,0.00000000D+00
     .,0.10664469D+04,0.137D+03,0.680D+02,0.97380000D+00,0.00000000D+00
     .,0.10446045D+04,0.137D+03,0.690D+02,0.97380000D+00,0.00000000D+00
     .,0.10333196D+04,0.137D+03,0.700D+02,0.97380000D+00,0.00000000D+00
     .,0.85724780D+03,0.137D+03,0.710D+02,0.97380000D+00,0.00000000D+00
     .,0.82420500D+03,0.137D+03,0.720D+02,0.97380000D+00,0.00000000D+00
     .,0.74273970D+03,0.137D+03,0.730D+02,0.97380000D+00,0.00000000D+00
     .,0.62209770D+03,0.137D+03,0.740D+02,0.97380000D+00,0.00000000D+00
     .,0.62947280D+03,0.137D+03,0.750D+02,0.97380000D+00,0.00000000D+00
     .,0.56455160D+03,0.137D+03,0.760D+02,0.97380000D+00,0.00000000D+00
     .,0.51279660D+03,0.137D+03,0.770D+02,0.97380000D+00,0.00000000D+00
     .,0.42290190D+03,0.137D+03,0.780D+02,0.97380000D+00,0.00000000D+00
     .,0.39392880D+03,0.137D+03,0.790D+02,0.97380000D+00,0.00000000D+00
     ./)
      pars(  40401:  40500)=(/
     . 0.40326450D+03,0.137D+03,0.800D+02,0.97380000D+00,0.00000000D+00
     .,0.61404690D+03,0.137D+03,0.810D+02,0.97380000D+00,0.00000000D+00
     .,0.58982750D+03,0.137D+03,0.820D+02,0.97380000D+00,0.00000000D+00
     .,0.53239070D+03,0.137D+03,0.830D+02,0.97380000D+00,0.00000000D+00
     .,0.50286730D+03,0.137D+03,0.840D+02,0.97380000D+00,0.00000000D+00
     .,0.45900050D+03,0.137D+03,0.850D+02,0.97380000D+00,0.00000000D+00
     .,0.41673210D+03,0.137D+03,0.860D+02,0.97380000D+00,0.00000000D+00
     .,0.16476456D+04,0.137D+03,0.870D+02,0.97380000D+00,0.00000000D+00
     .,0.15223656D+04,0.137D+03,0.880D+02,0.97380000D+00,0.00000000D+00
     .,0.13176426D+04,0.137D+03,0.890D+02,0.97380000D+00,0.00000000D+00
     .,0.11578649D+04,0.137D+03,0.900D+02,0.97380000D+00,0.00000000D+00
     .,0.11634976D+04,0.137D+03,0.910D+02,0.97380000D+00,0.00000000D+00
     .,0.11259076D+04,0.137D+03,0.920D+02,0.97380000D+00,0.00000000D+00
     .,0.11753584D+04,0.137D+03,0.930D+02,0.97380000D+00,0.00000000D+00
     .,0.11351209D+04,0.137D+03,0.940D+02,0.97380000D+00,0.00000000D+00
     .,0.57903900D+02,0.137D+03,0.101D+03,0.97380000D+00,0.00000000D+00
     .,0.19967710D+03,0.137D+03,0.103D+03,0.97380000D+00,0.98650000D+00
     .,0.25295740D+03,0.137D+03,0.104D+03,0.97380000D+00,0.98080000D+00
     .,0.18572060D+03,0.137D+03,0.105D+03,0.97380000D+00,0.97060000D+00
     .,0.13721250D+03,0.137D+03,0.106D+03,0.97380000D+00,0.98680000D+00
     ./)
      pars(  40501:  40600)=(/
     . 0.93458400D+02,0.137D+03,0.107D+03,0.97380000D+00,0.99440000D+00
     .,0.67004500D+02,0.137D+03,0.108D+03,0.97380000D+00,0.99250000D+00
     .,0.45248900D+02,0.137D+03,0.109D+03,0.97380000D+00,0.99820000D+00
     .,0.29467950D+03,0.137D+03,0.111D+03,0.97380000D+00,0.96840000D+00
     .,0.45778310D+03,0.137D+03,0.112D+03,0.97380000D+00,0.96280000D+00
     .,0.45336360D+03,0.137D+03,0.113D+03,0.97380000D+00,0.96480000D+00
     .,0.35341420D+03,0.137D+03,0.114D+03,0.97380000D+00,0.95070000D+00
     .,0.28343680D+03,0.137D+03,0.115D+03,0.97380000D+00,0.99470000D+00
     .,0.23654670D+03,0.137D+03,0.116D+03,0.97380000D+00,0.99480000D+00
     .,0.19086020D+03,0.137D+03,0.117D+03,0.97380000D+00,0.99720000D+00
     .,0.40103710D+03,0.137D+03,0.119D+03,0.97380000D+00,0.97670000D+00
     .,0.81234890D+03,0.137D+03,0.120D+03,0.97380000D+00,0.98310000D+00
     .,0.38990570D+03,0.137D+03,0.121D+03,0.97380000D+00,0.18627000D+01
     .,0.37639190D+03,0.137D+03,0.122D+03,0.97380000D+00,0.18299000D+01
     .,0.36906490D+03,0.137D+03,0.123D+03,0.97380000D+00,0.19138000D+01
     .,0.36686860D+03,0.137D+03,0.124D+03,0.97380000D+00,0.18269000D+01
     .,0.33222150D+03,0.137D+03,0.125D+03,0.97380000D+00,0.16406000D+01
     .,0.30627200D+03,0.137D+03,0.126D+03,0.97380000D+00,0.16483000D+01
     .,0.29229920D+03,0.137D+03,0.127D+03,0.97380000D+00,0.17149000D+01
     .,0.28612410D+03,0.137D+03,0.128D+03,0.97380000D+00,0.17937000D+01
     ./)
      pars(  40601:  40700)=(/
     . 0.28601310D+03,0.137D+03,0.129D+03,0.97380000D+00,0.95760000D+00
     .,0.26273320D+03,0.137D+03,0.130D+03,0.97380000D+00,0.19419000D+01
     .,0.44607980D+03,0.137D+03,0.131D+03,0.97380000D+00,0.96010000D+00
     .,0.38250120D+03,0.137D+03,0.132D+03,0.97380000D+00,0.94340000D+00
     .,0.33721600D+03,0.137D+03,0.133D+03,0.97380000D+00,0.98890000D+00
     .,0.30472790D+03,0.137D+03,0.134D+03,0.97380000D+00,0.99010000D+00
     .,0.26546470D+03,0.137D+03,0.135D+03,0.97380000D+00,0.99740000D+00
     .,0.47625910D+03,0.137D+03,0.137D+03,0.97380000D+00,0.97380000D+00
     .,0.64406500D+02,0.138D+03,0.100D+01,0.98010000D+00,0.91180000D+00
     .,0.38833300D+02,0.138D+03,0.200D+01,0.98010000D+00,0.00000000D+00
     .,0.16182465D+04,0.138D+03,0.300D+01,0.98010000D+00,0.00000000D+00
     .,0.73479260D+03,0.138D+03,0.400D+01,0.98010000D+00,0.00000000D+00
     .,0.44112610D+03,0.138D+03,0.500D+01,0.98010000D+00,0.00000000D+00
     .,0.27470310D+03,0.138D+03,0.600D+01,0.98010000D+00,0.00000000D+00
     .,0.18119730D+03,0.138D+03,0.700D+01,0.98010000D+00,0.00000000D+00
     .,0.13178460D+03,0.138D+03,0.800D+01,0.98010000D+00,0.00000000D+00
     .,0.96508000D+02,0.138D+03,0.900D+01,0.98010000D+00,0.00000000D+00
     .,0.72270300D+02,0.138D+03,0.100D+02,0.98010000D+00,0.00000000D+00
     .,0.19079592D+04,0.138D+03,0.110D+02,0.98010000D+00,0.00000000D+00
     .,0.12181274D+04,0.138D+03,0.120D+02,0.98010000D+00,0.00000000D+00
     ./)
      pars(  40701:  40800)=(/
     . 0.10570218D+04,0.138D+03,0.130D+02,0.98010000D+00,0.00000000D+00
     .,0.76638470D+03,0.138D+03,0.140D+02,0.98010000D+00,0.00000000D+00
     .,0.55938210D+03,0.138D+03,0.150D+02,0.98010000D+00,0.00000000D+00
     .,0.44578600D+03,0.138D+03,0.160D+02,0.98010000D+00,0.00000000D+00
     .,0.35042730D+03,0.138D+03,0.170D+02,0.98010000D+00,0.00000000D+00
     .,0.27747590D+03,0.138D+03,0.180D+02,0.98010000D+00,0.00000000D+00
     .,0.33022504D+04,0.138D+03,0.190D+02,0.98010000D+00,0.00000000D+00
     .,0.23148420D+04,0.138D+03,0.200D+02,0.98010000D+00,0.00000000D+00
     .,0.18497319D+04,0.138D+03,0.210D+02,0.98010000D+00,0.00000000D+00
     .,0.17407410D+04,0.138D+03,0.220D+02,0.98010000D+00,0.00000000D+00
     .,0.15682263D+04,0.138D+03,0.230D+02,0.98010000D+00,0.00000000D+00
     .,0.12385036D+04,0.138D+03,0.240D+02,0.98010000D+00,0.00000000D+00
     .,0.13188096D+04,0.138D+03,0.250D+02,0.98010000D+00,0.00000000D+00
     .,0.10340602D+04,0.138D+03,0.260D+02,0.98010000D+00,0.00000000D+00
     .,0.10522336D+04,0.138D+03,0.270D+02,0.98010000D+00,0.00000000D+00
     .,0.11016666D+04,0.138D+03,0.280D+02,0.98010000D+00,0.00000000D+00
     .,0.84879520D+03,0.138D+03,0.290D+02,0.98010000D+00,0.00000000D+00
     .,0.82251670D+03,0.138D+03,0.300D+02,0.98010000D+00,0.00000000D+00
     .,0.99191230D+03,0.138D+03,0.310D+02,0.98010000D+00,0.00000000D+00
     .,0.81848870D+03,0.138D+03,0.320D+02,0.98010000D+00,0.00000000D+00
     ./)
      pars(  40801:  40900)=(/
     . 0.66009280D+03,0.138D+03,0.330D+02,0.98010000D+00,0.00000000D+00
     .,0.57308500D+03,0.138D+03,0.340D+02,0.98010000D+00,0.00000000D+00
     .,0.48506500D+03,0.138D+03,0.350D+02,0.98010000D+00,0.00000000D+00
     .,0.40937610D+03,0.138D+03,0.360D+02,0.98010000D+00,0.00000000D+00
     .,0.36709261D+04,0.138D+03,0.370D+02,0.98010000D+00,0.00000000D+00
     .,0.27824577D+04,0.138D+03,0.380D+02,0.98010000D+00,0.00000000D+00
     .,0.23056086D+04,0.138D+03,0.390D+02,0.98010000D+00,0.00000000D+00
     .,0.20037829D+04,0.138D+03,0.400D+02,0.98010000D+00,0.00000000D+00
     .,0.17897638D+04,0.138D+03,0.410D+02,0.98010000D+00,0.00000000D+00
     .,0.13350700D+04,0.138D+03,0.420D+02,0.98010000D+00,0.00000000D+00
     .,0.15092981D+04,0.138D+03,0.430D+02,0.98010000D+00,0.00000000D+00
     .,0.11061259D+04,0.138D+03,0.440D+02,0.98010000D+00,0.00000000D+00
     .,0.12034301D+04,0.138D+03,0.450D+02,0.98010000D+00,0.00000000D+00
     .,0.11005370D+04,0.138D+03,0.460D+02,0.98010000D+00,0.00000000D+00
     .,0.93160220D+03,0.138D+03,0.470D+02,0.98010000D+00,0.00000000D+00
     .,0.95275330D+03,0.138D+03,0.480D+02,0.98010000D+00,0.00000000D+00
     .,0.12501173D+04,0.138D+03,0.490D+02,0.98010000D+00,0.00000000D+00
     .,0.10916636D+04,0.138D+03,0.500D+02,0.98010000D+00,0.00000000D+00
     .,0.92413690D+03,0.138D+03,0.510D+02,0.98010000D+00,0.00000000D+00
     .,0.83250600D+03,0.138D+03,0.520D+02,0.98010000D+00,0.00000000D+00
     ./)
      pars(  40901:  41000)=(/
     . 0.72971760D+03,0.138D+03,0.530D+02,0.98010000D+00,0.00000000D+00
     .,0.63746600D+03,0.138D+03,0.540D+02,0.98010000D+00,0.00000000D+00
     .,0.44969846D+04,0.138D+03,0.550D+02,0.98010000D+00,0.00000000D+00
     .,0.36314540D+04,0.138D+03,0.560D+02,0.98010000D+00,0.00000000D+00
     .,0.30140781D+04,0.138D+03,0.570D+02,0.98010000D+00,0.00000000D+00
     .,0.11044325D+04,0.138D+03,0.580D+02,0.98010000D+00,0.27991000D+01
     .,0.31621198D+04,0.138D+03,0.590D+02,0.98010000D+00,0.00000000D+00
     .,0.29999283D+04,0.138D+03,0.600D+02,0.98010000D+00,0.00000000D+00
     .,0.29157949D+04,0.138D+03,0.610D+02,0.98010000D+00,0.00000000D+00
     .,0.28393501D+04,0.138D+03,0.620D+02,0.98010000D+00,0.00000000D+00
     .,0.27713234D+04,0.138D+03,0.630D+02,0.98010000D+00,0.00000000D+00
     .,0.20594692D+04,0.138D+03,0.640D+02,0.98010000D+00,0.00000000D+00
     .,0.25668729D+04,0.138D+03,0.650D+02,0.98010000D+00,0.00000000D+00
     .,0.24578402D+04,0.138D+03,0.660D+02,0.98010000D+00,0.00000000D+00
     .,0.24609470D+04,0.138D+03,0.670D+02,0.98010000D+00,0.00000000D+00
     .,0.24041111D+04,0.138D+03,0.680D+02,0.98010000D+00,0.00000000D+00
     .,0.23512803D+04,0.138D+03,0.690D+02,0.98010000D+00,0.00000000D+00
     .,0.23289369D+04,0.138D+03,0.700D+02,0.98010000D+00,0.00000000D+00
     .,0.18927447D+04,0.138D+03,0.710D+02,0.98010000D+00,0.00000000D+00
     .,0.17529565D+04,0.138D+03,0.720D+02,0.98010000D+00,0.00000000D+00
     ./)
      pars(  41001:  41100)=(/
     . 0.15479897D+04,0.138D+03,0.730D+02,0.98010000D+00,0.00000000D+00
     .,0.12804929D+04,0.138D+03,0.740D+02,0.98010000D+00,0.00000000D+00
     .,0.12840434D+04,0.138D+03,0.750D+02,0.98010000D+00,0.00000000D+00
     .,0.11312913D+04,0.138D+03,0.760D+02,0.98010000D+00,0.00000000D+00
     .,0.10131095D+04,0.138D+03,0.770D+02,0.98010000D+00,0.00000000D+00
     .,0.82466090D+03,0.138D+03,0.780D+02,0.98010000D+00,0.00000000D+00
     .,0.76388530D+03,0.138D+03,0.790D+02,0.98010000D+00,0.00000000D+00
     .,0.77550460D+03,0.138D+03,0.800D+02,0.98010000D+00,0.00000000D+00
     .,0.12742283D+04,0.138D+03,0.810D+02,0.98010000D+00,0.00000000D+00
     .,0.11873359D+04,0.138D+03,0.820D+02,0.98010000D+00,0.00000000D+00
     .,0.10400523D+04,0.138D+03,0.830D+02,0.98010000D+00,0.00000000D+00
     .,0.96630350D+03,0.138D+03,0.840D+02,0.98010000D+00,0.00000000D+00
     .,0.86500840D+03,0.138D+03,0.850D+02,0.98010000D+00,0.00000000D+00
     .,0.77178770D+03,0.138D+03,0.860D+02,0.98010000D+00,0.00000000D+00
     .,0.40533897D+04,0.138D+03,0.870D+02,0.98010000D+00,0.00000000D+00
     .,0.34910837D+04,0.138D+03,0.880D+02,0.98010000D+00,0.00000000D+00
     .,0.29248174D+04,0.138D+03,0.890D+02,0.98010000D+00,0.00000000D+00
     .,0.24812627D+04,0.138D+03,0.900D+02,0.98010000D+00,0.00000000D+00
     .,0.25425532D+04,0.138D+03,0.910D+02,0.98010000D+00,0.00000000D+00
     .,0.24572393D+04,0.138D+03,0.920D+02,0.98010000D+00,0.00000000D+00
     ./)
      pars(  41101:  41200)=(/
     . 0.26180036D+04,0.138D+03,0.930D+02,0.98010000D+00,0.00000000D+00
     .,0.25175053D+04,0.138D+03,0.940D+02,0.98010000D+00,0.00000000D+00
     .,0.11032750D+03,0.138D+03,0.101D+03,0.98010000D+00,0.00000000D+00
     .,0.42146100D+03,0.138D+03,0.103D+03,0.98010000D+00,0.98650000D+00
     .,0.52954940D+03,0.138D+03,0.104D+03,0.98010000D+00,0.98080000D+00
     .,0.36336070D+03,0.138D+03,0.105D+03,0.98010000D+00,0.97060000D+00
     .,0.25898850D+03,0.138D+03,0.106D+03,0.98010000D+00,0.98680000D+00
     .,0.16917720D+03,0.138D+03,0.107D+03,0.98010000D+00,0.99440000D+00
     .,0.11710180D+03,0.138D+03,0.108D+03,0.98010000D+00,0.99250000D+00
     .,0.75675800D+02,0.138D+03,0.109D+03,0.98010000D+00,0.99820000D+00
     .,0.63083040D+03,0.138D+03,0.111D+03,0.98010000D+00,0.96840000D+00
     .,0.98878570D+03,0.138D+03,0.112D+03,0.98010000D+00,0.96280000D+00
     .,0.94475350D+03,0.138D+03,0.113D+03,0.98010000D+00,0.96480000D+00
     .,0.70100090D+03,0.138D+03,0.114D+03,0.98010000D+00,0.95070000D+00
     .,0.54212620D+03,0.138D+03,0.115D+03,0.98010000D+00,0.99470000D+00
     .,0.44147190D+03,0.138D+03,0.116D+03,0.98010000D+00,0.99480000D+00
     .,0.34712550D+03,0.138D+03,0.117D+03,0.98010000D+00,0.99720000D+00
     .,0.84199830D+03,0.138D+03,0.119D+03,0.98010000D+00,0.97670000D+00
     .,0.18769417D+04,0.138D+03,0.120D+03,0.98010000D+00,0.98310000D+00
     .,0.77994880D+03,0.138D+03,0.121D+03,0.98010000D+00,0.18627000D+01
     ./)
      pars(  41201:  41300)=(/
     . 0.75467530D+03,0.138D+03,0.122D+03,0.98010000D+00,0.18299000D+01
     .,0.74017630D+03,0.138D+03,0.123D+03,0.98010000D+00,0.19138000D+01
     .,0.73983150D+03,0.138D+03,0.124D+03,0.98010000D+00,0.18269000D+01
     .,0.65184050D+03,0.138D+03,0.125D+03,0.98010000D+00,0.16406000D+01
     .,0.59697670D+03,0.138D+03,0.126D+03,0.98010000D+00,0.16483000D+01
     .,0.57030240D+03,0.138D+03,0.127D+03,0.98010000D+00,0.17149000D+01
     .,0.55938830D+03,0.138D+03,0.128D+03,0.98010000D+00,0.17937000D+01
     .,0.56991510D+03,0.138D+03,0.129D+03,0.98010000D+00,0.95760000D+00
     .,0.50526080D+03,0.138D+03,0.130D+03,0.98010000D+00,0.19419000D+01
     .,0.91623120D+03,0.138D+03,0.131D+03,0.98010000D+00,0.96010000D+00
     .,0.75505840D+03,0.138D+03,0.132D+03,0.98010000D+00,0.94340000D+00
     .,0.64702200D+03,0.138D+03,0.133D+03,0.98010000D+00,0.98890000D+00
     .,0.57371350D+03,0.138D+03,0.134D+03,0.98010000D+00,0.99010000D+00
     .,0.48934610D+03,0.138D+03,0.135D+03,0.98010000D+00,0.99740000D+00
     .,0.99279700D+03,0.138D+03,0.137D+03,0.98010000D+00,0.97380000D+00
     .,0.23126602D+04,0.138D+03,0.138D+03,0.98010000D+00,0.98010000D+00
     .,0.51235600D+02,0.139D+03,0.100D+01,0.19153000D+01,0.91180000D+00
     .,0.31988900D+02,0.139D+03,0.200D+01,0.19153000D+01,0.00000000D+00
     .,0.10745875D+04,0.139D+03,0.300D+01,0.19153000D+01,0.00000000D+00
     .,0.53251100D+03,0.139D+03,0.400D+01,0.19153000D+01,0.00000000D+00
     ./)
      pars(  41301:  41400)=(/
     . 0.33413510D+03,0.139D+03,0.500D+01,0.19153000D+01,0.00000000D+00
     .,0.21477600D+03,0.139D+03,0.600D+01,0.19153000D+01,0.00000000D+00
     .,0.14492070D+03,0.139D+03,0.700D+01,0.19153000D+01,0.00000000D+00
     .,0.10704380D+03,0.139D+03,0.800D+01,0.19153000D+01,0.00000000D+00
     .,0.79407600D+02,0.139D+03,0.900D+01,0.19153000D+01,0.00000000D+00
     .,0.60062800D+02,0.139D+03,0.100D+02,0.19153000D+01,0.00000000D+00
     .,0.12727028D+04,0.139D+03,0.110D+02,0.19153000D+01,0.00000000D+00
     .,0.86937390D+03,0.139D+03,0.120D+02,0.19153000D+01,0.00000000D+00
     .,0.77186060D+03,0.139D+03,0.130D+02,0.19153000D+01,0.00000000D+00
     .,0.57805500D+03,0.139D+03,0.140D+02,0.19153000D+01,0.00000000D+00
     .,0.43301710D+03,0.139D+03,0.150D+02,0.19153000D+01,0.00000000D+00
     .,0.35057750D+03,0.139D+03,0.160D+02,0.19153000D+01,0.00000000D+00
     .,0.27973550D+03,0.139D+03,0.170D+02,0.19153000D+01,0.00000000D+00
     .,0.22431260D+03,0.139D+03,0.180D+02,0.19153000D+01,0.00000000D+00
     .,0.21637031D+04,0.139D+03,0.190D+02,0.19153000D+01,0.00000000D+00
     .,0.16036362D+04,0.139D+03,0.200D+02,0.19153000D+01,0.00000000D+00
     .,0.12972964D+04,0.139D+03,0.210D+02,0.19153000D+01,0.00000000D+00
     .,0.12323433D+04,0.139D+03,0.220D+02,0.19153000D+01,0.00000000D+00
     .,0.11169531D+04,0.139D+03,0.230D+02,0.19153000D+01,0.00000000D+00
     .,0.88060690D+03,0.139D+03,0.240D+02,0.19153000D+01,0.00000000D+00
     ./)
      pars(  41401:  41500)=(/
     . 0.94758340D+03,0.139D+03,0.250D+02,0.19153000D+01,0.00000000D+00
     .,0.74266410D+03,0.139D+03,0.260D+02,0.19153000D+01,0.00000000D+00
     .,0.76821070D+03,0.139D+03,0.270D+02,0.19153000D+01,0.00000000D+00
     .,0.79937100D+03,0.139D+03,0.280D+02,0.19153000D+01,0.00000000D+00
     .,0.61409830D+03,0.139D+03,0.290D+02,0.19153000D+01,0.00000000D+00
     .,0.60904990D+03,0.139D+03,0.300D+02,0.19153000D+01,0.00000000D+00
     .,0.72958130D+03,0.139D+03,0.310D+02,0.19153000D+01,0.00000000D+00
     .,0.61799850D+03,0.139D+03,0.320D+02,0.19153000D+01,0.00000000D+00
     .,0.50955030D+03,0.139D+03,0.330D+02,0.19153000D+01,0.00000000D+00
     .,0.44813810D+03,0.139D+03,0.340D+02,0.19153000D+01,0.00000000D+00
     .,0.38429010D+03,0.139D+03,0.350D+02,0.19153000D+01,0.00000000D+00
     .,0.32813160D+03,0.139D+03,0.360D+02,0.19153000D+01,0.00000000D+00
     .,0.24114577D+04,0.139D+03,0.370D+02,0.19153000D+01,0.00000000D+00
     .,0.19209295D+04,0.139D+03,0.380D+02,0.19153000D+01,0.00000000D+00
     .,0.16243636D+04,0.139D+03,0.390D+02,0.19153000D+01,0.00000000D+00
     .,0.14293576D+04,0.139D+03,0.400D+02,0.19153000D+01,0.00000000D+00
     .,0.12864770D+04,0.139D+03,0.410D+02,0.19153000D+01,0.00000000D+00
     .,0.97159590D+03,0.139D+03,0.420D+02,0.19153000D+01,0.00000000D+00
     .,0.10932424D+04,0.139D+03,0.430D+02,0.19153000D+01,0.00000000D+00
     .,0.81262860D+03,0.139D+03,0.440D+02,0.19153000D+01,0.00000000D+00
     ./)
      pars(  41501:  41600)=(/
     . 0.88627500D+03,0.139D+03,0.450D+02,0.19153000D+01,0.00000000D+00
     .,0.81484020D+03,0.139D+03,0.460D+02,0.19153000D+01,0.00000000D+00
     .,0.68497960D+03,0.139D+03,0.470D+02,0.19153000D+01,0.00000000D+00
     .,0.71021720D+03,0.139D+03,0.480D+02,0.19153000D+01,0.00000000D+00
     .,0.91615920D+03,0.139D+03,0.490D+02,0.19153000D+01,0.00000000D+00
     .,0.81841300D+03,0.139D+03,0.500D+02,0.19153000D+01,0.00000000D+00
     .,0.70709330D+03,0.139D+03,0.510D+02,0.19153000D+01,0.00000000D+00
     .,0.64444940D+03,0.139D+03,0.520D+02,0.19153000D+01,0.00000000D+00
     .,0.57184320D+03,0.139D+03,0.530D+02,0.19153000D+01,0.00000000D+00
     .,0.50522620D+03,0.139D+03,0.540D+02,0.19153000D+01,0.00000000D+00
     .,0.29502475D+04,0.139D+03,0.550D+02,0.19153000D+01,0.00000000D+00
     .,0.24856310D+04,0.139D+03,0.560D+02,0.19153000D+01,0.00000000D+00
     .,0.21057491D+04,0.139D+03,0.570D+02,0.19153000D+01,0.00000000D+00
     .,0.84084100D+03,0.139D+03,0.580D+02,0.19153000D+01,0.27991000D+01
     .,0.21783229D+04,0.139D+03,0.590D+02,0.19153000D+01,0.00000000D+00
     .,0.20748807D+04,0.139D+03,0.600D+02,0.19153000D+01,0.00000000D+00
     .,0.20189224D+04,0.139D+03,0.610D+02,0.19153000D+01,0.00000000D+00
     .,0.19678827D+04,0.139D+03,0.620D+02,0.19153000D+01,0.00000000D+00
     .,0.19225226D+04,0.139D+03,0.630D+02,0.19153000D+01,0.00000000D+00
     .,0.14581969D+04,0.139D+03,0.640D+02,0.19153000D+01,0.00000000D+00
     ./)
      pars(  41601:  41700)=(/
     . 0.17521144D+04,0.139D+03,0.650D+02,0.19153000D+01,0.00000000D+00
     .,0.16830549D+04,0.139D+03,0.660D+02,0.19153000D+01,0.00000000D+00
     .,0.17170123D+04,0.139D+03,0.670D+02,0.19153000D+01,0.00000000D+00
     .,0.16785781D+04,0.139D+03,0.680D+02,0.19153000D+01,0.00000000D+00
     .,0.16432139D+04,0.139D+03,0.690D+02,0.19153000D+01,0.00000000D+00
     .,0.16263339D+04,0.139D+03,0.700D+02,0.19153000D+01,0.00000000D+00
     .,0.13398149D+04,0.139D+03,0.710D+02,0.19153000D+01,0.00000000D+00
     .,0.12689492D+04,0.139D+03,0.720D+02,0.19153000D+01,0.00000000D+00
     .,0.11346407D+04,0.139D+03,0.730D+02,0.19153000D+01,0.00000000D+00
     .,0.94540990D+03,0.139D+03,0.740D+02,0.19153000D+01,0.00000000D+00
     .,0.95356340D+03,0.139D+03,0.750D+02,0.19153000D+01,0.00000000D+00
     .,0.84938170D+03,0.139D+03,0.760D+02,0.19153000D+01,0.00000000D+00
     .,0.76730030D+03,0.139D+03,0.770D+02,0.19153000D+01,0.00000000D+00
     .,0.62916050D+03,0.139D+03,0.780D+02,0.19153000D+01,0.00000000D+00
     .,0.58463530D+03,0.139D+03,0.790D+02,0.19153000D+01,0.00000000D+00
     .,0.59694450D+03,0.139D+03,0.800D+02,0.19153000D+01,0.00000000D+00
     .,0.93562990D+03,0.139D+03,0.810D+02,0.19153000D+01,0.00000000D+00
     .,0.88887560D+03,0.139D+03,0.820D+02,0.19153000D+01,0.00000000D+00
     .,0.79367010D+03,0.139D+03,0.830D+02,0.19153000D+01,0.00000000D+00
     .,0.74514710D+03,0.139D+03,0.840D+02,0.19153000D+01,0.00000000D+00
     ./)
      pars(  41701:  41800)=(/
     . 0.67518420D+03,0.139D+03,0.850D+02,0.19153000D+01,0.00000000D+00
     .,0.60886250D+03,0.139D+03,0.860D+02,0.19153000D+01,0.00000000D+00
     .,0.26980970D+04,0.139D+03,0.870D+02,0.19153000D+01,0.00000000D+00
     .,0.24134225D+04,0.139D+03,0.880D+02,0.19153000D+01,0.00000000D+00
     .,0.20622685D+04,0.139D+03,0.890D+02,0.19153000D+01,0.00000000D+00
     .,0.17871863D+04,0.139D+03,0.900D+02,0.19153000D+01,0.00000000D+00
     .,0.18092861D+04,0.139D+03,0.910D+02,0.19153000D+01,0.00000000D+00
     .,0.17497379D+04,0.139D+03,0.920D+02,0.19153000D+01,0.00000000D+00
     .,0.18406901D+04,0.139D+03,0.930D+02,0.19153000D+01,0.00000000D+00
     .,0.17746391D+04,0.139D+03,0.940D+02,0.19153000D+01,0.00000000D+00
     .,0.85681500D+02,0.139D+03,0.101D+03,0.19153000D+01,0.00000000D+00
     .,0.30688620D+03,0.139D+03,0.103D+03,0.19153000D+01,0.98650000D+00
     .,0.38765640D+03,0.139D+03,0.104D+03,0.19153000D+01,0.98080000D+00
     .,0.27740500D+03,0.139D+03,0.105D+03,0.19153000D+01,0.97060000D+00
     .,0.20197440D+03,0.139D+03,0.106D+03,0.19153000D+01,0.98680000D+00
     .,0.13518390D+03,0.139D+03,0.107D+03,0.19153000D+01,0.99440000D+00
     .,0.95445800D+02,0.139D+03,0.108D+03,0.19153000D+01,0.99250000D+00
     .,0.63188900D+02,0.139D+03,0.109D+03,0.19153000D+01,0.99820000D+00
     .,0.45499280D+03,0.139D+03,0.111D+03,0.19153000D+01,0.96840000D+00
     .,0.70969610D+03,0.139D+03,0.112D+03,0.19153000D+01,0.96280000D+00
     ./)
      pars(  41801:  41900)=(/
     . 0.69347320D+03,0.139D+03,0.113D+03,0.19153000D+01,0.96480000D+00
     .,0.53081800D+03,0.139D+03,0.114D+03,0.19153000D+01,0.95070000D+00
     .,0.41987180D+03,0.139D+03,0.115D+03,0.19153000D+01,0.99470000D+00
     .,0.34697100D+03,0.139D+03,0.116D+03,0.19153000D+01,0.99480000D+00
     .,0.27697730D+03,0.139D+03,0.117D+03,0.19153000D+01,0.99720000D+00
     .,0.61384630D+03,0.139D+03,0.119D+03,0.19153000D+01,0.97670000D+00
     .,0.12935159D+04,0.139D+03,0.120D+03,0.19153000D+01,0.98310000D+00
     .,0.58631350D+03,0.139D+03,0.121D+03,0.19153000D+01,0.18627000D+01
     .,0.56686700D+03,0.139D+03,0.122D+03,0.19153000D+01,0.18299000D+01
     .,0.55572150D+03,0.139D+03,0.123D+03,0.19153000D+01,0.19138000D+01
     .,0.55347640D+03,0.139D+03,0.124D+03,0.19153000D+01,0.18269000D+01
     .,0.49633310D+03,0.139D+03,0.125D+03,0.19153000D+01,0.16406000D+01
     .,0.45638780D+03,0.139D+03,0.126D+03,0.19153000D+01,0.16483000D+01
     .,0.43567010D+03,0.139D+03,0.127D+03,0.19153000D+01,0.17149000D+01
     .,0.42674050D+03,0.139D+03,0.128D+03,0.19153000D+01,0.17937000D+01
     .,0.42937430D+03,0.139D+03,0.129D+03,0.19153000D+01,0.95760000D+00
     .,0.38958110D+03,0.139D+03,0.130D+03,0.19153000D+01,0.19419000D+01
     .,0.67825900D+03,0.139D+03,0.131D+03,0.19153000D+01,0.96010000D+00
     .,0.57321390D+03,0.139D+03,0.132D+03,0.19153000D+01,0.94340000D+00
     .,0.50000710D+03,0.139D+03,0.133D+03,0.19153000D+01,0.98890000D+00
     ./)
      pars(  41901:  42000)=(/
     . 0.44848720D+03,0.139D+03,0.134D+03,0.19153000D+01,0.99010000D+00
     .,0.38738760D+03,0.139D+03,0.135D+03,0.19153000D+01,0.99740000D+00
     .,0.72686130D+03,0.139D+03,0.137D+03,0.19153000D+01,0.97380000D+00
     .,0.15865402D+04,0.139D+03,0.138D+03,0.19153000D+01,0.98010000D+00
     .,0.11322983D+04,0.139D+03,0.139D+03,0.19153000D+01,0.19153000D+01
     .,0.39666900D+02,0.140D+03,0.100D+01,0.19355000D+01,0.91180000D+00
     .,0.25703700D+02,0.140D+03,0.200D+01,0.19355000D+01,0.00000000D+00
     .,0.67848100D+03,0.140D+03,0.300D+01,0.19355000D+01,0.00000000D+00
     .,0.37401350D+03,0.140D+03,0.400D+01,0.19355000D+01,0.00000000D+00
     .,0.24601470D+03,0.140D+03,0.500D+01,0.19355000D+01,0.00000000D+00
     .,0.16345480D+03,0.140D+03,0.600D+01,0.19355000D+01,0.00000000D+00
     .,0.11293390D+03,0.140D+03,0.700D+01,0.19355000D+01,0.00000000D+00
     .,0.84787600D+02,0.140D+03,0.800D+01,0.19355000D+01,0.00000000D+00
     .,0.63768000D+02,0.140D+03,0.900D+01,0.19355000D+01,0.00000000D+00
     .,0.48760000D+02,0.140D+03,0.100D+02,0.19355000D+01,0.00000000D+00
     .,0.80899730D+03,0.140D+03,0.110D+02,0.19355000D+01,0.00000000D+00
     .,0.60055110D+03,0.140D+03,0.120D+02,0.19355000D+01,0.00000000D+00
     .,0.54671760D+03,0.140D+03,0.130D+02,0.19355000D+01,0.00000000D+00
     .,0.42369920D+03,0.140D+03,0.140D+02,0.19355000D+01,0.00000000D+00
     .,0.32600870D+03,0.140D+03,0.150D+02,0.19355000D+01,0.00000000D+00
     ./)
      pars(  42001:  42100)=(/
     . 0.26830770D+03,0.140D+03,0.160D+02,0.19355000D+01,0.00000000D+00
     .,0.21743220D+03,0.140D+03,0.170D+02,0.19355000D+01,0.00000000D+00
     .,0.17666900D+03,0.140D+03,0.180D+02,0.19355000D+01,0.00000000D+00
     .,0.13359899D+04,0.140D+03,0.190D+02,0.19355000D+01,0.00000000D+00
     .,0.10709104D+04,0.140D+03,0.200D+02,0.19355000D+01,0.00000000D+00
     .,0.87901650D+03,0.140D+03,0.210D+02,0.19355000D+01,0.00000000D+00
     .,0.84415650D+03,0.140D+03,0.220D+02,0.19355000D+01,0.00000000D+00
     .,0.77041850D+03,0.140D+03,0.230D+02,0.19355000D+01,0.00000000D+00
     .,0.60670400D+03,0.140D+03,0.240D+02,0.19355000D+01,0.00000000D+00
     .,0.66010050D+03,0.140D+03,0.250D+02,0.19355000D+01,0.00000000D+00
     .,0.51758120D+03,0.140D+03,0.260D+02,0.19355000D+01,0.00000000D+00
     .,0.54466430D+03,0.140D+03,0.270D+02,0.19355000D+01,0.00000000D+00
     .,0.56293440D+03,0.140D+03,0.280D+02,0.19355000D+01,0.00000000D+00
     .,0.43155440D+03,0.140D+03,0.290D+02,0.19355000D+01,0.00000000D+00
     .,0.43840800D+03,0.140D+03,0.300D+02,0.19355000D+01,0.00000000D+00
     .,0.52118020D+03,0.140D+03,0.310D+02,0.19355000D+01,0.00000000D+00
     .,0.45369050D+03,0.140D+03,0.320D+02,0.19355000D+01,0.00000000D+00
     .,0.38264580D+03,0.140D+03,0.330D+02,0.19355000D+01,0.00000000D+00
     .,0.34103380D+03,0.140D+03,0.340D+02,0.19355000D+01,0.00000000D+00
     .,0.29639440D+03,0.140D+03,0.350D+02,0.19355000D+01,0.00000000D+00
     ./)
      pars(  42101:  42200)=(/
     . 0.25615010D+03,0.140D+03,0.360D+02,0.19355000D+01,0.00000000D+00
     .,0.14937993D+04,0.140D+03,0.370D+02,0.19355000D+01,0.00000000D+00
     .,0.12775813D+04,0.140D+03,0.380D+02,0.19355000D+01,0.00000000D+00
     .,0.11066630D+04,0.140D+03,0.390D+02,0.19355000D+01,0.00000000D+00
     .,0.98782570D+03,0.140D+03,0.400D+02,0.19355000D+01,0.00000000D+00
     .,0.89679520D+03,0.140D+03,0.410D+02,0.19355000D+01,0.00000000D+00
     .,0.68718480D+03,0.140D+03,0.420D+02,0.19355000D+01,0.00000000D+00
     .,0.76884280D+03,0.140D+03,0.430D+02,0.19355000D+01,0.00000000D+00
     .,0.58090070D+03,0.140D+03,0.440D+02,0.19355000D+01,0.00000000D+00
     .,0.63480240D+03,0.140D+03,0.450D+02,0.19355000D+01,0.00000000D+00
     .,0.58702020D+03,0.140D+03,0.460D+02,0.19355000D+01,0.00000000D+00
     .,0.49038170D+03,0.140D+03,0.470D+02,0.19355000D+01,0.00000000D+00
     .,0.51538610D+03,0.140D+03,0.480D+02,0.19355000D+01,0.00000000D+00
     .,0.65256540D+03,0.140D+03,0.490D+02,0.19355000D+01,0.00000000D+00
     .,0.59689190D+03,0.140D+03,0.500D+02,0.19355000D+01,0.00000000D+00
     .,0.52654260D+03,0.140D+03,0.510D+02,0.19355000D+01,0.00000000D+00
     .,0.48561760D+03,0.140D+03,0.520D+02,0.19355000D+01,0.00000000D+00
     .,0.43630460D+03,0.140D+03,0.530D+02,0.19355000D+01,0.00000000D+00
     .,0.38993850D+03,0.140D+03,0.540D+02,0.19355000D+01,0.00000000D+00
     .,0.18190406D+04,0.140D+03,0.550D+02,0.19355000D+01,0.00000000D+00
     ./)
      pars(  42201:  42300)=(/
     . 0.16353622D+04,0.140D+03,0.560D+02,0.19355000D+01,0.00000000D+00
     .,0.14206192D+04,0.140D+03,0.570D+02,0.19355000D+01,0.00000000D+00
     .,0.62328410D+03,0.140D+03,0.580D+02,0.19355000D+01,0.27991000D+01
     .,0.14429680D+04,0.140D+03,0.590D+02,0.19355000D+01,0.00000000D+00
     .,0.13832190D+04,0.140D+03,0.600D+02,0.19355000D+01,0.00000000D+00
     .,0.13477804D+04,0.140D+03,0.610D+02,0.19355000D+01,0.00000000D+00
     .,0.13152670D+04,0.140D+03,0.620D+02,0.19355000D+01,0.00000000D+00
     .,0.12864106D+04,0.140D+03,0.630D+02,0.19355000D+01,0.00000000D+00
     .,0.99936150D+03,0.140D+03,0.640D+02,0.19355000D+01,0.00000000D+00
     .,0.11461276D+04,0.140D+03,0.650D+02,0.19355000D+01,0.00000000D+00
     .,0.11025614D+04,0.140D+03,0.660D+02,0.19355000D+01,0.00000000D+00
     .,0.11568729D+04,0.140D+03,0.670D+02,0.19355000D+01,0.00000000D+00
     .,0.11319421D+04,0.140D+03,0.680D+02,0.19355000D+01,0.00000000D+00
     .,0.11093035D+04,0.140D+03,0.690D+02,0.19355000D+01,0.00000000D+00
     .,0.10968681D+04,0.140D+03,0.700D+02,0.19355000D+01,0.00000000D+00
     .,0.91629250D+03,0.140D+03,0.710D+02,0.19355000D+01,0.00000000D+00
     .,0.89141000D+03,0.140D+03,0.720D+02,0.19355000D+01,0.00000000D+00
     .,0.80810660D+03,0.140D+03,0.730D+02,0.19355000D+01,0.00000000D+00
     .,0.67904580D+03,0.140D+03,0.740D+02,0.19355000D+01,0.00000000D+00
     .,0.68897190D+03,0.140D+03,0.750D+02,0.19355000D+01,0.00000000D+00
     ./)
      pars(  42301:  42400)=(/
     . 0.62088940D+03,0.140D+03,0.760D+02,0.19355000D+01,0.00000000D+00
     .,0.56604620D+03,0.140D+03,0.770D+02,0.19355000D+01,0.00000000D+00
     .,0.46802460D+03,0.140D+03,0.780D+02,0.19355000D+01,0.00000000D+00
     .,0.43643220D+03,0.140D+03,0.790D+02,0.19355000D+01,0.00000000D+00
     .,0.44796630D+03,0.140D+03,0.800D+02,0.19355000D+01,0.00000000D+00
     .,0.66836720D+03,0.140D+03,0.810D+02,0.19355000D+01,0.00000000D+00
     .,0.64781650D+03,0.140D+03,0.820D+02,0.19355000D+01,0.00000000D+00
     .,0.58976480D+03,0.140D+03,0.830D+02,0.19355000D+01,0.00000000D+00
     .,0.55956430D+03,0.140D+03,0.840D+02,0.19355000D+01,0.00000000D+00
     .,0.51325740D+03,0.140D+03,0.850D+02,0.19355000D+01,0.00000000D+00
     .,0.46784580D+03,0.140D+03,0.860D+02,0.19355000D+01,0.00000000D+00
     .,0.17028592D+04,0.140D+03,0.870D+02,0.19355000D+01,0.00000000D+00
     .,0.16078178D+04,0.140D+03,0.880D+02,0.19355000D+01,0.00000000D+00
     .,0.14061219D+04,0.140D+03,0.890D+02,0.19355000D+01,0.00000000D+00
     .,0.12490250D+04,0.140D+03,0.900D+02,0.19355000D+01,0.00000000D+00
     .,0.12473212D+04,0.140D+03,0.910D+02,0.19355000D+01,0.00000000D+00
     .,0.12073349D+04,0.140D+03,0.920D+02,0.19355000D+01,0.00000000D+00
     .,0.12519154D+04,0.140D+03,0.930D+02,0.19355000D+01,0.00000000D+00
     .,0.12106981D+04,0.140D+03,0.940D+02,0.19355000D+01,0.00000000D+00
     .,0.64706800D+02,0.140D+03,0.101D+03,0.19355000D+01,0.00000000D+00
     ./)
      pars(  42401:  42500)=(/
     . 0.21671700D+03,0.140D+03,0.103D+03,0.19355000D+01,0.98650000D+00
     .,0.27536440D+03,0.140D+03,0.104D+03,0.19355000D+01,0.98080000D+00
     .,0.20600140D+03,0.140D+03,0.105D+03,0.19355000D+01,0.97060000D+00
     .,0.15342290D+03,0.140D+03,0.106D+03,0.19355000D+01,0.98680000D+00
     .,0.10531510D+03,0.140D+03,0.107D+03,0.19355000D+01,0.99440000D+00
     .,0.75899400D+02,0.140D+03,0.108D+03,0.19355000D+01,0.99250000D+00
     .,0.51515800D+02,0.140D+03,0.109D+03,0.19355000D+01,0.99820000D+00
     .,0.31816780D+03,0.140D+03,0.111D+03,0.19355000D+01,0.96840000D+00
     .,0.49330570D+03,0.140D+03,0.112D+03,0.19355000D+01,0.96280000D+00
     .,0.49402450D+03,0.140D+03,0.113D+03,0.19355000D+01,0.96480000D+00
     .,0.39071920D+03,0.140D+03,0.114D+03,0.19355000D+01,0.95070000D+00
     .,0.31631690D+03,0.140D+03,0.115D+03,0.19355000D+01,0.99470000D+00
     .,0.26540890D+03,0.140D+03,0.116D+03,0.19355000D+01,0.99480000D+00
     .,0.21521010D+03,0.140D+03,0.117D+03,0.19355000D+01,0.99720000D+00
     .,0.43506490D+03,0.140D+03,0.119D+03,0.19355000D+01,0.97670000D+00
     .,0.85671310D+03,0.140D+03,0.120D+03,0.19355000D+01,0.98310000D+00
     .,0.42930190D+03,0.140D+03,0.121D+03,0.19355000D+01,0.18627000D+01
     .,0.41420300D+03,0.140D+03,0.122D+03,0.19355000D+01,0.18299000D+01
     .,0.40602230D+03,0.140D+03,0.123D+03,0.19355000D+01,0.19138000D+01
     .,0.40292000D+03,0.140D+03,0.124D+03,0.19355000D+01,0.18269000D+01
     ./)
      pars(  42501:  42600)=(/
     . 0.36779370D+03,0.140D+03,0.125D+03,0.19355000D+01,0.16406000D+01
     .,0.33962780D+03,0.140D+03,0.126D+03,0.19355000D+01,0.16483000D+01
     .,0.32399210D+03,0.140D+03,0.127D+03,0.19355000D+01,0.17149000D+01
     .,0.31694170D+03,0.140D+03,0.128D+03,0.19355000D+01,0.17937000D+01
     .,0.31499080D+03,0.140D+03,0.129D+03,0.19355000D+01,0.95760000D+00
     .,0.29240050D+03,0.140D+03,0.130D+03,0.19355000D+01,0.19419000D+01
     .,0.48787590D+03,0.140D+03,0.131D+03,0.19355000D+01,0.96010000D+00
     .,0.42320760D+03,0.140D+03,0.132D+03,0.19355000D+01,0.94340000D+00
     .,0.37591790D+03,0.140D+03,0.133D+03,0.19355000D+01,0.98890000D+00
     .,0.34120400D+03,0.140D+03,0.134D+03,0.19355000D+01,0.99010000D+00
     .,0.29856340D+03,0.140D+03,0.135D+03,0.19355000D+01,0.99740000D+00
     .,0.51769610D+03,0.140D+03,0.137D+03,0.19355000D+01,0.97380000D+00
     .,0.10443401D+04,0.140D+03,0.138D+03,0.19355000D+01,0.98010000D+00
     .,0.78082120D+03,0.140D+03,0.139D+03,0.19355000D+01,0.19153000D+01
     .,0.56867890D+03,0.140D+03,0.140D+03,0.19355000D+01,0.19355000D+01
     .,0.40030400D+02,0.141D+03,0.100D+01,0.19545000D+01,0.91180000D+00
     .,0.25953400D+02,0.141D+03,0.200D+01,0.19545000D+01,0.00000000D+00
     .,0.68410400D+03,0.141D+03,0.300D+01,0.19545000D+01,0.00000000D+00
     .,0.37796930D+03,0.141D+03,0.400D+01,0.19545000D+01,0.00000000D+00
     .,0.24842660D+03,0.141D+03,0.500D+01,0.19545000D+01,0.00000000D+00
     ./)
      pars(  42601:  42700)=(/
     . 0.16499800D+03,0.141D+03,0.600D+01,0.19545000D+01,0.00000000D+00
     .,0.11399890D+03,0.141D+03,0.700D+01,0.19545000D+01,0.00000000D+00
     .,0.85602400D+02,0.141D+03,0.800D+01,0.19545000D+01,0.00000000D+00
     .,0.64401000D+02,0.141D+03,0.900D+01,0.19545000D+01,0.00000000D+00
     .,0.49263500D+02,0.141D+03,0.100D+02,0.19545000D+01,0.00000000D+00
     .,0.81603170D+03,0.141D+03,0.110D+02,0.19545000D+01,0.00000000D+00
     .,0.60700730D+03,0.141D+03,0.120D+02,0.19545000D+01,0.00000000D+00
     .,0.55235500D+03,0.141D+03,0.130D+02,0.19545000D+01,0.00000000D+00
     .,0.42788370D+03,0.141D+03,0.140D+02,0.19545000D+01,0.00000000D+00
     .,0.32909470D+03,0.141D+03,0.150D+02,0.19545000D+01,0.00000000D+00
     .,0.27080590D+03,0.141D+03,0.160D+02,0.19545000D+01,0.00000000D+00
     .,0.21944620D+03,0.141D+03,0.170D+02,0.19545000D+01,0.00000000D+00
     .,0.17831830D+03,0.141D+03,0.180D+02,0.19545000D+01,0.00000000D+00
     .,0.13433714D+04,0.141D+03,0.190D+02,0.19545000D+01,0.00000000D+00
     .,0.10819930D+04,0.141D+03,0.200D+02,0.19545000D+01,0.00000000D+00
     .,0.88833560D+03,0.141D+03,0.210D+02,0.19545000D+01,0.00000000D+00
     .,0.85304640D+03,0.141D+03,0.220D+02,0.19545000D+01,0.00000000D+00
     .,0.77852660D+03,0.141D+03,0.230D+02,0.19545000D+01,0.00000000D+00
     .,0.61302240D+03,0.141D+03,0.240D+02,0.19545000D+01,0.00000000D+00
     .,0.66703050D+03,0.141D+03,0.250D+02,0.19545000D+01,0.00000000D+00
     ./)
      pars(  42701:  42800)=(/
     . 0.52300200D+03,0.141D+03,0.260D+02,0.19545000D+01,0.00000000D+00
     .,0.55040920D+03,0.141D+03,0.270D+02,0.19545000D+01,0.00000000D+00
     .,0.56890880D+03,0.141D+03,0.280D+02,0.19545000D+01,0.00000000D+00
     .,0.43610420D+03,0.141D+03,0.290D+02,0.19545000D+01,0.00000000D+00
     .,0.44298860D+03,0.141D+03,0.300D+02,0.19545000D+01,0.00000000D+00
     .,0.52649640D+03,0.141D+03,0.310D+02,0.19545000D+01,0.00000000D+00
     .,0.45819200D+03,0.141D+03,0.320D+02,0.19545000D+01,0.00000000D+00
     .,0.38630910D+03,0.141D+03,0.330D+02,0.19545000D+01,0.00000000D+00
     .,0.34424250D+03,0.141D+03,0.340D+02,0.19545000D+01,0.00000000D+00
     .,0.29915350D+03,0.141D+03,0.350D+02,0.19545000D+01,0.00000000D+00
     .,0.25853040D+03,0.141D+03,0.360D+02,0.19545000D+01,0.00000000D+00
     .,0.15017789D+04,0.141D+03,0.370D+02,0.19545000D+01,0.00000000D+00
     .,0.12904826D+04,0.141D+03,0.380D+02,0.19545000D+01,0.00000000D+00
     .,0.11181402D+04,0.141D+03,0.390D+02,0.19545000D+01,0.00000000D+00
     .,0.99812240D+03,0.141D+03,0.400D+02,0.19545000D+01,0.00000000D+00
     .,0.90608010D+03,0.141D+03,0.410D+02,0.19545000D+01,0.00000000D+00
     .,0.69429830D+03,0.141D+03,0.420D+02,0.19545000D+01,0.00000000D+00
     .,0.77673680D+03,0.141D+03,0.430D+02,0.19545000D+01,0.00000000D+00
     .,0.58689450D+03,0.141D+03,0.440D+02,0.19545000D+01,0.00000000D+00
     .,0.64141640D+03,0.141D+03,0.450D+02,0.19545000D+01,0.00000000D+00
     ./)
      pars(  42801:  42900)=(/
     . 0.59314140D+03,0.141D+03,0.460D+02,0.19545000D+01,0.00000000D+00
     .,0.49548000D+03,0.141D+03,0.470D+02,0.19545000D+01,0.00000000D+00
     .,0.52076290D+03,0.141D+03,0.480D+02,0.19545000D+01,0.00000000D+00
     .,0.65928650D+03,0.141D+03,0.490D+02,0.19545000D+01,0.00000000D+00
     .,0.60296920D+03,0.141D+03,0.500D+02,0.19545000D+01,0.00000000D+00
     .,0.53174240D+03,0.141D+03,0.510D+02,0.19545000D+01,0.00000000D+00
     .,0.49032060D+03,0.141D+03,0.520D+02,0.19545000D+01,0.00000000D+00
     .,0.44045990D+03,0.141D+03,0.530D+02,0.19545000D+01,0.00000000D+00
     .,0.39361210D+03,0.141D+03,0.540D+02,0.19545000D+01,0.00000000D+00
     .,0.18264461D+04,0.141D+03,0.550D+02,0.19545000D+01,0.00000000D+00
     .,0.16509334D+04,0.141D+03,0.560D+02,0.19545000D+01,0.00000000D+00
     .,0.14348956D+04,0.141D+03,0.570D+02,0.19545000D+01,0.00000000D+00
     .,0.62949770D+03,0.141D+03,0.580D+02,0.19545000D+01,0.27991000D+01
     .,0.14565113D+04,0.141D+03,0.590D+02,0.19545000D+01,0.00000000D+00
     .,0.13970298D+04,0.141D+03,0.600D+02,0.19545000D+01,0.00000000D+00
     .,0.13613178D+04,0.141D+03,0.610D+02,0.19545000D+01,0.00000000D+00
     .,0.13285415D+04,0.141D+03,0.620D+02,0.19545000D+01,0.00000000D+00
     .,0.12994488D+04,0.141D+03,0.630D+02,0.19545000D+01,0.00000000D+00
     .,0.10094419D+04,0.141D+03,0.640D+02,0.19545000D+01,0.00000000D+00
     .,0.11553947D+04,0.141D+03,0.650D+02,0.19545000D+01,0.00000000D+00
     ./)
      pars(  42901:  43000)=(/
     . 0.11106913D+04,0.141D+03,0.660D+02,0.19545000D+01,0.00000000D+00
     .,0.11688393D+04,0.141D+03,0.670D+02,0.19545000D+01,0.00000000D+00
     .,0.11436857D+04,0.141D+03,0.680D+02,0.19545000D+01,0.00000000D+00
     .,0.11208436D+04,0.141D+03,0.690D+02,0.19545000D+01,0.00000000D+00
     .,0.11082791D+04,0.141D+03,0.700D+02,0.19545000D+01,0.00000000D+00
     .,0.92522660D+03,0.141D+03,0.710D+02,0.19545000D+01,0.00000000D+00
     .,0.90068080D+03,0.141D+03,0.720D+02,0.19545000D+01,0.00000000D+00
     .,0.81649740D+03,0.141D+03,0.730D+02,0.19545000D+01,0.00000000D+00
     .,0.68598560D+03,0.141D+03,0.740D+02,0.19545000D+01,0.00000000D+00
     .,0.69600400D+03,0.141D+03,0.750D+02,0.19545000D+01,0.00000000D+00
     .,0.62720650D+03,0.141D+03,0.760D+02,0.19545000D+01,0.00000000D+00
     .,0.57177930D+03,0.141D+03,0.770D+02,0.19545000D+01,0.00000000D+00
     .,0.47276430D+03,0.141D+03,0.780D+02,0.19545000D+01,0.00000000D+00
     .,0.44085970D+03,0.141D+03,0.790D+02,0.19545000D+01,0.00000000D+00
     .,0.45247610D+03,0.141D+03,0.800D+02,0.19545000D+01,0.00000000D+00
     .,0.67515180D+03,0.141D+03,0.810D+02,0.19545000D+01,0.00000000D+00
     .,0.65444510D+03,0.141D+03,0.820D+02,0.19545000D+01,0.00000000D+00
     .,0.59566400D+03,0.141D+03,0.830D+02,0.19545000D+01,0.00000000D+00
     .,0.56506480D+03,0.141D+03,0.840D+02,0.19545000D+01,0.00000000D+00
     .,0.51821240D+03,0.141D+03,0.850D+02,0.19545000D+01,0.00000000D+00
     ./)
      pars(  43001:  43100)=(/
     . 0.47230580D+03,0.141D+03,0.860D+02,0.19545000D+01,0.00000000D+00
     .,0.17127050D+04,0.141D+03,0.870D+02,0.19545000D+01,0.00000000D+00
     .,0.16235376D+04,0.141D+03,0.880D+02,0.19545000D+01,0.00000000D+00
     .,0.14203983D+04,0.141D+03,0.890D+02,0.19545000D+01,0.00000000D+00
     .,0.12619867D+04,0.141D+03,0.900D+02,0.19545000D+01,0.00000000D+00
     .,0.12600147D+04,0.141D+03,0.910D+02,0.19545000D+01,0.00000000D+00
     .,0.12197064D+04,0.141D+03,0.920D+02,0.19545000D+01,0.00000000D+00
     .,0.12648378D+04,0.141D+03,0.930D+02,0.19545000D+01,0.00000000D+00
     .,0.12232638D+04,0.141D+03,0.940D+02,0.19545000D+01,0.00000000D+00
     .,0.65312900D+02,0.141D+03,0.101D+03,0.19545000D+01,0.00000000D+00
     .,0.21897060D+03,0.141D+03,0.103D+03,0.19545000D+01,0.98650000D+00
     .,0.27808810D+03,0.141D+03,0.104D+03,0.19545000D+01,0.98080000D+00
     .,0.20799430D+03,0.141D+03,0.105D+03,0.19545000D+01,0.97060000D+00
     .,0.15487640D+03,0.141D+03,0.106D+03,0.19545000D+01,0.98680000D+00
     .,0.10631410D+03,0.141D+03,0.107D+03,0.19545000D+01,0.99440000D+00
     .,0.76634700D+02,0.141D+03,0.108D+03,0.19545000D+01,0.99250000D+00
     .,0.52041200D+02,0.141D+03,0.109D+03,0.19545000D+01,0.99820000D+00
     .,0.32153780D+03,0.141D+03,0.111D+03,0.19545000D+01,0.96840000D+00
     .,0.49834860D+03,0.141D+03,0.112D+03,0.19545000D+01,0.96280000D+00
     .,0.49907610D+03,0.141D+03,0.113D+03,0.19545000D+01,0.96480000D+00
     ./)
      pars(  43101:  43200)=(/
     . 0.39454030D+03,0.141D+03,0.114D+03,0.19545000D+01,0.95070000D+00
     .,0.31930990D+03,0.141D+03,0.115D+03,0.19545000D+01,0.99470000D+00
     .,0.26788580D+03,0.141D+03,0.116D+03,0.19545000D+01,0.99480000D+00
     .,0.21720780D+03,0.141D+03,0.117D+03,0.19545000D+01,0.99720000D+00
     .,0.43952280D+03,0.141D+03,0.119D+03,0.19545000D+01,0.97670000D+00
     .,0.86414110D+03,0.141D+03,0.120D+03,0.19545000D+01,0.98310000D+00
     .,0.43378650D+03,0.141D+03,0.121D+03,0.19545000D+01,0.18627000D+01
     .,0.41830680D+03,0.141D+03,0.122D+03,0.19545000D+01,0.18299000D+01
     .,0.41010330D+03,0.141D+03,0.123D+03,0.19545000D+01,0.19138000D+01
     .,0.40698940D+03,0.141D+03,0.124D+03,0.19545000D+01,0.18269000D+01
     .,0.37146910D+03,0.141D+03,0.125D+03,0.19545000D+01,0.16406000D+01
     .,0.34300120D+03,0.141D+03,0.126D+03,0.19545000D+01,0.16483000D+01
     .,0.32719950D+03,0.141D+03,0.127D+03,0.19545000D+01,0.17149000D+01
     .,0.32009600D+03,0.141D+03,0.128D+03,0.19545000D+01,0.17937000D+01
     .,0.31821360D+03,0.141D+03,0.129D+03,0.19545000D+01,0.95760000D+00
     .,0.29527620D+03,0.141D+03,0.130D+03,0.19545000D+01,0.19419000D+01
     .,0.49283160D+03,0.141D+03,0.131D+03,0.19545000D+01,0.96010000D+00
     .,0.42736220D+03,0.141D+03,0.132D+03,0.19545000D+01,0.94340000D+00
     .,0.37950970D+03,0.141D+03,0.133D+03,0.19545000D+01,0.98890000D+00
     .,0.34441690D+03,0.141D+03,0.134D+03,0.19545000D+01,0.99010000D+00
     ./)
      pars(  43201:  43300)=(/
     . 0.30134630D+03,0.141D+03,0.135D+03,0.19545000D+01,0.99740000D+00
     .,0.52294490D+03,0.141D+03,0.137D+03,0.19545000D+01,0.97380000D+00
     .,0.10528629D+04,0.141D+03,0.138D+03,0.19545000D+01,0.98010000D+00
     .,0.78749500D+03,0.141D+03,0.139D+03,0.19545000D+01,0.19153000D+01
     .,0.57445910D+03,0.141D+03,0.140D+03,0.19545000D+01,0.19355000D+01
     .,0.58058240D+03,0.141D+03,0.141D+03,0.19545000D+01,0.19545000D+01
     .,0.37547800D+02,0.142D+03,0.100D+01,0.19420000D+01,0.91180000D+00
     .,0.24553000D+02,0.142D+03,0.200D+01,0.19420000D+01,0.00000000D+00
     .,0.62818170D+03,0.142D+03,0.300D+01,0.19420000D+01,0.00000000D+00
     .,0.34918970D+03,0.142D+03,0.400D+01,0.19420000D+01,0.00000000D+00
     .,0.23104490D+03,0.142D+03,0.500D+01,0.19420000D+01,0.00000000D+00
     .,0.15432880D+03,0.142D+03,0.600D+01,0.19420000D+01,0.00000000D+00
     .,0.10713040D+03,0.142D+03,0.700D+01,0.19420000D+01,0.00000000D+00
     .,0.80736000D+02,0.142D+03,0.800D+01,0.19420000D+01,0.00000000D+00
     .,0.60941400D+02,0.142D+03,0.900D+01,0.19420000D+01,0.00000000D+00
     .,0.46749800D+02,0.142D+03,0.100D+02,0.19420000D+01,0.00000000D+00
     .,0.74949740D+03,0.142D+03,0.110D+02,0.19420000D+01,0.00000000D+00
     .,0.55988560D+03,0.142D+03,0.120D+02,0.19420000D+01,0.00000000D+00
     .,0.51108100D+03,0.142D+03,0.130D+02,0.19420000D+01,0.00000000D+00
     .,0.39766020D+03,0.142D+03,0.140D+02,0.19420000D+01,0.00000000D+00
     ./)
      pars(  43301:  43400)=(/
     . 0.30713840D+03,0.142D+03,0.150D+02,0.19420000D+01,0.00000000D+00
     .,0.25349850D+03,0.142D+03,0.160D+02,0.19420000D+01,0.00000000D+00
     .,0.20604590D+03,0.142D+03,0.170D+02,0.19420000D+01,0.00000000D+00
     .,0.16790140D+03,0.142D+03,0.180D+02,0.19420000D+01,0.00000000D+00
     .,0.12379643D+04,0.142D+03,0.190D+02,0.19420000D+01,0.00000000D+00
     .,0.99601570D+03,0.142D+03,0.200D+02,0.19420000D+01,0.00000000D+00
     .,0.81851610D+03,0.142D+03,0.210D+02,0.19420000D+01,0.00000000D+00
     .,0.78705650D+03,0.142D+03,0.220D+02,0.19420000D+01,0.00000000D+00
     .,0.71884690D+03,0.142D+03,0.230D+02,0.19420000D+01,0.00000000D+00
     .,0.56643000D+03,0.142D+03,0.240D+02,0.19420000D+01,0.00000000D+00
     .,0.61659930D+03,0.142D+03,0.250D+02,0.19420000D+01,0.00000000D+00
     .,0.48385370D+03,0.142D+03,0.260D+02,0.19420000D+01,0.00000000D+00
     .,0.50969990D+03,0.142D+03,0.270D+02,0.19420000D+01,0.00000000D+00
     .,0.52637950D+03,0.142D+03,0.280D+02,0.19420000D+01,0.00000000D+00
     .,0.40384000D+03,0.142D+03,0.290D+02,0.19420000D+01,0.00000000D+00
     .,0.41098660D+03,0.142D+03,0.300D+02,0.19420000D+01,0.00000000D+00
     .,0.48798580D+03,0.142D+03,0.310D+02,0.19420000D+01,0.00000000D+00
     .,0.42607030D+03,0.142D+03,0.320D+02,0.19420000D+01,0.00000000D+00
     .,0.36044960D+03,0.142D+03,0.330D+02,0.19420000D+01,0.00000000D+00
     .,0.32194150D+03,0.142D+03,0.340D+02,0.19420000D+01,0.00000000D+00
     ./)
      pars(  43401:  43500)=(/
     . 0.28047020D+03,0.142D+03,0.350D+02,0.19420000D+01,0.00000000D+00
     .,0.24296840D+03,0.142D+03,0.360D+02,0.19420000D+01,0.00000000D+00
     .,0.13851228D+04,0.142D+03,0.370D+02,0.19420000D+01,0.00000000D+00
     .,0.11881982D+04,0.142D+03,0.380D+02,0.19420000D+01,0.00000000D+00
     .,0.10314745D+04,0.142D+03,0.390D+02,0.19420000D+01,0.00000000D+00
     .,0.92206600D+03,0.142D+03,0.400D+02,0.19420000D+01,0.00000000D+00
     .,0.83798490D+03,0.142D+03,0.410D+02,0.19420000D+01,0.00000000D+00
     .,0.64347210D+03,0.142D+03,0.420D+02,0.19420000D+01,0.00000000D+00
     .,0.71936590D+03,0.142D+03,0.430D+02,0.19420000D+01,0.00000000D+00
     .,0.54480750D+03,0.142D+03,0.440D+02,0.19420000D+01,0.00000000D+00
     .,0.59515030D+03,0.142D+03,0.450D+02,0.19420000D+01,0.00000000D+00
     .,0.55075570D+03,0.142D+03,0.460D+02,0.19420000D+01,0.00000000D+00
     .,0.46023670D+03,0.142D+03,0.470D+02,0.19420000D+01,0.00000000D+00
     .,0.48402860D+03,0.142D+03,0.480D+02,0.19420000D+01,0.00000000D+00
     .,0.61139110D+03,0.142D+03,0.490D+02,0.19420000D+01,0.00000000D+00
     .,0.56051040D+03,0.142D+03,0.500D+02,0.19420000D+01,0.00000000D+00
     .,0.49569480D+03,0.142D+03,0.510D+02,0.19420000D+01,0.00000000D+00
     .,0.45793020D+03,0.142D+03,0.520D+02,0.19420000D+01,0.00000000D+00
     .,0.41222750D+03,0.142D+03,0.530D+02,0.19420000D+01,0.00000000D+00
     .,0.36915310D+03,0.142D+03,0.540D+02,0.19420000D+01,0.00000000D+00
     ./)
      pars(  43501:  43600)=(/
     . 0.16877490D+04,0.142D+03,0.550D+02,0.19420000D+01,0.00000000D+00
     .,0.15201317D+04,0.142D+03,0.560D+02,0.19420000D+01,0.00000000D+00
     .,0.13232583D+04,0.142D+03,0.570D+02,0.19420000D+01,0.00000000D+00
     .,0.58664300D+03,0.142D+03,0.580D+02,0.19420000D+01,0.27991000D+01
     .,0.13428116D+04,0.142D+03,0.590D+02,0.19420000D+01,0.00000000D+00
     .,0.12872763D+04,0.142D+03,0.600D+02,0.19420000D+01,0.00000000D+00
     .,0.12544039D+04,0.142D+03,0.610D+02,0.19420000D+01,0.00000000D+00
     .,0.12242314D+04,0.142D+03,0.620D+02,0.19420000D+01,0.00000000D+00
     .,0.11974564D+04,0.142D+03,0.630D+02,0.19420000D+01,0.00000000D+00
     .,0.93283320D+03,0.142D+03,0.640D+02,0.19420000D+01,0.00000000D+00
     .,0.10667077D+04,0.142D+03,0.650D+02,0.19420000D+01,0.00000000D+00
     .,0.10271026D+04,0.142D+03,0.660D+02,0.19420000D+01,0.00000000D+00
     .,0.10774205D+04,0.142D+03,0.670D+02,0.19420000D+01,0.00000000D+00
     .,0.10542484D+04,0.142D+03,0.680D+02,0.19420000D+01,0.00000000D+00
     .,0.10332364D+04,0.142D+03,0.690D+02,0.19420000D+01,0.00000000D+00
     .,0.10215311D+04,0.142D+03,0.700D+02,0.19420000D+01,0.00000000D+00
     .,0.85524570D+03,0.142D+03,0.710D+02,0.19420000D+01,0.00000000D+00
     .,0.83345230D+03,0.142D+03,0.720D+02,0.19420000D+01,0.00000000D+00
     .,0.75675320D+03,0.142D+03,0.730D+02,0.19420000D+01,0.00000000D+00
     .,0.63690770D+03,0.142D+03,0.740D+02,0.19420000D+01,0.00000000D+00
     ./)
      pars(  43601:  43700)=(/
     . 0.64654950D+03,0.142D+03,0.750D+02,0.19420000D+01,0.00000000D+00
     .,0.58352570D+03,0.142D+03,0.760D+02,0.19420000D+01,0.00000000D+00
     .,0.53265790D+03,0.142D+03,0.770D+02,0.19420000D+01,0.00000000D+00
     .,0.44119610D+03,0.142D+03,0.780D+02,0.19420000D+01,0.00000000D+00
     .,0.41171360D+03,0.142D+03,0.790D+02,0.19420000D+01,0.00000000D+00
     .,0.42271910D+03,0.142D+03,0.800D+02,0.19420000D+01,0.00000000D+00
     .,0.62693260D+03,0.142D+03,0.810D+02,0.19420000D+01,0.00000000D+00
     .,0.60866200D+03,0.142D+03,0.820D+02,0.19420000D+01,0.00000000D+00
     .,0.55532550D+03,0.142D+03,0.830D+02,0.19420000D+01,0.00000000D+00
     .,0.52759970D+03,0.142D+03,0.840D+02,0.19420000D+01,0.00000000D+00
     .,0.48478900D+03,0.142D+03,0.850D+02,0.19420000D+01,0.00000000D+00
     .,0.44266740D+03,0.142D+03,0.860D+02,0.19420000D+01,0.00000000D+00
     .,0.15815449D+04,0.142D+03,0.870D+02,0.19420000D+01,0.00000000D+00
     .,0.14962077D+04,0.142D+03,0.880D+02,0.19420000D+01,0.00000000D+00
     .,0.13112088D+04,0.142D+03,0.890D+02,0.19420000D+01,0.00000000D+00
     .,0.11676309D+04,0.142D+03,0.900D+02,0.19420000D+01,0.00000000D+00
     .,0.11649288D+04,0.142D+03,0.910D+02,0.19420000D+01,0.00000000D+00
     .,0.11276927D+04,0.142D+03,0.920D+02,0.19420000D+01,0.00000000D+00
     .,0.11676383D+04,0.142D+03,0.930D+02,0.19420000D+01,0.00000000D+00
     .,0.11294830D+04,0.142D+03,0.940D+02,0.19420000D+01,0.00000000D+00
     ./)
      pars(  43701:  43800)=(/
     . 0.60977300D+02,0.142D+03,0.101D+03,0.19420000D+01,0.00000000D+00
     .,0.20251730D+03,0.142D+03,0.103D+03,0.19420000D+01,0.98650000D+00
     .,0.25760880D+03,0.142D+03,0.104D+03,0.19420000D+01,0.98080000D+00
     .,0.19375570D+03,0.142D+03,0.105D+03,0.19420000D+01,0.97060000D+00
     .,0.14487380D+03,0.142D+03,0.106D+03,0.19420000D+01,0.98680000D+00
     .,0.99938100D+02,0.142D+03,0.107D+03,0.19420000D+01,0.99440000D+00
     .,0.72352000D+02,0.142D+03,0.108D+03,0.19420000D+01,0.99250000D+00
     .,0.49416600D+02,0.142D+03,0.109D+03,0.19420000D+01,0.99820000D+00
     .,0.29717130D+03,0.142D+03,0.111D+03,0.19420000D+01,0.96840000D+00
     .,0.46042340D+03,0.142D+03,0.112D+03,0.19420000D+01,0.96280000D+00
     .,0.46216780D+03,0.142D+03,0.113D+03,0.19420000D+01,0.96480000D+00
     .,0.36695490D+03,0.142D+03,0.114D+03,0.19420000D+01,0.95070000D+00
     .,0.29806610D+03,0.142D+03,0.115D+03,0.19420000D+01,0.99470000D+00
     .,0.25075730D+03,0.142D+03,0.116D+03,0.19420000D+01,0.99480000D+00
     .,0.20393970D+03,0.142D+03,0.117D+03,0.19420000D+01,0.99720000D+00
     .,0.40759060D+03,0.142D+03,0.119D+03,0.19420000D+01,0.97670000D+00
     .,0.79747730D+03,0.142D+03,0.120D+03,0.19420000D+01,0.98310000D+00
     .,0.40315400D+03,0.142D+03,0.121D+03,0.19420000D+01,0.18627000D+01
     .,0.38917870D+03,0.142D+03,0.122D+03,0.19420000D+01,0.18299000D+01
     .,0.38148040D+03,0.142D+03,0.123D+03,0.19420000D+01,0.19138000D+01
     ./)
      pars(  43801:  43900)=(/
     . 0.37844740D+03,0.142D+03,0.124D+03,0.19420000D+01,0.18269000D+01
     .,0.34604230D+03,0.142D+03,0.125D+03,0.19420000D+01,0.16406000D+01
     .,0.31976370D+03,0.142D+03,0.126D+03,0.19420000D+01,0.16483000D+01
     .,0.30508870D+03,0.142D+03,0.127D+03,0.19420000D+01,0.17149000D+01
     .,0.29841290D+03,0.142D+03,0.128D+03,0.19420000D+01,0.17937000D+01
     .,0.29619890D+03,0.142D+03,0.129D+03,0.19420000D+01,0.95760000D+00
     .,0.27560550D+03,0.142D+03,0.130D+03,0.19420000D+01,0.19419000D+01
     .,0.45716410D+03,0.142D+03,0.131D+03,0.19420000D+01,0.96010000D+00
     .,0.39775920D+03,0.142D+03,0.132D+03,0.19420000D+01,0.94340000D+00
     .,0.35418680D+03,0.142D+03,0.133D+03,0.19420000D+01,0.98890000D+00
     .,0.32209780D+03,0.142D+03,0.134D+03,0.19420000D+01,0.99010000D+00
     .,0.28249330D+03,0.142D+03,0.135D+03,0.19420000D+01,0.99740000D+00
     .,0.48541290D+03,0.142D+03,0.137D+03,0.19420000D+01,0.97380000D+00
     .,0.97212330D+03,0.142D+03,0.138D+03,0.19420000D+01,0.98010000D+00
     .,0.73014460D+03,0.142D+03,0.139D+03,0.19420000D+01,0.19153000D+01
     .,0.53401080D+03,0.142D+03,0.140D+03,0.19420000D+01,0.19355000D+01
     .,0.53935320D+03,0.142D+03,0.141D+03,0.19420000D+01,0.19545000D+01
     .,0.50200730D+03,0.142D+03,0.142D+03,0.19420000D+01,0.19420000D+01
     .,0.41338500D+02,0.143D+03,0.100D+01,0.16682000D+01,0.91180000D+00
     .,0.26621700D+02,0.143D+03,0.200D+01,0.16682000D+01,0.00000000D+00
     ./)
      pars(  43901:  44000)=(/
     . 0.76378140D+03,0.143D+03,0.300D+01,0.16682000D+01,0.00000000D+00
     .,0.40277270D+03,0.143D+03,0.400D+01,0.16682000D+01,0.00000000D+00
     .,0.26037960D+03,0.143D+03,0.500D+01,0.16682000D+01,0.00000000D+00
     .,0.17125810D+03,0.143D+03,0.600D+01,0.16682000D+01,0.00000000D+00
     .,0.11764790D+03,0.143D+03,0.700D+01,0.16682000D+01,0.00000000D+00
     .,0.88060900D+02,0.143D+03,0.800D+01,0.16682000D+01,0.00000000D+00
     .,0.66109300D+02,0.143D+03,0.900D+01,0.16682000D+01,0.00000000D+00
     .,0.50508700D+02,0.143D+03,0.100D+02,0.16682000D+01,0.00000000D+00
     .,0.90840250D+03,0.143D+03,0.110D+02,0.16682000D+01,0.00000000D+00
     .,0.65125640D+03,0.143D+03,0.120D+02,0.16682000D+01,0.00000000D+00
     .,0.58698620D+03,0.143D+03,0.130D+02,0.16682000D+01,0.00000000D+00
     .,0.44911910D+03,0.143D+03,0.140D+02,0.16682000D+01,0.00000000D+00
     .,0.34248830D+03,0.143D+03,0.150D+02,0.16682000D+01,0.00000000D+00
     .,0.28056130D+03,0.143D+03,0.160D+02,0.16682000D+01,0.00000000D+00
     .,0.22648330D+03,0.143D+03,0.170D+02,0.16682000D+01,0.00000000D+00
     .,0.18352050D+03,0.143D+03,0.180D+02,0.16682000D+01,0.00000000D+00
     .,0.15188755D+04,0.143D+03,0.190D+02,0.16682000D+01,0.00000000D+00
     .,0.11783302D+04,0.143D+03,0.200D+02,0.16682000D+01,0.00000000D+00
     .,0.96136570D+03,0.143D+03,0.210D+02,0.16682000D+01,0.00000000D+00
     .,0.91930990D+03,0.143D+03,0.220D+02,0.16682000D+01,0.00000000D+00
     ./)
      pars(  44001:  44100)=(/
     . 0.83673410D+03,0.143D+03,0.230D+02,0.16682000D+01,0.00000000D+00
     .,0.65972010D+03,0.143D+03,0.240D+02,0.16682000D+01,0.00000000D+00
     .,0.71418300D+03,0.143D+03,0.250D+02,0.16682000D+01,0.00000000D+00
     .,0.56036480D+03,0.143D+03,0.260D+02,0.16682000D+01,0.00000000D+00
     .,0.58529350D+03,0.143D+03,0.270D+02,0.16682000D+01,0.00000000D+00
     .,0.60647400D+03,0.143D+03,0.280D+02,0.16682000D+01,0.00000000D+00
     .,0.46581470D+03,0.143D+03,0.290D+02,0.16682000D+01,0.00000000D+00
     .,0.46845290D+03,0.143D+03,0.300D+02,0.16682000D+01,0.00000000D+00
     .,0.55815220D+03,0.143D+03,0.310D+02,0.16682000D+01,0.00000000D+00
     .,0.48089260D+03,0.143D+03,0.320D+02,0.16682000D+01,0.00000000D+00
     .,0.40245400D+03,0.143D+03,0.330D+02,0.16682000D+01,0.00000000D+00
     .,0.35725110D+03,0.143D+03,0.340D+02,0.16682000D+01,0.00000000D+00
     .,0.30935300D+03,0.143D+03,0.350D+02,0.16682000D+01,0.00000000D+00
     .,0.26657770D+03,0.143D+03,0.360D+02,0.16682000D+01,0.00000000D+00
     .,0.16961214D+04,0.143D+03,0.370D+02,0.16682000D+01,0.00000000D+00
     .,0.14083895D+04,0.143D+03,0.380D+02,0.16682000D+01,0.00000000D+00
     .,0.12079868D+04,0.143D+03,0.390D+02,0.16682000D+01,0.00000000D+00
     .,0.10721935D+04,0.143D+03,0.400D+02,0.16682000D+01,0.00000000D+00
     .,0.97020700D+03,0.143D+03,0.410D+02,0.16682000D+01,0.00000000D+00
     .,0.73979150D+03,0.143D+03,0.420D+02,0.16682000D+01,0.00000000D+00
     ./)
      pars(  44101:  44200)=(/
     . 0.82924080D+03,0.143D+03,0.430D+02,0.16682000D+01,0.00000000D+00
     .,0.62313860D+03,0.143D+03,0.440D+02,0.16682000D+01,0.00000000D+00
     .,0.68005240D+03,0.143D+03,0.450D+02,0.16682000D+01,0.00000000D+00
     .,0.62758100D+03,0.143D+03,0.460D+02,0.16682000D+01,0.00000000D+00
     .,0.52609770D+03,0.143D+03,0.470D+02,0.16682000D+01,0.00000000D+00
     .,0.54963380D+03,0.143D+03,0.480D+02,0.16682000D+01,0.00000000D+00
     .,0.70042290D+03,0.143D+03,0.490D+02,0.16682000D+01,0.00000000D+00
     .,0.63478310D+03,0.143D+03,0.500D+02,0.16682000D+01,0.00000000D+00
     .,0.55577940D+03,0.143D+03,0.510D+02,0.16682000D+01,0.00000000D+00
     .,0.51056820D+03,0.143D+03,0.520D+02,0.16682000D+01,0.00000000D+00
     .,0.45697070D+03,0.143D+03,0.530D+02,0.16682000D+01,0.00000000D+00
     .,0.40709700D+03,0.143D+03,0.540D+02,0.16682000D+01,0.00000000D+00
     .,0.20688557D+04,0.143D+03,0.550D+02,0.16682000D+01,0.00000000D+00
     .,0.18110626D+04,0.143D+03,0.560D+02,0.16682000D+01,0.00000000D+00
     .,0.15570674D+04,0.143D+03,0.570D+02,0.16682000D+01,0.00000000D+00
     .,0.65923480D+03,0.143D+03,0.580D+02,0.16682000D+01,0.27991000D+01
     .,0.15936331D+04,0.143D+03,0.590D+02,0.16682000D+01,0.00000000D+00
     .,0.15238714D+04,0.143D+03,0.600D+02,0.16682000D+01,0.00000000D+00
     .,0.14839817D+04,0.143D+03,0.610D+02,0.16682000D+01,0.00000000D+00
     .,0.14474665D+04,0.143D+03,0.620D+02,0.16682000D+01,0.00000000D+00
     ./)
      pars(  44201:  44300)=(/
     . 0.14150358D+04,0.143D+03,0.630D+02,0.16682000D+01,0.00000000D+00
     .,0.10889839D+04,0.143D+03,0.640D+02,0.16682000D+01,0.00000000D+00
     .,0.12727235D+04,0.143D+03,0.650D+02,0.16682000D+01,0.00000000D+00
     .,0.12232739D+04,0.143D+03,0.660D+02,0.16682000D+01,0.00000000D+00
     .,0.12689174D+04,0.143D+03,0.670D+02,0.16682000D+01,0.00000000D+00
     .,0.12411151D+04,0.143D+03,0.680D+02,0.16682000D+01,0.00000000D+00
     .,0.12157298D+04,0.143D+03,0.690D+02,0.16682000D+01,0.00000000D+00
     .,0.12025314D+04,0.143D+03,0.700D+02,0.16682000D+01,0.00000000D+00
     .,0.99886140D+03,0.143D+03,0.710D+02,0.16682000D+01,0.00000000D+00
     .,0.96131920D+03,0.143D+03,0.720D+02,0.16682000D+01,0.00000000D+00
     .,0.86700540D+03,0.143D+03,0.730D+02,0.16682000D+01,0.00000000D+00
     .,0.72670660D+03,0.143D+03,0.740D+02,0.16682000D+01,0.00000000D+00
     .,0.73557240D+03,0.143D+03,0.750D+02,0.16682000D+01,0.00000000D+00
     .,0.66018700D+03,0.143D+03,0.760D+02,0.16682000D+01,0.00000000D+00
     .,0.60002740D+03,0.143D+03,0.770D+02,0.16682000D+01,0.00000000D+00
     .,0.49511760D+03,0.143D+03,0.780D+02,0.16682000D+01,0.00000000D+00
     .,0.46130260D+03,0.143D+03,0.790D+02,0.16682000D+01,0.00000000D+00
     .,0.47242330D+03,0.143D+03,0.800D+02,0.16682000D+01,0.00000000D+00
     .,0.71722290D+03,0.143D+03,0.810D+02,0.16682000D+01,0.00000000D+00
     .,0.68964350D+03,0.143D+03,0.820D+02,0.16682000D+01,0.00000000D+00
     ./)
      pars(  44301:  44400)=(/
     . 0.62335390D+03,0.143D+03,0.830D+02,0.16682000D+01,0.00000000D+00
     .,0.58927250D+03,0.143D+03,0.840D+02,0.16682000D+01,0.00000000D+00
     .,0.53837470D+03,0.143D+03,0.850D+02,0.16682000D+01,0.00000000D+00
     .,0.48918620D+03,0.143D+03,0.860D+02,0.16682000D+01,0.00000000D+00
     .,0.19184029D+04,0.143D+03,0.870D+02,0.16682000D+01,0.00000000D+00
     .,0.17715639D+04,0.143D+03,0.880D+02,0.16682000D+01,0.00000000D+00
     .,0.15346792D+04,0.143D+03,0.890D+02,0.16682000D+01,0.00000000D+00
     .,0.13502400D+04,0.143D+03,0.900D+02,0.16682000D+01,0.00000000D+00
     .,0.13560504D+04,0.143D+03,0.910D+02,0.16682000D+01,0.00000000D+00
     .,0.13122121D+04,0.143D+03,0.920D+02,0.16682000D+01,0.00000000D+00
     .,0.13686610D+04,0.143D+03,0.930D+02,0.16682000D+01,0.00000000D+00
     .,0.13219594D+04,0.143D+03,0.940D+02,0.16682000D+01,0.00000000D+00
     .,0.67892400D+02,0.143D+03,0.101D+03,0.16682000D+01,0.00000000D+00
     .,0.23295390D+03,0.143D+03,0.103D+03,0.16682000D+01,0.98650000D+00
     .,0.29538160D+03,0.143D+03,0.104D+03,0.16682000D+01,0.98080000D+00
     .,0.21747620D+03,0.143D+03,0.105D+03,0.16682000D+01,0.97060000D+00
     .,0.16092230D+03,0.143D+03,0.106D+03,0.16682000D+01,0.98680000D+00
     .,0.10977550D+03,0.143D+03,0.107D+03,0.16682000D+01,0.99440000D+00
     .,0.78792000D+02,0.143D+03,0.108D+03,0.16682000D+01,0.99250000D+00
     .,0.53288000D+02,0.143D+03,0.109D+03,0.16682000D+01,0.99820000D+00
     ./)
      pars(  44401:  44500)=(/
     . 0.34355920D+03,0.143D+03,0.111D+03,0.16682000D+01,0.96840000D+00
     .,0.53370850D+03,0.143D+03,0.112D+03,0.16682000D+01,0.96280000D+00
     .,0.52930430D+03,0.143D+03,0.113D+03,0.16682000D+01,0.96480000D+00
     .,0.41359910D+03,0.143D+03,0.114D+03,0.16682000D+01,0.95070000D+00
     .,0.33227830D+03,0.143D+03,0.115D+03,0.16682000D+01,0.99470000D+00
     .,0.27760020D+03,0.143D+03,0.116D+03,0.16682000D+01,0.99480000D+00
     .,0.22420950D+03,0.143D+03,0.117D+03,0.16682000D+01,0.99720000D+00
     .,0.46809770D+03,0.143D+03,0.119D+03,0.16682000D+01,0.97670000D+00
     .,0.94599430D+03,0.143D+03,0.120D+03,0.16682000D+01,0.98310000D+00
     .,0.45591090D+03,0.143D+03,0.121D+03,0.16682000D+01,0.18627000D+01
     .,0.44025610D+03,0.143D+03,0.122D+03,0.16682000D+01,0.18299000D+01
     .,0.43163910D+03,0.143D+03,0.123D+03,0.16682000D+01,0.19138000D+01
     .,0.42896620D+03,0.143D+03,0.124D+03,0.16682000D+01,0.18269000D+01
     .,0.38890660D+03,0.143D+03,0.125D+03,0.16682000D+01,0.16406000D+01
     .,0.35865620D+03,0.143D+03,0.126D+03,0.16682000D+01,0.16483000D+01
     .,0.34229790D+03,0.143D+03,0.127D+03,0.16682000D+01,0.17149000D+01
     .,0.33502870D+03,0.143D+03,0.128D+03,0.16682000D+01,0.17937000D+01
     .,0.33456150D+03,0.143D+03,0.129D+03,0.16682000D+01,0.95760000D+00
     .,0.30788150D+03,0.143D+03,0.130D+03,0.16682000D+01,0.19419000D+01
     .,0.52114410D+03,0.143D+03,0.131D+03,0.16682000D+01,0.96010000D+00
     ./)
      pars(  44501:  44600)=(/
     . 0.44771310D+03,0.143D+03,0.132D+03,0.16682000D+01,0.94340000D+00
     .,0.39524720D+03,0.143D+03,0.133D+03,0.16682000D+01,0.98890000D+00
     .,0.35747370D+03,0.143D+03,0.134D+03,0.16682000D+01,0.99010000D+00
     .,0.31169010D+03,0.143D+03,0.135D+03,0.16682000D+01,0.99740000D+00
     .,0.55612550D+03,0.143D+03,0.137D+03,0.16682000D+01,0.97380000D+00
     .,0.11562604D+04,0.143D+03,0.138D+03,0.16682000D+01,0.98010000D+00
     .,0.84837270D+03,0.143D+03,0.139D+03,0.16682000D+01,0.19153000D+01
     .,0.60533890D+03,0.143D+03,0.140D+03,0.16682000D+01,0.19355000D+01
     .,0.61127640D+03,0.143D+03,0.141D+03,0.16682000D+01,0.19545000D+01
     .,0.56776700D+03,0.143D+03,0.142D+03,0.16682000D+01,0.19420000D+01
     .,0.64971430D+03,0.143D+03,0.143D+03,0.16682000D+01,0.16682000D+01
     .,0.33187500D+02,0.144D+03,0.100D+01,0.18584000D+01,0.91180000D+00
     .,0.22061600D+02,0.144D+03,0.200D+01,0.18584000D+01,0.00000000D+00
     .,0.51733300D+03,0.144D+03,0.300D+01,0.18584000D+01,0.00000000D+00
     .,0.29751860D+03,0.144D+03,0.400D+01,0.18584000D+01,0.00000000D+00
     .,0.20032990D+03,0.144D+03,0.500D+01,0.18584000D+01,0.00000000D+00
     .,0.13555170D+03,0.144D+03,0.600D+01,0.18584000D+01,0.00000000D+00
     .,0.95019000D+02,0.144D+03,0.700D+01,0.18584000D+01,0.00000000D+00
     .,0.72117300D+02,0.144D+03,0.800D+01,0.18584000D+01,0.00000000D+00
     .,0.54778100D+02,0.144D+03,0.900D+01,0.18584000D+01,0.00000000D+00
     ./)
      pars(  44601:  44700)=(/
     . 0.42243200D+02,0.144D+03,0.100D+02,0.18584000D+01,0.00000000D+00
     .,0.61866690D+03,0.144D+03,0.110D+02,0.18584000D+01,0.00000000D+00
     .,0.47441800D+03,0.144D+03,0.120D+02,0.18584000D+01,0.00000000D+00
     .,0.43702090D+03,0.144D+03,0.130D+02,0.18584000D+01,0.00000000D+00
     .,0.34421130D+03,0.144D+03,0.140D+02,0.18584000D+01,0.00000000D+00
     .,0.26854160D+03,0.144D+03,0.150D+02,0.18584000D+01,0.00000000D+00
     .,0.22308830D+03,0.144D+03,0.160D+02,0.18584000D+01,0.00000000D+00
     .,0.18247210D+03,0.144D+03,0.170D+02,0.18584000D+01,0.00000000D+00
     .,0.14952220D+03,0.144D+03,0.180D+02,0.18584000D+01,0.00000000D+00
     .,0.10157997D+04,0.144D+03,0.190D+02,0.18584000D+01,0.00000000D+00
     .,0.83526920D+03,0.144D+03,0.200D+02,0.18584000D+01,0.00000000D+00
     .,0.68960880D+03,0.144D+03,0.210D+02,0.18584000D+01,0.00000000D+00
     .,0.66581430D+03,0.144D+03,0.220D+02,0.18584000D+01,0.00000000D+00
     .,0.60961590D+03,0.144D+03,0.230D+02,0.18584000D+01,0.00000000D+00
     .,0.48063820D+03,0.144D+03,0.240D+02,0.18584000D+01,0.00000000D+00
     .,0.52477480D+03,0.144D+03,0.250D+02,0.18584000D+01,0.00000000D+00
     .,0.41224680D+03,0.144D+03,0.260D+02,0.18584000D+01,0.00000000D+00
     .,0.43640540D+03,0.144D+03,0.270D+02,0.18584000D+01,0.00000000D+00
     .,0.44958060D+03,0.144D+03,0.280D+02,0.18584000D+01,0.00000000D+00
     .,0.34510430D+03,0.144D+03,0.290D+02,0.18584000D+01,0.00000000D+00
     ./)
      pars(  44701:  44800)=(/
     . 0.35378800D+03,0.144D+03,0.300D+02,0.18584000D+01,0.00000000D+00
     .,0.41885730D+03,0.144D+03,0.310D+02,0.18584000D+01,0.00000000D+00
     .,0.36917340D+03,0.144D+03,0.320D+02,0.18584000D+01,0.00000000D+00
     .,0.31492530D+03,0.144D+03,0.330D+02,0.18584000D+01,0.00000000D+00
     .,0.28273020D+03,0.144D+03,0.340D+02,0.18584000D+01,0.00000000D+00
     .,0.24762030D+03,0.144D+03,0.350D+02,0.18584000D+01,0.00000000D+00
     .,0.21556830D+03,0.144D+03,0.360D+02,0.18584000D+01,0.00000000D+00
     .,0.11385265D+04,0.144D+03,0.370D+02,0.18584000D+01,0.00000000D+00
     .,0.99561160D+03,0.144D+03,0.380D+02,0.18584000D+01,0.00000000D+00
     .,0.87126000D+03,0.144D+03,0.390D+02,0.18584000D+01,0.00000000D+00
     .,0.78271660D+03,0.144D+03,0.400D+02,0.18584000D+01,0.00000000D+00
     .,0.71364770D+03,0.144D+03,0.410D+02,0.18584000D+01,0.00000000D+00
     .,0.55112400D+03,0.144D+03,0.420D+02,0.18584000D+01,0.00000000D+00
     .,0.61481850D+03,0.144D+03,0.430D+02,0.18584000D+01,0.00000000D+00
     .,0.46856950D+03,0.144D+03,0.440D+02,0.18584000D+01,0.00000000D+00
     .,0.51183150D+03,0.144D+03,0.450D+02,0.18584000D+01,0.00000000D+00
     .,0.47463690D+03,0.144D+03,0.460D+02,0.18584000D+01,0.00000000D+00
     .,0.39627420D+03,0.144D+03,0.470D+02,0.18584000D+01,0.00000000D+00
     .,0.41825790D+03,0.144D+03,0.480D+02,0.18584000D+01,0.00000000D+00
     .,0.52482360D+03,0.144D+03,0.490D+02,0.18584000D+01,0.00000000D+00
     ./)
      pars(  44801:  44900)=(/
     . 0.48488550D+03,0.144D+03,0.500D+02,0.18584000D+01,0.00000000D+00
     .,0.43198650D+03,0.144D+03,0.510D+02,0.18584000D+01,0.00000000D+00
     .,0.40083190D+03,0.144D+03,0.520D+02,0.18584000D+01,0.00000000D+00
     .,0.36253740D+03,0.144D+03,0.530D+02,0.18584000D+01,0.00000000D+00
     .,0.32611570D+03,0.144D+03,0.540D+02,0.18584000D+01,0.00000000D+00
     .,0.13871242D+04,0.144D+03,0.550D+02,0.18584000D+01,0.00000000D+00
     .,0.12701002D+04,0.144D+03,0.560D+02,0.18584000D+01,0.00000000D+00
     .,0.11145962D+04,0.144D+03,0.570D+02,0.18584000D+01,0.00000000D+00
     .,0.51058640D+03,0.144D+03,0.580D+02,0.18584000D+01,0.27991000D+01
     .,0.11251127D+04,0.144D+03,0.590D+02,0.18584000D+01,0.00000000D+00
     .,0.10801628D+04,0.144D+03,0.600D+02,0.18584000D+01,0.00000000D+00
     .,0.10529974D+04,0.144D+03,0.610D+02,0.18584000D+01,0.00000000D+00
     .,0.10280158D+04,0.144D+03,0.620D+02,0.18584000D+01,0.00000000D+00
     .,0.10058586D+04,0.144D+03,0.630D+02,0.18584000D+01,0.00000000D+00
     .,0.79051760D+03,0.144D+03,0.640D+02,0.18584000D+01,0.00000000D+00
     .,0.89187640D+03,0.144D+03,0.650D+02,0.18584000D+01,0.00000000D+00
     .,0.85997290D+03,0.144D+03,0.660D+02,0.18584000D+01,0.00000000D+00
     .,0.90692140D+03,0.144D+03,0.670D+02,0.18584000D+01,0.00000000D+00
     .,0.88762020D+03,0.144D+03,0.680D+02,0.18584000D+01,0.00000000D+00
     .,0.87020630D+03,0.144D+03,0.690D+02,0.18584000D+01,0.00000000D+00
     ./)
      pars(  44901:  45000)=(/
     . 0.86003450D+03,0.144D+03,0.700D+02,0.18584000D+01,0.00000000D+00
     .,0.72429040D+03,0.144D+03,0.710D+02,0.18584000D+01,0.00000000D+00
     .,0.71147790D+03,0.144D+03,0.720D+02,0.18584000D+01,0.00000000D+00
     .,0.64914710D+03,0.144D+03,0.730D+02,0.18584000D+01,0.00000000D+00
     .,0.54842120D+03,0.144D+03,0.740D+02,0.18584000D+01,0.00000000D+00
     .,0.55775610D+03,0.144D+03,0.750D+02,0.18584000D+01,0.00000000D+00
     .,0.50549330D+03,0.144D+03,0.760D+02,0.18584000D+01,0.00000000D+00
     .,0.46298700D+03,0.144D+03,0.770D+02,0.18584000D+01,0.00000000D+00
     .,0.38491460D+03,0.144D+03,0.780D+02,0.18584000D+01,0.00000000D+00
     .,0.35974270D+03,0.144D+03,0.790D+02,0.18584000D+01,0.00000000D+00
     .,0.36989520D+03,0.144D+03,0.800D+02,0.18584000D+01,0.00000000D+00
     .,0.53929820D+03,0.144D+03,0.810D+02,0.18584000D+01,0.00000000D+00
     .,0.52677380D+03,0.144D+03,0.820D+02,0.18584000D+01,0.00000000D+00
     .,0.48381430D+03,0.144D+03,0.830D+02,0.18584000D+01,0.00000000D+00
     .,0.46139360D+03,0.144D+03,0.840D+02,0.18584000D+01,0.00000000D+00
     .,0.42587210D+03,0.144D+03,0.850D+02,0.18584000D+01,0.00000000D+00
     .,0.39047110D+03,0.144D+03,0.860D+02,0.18584000D+01,0.00000000D+00
     .,0.13086904D+04,0.144D+03,0.870D+02,0.18584000D+01,0.00000000D+00
     .,0.12553593D+04,0.144D+03,0.880D+02,0.18584000D+01,0.00000000D+00
     .,0.11084934D+04,0.144D+03,0.890D+02,0.18584000D+01,0.00000000D+00
     ./)
      pars(  45001:  45100)=(/
     . 0.99548230D+03,0.144D+03,0.900D+02,0.18584000D+01,0.00000000D+00
     .,0.98913640D+03,0.144D+03,0.910D+02,0.18584000D+01,0.00000000D+00
     .,0.95777810D+03,0.144D+03,0.920D+02,0.18584000D+01,0.00000000D+00
     .,0.98673520D+03,0.144D+03,0.930D+02,0.18584000D+01,0.00000000D+00
     .,0.95542490D+03,0.144D+03,0.940D+02,0.18584000D+01,0.00000000D+00
     .,0.53364800D+02,0.144D+03,0.101D+03,0.18584000D+01,0.00000000D+00
     .,0.17297080D+03,0.144D+03,0.103D+03,0.18584000D+01,0.98650000D+00
     .,0.22067460D+03,0.144D+03,0.104D+03,0.18584000D+01,0.98080000D+00
     .,0.16859440D+03,0.144D+03,0.105D+03,0.18584000D+01,0.97060000D+00
     .,0.12721250D+03,0.144D+03,0.106D+03,0.18584000D+01,0.98680000D+00
     .,0.88661300D+02,0.144D+03,0.107D+03,0.18584000D+01,0.99440000D+00
     .,0.64744800D+02,0.144D+03,0.108D+03,0.18584000D+01,0.99250000D+00
     .,0.44708700D+02,0.144D+03,0.109D+03,0.18584000D+01,0.99820000D+00
     .,0.25309000D+03,0.144D+03,0.111D+03,0.18584000D+01,0.96840000D+00
     .,0.39136060D+03,0.144D+03,0.112D+03,0.18584000D+01,0.96280000D+00
     .,0.39606340D+03,0.144D+03,0.113D+03,0.18584000D+01,0.96480000D+00
     .,0.31817950D+03,0.144D+03,0.114D+03,0.18584000D+01,0.95070000D+00
     .,0.26070120D+03,0.144D+03,0.115D+03,0.18584000D+01,0.99470000D+00
     .,0.22064770D+03,0.144D+03,0.116D+03,0.18584000D+01,0.99480000D+00
     .,0.18059160D+03,0.144D+03,0.117D+03,0.18584000D+01,0.99720000D+00
     ./)
      pars(  45101:  45200)=(/
     . 0.34947530D+03,0.144D+03,0.119D+03,0.18584000D+01,0.97670000D+00
     .,0.66876460D+03,0.144D+03,0.120D+03,0.18584000D+01,0.98310000D+00
     .,0.34907020D+03,0.144D+03,0.121D+03,0.18584000D+01,0.18627000D+01
     .,0.33705530D+03,0.144D+03,0.122D+03,0.18584000D+01,0.18299000D+01
     .,0.33035670D+03,0.144D+03,0.123D+03,0.18584000D+01,0.19138000D+01
     .,0.32735530D+03,0.144D+03,0.124D+03,0.18584000D+01,0.18269000D+01
     .,0.30103970D+03,0.144D+03,0.125D+03,0.18584000D+01,0.16406000D+01
     .,0.27865260D+03,0.144D+03,0.126D+03,0.18584000D+01,0.16483000D+01
     .,0.26587520D+03,0.144D+03,0.127D+03,0.18584000D+01,0.17149000D+01
     .,0.25994720D+03,0.144D+03,0.128D+03,0.18584000D+01,0.17937000D+01
     .,0.25694800D+03,0.144D+03,0.129D+03,0.18584000D+01,0.95760000D+00
     .,0.24091240D+03,0.144D+03,0.130D+03,0.18584000D+01,0.19419000D+01
     .,0.39335710D+03,0.144D+03,0.131D+03,0.18584000D+01,0.96010000D+00
     .,0.34538720D+03,0.144D+03,0.132D+03,0.18584000D+01,0.94340000D+00
     .,0.30960350D+03,0.144D+03,0.133D+03,0.18584000D+01,0.98890000D+00
     .,0.28284600D+03,0.144D+03,0.134D+03,0.18584000D+01,0.99010000D+00
     .,0.24934040D+03,0.144D+03,0.135D+03,0.18584000D+01,0.99740000D+00
     .,0.41708310D+03,0.144D+03,0.137D+03,0.18584000D+01,0.97380000D+00
     .,0.81419750D+03,0.144D+03,0.138D+03,0.18584000D+01,0.98010000D+00
     .,0.62130580D+03,0.144D+03,0.139D+03,0.18584000D+01,0.19153000D+01
     ./)
      pars(  45201:  45300)=(/
     . 0.46191410D+03,0.144D+03,0.140D+03,0.18584000D+01,0.19355000D+01
     .,0.46653580D+03,0.144D+03,0.141D+03,0.18584000D+01,0.19545000D+01
     .,0.43510620D+03,0.144D+03,0.142D+03,0.18584000D+01,0.19420000D+01
     .,0.48832640D+03,0.144D+03,0.143D+03,0.18584000D+01,0.16682000D+01
     .,0.37926550D+03,0.144D+03,0.144D+03,0.18584000D+01,0.18584000D+01
     .,0.31102400D+02,0.145D+03,0.100D+01,0.19003000D+01,0.91180000D+00
     .,0.20777600D+02,0.145D+03,0.200D+01,0.19003000D+01,0.00000000D+00
     .,0.48243420D+03,0.145D+03,0.300D+01,0.19003000D+01,0.00000000D+00
     .,0.27741740D+03,0.145D+03,0.400D+01,0.19003000D+01,0.00000000D+00
     .,0.18713270D+03,0.145D+03,0.500D+01,0.19003000D+01,0.00000000D+00
     .,0.12690850D+03,0.145D+03,0.600D+01,0.19003000D+01,0.00000000D+00
     .,0.89165900D+02,0.145D+03,0.700D+01,0.19003000D+01,0.00000000D+00
     .,0.67811400D+02,0.145D+03,0.800D+01,0.19003000D+01,0.00000000D+00
     .,0.51611700D+02,0.145D+03,0.900D+01,0.19003000D+01,0.00000000D+00
     .,0.39876200D+02,0.145D+03,0.100D+02,0.19003000D+01,0.00000000D+00
     .,0.57701810D+03,0.145D+03,0.110D+02,0.19003000D+01,0.00000000D+00
     .,0.44233810D+03,0.145D+03,0.120D+02,0.19003000D+01,0.00000000D+00
     .,0.40769990D+03,0.145D+03,0.130D+02,0.19003000D+01,0.00000000D+00
     .,0.32145940D+03,0.145D+03,0.140D+02,0.19003000D+01,0.00000000D+00
     .,0.25114770D+03,0.145D+03,0.150D+02,0.19003000D+01,0.00000000D+00
     ./)
      pars(  45301:  45400)=(/
     . 0.20890670D+03,0.145D+03,0.160D+02,0.19003000D+01,0.00000000D+00
     .,0.17111870D+03,0.145D+03,0.170D+02,0.19003000D+01,0.00000000D+00
     .,0.14042730D+03,0.145D+03,0.180D+02,0.19003000D+01,0.00000000D+00
     .,0.94851400D+03,0.145D+03,0.190D+02,0.19003000D+01,0.00000000D+00
     .,0.77905470D+03,0.145D+03,0.200D+02,0.19003000D+01,0.00000000D+00
     .,0.64318930D+03,0.145D+03,0.210D+02,0.19003000D+01,0.00000000D+00
     .,0.62118070D+03,0.145D+03,0.220D+02,0.19003000D+01,0.00000000D+00
     .,0.56883480D+03,0.145D+03,0.230D+02,0.19003000D+01,0.00000000D+00
     .,0.44875270D+03,0.145D+03,0.240D+02,0.19003000D+01,0.00000000D+00
     .,0.48979330D+03,0.145D+03,0.250D+02,0.19003000D+01,0.00000000D+00
     .,0.38501530D+03,0.145D+03,0.260D+02,0.19003000D+01,0.00000000D+00
     .,0.40745740D+03,0.145D+03,0.270D+02,0.19003000D+01,0.00000000D+00
     .,0.41967090D+03,0.145D+03,0.280D+02,0.19003000D+01,0.00000000D+00
     .,0.32239880D+03,0.145D+03,0.290D+02,0.19003000D+01,0.00000000D+00
     .,0.33047130D+03,0.145D+03,0.300D+02,0.19003000D+01,0.00000000D+00
     .,0.39105080D+03,0.145D+03,0.310D+02,0.19003000D+01,0.00000000D+00
     .,0.34491000D+03,0.145D+03,0.320D+02,0.19003000D+01,0.00000000D+00
     .,0.29454210D+03,0.145D+03,0.330D+02,0.19003000D+01,0.00000000D+00
     .,0.26467170D+03,0.145D+03,0.340D+02,0.19003000D+01,0.00000000D+00
     .,0.23205800D+03,0.145D+03,0.350D+02,0.19003000D+01,0.00000000D+00
     ./)
      pars(  45401:  45500)=(/
     . 0.20225670D+03,0.145D+03,0.360D+02,0.19003000D+01,0.00000000D+00
     .,0.10633477D+04,0.145D+03,0.370D+02,0.19003000D+01,0.00000000D+00
     .,0.92882550D+03,0.145D+03,0.380D+02,0.19003000D+01,0.00000000D+00
     .,0.81293380D+03,0.145D+03,0.390D+02,0.19003000D+01,0.00000000D+00
     .,0.73049700D+03,0.145D+03,0.400D+02,0.19003000D+01,0.00000000D+00
     .,0.66622590D+03,0.145D+03,0.410D+02,0.19003000D+01,0.00000000D+00
     .,0.51491280D+03,0.145D+03,0.420D+02,0.19003000D+01,0.00000000D+00
     .,0.57424560D+03,0.145D+03,0.430D+02,0.19003000D+01,0.00000000D+00
     .,0.43804260D+03,0.145D+03,0.440D+02,0.19003000D+01,0.00000000D+00
     .,0.47828480D+03,0.145D+03,0.450D+02,0.19003000D+01,0.00000000D+00
     .,0.44362800D+03,0.145D+03,0.460D+02,0.19003000D+01,0.00000000D+00
     .,0.37063780D+03,0.145D+03,0.470D+02,0.19003000D+01,0.00000000D+00
     .,0.39106750D+03,0.145D+03,0.480D+02,0.19003000D+01,0.00000000D+00
     .,0.49033250D+03,0.145D+03,0.490D+02,0.19003000D+01,0.00000000D+00
     .,0.45317940D+03,0.145D+03,0.500D+02,0.19003000D+01,0.00000000D+00
     .,0.40403840D+03,0.145D+03,0.510D+02,0.19003000D+01,0.00000000D+00
     .,0.37513030D+03,0.145D+03,0.520D+02,0.19003000D+01,0.00000000D+00
     .,0.33956200D+03,0.145D+03,0.530D+02,0.19003000D+01,0.00000000D+00
     .,0.30572010D+03,0.145D+03,0.540D+02,0.19003000D+01,0.00000000D+00
     .,0.12958507D+04,0.145D+03,0.550D+02,0.19003000D+01,0.00000000D+00
     ./)
      pars(  45501:  45600)=(/
     . 0.11851747D+04,0.145D+03,0.560D+02,0.19003000D+01,0.00000000D+00
     .,0.10401066D+04,0.145D+03,0.570D+02,0.19003000D+01,0.00000000D+00
     .,0.47759450D+03,0.145D+03,0.580D+02,0.19003000D+01,0.27991000D+01
     .,0.10500763D+04,0.145D+03,0.590D+02,0.19003000D+01,0.00000000D+00
     .,0.10080521D+04,0.145D+03,0.600D+02,0.19003000D+01,0.00000000D+00
     .,0.98268750D+03,0.145D+03,0.610D+02,0.19003000D+01,0.00000000D+00
     .,0.95935980D+03,0.145D+03,0.620D+02,0.19003000D+01,0.00000000D+00
     .,0.93866880D+03,0.145D+03,0.630D+02,0.19003000D+01,0.00000000D+00
     .,0.73813170D+03,0.145D+03,0.640D+02,0.19003000D+01,0.00000000D+00
     .,0.83296700D+03,0.145D+03,0.650D+02,0.19003000D+01,0.00000000D+00
     .,0.80325110D+03,0.145D+03,0.660D+02,0.19003000D+01,0.00000000D+00
     .,0.84631030D+03,0.145D+03,0.670D+02,0.19003000D+01,0.00000000D+00
     .,0.82828260D+03,0.145D+03,0.680D+02,0.19003000D+01,0.00000000D+00
     .,0.81202140D+03,0.145D+03,0.690D+02,0.19003000D+01,0.00000000D+00
     .,0.80250260D+03,0.145D+03,0.700D+02,0.19003000D+01,0.00000000D+00
     .,0.67613200D+03,0.145D+03,0.710D+02,0.19003000D+01,0.00000000D+00
     .,0.66419530D+03,0.145D+03,0.720D+02,0.19003000D+01,0.00000000D+00
     .,0.60624630D+03,0.145D+03,0.730D+02,0.19003000D+01,0.00000000D+00
     .,0.51256560D+03,0.145D+03,0.740D+02,0.19003000D+01,0.00000000D+00
     .,0.52130610D+03,0.145D+03,0.750D+02,0.19003000D+01,0.00000000D+00
     ./)
      pars(  45601:  45700)=(/
     . 0.47268090D+03,0.145D+03,0.760D+02,0.19003000D+01,0.00000000D+00
     .,0.43312920D+03,0.145D+03,0.770D+02,0.19003000D+01,0.00000000D+00
     .,0.36041730D+03,0.145D+03,0.780D+02,0.19003000D+01,0.00000000D+00
     .,0.33697140D+03,0.145D+03,0.790D+02,0.19003000D+01,0.00000000D+00
     .,0.34645260D+03,0.145D+03,0.800D+02,0.19003000D+01,0.00000000D+00
     .,0.50424040D+03,0.145D+03,0.810D+02,0.19003000D+01,0.00000000D+00
     .,0.49256890D+03,0.145D+03,0.820D+02,0.19003000D+01,0.00000000D+00
     .,0.45264510D+03,0.145D+03,0.830D+02,0.19003000D+01,0.00000000D+00
     .,0.43185880D+03,0.145D+03,0.840D+02,0.19003000D+01,0.00000000D+00
     .,0.39887680D+03,0.145D+03,0.850D+02,0.19003000D+01,0.00000000D+00
     .,0.36599140D+03,0.145D+03,0.860D+02,0.19003000D+01,0.00000000D+00
     .,0.12224485D+04,0.145D+03,0.870D+02,0.19003000D+01,0.00000000D+00
     .,0.11715378D+04,0.145D+03,0.880D+02,0.19003000D+01,0.00000000D+00
     .,0.10345633D+04,0.145D+03,0.890D+02,0.19003000D+01,0.00000000D+00
     .,0.92953190D+03,0.145D+03,0.900D+02,0.19003000D+01,0.00000000D+00
     .,0.92366730D+03,0.145D+03,0.910D+02,0.19003000D+01,0.00000000D+00
     .,0.89441350D+03,0.145D+03,0.920D+02,0.19003000D+01,0.00000000D+00
     .,0.92128530D+03,0.145D+03,0.930D+02,0.19003000D+01,0.00000000D+00
     .,0.89206410D+03,0.145D+03,0.940D+02,0.19003000D+01,0.00000000D+00
     .,0.49909000D+02,0.145D+03,0.101D+03,0.19003000D+01,0.00000000D+00
     ./)
      pars(  45701:  45800)=(/
     . 0.16134990D+03,0.145D+03,0.103D+03,0.19003000D+01,0.98650000D+00
     .,0.20595300D+03,0.145D+03,0.104D+03,0.19003000D+01,0.98080000D+00
     .,0.15759490D+03,0.145D+03,0.105D+03,0.19003000D+01,0.97060000D+00
     .,0.11912650D+03,0.145D+03,0.106D+03,0.19003000D+01,0.98680000D+00
     .,0.83222700D+02,0.145D+03,0.107D+03,0.19003000D+01,0.99440000D+00
     .,0.60915900D+02,0.145D+03,0.108D+03,0.19003000D+01,0.99250000D+00
     .,0.42208400D+02,0.145D+03,0.109D+03,0.19003000D+01,0.99820000D+00
     .,0.23614440D+03,0.145D+03,0.111D+03,0.19003000D+01,0.96840000D+00
     .,0.36507050D+03,0.145D+03,0.112D+03,0.19003000D+01,0.96280000D+00
     .,0.36958140D+03,0.145D+03,0.113D+03,0.19003000D+01,0.96480000D+00
     .,0.29723390D+03,0.145D+03,0.114D+03,0.19003000D+01,0.95070000D+00
     .,0.24384390D+03,0.145D+03,0.115D+03,0.19003000D+01,0.99470000D+00
     .,0.20662580D+03,0.145D+03,0.116D+03,0.19003000D+01,0.99480000D+00
     .,0.16935840D+03,0.145D+03,0.117D+03,0.19003000D+01,0.99720000D+00
     .,0.32659470D+03,0.145D+03,0.119D+03,0.19003000D+01,0.97670000D+00
     .,0.62428500D+03,0.145D+03,0.120D+03,0.19003000D+01,0.98310000D+00
     .,0.32625030D+03,0.145D+03,0.121D+03,0.19003000D+01,0.18627000D+01
     .,0.31509820D+03,0.145D+03,0.122D+03,0.19003000D+01,0.18299000D+01
     .,0.30884820D+03,0.145D+03,0.123D+03,0.19003000D+01,0.19138000D+01
     .,0.30603400D+03,0.145D+03,0.124D+03,0.19003000D+01,0.18269000D+01
     ./)
      pars(  45801:  45900)=(/
     . 0.28151340D+03,0.145D+03,0.125D+03,0.19003000D+01,0.16406000D+01
     .,0.26065670D+03,0.145D+03,0.126D+03,0.19003000D+01,0.16483000D+01
     .,0.24873880D+03,0.145D+03,0.127D+03,0.19003000D+01,0.17149000D+01
     .,0.24319090D+03,0.145D+03,0.128D+03,0.19003000D+01,0.17937000D+01
     .,0.24032600D+03,0.145D+03,0.129D+03,0.19003000D+01,0.95760000D+00
     .,0.22543980D+03,0.145D+03,0.130D+03,0.19003000D+01,0.19419000D+01
     .,0.36731970D+03,0.145D+03,0.131D+03,0.19003000D+01,0.96010000D+00
     .,0.32278160D+03,0.145D+03,0.132D+03,0.19003000D+01,0.94340000D+00
     .,0.28959340D+03,0.145D+03,0.133D+03,0.19003000D+01,0.98890000D+00
     .,0.26478150D+03,0.145D+03,0.134D+03,0.19003000D+01,0.99010000D+00
     .,0.23366080D+03,0.145D+03,0.135D+03,0.19003000D+01,0.99740000D+00
     .,0.38991830D+03,0.145D+03,0.137D+03,0.19003000D+01,0.97380000D+00
     .,0.76024370D+03,0.145D+03,0.138D+03,0.19003000D+01,0.98010000D+00
     .,0.58041450D+03,0.145D+03,0.139D+03,0.19003000D+01,0.19153000D+01
     .,0.43182880D+03,0.145D+03,0.140D+03,0.19003000D+01,0.19355000D+01
     .,0.43614400D+03,0.145D+03,0.141D+03,0.19003000D+01,0.19545000D+01
     .,0.40693380D+03,0.145D+03,0.142D+03,0.19003000D+01,0.19420000D+01
     .,0.45656180D+03,0.145D+03,0.143D+03,0.19003000D+01,0.16682000D+01
     .,0.35492820D+03,0.145D+03,0.144D+03,0.19003000D+01,0.18584000D+01
     .,0.33224250D+03,0.145D+03,0.145D+03,0.19003000D+01,0.19003000D+01
     ./)
      pars(  45901:  46000)=(/
     . 0.28966100D+02,0.146D+03,0.100D+01,0.18630000D+01,0.91180000D+00
     .,0.19468900D+02,0.146D+03,0.200D+01,0.18630000D+01,0.00000000D+00
     .,0.44391060D+03,0.146D+03,0.300D+01,0.18630000D+01,0.00000000D+00
     .,0.25623270D+03,0.146D+03,0.400D+01,0.18630000D+01,0.00000000D+00
     .,0.17343600D+03,0.146D+03,0.500D+01,0.18630000D+01,0.00000000D+00
     .,0.11801370D+03,0.146D+03,0.600D+01,0.18630000D+01,0.00000000D+00
     .,0.83171000D+02,0.146D+03,0.700D+01,0.18630000D+01,0.00000000D+00
     .,0.63412500D+02,0.146D+03,0.800D+01,0.18630000D+01,0.00000000D+00
     .,0.48382400D+02,0.146D+03,0.900D+01,0.18630000D+01,0.00000000D+00
     .,0.37464700D+02,0.146D+03,0.100D+02,0.18630000D+01,0.00000000D+00
     .,0.53116760D+03,0.146D+03,0.110D+02,0.18630000D+01,0.00000000D+00
     .,0.40830220D+03,0.146D+03,0.120D+02,0.18630000D+01,0.00000000D+00
     .,0.37687760D+03,0.146D+03,0.130D+02,0.18630000D+01,0.00000000D+00
     .,0.29781740D+03,0.146D+03,0.140D+02,0.18630000D+01,0.00000000D+00
     .,0.23321060D+03,0.146D+03,0.150D+02,0.18630000D+01,0.00000000D+00
     .,0.19433730D+03,0.146D+03,0.160D+02,0.18630000D+01,0.00000000D+00
     .,0.15949140D+03,0.146D+03,0.170D+02,0.18630000D+01,0.00000000D+00
     .,0.13113380D+03,0.146D+03,0.180D+02,0.18630000D+01,0.00000000D+00
     .,0.87329840D+03,0.146D+03,0.190D+02,0.18630000D+01,0.00000000D+00
     .,0.71849830D+03,0.146D+03,0.200D+02,0.18630000D+01,0.00000000D+00
     ./)
      pars(  46001:  46100)=(/
     . 0.59349940D+03,0.146D+03,0.210D+02,0.18630000D+01,0.00000000D+00
     .,0.57359720D+03,0.146D+03,0.220D+02,0.18630000D+01,0.00000000D+00
     .,0.52547320D+03,0.146D+03,0.230D+02,0.18630000D+01,0.00000000D+00
     .,0.41479180D+03,0.146D+03,0.240D+02,0.18630000D+01,0.00000000D+00
     .,0.45273530D+03,0.146D+03,0.250D+02,0.18630000D+01,0.00000000D+00
     .,0.35613440D+03,0.146D+03,0.260D+02,0.18630000D+01,0.00000000D+00
     .,0.37699280D+03,0.146D+03,0.270D+02,0.18630000D+01,0.00000000D+00
     .,0.38812000D+03,0.146D+03,0.280D+02,0.18630000D+01,0.00000000D+00
     .,0.29838940D+03,0.146D+03,0.290D+02,0.18630000D+01,0.00000000D+00
     .,0.30606330D+03,0.146D+03,0.300D+02,0.18630000D+01,0.00000000D+00
     .,0.36187800D+03,0.146D+03,0.310D+02,0.18630000D+01,0.00000000D+00
     .,0.31969460D+03,0.146D+03,0.320D+02,0.18630000D+01,0.00000000D+00
     .,0.27350100D+03,0.146D+03,0.330D+02,0.18630000D+01,0.00000000D+00
     .,0.24609210D+03,0.146D+03,0.340D+02,0.18630000D+01,0.00000000D+00
     .,0.21609450D+03,0.146D+03,0.350D+02,0.18630000D+01,0.00000000D+00
     .,0.18863390D+03,0.146D+03,0.360D+02,0.18630000D+01,0.00000000D+00
     .,0.97940320D+03,0.146D+03,0.370D+02,0.18630000D+01,0.00000000D+00
     .,0.85671690D+03,0.146D+03,0.380D+02,0.18630000D+01,0.00000000D+00
     .,0.75059330D+03,0.146D+03,0.390D+02,0.18630000D+01,0.00000000D+00
     .,0.67499290D+03,0.146D+03,0.400D+02,0.18630000D+01,0.00000000D+00
     ./)
      pars(  46101:  46200)=(/
     . 0.61597590D+03,0.146D+03,0.410D+02,0.18630000D+01,0.00000000D+00
     .,0.47670900D+03,0.146D+03,0.420D+02,0.18630000D+01,0.00000000D+00
     .,0.53137000D+03,0.146D+03,0.430D+02,0.18630000D+01,0.00000000D+00
     .,0.40593980D+03,0.146D+03,0.440D+02,0.18630000D+01,0.00000000D+00
     .,0.44306590D+03,0.146D+03,0.450D+02,0.18630000D+01,0.00000000D+00
     .,0.41113610D+03,0.146D+03,0.460D+02,0.18630000D+01,0.00000000D+00
     .,0.34367070D+03,0.146D+03,0.470D+02,0.18630000D+01,0.00000000D+00
     .,0.36264320D+03,0.146D+03,0.480D+02,0.18630000D+01,0.00000000D+00
     .,0.45405760D+03,0.146D+03,0.490D+02,0.18630000D+01,0.00000000D+00
     .,0.42012680D+03,0.146D+03,0.500D+02,0.18630000D+01,0.00000000D+00
     .,0.37509950D+03,0.146D+03,0.510D+02,0.18630000D+01,0.00000000D+00
     .,0.34860740D+03,0.146D+03,0.520D+02,0.18630000D+01,0.00000000D+00
     .,0.31592880D+03,0.146D+03,0.530D+02,0.18630000D+01,0.00000000D+00
     .,0.28479640D+03,0.146D+03,0.540D+02,0.18630000D+01,0.00000000D+00
     .,0.11937040D+04,0.146D+03,0.550D+02,0.18630000D+01,0.00000000D+00
     .,0.10930044D+04,0.146D+03,0.560D+02,0.18630000D+01,0.00000000D+00
     .,0.96013460D+03,0.146D+03,0.570D+02,0.18630000D+01,0.00000000D+00
     .,0.44336290D+03,0.146D+03,0.580D+02,0.18630000D+01,0.27991000D+01
     .,0.96887580D+03,0.146D+03,0.590D+02,0.18630000D+01,0.00000000D+00
     .,0.93022500D+03,0.146D+03,0.600D+02,0.18630000D+01,0.00000000D+00
     ./)
      pars(  46201:  46300)=(/
     . 0.90685110D+03,0.146D+03,0.610D+02,0.18630000D+01,0.00000000D+00
     .,0.88534770D+03,0.146D+03,0.620D+02,0.18630000D+01,0.00000000D+00
     .,0.86627480D+03,0.146D+03,0.630D+02,0.18630000D+01,0.00000000D+00
     .,0.68220730D+03,0.146D+03,0.640D+02,0.18630000D+01,0.00000000D+00
     .,0.76879040D+03,0.146D+03,0.650D+02,0.18630000D+01,0.00000000D+00
     .,0.74151490D+03,0.146D+03,0.660D+02,0.18630000D+01,0.00000000D+00
     .,0.78120400D+03,0.146D+03,0.670D+02,0.18630000D+01,0.00000000D+00
     .,0.76457050D+03,0.146D+03,0.680D+02,0.18630000D+01,0.00000000D+00
     .,0.74957790D+03,0.146D+03,0.690D+02,0.18630000D+01,0.00000000D+00
     .,0.74073920D+03,0.146D+03,0.700D+02,0.18630000D+01,0.00000000D+00
     .,0.62471570D+03,0.146D+03,0.710D+02,0.18630000D+01,0.00000000D+00
     .,0.61425310D+03,0.146D+03,0.720D+02,0.18630000D+01,0.00000000D+00
     .,0.56115170D+03,0.146D+03,0.730D+02,0.18630000D+01,0.00000000D+00
     .,0.47494930D+03,0.146D+03,0.740D+02,0.18630000D+01,0.00000000D+00
     .,0.48315690D+03,0.146D+03,0.750D+02,0.18630000D+01,0.00000000D+00
     .,0.43846770D+03,0.146D+03,0.760D+02,0.18630000D+01,0.00000000D+00
     .,0.40208340D+03,0.146D+03,0.770D+02,0.18630000D+01,0.00000000D+00
     .,0.33498540D+03,0.146D+03,0.780D+02,0.18630000D+01,0.00000000D+00
     .,0.31334830D+03,0.146D+03,0.790D+02,0.18630000D+01,0.00000000D+00
     .,0.32218840D+03,0.146D+03,0.800D+02,0.18630000D+01,0.00000000D+00
     ./)
      pars(  46301:  46400)=(/
     . 0.46735710D+03,0.146D+03,0.810D+02,0.18630000D+01,0.00000000D+00
     .,0.45686740D+03,0.146D+03,0.820D+02,0.18630000D+01,0.00000000D+00
     .,0.42032740D+03,0.146D+03,0.830D+02,0.18630000D+01,0.00000000D+00
     .,0.40133490D+03,0.146D+03,0.840D+02,0.18630000D+01,0.00000000D+00
     .,0.37107280D+03,0.146D+03,0.850D+02,0.18630000D+01,0.00000000D+00
     .,0.34084580D+03,0.146D+03,0.860D+02,0.18630000D+01,0.00000000D+00
     .,0.11269549D+04,0.146D+03,0.870D+02,0.18630000D+01,0.00000000D+00
     .,0.10810405D+04,0.146D+03,0.880D+02,0.18630000D+01,0.00000000D+00
     .,0.95553590D+03,0.146D+03,0.890D+02,0.18630000D+01,0.00000000D+00
     .,0.85969500D+03,0.146D+03,0.900D+02,0.18630000D+01,0.00000000D+00
     .,0.85393730D+03,0.146D+03,0.910D+02,0.18630000D+01,0.00000000D+00
     .,0.82694450D+03,0.146D+03,0.920D+02,0.18630000D+01,0.00000000D+00
     .,0.85119160D+03,0.146D+03,0.930D+02,0.18630000D+01,0.00000000D+00
     .,0.82429470D+03,0.146D+03,0.940D+02,0.18630000D+01,0.00000000D+00
     .,0.46348900D+02,0.146D+03,0.101D+03,0.18630000D+01,0.00000000D+00
     .,0.14912090D+03,0.146D+03,0.103D+03,0.18630000D+01,0.98650000D+00
     .,0.19048260D+03,0.146D+03,0.104D+03,0.18630000D+01,0.98080000D+00
     .,0.14620200D+03,0.146D+03,0.105D+03,0.18630000D+01,0.97060000D+00
     .,0.11079460D+03,0.146D+03,0.106D+03,0.18630000D+01,0.98680000D+00
     .,0.77647900D+02,0.146D+03,0.107D+03,0.18630000D+01,0.99440000D+00
     ./)
      pars(  46401:  46500)=(/
     . 0.57004900D+02,0.146D+03,0.108D+03,0.18630000D+01,0.99250000D+00
     .,0.39663600D+02,0.146D+03,0.109D+03,0.18630000D+01,0.99820000D+00
     .,0.21823330D+03,0.146D+03,0.111D+03,0.18630000D+01,0.96840000D+00
     .,0.33722560D+03,0.146D+03,0.112D+03,0.18630000D+01,0.96280000D+00
     .,0.34179460D+03,0.146D+03,0.113D+03,0.18630000D+01,0.96480000D+00
     .,0.27549250D+03,0.146D+03,0.114D+03,0.18630000D+01,0.95070000D+00
     .,0.22646050D+03,0.146D+03,0.115D+03,0.18630000D+01,0.99470000D+00
     .,0.19221720D+03,0.146D+03,0.116D+03,0.18630000D+01,0.99480000D+00
     .,0.15785240D+03,0.146D+03,0.117D+03,0.18630000D+01,0.99720000D+00
     .,0.30246350D+03,0.146D+03,0.119D+03,0.18630000D+01,0.97670000D+00
     .,0.57613740D+03,0.146D+03,0.120D+03,0.18630000D+01,0.98310000D+00
     .,0.30248980D+03,0.146D+03,0.121D+03,0.18630000D+01,0.18627000D+01
     .,0.29221350D+03,0.146D+03,0.122D+03,0.18630000D+01,0.18299000D+01
     .,0.28642780D+03,0.146D+03,0.123D+03,0.18630000D+01,0.19138000D+01
     .,0.28377850D+03,0.146D+03,0.124D+03,0.18630000D+01,0.18269000D+01
     .,0.26126350D+03,0.146D+03,0.125D+03,0.18630000D+01,0.16406000D+01
     .,0.24201470D+03,0.146D+03,0.126D+03,0.18630000D+01,0.16483000D+01
     .,0.23097880D+03,0.146D+03,0.127D+03,0.18630000D+01,0.17149000D+01
     .,0.22581650D+03,0.146D+03,0.128D+03,0.18630000D+01,0.17937000D+01
     .,0.22301540D+03,0.146D+03,0.129D+03,0.18630000D+01,0.95760000D+00
     ./)
      pars(  46501:  46600)=(/
     . 0.20945460D+03,0.146D+03,0.130D+03,0.18630000D+01,0.19419000D+01
     .,0.34006750D+03,0.146D+03,0.131D+03,0.18630000D+01,0.96010000D+00
     .,0.29932800D+03,0.146D+03,0.132D+03,0.18630000D+01,0.94340000D+00
     .,0.26894270D+03,0.146D+03,0.133D+03,0.18630000D+01,0.98890000D+00
     .,0.24619370D+03,0.146D+03,0.134D+03,0.18630000D+01,0.99010000D+00
     .,0.21757380D+03,0.146D+03,0.135D+03,0.18630000D+01,0.99740000D+00
     .,0.36130470D+03,0.146D+03,0.137D+03,0.18630000D+01,0.97380000D+00
     .,0.70165500D+03,0.146D+03,0.138D+03,0.18630000D+01,0.98010000D+00
     .,0.53685660D+03,0.146D+03,0.139D+03,0.18630000D+01,0.19153000D+01
     .,0.40042090D+03,0.146D+03,0.140D+03,0.18630000D+01,0.19355000D+01
     .,0.40443000D+03,0.146D+03,0.141D+03,0.18630000D+01,0.19545000D+01
     .,0.37755350D+03,0.146D+03,0.142D+03,0.18630000D+01,0.19420000D+01
     .,0.42312790D+03,0.146D+03,0.143D+03,0.18630000D+01,0.16682000D+01
     .,0.32968260D+03,0.146D+03,0.144D+03,0.18630000D+01,0.18584000D+01
     .,0.30870590D+03,0.146D+03,0.145D+03,0.18630000D+01,0.19003000D+01
     .,0.28695190D+03,0.146D+03,0.146D+03,0.18630000D+01,0.18630000D+01
     .,0.27951800D+02,0.147D+03,0.100D+01,0.96790000D+00,0.91180000D+00
     .,0.18773100D+02,0.147D+03,0.200D+01,0.96790000D+00,0.00000000D+00
     .,0.43260780D+03,0.147D+03,0.300D+01,0.96790000D+00,0.00000000D+00
     .,0.24865730D+03,0.147D+03,0.400D+01,0.96790000D+00,0.00000000D+00
     ./)
      pars(  46601:  46700)=(/
     . 0.16783500D+03,0.147D+03,0.500D+01,0.96790000D+00,0.00000000D+00
     .,0.11399930D+03,0.147D+03,0.600D+01,0.96790000D+00,0.00000000D+00
     .,0.80266300D+02,0.147D+03,0.700D+01,0.96790000D+00,0.00000000D+00
     .,0.61174800D+02,0.147D+03,0.800D+01,0.96790000D+00,0.00000000D+00
     .,0.46672600D+02,0.147D+03,0.900D+01,0.96790000D+00,0.00000000D+00
     .,0.36148600D+02,0.147D+03,0.100D+02,0.96790000D+00,0.00000000D+00
     .,0.51756800D+03,0.147D+03,0.110D+02,0.96790000D+00,0.00000000D+00
     .,0.39661250D+03,0.147D+03,0.120D+02,0.96790000D+00,0.00000000D+00
     .,0.36552690D+03,0.147D+03,0.130D+02,0.96790000D+00,0.00000000D+00
     .,0.28828630D+03,0.147D+03,0.140D+02,0.96790000D+00,0.00000000D+00
     .,0.22539450D+03,0.147D+03,0.150D+02,0.96790000D+00,0.00000000D+00
     .,0.18765970D+03,0.147D+03,0.160D+02,0.96790000D+00,0.00000000D+00
     .,0.15390320D+03,0.147D+03,0.170D+02,0.96790000D+00,0.00000000D+00
     .,0.12648250D+03,0.147D+03,0.180D+02,0.96790000D+00,0.00000000D+00
     .,0.85100880D+03,0.147D+03,0.190D+02,0.96790000D+00,0.00000000D+00
     .,0.69886180D+03,0.147D+03,0.200D+02,0.96790000D+00,0.00000000D+00
     .,0.57696360D+03,0.147D+03,0.210D+02,0.96790000D+00,0.00000000D+00
     .,0.55728800D+03,0.147D+03,0.220D+02,0.96790000D+00,0.00000000D+00
     .,0.51036440D+03,0.147D+03,0.230D+02,0.96790000D+00,0.00000000D+00
     .,0.40283830D+03,0.147D+03,0.240D+02,0.96790000D+00,0.00000000D+00
     ./)
      pars(  46701:  46800)=(/
     . 0.43950990D+03,0.147D+03,0.250D+02,0.96790000D+00,0.00000000D+00
     .,0.34569950D+03,0.147D+03,0.260D+02,0.96790000D+00,0.00000000D+00
     .,0.36569250D+03,0.147D+03,0.270D+02,0.96790000D+00,0.00000000D+00
     .,0.37662810D+03,0.147D+03,0.280D+02,0.96790000D+00,0.00000000D+00
     .,0.28955800D+03,0.147D+03,0.290D+02,0.96790000D+00,0.00000000D+00
     .,0.29667280D+03,0.147D+03,0.300D+02,0.96790000D+00,0.00000000D+00
     .,0.35080100D+03,0.147D+03,0.310D+02,0.96790000D+00,0.00000000D+00
     .,0.30945430D+03,0.147D+03,0.320D+02,0.96790000D+00,0.00000000D+00
     .,0.26439570D+03,0.147D+03,0.330D+02,0.96790000D+00,0.00000000D+00
     .,0.23772170D+03,0.147D+03,0.340D+02,0.96790000D+00,0.00000000D+00
     .,0.20860440D+03,0.147D+03,0.350D+02,0.96790000D+00,0.00000000D+00
     .,0.18200160D+03,0.147D+03,0.360D+02,0.96790000D+00,0.00000000D+00
     .,0.95413200D+03,0.147D+03,0.370D+02,0.96790000D+00,0.00000000D+00
     .,0.83331680D+03,0.147D+03,0.380D+02,0.96790000D+00,0.00000000D+00
     .,0.72933160D+03,0.147D+03,0.390D+02,0.96790000D+00,0.00000000D+00
     .,0.65543290D+03,0.147D+03,0.400D+02,0.96790000D+00,0.00000000D+00
     .,0.59785280D+03,0.147D+03,0.410D+02,0.96790000D+00,0.00000000D+00
     .,0.46233830D+03,0.147D+03,0.420D+02,0.96790000D+00,0.00000000D+00
     .,0.51549530D+03,0.147D+03,0.430D+02,0.96790000D+00,0.00000000D+00
     .,0.39350880D+03,0.147D+03,0.440D+02,0.96790000D+00,0.00000000D+00
     ./)
      pars(  46801:  46900)=(/
     . 0.42951430D+03,0.147D+03,0.450D+02,0.96790000D+00,0.00000000D+00
     .,0.39846620D+03,0.147D+03,0.460D+02,0.96790000D+00,0.00000000D+00
     .,0.33315830D+03,0.147D+03,0.470D+02,0.96790000D+00,0.00000000D+00
     .,0.35137120D+03,0.147D+03,0.480D+02,0.96790000D+00,0.00000000D+00
     .,0.44027330D+03,0.147D+03,0.490D+02,0.96790000D+00,0.00000000D+00
     .,0.40690290D+03,0.147D+03,0.500D+02,0.96790000D+00,0.00000000D+00
     .,0.36287400D+03,0.147D+03,0.510D+02,0.96790000D+00,0.00000000D+00
     .,0.33701310D+03,0.147D+03,0.520D+02,0.96790000D+00,0.00000000D+00
     .,0.30521220D+03,0.147D+03,0.530D+02,0.96790000D+00,0.00000000D+00
     .,0.27497440D+03,0.147D+03,0.540D+02,0.96790000D+00,0.00000000D+00
     .,0.11625520D+04,0.147D+03,0.550D+02,0.96790000D+00,0.00000000D+00
     .,0.10633544D+04,0.147D+03,0.560D+02,0.96790000D+00,0.00000000D+00
     .,0.93316800D+03,0.147D+03,0.570D+02,0.96790000D+00,0.00000000D+00
     .,0.42899800D+03,0.147D+03,0.580D+02,0.96790000D+00,0.27991000D+01
     .,0.94235470D+03,0.147D+03,0.590D+02,0.96790000D+00,0.00000000D+00
     .,0.90466880D+03,0.147D+03,0.600D+02,0.96790000D+00,0.00000000D+00
     .,0.88190710D+03,0.147D+03,0.610D+02,0.96790000D+00,0.00000000D+00
     .,0.86096980D+03,0.147D+03,0.620D+02,0.96790000D+00,0.00000000D+00
     .,0.84239660D+03,0.147D+03,0.630D+02,0.96790000D+00,0.00000000D+00
     .,0.66260290D+03,0.147D+03,0.640D+02,0.96790000D+00,0.00000000D+00
     ./)
      pars(  46901:  47000)=(/
     . 0.74772470D+03,0.147D+03,0.650D+02,0.96790000D+00,0.00000000D+00
     .,0.72101030D+03,0.147D+03,0.660D+02,0.96790000D+00,0.00000000D+00
     .,0.75950840D+03,0.147D+03,0.670D+02,0.96790000D+00,0.00000000D+00
     .,0.74332210D+03,0.147D+03,0.680D+02,0.96790000D+00,0.00000000D+00
     .,0.72872030D+03,0.147D+03,0.690D+02,0.96790000D+00,0.00000000D+00
     .,0.72016540D+03,0.147D+03,0.700D+02,0.96790000D+00,0.00000000D+00
     .,0.60683130D+03,0.147D+03,0.710D+02,0.96790000D+00,0.00000000D+00
     .,0.59598520D+03,0.147D+03,0.720D+02,0.96790000D+00,0.00000000D+00
     .,0.54409850D+03,0.147D+03,0.730D+02,0.96790000D+00,0.00000000D+00
     .,0.46026700D+03,0.147D+03,0.740D+02,0.96790000D+00,0.00000000D+00
     .,0.46810840D+03,0.147D+03,0.750D+02,0.96790000D+00,0.00000000D+00
     .,0.42458230D+03,0.147D+03,0.760D+02,0.96790000D+00,0.00000000D+00
     .,0.38918700D+03,0.147D+03,0.770D+02,0.96790000D+00,0.00000000D+00
     .,0.32412400D+03,0.147D+03,0.780D+02,0.96790000D+00,0.00000000D+00
     .,0.30315550D+03,0.147D+03,0.790D+02,0.96790000D+00,0.00000000D+00
     .,0.31163540D+03,0.147D+03,0.800D+02,0.96790000D+00,0.00000000D+00
     .,0.45304550D+03,0.147D+03,0.810D+02,0.96790000D+00,0.00000000D+00
     .,0.44250950D+03,0.147D+03,0.820D+02,0.96790000D+00,0.00000000D+00
     .,0.40670840D+03,0.147D+03,0.830D+02,0.96790000D+00,0.00000000D+00
     .,0.38810010D+03,0.147D+03,0.840D+02,0.96790000D+00,0.00000000D+00
     ./)
      pars(  47001:  47100)=(/
     . 0.35859540D+03,0.147D+03,0.850D+02,0.96790000D+00,0.00000000D+00
     .,0.32919900D+03,0.147D+03,0.860D+02,0.96790000D+00,0.00000000D+00
     .,0.10968369D+04,0.147D+03,0.870D+02,0.96790000D+00,0.00000000D+00
     .,0.10511515D+04,0.147D+03,0.880D+02,0.96790000D+00,0.00000000D+00
     .,0.92833020D+03,0.147D+03,0.890D+02,0.96790000D+00,0.00000000D+00
     .,0.83431790D+03,0.147D+03,0.900D+02,0.96790000D+00,0.00000000D+00
     .,0.82919230D+03,0.147D+03,0.910D+02,0.96790000D+00,0.00000000D+00
     .,0.80298090D+03,0.147D+03,0.920D+02,0.96790000D+00,0.00000000D+00
     .,0.82716130D+03,0.147D+03,0.930D+02,0.96790000D+00,0.00000000D+00
     .,0.80093320D+03,0.147D+03,0.940D+02,0.96790000D+00,0.00000000D+00
     .,0.44783500D+02,0.147D+03,0.101D+03,0.96790000D+00,0.00000000D+00
     .,0.14465630D+03,0.147D+03,0.103D+03,0.96790000D+00,0.98650000D+00
     .,0.18465300D+03,0.147D+03,0.104D+03,0.96790000D+00,0.98080000D+00
     .,0.14140600D+03,0.147D+03,0.105D+03,0.96790000D+00,0.97060000D+00
     .,0.10702860D+03,0.147D+03,0.106D+03,0.96790000D+00,0.98680000D+00
     .,0.74935400D+02,0.147D+03,0.107D+03,0.96790000D+00,0.99440000D+00
     .,0.54985600D+02,0.147D+03,0.108D+03,0.96790000D+00,0.99250000D+00
     .,0.38253900D+02,0.147D+03,0.109D+03,0.96790000D+00,0.99820000D+00
     .,0.21184190D+03,0.147D+03,0.111D+03,0.96790000D+00,0.96840000D+00
     .,0.32735090D+03,0.147D+03,0.112D+03,0.96790000D+00,0.96280000D+00
     ./)
      pars(  47101:  47200)=(/
     . 0.33138320D+03,0.147D+03,0.113D+03,0.96790000D+00,0.96480000D+00
     .,0.26660080D+03,0.147D+03,0.114D+03,0.96790000D+00,0.95070000D+00
     .,0.21886170D+03,0.147D+03,0.115D+03,0.96790000D+00,0.99470000D+00
     .,0.18561970D+03,0.147D+03,0.116D+03,0.96790000D+00,0.99480000D+00
     .,0.15232700D+03,0.147D+03,0.117D+03,0.96790000D+00,0.99720000D+00
     .,0.29329110D+03,0.147D+03,0.119D+03,0.96790000D+00,0.97670000D+00
     .,0.56012980D+03,0.147D+03,0.120D+03,0.96790000D+00,0.98310000D+00
     .,0.29290630D+03,0.147D+03,0.121D+03,0.96790000D+00,0.18627000D+01
     .,0.28293320D+03,0.147D+03,0.122D+03,0.96790000D+00,0.18299000D+01
     .,0.27735250D+03,0.147D+03,0.123D+03,0.96790000D+00,0.19138000D+01
     .,0.27484520D+03,0.147D+03,0.124D+03,0.96790000D+00,0.18269000D+01
     .,0.25282980D+03,0.147D+03,0.125D+03,0.96790000D+00,0.16406000D+01
     .,0.23415110D+03,0.147D+03,0.126D+03,0.96790000D+00,0.16483000D+01
     .,0.22348050D+03,0.147D+03,0.127D+03,0.96790000D+00,0.17149000D+01
     .,0.21850750D+03,0.147D+03,0.128D+03,0.96790000D+00,0.17937000D+01
     .,0.21595810D+03,0.147D+03,0.129D+03,0.96790000D+00,0.95760000D+00
     .,0.20256850D+03,0.147D+03,0.130D+03,0.96790000D+00,0.19419000D+01
     .,0.32953870D+03,0.147D+03,0.131D+03,0.96790000D+00,0.96010000D+00
     .,0.28964110D+03,0.147D+03,0.132D+03,0.96790000D+00,0.94340000D+00
     .,0.25997060D+03,0.147D+03,0.133D+03,0.96790000D+00,0.98890000D+00
     ./)
      pars(  47201:  47300)=(/
     . 0.23782380D+03,0.147D+03,0.134D+03,0.96790000D+00,0.99010000D+00
     .,0.21004110D+03,0.147D+03,0.135D+03,0.96790000D+00,0.99740000D+00
     .,0.35022950D+03,0.147D+03,0.137D+03,0.96790000D+00,0.97380000D+00
     .,0.68218170D+03,0.147D+03,0.138D+03,0.96790000D+00,0.98010000D+00
     .,0.52087240D+03,0.147D+03,0.139D+03,0.96790000D+00,0.19153000D+01
     .,0.38773360D+03,0.147D+03,0.140D+03,0.96790000D+00,0.19355000D+01
     .,0.39166370D+03,0.147D+03,0.141D+03,0.96790000D+00,0.19545000D+01
     .,0.36552610D+03,0.147D+03,0.142D+03,0.96790000D+00,0.19420000D+01
     .,0.41006670D+03,0.147D+03,0.143D+03,0.96790000D+00,0.16682000D+01
     .,0.31896650D+03,0.147D+03,0.144D+03,0.96790000D+00,0.18584000D+01
     .,0.29866780D+03,0.147D+03,0.145D+03,0.96790000D+00,0.19003000D+01
     .,0.27760230D+03,0.147D+03,0.146D+03,0.96790000D+00,0.18630000D+01
     .,0.26861300D+03,0.147D+03,0.147D+03,0.96790000D+00,0.96790000D+00
     .,0.27928100D+02,0.148D+03,0.100D+01,0.19539000D+01,0.91180000D+00
     .,0.18860700D+02,0.148D+03,0.200D+01,0.19539000D+01,0.00000000D+00
     .,0.40844290D+03,0.148D+03,0.300D+01,0.19539000D+01,0.00000000D+00
     .,0.24209500D+03,0.148D+03,0.400D+01,0.19539000D+01,0.00000000D+00
     .,0.16565870D+03,0.148D+03,0.500D+01,0.19539000D+01,0.00000000D+00
     .,0.11343690D+03,0.148D+03,0.600D+01,0.19539000D+01,0.00000000D+00
     .,0.80249200D+02,0.148D+03,0.700D+01,0.19539000D+01,0.00000000D+00
     ./)
      pars(  47301:  47400)=(/
     . 0.61323400D+02,0.148D+03,0.800D+01,0.19539000D+01,0.00000000D+00
     .,0.46869200D+02,0.148D+03,0.900D+01,0.19539000D+01,0.00000000D+00
     .,0.36340100D+02,0.148D+03,0.100D+02,0.19539000D+01,0.00000000D+00
     .,0.48951290D+03,0.148D+03,0.110D+02,0.19539000D+01,0.00000000D+00
     .,0.38417280D+03,0.148D+03,0.120D+02,0.19539000D+01,0.00000000D+00
     .,0.35686540D+03,0.148D+03,0.130D+02,0.19539000D+01,0.00000000D+00
     .,0.28421390D+03,0.148D+03,0.140D+02,0.19539000D+01,0.00000000D+00
     .,0.22377830D+03,0.148D+03,0.150D+02,0.19539000D+01,0.00000000D+00
     .,0.18701630D+03,0.148D+03,0.160D+02,0.19539000D+01,0.00000000D+00
     .,0.15385920D+03,0.148D+03,0.170D+02,0.19539000D+01,0.00000000D+00
     .,0.12673560D+03,0.148D+03,0.180D+02,0.19539000D+01,0.00000000D+00
     .,0.80025930D+03,0.148D+03,0.190D+02,0.19539000D+01,0.00000000D+00
     .,0.67026330D+03,0.148D+03,0.200D+02,0.19539000D+01,0.00000000D+00
     .,0.55566790D+03,0.148D+03,0.210D+02,0.19539000D+01,0.00000000D+00
     .,0.53855020D+03,0.148D+03,0.220D+02,0.19539000D+01,0.00000000D+00
     .,0.49422370D+03,0.148D+03,0.230D+02,0.19539000D+01,0.00000000D+00
     .,0.38999530D+03,0.148D+03,0.240D+02,0.19539000D+01,0.00000000D+00
     .,0.42685790D+03,0.148D+03,0.250D+02,0.19539000D+01,0.00000000D+00
     .,0.33576420D+03,0.148D+03,0.260D+02,0.19539000D+01,0.00000000D+00
     .,0.35691720D+03,0.148D+03,0.270D+02,0.19539000D+01,0.00000000D+00
     ./)
      pars(  47401:  47500)=(/
     . 0.36686680D+03,0.148D+03,0.280D+02,0.19539000D+01,0.00000000D+00
     .,0.28186750D+03,0.148D+03,0.290D+02,0.19539000D+01,0.00000000D+00
     .,0.29078950D+03,0.148D+03,0.300D+02,0.19539000D+01,0.00000000D+00
     .,0.34329430D+03,0.148D+03,0.310D+02,0.19539000D+01,0.00000000D+00
     .,0.30514680D+03,0.148D+03,0.320D+02,0.19539000D+01,0.00000000D+00
     .,0.26228400D+03,0.148D+03,0.330D+02,0.19539000D+01,0.00000000D+00
     .,0.23657790D+03,0.148D+03,0.340D+02,0.19539000D+01,0.00000000D+00
     .,0.20821130D+03,0.148D+03,0.350D+02,0.19539000D+01,0.00000000D+00
     .,0.18208670D+03,0.148D+03,0.360D+02,0.19539000D+01,0.00000000D+00
     .,0.89853420D+03,0.148D+03,0.370D+02,0.19539000D+01,0.00000000D+00
     .,0.79845280D+03,0.148D+03,0.380D+02,0.19539000D+01,0.00000000D+00
     .,0.70379920D+03,0.148D+03,0.390D+02,0.19539000D+01,0.00000000D+00
     .,0.63513700D+03,0.148D+03,0.400D+02,0.19539000D+01,0.00000000D+00
     .,0.58084120D+03,0.148D+03,0.410D+02,0.19539000D+01,0.00000000D+00
     .,0.45097760D+03,0.148D+03,0.420D+02,0.19539000D+01,0.00000000D+00
     .,0.50211680D+03,0.148D+03,0.430D+02,0.19539000D+01,0.00000000D+00
     .,0.38494360D+03,0.148D+03,0.440D+02,0.19539000D+01,0.00000000D+00
     .,0.42037460D+03,0.148D+03,0.450D+02,0.19539000D+01,0.00000000D+00
     .,0.39057700D+03,0.148D+03,0.460D+02,0.19539000D+01,0.00000000D+00
     .,0.32596620D+03,0.148D+03,0.470D+02,0.19539000D+01,0.00000000D+00
     ./)
      pars(  47501:  47600)=(/
     . 0.34505990D+03,0.148D+03,0.480D+02,0.19539000D+01,0.00000000D+00
     .,0.43036070D+03,0.148D+03,0.490D+02,0.19539000D+01,0.00000000D+00
     .,0.40034740D+03,0.148D+03,0.500D+02,0.19539000D+01,0.00000000D+00
     .,0.35905550D+03,0.148D+03,0.510D+02,0.19539000D+01,0.00000000D+00
     .,0.33449170D+03,0.148D+03,0.520D+02,0.19539000D+01,0.00000000D+00
     .,0.30384100D+03,0.148D+03,0.530D+02,0.19539000D+01,0.00000000D+00
     .,0.27444070D+03,0.148D+03,0.540D+02,0.19539000D+01,0.00000000D+00
     .,0.10949943D+04,0.148D+03,0.550D+02,0.19539000D+01,0.00000000D+00
     .,0.10161802D+04,0.148D+03,0.560D+02,0.19539000D+01,0.00000000D+00
     .,0.89820920D+03,0.148D+03,0.570D+02,0.19539000D+01,0.00000000D+00
     .,0.42389100D+03,0.148D+03,0.580D+02,0.19539000D+01,0.27991000D+01
     .,0.90262940D+03,0.148D+03,0.590D+02,0.19539000D+01,0.00000000D+00
     .,0.86759380D+03,0.148D+03,0.600D+02,0.19539000D+01,0.00000000D+00
     .,0.84606160D+03,0.148D+03,0.610D+02,0.19539000D+01,0.00000000D+00
     .,0.82622610D+03,0.148D+03,0.620D+02,0.19539000D+01,0.00000000D+00
     .,0.80864130D+03,0.148D+03,0.630D+02,0.19539000D+01,0.00000000D+00
     .,0.64073930D+03,0.148D+03,0.640D+02,0.19539000D+01,0.00000000D+00
     .,0.71462060D+03,0.148D+03,0.650D+02,0.19539000D+01,0.00000000D+00
     .,0.69004230D+03,0.148D+03,0.660D+02,0.19539000D+01,0.00000000D+00
     .,0.73042860D+03,0.148D+03,0.670D+02,0.19539000D+01,0.00000000D+00
     ./)
      pars(  47601:  47700)=(/
     . 0.71501980D+03,0.148D+03,0.680D+02,0.19539000D+01,0.00000000D+00
     .,0.70118290D+03,0.148D+03,0.690D+02,0.19539000D+01,0.00000000D+00
     .,0.69275040D+03,0.148D+03,0.700D+02,0.19539000D+01,0.00000000D+00
     .,0.58668300D+03,0.148D+03,0.710D+02,0.19539000D+01,0.00000000D+00
     .,0.58027960D+03,0.148D+03,0.720D+02,0.19539000D+01,0.00000000D+00
     .,0.53179240D+03,0.148D+03,0.730D+02,0.19539000D+01,0.00000000D+00
     .,0.45093600D+03,0.148D+03,0.740D+02,0.19539000D+01,0.00000000D+00
     .,0.45935930D+03,0.148D+03,0.750D+02,0.19539000D+01,0.00000000D+00
     .,0.41790820D+03,0.148D+03,0.760D+02,0.19539000D+01,0.00000000D+00
     .,0.38396000D+03,0.148D+03,0.770D+02,0.19539000D+01,0.00000000D+00
     .,0.32036400D+03,0.148D+03,0.780D+02,0.19539000D+01,0.00000000D+00
     .,0.29986260D+03,0.148D+03,0.790D+02,0.19539000D+01,0.00000000D+00
     .,0.30870590D+03,0.148D+03,0.800D+02,0.19539000D+01,0.00000000D+00
     .,0.44321180D+03,0.148D+03,0.810D+02,0.19539000D+01,0.00000000D+00
     .,0.43519640D+03,0.148D+03,0.820D+02,0.19539000D+01,0.00000000D+00
     .,0.40209000D+03,0.148D+03,0.830D+02,0.19539000D+01,0.00000000D+00
     .,0.38476220D+03,0.148D+03,0.840D+02,0.19539000D+01,0.00000000D+00
     .,0.35659780D+03,0.148D+03,0.850D+02,0.19539000D+01,0.00000000D+00
     .,0.32818400D+03,0.148D+03,0.860D+02,0.19539000D+01,0.00000000D+00
     .,0.10391920D+04,0.148D+03,0.870D+02,0.19539000D+01,0.00000000D+00
     ./)
      pars(  47701:  47800)=(/
     . 0.10081792D+04,0.148D+03,0.880D+02,0.19539000D+01,0.00000000D+00
     .,0.89630090D+03,0.148D+03,0.890D+02,0.19539000D+01,0.00000000D+00
     .,0.81114820D+03,0.148D+03,0.900D+02,0.19539000D+01,0.00000000D+00
     .,0.80315200D+03,0.148D+03,0.910D+02,0.19539000D+01,0.00000000D+00
     .,0.77787630D+03,0.148D+03,0.920D+02,0.19539000D+01,0.00000000D+00
     .,0.79775990D+03,0.148D+03,0.930D+02,0.19539000D+01,0.00000000D+00
     .,0.77311930D+03,0.148D+03,0.940D+02,0.19539000D+01,0.00000000D+00
     .,0.44500600D+02,0.148D+03,0.101D+03,0.19539000D+01,0.00000000D+00
     .,0.14109010D+03,0.148D+03,0.103D+03,0.19539000D+01,0.98650000D+00
     .,0.18052070D+03,0.148D+03,0.104D+03,0.19539000D+01,0.98080000D+00
     .,0.13987930D+03,0.148D+03,0.105D+03,0.19539000D+01,0.97060000D+00
     .,0.10643320D+03,0.148D+03,0.106D+03,0.19539000D+01,0.98680000D+00
     .,0.74892500D+02,0.148D+03,0.107D+03,0.19539000D+01,0.99440000D+00
     .,0.55139200D+02,0.148D+03,0.108D+03,0.19539000D+01,0.99250000D+00
     .,0.38490600D+02,0.148D+03,0.109D+03,0.19539000D+01,0.99820000D+00
     .,0.20596470D+03,0.148D+03,0.111D+03,0.19539000D+01,0.96840000D+00
     .,0.31790020D+03,0.148D+03,0.112D+03,0.19539000D+01,0.96280000D+00
     .,0.32408300D+03,0.148D+03,0.113D+03,0.19539000D+01,0.96480000D+00
     .,0.26314380D+03,0.148D+03,0.114D+03,0.19539000D+01,0.95070000D+00
     .,0.21731990D+03,0.148D+03,0.115D+03,0.19539000D+01,0.99470000D+00
     ./)
      pars(  47801:  47900)=(/
     . 0.18495220D+03,0.148D+03,0.116D+03,0.19539000D+01,0.99480000D+00
     .,0.15226460D+03,0.148D+03,0.117D+03,0.19539000D+01,0.99720000D+00
     .,0.28627350D+03,0.148D+03,0.119D+03,0.19539000D+01,0.97670000D+00
     .,0.53708890D+03,0.148D+03,0.120D+03,0.19539000D+01,0.98310000D+00
     .,0.28838750D+03,0.148D+03,0.121D+03,0.19539000D+01,0.18627000D+01
     .,0.27858690D+03,0.148D+03,0.122D+03,0.19539000D+01,0.18299000D+01
     .,0.27303310D+03,0.148D+03,0.123D+03,0.19539000D+01,0.19138000D+01
     .,0.27028900D+03,0.148D+03,0.124D+03,0.19539000D+01,0.18269000D+01
     .,0.24982010D+03,0.148D+03,0.125D+03,0.19539000D+01,0.16406000D+01
     .,0.23161700D+03,0.148D+03,0.126D+03,0.19539000D+01,0.16483000D+01
     .,0.22102480D+03,0.148D+03,0.127D+03,0.19539000D+01,0.17149000D+01
     .,0.21602110D+03,0.148D+03,0.128D+03,0.19539000D+01,0.17937000D+01
     .,0.21274680D+03,0.148D+03,0.129D+03,0.19539000D+01,0.95760000D+00
     .,0.20082240D+03,0.148D+03,0.130D+03,0.19539000D+01,0.19419000D+01
     .,0.32310800D+03,0.148D+03,0.131D+03,0.19539000D+01,0.96010000D+00
     .,0.28605510D+03,0.148D+03,0.132D+03,0.19539000D+01,0.94340000D+00
     .,0.25796890D+03,0.148D+03,0.133D+03,0.19539000D+01,0.98890000D+00
     .,0.23666040D+03,0.148D+03,0.134D+03,0.19539000D+01,0.99010000D+00
     .,0.20960830D+03,0.148D+03,0.135D+03,0.19539000D+01,0.99740000D+00
     .,0.34234100D+03,0.148D+03,0.137D+03,0.19539000D+01,0.97380000D+00
     ./)
      pars(  47901:  48000)=(/
     . 0.65327820D+03,0.148D+03,0.138D+03,0.19539000D+01,0.98010000D+00
     .,0.50567470D+03,0.148D+03,0.139D+03,0.19539000D+01,0.19153000D+01
     .,0.38134050D+03,0.148D+03,0.140D+03,0.19539000D+01,0.19355000D+01
     .,0.38512060D+03,0.148D+03,0.141D+03,0.19539000D+01,0.19545000D+01
     .,0.35990590D+03,0.148D+03,0.142D+03,0.19539000D+01,0.19420000D+01
     .,0.40123820D+03,0.148D+03,0.143D+03,0.19539000D+01,0.16682000D+01
     .,0.31533650D+03,0.148D+03,0.144D+03,0.19539000D+01,0.18584000D+01
     .,0.29530360D+03,0.148D+03,0.145D+03,0.19539000D+01,0.19003000D+01
     .,0.27461540D+03,0.148D+03,0.146D+03,0.19539000D+01,0.18630000D+01
     .,0.26554500D+03,0.148D+03,0.147D+03,0.19539000D+01,0.96790000D+00
     .,0.26345080D+03,0.148D+03,0.148D+03,0.19539000D+01,0.19539000D+01
     .,0.42282700D+02,0.149D+03,0.100D+01,0.96330000D+00,0.91180000D+00
     .,0.27023600D+02,0.149D+03,0.200D+01,0.96330000D+00,0.00000000D+00
     .,0.76925140D+03,0.149D+03,0.300D+01,0.96330000D+00,0.00000000D+00
     .,0.41349650D+03,0.149D+03,0.400D+01,0.96330000D+00,0.00000000D+00
     .,0.26747500D+03,0.149D+03,0.500D+01,0.96330000D+00,0.00000000D+00
     .,0.17545850D+03,0.149D+03,0.600D+01,0.96330000D+00,0.00000000D+00
     .,0.12012790D+03,0.149D+03,0.700D+01,0.96330000D+00,0.00000000D+00
     .,0.89652000D+02,0.149D+03,0.800D+01,0.96330000D+00,0.00000000D+00
     .,0.67123200D+02,0.149D+03,0.900D+01,0.96330000D+00,0.00000000D+00
     ./)
      pars(  48001:  48100)=(/
     . 0.51174700D+02,0.149D+03,0.100D+02,0.96330000D+00,0.00000000D+00
     .,0.91594130D+03,0.149D+03,0.110D+02,0.96330000D+00,0.00000000D+00
     .,0.66728440D+03,0.149D+03,0.120D+02,0.96330000D+00,0.00000000D+00
     .,0.60241430D+03,0.149D+03,0.130D+02,0.96330000D+00,0.00000000D+00
     .,0.46150930D+03,0.149D+03,0.140D+02,0.96330000D+00,0.00000000D+00
     .,0.35151570D+03,0.149D+03,0.150D+02,0.96330000D+00,0.00000000D+00
     .,0.28739220D+03,0.149D+03,0.160D+02,0.96330000D+00,0.00000000D+00
     .,0.23145540D+03,0.149D+03,0.170D+02,0.96330000D+00,0.00000000D+00
     .,0.18709160D+03,0.149D+03,0.180D+02,0.96330000D+00,0.00000000D+00
     .,0.15173687D+04,0.149D+03,0.190D+02,0.96330000D+00,0.00000000D+00
     .,0.11989287D+04,0.149D+03,0.200D+02,0.96330000D+00,0.00000000D+00
     .,0.98084640D+03,0.149D+03,0.210D+02,0.96330000D+00,0.00000000D+00
     .,0.93876410D+03,0.149D+03,0.220D+02,0.96330000D+00,0.00000000D+00
     .,0.85507440D+03,0.149D+03,0.230D+02,0.96330000D+00,0.00000000D+00
     .,0.67285260D+03,0.149D+03,0.240D+02,0.96330000D+00,0.00000000D+00
     .,0.73052210D+03,0.149D+03,0.250D+02,0.96330000D+00,0.00000000D+00
     .,0.57217290D+03,0.149D+03,0.260D+02,0.96330000D+00,0.00000000D+00
     .,0.59982550D+03,0.149D+03,0.270D+02,0.96330000D+00,0.00000000D+00
     .,0.62133050D+03,0.149D+03,0.280D+02,0.96330000D+00,0.00000000D+00
     .,0.47601650D+03,0.149D+03,0.290D+02,0.96330000D+00,0.00000000D+00
     ./)
      pars(  48101:  48200)=(/
     . 0.48059720D+03,0.149D+03,0.300D+02,0.96330000D+00,0.00000000D+00
     .,0.57226580D+03,0.149D+03,0.310D+02,0.96330000D+00,0.00000000D+00
     .,0.49379840D+03,0.149D+03,0.320D+02,0.96330000D+00,0.00000000D+00
     .,0.41301260D+03,0.149D+03,0.330D+02,0.96330000D+00,0.00000000D+00
     .,0.36615190D+03,0.149D+03,0.340D+02,0.96330000D+00,0.00000000D+00
     .,0.31652310D+03,0.149D+03,0.350D+02,0.96330000D+00,0.00000000D+00
     .,0.27222880D+03,0.149D+03,0.360D+02,0.96330000D+00,0.00000000D+00
     .,0.16942091D+04,0.149D+03,0.370D+02,0.96330000D+00,0.00000000D+00
     .,0.14306640D+04,0.149D+03,0.380D+02,0.96330000D+00,0.00000000D+00
     .,0.12316979D+04,0.149D+03,0.390D+02,0.96330000D+00,0.00000000D+00
     .,0.10950419D+04,0.149D+03,0.400D+02,0.96330000D+00,0.00000000D+00
     .,0.99140130D+03,0.149D+03,0.410D+02,0.96330000D+00,0.00000000D+00
     .,0.75589180D+03,0.149D+03,0.420D+02,0.96330000D+00,0.00000000D+00
     .,0.84735570D+03,0.149D+03,0.430D+02,0.96330000D+00,0.00000000D+00
     .,0.63674800D+03,0.149D+03,0.440D+02,0.96330000D+00,0.00000000D+00
     .,0.69613210D+03,0.149D+03,0.450D+02,0.96330000D+00,0.00000000D+00
     .,0.64264560D+03,0.149D+03,0.460D+02,0.96330000D+00,0.00000000D+00
     .,0.53720580D+03,0.149D+03,0.470D+02,0.96330000D+00,0.00000000D+00
     .,0.56303120D+03,0.149D+03,0.480D+02,0.96330000D+00,0.00000000D+00
     .,0.71679570D+03,0.149D+03,0.490D+02,0.96330000D+00,0.00000000D+00
     ./)
      pars(  48201:  48300)=(/
     . 0.65114030D+03,0.149D+03,0.500D+02,0.96330000D+00,0.00000000D+00
     .,0.57030150D+03,0.149D+03,0.510D+02,0.96330000D+00,0.00000000D+00
     .,0.52361310D+03,0.149D+03,0.520D+02,0.96330000D+00,0.00000000D+00
     .,0.46816800D+03,0.149D+03,0.530D+02,0.96330000D+00,0.00000000D+00
     .,0.41650530D+03,0.149D+03,0.540D+02,0.96330000D+00,0.00000000D+00
     .,0.20626273D+04,0.149D+03,0.550D+02,0.96330000D+00,0.00000000D+00
     .,0.18342595D+04,0.149D+03,0.560D+02,0.96330000D+00,0.00000000D+00
     .,0.15838792D+04,0.149D+03,0.570D+02,0.96330000D+00,0.00000000D+00
     .,0.67579110D+03,0.149D+03,0.580D+02,0.96330000D+00,0.27991000D+01
     .,0.16159882D+04,0.149D+03,0.590D+02,0.96330000D+00,0.00000000D+00
     .,0.15472492D+04,0.149D+03,0.600D+02,0.96330000D+00,0.00000000D+00
     .,0.15072236D+04,0.149D+03,0.610D+02,0.96330000D+00,0.00000000D+00
     .,0.14705492D+04,0.149D+03,0.620D+02,0.96330000D+00,0.00000000D+00
     .,0.14379844D+04,0.149D+03,0.630D+02,0.96330000D+00,0.00000000D+00
     .,0.11092250D+04,0.149D+03,0.640D+02,0.96330000D+00,0.00000000D+00
     .,0.12845648D+04,0.149D+03,0.650D+02,0.96330000D+00,0.00000000D+00
     .,0.12348582D+04,0.149D+03,0.660D+02,0.96330000D+00,0.00000000D+00
     .,0.12913513D+04,0.149D+03,0.670D+02,0.96330000D+00,0.00000000D+00
     .,0.12633503D+04,0.149D+03,0.680D+02,0.96330000D+00,0.00000000D+00
     .,0.12378116D+04,0.149D+03,0.690D+02,0.96330000D+00,0.00000000D+00
     ./)
      pars(  48301:  48400)=(/
     . 0.12243143D+04,0.149D+03,0.700D+02,0.96330000D+00,0.00000000D+00
     .,0.10181407D+04,0.149D+03,0.710D+02,0.96330000D+00,0.00000000D+00
     .,0.98356470D+03,0.149D+03,0.720D+02,0.96330000D+00,0.00000000D+00
     .,0.88792510D+03,0.149D+03,0.730D+02,0.96330000D+00,0.00000000D+00
     .,0.74340350D+03,0.149D+03,0.740D+02,0.96330000D+00,0.00000000D+00
     .,0.75315120D+03,0.149D+03,0.750D+02,0.96330000D+00,0.00000000D+00
     .,0.67623400D+03,0.149D+03,0.760D+02,0.96330000D+00,0.00000000D+00
     .,0.61465820D+03,0.149D+03,0.770D+02,0.96330000D+00,0.00000000D+00
     .,0.50654420D+03,0.149D+03,0.780D+02,0.96330000D+00,0.00000000D+00
     .,0.47177480D+03,0.149D+03,0.790D+02,0.96330000D+00,0.00000000D+00
     .,0.48363130D+03,0.149D+03,0.800D+02,0.96330000D+00,0.00000000D+00
     .,0.73258080D+03,0.149D+03,0.810D+02,0.96330000D+00,0.00000000D+00
     .,0.70644460D+03,0.149D+03,0.820D+02,0.96330000D+00,0.00000000D+00
     .,0.63913150D+03,0.149D+03,0.830D+02,0.96330000D+00,0.00000000D+00
     .,0.60410820D+03,0.149D+03,0.840D+02,0.96330000D+00,0.00000000D+00
     .,0.55158440D+03,0.149D+03,0.850D+02,0.96330000D+00,0.00000000D+00
     .,0.50067860D+03,0.149D+03,0.860D+02,0.96330000D+00,0.00000000D+00
     .,0.19210063D+04,0.149D+03,0.870D+02,0.96330000D+00,0.00000000D+00
     .,0.17975569D+04,0.149D+03,0.880D+02,0.96330000D+00,0.00000000D+00
     .,0.15637265D+04,0.149D+03,0.890D+02,0.96330000D+00,0.00000000D+00
     ./)
      pars(  48401:  48500)=(/
     . 0.13796361D+04,0.149D+03,0.900D+02,0.96330000D+00,0.00000000D+00
     .,0.13822903D+04,0.149D+03,0.910D+02,0.96330000D+00,0.00000000D+00
     .,0.13377839D+04,0.149D+03,0.920D+02,0.96330000D+00,0.00000000D+00
     .,0.13931101D+04,0.149D+03,0.930D+02,0.96330000D+00,0.00000000D+00
     .,0.13462697D+04,0.149D+03,0.940D+02,0.96330000D+00,0.00000000D+00
     .,0.69675400D+02,0.149D+03,0.101D+03,0.96330000D+00,0.00000000D+00
     .,0.23904190D+03,0.149D+03,0.103D+03,0.96330000D+00,0.98650000D+00
     .,0.30266780D+03,0.149D+03,0.104D+03,0.96330000D+00,0.98080000D+00
     .,0.22316290D+03,0.149D+03,0.105D+03,0.96330000D+00,0.97060000D+00
     .,0.16468120D+03,0.149D+03,0.106D+03,0.96330000D+00,0.98680000D+00
     .,0.11197070D+03,0.149D+03,0.107D+03,0.96330000D+00,0.99440000D+00
     .,0.80098600D+02,0.149D+03,0.108D+03,0.96330000D+00,0.99250000D+00
     .,0.53936400D+02,0.149D+03,0.109D+03,0.96330000D+00,0.99820000D+00
     .,0.35200740D+03,0.149D+03,0.111D+03,0.96330000D+00,0.96840000D+00
     .,0.54627620D+03,0.149D+03,0.112D+03,0.96330000D+00,0.96280000D+00
     .,0.54321880D+03,0.149D+03,0.113D+03,0.96330000D+00,0.96480000D+00
     .,0.42482930D+03,0.149D+03,0.114D+03,0.96330000D+00,0.95070000D+00
     .,0.34093960D+03,0.149D+03,0.115D+03,0.96330000D+00,0.99470000D+00
     .,0.28433500D+03,0.149D+03,0.116D+03,0.96330000D+00,0.99480000D+00
     .,0.22912290D+03,0.149D+03,0.117D+03,0.96330000D+00,0.99720000D+00
     ./)
      pars(  48501:  48600)=(/
     . 0.47801410D+03,0.149D+03,0.119D+03,0.96330000D+00,0.97670000D+00
     .,0.95798970D+03,0.149D+03,0.120D+03,0.96330000D+00,0.98310000D+00
     .,0.46767000D+03,0.149D+03,0.121D+03,0.96330000D+00,0.18627000D+01
     .,0.45116580D+03,0.149D+03,0.122D+03,0.96330000D+00,0.18299000D+01
     .,0.44234200D+03,0.149D+03,0.123D+03,0.96330000D+00,0.19138000D+01
     .,0.43947280D+03,0.149D+03,0.124D+03,0.96330000D+00,0.18269000D+01
     .,0.39910790D+03,0.149D+03,0.125D+03,0.96330000D+00,0.16406000D+01
     .,0.36795920D+03,0.149D+03,0.126D+03,0.96330000D+00,0.16483000D+01
     .,0.35103000D+03,0.149D+03,0.127D+03,0.96330000D+00,0.17149000D+01
     .,0.34356200D+03,0.149D+03,0.128D+03,0.96330000D+00,0.17937000D+01
     .,0.34289310D+03,0.149D+03,0.129D+03,0.96330000D+00,0.95760000D+00
     .,0.31591660D+03,0.149D+03,0.130D+03,0.96330000D+00,0.19419000D+01
     .,0.53451270D+03,0.149D+03,0.131D+03,0.96330000D+00,0.96010000D+00
     .,0.45962850D+03,0.149D+03,0.132D+03,0.96330000D+00,0.94340000D+00
     .,0.40554530D+03,0.149D+03,0.133D+03,0.96330000D+00,0.98890000D+00
     .,0.36636520D+03,0.149D+03,0.134D+03,0.96330000D+00,0.99010000D+00
     .,0.31893050D+03,0.149D+03,0.135D+03,0.96330000D+00,0.99740000D+00
     .,0.56760020D+03,0.149D+03,0.137D+03,0.96330000D+00,0.97380000D+00
     .,0.11685036D+04,0.149D+03,0.138D+03,0.96330000D+00,0.98010000D+00
     .,0.86288330D+03,0.149D+03,0.139D+03,0.96330000D+00,0.19153000D+01
     ./)
      pars(  48601:  48700)=(/
     . 0.61965370D+03,0.149D+03,0.140D+03,0.96330000D+00,0.19355000D+01
     .,0.62602540D+03,0.149D+03,0.141D+03,0.96330000D+00,0.19545000D+01
     .,0.58105490D+03,0.149D+03,0.142D+03,0.96330000D+00,0.19420000D+01
     .,0.66303800D+03,0.149D+03,0.143D+03,0.96330000D+00,0.16682000D+01
     .,0.50016280D+03,0.149D+03,0.144D+03,0.96330000D+00,0.18584000D+01
     .,0.46737710D+03,0.149D+03,0.145D+03,0.96330000D+00,0.19003000D+01
     .,0.43299730D+03,0.149D+03,0.146D+03,0.96330000D+00,0.18630000D+01
     .,0.41968680D+03,0.149D+03,0.147D+03,0.96330000D+00,0.96790000D+00
     .,0.41115760D+03,0.149D+03,0.148D+03,0.96330000D+00,0.19539000D+01
     .,0.67899200D+03,0.149D+03,0.149D+03,0.96330000D+00,0.96330000D+00
     .,0.39050000D+02,0.150D+03,0.100D+01,0.95140000D+00,0.91180000D+00
     .,0.25410500D+02,0.150D+03,0.200D+01,0.95140000D+00,0.00000000D+00
     .,0.64241900D+03,0.150D+03,0.300D+01,0.95140000D+00,0.00000000D+00
     .,0.36274340D+03,0.150D+03,0.400D+01,0.95140000D+00,0.00000000D+00
     .,0.24059340D+03,0.150D+03,0.500D+01,0.95140000D+00,0.00000000D+00
     .,0.16058760D+03,0.150D+03,0.600D+01,0.95140000D+00,0.00000000D+00
     .,0.11127680D+03,0.150D+03,0.700D+01,0.95140000D+00,0.00000000D+00
     .,0.83712000D+02,0.150D+03,0.800D+01,0.95140000D+00,0.00000000D+00
     .,0.63080600D+02,0.150D+03,0.900D+01,0.95140000D+00,0.00000000D+00
     .,0.48325800D+02,0.150D+03,0.100D+02,0.95140000D+00,0.00000000D+00
     ./)
      pars(  48701:  48800)=(/
     . 0.76709550D+03,0.150D+03,0.110D+02,0.95140000D+00,0.00000000D+00
     .,0.58045000D+03,0.150D+03,0.120D+02,0.95140000D+00,0.00000000D+00
     .,0.53107940D+03,0.150D+03,0.130D+02,0.95140000D+00,0.00000000D+00
     .,0.41414030D+03,0.150D+03,0.140D+02,0.95140000D+00,0.00000000D+00
     .,0.31993070D+03,0.150D+03,0.150D+02,0.95140000D+00,0.00000000D+00
     .,0.26382770D+03,0.150D+03,0.160D+02,0.95140000D+00,0.00000000D+00
     .,0.21417230D+03,0.150D+03,0.170D+02,0.95140000D+00,0.00000000D+00
     .,0.17426390D+03,0.150D+03,0.180D+02,0.95140000D+00,0.00000000D+00
     .,0.12595964D+04,0.150D+03,0.190D+02,0.95140000D+00,0.00000000D+00
     .,0.10266097D+04,0.150D+03,0.200D+02,0.95140000D+00,0.00000000D+00
     .,0.84554470D+03,0.150D+03,0.210D+02,0.95140000D+00,0.00000000D+00
     .,0.81389640D+03,0.150D+03,0.220D+02,0.95140000D+00,0.00000000D+00
     .,0.74392530D+03,0.150D+03,0.230D+02,0.95140000D+00,0.00000000D+00
     .,0.58540690D+03,0.150D+03,0.240D+02,0.95140000D+00,0.00000000D+00
     .,0.63875870D+03,0.150D+03,0.250D+02,0.95140000D+00,0.00000000D+00
     .,0.50063040D+03,0.150D+03,0.260D+02,0.95140000D+00,0.00000000D+00
     .,0.52899650D+03,0.150D+03,0.270D+02,0.95140000D+00,0.00000000D+00
     .,0.54604730D+03,0.150D+03,0.280D+02,0.95140000D+00,0.00000000D+00
     .,0.41817300D+03,0.150D+03,0.290D+02,0.95140000D+00,0.00000000D+00
     .,0.42710250D+03,0.150D+03,0.300D+02,0.95140000D+00,0.00000000D+00
     ./)
      pars(  48801:  48900)=(/
     . 0.50688380D+03,0.150D+03,0.310D+02,0.95140000D+00,0.00000000D+00
     .,0.44348920D+03,0.150D+03,0.320D+02,0.95140000D+00,0.00000000D+00
     .,0.37536940D+03,0.150D+03,0.330D+02,0.95140000D+00,0.00000000D+00
     .,0.33511820D+03,0.150D+03,0.340D+02,0.95140000D+00,0.00000000D+00
     .,0.29171790D+03,0.150D+03,0.350D+02,0.95140000D+00,0.00000000D+00
     .,0.25244080D+03,0.150D+03,0.360D+02,0.95140000D+00,0.00000000D+00
     .,0.14095386D+04,0.150D+03,0.370D+02,0.95140000D+00,0.00000000D+00
     .,0.12233073D+04,0.150D+03,0.380D+02,0.95140000D+00,0.00000000D+00
     .,0.10654342D+04,0.150D+03,0.390D+02,0.95140000D+00,0.00000000D+00
     .,0.95393220D+03,0.150D+03,0.400D+02,0.95140000D+00,0.00000000D+00
     .,0.86754750D+03,0.150D+03,0.410D+02,0.95140000D+00,0.00000000D+00
     .,0.66641970D+03,0.150D+03,0.420D+02,0.95140000D+00,0.00000000D+00
     .,0.74499930D+03,0.150D+03,0.430D+02,0.95140000D+00,0.00000000D+00
     .,0.56445890D+03,0.150D+03,0.440D+02,0.95140000D+00,0.00000000D+00
     .,0.61737910D+03,0.150D+03,0.450D+02,0.95140000D+00,0.00000000D+00
     .,0.57156020D+03,0.150D+03,0.460D+02,0.95140000D+00,0.00000000D+00
     .,0.47659660D+03,0.150D+03,0.470D+02,0.95140000D+00,0.00000000D+00
     .,0.50255130D+03,0.150D+03,0.480D+02,0.95140000D+00,0.00000000D+00
     .,0.63411950D+03,0.150D+03,0.490D+02,0.95140000D+00,0.00000000D+00
     .,0.58276930D+03,0.150D+03,0.500D+02,0.95140000D+00,0.00000000D+00
     ./)
      pars(  48901:  49000)=(/
     . 0.51594670D+03,0.150D+03,0.510D+02,0.95140000D+00,0.00000000D+00
     .,0.47668110D+03,0.150D+03,0.520D+02,0.95140000D+00,0.00000000D+00
     .,0.42899630D+03,0.150D+03,0.530D+02,0.95140000D+00,0.00000000D+00
     .,0.38394570D+03,0.150D+03,0.540D+02,0.95140000D+00,0.00000000D+00
     .,0.17161526D+04,0.150D+03,0.550D+02,0.95140000D+00,0.00000000D+00
     .,0.15618289D+04,0.150D+03,0.560D+02,0.95140000D+00,0.00000000D+00
     .,0.13644290D+04,0.150D+03,0.570D+02,0.95140000D+00,0.00000000D+00
     .,0.61000420D+03,0.150D+03,0.580D+02,0.95140000D+00,0.27991000D+01
     .,0.13811677D+04,0.150D+03,0.590D+02,0.95140000D+00,0.00000000D+00
     .,0.13251151D+04,0.150D+03,0.600D+02,0.95140000D+00,0.00000000D+00
     .,0.12915707D+04,0.150D+03,0.610D+02,0.95140000D+00,0.00000000D+00
     .,0.12607608D+04,0.150D+03,0.620D+02,0.95140000D+00,0.00000000D+00
     .,0.12334280D+04,0.150D+03,0.630D+02,0.95140000D+00,0.00000000D+00
     .,0.96322340D+03,0.150D+03,0.640D+02,0.95140000D+00,0.00000000D+00
     .,0.10940703D+04,0.150D+03,0.650D+02,0.95140000D+00,0.00000000D+00
     .,0.10539748D+04,0.150D+03,0.660D+02,0.95140000D+00,0.00000000D+00
     .,0.11110005D+04,0.150D+03,0.670D+02,0.95140000D+00,0.00000000D+00
     .,0.10872901D+04,0.150D+03,0.680D+02,0.95140000D+00,0.00000000D+00
     .,0.10658196D+04,0.150D+03,0.690D+02,0.95140000D+00,0.00000000D+00
     .,0.10536773D+04,0.150D+03,0.700D+02,0.95140000D+00,0.00000000D+00
     ./)
      pars(  49001:  49100)=(/
     . 0.88359210D+03,0.150D+03,0.710D+02,0.95140000D+00,0.00000000D+00
     .,0.86380750D+03,0.150D+03,0.720D+02,0.95140000D+00,0.00000000D+00
     .,0.78517710D+03,0.150D+03,0.730D+02,0.95140000D+00,0.00000000D+00
     .,0.66055390D+03,0.150D+03,0.740D+02,0.95140000D+00,0.00000000D+00
     .,0.67110130D+03,0.150D+03,0.750D+02,0.95140000D+00,0.00000000D+00
     .,0.60606460D+03,0.150D+03,0.760D+02,0.95140000D+00,0.00000000D+00
     .,0.55342010D+03,0.150D+03,0.770D+02,0.95140000D+00,0.00000000D+00
     .,0.45807960D+03,0.150D+03,0.780D+02,0.95140000D+00,0.00000000D+00
     .,0.42739480D+03,0.150D+03,0.790D+02,0.95140000D+00,0.00000000D+00
     .,0.43923750D+03,0.150D+03,0.800D+02,0.95140000D+00,0.00000000D+00
     .,0.64947520D+03,0.150D+03,0.810D+02,0.95140000D+00,0.00000000D+00
     .,0.63216060D+03,0.150D+03,0.820D+02,0.95140000D+00,0.00000000D+00
     .,0.57757500D+03,0.150D+03,0.830D+02,0.95140000D+00,0.00000000D+00
     .,0.54893280D+03,0.150D+03,0.840D+02,0.95140000D+00,0.00000000D+00
     .,0.50441400D+03,0.150D+03,0.850D+02,0.95140000D+00,0.00000000D+00
     .,0.46043530D+03,0.150D+03,0.860D+02,0.95140000D+00,0.00000000D+00
     .,0.16132050D+04,0.150D+03,0.870D+02,0.95140000D+00,0.00000000D+00
     .,0.15396549D+04,0.150D+03,0.880D+02,0.95140000D+00,0.00000000D+00
     .,0.13539743D+04,0.150D+03,0.890D+02,0.95140000D+00,0.00000000D+00
     .,0.12088936D+04,0.150D+03,0.900D+02,0.95140000D+00,0.00000000D+00
     ./)
      pars(  49101:  49200)=(/
     . 0.12037043D+04,0.150D+03,0.910D+02,0.95140000D+00,0.00000000D+00
     .,0.11652987D+04,0.150D+03,0.920D+02,0.95140000D+00,0.00000000D+00
     .,0.12045686D+04,0.150D+03,0.930D+02,0.95140000D+00,0.00000000D+00
     .,0.11657062D+04,0.150D+03,0.940D+02,0.95140000D+00,0.00000000D+00
     .,0.63518900D+02,0.150D+03,0.101D+03,0.95140000D+00,0.00000000D+00
     .,0.21038150D+03,0.150D+03,0.103D+03,0.95140000D+00,0.98650000D+00
     .,0.26750360D+03,0.150D+03,0.104D+03,0.95140000D+00,0.98080000D+00
     .,0.20167580D+03,0.150D+03,0.105D+03,0.95140000D+00,0.97060000D+00
     .,0.15060400D+03,0.150D+03,0.106D+03,0.95140000D+00,0.98680000D+00
     .,0.10370700D+03,0.150D+03,0.107D+03,0.95140000D+00,0.99440000D+00
     .,0.74928900D+02,0.150D+03,0.108D+03,0.95140000D+00,0.99250000D+00
     .,0.51049200D+02,0.150D+03,0.109D+03,0.95140000D+00,0.99820000D+00
     .,0.30824700D+03,0.150D+03,0.111D+03,0.95140000D+00,0.96840000D+00
     .,0.47728480D+03,0.150D+03,0.112D+03,0.95140000D+00,0.96280000D+00
     .,0.48037350D+03,0.150D+03,0.113D+03,0.95140000D+00,0.96480000D+00
     .,0.38212550D+03,0.150D+03,0.114D+03,0.95140000D+00,0.95070000D+00
     .,0.31042380D+03,0.150D+03,0.115D+03,0.95140000D+00,0.99470000D+00
     .,0.26094970D+03,0.150D+03,0.116D+03,0.95140000D+00,0.99480000D+00
     .,0.21197090D+03,0.150D+03,0.117D+03,0.95140000D+00,0.99720000D+00
     .,0.42200550D+03,0.150D+03,0.119D+03,0.95140000D+00,0.97670000D+00
     ./)
      pars(  49201:  49300)=(/
     . 0.81999260D+03,0.150D+03,0.120D+03,0.95140000D+00,0.98310000D+00
     .,0.41914320D+03,0.150D+03,0.121D+03,0.95140000D+00,0.18627000D+01
     .,0.40443780D+03,0.150D+03,0.122D+03,0.95140000D+00,0.18299000D+01
     .,0.39640790D+03,0.150D+03,0.123D+03,0.95140000D+00,0.19138000D+01
     .,0.39312150D+03,0.150D+03,0.124D+03,0.95140000D+00,0.18269000D+01
     .,0.36011140D+03,0.150D+03,0.125D+03,0.95140000D+00,0.16406000D+01
     .,0.33275950D+03,0.150D+03,0.126D+03,0.95140000D+00,0.16483000D+01
     .,0.31739780D+03,0.150D+03,0.127D+03,0.95140000D+00,0.17149000D+01
     .,0.31042370D+03,0.150D+03,0.128D+03,0.95140000D+00,0.17937000D+01
     .,0.30781940D+03,0.150D+03,0.129D+03,0.95140000D+00,0.95760000D+00
     .,0.28694640D+03,0.150D+03,0.130D+03,0.95140000D+00,0.19419000D+01
     .,0.47510140D+03,0.150D+03,0.131D+03,0.95140000D+00,0.96010000D+00
     .,0.41406130D+03,0.150D+03,0.132D+03,0.95140000D+00,0.94340000D+00
     .,0.36881960D+03,0.150D+03,0.133D+03,0.95140000D+00,0.98890000D+00
     .,0.33526670D+03,0.150D+03,0.134D+03,0.95140000D+00,0.99010000D+00
     .,0.29382390D+03,0.150D+03,0.135D+03,0.95140000D+00,0.99740000D+00
     .,0.50249860D+03,0.150D+03,0.137D+03,0.95140000D+00,0.97380000D+00
     .,0.99819190D+03,0.150D+03,0.138D+03,0.95140000D+00,0.98010000D+00
     .,0.75411460D+03,0.150D+03,0.139D+03,0.95140000D+00,0.19153000D+01
     .,0.55441070D+03,0.150D+03,0.140D+03,0.95140000D+00,0.19355000D+01
     ./)
      pars(  49301:  49400)=(/
     . 0.55998500D+03,0.150D+03,0.141D+03,0.95140000D+00,0.19545000D+01
     .,0.52115630D+03,0.150D+03,0.142D+03,0.95140000D+00,0.19420000D+01
     .,0.58793550D+03,0.150D+03,0.143D+03,0.95140000D+00,0.16682000D+01
     .,0.45216820D+03,0.150D+03,0.144D+03,0.95140000D+00,0.18584000D+01
     .,0.42274380D+03,0.150D+03,0.145D+03,0.95140000D+00,0.19003000D+01
     .,0.39214710D+03,0.150D+03,0.146D+03,0.95140000D+00,0.18630000D+01
     .,0.37963350D+03,0.150D+03,0.147D+03,0.95140000D+00,0.96790000D+00
     .,0.37434300D+03,0.150D+03,0.148D+03,0.95140000D+00,0.19539000D+01
     .,0.60309650D+03,0.150D+03,0.149D+03,0.95140000D+00,0.96330000D+00
     .,0.54204390D+03,0.150D+03,0.150D+03,0.95140000D+00,0.95140000D+00
     .,0.37118600D+02,0.151D+03,0.100D+01,0.97490000D+00,0.91180000D+00
     .,0.24506800D+02,0.151D+03,0.200D+01,0.97490000D+00,0.00000000D+00
     .,0.56950200D+03,0.151D+03,0.300D+01,0.97490000D+00,0.00000000D+00
     .,0.33206650D+03,0.151D+03,0.400D+01,0.97490000D+00,0.00000000D+00
     .,0.22424380D+03,0.151D+03,0.500D+01,0.97490000D+00,0.00000000D+00
     .,0.15163340D+03,0.151D+03,0.600D+01,0.97490000D+00,0.00000000D+00
     .,0.10605380D+03,0.151D+03,0.700D+01,0.97490000D+00,0.00000000D+00
     .,0.80289900D+02,0.151D+03,0.800D+01,0.97490000D+00,0.00000000D+00
     .,0.60817400D+02,0.151D+03,0.900D+01,0.97490000D+00,0.00000000D+00
     .,0.46777500D+02,0.151D+03,0.100D+02,0.97490000D+00,0.00000000D+00
     ./)
      pars(  49401:  49500)=(/
     . 0.68136500D+03,0.151D+03,0.110D+02,0.97490000D+00,0.00000000D+00
     .,0.52835450D+03,0.151D+03,0.120D+02,0.97490000D+00,0.00000000D+00
     .,0.48798470D+03,0.151D+03,0.130D+02,0.97490000D+00,0.00000000D+00
     .,0.38531020D+03,0.151D+03,0.140D+02,0.97490000D+00,0.00000000D+00
     .,0.30075580D+03,0.151D+03,0.150D+02,0.97490000D+00,0.00000000D+00
     .,0.24964690D+03,0.151D+03,0.160D+02,0.97490000D+00,0.00000000D+00
     .,0.20391030D+03,0.151D+03,0.170D+02,0.97490000D+00,0.00000000D+00
     .,0.16677960D+03,0.151D+03,0.180D+02,0.97490000D+00,0.00000000D+00
     .,0.11139369D+04,0.151D+03,0.190D+02,0.97490000D+00,0.00000000D+00
     .,0.92532660D+03,0.151D+03,0.200D+02,0.97490000D+00,0.00000000D+00
     .,0.76543030D+03,0.151D+03,0.210D+02,0.97490000D+00,0.00000000D+00
     .,0.73977510D+03,0.151D+03,0.220D+02,0.97490000D+00,0.00000000D+00
     .,0.67780140D+03,0.151D+03,0.230D+02,0.97490000D+00,0.00000000D+00
     .,0.53372530D+03,0.151D+03,0.240D+02,0.97490000D+00,0.00000000D+00
     .,0.58400790D+03,0.151D+03,0.250D+02,0.97490000D+00,0.00000000D+00
     .,0.45821350D+03,0.151D+03,0.260D+02,0.97490000D+00,0.00000000D+00
     .,0.48646450D+03,0.151D+03,0.270D+02,0.97490000D+00,0.00000000D+00
     .,0.50090110D+03,0.151D+03,0.280D+02,0.97490000D+00,0.00000000D+00
     .,0.38380100D+03,0.151D+03,0.290D+02,0.97490000D+00,0.00000000D+00
     .,0.39484340D+03,0.151D+03,0.300D+02,0.97490000D+00,0.00000000D+00
     ./)
      pars(  49501:  49600)=(/
     . 0.46751190D+03,0.151D+03,0.310D+02,0.97490000D+00,0.00000000D+00
     .,0.41296610D+03,0.151D+03,0.320D+02,0.97490000D+00,0.00000000D+00
     .,0.35254270D+03,0.151D+03,0.330D+02,0.97490000D+00,0.00000000D+00
     .,0.31639840D+03,0.151D+03,0.340D+02,0.97490000D+00,0.00000000D+00
     .,0.27688560D+03,0.151D+03,0.350D+02,0.97490000D+00,0.00000000D+00
     .,0.24075150D+03,0.151D+03,0.360D+02,0.97490000D+00,0.00000000D+00
     .,0.12488109D+04,0.151D+03,0.370D+02,0.97490000D+00,0.00000000D+00
     .,0.11019644D+04,0.151D+03,0.380D+02,0.97490000D+00,0.00000000D+00
     .,0.96716720D+03,0.151D+03,0.390D+02,0.97490000D+00,0.00000000D+00
     .,0.87013340D+03,0.151D+03,0.400D+02,0.97490000D+00,0.00000000D+00
     .,0.79387710D+03,0.151D+03,0.410D+02,0.97490000D+00,0.00000000D+00
     .,0.61324700D+03,0.151D+03,0.420D+02,0.97490000D+00,0.00000000D+00
     .,0.68412920D+03,0.151D+03,0.430D+02,0.97490000D+00,0.00000000D+00
     .,0.52150710D+03,0.151D+03,0.440D+02,0.97490000D+00,0.00000000D+00
     .,0.57027900D+03,0.151D+03,0.450D+02,0.97490000D+00,0.00000000D+00
     .,0.52899540D+03,0.151D+03,0.460D+02,0.97490000D+00,0.00000000D+00
     .,0.44070410D+03,0.151D+03,0.470D+02,0.97490000D+00,0.00000000D+00
     .,0.46629410D+03,0.151D+03,0.480D+02,0.97490000D+00,0.00000000D+00
     .,0.58471100D+03,0.151D+03,0.490D+02,0.97490000D+00,0.00000000D+00
     .,0.54152090D+03,0.151D+03,0.500D+02,0.97490000D+00,0.00000000D+00
     ./)
      pars(  49601:  49700)=(/
     . 0.48305960D+03,0.151D+03,0.510D+02,0.97490000D+00,0.00000000D+00
     .,0.44833330D+03,0.151D+03,0.520D+02,0.97490000D+00,0.00000000D+00
     .,0.40544000D+03,0.151D+03,0.530D+02,0.97490000D+00,0.00000000D+00
     .,0.36450640D+03,0.151D+03,0.540D+02,0.97490000D+00,0.00000000D+00
     .,0.15212386D+04,0.151D+03,0.550D+02,0.97490000D+00,0.00000000D+00
     .,0.14035743D+04,0.151D+03,0.560D+02,0.97490000D+00,0.00000000D+00
     .,0.12355723D+04,0.151D+03,0.570D+02,0.97490000D+00,0.00000000D+00
     .,0.57038460D+03,0.151D+03,0.580D+02,0.97490000D+00,0.27991000D+01
     .,0.12443690D+04,0.151D+03,0.590D+02,0.97490000D+00,0.00000000D+00
     .,0.11953394D+04,0.151D+03,0.600D+02,0.97490000D+00,0.00000000D+00
     .,0.11654811D+04,0.151D+03,0.610D+02,0.97490000D+00,0.00000000D+00
     .,0.11380116D+04,0.151D+03,0.620D+02,0.97490000D+00,0.00000000D+00
     .,0.11136574D+04,0.151D+03,0.630D+02,0.97490000D+00,0.00000000D+00
     .,0.87723370D+03,0.151D+03,0.640D+02,0.97490000D+00,0.00000000D+00
     .,0.98436690D+03,0.151D+03,0.650D+02,0.97490000D+00,0.00000000D+00
     .,0.94971600D+03,0.151D+03,0.660D+02,0.97490000D+00,0.00000000D+00
     .,0.10050059D+04,0.151D+03,0.670D+02,0.97490000D+00,0.00000000D+00
     .,0.98375110D+03,0.151D+03,0.680D+02,0.97490000D+00,0.00000000D+00
     .,0.96460520D+03,0.151D+03,0.690D+02,0.97490000D+00,0.00000000D+00
     .,0.95326960D+03,0.151D+03,0.700D+02,0.97490000D+00,0.00000000D+00
     ./)
      pars(  49701:  49800)=(/
     . 0.80412310D+03,0.151D+03,0.710D+02,0.97490000D+00,0.00000000D+00
     .,0.79227200D+03,0.151D+03,0.720D+02,0.97490000D+00,0.00000000D+00
     .,0.72359370D+03,0.151D+03,0.730D+02,0.97490000D+00,0.00000000D+00
     .,0.61110290D+03,0.151D+03,0.740D+02,0.97490000D+00,0.00000000D+00
     .,0.62195650D+03,0.151D+03,0.750D+02,0.97490000D+00,0.00000000D+00
     .,0.56397010D+03,0.151D+03,0.760D+02,0.97490000D+00,0.00000000D+00
     .,0.51667170D+03,0.151D+03,0.770D+02,0.97490000D+00,0.00000000D+00
     .,0.42917250D+03,0.151D+03,0.780D+02,0.97490000D+00,0.00000000D+00
     .,0.40097810D+03,0.151D+03,0.790D+02,0.97490000D+00,0.00000000D+00
     .,0.41268870D+03,0.151D+03,0.800D+02,0.97490000D+00,0.00000000D+00
     .,0.60016760D+03,0.151D+03,0.810D+02,0.97490000D+00,0.00000000D+00
     .,0.58760720D+03,0.151D+03,0.820D+02,0.97490000D+00,0.00000000D+00
     .,0.54048320D+03,0.151D+03,0.830D+02,0.97490000D+00,0.00000000D+00
     .,0.51568470D+03,0.151D+03,0.840D+02,0.97490000D+00,0.00000000D+00
     .,0.47605910D+03,0.151D+03,0.850D+02,0.97490000D+00,0.00000000D+00
     .,0.43636860D+03,0.151D+03,0.860D+02,0.97490000D+00,0.00000000D+00
     .,0.14387730D+04,0.151D+03,0.870D+02,0.97490000D+00,0.00000000D+00
     .,0.13892118D+04,0.151D+03,0.880D+02,0.97490000D+00,0.00000000D+00
     .,0.12302614D+04,0.151D+03,0.890D+02,0.97490000D+00,0.00000000D+00
     .,0.11073931D+04,0.151D+03,0.900D+02,0.97490000D+00,0.00000000D+00
     ./)
      pars(  49801:  49900)=(/
     . 0.10983173D+04,0.151D+03,0.910D+02,0.97490000D+00,0.00000000D+00
     .,0.10634833D+04,0.151D+03,0.920D+02,0.97490000D+00,0.00000000D+00
     .,0.10937836D+04,0.151D+03,0.930D+02,0.97490000D+00,0.00000000D+00
     .,0.10594592D+04,0.151D+03,0.940D+02,0.97490000D+00,0.00000000D+00
     .,0.59779600D+02,0.151D+03,0.101D+03,0.97490000D+00,0.00000000D+00
     .,0.19308010D+03,0.151D+03,0.103D+03,0.97490000D+00,0.98650000D+00
     .,0.24635800D+03,0.151D+03,0.104D+03,0.97490000D+00,0.98080000D+00
     .,0.18865770D+03,0.151D+03,0.105D+03,0.97490000D+00,0.97060000D+00
     .,0.14217370D+03,0.151D+03,0.106D+03,0.97490000D+00,0.98680000D+00
     .,0.98858300D+02,0.151D+03,0.107D+03,0.97490000D+00,0.99440000D+00
     .,0.71981600D+02,0.151D+03,0.108D+03,0.97490000D+00,0.99250000D+00
     .,0.49497000D+02,0.151D+03,0.109D+03,0.97490000D+00,0.99820000D+00
     .,0.28199230D+03,0.151D+03,0.111D+03,0.97490000D+00,0.96840000D+00
     .,0.43599230D+03,0.151D+03,0.112D+03,0.97490000D+00,0.96280000D+00
     .,0.44239460D+03,0.151D+03,0.113D+03,0.97490000D+00,0.96480000D+00
     .,0.35616810D+03,0.151D+03,0.114D+03,0.97490000D+00,0.95070000D+00
     .,0.29192250D+03,0.151D+03,0.115D+03,0.97490000D+00,0.99470000D+00
     .,0.24688440D+03,0.151D+03,0.116D+03,0.97490000D+00,0.99480000D+00
     .,0.20178980D+03,0.151D+03,0.117D+03,0.97490000D+00,0.99720000D+00
     .,0.38878440D+03,0.151D+03,0.119D+03,0.97490000D+00,0.97670000D+00
     ./)
      pars(  49901:  50000)=(/
     . 0.73983800D+03,0.151D+03,0.120D+03,0.97490000D+00,0.98310000D+00
     .,0.38986450D+03,0.151D+03,0.121D+03,0.97490000D+00,0.18627000D+01
     .,0.37633360D+03,0.151D+03,0.122D+03,0.97490000D+00,0.18299000D+01
     .,0.36879080D+03,0.151D+03,0.123D+03,0.97490000D+00,0.19138000D+01
     .,0.36529400D+03,0.151D+03,0.124D+03,0.97490000D+00,0.18269000D+01
     .,0.33651780D+03,0.151D+03,0.125D+03,0.97490000D+00,0.16406000D+01
     .,0.31148590D+03,0.151D+03,0.126D+03,0.97490000D+00,0.16483000D+01
     .,0.29711100D+03,0.151D+03,0.127D+03,0.97490000D+00,0.17149000D+01
     .,0.29044330D+03,0.151D+03,0.128D+03,0.97490000D+00,0.17937000D+01
     .,0.28674360D+03,0.151D+03,0.129D+03,0.97490000D+00,0.95760000D+00
     .,0.26942080D+03,0.151D+03,0.130D+03,0.97490000D+00,0.19419000D+01
     .,0.43926920D+03,0.151D+03,0.131D+03,0.97490000D+00,0.96010000D+00
     .,0.38642600D+03,0.151D+03,0.132D+03,0.97490000D+00,0.94340000D+00
     .,0.34656560D+03,0.151D+03,0.133D+03,0.97490000D+00,0.98890000D+00
     .,0.31651250D+03,0.151D+03,0.134D+03,0.97490000D+00,0.99010000D+00
     .,0.27880760D+03,0.151D+03,0.135D+03,0.97490000D+00,0.99740000D+00
     .,0.46396320D+03,0.151D+03,0.137D+03,0.97490000D+00,0.97380000D+00
     .,0.89977960D+03,0.151D+03,0.138D+03,0.97490000D+00,0.98010000D+00
     .,0.69024350D+03,0.151D+03,0.139D+03,0.97490000D+00,0.19153000D+01
     .,0.51535410D+03,0.151D+03,0.140D+03,0.97490000D+00,0.19355000D+01
     ./)
      pars(  50001:  50100)=(/
     . 0.52041820D+03,0.151D+03,0.141D+03,0.97490000D+00,0.19545000D+01
     .,0.48533740D+03,0.151D+03,0.142D+03,0.97490000D+00,0.19420000D+01
     .,0.54348220D+03,0.151D+03,0.143D+03,0.97490000D+00,0.16682000D+01
     .,0.42337510D+03,0.151D+03,0.144D+03,0.97490000D+00,0.18584000D+01
     .,0.39603590D+03,0.151D+03,0.145D+03,0.97490000D+00,0.19003000D+01
     .,0.36774930D+03,0.151D+03,0.146D+03,0.97490000D+00,0.18630000D+01
     .,0.35570800D+03,0.151D+03,0.147D+03,0.97490000D+00,0.96790000D+00
     .,0.35221320D+03,0.151D+03,0.148D+03,0.97490000D+00,0.19539000D+01
     .,0.55762720D+03,0.151D+03,0.149D+03,0.97490000D+00,0.96330000D+00
     .,0.50519000D+03,0.151D+03,0.150D+03,0.97490000D+00,0.95140000D+00
     .,0.47347030D+03,0.151D+03,0.151D+03,0.97490000D+00,0.97490000D+00
     .,0.35486100D+02,0.152D+03,0.100D+01,0.98110000D+00,0.91180000D+00
     .,0.23710000D+02,0.152D+03,0.200D+01,0.98110000D+00,0.00000000D+00
     .,0.51945180D+03,0.152D+03,0.300D+01,0.98110000D+00,0.00000000D+00
     .,0.30894260D+03,0.152D+03,0.400D+01,0.98110000D+00,0.00000000D+00
     .,0.21127990D+03,0.152D+03,0.500D+01,0.98110000D+00,0.00000000D+00
     .,0.14426000D+03,0.152D+03,0.600D+01,0.98110000D+00,0.00000000D+00
     .,0.10163150D+03,0.152D+03,0.700D+01,0.98110000D+00,0.00000000D+00
     .,0.77336500D+02,0.152D+03,0.800D+01,0.98110000D+00,0.00000000D+00
     .,0.58833200D+02,0.152D+03,0.900D+01,0.98110000D+00,0.00000000D+00
     ./)
      pars(  50101:  50200)=(/
     . 0.45404400D+02,0.152D+03,0.100D+02,0.98110000D+00,0.00000000D+00
     .,0.62230990D+03,0.152D+03,0.110D+02,0.98110000D+00,0.00000000D+00
     .,0.48978200D+03,0.152D+03,0.120D+02,0.98110000D+00,0.00000000D+00
     .,0.45524770D+03,0.152D+03,0.130D+02,0.98110000D+00,0.00000000D+00
     .,0.36255610D+03,0.152D+03,0.140D+02,0.98110000D+00,0.00000000D+00
     .,0.28512390D+03,0.152D+03,0.150D+02,0.98110000D+00,0.00000000D+00
     .,0.23785710D+03,0.152D+03,0.160D+02,0.98110000D+00,0.00000000D+00
     .,0.19521310D+03,0.152D+03,0.170D+02,0.98110000D+00,0.00000000D+00
     .,0.16033310D+03,0.152D+03,0.180D+02,0.98110000D+00,0.00000000D+00
     .,0.10157738D+04,0.152D+03,0.190D+02,0.98110000D+00,0.00000000D+00
     .,0.85290360D+03,0.152D+03,0.200D+02,0.98110000D+00,0.00000000D+00
     .,0.70737690D+03,0.152D+03,0.210D+02,0.98110000D+00,0.00000000D+00
     .,0.68556870D+03,0.152D+03,0.220D+02,0.98110000D+00,0.00000000D+00
     .,0.62913900D+03,0.152D+03,0.230D+02,0.98110000D+00,0.00000000D+00
     .,0.49588010D+03,0.152D+03,0.240D+02,0.98110000D+00,0.00000000D+00
     .,0.54334380D+03,0.152D+03,0.250D+02,0.98110000D+00,0.00000000D+00
     .,0.42683610D+03,0.152D+03,0.260D+02,0.98110000D+00,0.00000000D+00
     .,0.45431700D+03,0.152D+03,0.270D+02,0.98110000D+00,0.00000000D+00
     .,0.46700090D+03,0.152D+03,0.280D+02,0.98110000D+00,0.00000000D+00
     .,0.35818700D+03,0.152D+03,0.290D+02,0.98110000D+00,0.00000000D+00
     ./)
      pars(  50201:  50300)=(/
     . 0.37006740D+03,0.152D+03,0.300D+02,0.98110000D+00,0.00000000D+00
     .,0.43744170D+03,0.152D+03,0.310D+02,0.98110000D+00,0.00000000D+00
     .,0.38889460D+03,0.152D+03,0.320D+02,0.98110000D+00,0.00000000D+00
     .,0.33402630D+03,0.152D+03,0.330D+02,0.98110000D+00,0.00000000D+00
     .,0.30096170D+03,0.152D+03,0.340D+02,0.98110000D+00,0.00000000D+00
     .,0.26444460D+03,0.152D+03,0.350D+02,0.98110000D+00,0.00000000D+00
     .,0.23079470D+03,0.152D+03,0.360D+02,0.98110000D+00,0.00000000D+00
     .,0.11403397D+04,0.152D+03,0.370D+02,0.98110000D+00,0.00000000D+00
     .,0.10156122D+04,0.152D+03,0.380D+02,0.98110000D+00,0.00000000D+00
     .,0.89573040D+03,0.152D+03,0.390D+02,0.98110000D+00,0.00000000D+00
     .,0.80842520D+03,0.152D+03,0.400D+02,0.98110000D+00,0.00000000D+00
     .,0.73920970D+03,0.152D+03,0.410D+02,0.98110000D+00,0.00000000D+00
     .,0.57337000D+03,0.152D+03,0.420D+02,0.98110000D+00,0.00000000D+00
     .,0.63864620D+03,0.152D+03,0.430D+02,0.98110000D+00,0.00000000D+00
     .,0.48901000D+03,0.152D+03,0.440D+02,0.98110000D+00,0.00000000D+00
     .,0.53445110D+03,0.152D+03,0.450D+02,0.98110000D+00,0.00000000D+00
     .,0.49643130D+03,0.152D+03,0.460D+02,0.98110000D+00,0.00000000D+00
     .,0.41359130D+03,0.152D+03,0.470D+02,0.98110000D+00,0.00000000D+00
     .,0.43835340D+03,0.152D+03,0.480D+02,0.98110000D+00,0.00000000D+00
     .,0.54729130D+03,0.152D+03,0.490D+02,0.98110000D+00,0.00000000D+00
     ./)
      pars(  50301:  50400)=(/
     . 0.50938490D+03,0.152D+03,0.500D+02,0.98110000D+00,0.00000000D+00
     .,0.45675890D+03,0.152D+03,0.510D+02,0.98110000D+00,0.00000000D+00
     .,0.42531520D+03,0.152D+03,0.520D+02,0.98110000D+00,0.00000000D+00
     .,0.38599880D+03,0.152D+03,0.530D+02,0.98110000D+00,0.00000000D+00
     .,0.34821640D+03,0.152D+03,0.540D+02,0.98110000D+00,0.00000000D+00
     .,0.13899562D+04,0.152D+03,0.550D+02,0.98110000D+00,0.00000000D+00
     .,0.12920505D+04,0.152D+03,0.560D+02,0.98110000D+00,0.00000000D+00
     .,0.11428009D+04,0.152D+03,0.570D+02,0.98110000D+00,0.00000000D+00
     .,0.53897800D+03,0.152D+03,0.580D+02,0.98110000D+00,0.27991000D+01
     .,0.11473718D+04,0.152D+03,0.590D+02,0.98110000D+00,0.00000000D+00
     .,0.11029411D+04,0.152D+03,0.600D+02,0.98110000D+00,0.00000000D+00
     .,0.10755989D+04,0.152D+03,0.610D+02,0.98110000D+00,0.00000000D+00
     .,0.10504168D+04,0.152D+03,0.620D+02,0.98110000D+00,0.00000000D+00
     .,0.10280990D+04,0.152D+03,0.630D+02,0.98110000D+00,0.00000000D+00
     .,0.81457180D+03,0.152D+03,0.640D+02,0.98110000D+00,0.00000000D+00
     .,0.90749290D+03,0.152D+03,0.650D+02,0.98110000D+00,0.00000000D+00
     .,0.87640540D+03,0.152D+03,0.660D+02,0.98110000D+00,0.00000000D+00
     .,0.92881650D+03,0.152D+03,0.670D+02,0.98110000D+00,0.00000000D+00
     .,0.90926160D+03,0.152D+03,0.680D+02,0.98110000D+00,0.00000000D+00
     .,0.89171060D+03,0.152D+03,0.690D+02,0.98110000D+00,0.00000000D+00
     ./)
      pars(  50401:  50500)=(/
     . 0.88100370D+03,0.152D+03,0.700D+02,0.98110000D+00,0.00000000D+00
     .,0.74613080D+03,0.152D+03,0.710D+02,0.98110000D+00,0.00000000D+00
     .,0.73873540D+03,0.152D+03,0.720D+02,0.98110000D+00,0.00000000D+00
     .,0.67688650D+03,0.152D+03,0.730D+02,0.98110000D+00,0.00000000D+00
     .,0.57338870D+03,0.152D+03,0.740D+02,0.98110000D+00,0.00000000D+00
     .,0.58418960D+03,0.152D+03,0.750D+02,0.98110000D+00,0.00000000D+00
     .,0.53122090D+03,0.152D+03,0.760D+02,0.98110000D+00,0.00000000D+00
     .,0.48779780D+03,0.152D+03,0.770D+02,0.98110000D+00,0.00000000D+00
     .,0.40633480D+03,0.152D+03,0.780D+02,0.98110000D+00,0.00000000D+00
     .,0.38005290D+03,0.152D+03,0.790D+02,0.98110000D+00,0.00000000D+00
     .,0.39145330D+03,0.152D+03,0.800D+02,0.98110000D+00,0.00000000D+00
     .,0.56288150D+03,0.152D+03,0.810D+02,0.98110000D+00,0.00000000D+00
     .,0.55306450D+03,0.152D+03,0.820D+02,0.98110000D+00,0.00000000D+00
     .,0.51099880D+03,0.152D+03,0.830D+02,0.98110000D+00,0.00000000D+00
     .,0.48888150D+03,0.152D+03,0.840D+02,0.98110000D+00,0.00000000D+00
     .,0.45282060D+03,0.152D+03,0.850D+02,0.98110000D+00,0.00000000D+00
     .,0.41635580D+03,0.152D+03,0.860D+02,0.98110000D+00,0.00000000D+00
     .,0.13196282D+04,0.152D+03,0.870D+02,0.98110000D+00,0.00000000D+00
     .,0.12821564D+04,0.152D+03,0.880D+02,0.98110000D+00,0.00000000D+00
     .,0.11403381D+04,0.152D+03,0.890D+02,0.98110000D+00,0.00000000D+00
     ./)
      pars(  50501:  50600)=(/
     . 0.10319646D+04,0.152D+03,0.900D+02,0.98110000D+00,0.00000000D+00
     .,0.10211167D+04,0.152D+03,0.910D+02,0.98110000D+00,0.00000000D+00
     .,0.98886570D+03,0.152D+03,0.920D+02,0.98110000D+00,0.00000000D+00
     .,0.10136901D+04,0.152D+03,0.930D+02,0.98110000D+00,0.00000000D+00
     .,0.98242950D+03,0.152D+03,0.940D+02,0.98110000D+00,0.00000000D+00
     .,0.56715600D+02,0.152D+03,0.101D+03,0.98110000D+00,0.00000000D+00
     .,0.17998120D+03,0.152D+03,0.103D+03,0.98110000D+00,0.98650000D+00
     .,0.23026410D+03,0.152D+03,0.104D+03,0.98110000D+00,0.98080000D+00
     .,0.17825090D+03,0.152D+03,0.105D+03,0.98110000D+00,0.97060000D+00
     .,0.13527200D+03,0.152D+03,0.106D+03,0.98110000D+00,0.98680000D+00
     .,0.94770500D+02,0.152D+03,0.107D+03,0.98110000D+00,0.99440000D+00
     .,0.69431700D+02,0.152D+03,0.108D+03,0.98110000D+00,0.99250000D+00
     .,0.48103700D+02,0.152D+03,0.109D+03,0.98110000D+00,0.99820000D+00
     .,0.26236970D+03,0.152D+03,0.111D+03,0.98110000D+00,0.96840000D+00
     .,0.40525290D+03,0.152D+03,0.112D+03,0.98110000D+00,0.96280000D+00
     .,0.41338560D+03,0.152D+03,0.113D+03,0.98110000D+00,0.96480000D+00
     .,0.33559140D+03,0.152D+03,0.114D+03,0.98110000D+00,0.95070000D+00
     .,0.27683720D+03,0.152D+03,0.115D+03,0.98110000D+00,0.99470000D+00
     .,0.23520590D+03,0.152D+03,0.116D+03,0.98110000D+00,0.99480000D+00
     .,0.19317040D+03,0.152D+03,0.117D+03,0.98110000D+00,0.99720000D+00
     ./)
      pars(  50601:  50700)=(/
     . 0.36384820D+03,0.152D+03,0.119D+03,0.98110000D+00,0.97670000D+00
     .,0.68296210D+03,0.152D+03,0.120D+03,0.98110000D+00,0.98310000D+00
     .,0.36699000D+03,0.152D+03,0.121D+03,0.98110000D+00,0.18627000D+01
     .,0.35440290D+03,0.152D+03,0.122D+03,0.98110000D+00,0.18299000D+01
     .,0.34726480D+03,0.152D+03,0.123D+03,0.98110000D+00,0.19138000D+01
     .,0.34370870D+03,0.152D+03,0.124D+03,0.98110000D+00,0.18269000D+01
     .,0.31777500D+03,0.152D+03,0.125D+03,0.98110000D+00,0.16406000D+01
     .,0.29450790D+03,0.152D+03,0.126D+03,0.98110000D+00,0.16483000D+01
     .,0.28094820D+03,0.152D+03,0.127D+03,0.98110000D+00,0.17149000D+01
     .,0.27455800D+03,0.152D+03,0.128D+03,0.98110000D+00,0.17937000D+01
     .,0.27027830D+03,0.152D+03,0.129D+03,0.98110000D+00,0.95760000D+00
     .,0.25527130D+03,0.152D+03,0.130D+03,0.98110000D+00,0.19419000D+01
     .,0.41170180D+03,0.152D+03,0.131D+03,0.98110000D+00,0.96010000D+00
     .,0.36448820D+03,0.152D+03,0.132D+03,0.98110000D+00,0.94340000D+00
     .,0.32849090D+03,0.152D+03,0.133D+03,0.98110000D+00,0.98890000D+00
     .,0.30105670D+03,0.152D+03,0.134D+03,0.98110000D+00,0.99010000D+00
     .,0.26622780D+03,0.152D+03,0.135D+03,0.98110000D+00,0.99740000D+00
     .,0.43493910D+03,0.152D+03,0.137D+03,0.98110000D+00,0.97380000D+00
     .,0.83036610D+03,0.152D+03,0.138D+03,0.98110000D+00,0.98010000D+00
     .,0.64326970D+03,0.152D+03,0.139D+03,0.98110000D+00,0.19153000D+01
     ./)
      pars(  50701:  50800)=(/
     . 0.48508560D+03,0.152D+03,0.140D+03,0.98110000D+00,0.19355000D+01
     .,0.48977500D+03,0.152D+03,0.141D+03,0.98110000D+00,0.19545000D+01
     .,0.45747170D+03,0.152D+03,0.142D+03,0.98110000D+00,0.19420000D+01
     .,0.50987810D+03,0.152D+03,0.143D+03,0.98110000D+00,0.16682000D+01
     .,0.40053650D+03,0.152D+03,0.144D+03,0.98110000D+00,0.18584000D+01
     .,0.37486560D+03,0.152D+03,0.145D+03,0.98110000D+00,0.19003000D+01
     .,0.34838020D+03,0.152D+03,0.146D+03,0.98110000D+00,0.18630000D+01
     .,0.33677710D+03,0.152D+03,0.147D+03,0.98110000D+00,0.96790000D+00
     .,0.33432410D+03,0.152D+03,0.148D+03,0.98110000D+00,0.19539000D+01
     .,0.52288630D+03,0.152D+03,0.149D+03,0.98110000D+00,0.96330000D+00
     .,0.47620870D+03,0.152D+03,0.150D+03,0.98110000D+00,0.95140000D+00
     .,0.44801730D+03,0.152D+03,0.151D+03,0.98110000D+00,0.97490000D+00
     .,0.42510150D+03,0.152D+03,0.152D+03,0.98110000D+00,0.98110000D+00
     .,0.32815100D+02,0.153D+03,0.100D+01,0.99680000D+00,0.91180000D+00
     .,0.22271700D+02,0.153D+03,0.200D+01,0.99680000D+00,0.00000000D+00
     .,0.45439540D+03,0.153D+03,0.300D+01,0.99680000D+00,0.00000000D+00
     .,0.27643090D+03,0.153D+03,0.400D+01,0.99680000D+00,0.00000000D+00
     .,0.19192730D+03,0.153D+03,0.500D+01,0.99680000D+00,0.00000000D+00
     .,0.13262510D+03,0.153D+03,0.600D+01,0.99680000D+00,0.00000000D+00
     .,0.94299100D+02,0.153D+03,0.700D+01,0.99680000D+00,0.00000000D+00
     ./)
      pars(  50801:  50900)=(/
     . 0.72235800D+02,0.153D+03,0.800D+01,0.99680000D+00,0.00000000D+00
     .,0.55268900D+02,0.153D+03,0.900D+01,0.99680000D+00,0.00000000D+00
     .,0.42851200D+02,0.153D+03,0.100D+02,0.99680000D+00,0.00000000D+00
     .,0.54528550D+03,0.153D+03,0.110D+02,0.99680000D+00,0.00000000D+00
     .,0.43644520D+03,0.153D+03,0.120D+02,0.99680000D+00,0.00000000D+00
     .,0.40871490D+03,0.153D+03,0.130D+02,0.99680000D+00,0.00000000D+00
     .,0.32881570D+03,0.153D+03,0.140D+02,0.99680000D+00,0.00000000D+00
     .,0.26094460D+03,0.153D+03,0.150D+02,0.99680000D+00,0.00000000D+00
     .,0.21904300D+03,0.153D+03,0.160D+02,0.99680000D+00,0.00000000D+00
     .,0.18086040D+03,0.153D+03,0.170D+02,0.99680000D+00,0.00000000D+00
     .,0.14934300D+03,0.153D+03,0.180D+02,0.99680000D+00,0.00000000D+00
     .,0.88900260D+03,0.153D+03,0.190D+02,0.99680000D+00,0.00000000D+00
     .,0.75534780D+03,0.153D+03,0.200D+02,0.99680000D+00,0.00000000D+00
     .,0.62833170D+03,0.153D+03,0.210D+02,0.99680000D+00,0.00000000D+00
     .,0.61099490D+03,0.153D+03,0.220D+02,0.99680000D+00,0.00000000D+00
     .,0.56176430D+03,0.153D+03,0.230D+02,0.99680000D+00,0.00000000D+00
     .,0.44344010D+03,0.153D+03,0.240D+02,0.99680000D+00,0.00000000D+00
     .,0.48650250D+03,0.153D+03,0.250D+02,0.99680000D+00,0.00000000D+00
     .,0.38289000D+03,0.153D+03,0.260D+02,0.99680000D+00,0.00000000D+00
     .,0.40860640D+03,0.153D+03,0.270D+02,0.99680000D+00,0.00000000D+00
     ./)
      pars(  50901:  51000)=(/
     . 0.41915920D+03,0.153D+03,0.280D+02,0.99680000D+00,0.00000000D+00
     .,0.32203460D+03,0.153D+03,0.290D+02,0.99680000D+00,0.00000000D+00
     .,0.33424140D+03,0.153D+03,0.300D+02,0.99680000D+00,0.00000000D+00
     .,0.39422890D+03,0.153D+03,0.310D+02,0.99680000D+00,0.00000000D+00
     .,0.35311560D+03,0.153D+03,0.320D+02,0.99680000D+00,0.00000000D+00
     .,0.30552140D+03,0.153D+03,0.330D+02,0.99680000D+00,0.00000000D+00
     .,0.27661090D+03,0.153D+03,0.340D+02,0.99680000D+00,0.00000000D+00
     .,0.24427520D+03,0.153D+03,0.350D+02,0.99680000D+00,0.00000000D+00
     .,0.21420040D+03,0.153D+03,0.360D+02,0.99680000D+00,0.00000000D+00
     .,0.99978710D+03,0.153D+03,0.370D+02,0.99680000D+00,0.00000000D+00
     .,0.89949300D+03,0.153D+03,0.380D+02,0.99680000D+00,0.00000000D+00
     .,0.79779010D+03,0.153D+03,0.390D+02,0.99680000D+00,0.00000000D+00
     .,0.72272400D+03,0.153D+03,0.400D+02,0.99680000D+00,0.00000000D+00
     .,0.66260680D+03,0.153D+03,0.410D+02,0.99680000D+00,0.00000000D+00
     .,0.51659320D+03,0.153D+03,0.420D+02,0.99680000D+00,0.00000000D+00
     .,0.57428050D+03,0.153D+03,0.430D+02,0.99680000D+00,0.00000000D+00
     .,0.44217100D+03,0.153D+03,0.440D+02,0.99680000D+00,0.00000000D+00
     .,0.48282170D+03,0.153D+03,0.450D+02,0.99680000D+00,0.00000000D+00
     .,0.44920610D+03,0.153D+03,0.460D+02,0.99680000D+00,0.00000000D+00
     .,0.37444670D+03,0.153D+03,0.470D+02,0.99680000D+00,0.00000000D+00
     ./)
      pars(  51001:  51100)=(/
     . 0.39749790D+03,0.153D+03,0.480D+02,0.99680000D+00,0.00000000D+00
     .,0.49366250D+03,0.153D+03,0.490D+02,0.99680000D+00,0.00000000D+00
     .,0.46207160D+03,0.153D+03,0.500D+02,0.99680000D+00,0.00000000D+00
     .,0.41687150D+03,0.153D+03,0.510D+02,0.99680000D+00,0.00000000D+00
     .,0.38970290D+03,0.153D+03,0.520D+02,0.99680000D+00,0.00000000D+00
     .,0.35521750D+03,0.153D+03,0.530D+02,0.99680000D+00,0.00000000D+00
     .,0.32180290D+03,0.153D+03,0.540D+02,0.99680000D+00,0.00000000D+00
     .,0.12196272D+04,0.153D+03,0.550D+02,0.99680000D+00,0.00000000D+00
     .,0.11429378D+04,0.153D+03,0.560D+02,0.99680000D+00,0.00000000D+00
     .,0.10164155D+04,0.153D+03,0.570D+02,0.99680000D+00,0.00000000D+00
     .,0.49161500D+03,0.153D+03,0.580D+02,0.99680000D+00,0.27991000D+01
     .,0.10169588D+04,0.153D+03,0.590D+02,0.99680000D+00,0.00000000D+00
     .,0.97834710D+03,0.153D+03,0.600D+02,0.99680000D+00,0.00000000D+00
     .,0.95429670D+03,0.153D+03,0.610D+02,0.99680000D+00,0.00000000D+00
     .,0.93211660D+03,0.153D+03,0.620D+02,0.99680000D+00,0.00000000D+00
     .,0.91246710D+03,0.153D+03,0.630D+02,0.99680000D+00,0.00000000D+00
     .,0.72800530D+03,0.153D+03,0.640D+02,0.99680000D+00,0.00000000D+00
     .,0.80453430D+03,0.153D+03,0.650D+02,0.99680000D+00,0.00000000D+00
     .,0.77785480D+03,0.153D+03,0.660D+02,0.99680000D+00,0.00000000D+00
     .,0.82536440D+03,0.153D+03,0.670D+02,0.99680000D+00,0.00000000D+00
     ./)
      pars(  51101:  51200)=(/
     . 0.80806540D+03,0.153D+03,0.680D+02,0.99680000D+00,0.00000000D+00
     .,0.79260610D+03,0.153D+03,0.690D+02,0.99680000D+00,0.00000000D+00
     .,0.78284050D+03,0.153D+03,0.700D+02,0.99680000D+00,0.00000000D+00
     .,0.66614790D+03,0.153D+03,0.710D+02,0.99680000D+00,0.00000000D+00
     .,0.66319200D+03,0.153D+03,0.720D+02,0.99680000D+00,0.00000000D+00
     .,0.61001430D+03,0.153D+03,0.730D+02,0.99680000D+00,0.00000000D+00
     .,0.51873530D+03,0.153D+03,0.740D+02,0.99680000D+00,0.00000000D+00
     .,0.52912210D+03,0.153D+03,0.750D+02,0.99680000D+00,0.00000000D+00
     .,0.48278010D+03,0.153D+03,0.760D+02,0.99680000D+00,0.00000000D+00
     .,0.44456400D+03,0.153D+03,0.770D+02,0.99680000D+00,0.00000000D+00
     .,0.37168100D+03,0.153D+03,0.780D+02,0.99680000D+00,0.00000000D+00
     .,0.34813080D+03,0.153D+03,0.790D+02,0.99680000D+00,0.00000000D+00
     .,0.35883830D+03,0.153D+03,0.800D+02,0.99680000D+00,0.00000000D+00
     .,0.50911500D+03,0.153D+03,0.810D+02,0.99680000D+00,0.00000000D+00
     .,0.50219660D+03,0.153D+03,0.820D+02,0.99680000D+00,0.00000000D+00
     .,0.46641830D+03,0.153D+03,0.830D+02,0.99680000D+00,0.00000000D+00
     .,0.44766490D+03,0.153D+03,0.840D+02,0.99680000D+00,0.00000000D+00
     .,0.41630430D+03,0.153D+03,0.850D+02,0.99680000D+00,0.00000000D+00
     .,0.38423110D+03,0.153D+03,0.860D+02,0.99680000D+00,0.00000000D+00
     .,0.11630702D+04,0.153D+03,0.870D+02,0.99680000D+00,0.00000000D+00
     ./)
      pars(  51201:  51300)=(/
     . 0.11376470D+04,0.153D+03,0.880D+02,0.99680000D+00,0.00000000D+00
     .,0.10167803D+04,0.153D+03,0.890D+02,0.99680000D+00,0.00000000D+00
     .,0.92598120D+03,0.153D+03,0.900D+02,0.99680000D+00,0.00000000D+00
     .,0.91390750D+03,0.153D+03,0.910D+02,0.99680000D+00,0.00000000D+00
     .,0.88520210D+03,0.153D+03,0.920D+02,0.99680000D+00,0.00000000D+00
     .,0.90396880D+03,0.153D+03,0.930D+02,0.99680000D+00,0.00000000D+00
     .,0.87665190D+03,0.153D+03,0.940D+02,0.99680000D+00,0.00000000D+00
     .,0.51949600D+02,0.153D+03,0.101D+03,0.99680000D+00,0.00000000D+00
     .,0.16143030D+03,0.153D+03,0.103D+03,0.99680000D+00,0.98650000D+00
     .,0.20721850D+03,0.153D+03,0.104D+03,0.99680000D+00,0.98080000D+00
     .,0.16249390D+03,0.153D+03,0.105D+03,0.99680000D+00,0.97060000D+00
     .,0.12439580D+03,0.153D+03,0.106D+03,0.99680000D+00,0.98680000D+00
     .,0.87986900D+02,0.153D+03,0.107D+03,0.99680000D+00,0.99440000D+00
     .,0.64976400D+02,0.153D+03,0.108D+03,0.99680000D+00,0.99250000D+00
     .,0.45461800D+02,0.153D+03,0.109D+03,0.99680000D+00,0.99820000D+00
     .,0.23487760D+03,0.153D+03,0.111D+03,0.99680000D+00,0.96840000D+00
     .,0.36232580D+03,0.153D+03,0.112D+03,0.99680000D+00,0.96280000D+00
     .,0.37186150D+03,0.153D+03,0.113D+03,0.99680000D+00,0.96480000D+00
     .,0.30487150D+03,0.153D+03,0.114D+03,0.99680000D+00,0.95070000D+00
     .,0.25346770D+03,0.153D+03,0.115D+03,0.99680000D+00,0.99470000D+00
     ./)
      pars(  51301:  51400)=(/
     . 0.21658660D+03,0.153D+03,0.116D+03,0.99680000D+00,0.99480000D+00
     .,0.17895810D+03,0.153D+03,0.117D+03,0.99680000D+00,0.99720000D+00
     .,0.32822070D+03,0.153D+03,0.119D+03,0.99680000D+00,0.97670000D+00
     .,0.60618590D+03,0.153D+03,0.120D+03,0.99680000D+00,0.98310000D+00
     .,0.33318010D+03,0.153D+03,0.121D+03,0.99680000D+00,0.18627000D+01
     .,0.32194330D+03,0.153D+03,0.122D+03,0.99680000D+00,0.18299000D+01
     .,0.31543490D+03,0.153D+03,0.123D+03,0.99680000D+00,0.19138000D+01
     .,0.31193950D+03,0.153D+03,0.124D+03,0.99680000D+00,0.18269000D+01
     .,0.28957840D+03,0.153D+03,0.125D+03,0.99680000D+00,0.16406000D+01
     .,0.26879340D+03,0.153D+03,0.126D+03,0.99680000D+00,0.16483000D+01
     .,0.25647210D+03,0.153D+03,0.127D+03,0.99680000D+00,0.17149000D+01
     .,0.25055360D+03,0.153D+03,0.128D+03,0.99680000D+00,0.17937000D+01
     .,0.24584350D+03,0.153D+03,0.129D+03,0.99680000D+00,0.95760000D+00
     .,0.2335632