/*
 * SPDX-License-Identifier: AGPL-3.0-only
 * Copyright 2005 - 2016 Zarafa and its licensors
 */
#ifndef ECGENERICPROP_H
#define ECGENERICPROP_H

#include <memory>
#include <mutex>
#include <kopano/ECUnknown.h>
#include "IECPropStorage.h"
#include "ECPropertyEntry.h"
#include <kopano/IECInterfaces.hpp>
#include <kopano/memory.hpp>
#include <list>
#include <map>
#include <set>

// These are the callback functions called when a software-handled property is requested
typedef HRESULT (*SetPropCallBack)(ULONG ulPropTag, void *lpProvider, const SPropValue *lpsPropValue, void *lpParam);
typedef HRESULT (* GetPropCallBack)(ULONG ulPropTag, void* lpProvider, ULONG ulFlags, LPSPropValue lpsPropValue, void *lpParam, void *lpBase);

struct PROPCALLBACK {
	ULONG ulPropTag;
	SetPropCallBack lpfnSetProp;
	GetPropCallBack lpfnGetProp;
	void *			lpParam;
	BOOL			fRemovable;
	BOOL			fHidden; // hidden from GetPropList

	bool operator==(const PROPCALLBACK &callback) const noexcept
	{
		return callback.ulPropTag == ulPropTag;
	}
};

typedef std::map<short, PROPCALLBACK>			ECPropCallBackMap;
typedef ECPropCallBackMap::iterator				ECPropCallBackIterator;
typedef std::map<short, ECPropertyEntry>		ECPropertyEntryMap;
typedef ECPropertyEntryMap::iterator			ECPropertyEntryIterator;

class ECGenericProp :
    public KC::ECUnknown, public virtual IMAPIProp,
    public KC::IECSingleInstance {
protected:
	ECGenericProp(void *lpProvider, ULONG ulObjType, BOOL fModify, const char *szClassName = NULL);
	virtual ~ECGenericProp() = default;

public:
	virtual HRESULT QueryInterface(const IID &, void **) override;

	HRESULT SetProvider(void* lpProvider);
	HRESULT SetEntryId(ULONG eid_size, const ENTRYID *eid);
	static HRESULT DefaultGetPropGetReal(ULONG ulPropTag, void *lpProvider, ULONG ulFlags, LPSPropValue lpsPropValue, void *lpParam, void *lpBase);
	static HRESULT DefaultSetPropComputed(ULONG ulPropTag, void *lpProvider, const SPropValue *lpsPropValue, void *lpParam);
	static HRESULT DefaultSetPropIgnore(ULONG ulPropTag, void *lpProvider, const SPropValue *lpsPropValue, void *lpParam);
	static HRESULT DefaultSetPropSetReal(ULONG ulPropTag, void *lpProvider, const SPropValue *lpsPropValue, void *lpParam);
	static HRESULT		DefaultGetProp(ULONG ulPropTag, void* lpProvider, ULONG ulFlags, LPSPropValue lpsPropValue, void *lpParam, void *lpBase);

	// Our table-row getprop handler (handles client-side generation of table columns)
	static HRESULT TableRowGetProp(void *prov, const struct propVal *src, SPropValue *dst, void **base, ULONG type);
	virtual HRESULT HrSetPropStorage(IECPropStorage *lpStorage, BOOL fLoadProps);
	virtual HRESULT HrSetRealProp(const SPropValue *lpsPropValue);
	virtual HRESULT		HrGetRealProp(ULONG ulPropTag, ULONG ulFlags, void *lpBase, LPSPropValue lpsPropValue, ULONG ulMaxSize = 0);
	virtual HRESULT		HrAddPropHandlers(ULONG ulPropTag, GetPropCallBack lpfnGetProp, SetPropCallBack lpfnSetProp, void *lpParam, BOOL fRemovable = FALSE, BOOL fHidden = FALSE);
	virtual HRESULT 	HrLoadEmptyProps();

	bool IsReadOnly() const;

protected: ///?
	virtual HRESULT		HrLoadProps();
	HRESULT				HrLoadProp(ULONG ulPropTag);
	virtual HRESULT		HrDeleteRealProp(ULONG ulPropTag, BOOL fOverwriteRO);
	HRESULT				HrGetHandler(ULONG ulPropTag, SetPropCallBack *lpfnSetProp, GetPropCallBack *lpfnGetProp, void **lpParam);
	HRESULT				IsPropDirty(ULONG ulPropTag, BOOL *lpbDirty);
	HRESULT				HrSetClean();
	HRESULT				HrSetCleanProperty(ULONG ulPropTag);

	/* used by child to save here in it's parent */
	friend class ECParentStorage;
	virtual HRESULT HrSaveChild(ULONG ulFlags, MAPIOBJECT *lpsMapiObject);
	virtual HRESULT HrRemoveModifications(MAPIOBJECT *lpsMapiObject, ULONG ulPropTag);

	// For IECSingleInstance
	virtual HRESULT GetSingleInstanceId(ULONG *id_size, ENTRYID **id) override;
	virtual HRESULT SetSingleInstanceId(ULONG eid_size, const ENTRYID *eid) override;

public:
	// From IMAPIProp
	virtual HRESULT GetLastError(HRESULT, ULONG flags, MAPIERROR **) override;
	virtual HRESULT SaveChanges(ULONG flags) override;
	virtual HRESULT GetProps(const SPropTagArray *, ULONG flags, ULONG *nprops, SPropValue **) override;
	virtual HRESULT GetPropList(ULONG flags, SPropTagArray **) override;

	/**
	 * \brief Opens a property.
	 *
	 * \param ulPropTag				The proptag of the property to open.
	 * \param lpiid					Pointer to the requested interface for the object to be opened.
	 * \param ulInterfaceOptions	Interface options.
	 * \param ulFlags				Flags.
	 * \param lppUnk				Pointer to an IUnknown pointer where the opened object will be stored.
	 *
	 * \return hrSuccess on success.
	 */
	virtual HRESULT OpenProperty(ULONG tag, const IID *intf, ULONG intf_opts, ULONG flags, IUnknown **) override;
	virtual HRESULT SetProps(ULONG nprops, const SPropValue *props, SPropProblemArray **) override;
	virtual HRESULT DeleteProps(const SPropTagArray *, LPSPropProblemArray *) override;
	virtual HRESULT CopyTo(ULONG nexcl, const IID *excl, const SPropTagArray *exclprop, ULONG ui_param, IMAPIProgress *, const IID *intf, void *dest, ULONG flags, SPropProblemArray **) override;
	virtual HRESULT CopyProps(const SPropTagArray *inclprop, ULONG ui_param, IMAPIProgress *, const IID *intf, void *dest, ULONG flags, SPropProblemArray **) override;
	virtual HRESULT GetNamesFromIDs(SPropTagArray **tags, const GUID *propset, ULONG flags, ULONG *nvals, MAPINAMEID ***names) override;
	virtual HRESULT GetIDsFromNames(ULONG n, MAPINAMEID **, ULONG flags, SPropTagArray **) override;

protected:
	ECPropertyEntryMap lstProps;
	std::set<ULONG>			m_setDeletedProps;
	ECPropCallBackMap		lstCallBack;
	DWORD dwLastError = hrSuccess;
	BOOL fSaved = false; // only 0 if just created, // not saved until we either read or write from/to disk
	ULONG					ulObjType;
	ULONG ulObjFlags = 0; // message: MAPI_ASSOCIATED, folder: FOLDER_SEARCH (last?)
	BOOL					fModify;
	void*					lpProvider;
	BOOL isTransactedObject = true; // only ECMsgStore and ECMAPIFolder are not transacted
	ULONG m_ulMaxPropSize = 8192;
	bool m_props_loaded = false;

public:
	// Current entryid of object
	ULONG m_cbEntryId = 0;
	std::recursive_mutex m_hMutexMAPIObject; /* Mutex for locking the MAPIObject */
	BOOL m_bReload = false, m_bLoading = false;
	KC::memory_ptr<ENTRYID> m_lpEntryId;
	KC::object_ptr<IECPropStorage> lpStorage;
	std::unique_ptr<MAPIOBJECT> m_sMapiObject;
};

inline bool ECGenericProp::IsReadOnly() const {
	return !fModify;
}

#endif
