{-
    Kaya - My favourite toy language.
    Copyright (C) 2004-2006 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module Chaser where

import Parser
import Language
import Options
import Module
import Lib
import Lexer

import Char
import IO
import System
import System.IO
import System.Directory
import Debug.Trace
import Monad

import System.IO.Unsafe
import Foreign
import Data.Char

--import Data.ByteString.Char8(readFile,unpack)

-- File name, module name and dependencies
data ModuleTree = Mod FilePath InputType Name [ModuleTree]
   deriving Show

data ModuleData = MD { mod_inputtype :: InputType,
                       mod_name :: String,
                       mod_imports :: [String] }

allDeps [] = []
allDeps ((Mod f t mod deps):xs) = f:(allDeps deps ++ allDeps xs)

instance Show ModuleData where
    show (MD int nm imp) = show int ++ " " ++ nm ++ " depends on " ++ show imp

data CompileNeeded = Comp { cn_filename :: FilePath,
                            cn_inputtype :: InputType,
                            cn_module :: Name,
                            cn_build :: Bool }
   deriving Show

-- Given a top level file, return a tree describing which modules depend 
-- on which others. Look out for circular dependencies and ignore them.
getDeps :: FilePath -> Options -> [FilePath] -> IO ModuleTree
getDeps fn opts done = 
    do libdirs <- getAllLibDirs opts
       prog <- strictReadFile fn
--       let imppt = parse (getroot fn) libdirs prog fn
       let moddata = quickParse (getroot fn) prog (nochase opts)
       when (dumpdeps opts) $ do
               case moddata of
                        (Success x) -> do 
                           putStrLn $ show x
                           when (length done > 0) $ 
                             putStrLn $ "\tThis path contains " ++ show done
                        _ -> return ()
       -- Dig out all of the imports from imppt
       if (nochase opts)
          then case moddata of
                   (Success (MD ty mod pt)) -> return $ Mod fn ty (UN mod) []
                   (Failure _ _ _) -> return $ Mod fn Module (UN (getroot fn)) []
          else findImports moddata libdirs
  where getroot ".k" = ""
	getroot (x:xs) = x:(getroot xs)
	getroot [] = ""

        findImports (Failure _ _ _) libdirs = return $ Mod fn Module (UN "error") []
  -- fail "Can't happen (findImports)" -- actually can happen!
        findImports (Success (MD ty mod pt)) libdirs = do
            deps <- fi libdirs pt
            return $ Mod fn ty (UN mod) deps
        fi libdirs [] = return []
        fi libdirs (x:xs) =
            -- look in current directory only
            do mimpfn <- findLib [""] (x++".k")
               case mimpfn of
                 Just impfn -> 
                          if (impfn `elem` (fn:done))
                            then do putStrLn $ "Warning: " ++
                                               "Circular dependency (in " 
                                                ++ show fn ++ ")"
                                    return []
                            else do
                              modtreeRec <- getDeps impfn opts (fn:done)
                              rest <- fi libdirs xs
                              return (modtreeRec:rest)
                 Nothing -> do
                  -- Make sure the ki exists at least!
                             mfn <- findLib libdirs (x++".ki")
                             case mfn of
                               Just fn -> fi libdirs xs
                               Nothing -> return [] --fail $ "Can't find " ++ x++".k" ++ 
                                                 -- " or " ++ x ++ ".ki in library search path"

{-
strictReadFile :: FilePath -> IO String
strictReadFile fp = do catch (do h <- openFile fp ReadMode
                                 contents <- getLines [] h
                                 hClose h
                                 return (unlines (reverse contents)))
                         (\e -> do putStrLn (show e)
                                   exitWith (ExitFailure 1))
   where getLines acc h = do b <- hIsEOF h
                             if b then return acc else
                                 do l <- hGetLine h
                                    getLines (l:acc) h
-}

{-
Aargh, 6.6 only!
strictReadFile fn = do c <- Data.ByteString.Char8.readFile fn
                       return (unpack c)
-}

strictReadFile f = do
   h <- openFile f ReadMode
   s <- hFileSize h
   fp <- mallocForeignPtrBytes (fromIntegral s)
   len <- withForeignPtr fp $ \buf -> hGetBuf h buf (fromIntegral s)
   hClose h
   lazySlurp fp 0 len

buf_size = 4096 :: Int

lazySlurp :: ForeignPtr Word8 -> Int -> Int -> IO String
lazySlurp fp ix len
   | fp `seq` False = undefined
   | ix >= len = return []
   | otherwise = do
       cs <- unsafeInterleaveIO (lazySlurp fp (ix + buf_size) len)
       ws <- withForeignPtr fp $ \p -> loop (min (len-ix) buf_size - 1) ((p :: Ptr Word8) `plusPtr` ix) cs
       return ws
  where
   loop :: Int -> Ptr Word8 -> String -> IO String
   loop len p acc
     | len `seq` p `seq` False = undefined
     | len < 0 = return acc
     | otherwise = do
        w <- peekElemOff p len
        loop (len-1) p (chr (fromIntegral w):acc)


buildWhat :: Options -> ModuleTree -> IO [CompileNeeded]
buildWhat opts (Mod fn ty mod deps) = do
    if (nochase opts)
       then return [Comp fn ty mod True]
       else do
            anyNeeded <- mapM (uptodateCheck ty mod) (fn:(allDeps deps))
            let needed = or anyNeeded
            builddeps <- mapM (buildWhat opts) deps
            let alldeps = undup $ concat builddeps
            let buildit = (forcebuild opts) || needed ||
                          (or $ map cn_build alldeps)
            return $ (Comp fn ty mod buildit) : alldeps
 where undup [] = []
       undup (x:xs) | (cn_filename x) `elem` (map cn_filename xs) = undup xs
                    | otherwise = x:(undup xs)

uptodateCheck t mod infile =
    do let outfile = outputfile t mod
       ex <- doesFileExist outfile	  
       if (not ex) then return True
	  else do inmod <- getModificationTime infile
		  outmod <- getModificationTime outfile
		  return (inmod>outmod)

outputfile Module mod = showuser mod ++ ".o"
outputfile SharedLib mod = showuser mod ++ ".o"
-- TMP HACK: This should probably be a %extension "cgi" directive in the .ks
outputfile (Program "webapp") mod = showuser mod ++ ".cgi"
outputfile (Program "webprog") mod = showuser mod ++ ".cgi"
outputfile (Program _) mod = showuser mod
outputfile Shebang mod = showuser mod
-- outputfile Webapp mod = showuser mod ++ ".cgi"
-- outputfile Webprog mod = showuser mod ++ ".cgi"

-- Extract module name and import data from a file.
-- FIXME: It would really be better to use the real parser, if we can
-- find a way of not parsing the bits we don't actually care about. I doubt
-- this is really possible, and this hack should be fine.

quickParse :: String -> String -> Bool -> Result ModuleData
quickParse root inf quick 
    = do let stripped = stripComments inf
         (inf, mty, modname) <- parseModName root stripped
         imports <- case quick of
                      True -> return []
                      _ -> parseImports inf []
         return $ MD mty modname imports

-- strip Comments and spaces (i.e. non code things)

stripComments [] = []
stripComments ('/':'/':xs) = stripComments (dropToEndLine xs)
stripComments ('/':'*':xs) = stripComments (dropToEndComment xs)
stripComments ('"':xs) = case getstr False xs of
                            Just (_,rest,_) -> stripComments rest
                            _ -> ""
stripComments (x:xs) = x:(stripComments xs)

dropToEndLine [] = []
dropToEndLine l@('\n':xs) = l
dropToEndLine (x:xs) = dropToEndLine xs

dropToEndComment [] = []
dropToEndComment ('*':'/':xs) = xs
dropToEndComment (x:xs) = dropToEndComment xs

parseModName :: String -> String -> Result (String, InputType, String)
parseModName root [] = fail "Not a module"
parseModName root ('#':'!':cs) = return (cs, Shebang, root)
parseModName root (c:cs) = case span isAlpha (c:cs) of
   ("", _) -> parseModName root cs
   (x,rest) -> if (isHdr x)
                 then do (rest, nm) <- getName rest "module"
                         return (rest, getty x, nm)
                 else parseModName root rest
  where getty "module" = Module
        getty ty = Program ty
--        getty "webapp" = Webapp
--        getty "webprog" = Webprog
        isHdr ('#':'!':xs) = False
        isHdr _ = True

getName cs thing = case span isAllowed (stripspace cs) of
   ("", _) -> fail $ "Can't get " ++ thing ++ " name"
   ("public", rest) -> getName rest thing
   (x, rest) -> if (head (stripspace rest) == ';') 
                   then return (rest, x)
                   else fail "Not an import"
  where stripspace (x:xs) | isSpace x || x == '\n' = stripspace xs
        stripspace xs = xs

-- only chase imports at the start of a line.

parseImports :: String -> [String] -> Result [String]
parseImports [] acc = return acc
parseImports ('\n':c:cs) acc = case span isAlpha (c:cs) of
   ("",_) -> parseImports (c:cs) acc
   ("import",rest) -> case getName rest "import" of
                        (Success (rest, nextImp)) ->
                           parseImports rest (nextImp:acc)
                        _ -> parseImports rest acc
   (_,rest) -> parseImports rest acc
parseImports (c:cs) acc -- = parseImports cs acc
   | isSpace c || c==';' = parseImports cs acc
   | otherwise = return acc