/*
 * Copyright (C) 2014 Guitarix project MOD project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */


////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_common.h"      // faust support and denormal protection (SSE)
#include "gx_vibe.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "vibe.cc"    // dsp class generated by faust -> dsp2cc

////////////////////////////// PLUG-IN CLASS ///////////////////////////

namespace vibe {

class Gx_vibe_
{
private:
  bool        Pstereo;
  // pointer to buffer
  float*      output;
  float*      input;
  float*      output1;
  float*      input1;
  // pointer to dsp class
  PluginLV2*  vibe;
  // private functions
  inline void run_dsp_(uint32_t n_samples);
  inline void run_dsp_mono(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate);
  inline void init_dsp_mono(uint32_t rate);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  static const LV2_Descriptor descriptor1;
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);
  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  static LV2_Handle instantiateM(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);
  Gx_vibe_();
  ~Gx_vibe_();
};

// constructor
Gx_vibe_::Gx_vibe_() :
  output(NULL),
  input(NULL),
  output1(NULL),
  input1(NULL) {};

// destructor
Gx_vibe_::~Gx_vibe_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (vibe->activate_plugin !=0)
    vibe->activate_plugin(false, vibe);
  // delete DSP class
  vibe->delete_instance(vibe);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_vibe_::init_dsp_(uint32_t rate)
{
  Pstereo = true;
  vibe=vibe::plugin_stereo();
  AVOIDDENORMALS(); // init the SSE denormal protection
  vibe->set_samplerate(rate, vibe); // init the DSP class
}

void Gx_vibe_::init_dsp_mono(uint32_t rate)
{
  Pstereo = false;
  vibe=vibe::plugin_mono();
  AVOIDDENORMALS(); // init the SSE denormal protection
  vibe->set_samplerate(rate, vibe); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_vibe_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
    case EFFECTS_OUTPUT1:
      output1 = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT1:
      input1 = static_cast<float*>(data);
      break;
    default:
      break;
    }
}

void Gx_vibe_::activate_f()
{
  // allocate the internal DSP mem
  if (vibe->activate_plugin !=0)
    vibe->activate_plugin(true, vibe);
}

void Gx_vibe_::clean_up()
{
  // delete the internal DSP mem
  if (vibe->activate_plugin !=0)
    vibe->activate_plugin(false, vibe);
}

void Gx_vibe_::deactivate_f()
{
  // delete the internal DSP mem
  if (vibe->activate_plugin !=0)
    vibe->activate_plugin(false, vibe);
}

void Gx_vibe_::run_dsp_(uint32_t n_samples)
{
  if (n_samples< 1) return;
  vibe->stereo_audio(static_cast<int>(n_samples), input, input1, output, output1, vibe);
}

void Gx_vibe_::run_dsp_mono(uint32_t n_samples)
{
  vibe->mono_audio(static_cast<int>(n_samples), input, output, vibe);
}

void Gx_vibe_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  vibe->connect_ports(port,  data, vibe);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Handle 
Gx_vibe_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_vibe_ *self = new Gx_vibe_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_((uint32_t)rate);

  return (LV2_Handle)self;
}

LV2_Handle 
Gx_vibe_::instantiateM(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_vibe_ *self = new Gx_vibe_();
  if (!self)
    {
      return NULL;
    }

  self->init_dsp_mono((uint32_t)rate);

  return (LV2_Handle)self;
}

void Gx_vibe_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_vibe_*>(instance)->connect_all__ports(port, data);
}

void Gx_vibe_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_vibe_*>(instance)->activate_f();
}

void Gx_vibe_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  Gx_vibe_ * self = static_cast<Gx_vibe_*>(instance);
  if (self->Pstereo)
    self->run_dsp_(n_samples);
  else
    self->run_dsp_mono(n_samples);
}

void Gx_vibe_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_vibe_*>(instance)->deactivate_f();
}

void Gx_vibe_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_vibe_* self = static_cast<Gx_vibe_*>(instance);
  self->clean_up();
  delete self;
}

const LV2_Descriptor Gx_vibe_::descriptor =
{
  GXPLUGIN_URI "#_vibe_",
  Gx_vibe_::instantiate,
  Gx_vibe_::connect_port,
  Gx_vibe_::activate,
  Gx_vibe_::run,
  Gx_vibe_::deactivate,
  Gx_vibe_::cleanup,
  NULL
};

const LV2_Descriptor Gx_vibe_::descriptor1 =
{
  GXPLUGIN_URI "#_vibe_mono",
  Gx_vibe_::instantiateM,
  Gx_vibe_::connect_port,
  Gx_vibe_::activate,
  Gx_vibe_::run,
  Gx_vibe_::deactivate,
  Gx_vibe_::cleanup,
  NULL
};

} // end namespace vibe

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

extern "C"
LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &vibe::Gx_vibe_::descriptor;
    case 1:
      return &vibe::Gx_vibe_::descriptor1;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
