#ifndef __GM_WORLD_H__
#define __GM_WORLD_H__

#include <gtk/gtk.h>
#include <regex.h>

#include "gm-options.h"
#include "gm-net.h"
#include "gm-editor.h"
#include "gm-triggers.h"
#include "mcp/gm-mcp-session.h"

G_BEGIN_DECLS

/*
 * Type checking and casting macros
 */
#define GM_TYPE_WORLD				(gm_world_get_type())
#define GM_WORLD(obj)				(G_TYPE_CHECK_INSTANCE_CAST((obj), \
		GM_TYPE_WORLD, GmWorld))
#define GM_WORLD_CONST(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj), \
		GM_TYPE_WORLD, GmWorld const))
#define GM_WORLD_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST((klass), \
		GM_TYPE_WORLD, GmWorldClass))
#define GM_IS_WORLD(obj)			(G_TYPE_CHECK_INSTANCE_TYPE((obj), \
		GM_TYPE_WORLD))
#define GM_IS_WORLD_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE ((klass), \
		GM_TYPE_WORLD))
#define GM_WORLD_GET_CLASS(obj)		(G_TYPE_INSTANCE_GET_CLASS((obj), \
		GM_TYPE_WORLD, GmWorldClass))

/** \ingroup world
 *  \brief struct for world editing information
 *
 *  Struct which contains various fields for storing editing information. This
 *  is only used if editing via mcp_simpleedit can't be used.
 */
typedef struct _GmEditingInfo {
	gboolean is_editing; /**< are we editing at the moment? */
	GList *lines; /**< the lines to be edited */
	gchar *name; /**< the name of the editor */
	gchar *upload; /**< the command to send when sending the editor contents */
} GmEditingInfo;

/** \ingroup world
 *  \brief enum indicating log type
 *
 *  Enumeration which indicates the log type
 */
typedef enum _GmLogType GmLogType;
enum _GmLogType {
	LOG_NONE, /**< don't log */
	LOG_IN, /**< incoming lines */
	LOG_OUT, /**< outgoing lines */
	LOG_STATUS, /**< status lines (such as connecting information) */
	LOG_MCP_IN, /**< mcp incoming lines */
	LOG_MCP_OUT, /**< mcp outgoing lines */
	LOG_MCP_STATUS /**< mcp status lines */
};

/* Private structure type */
typedef struct _GmWorldPrivate GmWorldPrivate;

/*
 * Main object structure
 */
typedef struct _GmWorld GmWorld;
 
struct _GmWorld {
	GObject object;
	
	/*< private > */
	GmWorldPrivate *priv;
};

/*
 * Class definition
 */
typedef struct _GmWorldClass 	GmWorldClass;

struct _GmWorldClass {
	GObjectClass parent_class;

	/* Signals */
	void (* activate_request) (GmWorld *world);
	void (* load) (GmWorld *world);
	void (* unload) (GmWorld *world);
	void (* state_changing) (GmWorld *world, guint state);
	void (* world_error) (GmWorld *world, gchar const *error, gint code);
	void (* text_received) (GmWorld *world, gchar const *text);
	void (* editor_added) (GmWorld *world, GObject *editor);
	void (* editor_removed) (GmWorld *world, GObject *editor);
	void (* highlight) (GmWorld *world, gint start, gint end, 
			gchar const *color);
	void (* notify_message) (GmWorld *world, gchar const *message);
};

GType gm_world_get_type(void) G_GNUC_CONST;
GmWorld *gm_world_new(gchar *path);

GmWorld *gm_world_dup(GmWorld *source);
void gm_world_load(GmWorld *world);
void gm_world_unload(GmWorld *world);

const gchar *gm_world_name(GmWorld *world);
GmOptions *gm_world_options(GmWorld *world);
const gchar *gm_world_path(GmWorld *world);
GList **gm_world_history(GmWorld *world);
gint gm_world_activity(GmWorld *world);
gboolean gm_world_active(GmWorld *world);
GmTriggers *gm_world_triggers(GmWorld *world);

const gchar *gm_world_current_host(GmWorld *world);
const gchar *gm_world_current_port(GmWorld *world);

void gm_world_set_name(GmWorld *world, const gchar *name);
void gm_world_set_activity(GmWorld *world, gint activity);
void gm_world_set_active(GmWorld *world, gboolean active);

GmMcpSession *gm_world_get_mcp_session(GmWorld *world);

gboolean gm_world_loaded(GmWorld *world);
GmNetState gm_world_state(GmWorld *world);
gboolean gm_world_connected(GmWorld *world);
gboolean gm_world_disconnected(GmWorld *world);

void gm_world_connect(GmWorld *world);
void gm_world_connect_to(GmWorld *world, gchar const *host, gchar const *port);
void gm_world_disconnect(GmWorld *world);
void gm_world_prepare_disconnect(GmWorld *world);
void gm_world_add_editor(GmWorld *world, GmEditor *editor);
void gm_world_remove_editor(GmWorld *world, GmEditor *editor);
GSList const *gm_world_editors(GmWorld *world);
void gm_world_sendln_log(GmWorld *world, gchar const *text, GmLogType logtype);
void gm_world_sendln(GmWorld *world, gchar const *text);
void gm_world_writeln(GmWorld *world, gchar const *text);
void gm_world_status(GmWorld *world, gchar const *text);
void gm_world_process_input(GmWorld *world, gchar const *text);
void gm_world_log(GmWorld *world, GmLogType type, gchar const *text);
void gm_world_apply_trigger(GmWorld *world, GmTrigger *trigger, 
		gchar const *text, regmatch_t *matches, gint nummatches);

G_END_DECLS
#endif /* __GM_WORLD_H__ */
