#
# Locking.py - functions for locking/unlocking files
#
# Copyright (c) 2006-2007 The DITrack Project, www.ditrack.org.
#
# $Id: Locking.py 1696 2007-07-10 22:45:31Z vss $
# $HeadURL: https://svn.xiolabs.com/ditrack/src/tags/0.8/DITrack/Util/Locking.py $
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#  * Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
#  * Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

import datetime
import errno
import os
import time

class FileIsLockedError(Exception):
    message = "File is locked"

if os.name == "posix":
    import fcntl

class LockObject:
    def __init__(self, mode, fileobj):
        self.fileobj = fileobj
        self.mode = mode
        self.datetime = datetime.datetime.now()
        self.pid = os.getpid

        if os.name == "posix":
            if mode == "w":
                operation = fcntl.LOCK_EX | fcntl.LOCK_NB
            else:
                operation = fcntl.LOCK_SH | fcntl.LOCK_NB

            try:
                fcntl.flock(self.fileobj.fileno(), operation)
            except IOError, (errid, strerror):
                if ((errid == errno.EACCES) or (errid == errno.EAGAIN)):
                    raise FileIsLockedError
                else:
                    raise

    def __del__(self):
        self.unlock()

    def unlock(self):
        if os.name == "posix":
            fcntl.flock(self.fileobj.fileno(), fcntl.LOCK_UN)


def lock(fileobj, mode, timeout=0):
    """
    Lock the file object.
    The mode parameter should be "r" (shared/read lock) or
    "w" (exclusive/write lock).
    FileIsLockedError would be raised if:
     - there's already a read lock on fileobj and
      this is an attempt to create write lock
     - there's already a write lock on fileobj and
      this is an attempt to create read/write lock
    One fileobj is permitted to have several shared locks
    at once, but it is not so for exclusive lock.
    """

    if os.name == "posix":
        for i in range(timeout+1):
            try:
                lockobj = LockObject(mode, fileobj)
                return lockobj
            except FileIsLockedError:
                pass
            if timeout:
                time.sleep(1)
        else:
            raise FileIsLockedError
    else:
        return None

def unlock(lockobj):
    if os.name == "posix":
        if lockobj:
            lockobj.unlock()

