/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */



/**
 * @file diagnostics/frame/logging_implementation.hpp
 *
 * $Id: logging_implementation.hpp,v 1.11 2005/06/23 09:54:28 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] Interface of @ref diagnostics::internal::Logging_Implementation
 */

#ifndef DIAGNOSTICS__FRAME__LOGGING_IMPLEMENTATION_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__FRAME__LOGGING_IMPLEMENTATION_HPP__INCLUDE_GUARD

#include <diagnostics/frame/pp_config.hpp>

// used in the implementation by inlining
#include <diagnostics/frame/platform.hpp>

// used in the implementation by value
#include <diagnostics/frame/logger.hpp>

// used in the implementation by value
#include <diagnostics/frame/record.hpp>

#include <vector>

DIAGNOSTICS_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

class AVOID_WARNING_FRIEND;

/**
 * @brief The Implementation of @ref Logging_Config and @ref Logging_Facility
 *
 * This class implements both, the config and the facility.
 *
 * However, it leaves the mutexing open -- it handled by the the
 * methods of these two classes directly with the help of the @ref
 * DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD and @ref
 * DIAGNOSTICS_LOGGING_FACILITY_LOCK_IS_NESTED. Thus, this class can
 * be used by components which live behind the @ref
 * DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD to log. 
 *
 * @nosubgrouping
 */
class Logging_Implementation
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Singleton Implementation
     * @{
     */
private:
    typedef Logging_Implementation Self;

    /**
     * @brief Initializes the logging framework
     *
     * Fetches pid, hostname and startup time from the system.  Calls
     * @ref set_initial_loggers to obtain the initial set of loggers sends
     * them a @ref TYPE_LOG_OPEN message. 
     *
     * @throw never -- it aborts on problems after using @ref
     * DIAGNOSTICS_PANIC_LOG.
     */
    Logging_Implementation();

    /**
     * @brief Shuts the logging framework down
     *
     * Sends a @ref TYPE_LOG_CLOSE to all loggers, calls their @ref
     * Logger::close method and unregisters them.
     *
     * @throw never -- it tries to go on and calls @ref
     * DIAGNOSTICS_PANIC_LOG
     */
    ~Logging_Implementation();
    

    /**
     * @brief To avoid warning since cstr and dstr are both private.
     */
    friend class AVOID_WARNING_FRIEND;

public:
    /**
     * @brief Singleton implementation
     *
     * It is registering @ref atexit_handler -- however,
     * I have to check out the phoenix behavior.
     */
    inline static Self* instance();
private:
    static Self * m_instance;

    /**
     * @brief sets up the instance
     */
    static Self* setup_instance();

    /**
     * @brief delete @ref m_instance
     *
     * @todo make it clean -- phoenix pattern
     */
    static void atexit_handler();
    /* @} */

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Logging_Facility and Logging_Config support
     * @{
     */
public:
    /**
     * @brief panic_log is used inside of the logging framework and by
     * implementations @ref Logger to generate error messages.
     * 
     * For the arguments, see @ref log. The effect of calling @ref
     * panic_log is simple: The arugments are printed via ::std::cerr. 
     *
     * @throw never -- and if it throws, everything is lost. 
     */
    static void panic_log(::std::string const & what,
						  char const * const base_file_name,
						  char const * const file_name,
						  int const line);
    
    /**
     * @brief Construts a @ref Record and sends to each register @ref
     * Logger. This method is the inner implementation of @ref
     * Logging_Facility::log.
     *
     * It has no try/catch -- that's at the level of @ref
     * Logging_Facility::log, i.e., a panic_log is wrapped around. The
     * arguments are identical.
     *
     * @throw this method lets the exceptions of the loggers (WHICH
     * SHOULD NOT OCCUR) pass through
     */
    inline void log(Level_t const level,
		    Type_t const type,
		    int const nr_what,
		    ::std::string const & str_what,
		    char const * const base_file_name,
		    char const * const file_name,
		    int const line);
    /**
     * @brief Implementation of @ref Logging_Config::register_logger
     *
     * Specs are identical. -- mutexing is handled in the surrounding context
     */
    void register_logger(Logger * const logger);

    /**
     * @brief Implementation of @ref Logging_Config::register_logger
     *
     * Specs are identical. -- mutexing is handled in the surrounding context
     */
    void unregister_logger(Logger * const logger);

    /* @} */

	////////////////////////////////////////////////////////////////////////////////
    /**
     * @name State
     * @{
     */
private:
    typedef ::std::vector<Logger *> Loggers_t;
    /*
     * @brief the registered loggers.
     */
    Loggers_t m_loggers;

#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    /*
     * @brief process id of the process
     */
    Pid_t m_pid;
    static int const m_HOSTNAME_MAX_LEN=512;
    /*
     * @brief hostname where the process is running
     */
    char m_hostname[m_HOSTNAME_MAX_LEN];
    /*
     * @brief time when the singleton is constructed
     */
    Sec_t m_startup_sec;
    Usec_t m_startup_usec;
#endif
	// @}
};

////////////////////////////////////////////////////////////////////////////////


Logging_Implementation* Logging_Implementation::instance()
{
    if(m_instance!=NULL) return m_instance;
    return setup_instance();
}


////////////////////////////////////////////////////////////////////////////////

#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
#  define DIAGNOSTICS_RECORD(level,type,nr_what,str_what,base_file_name,file_name,line) \
    Sec_t sec; \
    Usec_t usec; \
    Platform::get_time(sec,usec); \
    Record record(level,type,nr_what,str_what,base_file_name,file_name,line, \
		  m_pid,Platform::get_tid(),sec,usec,m_hostname)
#else 
#  define DIAGNOSTICS_RECORD(level,type,nr_what,str_what,base_file_name,file_name,line) \
    Record record(level,type,nr_what,str_what,base_file_name,file_name,line)
#endif


void Logging_Implementation::log(Level_t const level,
								 Type_t const type,
								 int const nr_what,
								 ::std::string const & str_what,
								 char const * const base_file_name,
								 char const * const file_name,
								 int const line)
{
    DIAGNOSTICS_RECORD(level,type,nr_what,str_what,base_file_name,file_name,line);
    Loggers_t::iterator current(m_loggers.begin());
    Loggers_t::iterator const end(m_loggers.end());
    for(;current!=end;++current) (*current)->log(record);
}


INTERNAL_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

/**
 * @brief For internal panic logs (i.e., logging behind the @ref
 * DIAGNOSTICS_LOGGING_FACILITY_LOCK_GUARD).
 */ 
#define DIAGNOSTICS_PANIC_LOG_INTERNAL(WHAT)\
  ::DIAGNOSTICS_NAMESPACE::INTERNAL_NAMESPACE::Logging_Implementation::panic_log( \
   WHAT, \
   DIAGNOSTICS_BASE_FILE, \
  __FILE__, \
  __LINE__)


#endif
// vim:ts=4:sw=4
