/*
 * Copyright (C) 2010-2014 Codership Oy <info@codership.com>
 */

#include "wsdb.hpp"
#include "trx_handle.hpp"
#include "write_set.hpp"

#include "gu_lock.hpp"
#include "gu_throw.hpp"

#include <algorithm>

void galera::Wsdb::print(std::ostream& os) const
{
    os << "trx map:\n";
    for (galera::Wsdb::TrxMap::const_iterator i = trx_map_.begin();
         i != trx_map_.end();
         ++i)
    {
        os << i->first << " " << *i->second << "\n";
    }
    os << "conn query map:\n";
    for (galera::Wsdb::ConnMap::const_iterator i = conn_map_.begin();
         i != conn_map_.end();
         ++i)
    {
        os << i->first << " ";
    }
    os << "\n";
}


galera::Wsdb::Wsdb()
    :
    trx_pool_  (TrxHandle::LOCAL_STORAGE_SIZE(), 512, "LocalTrxHandle"),
    trx_map_     (),
    conn_trx_map_(),
    trx_mutex_   (),
    conn_map_    (),
    conn_mutex_  ()
{}


galera::Wsdb::~Wsdb()
{
    log_info << "wsdb trx map usage " << trx_map_.size()
             << " conn query map usage " << conn_map_.size();
    log_info << trx_pool_;

    /* There is potential race when a user triggers update of wsrep_provider
    that leads to deinit/unload of the provider. deinit/unload action of
    provider waits for replication to end. stop_replication routine waits
    for any active monitors for get released. But once monitors are
    released before the connection or transaction handle is discarded
    if deinit/unload sequence try to free up/destruct the provider user may
    hit the below mentioned assert. (that too in release mode only).

    In normal flow, case shouldn't arise but if the case shows-up then
    waiting for few seconds should help schedule release of connection and
    transaction handle.
    Even if wait doesn't help then it suggest some other serious issue
    that is blocking release of connection/transaction handle.
    In such case let the server assert as per the original flow.
    assert at this level should be generally safe given provider
    is unloading. */

    uint count = 5;
    while((trx_map_.size() != 0 || conn_map_.size() != 0) && count != 0)
    {
        log_info << "giving timeslice for connection/transaction handle"
                 << " to get released";
        sleep(1);
        --count;
    }

    // With debug builds just print trx and query maps to stderr
    // and don't clean up to let valgrind etc to detect leaks.
#ifndef NDEBUG
    log_info << *this;
    assert(trx_map_.size() == 0);
    assert(conn_map_.size() == 0);
#else
    for_each(trx_map_.begin(), trx_map_.end(), Unref2nd<TrxMap::value_type>());
    for_each(conn_trx_map_.begin(),
             conn_trx_map_.end(),
             Unref2nd<ConnTrxMap::value_type>());
#endif // !NDEBUG
}


inline galera::TrxHandle*
galera::Wsdb::find_trx(wsrep_trx_id_t const trx_id)
{
    gu::Lock lock(trx_mutex_);

    galera::TrxHandle* trx;

    if (trx_id != wsrep_trx_id_t(-1))
    {
        /* trx_id is valid and valid ids are unique.
        Search for valid trx_id in trx_id -> trx map. */
        TrxMap::iterator const i(trx_map_.find(trx_id));
        trx = (trx_map_.end() == i ? NULL : i->second);
    }
    else
    {
        /* trx_id is default so search for repsective connection id
        in connection-transaction map. */
        pthread_t const id = pthread_self();
        ConnTrxMap::iterator const i(conn_trx_map_.find(id));
        trx = (conn_trx_map_.end() == i ? NULL : i->second);
    }

    return (trx);
}


inline galera::TrxHandle*
galera::Wsdb::create_trx(const TrxHandle::Params& params,
                         const wsrep_uuid_t&  source_id,
                         wsrep_trx_id_t const trx_id)
{
    TrxHandle* trx(TrxHandle::New(trx_pool_, params, source_id, -1, trx_id));

    gu::Lock lock(trx_mutex_);

    galera::TrxHandle* trx_ref;
    if (trx_id != wsrep_trx_id_t(-1))
    {
        /* trx_id is valid add it to trx-map as valid trx_id is unique
        accross connections. */
        std::pair<TrxMap::iterator, bool> i
            (trx_map_.insert(std::make_pair(trx_id, trx)));
        if (gu_unlikely(i.second == false)) gu_throw_fatal;
        trx_ref = i.first->second;
    }
    else
    {
        /* trx_id is default so add trx object to connection map
        that is maintained based on pthread_id (alias for connection_id). */
         std::pair<ConnTrxMap::iterator, bool> i
             (conn_trx_map_.insert(std::make_pair(pthread_self(), trx)));
        if (gu_unlikely(i.second == false)) gu_throw_fatal;
        trx_ref = i.first->second;
    }

    return (trx_ref);
}


galera::TrxHandle*
galera::Wsdb::get_trx(const TrxHandle::Params& params,
                      const wsrep_uuid_t&  source_id,
                      wsrep_trx_id_t const trx_id,
                      bool const           create)
{
    TrxHandle* retval(find_trx(trx_id));

    if (0 == retval && create) retval = create_trx(params, source_id, trx_id);

    if (retval != 0) retval->ref();

    return retval;
}


galera::Wsdb::Conn*
galera::Wsdb::get_conn(wsrep_conn_id_t const conn_id, bool const create)
{
    gu::Lock lock(conn_mutex_);

    ConnMap::iterator i(conn_map_.find(conn_id));

    if (conn_map_.end() == i)
    {
        if (create == true)
        {
            std::pair<ConnMap::iterator, bool> p
                (conn_map_.insert(std::make_pair(conn_id, Conn(conn_id))));

            if (gu_unlikely(p.second == false)) gu_throw_fatal;

            return &p.first->second;
        }

        return 0;
    }

    return &(i->second);
}


galera::TrxHandle*
galera::Wsdb::get_conn_query(const TrxHandle::Params& params,
                             const wsrep_uuid_t&  source_id,
                             wsrep_trx_id_t const conn_id,
                             bool const           create)
{
    Conn* const conn(get_conn(conn_id, create));

    if (0 == conn) return 0;

    if (conn->get_trx() == 0 && create == true)
    {
        TrxHandle* trx
            (TrxHandle::New(trx_pool_, params, source_id, conn_id, -1));
        conn->assign_trx(trx);
    }

    return conn->get_trx();
}


void galera::Wsdb::discard_trx(wsrep_trx_id_t trx_id)
{
    gu::Lock lock(trx_mutex_);
    if (trx_id != wsrep_trx_id_t(-1))
    {
        TrxMap::iterator i;
        if ((i = trx_map_.find(trx_id)) != trx_map_.end())
        {
            i->second->unref();
            trx_map_.erase(i);
        }
    }
    else
    {
        ConnTrxMap::iterator i;
        pthread_t id = pthread_self();
        if ((i = conn_trx_map_.find(id)) != conn_trx_map_.end())
        {
            i->second->unref();
            conn_trx_map_.erase(i);
        }
    }
}


void galera::Wsdb::discard_conn_query(wsrep_conn_id_t conn_id)
{
    gu::Lock lock(conn_mutex_);
    ConnMap::iterator i;
    if ((i = conn_map_.find(conn_id)) != conn_map_.end())
    {
        i->second.assign_trx(0);
        conn_map_.erase(i);
    }
}
